/*
 * Copyright (C) 2010 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.exoplatform.gwtframework.editor.codemirror;

import java.util.Stack;
import java.util.Vector;

import org.exoplatform.gwtframework.commons.rest.MimeType;
import org.exoplatform.gwtframework.editor.api.Token;
import org.exoplatform.gwtframework.editor.api.Token.TokenType;
import org.exoplatform.gwtframework.editor.codemirror.Parser.Node;

import com.google.gwt.core.client.JavaScriptObject;

/**
 * @author <a href="mailto:dmitry.ndp@gmail.com">Dmytro Nochevnov</a>
 * @version $Id: $
 *
 */
public class JavaScriptParser extends Parser
{      
   
   private Stack<Node> nodeStack = new Stack<Node>();
   
   /**
    * Stack of blocks "{... {...} ...}"
    */
   private Stack<TokenType> enclosers = new Stack<TokenType>();

   /**
    * Indicate the position within the object creation statement like "var a = UWA._Data()"
    */
   boolean isObjectCreation;
   
   @Override
   public void init()
   {
      super.init();

      isObjectCreation = false;
      
      nodeStack.clear();
   }
   
   @Override
   Token parseLine(JavaScriptObject javaScriptNode, int lineNumber, Token currentToken, boolean hasParentParser)
   {
      // interrupt at the end of content
      if (javaScriptNode == null)
      {
         return currentToken;
      }
      
      // interrupt at the end of the line
      else if (getName(javaScriptNode).equals("BR"))
      {
         isObjectCreation = false;
         nodeStack.push(new Node("BR", ""));
      }
      
      else
      {
         nodeStack.push(new Node(javaScriptNode));
      }
      
      Node possibleNode = null;
      
      // to recognize "var a;"
      if (!nodeStack.isEmpty())
      {
         possibleNode = isVariableWithoutAssigmentStatement(nodeStack);
         if (possibleNode != null)
         {
            Token newVariable = new Token(possibleNode.getContent(), TokenType.VARIABLE, lineNumber, MimeType.APPLICATION_JAVASCRIPT);
            currentToken.addSubToken(newVariable);
         }
      }

      // to recognize "var a = "
      if (!nodeStack.isEmpty())
      {
         possibleNode = isVariableWithAssigmentStatement(nodeStack);
         if (possibleNode != null)
         {
            Token newVariable = new Token(possibleNode.getContent(), TokenType.VARIABLE, lineNumber, MimeType.APPLICATION_JAVASCRIPT);
            newVariable.setElementType("Object");
            currentToken.addSubToken(newVariable);
         }
      }      

      // to recognize function definition like "var a = function() {"
      if (!nodeStack.isEmpty())
      {
         if (isVariableWithFunctionAssigmentStatement(nodeStack) 
                  && currentToken.getLastSubToken() != null)
         {
            enclosers.push(TokenType.FUNCTION);
            currentToken.getLastSubToken().setType(TokenType.FUNCTION);
            currentToken.getLastSubToken().setName(currentToken.getLastSubToken().getName() + "()");
            currentToken.getLastSubToken().setElementType(null);
            currentToken = currentToken.getLastSubToken();
            
            nodeStack.clear();
         }
      }         
      
      // to recognize object creation like "var a = new UWA.Data()"
      if (!nodeStack.isEmpty() 
               && isObjectCreation
               && (isPoint(nodeStack.lastElement().getType(), nodeStack.lastElement().getContent())
                   || isJsVariable(nodeStack.lastElement().getType())
                   || isJsProperty(nodeStack.lastElement().getType()))
         )
      {
         currentToken.concatElementTypeOfLastSubToken(nodeStack.lastElement().getContent());
      }
      else
      {
         // to recognize start of object creation like "var a = new"
         if (isObjectCreation(nodeStack)
                  && currentToken.getLastSubToken() != null)
         {
            isObjectCreation = true;
            currentToken.getLastSubToken().setElementType(null);
         }
         else
         {
            isObjectCreation = false;
         }
      }
      
      // recognize open brace "{"
      if (!nodeStack.isEmpty()
               && isOpenBrace(nodeStack.lastElement()))
      {
         possibleNode = isFunctionStatement(nodeStack);
         if (possibleNode != null)
         {
            enclosers.push(TokenType.FUNCTION);
            Token newFunction = new Token(possibleNode.getContent() + "()", TokenType.FUNCTION, lineNumber, MimeType.APPLICATION_JAVASCRIPT);
            currentToken.addSubToken(newFunction);
            currentToken = newFunction;
            
            nodeStack.clear();
         }
           
         // to recognize anonymous function definition like "function() {"
         else if (isAnonymousFunctionStatement(nodeStack) != -1) 
         {
            enclosers.push(TokenType.FUNCTION);
            Token newFunction = new Token("function()", TokenType.FUNCTION, lineNumber, MimeType.APPLICATION_JAVASCRIPT);
            currentToken.addSubToken(newFunction);
            currentToken = newFunction;            
            
            nodeStack.clear();
         }   
         
         else
         {
            enclosers.push(TokenType.BLOCK);
         }
      }
      
      // recognize close brace "}"      
      else if (!nodeStack.isEmpty()
               && isCloseBrace(nodeStack.lastElement()))
      {         
         if (! enclosers.isEmpty())
         {
            if (TokenType.FUNCTION.equals(enclosers.lastElement()))
            {
               currentToken.setLastLineNumber(lineNumber);
               
               if (currentToken.getParentToken() != null)
               {         
                  currentToken = currentToken.getParentToken();                
               }
            }
            
            enclosers.pop();
         }
      }
      
      if (hasParentParser 
               || (!nodeStack.isEmpty() && isLineBreak(nodeStack.lastElement()))) 
      {
         return currentToken; // return current token to parent parser
      } 
      else
      {
         return parseLine(getNext(javaScriptNode), lineNumber, currentToken, false);
      }
   
   }

   /**
    * Recognize js property
    * @param nodeType
    * @return 
    */
   private boolean isJsProperty(String nodeType)
   {
      return "js-property".equals(nodeType);
   }

   /**
    * 
    * @param nodeStack
    * @return true in case like "var a = new "
    */
   private boolean isObjectCreation(Stack<Node> nodeStack)
   {     
      if (nodeStack.size() > 3)
      {
         return isNewKeyword(nodeStack.get(nodeStack.size() - 1))
            && isEqualNode(nodeStack.get(nodeStack.size() - 2))
            && (isJsVariable(nodeStack.get(nodeStack.size() - 3).getType()) 
                     || isJsLocalVariableDef(nodeStack.get(nodeStack.size() - 3).getType()))
            && isVarNode(nodeStack.get(nodeStack.size() - 4));
      }
      
      return false;
   }

   private boolean isVarNode(Node node)
   {
      return "js-keyword".equals(node.getType()) && "var".equals(node.getContent());
   }

   /**
    * Recognize "function" keyword
    * @param node
    * @return
    */
   private boolean isFunctionNode(Node node)
   {
      return "js-keyword".equals(node.getType()) && "function".equals(node.getContent());
   }

   private boolean isEqualNode(Node node)
   {
      return "js-operator".equals(node.getType()) && "=".equals(node.getContent());
   }
   
   /**
    * Recognize ";" node
    * @param node
    * @return
    */
   private boolean isSemicolonNode(Node node)
   {
      return "js-punctuation".equals(node.getType()) && ";".equals(node.getContent());
   };
   
   
   /**
    * Recognize ":" node
    * @param node
    * @return
    */
   private boolean isColonNode(Node node)
   {
      return "js-punctuation".equals(node.getType()) && ":".equals(node.getContent());
   };
   
   /**
    * Recognize variable out of the function
    * @param nodeType
    * @return
    */
   public static boolean isJsVariable(String nodeType)
   {
      return "js-variable".equals(nodeType);
   }   

   /**
    * Recognize local variable definition within the function like "function a() { var b = 1;  }"
    * @param nodeType
    * @return
    */
   public static boolean isJsLocalVariableDef(String nodeType)
   {
      return "js-variabledef".equals(nodeType);
   }

   /**
    * Recognize local variable within the function like "function a() { b = 1;  }"
    * @param nodeType
    * @return
    */
   public static boolean isJsLocalVariable(String nodeType)
   {
      return "js-localvariable".equals(nodeType);
   }
   
   /**
    * Recognize "new" keyword
    * @param nodeType
    * @param nodeContent
    * @return
    */
   private boolean isNewKeyword(Node node)
   {
      return "js-keyword".equals(node.getType()) && "new".equals(node.getContent());
   };
   
   /**
    * Recognize "." out of the js string
    * @return
    */
   public static boolean isPoint(String nodeType, String nodeContent)
   {
      return "js-punctuation".equals(nodeType) && ".".equals(nodeContent);
   }
   
   /**
    * Recognize "{"
    * @return true if there is open braces of method definition
    */
   private boolean isOpenBrace(Node node)
   {
      return "js-punctuation".equals(node.getType()) && "{".equals(node.getContent());
   }
   
   /**
    * Recognize "}"
    */
   private boolean isCloseBrace(Node node)
   {
      return "js-punctuation".equals(node.getType()) && "}".equals(node.getContent());
   }
   
   /**
    * Recognize function definition like function a(...){ and return its name
    * @param originalNodeStack
    * @return
    */
   private Node isFunctionStatement(Stack<Node> originalNodeStack)
   {
      if (originalNodeStack.size() > 4)
      {
         Stack<Node> cloneNodeStack = (Stack<Node>) originalNodeStack.clone(); 
         if (isOpenBrace(cloneNodeStack.pop()))
         {
            // pass BR or whitespace between ") ....  {"
            while (cloneNodeStack.size() > 3) {
               Node node = cloneNodeStack.pop();
               if (isCloseBracket(node))
               {
                  break;
               }
               
               // return if there is non-BR or non-whitespace node between ") ....  {"
               else if (!(isLineBreak(node) || isWhitespace(node)))
               {
                  return null;
               }
            }
            
            while (cloneNodeStack.size() > 2) {
               // test if this is like "function a (" 
               if (isOpenBracket(cloneNodeStack.lastElement()))
               {
                  if ( (isJsVariable(cloneNodeStack.get(cloneNodeStack.size() - 2).getType())
                           || isJsLocalVariableDef(cloneNodeStack.get(cloneNodeStack.size() - 2).getType())
                        )
                        && isFunctionNode(cloneNodeStack.get(cloneNodeStack.size() - 3))
                     )
                  {
                     return cloneNodeStack.get(cloneNodeStack.size() - 2);
                  }
               }
               
               cloneNodeStack.pop();
            }
         } 
      }
      
      return null;
   }

   /**
    * Recognize function definition like "function (...) {" and return index of function node "function"
    * @param originalNodeStack
    * @return index of function node "function"
    */
   private int isAnonymousFunctionStatement(Stack<Node> originalNodeStack)
   {
      if (originalNodeStack.size() > 3)
      {
         Stack<Node> cloneNodeStack = (Stack<Node>) originalNodeStack.clone(); 
         if (isOpenBrace(cloneNodeStack.pop()))
         {
            // pass BR or whitespace between ") ....  {"
            while (cloneNodeStack.size() > 2) {
               Node node = cloneNodeStack.pop();
               if (isCloseBracket(node))
               {
                  break;
               }
               
               // return if there is non-BR or non-whitespace node between ") ....  {"
               else if (!(isLineBreak(node) || isWhitespace(node)))
               {
                  return -1;
               }
            }
            
            while (cloneNodeStack.size() > 1) {
               // test if this is like "function (" 
               if (isOpenBracket(cloneNodeStack.lastElement()))
               {
                  if (isFunctionNode(cloneNodeStack.get(cloneNodeStack.size() - 2)))
                  {
                     return originalNodeStack.indexOf(cloneNodeStack.get(cloneNodeStack.size() - 2));
                  }
               }
               
               cloneNodeStack.pop();
            }
         } 
      }
      
      return -1;
   }

   /**
    * 
    * @param nodeStack
    * @return Node with variable in case like "var a/n" or "var a;"
    */
   private Node isVariableWithoutAssigmentStatement(Stack<Node> nodeStack)
   {
      if (nodeStack.size() > 2)
      {
         if ( (isLineBreak(nodeStack.get(nodeStack.size() - 1)) 
                  || isSemicolonNode(nodeStack.get(nodeStack.size() - 1)) 
               )
            && (isJsVariable(nodeStack.get(nodeStack.size() - 2).getType()) 
                  || isJsLocalVariableDef(nodeStack.get(nodeStack.size() - 2).getType()))
            && isVarNode(nodeStack.get(nodeStack.size() - 3))
         )
         {
            return nodeStack.get(nodeStack.size() - 2);
         }
      }
      
      return null;
   }

   /**
    * 
    * @param nodeStack
    * @return Node with variable in case like "var a = "
    */
   private Node isVariableWithAssigmentStatement(Stack<Node> nodeStack)
   {
      if (nodeStack.size() > 2)
      {
         if ( (isEqualNode(nodeStack.get(nodeStack.size() - 1))
               )
            && (isJsVariable(nodeStack.get(nodeStack.size() - 2).getType()) 
                  || isJsLocalVariableDef(nodeStack.get(nodeStack.size() - 2).getType()))
            && isVarNode(nodeStack.get(nodeStack.size() - 3))
         )
         {
            return nodeStack.get(nodeStack.size() - 2);
         }
      }
      
      return null;
   }

   /**
    * 
    * @param nodeStack
    * @return true in case like "var a = function(...) {"
    */
   private boolean isVariableWithFunctionAssigmentStatement(Stack<Node> nodeStack)
   {
      if (nodeStack.size() > 6)
      {
         // get indexOffunction node "function"
         int indexOfFunctionNode = isAnonymousFunctionStatement(nodeStack);
         
         if (indexOfFunctionNode > 2)
         {
            Stack<Node> cloneNodeStack = (Stack<Node>) nodeStack.clone();
            cloneNodeStack.setSize(indexOfFunctionNode);
            
            return isVariableWithAssigmentStatement(cloneNodeStack) != null;
         }
      }
      
      return false;
   }   

   /**
    * Recognize break line node with type "BR"
    * @param node
    * @return
    */
   private boolean isLineBreak(Node node)
   {
      return "BR".equals(node.getType());
   }

   /**
    * Recognize open brackets "(" 
    * @param node 
    * @return
    */
   private boolean isOpenBracket(Node node)
   {
      return "js-punctuation".equals(node.getType()) && "(".equals(node.getContent());
   }

   /**
    * Recognize open brackets ")" 
    * @param node
    * @return
    */
   private boolean isCloseBracket(Node node)
   {
      return "js-punctuation".equals(node.getType()) && ")".equals(node.getContent());
   }

   /**
    * 
    * @param node
    * @return true if this is whitespace node
    */
   private boolean isWhitespace(Node node)
   {
      return "whitespace".equals(node.getType());
   }
}