/**
 * Copyright (C) 2010 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

package org.exoplatform.gwtframework.editor.api;

import java.util.ArrayList;
import java.util.List;

public class Token
{
   private String name;

   private TokenType type;

   /**
    * Number of first (single) line with token
    */
   private int lineNumber;

   /**
    * Number of ended line of token in file content, like "}" for method or class
    */
   private int lastLineNumber;   

   private List<Token> subTokenList;

   /**
    * List of method's parameters 
    */
   private List<Token> parameters;

   /**
    * List of annotations of class, method, property or parameter
    */
   private List<Token> annotations;

   private String shortDescription;

   private String code;

   private String fullDescription;

   private String mimeType;

   private Token parentToken;

   private List<Modifier> modifier;

   /**
    * type of properties or methods within the groovy scripts
    */
   private String elementType;

   /**
    * Fully qualified name (fqn) of properties or methods within the groovy scripts
    */
   private String fqn;

   public Token(String name, TokenType type)
   {
      this.name = name;
      this.type = type;
   }

   public Token(String name, TokenType type, int lineNumber)
   {
      this(name, type, lineNumber, null, null, null, null, null, null, null, null);
   }

   public Token(String name, TokenType type, int lineNumber, String mimeType)
   {
      this(name, type, lineNumber, mimeType, null, null, null, null, null, null, null);
   }

   public Token(String name, TokenType type, int lineNumber, String mimeType, String elementType)
   {
      this(name, type, lineNumber, mimeType, null, null, null, null, elementType, null, null);
   }

   public Token(String name, TokenType type, int lineNumber, String mimeType, String elementType,
      List<Modifier> lavaModifiers)
   {
      this(name, type, lineNumber, mimeType, null, null, null, null, elementType, lavaModifiers, null);
   }

   public Token(String name, TokenType type, int lineNumber, String mimeType, String elementType,
      List<Modifier> lavaModifiers, String fqn)
   {
      this(name, type, lineNumber, mimeType, null, null, null, null, elementType, lavaModifiers, fqn);
   }

   public Token(String name, TokenType type, int lineNumber, List<Token> subTokenList)
   {
      this(name, type, lineNumber, null, null, null, null, subTokenList, null, null, null);
   }

   public Token(String name, TokenType type, String shortDescription, String code, String fullDescription)
   {
      this(name, type, 0, null, shortDescription, code, fullDescription, null, null, null, null);
   }

   public Token(String name, TokenType type, int lineNumber, String mimeType, String shortDescription, String code,
      String fullDescription, List<Token> subTokenList, String elementType, List<Modifier> modifier, String fqn)
   {
      this.name = name;
      this.type = type;
      this.lineNumber = lineNumber;
      this.mimeType = mimeType;
      this.shortDescription = shortDescription;
      this.code = code;
      this.fullDescription = fullDescription;
      this.subTokenList = subTokenList;
      this.elementType = elementType;
      this.modifier = modifier;
      this.fqn = fqn;
   }

   public Token()
   {
   }

   /**
    * Set token's name
    * @param name
    */
   public void setName(String name)
   {
      this.name = name;
   }

   /**
    * @return the shortDescription
    */
   public String getShortDescription()
   {
      return shortDescription;
   }

   /**
    * @return the code
    */
   public String getCode()
   {
      return code;
   }

   /**
    * @return the fullDescription
    */
   public String getFullDescription()
   {
      return fullDescription;
   }

   public String getName()
   {
      return name;
   }

   public TokenType getType()
   {
      return type;
   }

   public void setType(TokenType type)
   {
      this.type = type;
   }

   public int getLineNumber()
   {
      return lineNumber;
   }

   public void setLastLineNumber(int lastLineNumber)
   {
      this.lastLineNumber = lastLineNumber;
   }

   /**
    * @return number of ended line of token in file content, like "}" for method or class
    */
   public int getLastLineNumber()
   {
      return lastLineNumber;
   }
   
   public List<Token> getSubTokenList()
   {
      return subTokenList;
   }

   public String getMimeType()
   {
      return mimeType;
   }

   /**
    * @param lineNumber the lineNumber to set
    */
   public void setLineNumber(int lineNumber)
   {
      this.lineNumber = lineNumber;
   }

   /**
    * @param shortDescription the shortDescription to set
    */
   public void setShortDescription(String shortDescription)
   {
      this.shortDescription = shortDescription;
   }

   /**
    * @param code the code to set
    */
   public void setCode(String code)
   {
      this.code = code;
   }

   public void setParentToken(Token parentToken)
   {
      this.parentToken = parentToken;
   }

   public Token getParentToken()
   {
      return parentToken;
   }

   /**
    * @param fullDescription the fullDescription to set
    */
   public void setFullDescription(String fullDescription)
   {
      this.fullDescription = fullDescription;
   }

   /**
    * @param subTokenList the subTokenList to set
    */
   public void setSubTokenList(List<Token> subTokenList)
   {
      this.subTokenList = subTokenList;
   }

   /**
    * @param mimeType the mimeType to set
    */
   public void setMimeType(String mimeType)
   {
      this.mimeType = mimeType;
   }

   public void setElementType(String elementType)
   {
      this.elementType = elementType;
   }

   public String getElementType()
   {
      return elementType;
   }

   public void setFqn(String fqn)
   {
      this.fqn = fqn;
   }

   public String getFqn()
   {
      return fqn;
   }

   public List<Modifier> getModifiers()
   {
      return modifier;
   }

   public void addAnnotation(Token annotation)
   {
      if (this.annotations == null)
      {
         this.annotations = new ArrayList<Token>();
      }

      this.annotations.add(annotation);
   }

   public List<Token> getAnnotations()
   {
      return annotations;
   }

   public void setAnnotations(List<Token> annotations)
   {
      this.annotations = annotations;
   }

   public void addParameter(Token parameter)
   {
      if (this.parameters == null)
      {
         this.parameters = new ArrayList<Token>();
      }

      this.parameters.add(parameter);
   }

   public List<Token> getParameters()
   {
      return parameters;
   }

   public void setParameters(List<Token> parameters)
   {
      this.parameters = parameters;
   }

   /**
    * Adds subToken into the last sub token subTokenList
    * @param token
    * @param subToken
    */
   public void addSubTokenToTheLastSubToken(Token subToken)
   {
      if (subTokenList == null || subTokenList.size() == 0)
         return;

      Token lastToken = subTokenList.get(subTokenList.size() - 1);
      lastToken.addSubToken(subToken);

      subToken.parentToken = lastToken;
   }

   public void updateTypeOfLastSubTokenOfLastToken(TokenType newType)
   {
      if (subTokenList == null || subTokenList.size() == 0)
         return;

      Token lastSubToken = subTokenList.get(subTokenList.size() - 1);

      lastSubToken.updateTypeOfLastSubToken(newType);
   }
   
   public void updateTypeOfLastSubToken(TokenType newType)
   {
      if (subTokenList == null || subTokenList.size() == 0)
         return;

      Token lastSubToken = subTokenList.get(subTokenList.size() - 1);

      lastSubToken.setType(newType);
   }
   
   /**
    * Set token.elementType += elementType  
    * @param elementType
    */
   public void concatElementTypeOfLastSubToken(String elementType)
   {
      if (subTokenList == null || subTokenList.size() == 0)
         return;

      Token lastSubToken = subTokenList.get(subTokenList.size() - 1);

      if (lastSubToken.getElementType() == null)
      {
         lastSubToken.setElementType(elementType);
      }
      else
      {
         lastSubToken.setElementType(lastSubToken.getElementType() + elementType);
      }
   }
   
   public void addSubToken(Token subToken)
   {
      if (this.subTokenList == null)
      {
         this.subTokenList = new ArrayList<Token>();
      }

      this.subTokenList.add(subToken);
      subToken.setParentToken(this);
   }

   /**
    * remove last subtoken if subtoken list isn't empty
    */
   public void removeLastSubToken()
   {
      if (getSubTokenList() != null && getSubTokenList().size() != 0)
      {
         getSubTokenList().remove(getSubTokenList().size() - 1);
      }
   }

   public void setLastSubTokenName(String name)
   {
      Token token = getLastSubToken();

      if (token != null)
      {
         token.setName(name);
      }
   }

   /**
    * 
    * @return last subtoken or null if there is no any subtoken
    */
   public Token getLastSubToken()
   {
      if (getSubTokenList() != null && getSubTokenList().size() != 0)
      {
         return getSubTokenList().get(getSubTokenList().size() - 1);
      }

      return null;
   }

   /**
    * @return last annotation token or null if there is no any annotation token
    */
   public Token getLastAnnotationToken()
   {
      if (this.annotations != null && this.annotations.size() != 0)
      {
         return this.annotations.get(this.annotations.size() - 1);
      }

      return null;
   }

   /**
    * Concatenates the specified "string" to the end of name of last token in annotation list
    * @param string
    */
   public void lastAnnotationTokenNameConcat(String string)
   {
      if (getLastAnnotationToken() != null)
      {
         getLastAnnotationToken().setName(getLastAnnotationToken().getName().concat(string)); // update last  annotation token
      }
   }

   public enum TokenType {
      VARIABLE, FUNCTION, PROPERTY, METHOD, KEYWORD, TEMPLATE, TAG, ATTRIBUTE, CDATA, ROOT, TAG_BREAK, CLASS, GROOVY_TAG, ANNOTATION, PARAMETER, IMPORT, PACKAGE, TYPE, OBJECT, ARRAY, STRING, NUMBER, BOOLEAN, NULL, ERROR, INTERFACE, BLOCK;
   }

   public enum Modifier {
      STATIC(0x00000008), FINAL(0x00000010), PRIVATE(0x00000002), PUBLIC(0x00000001), PROTECTED(0x00000004), ABSTRACT(
         0x00000400), STRICTFP(0x00000800), SYNCHRONIZED(0x00000020), THREADSAFE(0), TRANSIENT(0x00000080), VOLATILE(
         0x00000040);
      private final int mod;

      Modifier(int i)
      {
         this.mod = i;
      }

      public int value()
      {
         return mod;
      }
   }
}