/*
 * Copyright (C) 2010 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.exoplatform.gwtframework.editor.codemirror;

import java.util.ArrayList;
import java.util.List;

import org.exoplatform.gwtframework.editor.api.Token;
import org.exoplatform.gwtframework.editor.api.Token.TokenType;

import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.json.client.JSONArray;
import com.google.gwt.json.client.JSONException;
import com.google.gwt.json.client.JSONObject;
import com.google.gwt.json.client.JSONValue;

/**
 * @author <a href="mailto:dmitry.ndp@gmail.com">Dmytro Nochevnov</a>
 * @version $Id: $
 *
 */
public class JSONParser extends Parser
{      

   @Override
   public List<Token> getTokenList(JavaScriptObject editor)
   {
      
      String jsonData = Parser.getTextFromEditor(editor);

      List<Token> tokenList = new ArrayList<Token>();
      
      Token JSONToken;
      
      if ((JSONToken = parseJSON(jsonData)) != null)
      {
         tokenList.add(JSONToken);         
      }
          
      return tokenList;
   }
   
   /**
    * @param jsonData
    * @return token constructed with aim of com.google.gwt.json.client.JSONParser, or null if jsonData is empty
    */
   Token parseJSON(String jsonData)
   {
      // test JSON file is empty 
      if (jsonData.isEmpty() || jsonData.equals("\n"))
      {
         return null;
      }
      
      JSONValue parsedObject;
      try
      {
         parsedObject = com.google.gwt.json.client.JSONParser.parse(jsonData);
      }
      catch (JSONException e)
      {
         // get short error description
         if (e.getMessage().split("\n").length > 0)
         {
            return new Token("JSON parse exception: " + e.getMessage().split("\n")[0].replace("com.google.gwt.core.client.JavaScriptException: (SyntaxError): ", ""), TokenType.ERROR);
         }
         else
         {
            return new Token("JSON parse exception!", TokenType.ERROR);
         }
      }      
      
      if (parsedObject.isObject() != null) {
         Token jsonToken = new Token("JSON", TokenType.OBJECT);
         jsonToken.setSubTokenList(parseJsonObject(parsedObject.isObject()));
         return jsonToken;
      }
      
      return null;
   }
   
   List<Token> parseJsonObject(JSONObject jsonObject)
   {
      List<Token> values = new ArrayList<Token>();
      
      if (jsonObject.size() > 0)
      {
         JSONValue jsonValue;
         Token tokenValue;
         
         for (String jsonProperty: jsonObject.keySet())
         {
            jsonValue = jsonObject.get(jsonProperty);

            tokenValue = parseJsonValue(jsonProperty, jsonValue);

            if (tokenValue != null)
            {
               values.add(tokenValue);
            }
         }
      }
      
      return values;
   }
   
   private List<Token> parseJsonArray(JSONArray jsonArray)
   {
      List<Token> values = new ArrayList<Token>();

      JSONValue jsonValue;
      Token tokenValue;

      for (int i = 0; i < jsonArray.size(); i++)
      {
         jsonValue = jsonArray.get(i);

         tokenValue = parseJsonValue(i + "", jsonValue);

         if (tokenValue != null)
         {
            values.add(tokenValue);
         }
      }
      
      return values;      
   }

   private Token parseJsonValue(String jsonProperty, JSONValue jsonValue)
   {
      Token tokenValue;
      if (jsonValue.isObject() != null)
      {
         tokenValue = new Token(jsonProperty, TokenType.OBJECT);
         tokenValue.setSubTokenList(parseJsonObject(jsonValue.isObject()));
      }
      
      else if (jsonValue.isArray() != null)
      {
         tokenValue = new Token(jsonProperty, TokenType.ARRAY);
         tokenValue.setSubTokenList(parseJsonArray(jsonValue.isArray()));
      }
      
      else if (jsonValue.isString() != null)
      {
         tokenValue = new Token(jsonProperty + ": \"" + jsonValue.isString().stringValue() + "\"", TokenType.STRING);
      }

      else if (jsonValue.isNumber() != null)
      {
         tokenValue = new Token(jsonProperty + ": " + jsonValue.isNumber().toString(), TokenType.NUMBER);
      }            

      else if (jsonValue.isBoolean() != null)
      {
         tokenValue = new Token(jsonProperty + ": " + jsonValue.isBoolean().toString(), TokenType.BOOLEAN);
      }

      else if (jsonValue.isNull() != null)
      {
         tokenValue = new Token(jsonProperty + ": " + jsonValue.isNull().toString(), TokenType.NULL);
      }
      
      else
      {
         return null;
      }
      
      return tokenValue;
   }
   
   
   
//   private String lastNodeContent;
//   
//   private String lastNodeType;
//   
//   private TokenType lastTokenType;
//   
//   @Override
//   Token parseLine(JavaScriptObject node, int lineNumber, Token currentToken, boolean hasParentParser)
//   {
//      // interrupt at the end of the line or content
//      if ((node == null) || getName(node).equals("BR"))
//         return currentToken;
//      
//      String nodeContent = getContent(node).trim(); // returns text without ended space " " in the text
//      String nodeType = getType(node);  
//      
//      
//      lastNodeContent = nodeContent;
//      lastNodeType = nodeType;
//
//      if (hasParentParser) 
//      {
//         return currentToken; // return current token to parent parser
//      } 
//      else
//      {
//         return parseLine(getNext(node), lineNumber, currentToken, false);
//      }
//   
//   }
   

}