/**
 * Copyright (C) 2010 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

package org.exoplatform.gwtframework.ui.client.button;

import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Element;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlowPanel;

/**
 * 
 * Created by The eXo Platform SAS .
 * 
 * @author <a href="mailto:gavrikvetal@gmail.com">Vitaliy Gulyy</a>
 * @version $
 */

public class IconButton extends Composite
{

   /**
    * Styles for component.
    */
   private interface Style
   {

      static final String BUTTON_PANEL = "exoIconButtonPanel";

      static final String BUTTON_PANEL_OVER = "exoIconButtonPanelOver";

      static final String BUTTON_PANEL_DOWN = "exoIconButtonPanelDown";

      static final String BUTTON_PANEL_SELECTED = "exoIconButtonPanelSelected";

      static final String BUTTON_PANEL_SELECTED_OVER = "exoIconButtonPanelSelectedOver";

      static final String BUTTON_PANEL_SELECTED_DOWN = "exoIconButtonPanelSelectedDown";

      static final String BUTTON_ICON = "exoIconButtonIcon";

   }

   /**
    * Button's panel.
    */
   private ButtonPanel buttonPanel;

   /**
    * Icon for enabled state.
    */
   protected String icon;

   /**
    * Icon for disabled state.
    */
   protected String disabledIcon;

   /**
    * Command which will be executed when button was pressed.
    */
   protected Command command;

   /**
    * Is enabled.
    */
   private boolean enabled = true;

   /**
    * Is button selected.
    */
   private boolean selected = false;

   /**
    * Create IconButton with icons. Icons is an HTML image and must be prepared like "<img ... />" tag
    * Use org.exoplatform.gwtframework.ui.client.util.ImageHelper to create Image from some sources.
    * 
    * @param icon icon for enabled state
    * @param disabledIcon icon for disabled state
    */
   public IconButton(String icon, String disabledIcon)
   {
      this(icon, disabledIcon, null);
   }

   /**
    * Create IconButton with icons and command. Icons is an HTML image and must be prepared like "<img ... />" tag
    * Use org.exoplatform.gwtframework.ui.client.util.ImageHelper to create Image from some sources.
    * 
    * @param icon icon for enabled state
    * @param disabledIcon icon for disabled state
    * @param command command which will be executed when button was pressed.
    */
   public IconButton(String icon, String disabledIcon, Command command)
   {
      this.icon = icon;
      this.disabledIcon = disabledIcon;
      this.command = command;

      buttonPanel = new ButtonPanel();
      initWidget(buttonPanel);
      buttonPanel.setStyleName(Style.BUTTON_PANEL);
      DOM.setElementAttribute(buttonPanel.getElement(), "enabled", "" + enabled);      

      renderIcon();
   }

   /**
    * Get command which will be executed when button was pressed.
    * 
    * @return command which will be executed when button was pressed
    */
   public Command getCommand()
   {
      return command;
   }

   /**
    * Set command which will be executed when button was pressed.
    * 
    * @param command command which will be executed when button was pressed
    */
   public void setCommand(Command command)
   {
      this.command = command;
   }

   /**
    * Get is enabled.
    * 
    * @return is enabled
    */
   public boolean isEnabled()
   {
      return enabled;
   }

   /**
    * Set is enabled.
    * 
    * @param enabled is enabled
    */
   public void setEnabled(boolean enabled)
   {
      this.enabled = enabled;
      buttonPanel.setStyleName(Style.BUTTON_PANEL);
      DOM.setElementAttribute(buttonPanel.getElement(), "enabled", "" + enabled);      
      renderIcon();
   }

   /**
    * Get icon for enabled state.
    * 
    * @return icon which is uses for enabled state 
    */
   public String getIcon()
   {
      return icon;
   }

   /**
    * Set icon as HTML image for enabled state. Image must be prepared like "<img ... />" tag
    * Use org.exoplatform.gwtframework.ui.client.util.ImageHelper for creation Image from some sources.
    * 
    * @param icon icon
    */
   public void setIcon(String icon)
   {
      this.icon = icon;
      if (enabled)
      {
         renderIcon();
      }
   }

   /**
    * Get icon for disabled state.
    * 
    * @return icon which is uses for disabled state
    */
   public String getDisabledIcon()
   {
      return disabledIcon;
   }

   /**
    * Set icon as HTML image for disabled state. Image must be prepared like "<img ... />" tag
    * Use org.exoplatform.gwtframework.ui.client.util.ImageHelper for creation Image from some sources.
    * 
    * @param disabledIcon icon which is uses for disabled state 
    */
   public void setDisabledIcon(String disabledIcon)
   {
      this.disabledIcon = disabledIcon;
      if (enabled)
      {
         return;
      }

      renderIcon();
   }

   /**
    * Get is button selected.
    * 
    * @return <code>true</code> if button selected, <code>false</code> otherwise. 
    */
   public boolean isSelected()
   {
      return selected;
   }

   /**
    * Set button is selected.
    * 
    * @param selected is button selected
    */
   public void setSelected(boolean selected)
   {
      this.selected = selected;
      if (selected)
      {
         buttonPanel.setStyleName(Style.BUTTON_PANEL_SELECTED);
      }
      else
      {
         buttonPanel.setStyleName(Style.BUTTON_PANEL);
      }
   }

   /**
    * Redraw icon.
    */
   private void renderIcon()
   {
      if (enabled)
      {
         if (icon != null)
         {
            buttonPanel.getElement().setInnerHTML(icon);
         }
         else
         {
            buttonPanel.getElement().setInnerHTML("");
            return;
         }
      }
      else
      {
         if (disabledIcon != null)
         {
            buttonPanel.getElement().setInnerHTML(disabledIcon);
         }
         else
         {
            buttonPanel.getElement().setInnerHTML("");
            return;
         }

      }

      Element e = buttonPanel.getElement();
      Element imageElement = DOM.getChild(e, 0);
      DOM.setElementAttribute(imageElement, "class", Style.BUTTON_ICON);
   }

   /**
    * Mouse Over handler.
    */
   private void onMouseOver()
   {
      if (selected)
      {
         buttonPanel.setStyleName(Style.BUTTON_PANEL_SELECTED_OVER);
      }
      else
      {
         buttonPanel.setStyleName(Style.BUTTON_PANEL_OVER);
      }
   }

   /**
    * Mouse Out handler.
    */
   private void onMouseOut()
   {
      if (selected)
      {
         buttonPanel.setStyleName(Style.BUTTON_PANEL_SELECTED);
      }
      else
      {
         buttonPanel.setStyleName(Style.BUTTON_PANEL);
      }
   }

   /**
    * Mouse Down handler.
    */
   private void onMouseDown()
   {
      if (selected)
      {
         buttonPanel.setStyleName(Style.BUTTON_PANEL_SELECTED_DOWN);
      }
      else
      {
         buttonPanel.setStyleName(Style.BUTTON_PANEL_DOWN);
      }
   }

   /**
    * Mouse Up handler.
    */
   private void onMouseUp()
   {
      if (selected)
      {
         buttonPanel.setStyleName(Style.BUTTON_PANEL_SELECTED_OVER);
      }
      else
      {
         buttonPanel.setStyleName(Style.BUTTON_PANEL_OVER);
      }

      if (command != null)
      {
         command.execute();
      }
   }

   /**
    *  ButtonPanel class uses for listening mouse events on button.
    */
   private class ButtonPanel extends FlowPanel
   {

      public ButtonPanel()
      {
         sinkEvents(Event.ONMOUSEOVER | Event.ONMOUSEOUT | Event.ONMOUSEDOWN | Event.ONMOUSEUP);
      }

      /**
       * Handle browser's events.
       */
      @Override
      public void onBrowserEvent(Event event)
      {
         if (!enabled)
         {
            return;
         }

         switch (DOM.eventGetType(event))
         {
            case Event.ONMOUSEOVER :
               onMouseOver();
               break;

            case Event.ONMOUSEOUT :
               onMouseOut();
               break;

            case Event.ONMOUSEDOWN :
               if (event.getButton() != Event.BUTTON_LEFT)
               {
                  return;
               }

               onMouseDown();
               break;

            case Event.ONMOUSEUP :
               onMouseUp();
               break;
         }
      }

   }

   /**
    * Sets the title associated with this button. The title is the 'tool-tip'
    * displayed to users when they hover over the object.
    * 
    * @param title the object's new title
    */
   public void setTitle(String title)
   {
      buttonPanel.setTitle(title);
   }

}
