/**
 * Copyright (C) 2010 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 *
 */

package org.exoplatform.gwtframework.ui.client.menu;

import java.util.ArrayList;
import java.util.List;

import org.exoplatform.gwtframework.ui.client.util.ExoStyle;

import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Element;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.SimplePanel;

/**
 * 
 * Created by The eXo Platform SAS .
 * 
 * PopupMenu is visual component represents all known Popup Menu. 
 * 
 * @author <a href="mailto:gavrikvetal@gmail.com">Vitaliy Gulyy</a>
 * @version $
 */

public class PopupMenu extends Composite
{

   /**
    * This table uses for handling mouse events.
    */
   private class PopupMenuTable extends FlexTable
   {

      public PopupMenuTable()
      {
         sinkEvents(Event.ONMOUSEOVER | Event.ONMOUSEDOWN);
      }

      @Override
      public void onBrowserEvent(Event event)
      {
         Element td = getEventTargetCell(event);

         if (td == null)
            return;
         Element tr = DOM.getParent(td);

         String index = DOM.getElementAttribute(tr, "item-index");
         if (index == null || "".equals(index))
         {
            return;
         }

         String enabled = DOM.getElementAttribute(tr, "item-enabled");
         if (enabled == null || "".equals(enabled) || "false".equals(enabled))
         {
            return;
         }

         int itemIndex = Integer.parseInt(index);
         PopupMenuItem menuItem = (PopupMenuItem)menuItems.get(itemIndex);
         if (!menuItem.isEnabled())
         {
            return;
         }

         switch (DOM.eventGetType(event))
         {
            case Event.ONMOUSEOVER :
               onMouseOver(tr);
               break;

            case Event.ONMOUSEDOWN :
               onMouseDown(tr);
               break;

            default :
               break;
         }

      }
   }

   /**
    * Working variable is needs to indicate when PopupMenu has at list one MenuItem with selected state. 
    */
   private boolean hasCheckedItems;

   /**
    * Callback uses for notify Parent Menu when menu item is selecred.
    */
   private ItemSelectedHandler itemSelectedCallback;

   /**
    * Lock layer uses as root for displaying this PopupMenu and uses for locking screen and hiding menu when user just clicked outside menu.
    */
   private MenuLockLayer lockLayer;

   /**
    * List of Menu Items.
    */
   private List<MenuItem> menuItems;

   /**
    * Contains opened sub Popup Menu.
    */
   private PopupMenu openedSubPopup;

   /**
    * Contains HTML element ( <TR> ) which is hovered for the current time.
    */
   private Element hoveredTR;

   /**
    * Working variable.
    * PopupMenu panel.
    */
   private SimplePanel popupMenuPanel;

   /**
    * Working variable. Special table uses for handling mouse events.
    */
   private PopupMenuTable table;

   /**
    * Create PopupMenu
    * 
    * @param menuItems - list of menu items.
    * @param lockLayer - lock layer, uses as rot for attaching this popup menu.
    * @param itemSelectedCallback - callback, uses for notifying parent menu when menu item is selected.
    */
   public PopupMenu(List<MenuItem> menuItems, MenuLockLayer lockLayer, ItemSelectedHandler itemSelectedCallback)
   {
      this.menuItems = new ArrayList<MenuItem>();

      /*
       * show only visible items and delimiters
       */
      for (MenuItem item : menuItems)
      {
         if (item.getTitle() == null)
         {
            this.menuItems.add(item);
            continue;
         }

         if (item.isVisible())
         {
            this.menuItems.add(item);
         }
      }

      filterMenuItems();

      this.lockLayer = lockLayer;
      this.itemSelectedCallback = itemSelectedCallback;

      popupMenuPanel = new SimplePanel();
      initWidget(popupMenuPanel);
      popupMenuPanel.setStyleName(PopupMenuStyle.MENU_MAIN);

      hasCheckedItems = hasCheckedItems();

      redraw();
   }

   private void filterMenuItems()
   {
      /*
       * remove delimiters from start
       */
      while (menuItems.size() > 0 && menuItems.get(0).getTitle() == null)
      {
         menuItems.remove(0);
      }

      /*
       * remove delimiters from end
       */
      while (menuItems.size() > 0 && menuItems.get(menuItems.size() - 1).getTitle() == null)
      {
         menuItems.remove(menuItems.size() - 1);
      }

      /*
       * avoid sequence of two delimiters
       */
      boolean found = false;
      int index = 0;
      while (index < menuItems.size())
      {
         if (menuItems.get(index).getTitle() == null)
         {
            if (found)
            {
               menuItems.remove(index);
               continue;
            }

            found = true;
         }
         else
         {
            found = false;
         }

         index++;
      }

   }

   /**
    * Close this Popup Menu.
    */
   public void closePopup()
   {
      if (openedSubPopup != null)
      {
         openedSubPopup.closePopup();
      }
      removeFromParent();
   }

   /**
    * Render Popup Menu component.
    */
   private void redraw()
   {
      table = new PopupMenuTable();
      table.setStyleName(PopupMenuStyle.MENU_TABLE);
      table.setCellPadding(0);
      table.setCellSpacing(0);
      DOM.setElementAttribute(table.getElement(), "border", "0");

      for (int i = 0; i < menuItems.size(); i++)
      {
         PopupMenuItem menuItem = (PopupMenuItem)menuItems.get(i);

         if (menuItem.getTitle() == null)
         {
            table.getFlexCellFormatter().setColSpan(i, 0, hasCheckedItems ? 5 : 4);
            table.setHTML(i, 0, "<nobr><hr></nobr>");
            table.getCellFormatter().setStyleName(i, 0, PopupMenuStyle.DELIMITER);
         }
         else
         {
            table.setHTML(i, 0, menuItem.getIcon());
            table.getCellFormatter().setStyleName(i, 0,
               menuItem.isEnabled() ? PopupMenuStyle.ICON_FIELD : PopupMenuStyle.ICON_FIELD_DISABLED);

            int work = 1;

            if (hasCheckedItems)
            {
               String checkImage;
               if (menuItem.isSelected())
               {
                  checkImage = ExoStyle.getEXoStyleURL() + "popupMenu/check.gif";
               }
               else
               {
                  checkImage = ExoStyle.getEXoStyleURL() + "blank.gif";
               }

               table.setHTML(i, work, "<img src=\"" + checkImage + "\">");
               table.getCellFormatter().setStyleName(i, work,
                  menuItem.isEnabled() ? PopupMenuStyle.CHECK_FIELD : PopupMenuStyle.CHECK_FIELD_DISABLED);
               work++;
            }

            table.setHTML(i, work, "<nobr>" + menuItem.getTitle() + "</nobr>");
            table.getCellFormatter().setStyleName(i, work,
               menuItem.isEnabled() ? PopupMenuStyle.TITLE_FIELD : PopupMenuStyle.TITLE_FIELD_DISABLED);

            work++;

            String hotKey = menuItem.getHotKey();
            if (hotKey == null)
            {
               hotKey = "&nbsp;";
            }
            else
            {
               hotKey =
                  "<nobr><font color=\"" + (menuItem.isEnabled() ? "#000088" : "#AAAAAA") + "\">&nbsp;[" + hotKey
                     + "]&nbsp;</font></nobr>";
            }

            table.setHTML(i, work, hotKey);
            table.getCellFormatter().setStyleName(i, work,
               menuItem.isEnabled() ? PopupMenuStyle.KEY_FIELD : PopupMenuStyle.KEY_FIELD_DISABLED);

            work++;

            if (menuItem.getItems().size() == 0)
            {
               table.setHTML(i, work, "<img src=\"" + ExoStyle.getEXoStyleURL() + "blank.gif" + "\" class=\""
                  + PopupMenuStyle.SUBMENU_IMAGE + "\" />");
               table.getCellFormatter().setStyleName(i, work,
                  menuItem.isEnabled() ? PopupMenuStyle.SUBMENU_FIELD : PopupMenuStyle.SUBMENU_FIELD_DISABLED);
            }
            else
            {
               table.setHTML(i, work, "<img src=\"" + ExoStyle.getEXoStyleURL() + "popupMenu/submenu.gif\" class=\""
                  + PopupMenuStyle.SUBMENU_IMAGE + "\" />");
               table.getCellFormatter().setStyleName(i, work,
                  menuItem.isEnabled() ? PopupMenuStyle.SUBMENU_FIELD : PopupMenuStyle.SUBMENU_FIELD_DISABLED);
            }

            work++;

            DOM.setElementAttribute(table.getRowFormatter().getElement(i), "item-index", "" + i);
            DOM.setElementAttribute(table.getRowFormatter().getElement(i), "item-enabled", "" + menuItem.isEnabled());
         }

      }

      popupMenuPanel.add(table);
   }

   /**
    * @return true when at list one item from list of menu items has selected state.
    */
   private boolean hasCheckedItems()
   {
      for (int i = 0; i < menuItems.size(); i++)
      {
         PopupMenuItem menuItem = (PopupMenuItem)menuItems.get(i);
         if (menuItem.isSelected())
         {
            return true;
         }
      }

      return false;
   }

   /**
    * Handling MouseOut event.
    * 
    * @param tr - element to be processed.
    */
   protected void onMouseOut(Element tr)
   {
      if (hasCheckedItems)
      {
         Element iconTD = DOM.getChild(tr, 0);
         Element checkTD = DOM.getChild(tr, 1);
         Element titleTD = DOM.getChild(tr, 2);
         Element hotKeyTD = DOM.getChild(tr, 3);
         Element submenuTD = DOM.getChild(tr, 4);

         iconTD.setClassName(PopupMenuStyle.ICON_FIELD);
         checkTD.setClassName(PopupMenuStyle.CHECK_FIELD);
         titleTD.setClassName(PopupMenuStyle.TITLE_FIELD);
         hotKeyTD.setClassName(PopupMenuStyle.KEY_FIELD);
         submenuTD.setClassName(PopupMenuStyle.SUBMENU_FIELD);
      }
      else
      {
         Element iconTD = DOM.getChild(tr, 0);
         Element titleTD = DOM.getChild(tr, 1);
         Element hotKeyTD = DOM.getChild(tr, 2);
         Element submenuTD = DOM.getChild(tr, 3);

         iconTD.setClassName(PopupMenuStyle.ICON_FIELD);
         titleTD.setClassName(PopupMenuStyle.TITLE_FIELD);
         hotKeyTD.setClassName(PopupMenuStyle.KEY_FIELD);
         submenuTD.setClassName(PopupMenuStyle.SUBMENU_FIELD);
      }

   }

   /**
    * Handling MouseOver event.
    * 
    * @param tr - element to be processed.
    */
   protected void onMouseOver(Element tr)
   {
      //      if (parentRemoveSubPopupTimer != null)
      //      {
      //         parentRemoveSubPopupTimer.cancel();
      //      }

      if (tr == hoveredTR)
      {
         return;
      }

      if (hoveredTR != null)
      {
         onMouseOut(hoveredTR);
      }

      hoveredTR = tr;

      if (hasCheckedItems)
      {
         Element iconTD = DOM.getChild(tr, 0);
         Element checkTD = DOM.getChild(tr, 1);
         Element titleTD = DOM.getChild(tr, 2);
         Element hotKeyTD = DOM.getChild(tr, 3);
         Element submenuTD = DOM.getChild(tr, 4);

         iconTD.setClassName(PopupMenuStyle.ICON_FIELD_OVER);
         checkTD.setClassName(PopupMenuStyle.CHECK_FIELD_OVER);
         titleTD.setClassName(PopupMenuStyle.TITLE_FIELD_OVER);
         hotKeyTD.setClassName(PopupMenuStyle.KEY_FIELD_OVER);
         submenuTD.setClassName(PopupMenuStyle.SUBMENU_FIELD_OVER);

      }
      else
      {
         Element iconTD = DOM.getChild(tr, 0);
         Element titleTD = DOM.getChild(tr, 1);
         Element hotKeyTD = DOM.getChild(tr, 2);
         Element submenuTD = DOM.getChild(tr, 3);

         iconTD.setClassName(PopupMenuStyle.ICON_FIELD_OVER);
         titleTD.setClassName(PopupMenuStyle.TITLE_FIELD_OVER);
         hotKeyTD.setClassName(PopupMenuStyle.KEY_FIELD_OVER);
         submenuTD.setClassName(PopupMenuStyle.SUBMENU_FIELD_OVER);
      }

   }

   /**
    * Handling MouseDown event.
    * 
    * @param tr - element to be processed.
    */
   protected void onMouseDown(Element tr)
   {
      int itemIndex = Integer.parseInt(DOM.getElementAttribute(tr, "item-index"));
      PopupMenuItem menuItem = (PopupMenuItem)menuItems.get(itemIndex);
      if (menuItem.getItems().size() == 0)
      {
         if (itemSelectedCallback != null)
         {
            itemSelectedCallback.onMenuItemSelected(menuItem);
         }

         if (menuItem.getCommand() != null)
         {
            menuItem.getCommand().execute();
         }
      }
      else
      {
         if (openedSubPopup != null)
         {
            openedSubPopup.closePopup();
         }

         int x = getAbsoluteLeft();
         int y = tr.getAbsoluteTop();
         int w = getOffsetWidth();
         openedSubPopup = new PopupMenu(menuItem.getItems(), lockLayer, itemSelectedCallback);
         lockLayer.add(openedSubPopup, x + w, y - lockLayer.getTopOffset() - 1);
      }
   }

}
