/*
 * Isomorphic SmartClient
 * Version SC_SNAPSHOT-2010-08-16 (2010-08-16)
 * Copyright(c) 1998 and beyond Isomorphic Software, Inc. All rights reserved.
 * "SmartClient" is a trademark of Isomorphic Software, Inc.
 *
 * licensing@smartclient.com
 *
 * http://smartclient.com/license
 */


// This file encapsulates SmartClient's HTML5 (and fallbacks) offline support.  For ease of 
// porting to pure GWT, isc.Offline is created as a plain JS object with no dependence on 
// the SmartClient class system



(function () {

var Offline = {

    // Offline detection
    // ---------------------------------------------------------------------------------------
    isOffline : function () {
        // Working via a local var so it can be flipped in the debugger for ease of testing
        var offline = window.navigator.onLine ? false : true;
        return offline;
    },
    
    // Overall cache managment
    // ---------------------------------------------------------------------------------------
    KEY_PREFIX: "isc:",
    
    LOCAL_STORAGE: "localStorage",                // HTML5 browsers
    GLOBAL_STORAGE: "globalStorage",              // Older versions of Firefox
    DATABASE_STORAGE: "databaseStorage",          // Safari 3.1 & 3.2 (HTML5 browsers as well,
                                                  // but these older Safari versions have no 
                                                  // localStorage support)
    GEARS_DATABASE_API: "gears database api",     // Built into pre-4 versions of Chrome
    USERDATA_PERSISTENCE: "userData persistence", // IE6 and IE7
    GOOGLE_GEARS: "google gears",                 // An explicitly installed Gears instance
    NO_MECHANISM: "no discovered mechanism",      // We won't get this very often!
    
    maxResponsesToPersist: 100,
    
    // The force param tells the function to remove even items that
    // look like we didn't put them there - useful for if you 
    // really need the cache to be clear
    localStorageType : function () {
        if (window.localStorage) return this.LOCAL_STORAGE;
        if (window.globalStorage) return this.GLOBAL_STORAGE;
        if (window.openDatabase) return this.DATABASE_STORAGE;
        // Not implementing anything else just yet
        return this.NO_MECHANISM;
    },

    getStoredValuesCount : function () {
        switch (this.localStorageType()) {
            case this.LOCAL_STORAGE:
                return localStorage.length;
            case this.GLOBAL_STORAGE:
            case this.DATABASE_STORAGE:
            case this.GEARS_DATABASE_API:
            case this.USERDATA_PERSISTENCE:
            case this.GOOGLE_GEARS:
                this.logError("Persistence method '" + this.localStorageType() + "' not yet supported");
                break;
        }
    },

    clearCache : function (force) {
        var count = this.getStoredValuesCount();
        while (this.getStoredValuesCount() > 0) {
            this.removeOldestEntry();
            // This just prevents an eternal loop if for whatever reason running 
            // removeOldestEntry() does not result in a different count 
            if (this.getStoredValuesCount() == count) break;
            count = this.getStoredValuesCount();
        }
    },
    
    logCacheContents : function (maxEntryLength) {
        var contents = this.getCacheContents();
        this.logWarn("Dumping the contents of the browser's local storage:");
        for (var key in contents) {
            var value = contents[key];
            if (value && value.length > maxEntryLength) {
                value = value.substring(0, maxEntryLength);
            }
            this.logWarn(key + " ===> " + value);
        }
    },
    
    getCacheContents : function () {
        var index = 0,
            contents = {},
            count = this.getStoredValuesCount();
        while (index < count) {
            var key = this.getKeyForIndex(index);
            if (key.indexOf(this.KEY_PREFIX) == 0) key = key.substring(this.KEY_PREFIX.length);
            contents[key] = this.get(key);
            index++;
        }
        return contents;
    },

    getCacheKeys : function () {
        var index = 0,
            keys = [],
            count = this.getStoredValuesCount();
        while (index < count) {
            var key = this.getKeyForIndex(index);
            keys[keys.length] = key;
            index++;
        }
        return keys;
    },
    
    getKeyForIndex : function (index) {
        switch (this.localStorageType()) {
            case this.LOCAL_STORAGE:
                return localStorage.key(index);
                break;
            case this.GLOBAL_STORAGE:
            case this.DATABASE_STORAGE:
            case this.GEARS_DATABASE_API:
            case this.USERDATA_PERSISTENCE:
            case this.GOOGLE_GEARS:
                this.logError("Persistence method '" + this.localStorageType() + "' not yet supported");
                break;
        }
    },
    
    // The force param tells the function to remove an item if it appears like we didn't 
    // put it there - useful for if you really need the cache to be clear
    removeOldestEntry : function (force) {
        switch (this.localStorageType()) {
            case this.LOCAL_STORAGE:
                
                var index = 0;
                while (true) {
                    var key = localStorage.key(index);
                    if (key && key.substring(0, this.KEY_PREFIX.length) == this.KEY_PREFIX ||
                                        force) {
                        localStorage.removeItem(key);
                        break;
                    } else if (!key) {
                        break;
                    }
                }
                break;
            case this.GLOBAL_STORAGE:
            case this.DATABASE_STORAGE:
            case this.GEARS_DATABASE_API:
            case this.USERDATA_PERSISTENCE:
            case this.GOOGLE_GEARS:
                this.logError("Persistence method '" + this.localStorageType() + "' not yet supported");
                break;
        }
    },

    // Managing entries
    // ---------------------------------------------------------------------------------------
    put : function (key, value) {
        var ts = new Date().getTime();
        switch (this.localStorageType()) {
            case this.LOCAL_STORAGE:
                localStorage.setItem(this.KEY_PREFIX + key, value);
                break;
            case this.GLOBAL_STORAGE:
            //    globalStorage[''][key] = value;
            //    break;
            case this.DATABASE_STORAGE:
            //    this.database.executeSql(
            //        "INSERT INTO ResponseCache (key, value) " + 
            //        "('" + key + "', '" + value + "')");
            //    break;
            case this.GEARS_DATABASE_API:
            case this.USERDATA_PERSISTENCE:
            case this.GOOGLE_GEARS:
                this.logError("Persistence method '" + this.localStorageType() + "' not yet supported");
                break;
        }
        var end = new Date().getTime();
        this.logWarn("put() with key: " + key + 
                     "\nitem: " + this.echoLeaf(value) + 
                     ": " + (end - ts) + "ms");
    },
    
    get : function (key) {
        var ts = new Date().getTime(),
            item;
        switch (this.localStorageType()) {
            case this.LOCAL_STORAGE:
                var item = localStorage.getItem(this.KEY_PREFIX + key);
                break;
            case this.GLOBAL_STORAGE:
            //    return globalStorage[''][key];
            case this.DATABASE_STORAGE:
                // Not sure what's involved yet
            case this.GEARS_DATABASE_API:
            case this.USERDATA_PERSISTENCE:
            case this.GOOGLE_GEARS:
                this.logError("Persistence method '" + this.localStorageType() + "' not yet supported");
                break;
        }
        var end = new Date().getTime();
        this.logWarn("get() with key: " + key + 
                     "\nitem is: " + this.echoLeaf(item) +
                     ": " + (end - ts) + "ms");
        return item;
    },
    
    remove : function (key) {
        this.logWarn("Removing item for key: " + key);
        switch (this.localStorageType()) {
            case this.LOCAL_STORAGE:
                localStorage.removeItem(this.KEY_PREFIX + key);
                break;
            case this.GLOBAL_STORAGE:
            //    return globalStorage[''][key];
            case this.DATABASE_STORAGE:
                // Not sure what's involved yet
            case this.GEARS_DATABASE_API:
            case this.USERDATA_PERSISTENCE:
            case this.GOOGLE_GEARS:
                this.logError("Persistence method '" + this.localStorageType() + "' not yet supported");
                break;
        }
    },

    // DataSource functionality
    // ---------------------------------------------------------------------------------------
    storeResponse : function (dsRequest, dsResponse) {
        var ts = new Date().getTime();

        var trimmedRequest = this.trimRequest(dsRequest),
            key = this.serialize(trimmedRequest),
            value = this.serialize(this.trimResponse(dsResponse));

        this.logWarn("storeResponse serializing: " + (new Date().getTime() - ts) + "ms");

        // Unless we are already storing a response for this request (in which case we'll just
        // freshen it), check if we're about to bust the maximum responses limit
        if (this.get(key) == null) {
            if (this.getStoredValuesCount() >= this.maxResponsesToPersist) {
                this.removeOldestEntry();
            }
        }
        this.put(key, value);
    },
    
    trimRequest : function (dsRequest) {
        var keyProps = ["dataSource", "operationType", "operationId",
                        "criteria", "values", "sortBy", 
                        "startRow", "endRow"],
            trimmed = {},
            undef;
        for (var i = 0; i < keyProps.length; i++) {
            if (dsRequest[keyProps[i]] !== undef) {
                trimmed[keyProps[i]] = dsRequest[keyProps[i]];
            }
        }
        return trimmed;
    },
    trimResponse : function (dsResponse) {
        var keyProps = ["dataSource", 
                        "startRow", "endRow", "totalRows", 
                        "data", 
                        "status", "errors", // note we don't actually cache errors yet
                        "invalidateCache", "cacheTimestamp"],
            trimmed = {},
            undef;
        for (var i = 0; i < keyProps.length; i++) {
            if (dsResponse[keyProps[i]] !== undef) {
                trimmed[keyProps[i]] = dsResponse[keyProps[i]];
            }
        }
        return trimmed;
    },

    getResponse : function (dsRequest) {
        //!DONTOBFUSCATE
        var trimmedRequest = this.trimRequest(dsRequest),
            key = this.serialize(trimmedRequest),
            value = this.get(key),
            returnValue;
        //this.logWarn("Attempting to eval stored response: " + value);
        // using raw eval() because SmartClient might not be available
        eval('returnValue = ' + value);
        return returnValue;
    },
    serialize : function (obj) {
        return isc.Comm.serialize(obj, false);
    }
};

if (window.isc) {
    isc.defineClass("Offline").addClassProperties(Offline);
} else {

//>GWT_Standalone
isc.addProperties = function (objOne, objTwo) {
    for (var propName in objTwo) objOne[propName] = objTwo[propName];
}

isc.addProperties(isc.Offline, {
    // utilities
    // ---------------------------------------------------------------------------------------
    serialize : function (object) {
        return isc.OfflineJSONEncoder.encode(object);
    },
    logWarn : function (message) {
        if (console) console.log(message);
    },
    logError : function (message) {
        if (console) console.log(message);
        alert(message);
    },
    echoLeaf : function (obj) {
    	var output = "", undefined;
		if (obj === undefined) return "undef";
        try {
            if (typeof obj == "Array") {
    			output += "Array[" + obj.length + "]";
            } else if (typeof obj == "Date") {
    			output += "Date(" + obj.toShortDate() + ")";
            } else if (typeof obj == "Function") {
                output += isc.Func.getName(obj, true) + "()";
    		} else {
    			switch (typeof obj) {
    			case "string" : 
                    // for shorter strings show the whole thing.  Also, in "longMode" don't
                    // shorten.
                    if (obj.length <= 40) { 
                        output += '"' + obj + '"'; 
                        break;
                    }
    
                    // for long strings, show an elipsis and the strings full length
                    output += '"' + obj.substring(0, 40) + '..."[' + obj.length + ']';

                    // convert CR/LF to avoid spanning several lines
                    output = output.replaceAll("\n", "\\n").replaceAll("\r", "\\r");
                    break;
    			case "object" :
    				// typeof null is "object"
    				if (obj == null) { output += "null"; break; }
    
                    // DOM object
                    if (obj.tagName != null) {
                        output += "[" + obj.tagName + "Element]";
                        break;
                    }
    
    			    var toString = "" + obj;
    			    if (toString != "" && toString != "[object Object]" && 
                        toString != "[object]") 
                    {
                        // someone went through the trouble of making a better toString(), so
                        // use it.  NOTE: check for "" because in IE an XmlNodeList among
                        // others will toString() to ""
                        output += toString;
                        break;
                    }
    
    			    // return generic "Obj"
                    output += "Obj";
                
    				break;
    			default: output += "" + obj; // invoke native toString()
    			}
    		}
    		return output;
        } catch (e) {
            var message = "[Error in echoLeaf: " + e + "]";
            output += message;
            return output;
        }            
	},

    echo : function (object) { return this.serialize(object); }

});

// Serialization support
// ---------------------------------------------------------------------------------------
// This code is from the SmartClient serializer - I've just stolen it and refactored it to 
// strip out the class system, API docs and any dependency on other SmartClient facilities.  
// Where this wasn't possible (for example, the serializer depends heavily on "isA" functions),
// I have factored across the necessary from wherever it exists in SmartClient into this 
// class.  This seemed like the only quick way to obtain a robust, dependency-free serializer 
// that serializes according to what the Isomorphic server is expecting to see (in terms of 
// Dates, for example)
isc.OfflineJSONEncoder = {

_serialize_remember : function (objRefs, object, path) {
	objRefs.obj.add(object);
	objRefs.path.add(path);
},
_serialize_cleanNode : function (object) {
    var treeId = object["_isc_tree"];
    if (treeId != null) {
        var theTree = window[treeId];
        if (theTree && theTree.parentProperty && object[theTree.parentProperty]) {
            object = theTree.getCleanNodeData(object);
        }
    }
    return object;
},
_serialize_alreadyReferenced : function (objRefs, object) {
	var rowNum = objRefs.obj.indexOf(object);
	if (rowNum == -1) return null;
	return objRefs.path[rowNum];
},
_serialize_addToPath : function (objPath, newIdentifier) {
	if (isc.isA.Number(newIdentifier)) {
		return objPath + "[" + newIdentifier + "]";
	} else if (! isc.Comm._simpleIdentifierRE.test(newIdentifier)) {
		return objPath + '["' + newIdentifier + '"]';
	} else {
		return objPath + "." + newIdentifier;
	}
}, 

encode : function (object) {
    this.objRefs = {obj:[],path:[]};
    var retVal = this._serialize(object, this.prettyPrint ? "" : null , null);
    this.objRefs = null;
    return retVal
},

dateFormat: "xmlSchema",

encodeDate : function (date) {
    if (this.dateFormat == "dateConstructor") { 
        return "new Date(" + date.getTime() + ")";
    } else { // quotes for xml schema
        return '"' + this.toSchemaDate(date) + '"';    
    }
},

toSchemaDate : function (date) {
    var dd = "" + date.getDate(),
        mm = "" + (date.getMonth() + 1),
        yyyy = "" + (date.getyear() + 1900),
        hh = "" + date.getHours(),
        mi = "" + date.getMinutes(),
        ss = "" + date.getSeconds();
        
    dd = dd.length == 1 ? "0" + dd : dd;    
    mm = mm.length == 1 ? "0" + mm : mm;    
    hh = hh.length == 1 ? "0" + hh : ff;    
    mi = mi.length == 1 ? "0" + mi : mi;    
    ss = ss.length == 1 ? "0" + ss : ss;    
        
   return yyyy + "-" + mm + "-" + dd + "T" + hh + ":" + mi + ":" + ss;
},

strictQuoting: true,
circularReferenceMode: "path",
circularReferenceMarker: "$$BACKREF$$",
prettyPrint: false,

_serialize : function (object, prefix, objPath) {	
    if (!objPath) {
        if (object && object.getID) objPath = object.getID();
        else objPath = "";
    }
	
	if (object == null) return null;

	// handle simple types
    // In Safari a cross-frame scripting bug means that the 'asSource' method may not always be
    // available as an instance method.
    // call the static version of the same method if this happens.
	if (this.isAString(object)) return this.asSource(object);
	if (this.isAFunction(object)) return null;
	if (this.isANumber(object) || this.isASpecialNumber(object)) return object;
	if (this.isABoolean(object)) return object;
	if (this.isADate(object)) return this.encodeDate(object);

    // skip instances (this is a simplification of the root code)
    if (this.isAnInstance(object)) return null;
	
	var prevPath = this._serialize_alreadyReferenced(this.objRefs, object);
    
	if (prevPath != null && objPath.contains(prevPath)) {
        
        var nextChar = objPath.substring(prevPath.length, prevPath.length+1);
        if (nextChar == "." || nextChar == "[" || nextChar == "]") {
            var mode = this.circularReferenceMode;
            if (mode == "marker") {
                return "'" + this.circularReferenceMarker + "'";    
            } else if (mode == "path") {
                return  "'" + this.circularReferenceMarker + ":" + prevPath + "'";   
            } else {
                return null;    
            }
        }
	}

    if (this.isAClassObject(object)) return null;

    if (object == window) return null;

	this._serialize_remember(this.objRefs, object, objPath);
	
	// if there is a serialize method associated with this object, call that
    // (NOTE: Leaving this in, in case we're running in a SmartClinet context)
	if (this.isAFunction(object._serialize)) {
        return object._serialize(prefix, this.objRefs, objPath);
    }

	// handle arrays as a special case
	if (this.isAnArray(object))	{
        return this._serializeArray(object, objPath, this.objRefs, prefix);
    }

    var data;
	// if the object has a getSerializeableFields, use whatever it returns, otherwise just use the object
    if (object.getSerializeableFields) {
		data = object.getSerializeableFields([], []);
    } else {
        data = object;
    }
	// and return anything else as a simple object
	return this._serializeObject(data, objPath, this.objRefs, prefix);
},

_serializeArray : function (object, objPath, objRefs, prefix) {
	// Replaced references to the SmartClient StringBuffer class with native strings throughout
    // this method, rather than port that class's functionality. Also removed pretty print 
    // facilities - we don't need them for this
	var output = "[";
	// for each element in the array
	for (var i = 0, len = object.length; i < len; i++) {
		var value = object[i];
        if (prefix != null) output += "\r" + prefix;

        var objPath = this._serialize_addToPath(objPath, i);
        var serializedValue =  
            this._serialize(value, (prefix != null ? prefix : null), objPath);
        output += serializedValue + ",";
        if (prefix != null) output += " ";
	}
	// get rid of the trailing comma, if any
	var commaChar = output.lastIndexOf(",");
	if (commaChar > -1) output = output.substring(0, commaChar);

	// add the end array marker
    if (prefix != null) output += "\r" + prefix;
    output += "]";

	// and return the output
	return output;	
},

_serializeObject : function (object, objPath, objRefs, prefix) {
	// Replaced references to the SmartClient StringBuffer class with native strings throughout
    // this method, rather than port that class's functionality. Also removed pretty print 
    // facilities - we don't need them for this
	var output = "{",
        undef;

        
    object = this._serialize_cleanNode(object);

    // Not trying to support every edge case at this point - skipping code that copes with
    // XMLNodes here, rather than porting isc.echoLeaf
    try {
        
    	for (var key in object) break;
    } catch (e) {
        return null;
    }

	for (var key in object) {
		if (key == null) continue;
        
        // Sorry about this hack - attempting to serialize certain sub-properties of an XHR 
        // causes Firefox (maybe others) to crash when examining the property's constructor.
        // Not sure why - needs a proper Javascript guru...
        if (key == "xmlHttpRequest") continue;
        
        // skip internal properties, if the flag is set (simplified this code to remove 
        // framework calls)
		if (key.substring(0,1) == "_" || key.substring(0,1) == "$") continue;
		var value = object[key];

		if (this.isAFunction(value)) continue;

        // omit instances entirely if so configured (this code is simplified, we always 
        // skip instances)
        if (this.isAnInstance(value)) continue;

		// convert the key to a string
		var keyStr = key.toString();
		// and quote it
		keyStr = '"' + keyStr + '"';
    
        var objPath = this._serialize_addToPath(objPath, key);
        var serializedValue;
        // don't try to serialize references to GWT Java objects
        if (key != "__ref") {
            serializedValue = 
                this._serialize(value, (prefix != null ? prefix : null), objPath);
        }

		// now output the key : value pair
        if (prefix != null) output += "\r" + prefix;
		output += keyStr + ":" + serializedValue + ",";
    
        if (prefix != null) output.append(" ");
	}
	// get rid of the trailing comma, if any
	var commaChar = output.lastIndexOf(",");
	if (commaChar > -1) output = output.substring(0, commaChar);

	// add the end object marker
    if (prefix != null) output += "\r" + prefix;
    output += "}";

	// and return the output
	return output;
},
    
// These methods ripped off from SmartClient's IsA class
isAString : function (object) {
    if (object == null) return false;
    if (typeof object == this._$function) return false;
    if (object.constructor && object.constructor.__nativeType != null) {
        return object.constructor.__nativeType == 4;
    }
    if (object.Class != null && object.Class == this._$String) return true;

    return typeof object == "string";
},
isAnArray : function (object) {
    if (object == null) return false;
    if (typeof object == this._$function) return false;
    if (object.constructor && object.constructor.__nativeType != null) {
        return object.constructor.__nativeType == 2;
    }
    if (isc.Browser.isSafari) return ""+object.splice == "(Internal function)";
    return ""+object.constructor == ""+Array;
},
isAFunction : function (object) {
    if (object == null) return false;
    if (isc.Browser.isIE && typeof object == this._$function) return true;
    var cons = object.constructor;
    if (cons && cons.__nativeType != null) {
        if (cons.__nativeType != 1) return false;
        if (cons === Function) return true;
    }

    return isc.Browser.isIE ? (isc.emptyString+object.constructor == Function.toString()) : 
                              (typeof object == this._$function);
},

isANumber : function (object) {
    if (object == null) return false;
    if (object.constructor && object.constructor.__nativeType != null) {
        if (object.constructor.__nativeType != 5) return false;
    } else {
        if (typeof object != "number") return false;
    }
    // it's a number, now check if it's a valid number
    return !isNaN(object) && 
        object != Number.POSITIVE_INFINITY && 
        object != Number.NEGATIVE_INFINITY;
},
isASpecialNumber : function (object) {
    if (object == null) return false;
    if (object.constructor && object.constructor.__nativeType != null) {
        if (object.constructor.__nativeType != 5) return false;
    } else {
        if (typeof object != "number") return false;
    }
    return (isNaN(object) || object == Number.POSITIVE_INFINITY ||
            object == Number.NEGATIVE_INFINITY);
},
isABoolean	: function (object) {
    if (object == null) return false;
    if (object.constructor && object.constructor.__nativeType != null) {
        return object.constructor.__nativeType == 6;
    }
    return typeof object == "boolean";
},

isADate : function (object) {
    if (object == null) return false;
    if (object.constructor && object.constructor.__nativeType != null) {
        return object.constructor.__nativeType == 3;
    }
    return (""+object.constructor) == (""+Date) &&
            object.getDate && isc.isA.Number(object.getDate()) 
},
isAnXMLNode : function (object) {
    if (object == null) return false;
    if (isc.Browser.isIE) {
        return object.specified != null && object.parsed != null && 
               object.nodeType != null && object.hasChildNodes != null;
    }
    var doc = object.ownerDocument;
    if (doc == null) return false;
    return doc.contentType == this._$textXML;
},
isAnInstance : function (object) {	
    return (object != null && object._scPrototype != null)
},
isAClassObject : function (object) {
	return (object != null && object._isClassObject == true)
}, 
    
    
// This method ripped out of the SmartClient String enhancements
asSource : function (string, singleQuote) {
    if (!this.isAString(string)) string = ""+string;

    var quoteRegex = singleQuote ? String._singleQuoteRegex : String._doubleQuoteRegex,
        outerQuote = singleQuote ? "'" : '"';
    return outerQuote +
               string.replace(/\\/g, "\\\\")
                     // quote whichever quote we use on the outside
                     .replace(quoteRegex, '\\' + outerQuote)
                     .replace(/\t/g, "\\t")
                     .replace(/\r/g, "\\r")
                     .replace(/\n/g, "\\n") + outerQuote;
}              
    
}; // close offline serializer definition

//<GWT_Standalone

} // close !window.isc

})(); // close wrapper function and execute
