/*
 * Isomorphic SmartClient
 * Version SC_SNAPSHOT-2011-08-02 (2011-08-02)
 * Copyright(c) 1998 and beyond Isomorphic Software, Inc. All rights reserved.
 * "SmartClient" is a trademark of Isomorphic Software, Inc.
 *
 * licensing@smartclient.com
 *
 * http://smartclient.com/license
 */




//>	@class	ListGrid
// A ListGrid is a +link{DataBoundComponent} that displays a list of objects in a grid, where
// each row represents one object and each cell in the row represents one property. 
//
//  @implements DataBoundComponent    
//  @treeLocation Client Reference/Grids
//  @visibility external
//<
isc.ClassFactory.defineClass("ListGrid", "Canvas", "DataBoundComponent");

// Synonym for backCompat.  NOTE: define an alias rather than make a subclass, otherwise, attempts
// to skin the class using the old name would only affect the subclass!
isc.addGlobal("ListViewer", isc.ListGrid);
// define groups for documentation purposes

    //> @groupDef data
    //<

    //> @groupDef databinding
    // DataBinding means the automatic, highly customizable process of 'binding' a UI component
    // to a DataSource, so that a UI component displays, edits and saves DataSource records
    // using appropriate formatters, editors, validation rules, and persistence logic.
    // 
    // @see interface:DataBoundComponent
    // @title DataBinding
    //<

    //> @groupDef sorting
    //< 

    //> @groupDef editing
    // Data being displayed by a grid may be edited within the grid, by showing editing
    // interfaces embedded inside the cells of the grid.
    // <P>
    // <b>Enabling editing</b>
    // <P>
    // Editing is enabled when +link{listGrid.canEdit,canEdit} is <code>true</code>.  When enabled,
    // the user can begin editing via the 
    // +link{listGrid.editEvent,editEvent}, typically click or double-click.  Editing can also be triggered
    // programmatically by a call to +link{listGrid.startEditing,startEditing()} or
    // +link{listGrid.startEditingNew,startEditingNew()}.
    // <P>
    // <b>New record creation</b>
    // <P>
    // By default, editing is restricted to existing records.  Setting +link{listGrid.listEndEditAction} to
    // "next" allows the user to create new records by simply navigating off the end of the dataset
    // with the keyboard.  Editing of new records can also be initiated with 
    // +link{listGrid.startEditingNew()}, for example, from a button outside the grid.
    // <P>
    // <b>Saving changes</b>
    // <P>
    // Saving of changes is triggered automatically when the user navigates out of the row or cell
    // being edited (based on +link{listGrid.saveByCell}) or when the user ends editing.   For
    // a "mass update" interface, automatic saving of changes can be disabled entirely via 
    // +link{listGrid.autoSaveEdits,autoSaveEdits:false}, in which case a manual call to 
    // +link{listGrid.saveEdits,saveEdits()} or +link{listGrid.saveAllEdits,saveAllEdits()} is required
    // to trigger saving. 
    // <P>
    // If a grid has no DataSource, saving means that the properties of the +link{ListGridRecord}s
    // in +link{listGrid.data,grid.data} are directly changed.
    // <P>
    // For a grid with a DataSource, saving will be accomplished by using DataSource "update"
    // operations for existing records, and DataSource "add" operations for new records.  If multiple
    // records have been edited and +link{listGrid.saveAllEdits,saveAllEdits()} is called, 
    // +link{rpcManager.startQueue,request queuing} will be automatically used to enable all
    // edits to be saved in one HTTP turnaround (if using the SmartClient Server).
    // <P>
    // By default, a grid will send only updated fields and the primaryKey field as part of 
    // +link{dsRequest.data} so that the server can discern which fields the user actually changed.
    // However, the grid always includes the original field values in the
    // dsRequest as +link{dsRequest.oldValues}.
    // <P>
    // Note that although it is possible to load DataSource data without actually declaring a
    // +link{dataSourceField.primaryKey,primaryKey field}, a primaryKey must be declared for
    // editing and saving.  The primaryKey value is how SmartClient identifies the changed
    // record to the server.
    // <P>
    // <b>Validation</b>
    // <P>
    // Any time saving is attempted, validation is automatically triggered.  Values entered by the
    // user will be checked against the +link{listGridField.validators} and the
    // +link{dataSourceField.validators}. Any invalid values abort an attempted save.
    // <P>
    // Similar to editing and saving, validation can be done on row transitions or on cell
    // transitions by setting +link{listGrid.validateByCell,validateByCell}, or can be disabled entirely
    // via +link{listGrid.neverValidate,neverValidate:true}.
    // <P>
    // <b>Editability of cells</b>
    // <P>
    // Editors will either be shown for the complete row or for a single cell based on 
    // +link{listGrid,editByCell,editByCell}.  Whether a cell can be edited can be controlled on a
    // per field basis by setting +link{listGridField.canEdit,field.canEdit}, or on a per-record basis
    // by setting +link{listGrid.recordEditProperty,recordEditProperty} on a 
    // +link{ListGridRecord,record}, or can be controlled on an arbitrary, programmatic basis via
    // an override of +link{listGrid.canEditCell()}.
    // <P>
    // Cells which are not editable just display the cell's current value.
    // <P>
    // <b>Keyboard Navigation</b>
    // <P>
    // Full keyboard navigation is supported by default, including Tab and Shift-Tab to navigate
    // between cells in a row, and Up Arrow and Down Arrow to traverse rows.  Several properties
    // on both grids and fields, all named *EditAction, control navigation behavior of certain keys
    // (eg Enter).
    // <P>
    // You can use +link{listGrid.startEditing,startEditing(<i>rowNum</i>, <i>colNum</i>)} to 
    // programmatically move editing to a particular cell, for example, during a 
    // +link{listGridField.changed,field.changed()} event.
    // <P>
    // <b>editValues (unsaved changes)</b>
    // <P>
    // The term "editValues" means changes that the user has made to the dataset which have not
    // been saved.  The grid manages and stores editValues separately from the data itself in order
    // to allow the user to revert to original values, and in order to enable to grid to send only
    // updated fields to the server.  
    // <P>
    // Because editValues are stored separately, if you directly access the dataset (eg via 
    // <code>grid.data.get()</code>) you will see the records without the user's unsaved changes.
    // Many APIs exist for retrieving and managing editValues (search for editValue).
    // For the common case of needing to access the record-as-edited, you can call 
    // +link{listGrid.getEditedRecord,grid.getEditedRecord(rowNum)}.
    // <P>
    // When accessing and manipulating edited data, you should think carefully about whether
    // you want to be working with the original data or with the edited version.  Values entered
    // by the user may not have been validated yet, or may have failed validation, hence you may
    // find a String value in a field of type "date" or "int", which could cause naive formatters or
    // totaling functions to crash.
    // <P>
    // Setting editValues is fully equivalent to the user making changes to data via the editing UI.
    // If you <i>also</i> allow editing external to the grid, setting editValues is one way to 
    // combine changes from external editors into the grid's edits, so that you can do a single 
    // save.
    // <P>
    // <b>Customizing Cell Editors</b>
    // <P>
    // When a cell is being edited, the editor displayed in the cell will be a +link{class:FormItem}.
    // The editor type for the cell will be determined by +link{listGrid.getEditorType()} based on the
    // specified +link{ListGridField.editorType} or +link{ListGridField.type, data type} for the field in
    // question.
    // <P>
    // You can customize the editor by setting +link{listGridField.editorProperties} to a set of
    // properties that is valid for that FormItem type.  Custom FormItem classes are also allowed,
    // for example, you may use +link{formItem.icons} to create an icon that launches a separate
    // +link{Dialog} in order to provide an arbitrary interface that allows the user to select the
    // value for a field.
    // <P>
    // <b>Events</b>
    // <P>
    // Editing triggers several events which you can provide handlers for in order to customize
    // editing behavior.  Some of the most popular are +link{listGridField.change,field.change()},
    // +link{listGridField.changed,field.changed()} for detecting changes made by the user,
    // +link{listGrid.cellChanged()} for detecting changes that have been successfully saved,
    // and +link{listGrid.editorEnter()} and +link{listGrid.editorExit,editorExit()} for detecting user
    // navigation during editing.
    // <P>
    // You can also install event handlers directly on the FormItem-based editors used in the grid
    // via +link{listGridField.editorProperties,editorProperties} as mentioned above.  When handling
    // events on items, or which involve items, be aware that in addition to standard 
    // +link{FormItem} APIs, editors have the following properties:
    // <P>
    // - <code>rowNum</code>: The rowNum of the record being edited.<br>
    // - <code>colNum</code>: The colNum of the cell being edited.<br>
    // - <code>grid</code>: A pointer back to the listGrid containing the record.
    //
    // @title Grid Editing
    // @treeLocation Client Reference/Grids/ListGrid
    // @visibility external
    //<
    // Note: we also include a pointer to the record on the FormItem - don't expose this for now
    // as the user will be more likely to want to work with the result of 'getEditedRecord()' than
    // the underlying record values.

	//> @groupDef imageColumns
	// Columns that show images either as their only appearance or in addition to text.
	//<

    //> @groupDef formulaFields
    // Fields with values calculated from other fields in the grid.
    //<

isc.defineClass("GridBody", isc.GridRenderer).addProperties({
    // suppress adjustOverflow while pending a redraw so we can resize smaller without seeing 
    // a scrollbar flash in and out of existence
    adjustOverflowWhileDirty:false,
    
    // adjustOverflow() - overridden to support 'autoFitData' behavior
    adjustOverflow : function (reason, a,b,c,d) {
        // we call 'getDelta' from this method which can fall back through to 'adjustOverflow'
        // Avoid infinite looping if we hit this case.
        if (this._calculatingDelta) return;
        
        
        var grid = this.grid;
        
        
        if (grid == null) return this.Super("adjustOverflow", arguments);
        
        var data = grid.data, isLoading = false;;
            
        if (isc.isA.ResultSet(data) && !data.lengthIsKnown()) {
            if (grid.emptyMessageHeight == null) {
                return this.invokeSuper(isc.GridBody, "adjustOverflow", reason,a,b,c,d);
            }
            isLoading = true;
        }
        
        var fitVertical = (this.autoFitData == "both"),
            fitHorizontal = fitVertical,
            frozen = grid && grid.frozenFields != null,
            isFrozenBody = frozen && grid && (grid.frozenBody == this);
        if (!fitVertical) fitVertical = (this.autoFitData == "vertical");
        if (!fitHorizontal) fitHorizontal = (this.autoFitData == "horizontal");
        // If we have frozen fields, the frozen body never shows scrollbars and always
        // gets sized to match the widths of the fields it contains (done as part of
        // setBodyFieldWidths). Don't worry about trying to run special auto-fit logic
        // on the frozen body.
        // - We do run auto-fit logic on the unfrozen body and take the size of the frozen
        //   body into account when doing so.
        // - We do still need to ensure the header layout is sized correctly when the frozen
        //   body is resized
        
        if (fitHorizontal || fitVertical) {
            var height, width, rowHeights, hscrollOn, vscrollOn, dX, dY;
            
            if (fitVertical) {
                var minHeight = this.grid.getAutoFitMinBodyHeight();
                height = minHeight;
                var totalRows = isLoading ? 0 : this.getTotalRows(),
                    rows = totalRows;
                rowHeights = 0;
                // ignore autoFitMaxRecords if set to zero - this means fit to all records!
                if (this.autoFitMaxRecords) rows = Math.min(rows, this.autoFitMaxRecords);
                
                if (rows > 0) {
                    // We need to handle variable rowHeights so we're going to have to look at
                    // the table element to determine the heights - we already have a method to
                    // do that
                    var drawnRowHeights = this._getDrawnRowHeights();
                        
                    // If we have any undrawn rows assume calculated sizes
                    
                    var firstDrawnRow = this._firstDrawnRow,
                        lastDrawnRow = this._lastDrawnRow;
                    // fdr / ldr unset implies no drawn rows - set such that we calculate
                    // theoretical heights only
                    if (this._firstDrawnRow == null) {
                        firstDrawnRow = rows;
                        lastDrawnRow = rows;
                    }
                    if (firstDrawnRow > 0) {
                        firstDrawnRow = Math.min(firstDrawnRow, rows);
                        for (var i = 0; i < firstDrawnRow; i++) {
                            rowHeights += this.getRowHeight ? 
                                            this.getRowHeight(i) : this.cellHeight;
                        }                        
                    }
                    if (lastDrawnRow < rows-1) {
                        for (var i = lastDrawnRow+1; i < rows; i++) {
                            rowHeights += this.getRowHeight ? 
                                            this.getRowHeight(i) : this.cellHeight;
                        }
                    }
                    // Measure the rendered rows and add up the heights.
                    // Note that getDrawnRowHeights() just returns an array of the heights of
                    // rendered rows so the first drawn row is the first entry in the array, not
                    // the _firstDrawnRow'th entry
                    lastDrawnRow = Math.min(lastDrawnRow, rows-1);
                    for (var i = 0; i <= lastDrawnRow-firstDrawnRow; i++) {                        
                        rowHeights += drawnRowHeights[i];
                    }
                    // If we are clipping off any rows we know we have a v-scrollbar
                    vscrollOn = totalRows > rows;
                    
                    // Treat autoFitMaxHeight:0 as unspecified - resize as large as necessary
                    var autoFitMaxHeight = this.getAutoFitMaxHeight();
                    if (autoFitMaxHeight && rowHeights > autoFitMaxHeight) {
                        rowHeights = autoFitMaxHeight;
                        vscrollOn = true;
                    }
                    
                    //this.logWarn("total rows to show:"+ rows +
                    // ", rendered:" + [this._firstDrawnRow,this._lastDrawnRow] +
                    // ", rowHeights total up to:"+ rowHeights + 
                    // ", current height:" + this.getHeight() + 
                    // ", body height based on ListGrid specified height:" + height);
                    
                } else {
                    // The emptyMessage renders in the available space. If emptyMessageHeight
                    // is explicitly set, leave that much space for it.
                    
                    if (this.grid.emptyMessageHeight != null) {
                        rowHeights = this.grid.emptyMessageHeight;
                    }
                }
                 

                // add some extra height if autoFitExtraRecords is set
                
                if (this.autoFitExtraRecords && this.autoFitExtraRecords > 0) {
                    var extraHeight = Math.round(this.autoFitExtraRecords * this.cellHeight);
                    rowHeights += extraHeight;
                }
 
            } else {
                vscrollOn = this.getScrollHeight() > this.getHeight();
            }
  
            if (fitHorizontal && !isFrozenBody) {
                
                var width = this.grid.getInnerWidth(),
                    frozenBodyWidth;
                if (frozen) {
                  
                    var frozenWidths = this.grid.getFrozenSlots(this.grid._fieldWidths);
                    frozenBodyWidth = frozenWidths.sum();
                    width -= frozenBodyWidth;
                    
                    // if the frozenWidths exceed the specified width for the grid as a whole,
                    // apply an arbitrary small positive min width for the unfrozen body
                }
                
                  
                // Note that we're calling getColumnSizes on the GridRenderer
                // So if we the LG is frozen body this gives us the cols within the 
                // appropriate body, not the total set of cols in the grid.
                var colSizes = this.getColumnSizes(),
                    contentWidth = colSizes.sum();
                
                if (this.autoFitMaxColumns) {
                    var maxCols = this.autoFitMaxColumns;
                    // bit of a hack - how to deal with maxCols specified as a number <= the
                    // number of frozen fields.
                    // For now we just enforce at least one unfrozen field
                    if (frozen) {
                        maxCols = Math.max(1, maxCols-this.grid.frozenFields.length);
                    }
                    
                    if (maxCols < colSizes.length) {
                        colSizes = colSizes.slice(0,this.autoFitMaxColumns);
                    }
                }
                var colWidths = colSizes.sum();
                if (this.autoFitMaxWidth) {
                    var maxWidth = this.autoFitMaxWidth;
                    if (frozen) maxWidth = Math.max(20, maxWidth - frozenBodyWidth);
                    colWidths = Math.min(maxWidth, colWidths);
                }
                
                hscrollOn = (this.overflow == isc.Canvas.SCROLL) ? true : 
                            (this.overflow == isc.Canvas.AUTO) ? (contentWidth > Math.max(width, colWidths)) :
                            false;
                                
            } else {
                hscrollOn = this.overflow == isc.Canvas.SCROLL ? true :
                            this.overflow == isc.Canvas.AUTO  ? this.getScrollWidth() > this.getWidth() :
                            false;
            }
            // Now we know if we have an h-scrollbar, adjust height and width for scrollbars /
            // borders / margin if appropriate
            if (fitVertical && rowHeights != null) {
                
                rowHeights += this.getVBorderPad() + this.getVMarginSize();
                if (hscrollOn) {
                    rowHeights += this.getScrollbarSize();
                    var autoFitMaxHeight = this.getAutoFitMaxHeight()
                    if (autoFitMaxHeight && rowHeights > autoFitMaxHeight) {
                        rowHeights = autoFitMaxHeight;
                    }
                }
                // Resize vertically if rowHeights (+ border etc) > the auto fit min height
                // (which is derived from the ListGrid's specified height)
                if (rowHeights > height) {
                    height = rowHeights;
                    this._vAutoFit = true;
                } else {
                    if (this._vAutoFit) delete this._vAutoFit; 
                }
            }
            if (fitHorizontal && !isFrozenBody && colWidths != null) {
                
                colWidths += this.getHBorderPad() + this.getHMarginSize();
                
                if (vscrollOn || this.alwaysShowVScrollbar) {
                    colWidths += this.getScrollbarSize();
                    if (this.autoFitMaxWidth) {
                        var maxWidth = this.autoFitMaxWidth;
                        if (frozen) maxWidth = Math.max(20, maxWidth - frozenBodyWidth);
                        colWidths = Math.min(maxWidth, colWidths);
                    }
                }
                // Resize horizontally iff colWidths > width
                if (colWidths > width) {
                    width = colWidths;
                    this._hAutoFit = true;
                } else {
                    if (this._hAutoFit) delete this._hAutoFit;
                }
            }

            // Calculate the delta with our current size.
            this._calculatingDelta = true;
            dY = this.getDelta(this._$height, height, this.getHeight());            
            dX = this.getDelta(this._$width, width, this.getWidth());
            delete this._calculatingDelta;
            
            // If necessary resize to accommodate content!
            if (dY != null || dX != null) this.resizeBy(dX, dY, null, null, true);
            
            // if width change != null, resize header to match body
            // Note that if isFrozenBody is true we skipped the dX calculation so
            // always resize the headerLayout to match
            if (dX != null || (isFrozenBody && fitHorizontal)) {
                var lg = this.grid,
                    headerWidth = (width - (vscrollOn ? lg.getScrollbarSize() : 0)),
                    totalHeaderWidth = headerWidth;
                if (frozen && lg.headerLayout) {
                    
                    if (isFrozenBody) {
                        totalHeaderWidth = this.getWidth() + lg.body.getWidth();
                        // If we go past the autoFitMaxWidth limit, run adjustOverflow on the body
                        // to force it to shrink/start scrolling
                        if (lg.autoFitMaxWidth != null && 
                            (totalHeaderWidth + lg.getHBorderPad() + 
                                lg.getHMarginSize() > lg.autoFitMaxWidth)) 
                        {
                            // don't bother to go on and resize the header - we'll do that
                            // when the body adjust overflow method runs
                            return lg.body.adjustOverflow();
                        }
                        totalHeaderWidth -= (lg.body.vscrollOn ? lg.getScrollbarSize() : 0);
                        
                    } else {
                        totalHeaderWidth = headerWidth + lg.frozenBody.getWidth();
                    }
                    lg.headerLayout.setWidth(totalHeaderWidth);
                }
                
                // We can skip resizing the frozen header - this is handled in setBodyFieldWidths
                if (!isFrozenBody) {
                    var header = lg.header;
    
                    if (header && header.isDrawn()) {
                        header.setWidth(headerWidth);
                    }
                }
                if (lg.sorter && lg._showSortButton() && !lg.isRTL()) {
                    lg.sorter.moveTo(totalHeaderWidth);
                    if (!lg.sorter.isVisible()) lg.sorter.show();
                } else if (lg.sorter) {
                    lg.sorter.setLeft(0 - lg.sorter.getWidth());
                }
            }
        
        // if autoFitData is null but we don't match our 'specified size', assume the property
        // has been modified and reset to specified size
        }
        
        // catch the case where autoFitData has been cleared in either direction and
        // reset to specified size.
        var verticalChanged = (!fitVertical && this._vAutoFit),
            horizontalChanged = (!fitHorizontal && this._hAutoFit);
        if (verticalChanged || horizontalChanged) {
            delete this._vAutoFit;
            delete this._hAutoFit;
            
            var standardHeight = verticalChanged ? this.grid.getAutoFitMinBodyHeight() : null,
                standardWidth = horizontalChanged ? 
                                (!frozen ? this.grid.getInnerWidth() : 
                                    (this.grid.getInnerWidth() - this.grid.frozenBody.getWidth()) )
                                                  : null;
            this.resizeTo(width,height);
            // reset field widths on the grid to resize the header to match the body
            this.grid._updateFieldWidths("autoFitData mode changed");
        }
        return this.invokeSuper(isc.GridBody, "adjustOverflow", reason, a,b,c,d);
    },
    
    getAutoFitMaxHeight : function () {
        return this.grid ? this.grid.getAutoFitMaxBodyHeight() : null;
    },
    
    resizeBy : function (deltaX, deltaY, animating, suppressHandleUpdate, autoFitSize) {
        
        // autoFitSize parameter: When autoFitData is true for this grid, we resize the
        // body to fit the data, and pass in the autoFitSize parameter to this method.
        // In the case of an explicit resize outside the autoFitData system, hang onto the
        // specified size so we can reset to it if the data shrinks, etc
        if (!autoFitSize) {
            this._specifiedWidth = this.getWidth() + (deltaX != null ? deltaX : 0);
        }

        // Note that return value of resizeBy indicates whether the size actually changed
        return this.invokeSuper(isc.GridBody, "resizeBy", 
                                deltaX, deltaY, animating, suppressHandleUpdate, autoFitSize);
    },
    
	// context menus (NOTE: ListGrid-level handling is identical for cell vs row context click)
    cellContextClick : function (record, rowNum, colNum) {
        var gridColNum = this.grid.getFieldNumFromLocal(colNum, this);
        return this.grid._cellContextClick(record, rowNum, gridColNum);
    },
    
    // this ensures that if we're not showing any records we can still scroll the header fields
    // into view.
    expandEmptyMessageToMatchFields:true,
    
    getInnerHTML : function () {
        // call bodyDrawing on the LG if we are the primary body
        this.grid.bodyDrawing(this);
        return this.Super("getInnerHTML", arguments);
    },
    
    // ------------------------------------------------------
    //
    // PrintHTML
    // This needs some tweaking to handle the following:
    // - printHTML can be generated asynchronously in 2 ways:
    //  - if number of rows exceeds printMaxRows we use timers to break up the HTML generation
    //  - if we have embeddedComponents fetching their printHTML may also be asynchronous
    //
    // In either case, 'getTableHTML()' will be fired more than once, asynchronously.
    // In the case of async embedded component printHTML generation, this is the standard
    // mechanism - see 'gotComponentPrintHTML' in GridRenderer.
    // In the case of splitting the printing into chunks, the _printingChunk
    // flag will be set and startRow/endRow will be shifted, then getTableHTML will be called
    // on a timer, repeatedly until all the rows' HTML is generated.
    //
    // We need to fire the 'prepareForPrinting' and 'donePrinting' methods on the ListGrid
    // around each of these blocks - this is required as the ListGrid relies on the body to
    // handle generating header HTML and if there are frozen fields, HTML from the frozen
    // body, and does so by setting various flags on the GR body which'll be read by
    // getTableHTML()
    //
    
    getTablePrintHTML : function (context) {
        // context contains startRow, endRow, callback, printProperties, printWidths
        var startRow = context.startRow,
            endRow = context.endRow,
            totalRows = endRow != null ? (endRow - startRow) : this.getTotalRows(),
            maxRows = this.printMaxRows,
            printWidths = context.printWidths,
            printProps = context.printProps;
            
        var asyncPrintCallback = {
            target:this,
            methodName:"gotTablePrintHTML",
            printContext:context,
            printCallback:context.callback
        }
        
        context.callback = asyncPrintCallback;
        
        if (maxRows < totalRows) {
            this.logDebug("get table print html - breaking HTML into chunks", "printing");
            if (startRow == null) startRow = context.startRow = 0;
            if (endRow == null) endRow = context.endRow = this.getTotalRows();
            this.getPrintHTMLChunk(context);
            
            return null;
        }
        
        // No chunks - can only be asynchronous due to getTableHTML directly going asynch
        // to get embeddedComponentHTML
        var suspendPrintingContext = this.grid._prepareForPrinting(printWidths, printProps);
        var printHTML = this.getTableHTML(null, startRow, endRow, null, asyncPrintCallback);
       
        // restore settings
        this.grid._donePrinting(suspendPrintingContext);
        return printHTML;
    },
    
    gotTablePrintHTML : function (HTML, asyncCallback) {
        var callback = asyncCallback.printCallback;
        if (callback) {
            this.fireCallback(callback, "HTML,callback", [HTML,callback]);
        }
    },
    
    // This is called repeatedly, asynchronously for each "chunk"
    // The first chunk may include fetches for component tableHTML so can also be asynchonous
    // itself.
    getPrintHTMLChunk : function (context) {
        
        var suspendPrintingContext = this.grid._prepareForPrinting(context.printWidths);
        // printing chunk flag - used by the GR to avoid writing out the outer table tags for each
        // chunk.
        this._printingChunk = true;
        
        // Second flag to indicate we are printing chunks. This is used only by
        // gotComponentPrintHTML() to reset the _printingChunk flag before calling
        // getTableHTML
        this._gettingPrintChunkHTML = true;
        
        var startRow = context.startRow, 
            endRow = context.endRow, 
            maxRows = this.printMaxRows, 
            callback = context.callback;
    
        this.currentPrintProperties = context.printProps;
            
        if (!context.html) context.html = [];
    
        var chunkEndRow = context.chunkEndRow = Math.min(endRow, (startRow + maxRows)),
            chunkHTML = this.getTableHTML(null, startRow, chunkEndRow, null, 
                {target:this, methodName:"gotPrintChunkHTML",
                    printContext:context, printCallback:context.callback
                });
            
        // restore settings
        this.grid._donePrinting(suspendPrintingContext);
        this._printingChunk = false;
        
        // chunkHTML will only be null if getTableHTML went asynchronous - can happen on the
        // first chunk while retrieving embedded componentHTML
        if (chunkHTML != null) {
            delete this._gettingPrintChunkHTML;
            this.gotPrintChunkHTML(chunkHTML, {printContext:context});
        }
    },
    gotPrintChunkHTML : function (HTML, callback) {
        var context = callback.printContext,
            startRow = context.startRow,
            endRow = context.endRow,
            chunkEndRow = context.chunkEndRow,
            maxRows = this.printMaxRows,
            gotHTMLCallback = context.callback;
            
        context.html.add(HTML);        
        
        if (chunkEndRow < endRow) {
            context.startRow = chunkEndRow;
            return this.delayCall("getPrintHTMLChunk", [context], 0);
        }
    
        if (gotHTMLCallback != null) {
            var html = context.html.join(isc.emptyString);
            this.fireCallback(gotHTMLCallback, "HTML,callback", [html,gotHTMLCallback]);
        }
    },
    
    // In GridRenderer.getTableHTML(), when printing, we generate all embedded components'
    // print HTML up front, then slot it into the actual HTML for the table.
    // component printHTML may be asynchronously generated in which case this callback is
    // fired when we have the component HTML - default implementation re-runs getTableHTML
    // which now recognizes it's got component HTML and continues to get the actual table
    // HTML then fire the async callback.
    // Overridden to call 'prepareForPrinting()' on the grid and reset the '_printingChunk'
    // flag if necessary
    gotComponentPrintHTML : function (HTML, callback) {
        
        var asyncCallback = callback.context.asyncCallback,
            context = asyncCallback.printContext;
            
        var printWidths = context.printWidths;
            
        var suspendPrintingContext = this.grid._prepareForPrinting(printWidths);
        if (this._gettingPrintChunkHTML) {
            this._printingChunk = true;
        }
        
        var HTML = this.Super("gotComponentPrintHTML", arguments);
        if (this._printingChunk) delete this._printingChunk;
        
        if (HTML != null) {
            delete this._gettingPrintChunkHTML;
        } else {
            this.grid._donePrinting(suspendPrintingContext);
        }
        
    },
    
    // Cell Alignment
    // ---------------------------------------------------------------------------------------
    
    // cellAlignment - override to account for the fact that with frozen fields, body 
    // colNum may be offset from ListGrid colNum
    getCellVAlign : function (record, field, rowNum, colNum) {
        if (this.grid && this.grid.getCellVAlign) {
            var gridColNum = this.grid.getFieldNumFromLocal(colNum, this);
            return this.grid.getCellVAlign(record, rowNum, gridColNum);
        }
    },
    getCellAlign : function (record, field, rowNum, colNum) {
        
        if (this.grid && this.grid.getCellAlign != null) {
            var gridColNum = this.grid.getFieldNumFromLocal(colNum, this);
            return this.grid.getCellAlign(record, rowNum, gridColNum);
            
        } else return field.cellAlign || field.align;
    },
 
    // Single Cell rows   
    // ---------------------------------------------------------------------------------------

    // if this is removed, DONTCOMBINE directive no longer needed in GridRenderer.js
    _drawRecordAsSingleCell : function (rowNum, record,c) {
        var lg = this.grid;
        if (lg.showNewRecordRow && lg._isNewRecordRow(rowNum)) return true;
        
        return isc.GridRenderer._instancePrototype.
            _drawRecordAsSingleCell.call(this, rowNum,record,c);    
        //return this.Super("_drawRecordAsSingleCell", arguments);
    },
    
    // showSingleCellCheckboxField()
    // If this record is showing a single cell value, should a checkbox field also show up next
    // to the record?
    showSingleCellCheckboxField : function (record) {
        var lg = this.grid;
        return lg && lg.showSingleCellCheckboxField(record);
    },
    
    // This method is called on records where _drawRecordAsSingleCell is true
    // returns the start/end col the single cell value should span.
    // Typically just spans all the cells we render out but if we're showing the
    // checkbox field we may want to NOT span over that field
    _getSingleCellSpan : function (record, rowNum, startCol, endCol) {
        // Span all columns if we're not showing a checkbox field
        
        if (rowNum == this._animatedShowStartRow ||
            !this.showSingleCellCheckboxField(record) || 
            (this.grid && this.grid.frozenBody != null && this.grid.frozenBody != this)) 
        {
            return [startCol,endCol];
        }
        
        
        return [Math.max(startCol, 1), endCol];
    },
    
    // Scrolling / Scroll Sync
    // ---------------------------------------------------------------------------------------

    // Have v-scrolling occur on the frozen body on mouseWheel
    // This essentially duplicates the mouseWheel handler at the Canvas level for
    // widgets with visible scrollbars.
    mouseWheel : function () {
        if (this.frozen && this.grid != null) {
            var wheelDelta = this.ns.EH.lastEvent.wheelDelta;
            var scrollTo = this.scrollTop + Math.round(wheelDelta * isc.Canvas.scrollWheelDelta);
            // Scroll the main body (we'll scroll in response to that) rather than
            // scrolling the frozen body directly.
            this.grid.body.scrollTo(null, scrollTo, "frozenMouseWheel");
            return false;
        }
        return this.Super("mouseWheel", arguments);
    },

    // Override _getDrawRows()
    // Have the frozen body rely on the unfrozen body to handle drawAhead / quickDrawAhead
    // etc and keep set of drawn rows in synch
    
    _getDrawRows : function () {
        if (this.frozen && this.grid) {
            var grid = this.grid;
            return grid.body._getDrawRows();
        }
        return this.Super("_getDrawRows", arguments);
    },

    // doneFastScrolling: ensure *both* bodies redraw without draw-ahead direction
    doneFastScrolling : function () {
        // we only expect to see this fire on the unfrozen body - the frozen body doesn't
        // show a scrollbar so won't get the thumb drag which initializes this method
        if (!this.frozen && this.grid != null && this.grid.frozenBody != null) {
            
            var redrawFrozenBody = this._appliedQuickDrawAhead;
            this.Super("doneFastScrolling", arguments);
            if (redrawFrozenBody) {
                this.grid.frozenBody._suppressDrawAheadDirection = true;
                this.grid.frozenBody.markForRedraw("Done fast scrolling on unfrozen body");
            }
        }
    },

    // observe the scroll routine of the body so we can sync up
    scrollTo : function (left, top, reason, animating) {
        if (isc._traceMarkers) arguments.__this = this;
        // Clamp the positions passed in to the edges of the viewport
        // (avoids the header from getting out of synch with the body.)
        
        if (left != null) {
            var maxScrollLeft = this.getScrollWidth() - this.getViewportWidth();
            left = Math.max(0, Math.min(maxScrollLeft, left));
        }
        if (top != null) {
            var maxScrollTop = this.getScrollHeight() - this.getViewportHeight();
            top = Math.max(0, Math.min(maxScrollTop, top));
        }            
        var lg = this.grid;

        //this.logWarn("body.scrollTo: " + this.getStackTrace());
        // dontReport when we're being called in response to bodyScrolled
        // observation!
        var dontReport = this._noScrollObservation;
        if (!dontReport) lg.bodyScrolled(left, top, this.frozen);
        
        
        this.invokeSuper(null, "scrollTo", left,top,reason,animating);
        
        // If the body scrolled without forcing a redraw, ensure any visible edit form 
        // items are notified that they have moved.
        
        if (!this.isDirty() && lg._editorShowing) {
            lg._editRowForm.itemsMoved();
        }

    },
    
    // Embedded Components
    // ---------------------------------------------------------------------------------------

    // embedded components can be per row or per cell.
    // When per-cell the GR APIs act by colNum only, not by field name.
    // However for us to handle field reorder, show/hide, etc it's useful to hang fieldName
    // onto the embeddedComponents as well
    addEmbeddedComponent : function (component, record, rowNum, colNum, position) {
        var comp = this.invokeSuper(isc.GridBody, "addEmbeddedComponent", component, record, 
                                    rowNum, colNum, position);
        if (component._currentColNum != null && component._currentColNum != -1) {
            var colNum = component._currentColNum;
            component._currentFieldName = this.fields[colNum].name;
        }
        return component;
    },
    
    // Override _getExtraEmbeddedComponentHeight() / upateHeightForEmbeddedComponents to
    // respect listGrid.recordComponentHeight if specified, even if there are no
    // embedded components for this record.
    
    updateHeightForEmbeddedComponents : function (record, rowNum, height) {
  
        if (record && !record._embeddedComponents && this.grid.showRecordComponents
            && this.grid.recordComponentHeight != null) 
        {
            // Reimplementing the superClass version, except that this logic is running even
            // when there are no embeddedComponents on the row.
            var details = this._getExtraEmbeddedComponentHeight(record, rowNum);
            if (details.allWithin && details.extraHeight > 0) {
                height = Math.max(height,details.extraHeight);
                //this.logWarn("in updateHeightForEmbeddedComponents ("+this.grid+"): details are "+isc.echoAll(details)+"\nheight is "+height);
            } else {
                height += details.extraHeight;
                //this.logWarn("in updateHeightForEmbeddedComponents ("+this.grid+"): details are "+isc.echoAll(details)+"\nheight is "+height);
            }
            
            return height;
        }
        
        return this.invokeSuper(isc.GridBody, "updateHeightForEmbeddedComponents", record, rowNum, height);
    },
    
    _getExtraEmbeddedComponentHeight : function (record, rowNum) {
        var heightConfig = this.invokeSuper(isc.GridBody, "_getExtraEmbeddedComponentHeight",
                                        record, rowNum);
        if (this.grid.showRecordComponents && this.grid.recordComponentHeight != null) {
            heightConfig.extraHeight = Math.max(heightConfig.extraHeight,
                                            this.grid.recordComponentHeight);
        }
        return heightConfig;
    },
    _writeEmbeddedComponentSpacer : function (record) {
        if (record && this.grid && this.grid.showRecordComponents 
            && this.grid.recordComponentHeight != null)
        {
            return true;
        }
        return this.invokeSuper(isc.GridBody, "_writeEmbeddedComponentSpacer", record);
    },

    
    getAvgRowHeight : function () {
        if (this.grid) return this.grid.getAvgRowHeight(this);
        return this.Super("getAvgRowHeight", arguments);
    },
    
    // override shouldShowAllColumns() - we can avoid showing all columns if row height
    // is variable *only* because of an expansion component expanding the entire row since
    // the heights won't vary per-cell.
    shouldShowAllColumns : function () {
        if (this.showAllColumns) {
            return true;
        }
        if (!this.fixedRowHeights && !this.showAllRows) {
            if (this.grid.canExpandRecords && this.grid._specifiedFixedRecordHeights) {
                return false;
            }
            return true;
        }
        if (this.overflow == isc.Canvas.VISIBLE) {
            return true;
        }
        return false;
         
    },

    // Editing
    // ---------------------------------------------------------------------------------------

    //> @method listGrid.markForRedraw()
    // @include canvas.markForRedraw
    // @visibility external
    //<
    
    // Redraw overridden:
	// - Update the editRow form items (we don't create more items than we need when
	//   rendering incrementally)
	// - Update the values in the edit row form.
    redraw : function (reason,b,c,d) {
        // flag to note we're redrawing - this is used by getDrawnFields() 
        this._redrawing = true;
        
        // If alwaysShowEditors is marked as true, but editorShowing is false it implies our
        // attempt to start editing on draw() failed - presumably there were no
        // editable cells in view.
        // See if we can start editing now in this case
        var lg = this.grid;
        if (lg.alwaysShowEditors && !lg._editorShowing) {
            lg.startEditing(null,null,true,null,true);
        }
        
        
        var editForm = lg._editRowForm,
            editing = lg._editorShowing,
            editColNum, editRowNum, editRecord,
            completeWidths,
            fieldsToRemove;
        
        // If the grid is showing inactive Editor HTML for any cells, we'll clear it
        // (and potentially regenerate it) as part of redraw(). Notify the grid so it can clear
        // up inactive contexts stored on the edit form items
        
        lg._clearingInactiveEditorHTML();
        
        // if body redraw came from data change, folder opening, or resizing of content,
        // it's likely to introduce a v-scrollbar.
        // If leaveScrollbarGap is false, call '_updateFieldWidths()' before the redraw occurs so
        // we leave a gap for the v-scrollbar, rather than redrawing with both V and H scrollbar,
        // then resizing the fields and redrawing without an H-scrollbar.
        if (!lg.leaveScrollbarGap && lg.predictScrollbarGap && (this.overflow == isc.Canvas.AUTO)) {
            var vScrollOn = this.vscrollOn,
                
                vScrollWillBeOn = !lg.isEmpty() && 
                                  (lg.getTotalRows() * lg.cellHeight)  > this.getInnerHeight();
            
            if (vScrollOn != vScrollWillBeOn) {
                // ensure we don't try to recalculate field widths a second time by clearing
                // the _fieldWidthsDirty flag
                delete this._fieldWidthsDirty;
                lg._updateFieldWidths("body redrawing with changed vertical scroll-state");
                
            }
        } 

        var nativeRowRefocus = false;

        if (editing) {
            
            this.logInfo("redraw with editors showing, editForm.hasFocus: " +
                         editForm.hasFocus, "gridEdit");
            editColNum = lg.getEditCol();                             
            
            // See comments near _storeFocusForRedraw() for how edit form item focus is handled
            // on redraw
            this._storeFocusForRedraw();

            // This will add the new edit items corresponding to the newly displayed fields
            // and return the items that need to be removed (after the body is actually redrawn,
            // which will hide them)
            // It also fires the "redrawing" notification on any items that are about to be redrawn
            fieldsToRemove = this._updateEditItems();
        } else if (isc.screenReader) {
            nativeRowRefocus = true;
        }   
        
        // refresh field widths if necessary
        if (this._fieldWidthsDirty != null) {
            
            delete this._fieldWidthsDirty;
            lg._updateFieldWidths(this._fieldWidthsDirty);
        }
        // store the new drawArea
        var newDrawArea = this.getDrawArea();
        
        
        var grid = this.grid,
            drawArea = this._oldDrawArea;

        if (!drawArea) drawArea = this._oldDrawArea = [0,0,0,0];

        var grid = this.grid,
            firstRecord = grid.getRecord(newDrawArea[0]),
            lastRecord = grid.getRecord(newDrawArea[1]),
            dataPresent = (firstRecord && firstRecord != Array.LOADING) 
                && (lastRecord && lastRecord != Array.LOADING);
        ;
        if (dataPresent && !drawArea.equals(newDrawArea)) 
        {
            // the old and new drawAreas differ and the extents of the new data are present - 
            // fire the notification method and update the stored _oldDrawArea
            
            if (!this.frozen) { 
                grid._drawAreaChanged(drawArea[0], drawArea[1], drawArea[2], drawArea[3], this);
                this._oldDrawArea = newDrawArea;
            }
        }
        
        // Always update all recordComponents on redraw().
        // don't rely on the draw area changing since we may be showing the same set of
        // rowNum/colNums but having underlying data or field meaning changes.
        // Note: updateRecordComponents() updates components in frozen and unfrozen bodies.
        // If this is a redraw of the frozen body, don't call updateRecordComponents() if
        // the redraw was tripped by scrolling or data change as in this case we'll also
        // get a redraw of the unfrozen body which can handle updating the RC's.
        // (DO still call the method in other cases as it may imply the fields in the frozen
        // body have changed, etc).
        if (!(this.frozen && (reason == this._$dataChanged || reason == "scrolled"))) {
            grid.updateRecordComponents();
        }
        

        this.invokeSuper(null, "redraw", reason,b,c,d);

        // clear the "redrawing" flag since the HTML is now up to date
        delete this._redrawing;

        if (editing) {
            // Remove the items that correspond to fields that are no longer rendered in the DOM
            if (fieldsToRemove != null && fieldsToRemove.length > 0) {
                editForm.removeItems(fieldsToRemove);
            }
            
            // Fire the method to notify form items that they have been drawn() / redrawn()
            // or cleared()
            
            lg._editItemsDrawingNotification(null, true, this);
            
            /*
            var itemColArray = [],
                items = lg._editRowForm.getItems();
            for (var i =0; i < items.length; i++) {
                itemColArray.add(items[i].colNum + " - " + items[i].name);
            }
            this.logWarn("After redraw - edit form covers these cols:" + itemColArray);
            */

        	
            lg.updateEditRow(lg.getEditRow());           
            
            // If the editRowForm is currently marked as having focus, or we took focus from it
            // before a redraw and no other widget has subsequently picked up focus, restore focus
            // to it
            // This catches both a simple redraw and the case where the user scrolls the edit item
            // out of view, it is cleared due to incremental rendering, then scrolls back into view
            // and it gets redrawn
        	
            if (editForm.hasFocus || 
                (this._editorSelection && isc.EH.getFocusCanvas() == null)) 
            {
                this._restoreFocusAfterRedraw(editColNum);
            } else {
                delete this._editorSelection;
            }
            
        } else if (editForm != null) {
            // notify the form that it's items have been cleared() (will no-op if they're
            // not currently drawn)
            lg._editItemsDrawingNotification(null, null, this);
        } else if (isc.screenReader && this.hasFocus && nativeRowRefocus) {
            // _nativeFocusRow was remembered last time putNativeFocusInRow was called
            this._putNativeFocusInRow(this._nativeFocusRow);
        }
    },


    // force redraw on setDisabled() if we're showing an edit form to ensure we 
    // redraw the items in enabled/disabled state
    setHandleDisabled : function (disabled) {
        var lg = this.grid;
        if (this.isDrawn() && lg && lg._editorShowing) {
            this.markForRedraw("Grid body disabled while editing");
        }
        return this.Super("setHandleDisabled", arguments);
    },
    
    // Add edit items corresponding to newly displayed fields (displayed due to incremental
    // rendering)
    // If any fields are to be hidden, do not remove these here, but return them in an array so
    // they can be removed from the form after the redraw completes
    // Note that the order of items in the form will not match the order of fields necessarily - 
    // acceptable since developers will be interacting with the items' colNum attribute rather than
    // index in the edit form fields array.
    
    _updateEditItems : function () {
    
        // We keep the set of items in the editForm in synch with the set of 
        // visible columns for performance.
        // Determine which items need to be created or removed here.
        var lg = this.grid, editForm = lg.getEditForm(),
            fieldsToRemove = [],
            editItems = editForm.getItems();
        if (!lg.editByCell) {

            // set up the vars used in creating form items
            var editRowNum = lg.getEditRow(),
                editRecord = lg.getRecord(editRowNum),
                
                completeWidths = lg.getEditFormItemFieldWidths(editRecord);
         
            // Determine what fields are rendered into the body
            // If we have frozen columns, we will always be showing them, in addition to whatever
            // fields are visible
            var editItems = editForm.getItems(),
                itemNames = editItems.getProperty(this.fieldIdProperty),
                fields = lg.getDrawnFields(),
                fieldNames = fields.getProperty(this.fieldIdProperty);
            
            // minor optimization - if possible, avoid iterating through both arrays
            var lengthsMatch = editItems.length == fields.length,
                changed = false;
                
            // fields that are no longer drawn should be removed
            for (var i = 0; i < editItems.length; i++) {
                // don't actually remove the items until they have been removed from the DOM via
                // redraw
                var index = fieldNames.indexOf(itemNames[i]);
                if (index == -1) {
                    changed = true;
                    fieldsToRemove.add(editItems[i]);
                } else {
                    // If we're keeping the item, just update width, and notify the item we're
                    // about to redraw
                    editItems[i].width = completeWidths[editItems[i].colNum];
                    editItems[i]._size = null;
                    editItems[i].redrawing();
                }
            }
            // newly rendered fields should be added
            if (!lengthsMatch || changed) {      
                var editedVals = lg.getEditedRecord(editRowNum, 0);
                for (var i = 0; i < fields.length; i++) {                    
                    if (!itemNames.contains(fieldNames[i])) {

                        var colNum = lg.fields.indexOf(fields[i]);
                        var item = lg.getEditItem(
                                        fields[i],
                                        editRecord, editedVals, editRowNum,
                                        colNum, completeWidths[colNum]
                                   );
                        editForm.addItem(item);
                    }
                }
            }
        } 
        // if editByCell is true this is not necessary - we consistantly have the editForm contain
        // only the necessary cell
        return fieldsToRemove;
    },  

    // _storeFocusForRedraw()
    // called when the edit form is showing and the body is being redrawn.
    // remember the current focus state / selection of the edit form so we can reset it after
    // redrawing the item in the DOM
    // blur the item (suppressing the handler if the item will be refocused after redraw) 


    
    
        
    _storeFocusForRedraw : function () {

        var lg = this.grid,
            editForm = lg.getEditForm(),
            editColNum = lg.getEditCol();
        if (editForm.hasFocus) {
            var focusItem = editForm.getFocusSubItem();
            if (focusItem) {
                focusItem.updateValue();
                var origFocusItem = focusItem;
                
                // We may be focused in a sub item, in which case we need to use the
                // parentItem to get the field name wrt our fields array
                while (focusItem.parentItem != null) {
                    focusItem = focusItem.parentItem;
                }
                
                // blur the focus item before removing it from the DOM.
                // If canEditCell for the current focus item returns false, we will
                // not redisplay it at the end of this method, so allow it to fire the
                // standard blur-handler
                
                if (!lg.canEditCell(focusItem.rowNum, focusItem.colNum) ||
                    editColNum != focusItem.colNum) {
                    
                    editForm.blur();
                } else {
                    
                    if (focusItem.hasFocus) {
                        // remember the current selection, so we can reset it after the redraw
                        // and refocus. [will have no effect if the item is not a text-item]
                        focusItem.rememberSelection();
                        this._editorSelection = 
                            [focusItem._lastSelectionStart, focusItem._lastSelectionEnd];
                    }
                    editForm._blurFocusItemWithoutHandler();

                }
                
            }
        }
        
        editForm._setValuesPending = true;        
    },
    
    // If the editForm is visible during a body redraw() this method ensures that after the
    // redraw completes, and the form items are present in the DOM, focus / selection is restored
    // to whatever it was before the redraw
    _restoreFocusAfterRedraw : function (editColNum) {
        var lg = this.grid,
            editForm = lg.getEditForm(),
            editItem = editForm.getItem(lg.getEditorName(lg.getEditRow(), editColNum));
        
        if (editItem != null && editItem.isDrawn()) {
            var scrollLeft = lg.body.getScrollLeft(), 
                scrollTop = lg.body.getScrollTop(),
                viewportWidth = lg.body.getViewportWidth(), 
                viewportHeight = lg.body.getViewportHeight(),
                rect = editItem.getRect(),
                // If we are partially out of the viewport, don't put focus into item -
                // forces a native scroll which can interfere with user scrolling.
                // Note: partially out of viewport actually could be ok for text items
                // where focus will only cause a scroll if the actual text is offscreen.
                
                outOfViewport = rect[0] < scrollLeft ||
                                rect[1] < scrollTop ||
                                rect[0] + rect[2] > (scrollLeft + viewportWidth) ||
                                rect[1] + rect[3] > (scrollTop + viewportHeight);
            if (!outOfViewport) {
                // Avoid selecting the focused value - we don't want rapid keypresses
                // to kill what was previously entered  
                editForm._focusInItemWithoutHandler(editItem);
                // Reset the selection / text insertion point to whatever was 
                // remembered before the redraw.
                
                if (this._editorSelection && this._editorSelection[0] != null) {
                    editItem.setSelectionRange(this._editorSelection[0], this._editorSelection[1]);
                }
                // clear up the _editorSelection flag so we don't try to restore focus again on
                // scroll
                delete this._editorSelection;
            }
        }
    },
    
	// Override isDirty to return true if the parent (ListGrid) is pending a redraw.
    isDirty : function (a,b,c) {
        return this.invokeSuper(null, "isDirty", a,b,c) || this.grid.isDirty();
    },
    
    // Override cellMove: We need to be able to show validation error HTML
    cellMove : function (record,rowNum,colNum) {
       
        var nativeTarget = isc.EH.lastEvent ? isc.EH.lastEvent.nativeTarget : null;
        if (nativeTarget && nativeTarget.getAttribute != null &&
            (nativeTarget.getAttribute("isErrorIcon") == "true")) 
        {
            if (this.grid._overErrorIcon != null) {
                var lastRow = this.grid._overErrorIcon[0],
                    lastCol = this.grid._overErrorIcon[1];
                if (lastRow != rowNum || lastCol != colNum) {
                    this.grid._handleErrorIconOut();
                }
            }
            
            if (this.grid._overErrorIcon == null) {
                this.grid._handleErrorIconOver(rowNum,colNum);
            }
        } else {
            if (this.grid._overErrorIcon != null) {
                this.grid._handleErrorIconOut();
            }
        }
    },

    // Override shouldShowRollOver to avoid styling the current edit cell with the over
    // style.
    // This avoids the issue where if you roll over the edit form items, the rollover style
    // would flash off as the body receives a mouseout (looks very weird).        
    shouldShowRollOver : function (rowNum, colNum,a,b) {

        //if (!this.invokeSuper(null, "shouldShowRollOver", rowNum,colNum,a,b)) return false;
        
        if (!this.grid.showRollOver || this._rowAnimationInfo) return false;
 
        // don't show rollover for the edit row if editing the whole row
        var lg = this.grid;
        if (lg._editorShowing && !lg.editByCell && rowNum == lg._editRowNum) return false;
        
        return true;
    },

    updateRollOver : function (rowNum, colNum, movingToNewCell) {

        var lg = this.grid;
        
        if (lg.showRollOverCanvas) {            
            // movingToNewCell param passed when the user rolled off one cell and over another
            // and this method is being called to clear the first cell hilight.
            // we can no-op in this case since we'll update the rollOverCanvas on the subsequent
            // call to this method, and that will avoid a clear/draw cycle (and flash)   
            
            if (!movingToNewCell) {
                var leaving = !(this.lastOverRow == rowNum && this.lastOverCol == colNum);
                lg.updateRollOverCanvas(rowNum, colNum, leaving);
            }
            // no support for frozen body / rollOverCanvas yet

        }

        this.setRowStyle(rowNum, null, (this.useCellRollOvers ? colNum : null));

        // frozen fields: if there is another body, force its rollover row to match and
        // update it
        var otherBody = (this == lg.body ? lg.frozenBody : lg.body);
        if (otherBody) {
            otherBody.lastOverRow = this.lastOverRow;
            otherBody.lastOverCol = this.lastOverCol;
            otherBody.setRowStyle(rowNum, null, (this.useCellRollOvers ? colNum : null));
        }
    },
    
    // override selectOnMouseDown/Up to disable selection when a row is clicked anywhere
    // besides the checkbox when selectionAppearance is checkbox.
    selectOnMouseDown : function (record, rowNum, colNum) {
        var shouldSelect = true,
            selApp = this.grid.selectionAppearance,
            cbSel = (selApp == "checkbox");
        if (cbSel) {
            // if frozen fields are showing, the cb field will show up in the frozen body!
            if ((this.grid.frozenFields != null && this.grid.frozenBody != this) || 
                (this.grid.getCheckboxFieldPosition() != colNum)) 
            {
                shouldSelect = false;
            }
        }
        
        if (shouldSelect) {
            this.invokeSuper(isc.GridBody, "selectOnMouseDown", record, rowNum, colNum);    
        }

        if (isc.screenReader) {
            this._putNativeFocusInRow(rowNum);
        }
    },
    
    selectOnMouseUp : function (record, rowNum, colNum) {
        var cbColNum = this.grid.getCheckboxFieldPosition(),
            selApp = this.grid.selectionAppearance;
        if (selApp != "checkbox" || (selApp == "checkbox" && cbColNum == colNum)) {
            this.invokeSuper(isc.GridBody, "selectOnMouseUp", record, rowNum, colNum);    
        }
    },
    
    // Override handleSelectionChanged() to fire our viewStateChanged method
    handleSelectionChanged : function (record,state) {
        var returnVal = this.Super("handleSelectionChanged", arguments);
        this.grid.handleViewStateChanged();
        return returnVal;
    },
    
    // When refreshing cellStyle, notify our edit items that the underlying cell style changed
    // so they can update if necessary
    _updateCellStyle : function (record, rowNum, colNum, cell, className, a,b,c) {
        this.invokeSuper(isc.GridBody, "_updateCellStyle", record, rowNum,colNum,cell,className,a,b,c);
        var lg = this.grid;        
        if (lg && lg.getEditRow() == rowNum) {
            var fieldName = lg.getFieldName(lg.getFieldNumFromLocal(colNum, this)),
                form = lg.getEditForm(),
                item = form ? form.getItem(fieldName) : null;                
            if (item && item.gridCellStyleChanged) {
                if (className == null) className = this.getCellStyle(record,rowNum,colNum);
                item.gridCellStyleChanged(record, rowNum, colNum, className);
            }
        }
        
    },
    
    // hovers: override getHoverTarget to return a pointer to our grid - this allows 
    // the developer to call 'updateHover' directly on the grid.
    getHoverTarget : function () {
        return this.grid;
    },

	// direct keyPresses to the ListGrid as a whole to implement arrow navigation,
	// selection, etc
	
    keyPress : function (event, eventInfo) {
        return this.grid.bodyKeyPress(event, eventInfo);
    },
    
    // getters for the current keyboard focus row for key-events.
    
    getFocusRow : function () {
        return this.grid.getFocusRow();
    },
    getFocusCol : function () {
        var colNum = this.grid._getKeyboardClickNum();
        return this.grid.getLocalFieldNum(colNum);
    },
    
	// Override _focusChanged to implement 'editOnFocus' - start editing the first
	// editable cell if appropriate.
	// See comments in 'editOnFocus' jsdoc comment for details of how this should work.
    _focusChanged : function (hasFocus) {
        
    	// use the Super implementation to set up this.hasFocus BEFORE we further 
    	// manipulate focus due to editing.
        var returnVal = this.Super("_focusChanged", arguments);

        var lastEvent = isc.EH.lastEvent;

        // if we're acquiring focus because we're in the middle of a click sequence on the body, 
        // the mouse handlers will correctly start editing or place focus on whatever row was hit, and we
        // should do nothing
        if (lastEvent.target == this &&
              (lastEvent.eventType == isc.EH.MOUSE_DOWN ||
               lastEvent.eventType == isc.EH.MOUSE_UP ||
               lastEvent.eventType == isc.EH.CLICK ||
               lastEvent.eventType == isc.EH.DOUBLE_CLICK)) return returnVal;
    
        // otherwise, entering focus due to a key event (tab, shift-tab) or something else (programmatic
        // including re-focus due to clickMask blur).
        var editCell,
            parent = this.grid;
        if (hasFocus && parent.isEditable()) {      
        	// editOnFocus enabled, but not currently editing
            if (parent.editOnFocus && parent.isEditable() &&     
                parent.getEditRow() == null)
            {
                if (this.logIsInfoEnabled("gridEdit")) {
            	    this.logInfo("Editing on focus: eventType: " + lastEvent.eventType +
            	                 ", lastTarget " + lastEvent.target, "gridEdit");
                }
            
            	// If we're explicitly suppressing edit on focus, don't start editing.
                if (parent._suppressEditOnFocus) {
                    delete parent._suppressEditOnFocus;
                } else {
                    // this'll run through every cell in every record until it finds one that's
                    // editable
                    editCell = parent.findNextEditCell(0,0,true,true);

                    if (editCell != null) 
                        parent.handleEditCellEvent(editCell[0], editCell[1], isc.ListGrid.FOCUS);
                }
            }
        }
          
        // In screenReader mode, if focus is moving into the grid but we're not going into editing mode,
        // put focus onto the row element rather than onto the GR body.
        if (hasFocus && editCell == null && isc.screenReader) {
            var lastEvent = isc.EH.lastEvent,
                eventType = lastEvent.eventType;

            // find the last hilited row if there is one
            var rowNum = parent.getFocusRow();
            // default to top of viewport
            if (rowNum == null) {
                rowNum = this.getVisibleRows()[0];
                // if first row is partially offscreen jump to next row to avoid auto-scroll
                if (this.getRowTop(rowNum) < this.getScrollTop()) rowNum += 1;
            }
            //this.logWarn("focus entering body - focusing in native row: " + rowNum +
            //             ", focus row was: " + parent.getFocusRow());
            parent._hiliteRecord(rowNum);
            this._putNativeFocusInRow(rowNum); 
        }

        return returnVal;
    },

    _putNativeFocusInRow : function (rowNum, suppressElementFocus) {
        this._nativeFocusRow = rowNum;

        if (suppressElementFocus || !this.hasFocus) return;

        var element = this.getTableElement(rowNum);
        if (element == null) return;

        
        isc.EH._focusCanvas = null;
        element.focus();
        isc.EH._focusCanvas = this;
    },

    getFocusHandle : function (newState) {
        if (!isc.screenReader) return this.Super("getFocusHandle", arguments);

        
        if (!newState) {
            var rowNum = this._nativeFocusRow;
            if (rowNum != null) {
                var nativeRow = this.getTableElement(rowNum);
                if (nativeRow != null) {
                    //this.logWarn("returning null for focus handle since focus is on row");
                    return null;
                }
            }
        }

        return this.Super("getFocusHandle", arguments);
    },
    
    // override updateRowSelection to update selectionCanvas if necessary
    updateRowSelection : function (rowNum) {
        var lg = this.grid;
        if (!lg) return;
        
        
        
        if (lg.showSelectionCanvas) lg.updateSelectionCanvas();
        if (lg._dontRefreshSelection) return;
        
        this.invokeSuper(isc.GridBody, "updateRowSelection", rowNum);

        if (isc.Canvas.ariaEnabled() && lg.selection) {
            this.setRowAriaState(rowNum, "selected", lg.selection.isSelected(lg.getRecord(rowNum)));
        }
        
        // with selectionAppearance:"checkbox", detect all rows selected and update checkbox in
        // header
        if (lg.getCurrentCheckboxField() != null) {
            var cellNum = lg.getCheckboxFieldPosition();
            if (lg) lg.refreshCell(rowNum, cellNum);
            var validData = (isc.isAn.Array(lg.data) || (isc.isA.ResultSet(lg.data) 
                        && lg.data.allMatchingRowsCached())),
                selection = lg.getSelection() || [];
                
            if (validData) {
                if (selection.length == lg.data.getLength()) {
                    lg._setCheckboxHeaderState(true);
                } else {
                    lg._setCheckboxHeaderState(false);
                }
            }
        } else if (lg.getTreeFieldNum && lg.selectionAppearance == "checkbox") {
            // in the TreeGrid, refresh the tree cell because that's where the checkbox is shown
            var treeCellNum = lg.getTreeFieldNum();
            lg.refreshCell(rowNum, treeCellNum);
        }
    },
    // ditto with _cellSelectionChanged
    _cellSelectionChanged : function (cellList,b,c,d) {
        var lg = this.grid;     
        if (lg && lg.showSelectionCanvas) lg.updateSelectionCanvas();
        return this.invokeSuper(isc.GridBody, "_cellSelectionChanged", cellList, b,c,d);
    },
    
    // Embedded components
    // -----------------------
    
    // animateShow selectionCanvas / rollOverCanvas if appropriate
    shouldAnimateEmbeddedComponent : function (component) {
        var grid = this.grid;
        if (component == grid.selectionCanvas) return grid.animateSelection;
        if (component == grid.selectionUnderCanvas) return grid.animateSelectionUnder;
        if (component == grid.rollOverCanvas) return grid.animateRollOver;
        if (component == grid.rollUnderCanvas) return grid.animateRollUnder;
            
        return false;
    },
    
    
    _handleEmbeddedComponentResize : function (component, deltaX, deltaY) {
        this.Super("_handleEmbeddedComponentResize", arguments);
        
        // Notify the grid - allows us to update the other body if we're showing
        // both a frozen and an unfrozen body
        this.grid._handleEmbeddedComponentResize(this, component, deltaX, deltaY);
    },
    
    // Override draw() to scroll to the appropriate cell if 'scrollCellIntoView' was called
	// before the body was drawn/created
	// Also update the edit form item rows if we're already editing.
    draw : function (a,b,c,d) {
        var lg = this.grid;
        
        if (lg.getEditRow() != null) {
            
            var rowNum = lg.getEditRow(),
                record = lg.getRecord(rowNum),
                fieldNum = lg.getEditCol(),
                form = lg._editRowForm,
                items = lg.getEditRowItems(record, rowNum, fieldNum, lg.editByCell),
                liveItems = form.getItems();
               
            var setItems = liveItems == null || items.length != liveItems.length;
            if (!setItems) {
                var liveItemNames = liveItems.getProperty("name");
                for (var i = 0; i < items.length; i++) {
                    if (!liveItemNames.contains(items[i].name)) {
                        setItems = true;
                        break;
                    }
                }
            }
            if (setItems) {
                this.logDebug("calling setItems on form from body draw","gridEdit");
                form.setItems(items);
            } else {
                this.logDebug("Skipping setItems() on form from body draw", "gridEdit");
            }
            
            
            form._setValuesPending = true;
            
        }
        this.invokeSuper(null, "draw", a,b,c,d);
        
        // If we are showing any edit form items, notify them that they have been written
        // into the DOM.
        
        if (lg._editRowForm) {
            lg._editItemsDrawingNotification(null, null, this);
        }
        // Tell the form to update its values (setItemValues())
        // (do this after the items have been notified that they're drawn to ensure items'
        // element values are set)
        lg.updateEditRow(lg.getEditRow());

        if (lg._scrollCell != null) {
            var scrollRowNum = isc.isAn.Array(lg._scrollCell) ? lg._scrollCell[0] : lg._scrollCell,
                scrollColNum = isc.isAn.Array(lg._scrollCell) ? lg._scrollCell[1] || 0 : 0
            ;
            lg.scrollCellIntoView(scrollRowNum, scrollColNum);
            delete lg._scrollCell;
        }

        // Call 'updateRecordComponents()' on initial draw to set up recordComponents
        // If this is a ResultSet rather than an array, the updateRecordComponents method
        // will be able to skip all records and we'll render out the components on redraw.
        this.grid.updateRecordComponents();
        
    },

	// rerun ListGrid-level layout if the body's scrolling state changes, to allow sizing
	// the header appropriately
    layoutChildren : function (reason,a,b,c) {
        this.invokeSuper(null, "layoutChildren", reason,a,b,c);
        // This method may be called with "scrolling state change" when a bodyLayout is 
        // currently undrawn but drawing out its children - we've seen this in FF 3
        // In this case bail now since if _updateFieldWidths() is fired on an undrawn body it
        // bails, leaving the body mis sized
        
        if (!this.isDrawn() || (this.grid.frozenFields && !this.grid.bodyLayout.isDrawn())) {   
            return;
        }
        if (reason == "scrolling state changed") {
                
            if (this._rowHeightAnimation == null) {
                this.grid.layoutChildren("body scroll changed");
                delete this._scrollbarChangeDuringAnimation;
    
            
            } else {
                this._scrollbarChangeDuringAnimation = true;
            }
        }
    },
    
    // Override rowAnimationComplete to call layoutChildren on the ListGrid if
    // scrollbars were introduced or cleared during animation.
    _rowAnimationComplete : function () {
        this.Super("_rowAnimationComplete", arguments);
        if (this._scrollbarChangeDuringAnimation) {    
            this.grid.layoutChildren("body scroll changed during animation");
            delete this._scrollbarChangeDuringAnimation;
        }
    },

    
    // Override moved to notify any edit form items that they have moved.
    handleMoved : function (a,b,c,d) {
        this.invokeSuper(null, "handleMoved", a,b,c,d);
        
        var lg = this.grid;
        if (lg._editorShowing) {
            lg._editRowForm.itemsMoved();
        }
        
    },
    
    handleParentMoved : function (a,b,c,d) {
        this.invokeSuper(null, "handleParentMoved", a,b,c,d);
        var lg = this.grid;
        if (lg._editorShowing) {
            lg._editRowForm.itemsMoved();
        }            
    },
    
    // Override show() / hide() / parentVisibilityChanged() / clear() to notify the Edit 
    // form items that they have been shown / hidden.
    setVisibility : function (newVisibility,b,c,d) {
        this.invokeSuper(null, "setVisibility", newVisibility,b,c,d);
        var lg = this.grid;
        if (lg._editorShowing) lg._editRowForm.itemsVisibilityChanged();
    },
    
    parentVisibilityChanged : function (newVisibility,b,c,d) {
        this.invokeSuper(null, "parentVisibilityChanged", newVisibility,b,c,d);
        var lg = this.grid;
        if (lg._editorShowing) lg._editRowForm.itemsVisibilityChanged();
    },
    
    clear : function () {
        var lg = this.grid;
        lg._clearingInactiveEditorHTML();
        this.Super("clear", arguments);
        if (lg._editorShowing) {
            // If we're showing the editRow form, notify the items that they have
            // been removed from the DOM.
            lg._editItemsDrawingNotification(null, null, this);

            // Separate mechanism to notify the form that items are no longer visible.
            
            lg._editRowForm.itemsVisibilityChanged();
        }
    },
    
    // Our tab-index is handled by our parent - similarly use that to manage next/prev
    // tab-widgets.
    _getNextTabWidget : function (backwards) {
        return this.grid._getNextTabWidget(backwards);
    },

    
    _setNextTabWidget : function () {
    },

    
    // also notify the edit form items of z index change
    zIndexChanged : function () {
        this.Super("zIndexChanged", arguments);
        var lg = this.grid;
        // Note: setZIndex can be called at init time to convert "auto" to a numeric
        // zIndex - we therefore can't assume that we've been added to the ListGrid as
        // a child yet.
        if (lg && lg._editorShowing) lg._editRowForm.itemsZIndexChanged();
        
    },
    parentZIndexChanged : function (a,b,c,d) {
        this.invokeSuper(null, "zIndexChanged", a,b,c,d);
        var lg = this.grid;
        if (lg._editorShowing) lg._editRowForm.itemsZIndexChanged();        
    },
    
    // Implement 'redrawFormItem()' - if one of the edit form items asks to redraw
    // we can simply refresh the cell rather than having the entire body redraw
    redrawFormItem : function (item, reason) {
        var lg = this.grid;
        if (lg && (item.form == lg._editRowForm)) {
            // determine which cell
            var row = lg.getEditRow(), col = lg.getColNum(item.getFieldName());
            
            // If the user has edited the cell, or setValue() has been called on the item
            // we don't want a call to redraw() on the item to drop that value
            if (lg.getEditCol() == col) {
                lg.storeUpdatedEditorValue();
            }
            
            lg.refreshCell(row, col, false, true);
            
        } else 
            return this.markForRedraw("Form Item Redraw " + (reason ? reason : isc.emptyString));
    },
	
    
    sizeFormItem : function (item) {
        var lg = this.grid;
        var width = item.width,
            finalWidth;
        
        if (isc.isA.String(width)) {
            var fieldWidths = lg.getEditFormItemFieldWidths(item.record),
                fieldWidth = fieldWidths[lg.getFieldNum(item.getFieldName())];
             if (width == "*") {
                finalWidth = fieldWidth;
             } else if (width[width.length-1] == "%") {
                var percentWidth = parseInt(width);
                if (isc.isA.Number(percentWidth)) {
                    finalWidth = Math.floor(fieldWidth * (percentWidth / 100));
                }
            }
        }
        
        var height = item.height,
            finalHeight;
        if (isc.isA.String(height)) {
            var cellHeight = lg.cellHeight;
            if (width == "*") {
                finalHeight = cellHeight;
            } else if (height[height.length-1] == "%") {
               var percentHeight = parseInt(height);
               if (isc.isA.Number(percentHeight)) {
                   finalHeight = Math.floor(cellHeight * (percentHeight / 100));
               }
            }
        }
        // Hang the calculated values on the _size attribute as we do when running
        // normal stretch-resize policy in form items.
        
        if (finalHeight != null || finalWidth != null) {
            item._size = [finalWidth == null ? item.width : finalWidth,
                          finalHeight == null ? item.height : finalHeight];
        }
        
    },
	
	//>Animation
    // Override startRowAnimation - if doing a delayed redraw to kick off a row animation
    // to close an open folder, we need to temporarily open the folder again to get the
    // appropriate HTML for the animation rows.
    startRowAnimation : function (show, startRow, endRow, callback, speed, duration, 
                                  effect, slideIn, delayed) 
    {
        this.finishRowAnimation();
        
        var shouldOpenFolder = (delayed && (this._openFolder != null)),
            tg = this.grid;
            
        if (shouldOpenFolder) {
            tg._suppressFolderToggleRedraw = true;
            tg.data.openFolder(this._openFolder);
            tg._suppressFolderToggleRedraw = null;
        }
        this.Super("startRowAnimation", arguments);
        if (shouldOpenFolder) {            
            tg._suppressFolderToggleRedraw = true;
            tg.data.closeFolder(this._openFolder);
            tg._suppressFolderToggleRedraw = null;
        }        
        delete this._openFolder;
    }
    //<Animation
});


isc.ListGrid.addClassProperties({
    

	//>	@type SortArrow
	//			Do we display an arrow for the sorted field ?
	//			@group	sorting, appearance
    //	@value	"none"   Don't show a sort arrow at all.
    //	@value	"corner" Display sort arrow in the upper-right corner (above the scrollbar) only.
	CORNER:"corner",	
    //	@value	"field"  Display sort arrow above each field header only.
	FIELD:"field",			
    //	@value	"both"   Display sort arrow above each field header AND in corner above scrollbar.
	//BOTH:"both", // NOTE: Canvas establishes this constant
    // @visibility external
	//<
	// NOTE: Canvas established the constant NONE ( == "none")

    //> @type ReorderPosition
    //  Controls where a drag-item should be dropped in relation to the target row
    //  @group dragdrop
    //  @value  ListGrid.BEFORE  Drop the drag-item before the target-row
    BEFORE:"before",
    //  @value  ListGrid.AFTER   Drop the drag-item after the target-row
    AFTER:"after",
    //  @value  ListGrid.OVER    Drop the drag-item over (onto) the target-row
    OVER:"over",
    // @visibility external
	//<
     
	//> @type RowEndEditAction
	//  While editing a ListGrid, what cell should we edit when the user attempts to navigate 
	//  into a cell past the end of an editable row, via a Tab keypress, or a programmatic
	//  saveAndEditNextCell() call?
	//
	// @value   "same"   navigate to the first editable cell in the same record
	// @value   "next"   navigate to the first editable cell in the next record
	// @value   "done"   complete the edit.
	// @value   "stop"   Leave focus in the cell being edited (take no action)
	// @value   "none"   take no action
	//
	// @visibility external
	// @group editing
	// @see ListGrid.rowEndEditAction
	//
	//<
    
	//>	@type EnterKeyEditAction
	// What to do when a user hits enter while editing a cell
	// @value "done" end editing
	// @value "nextCell" edit the next editable cell in the record
	// @value "nextRow" edit the same field in the next editable record
	// @value "nextRowStart" edit the first editable cell in next editable record
	//
	// @group editing
	// @visibility external
	//<
    
    //> @type EscapeKeyEditAction
    // What to do if the user hits escape while editing a cell.
    // @value "cancel" cancels the current edit and discards edit values
    // @value "done" end editing
    //
    // @group editing
    // @visibility external
    //<

	//>	@type EditCompletionEvent
	//			What event / user interaction type caused cell editing to complete.
	//			@visibility external
	//			@group	editing
	//
	//          @value  isc.ListGrid.CLICK_OUTSIDE  User clicked outside editor during edit.
	//          @value  isc.ListGrid.CLICK  User started editing another row by clicking on it
	//          @value  isc.ListGrid.DOUBLE_CLICK  User started editing another row by double
    //                               clicking 
	//          @value  isc.ListGrid.ENTER_KEYPRESS Enter pressed.
	//          @value  isc.ListGrid.ESCAPE_KEYPRESS    User pressed Escape.
	//          @value  isc.ListGrid.UP_ARROW_KEYPRESS  Up arrow key pressed.
	//          @value  isc.ListGrid.DOWN_ARROW_KEYPRESS    down arrow key.
	//          @value  isc.ListGrid.TAB_KEYPRESS   User pressed Tab.
	//          @value  isc.ListGrid.SHIFT_TAB_KEYPRESS   User pressed Shift+Tab.
	//          @value  isc.ListGrid.EDIT_FIELD_CHANGE      Edit moved to a different field (same row)
	//          @value  isc.ListGrid.PROGRAMMATIC   Edit completed via explicit function call
	// @visibility external
	//<
    CLICK_OUTSIDE:"click_outside",
    CLICK:"click",
    DOUBLE_CLICK:"doubleClick",
    ENTER_KEYPRESS:"enter",
    ESCAPE_KEYPRESS:"escape",
    UP_ARROW_KEYPRESS:"arrow_up",
    DOWN_ARROW_KEYPRESS:"arrow_down",
    // left/right only used in conjunction with moveEditorOnArrow
    LEFT_ARROW_KEYPRESS:"arrow_left",
    RIGHT_ARROW_KEYPRESS:"arrow_right",
    TAB_KEYPRESS:"tab",
    SHIFT_TAB_KEYPRESS:"shift_tab",
    EDIT_FIELD_CHANGE:"field_change",
    EDIT_ROW_CHANGE:"row_change",
    PROGRAMMATIC:"programmatic",
    // Focus is not a valid edit completion event - focusing in the grid can start an edit
    // if editOnFocus is true but this should not kill an existing edit.
    FOCUS:"focus",

	// GridRenderer passthrough
	// --------------------------------------------------------------------------------------------

	// the following properties, when set on the LV, are applied to the Grid
    _gridPassthroughProperties : [
    	// pass it a selection object (this enables selection behaviors)
        "selection", 
        "selectionType",
        "canSelectCells",
        "canDragSelect",
        "canSelectOnRightMouse",
        "recordCanSelectProperty",
                    
    	// D&D
        "canDrag",
        "canAcceptDrop",
        "canDrop",

    	// table geometry
        "autoFit",
        "wrapCells",
        "cellSpacing",
        "cellPadding",
        "cellHeight",
        "enforceVClipping",
        // autoFitData behavior implemented on GridBody class, not GR class
        "autoFitData",
        "autoFitMaxRecords",
        "autoFitMaxWidth",
        "autoFitMaxColumns",
        "autoFitMaxHeight",
        "autoFitExtraRecords",
        
    	// incremental rendering
    	// "showAllRecords" -> showAllRows done elsewhere
        "showAllColumns",
        "drawAllMaxCells",
        "drawAheadRatio",
        "quickDrawAheadRatio",
        "instantScrollTrackRedraw",
        "scrollRedrawDelay",
        
        // printing
        "printMaxRows",
        
        //>Animation
        // If we're doing a speed rather than duration based row animation allow the cap to
        // be specified on the ListGrid / TreeGrid
        // (Note that this is documented in the TreeGrid class).
        "animateRowsMaxTime",
        //<Animation

    	// documented by default setting
        "fastCellUpdates",

    	// rollover
        "showRollOver",
        "useCellRollOvers",

    	// hover
        "canHover",
        "showHover",
        "hoverDelay", 
        "hoverWidth", 
        "hoverHeight", 
        "hoverAlign", 
        "hoverVAlign", 
        "hoverStyle",
        "hoverOpacity",
        "hoverMoveWithMouse",
        
        "hoverByCell",
        "keepHoverActive",
        "cellHoverOutset",

    	// empty message
        "showEmptyMessage",
        "emptyMessageStyle",
        "emptyMessageTableStyle",    

    	// offline message
        "showOfflineMessage",
        "offlineMessageStyle",
        
        // special presentation of records
        "singleCellValueProperty",
        "isSeparatorProperty",

    	// Focus things -- note no need to pass tabIndex through - this is handled when
    	// we create the body / update the tabIndex on the parent
        "accessKey",
        "canFocus",
        "_useNativeTabIndex",
        "tableStyle",
        "baseStyle",
        "recordCustomStyleProperty",
        "showSelectedStyle",
        
        // whether to allow arbitrary rowSpanning.  Internal property
        "fullRowSpans",

        "showFocusOutline"
    ],

	// the following methods, when called on the LV, will call the same-named method on the
	// GridRenderer (this.body). 
    _lv2GridMethods : [
        // this makes it easier to override getCellStyle at the LV level, since you can call
        // these helpers as this.getCellStyleName()
        "getCellStyleName",
        "getCellStyleIndex",
        
        // setFastCellUpdates explicitly handled
        // in a method which keeps lg.fcu in synch with the 
        // body property value 
        //"setFastCellUpdates",
        
    	// checking table geometry   
        "getRowTop",
        "getRowPageTop",
        "getRowSize",
        "getDrawnRowHeight",
        "getCellPageRect",
        
        //> @method listGrid.getVisibleRows 
    	// @include gridRenderer.getVisibleRows()
        // @visibility external
    	//<
        "getVisibleRows",

        //> @method listGrid.getDrawnRows
    	// @include gridRenderer.getDrawnRows()
        // @visibility external
    	//<
        "getDrawnRows"
    ],

	// styling
	
	//>	@method	listGrid.getCellStyle()
	// @include gridRenderer.getCellStyle()
    // @see listGrid.getBaseStyle()
	//<

	//>	@method	listGrid.getCellCSSText()
	// @include gridRenderer.getCellCSSText()
    // @example addStyle
	//<

	// refresh
	//>	@method	listGrid.refreshCellStyle()    
	//  @include    gridRenderer.refreshCellStyle()
	//<

	// events
	//>	@method	listGrid.cellOver()
	// @include gridRenderer.cellOver()
	//<
	//>	@method	listGrid.rowOver()
	// @include gridRenderer.rowOver()
	//<

	//>	@method	listGrid.cellOut()
	// @include gridRenderer.cellOut()
	//<
	//>	@method	listGrid.rowOut()
	// @include gridRenderer.rowOut()
	//<
    
	//>	@method	listGrid.cellHover()
	// @include gridRenderer.cellHover()
	//<
	//>	@method	listGrid.rowHover()
	// @include gridRenderer.rowHover()
	//<
	//>	@method	listGrid.cellHoverHTML()
	// @include gridRenderer.cellHoverHTML()
	//<
    
	//>	@method	listGrid.cellContextClick()
	// @include gridRenderer.cellContextClick()
    // @example cellClicks
	//<
	//>	@method	listGrid.rowContextClick()
	// @include gridRenderer.rowContextClick()
    // @example recordClicks
	//<
    
	//>	@method	listGrid.cellMouseDown()
	// @include gridRenderer.cellMouseDown()
	//<
	//>	@method	listGrid.rowMouseDown()
	// @include gridRenderer.rowMouseDown()
	//<

	//>	@method	listGrid.cellMouseUp()
	// @include gridRenderer.cellMouseUp()
	//<
	//>	@method	listGrid.rowMouseUp()
	// @include gridRenderer.rowMouseUp()
	//<

	//>	@method	listGrid.cellClick()
	// @include gridRenderer.cellClick()
    // @example cellClicks
	//<

	//>	@method	listGrid.cellDoubleClick()
	// @include gridRenderer.cellDoubleClick()
    // @example cellClicks
	//<
    
    // Geometry
	//>	@method	listGrid.getRowTop()    
	//  @include    gridRenderer.getRowTop()
	//<
    
	//>	@method	listGrid.getRowPageTop()    
	//  @include    gridRenderer.getRowPageTop()
	//<

	// the following methods, when called on the GridRenderer used as LV.body, call the same-named
	// method on the ListGrid instance itself
    _grid2LVMethods : [
        
        "getTotalRows",
        "isEmpty",
        "cellIsEnabled",
        "willAcceptDrop",

    	// passed scroll change notification through
        "scrolled",

    	// native element naming
        "getTableElementId",
        "getRowElementId",
        "getCellElementId",

    	// shouldFixRowHeight - enables us to override the ListGrid level 'fixedRecordHeights'
    	// for individual rows
        "shouldFixRowHeight",

        "getEmptyMessage",
        "getCanHover",
        // bubble stopHover on the GR up to stopHover here.
        "stopHover",
        
        "updateEmbeddedComponentZIndex"

        // NOTE: These methods pick up their parameters from the stringMethodRegistry on the
        // GridRenderer class. If expanding this list ensure that any methods that take parameters
        // are registered as stringMethods on that class
    ]
});

isc.ListGrid.addClassMethods({
    makeBodyMethods : function (methodNames) {
        var funcTemplate = this._funcTemplate;
        if (funcTemplate == null) {
            funcTemplate = this._funcTemplate = [
                ,
                
                
                "this.grid._passthroughBody = this;" + 
                "var returnVal = this.grid.",,"(",,");" +
                "this.grid._passthroughBody=null;" +
                "return returnVal;"];
        }

        var methods = {};

		for (var i = 0; i < methodNames.length; i++) {
			var methodName = methodNames[i],
                argString = isc.GridRenderer.getArgString(methodName);
            
            if (isc.contains(argString, "colNum")) {
                // if there's a colNum argument, map it to the field index in the master
                funcTemplate[0] = "if (this.fields[colNum]) colNum = this.fields[colNum].masterIndex;"

            } else if (isc.isAn.emptyString(argString)) {
                // if there are no arguments, pass the body itself as a means of identifying
                // the calling body
                argString = "body";
                funcTemplate[0] = "body = this;";
            } else {
                funcTemplate[0] = null;
            }
    
			// create a function that routes a function call to the target object
            funcTemplate[2] = methodName;
            funcTemplate[4] = argString;
            var functionText = funcTemplate.join(isc.emptyString);
            
            //this.logWarn("for method: " + methodName + " with argString :"  + argString +  
            //             " function text is: " + functionText);

            methods[methodName] = 
                new Function(argString, functionText);
		}
        
        return methods;
    },
    
  
    classInit : function () {
    	// create functions to have methods on the ListGrid's body call methods on the ListGrid
    	// itself.  This is partly legacy support: the way to customize body rendering used to
    	// be to install functions that controlled body rendering directly on the ListGrid
    	// itself.

    	// make certain grid methods appear on the LV for convenience, so you don't have to go
    	// this.body.someGridMethod()
    	
    	
        this.addMethods(isc.ClassFactory.makePassthroughMethods(
            this._lv2GridMethods, "body"));

		// ----------------------------------------------------------------------------------------
    	// create methods that can be installed on the body to call methods on the LV itself, for:
        var passthroughMethods = {};

    	// - handlers (like cellOver) and overrides (like getCellCSSText) that we allow to be
    	//   defined on the LV but are really grid APIs
        var gridAPIs = isc.getKeys(isc.GridRenderer._gridAPIs),
            passthroughMethods = isc.ListGrid.makeBodyMethods(gridAPIs);

    	// - methods the grid needs to fulfill as the drag/drop target, which are really implemented
    	//   on the LV
        isc.addProperties(passthroughMethods,
                          isc.ListGrid.makeBodyMethods(this._grid2LVMethods));

        this._passthroughMethods = passthroughMethods;

    	

    	// create methods on the ListGrid to act as Super implementations for per-instance
    	// overrides of methods where we want to call the original GridRenderer implementation
    	// as Super.
        var passBackMethods = {},
            funcTemplate = [
                ,
                // _passthroughBody is set up by the body function that called back up the
                // the grid method - if present, we use it to ensure we call the original
                // implementation on the correct body.
                "var _passthroughBody = this._passthroughBody || this.body;" +
                " if (_passthroughBody == null) {" +
                    "return;" +
                "}" +
                "if(_passthroughBody.__orig_",,")return _passthroughBody.__orig_",,"(",,")"],
            origPrefix  = "__orig_",
            gridProto = isc.GridRenderer.getPrototype();
        for (var i = 0; i < gridAPIs.length; i++) {
            var methodName = gridAPIs[i],
                argString = isc.GridRenderer.getArgString(methodName);
            if (isc.ListGrid.getInstanceProperty(methodName) == null) {
                 
                if (isc.contains(argString, "colNum")) {
                    // if there's a colNum argument, map it to the field index in the body
                    
                    funcTemplate[0] = "if (colNum != null && colNum >= 0) colNum = this.getLocalFieldNum(colNum);"
                } else {
                    funcTemplate[0] = null;
                }
                funcTemplate[2] = funcTemplate[4] = methodName;
                funcTemplate[6] = argString
                
                passBackMethods[methodName] = new Function(argString, 
                    funcTemplate.join(isc.emptyString));
                // XXX this would also work, but imposes another Super call penalty, and is
                // odd (call to Super from outside of the object)
                //"return this.body.Super('" + methodName + "', arguments);");
            }
        	
            gridProto[origPrefix + methodName] = gridProto[methodName];
        }
        this._passBackMethods = passBackMethods;
        this.addMethods(passBackMethods);

    }
});

// add default properties to the class
isc.ListGrid.addProperties( {
    
	//>	@attr	listGrid.styleName		(CSSStyleName : "listGrid" : IRW)
	// Default CSS class
	// @group	appearance
    // @visibility external
	//<
    styleName:"listGrid",

	//>	@attr	listGrid.data		(List of ListGridRecord : null : IRW)
	// A List of ListGridRecord objects, specifying the data to be used to populate the
    // ListGrid.  In ListGrids, the data array specifies rows. Note that ListGrids
    // automatically observe changes to the data List and redraw accordingly.
    // <p>
    // This property is settable directly only as part of a +link{ListGrid} constructor.  If
    // you want to change the +link{ListGrid}'s data after initial creation, call
    // +link{listGrid.setData}.
    // <p>
    // This property will typically not be explicitly specified for databound ListGrids, where
    // the data is returned from the server via databound component methods such as
    // +link{fetchData()}. In this case the data objects will be set to a 
    // +link{class:ResultSet,resultSet} rather than a simple array.
	//
	// @group	data
    // @see ListGridRecord
    // @setter ListGrid.setData()
	// @visibility external
    // @example inlineData
    // @example localData
	//<
    
    // useCellRecords - Is our data model going to be one record per cell or one record per row?
    useCellRecords:false,

    //> @object ListGridRecord
    // A ListGridRecord is a JavaScript Object whose properties contain values for each
    // +link{ListGridField}.  A ListGridRecord may have additional properties which affect the
    // record's appearance or behavior, or which hold data for use by custom logic or other,
    // related components.
    // <p>
    // For example a ListGrid that defines the following fields:
    // <pre>
    // fields : [
    //     {name: "field1"},
    //     {name: "field2"}
    // ],
    // </pre>
    // Might have the following data:
    // <pre>
    // data : [
    //     {field1: "foo", field2: "bar", customProperty:5},
    //     {field1: "field1 value", field2: "field2 value", enabled:false}
    // ]
    // </pre>
    // Each line of code in the <code>data</code> array above creates one JavaScript Object via
    // JavaScript {type:ObjectLiteral,object literal} notation.  These JavaScript Objects are
    // used as ListGridRecords.
    // <P>
    // Both records shown above have properties whose names match the name property of a
    // ListGridField, as well as additional properties.  The second record will be disabled due to
    // <code>enabled:false</code>; the first record has a property "customProperty" which will
    // have no effect by default but which may be accessed by custom logic.
    // <P>
    // After a ListGrid is created and has loaded data, records may be accessed via
    // +link{listGrid.data}, for example, listGrid.data.get(0) retrieves the first record.
    // ListGridRecords are also passed to many events, such as
    // +link{ListGrid.cellClick,cellClick()}.
    // <P>
    // A ListGridRecord is always an ordinary JavaScript Object regardless of how the grid's
    // dataset is loaded (static data, java server, XML web service, etc), and so supports the
    // normal behaviors of JavaScript Objects, including accessing and assigning to properties
    // via dot notation:
    // <pre>
    //     var fieldValue = record.<i>fieldName</i>;
    //     record.<i>fieldName</i> = newValue;
    // </pre>
    // <P>
    // Note however that simply assigning a value to a record won't cause the display to be
    // automatically refreshed - +link{listGrid.refreshCell()} needs to be called.  Also,
    // consider +link{group:editing,editValues vs saved values} when directly modifying
    // ListGridRecords.
    // <P>
    // See the attributes in the API tab for the full list of special properties on
    // ListGridRecords that will affect the grid's behavior.
    //
    // @treeLocation Client Reference/Grids/ListGrid
    // @see ListGrid.data
    // @inheritsFrom Record
    // @visibility external
    //<
    

    
    //> @attr listGrid.recordEnabledProperty (string : "enabled" : IR)
    // Property name on a record that will be checked to determine whether a record is enabled.
    // <P>
    // Setting this property on a record will effect the visual style and interactivity of
    // the record.  If set to <code>false</code> the record (row in a +link{ListGrid} or
    // +link{TreeGrid}) will not highlight when the mouse moves over it, nor will it respond to
    // mouse clicks.
    //
    // @see listGridRecord.enabled
    // @example disabledRows
    // @visibility external
    //<
    recordEnabledProperty: "enabled",

    //> @attr listGridRecord.enabled (boolean : null : IR)
    //
    // Default property name denoting whether this record is enabled. Property name may be modified
    // for some grid via +link{listGrid.recordEnabledProperty}.
    //
    // @visibility external
    // @example disabledRows
    //<

    //> @attr listGrid.canExpandRecordProperty (string : "canExpand" : IR)
    // Property name on a record that will be checked to determine whether a record can be expanded.
    //
    // @see listGridRecord.canExpand
    // @group expansionField
    // @visibility external
    //<
    canExpandRecordProperty: "canExpand",

    //> @attr listGridRecord.canExpand (boolean : null : IR)
    //
    // Default property name denoting whether this record can be expanded. Property name may be 
    // modified for some grid via +link{listGrid.canExpandRecordProperty}.
    //
    // @group expansionField
    // @visibility external
    //<

    //> @attr listGridRecord.isSeparator (boolean : null : IR)
    //
    // Default property name denoting a separator row.<br>
    // When set to <code>true</code>, defines a horizontal separator in the listGrid
    // object. Typically this is specified as the only property of a record object, since a
    // record with <code>isSeparator:true</code> will not display any values.<br>
    // Note: this attribute name is governed by +link{ListGrid.isSeparatorProperty}.
    // @visibility external
    //<

    //> @attr listGridRecord.customStyle (CSSStyleName : null : IRW)
    // Name of a CSS style to use for all cells for this particular record.  
    // <P>
    // Note that using this property assigns a single, fixed style to the record, so rollover
    // and selection styling are disabled.  To provide a series of stateful styles for a record
    // use +link{listGridRecord._baseStyle} instead.
    // <P>
    // See +link{listGrid.getCellStyle()} for an overview of various ways to customize styling,
    // both declarative and programmatic.
    // <P>
    // If this property is changed after draw(), to refresh the grid call
    // +link{listGrid.refreshRow()} (or +link{listGrid.markForRedraw()} if several rows are
    // being refreshed).
    // <P>
    // If your application's data uses the "customStyle" attribute for something else, the
    // property name can be changed via +link{listGrid.recordCustomStyleProperty}.
    //
    // @visibility external
    //<

    //> @attr listGridRecord._baseStyle (CSSStyleName : null : IRW)
    // Name of a CSS style to use as the +link{listGrid.baseStyle} for all cells for this
    // particular record.  
    // <P>
    // The styleName specified with have suffixes appended to it as the record changes state
    // ("Over", "Selected" and so forth) as described by +link{listGrid.getCellStyle()}.  For a
    // single, fixed style for a record, use +link{listGridRecord.customStyle} instead.
    // <P>
    // See +link{listGrid.getCellStyle()} for an overview of various ways to customize styling,
    // both declarative and programmatic.
    // <P>
    // If this property is changed after draw(), to refresh the grid call
    // +link{listGrid.refreshRow()} (or +link{listGrid.markForRedraw()} if several rows are
    // being refreshed).
    // <P>
    // If your application's data uses the "_baseStyle" attribute for something else, the
    // property name can be changed via +link{listGrid.recordBaseStyleProperty}.
    //
    // @visibility external
    //<
    
    //> @attr listGridRecord.singleCellValue (HTML : null : IRW)
    // Default property name denoting the single value to display for all fields of this row.
    // If this property is set for some record, the record will be displayed as a single 
    // cell spanning every column in the grid, with contents set to the value of this
    // property.<br>
    // Note: this attribute name is governed by +link{ListGrid.singleCellValueProperty}.
    // @visibility external
    //<


    //> @attr listGridRecord.canDrag (boolean : null : IR)
    //
    // When set to <code>false</code>, this record cannot be dragged. If canDrag is false for
    // any record in the current selection, none of the records will be draggable.
    //
    // @visibility external
    //<

    //> @attr listGridRecord.canAcceptDrop (boolean : null : IR)
    //
    // When set to <code>false</code>, other records cannot be dropped on (i.e., inserted
    // via drag and drop) immediately before this record.
    //
    // @visibility external
    //<

	//> @attr listGridRecord.linkText (string : null : IRW)
    //
    //  The HTML to display in this row for fields with fieldType set to link. This overrides
    //  +link{attr:listGridField.linkText}.
    //
    //  @see type:ListGridFieldType
    //  @see type:FieldType
    //  @see attr:listGridField.linkText
    //  @see attr:listGrid.linkTextProperty
	//  @group  display_values
	//  @visibility external
	//<
    
	// Animation
    // ---------------------------------------------------------------------------------------
    // These apply to ListGrid grouping, which basically makes the data model into a Tree where animation
    // is applied for folder open/close.
    
    //> @attr listGrid.animateFolders (boolean : true : IRW)
    // If true, when folders are opened / closed children will be animated into view.
    // <P>
    // For a ListGrid, this property applies when +link{ListGrid.canGroupBy,grouping} is enabled.
    // @group animation
    // @visibility animation
    //<
    animateFolders:true,
    
    //> @attr listGrid.animateFolderMaxRows (integer : null : IRW)
    // If +link{animateFolders} is true for this grid, this number can be set to designate
    // the maximum number of rows to animate at a time when opening / closing a folder.
    // <P>
    // For a ListGrid, this property applies when +link{ListGrid.canGroupBy,grouping} is enabled.
    // @see getAnimateFolderMaxRows()
    // @group animation 
    // @visibility external
    //<

    //> @attr listGrid.animateFolderTime (number : 100 : IRW)
    // When animating folder opening / closing, if +link{treeGrid.animateFolderSpeed} is not
    // set, this property designates the duration of the animation in ms.
    // <P>
    // For a ListGrid, this property applies when +link{ListGrid.canGroupBy,grouping} is enabled.
    // @group animation
    // @visibility animation
    // @see listGrid.animateFolderSpeed
    //<
    animateFolderTime:100,

    //> @attr listGrid.animateFolderSpeed (number : 3000 : IRW)
    // When animating folder opening / closing, this property designates the speed of the
    // animation in pixels shown (or hidden) per second. Takes precedence over the 
    // +link{treeGrid.animateFolderTime} property, which allows the developer to specify a
    // duration for the animation rather than a speed.
    // <P>
    // For a ListGrid, this property applies when +link{ListGrid.canGroupBy,grouping} is enabled.
    // @group animation
    // @visibility animation
    // @see listGrid.animateFolderTime
    //<    
    animateFolderSpeed:3000,
    
    //> @attr listGrid.animateFolderEffect (AnimationAcceleration : null : IRW)
    // When animating folder opening / closing, this property can be set to apply an
    // animated acceleration effect. This allows the animation speed to be "weighted", for
    // example expanding or collapsing at a faster rate toward the beginning of the animation
    // than at the end.
    // <P>
    // For a ListGrid, this property applies when +link{ListGrid.canGroupBy,grouping} is enabled.
    // @group animation
    // @visibility animation
    //<    

    //> @attr listGrid.animateRowsMaxTime (number : 1000 : IRW)
    // If animateFolderSpeed is specified as a pixels / second value, this property will cap
    // the duration of the animation.
    // <P>
    // For a ListGrid, this property applies when +link{ListGrid.canGroupBy,grouping} is enabled.
    // @group animation
    // @visibility animation_advanced
    //<    
    animateRowsMaxTime:1000,

    // external: doc'd on TreeGrid
    shouldAnimateFolder : function (folder) {
        if (!this.animateFolders || !this.isDrawn()) return false;
        
        var children = this.data.isFolder(folder) ? this.data.getOpenList(folder) : null;
        
        // No children - bit arbitrary whether we "animate" or not!
        
        if (children == null || children.length <= 1) return false;
        return (children.length <= this.getAnimateFolderMaxRows());
    },
    
    // external: doc'd on TreeGrid
    getAnimateFolderMaxRows : function () {
        var maxRows = this.animateFolderMaxRows;
        if (maxRows == null) {
            var vfRs = this.body ? this.body._getViewportFillRows() : [0,0];
            maxRows = Math.min(75, (vfRs[1]-vfRs[0]) * 3);
        }
        return maxRows
    },
    
	
	// DataBinding
	// ----------------------------------------------------------------------------------------

	//>	@attr listGrid.fields (Array of ListGridField : null : [IRW])
	// An array of field objects, specifying the order, layout, formatting, and
	// sorting behavior of each field in the listGrid object.  In ListGrids, the fields
	// array specifies columns.  Each field in the fields array is a ListGridField object. 
    // Any listGrid that will display data should have at least one visible field.
    // <p>
    // If +link{ListGrid.dataSource} is also set, this value acts as a set of overrides as
    // explained in +link{attr:DataBoundComponent.fields}.
    //
	// @see    ListGridField
	// @see    setFields()
    // @group databinding
	// @visibility external
    // @example listGridFields
    // @example mergedFields
	//<
    
    //> @attr listGrid.defaultFields (Array of ListGridField Properties : null : IRA)
    // An array of listGrid field configuration objects.  When a listGrid is initialized, if this
    // property is set and there is no value for the <code>fields</code> attribute, this.fields will
    // be defaulted to a generated array of field objects duplicated from this array.
    // <P>
    // This property is useful for cases where a standard set of fields will be displayed
    // in multiple listGrids - for example a subclass of ListGrid intended to display a particular
    // type of data:<br>
    // In this example we would not assign a single +link{listGrid.fields} array directly to the
    // class via <code>addProperties()</code> as every generated instance of this class would
    // then point to the same fields array object. This would cause unexpected behavior such as
    // changes to the field order in one grid effecting other grids on the page.<br>
    // Instead we could use <code>addProperties()</code> on our new subclass to set
    // <code>defaultFields</code> to a standard array of fields to display. Each generated instance
    // of the subclass would then show up with default fields duplicated from this array.
    // @visibility external
    //<
    
	//>	@attr	listGrid.dataSource		(DataSource or ID : null : IRW)
    // @include dataBoundComponent.dataSource
	//<
    
	//> @attr listGrid.autoFetchDisplayMap (boolean : true : [IRW])
	// If true, for fields where +link{listGridField.optionDataSource} is specified,
    // a valueMap will be automatically created by making a +link{dataSource.fetchData()} call
    // against the specified dataSource and extracting a valueMap from the returned records
    // based on the displayField and valueField.
    // <P>
    // If set to false, valueMaps will not be automatically fetched.  In this case, setting
    // field.optionDataSource is effectively a shortcut for setting optionDataSource on
    // the editor via +link{listGridField.editorProperties}.
    // <P>
    // Can also be disabled on a per-field basis with +link{listGridField.autoFetchDisplayMap}.
    //
	// @group display_values
	// @see listGridField.autoFetchDisplayMap
	// @see listGridField.optionDataSource
    // @visibility external
	//<
    autoFetchDisplayMap:true,

    //> @attr ListGrid.saveLocally (boolean : null : IRA)
    // For grids with a specified +link{ListGrid.dataSource}, this property can be set to 
    // <code>true</code> to avoid the grid from attempting to save / retrieve data from the
    // server.  In this case the grid's data should be specified as an array via 
    // the +link{listGrid.data} attribute, and the datasource will simply act as a schema to
    // describe the set of fields visible in the grid. 
    // +link{listGrid.canEdit,Inline edits}, or removals via the
    // +link{listGrid.canRemoveRecords} mechanism will update the local data array rather than
    // attempting to perform operations against the dataSource.
    // @see useRemoteValidators
    // @visibility external
    // @group databinding
    //<
    
    
    //> @attr ListGrid.saveRequestProperties (DSRequest Properties : null : IRWA)
    // For editable grids with a specified +link{listGrid.dataSource}, where 
    // +link{listGrid.saveLocally} is false, this attribute may be used to specify standard
    // DSRequest properties to apply to all save operations performed by this grid (whether
    // triggered by user interaction, or explicit saveEdits or saveAllEdits call).
    // <P>
    // An example usage would be to customize the prompt displayed while saving is in progress
    // if +link{listGrid.waitForSave} is true.
    // <P>
    // Note that for more advanced customization of save operations,
    // +link{dataBoundComponent.addOperation} and +link{dataBoundComponent.updateOperation}
    // are available to developers, allowing specification of an explicit +link{operationBinding} 
    // for the add / update operation performed on save. 
    //
    // @visibility external
    // @group dataBinding
    // @group editing
    //<
    
    //> @attr ListGrid.useRemoteValidators (boolean : null : IRWA)
    // If +link{listGrid.saveLocally} is specified, but this grid is bound to a DataSource which
    // includes remote field validators, by default edits will be saved synchronously and
    // these validators will not be executed.<br>
    // Set this property to <code>true</code> to ensure these remote validators are called when 
    // saving edits in saveLocally mode. Note that since these remote validators need to run on 
    // the server, saving with this property set is asynchronous, even though the data that
    // ultimately gets updated is already present on the client.
    // @visibility external
    // @group databinding
    //<

	//>	@attr listGrid.useAllDataSourceFields (boolean : null : IRW)
    // @include dataBoundComponent.useAllDataSourceFields
    // @group databinding
	//<

	//>	@attr listGrid.showDetailFields (boolean : true : IR)
    // Whether to include fields marked <code>detail:true</code> from this component's 
    // <code>DataSource</code>.
    // <P>
    // Fields may also be included directly in this component's <code>fields</code> array in
    // which case they will be present regardless of the <code>detail</code> attribute.
    // <p>
    // When this property is <code>true</code>, <code>ListGrid</code>s will include all 
    // detail fields, but they will be initially hidden.  The user may show
    // these fields via the default header context menu (+link{listGrid.showHeaderContextMenu}).
    // Setting this property to false will completely exclude all detail fields from the list
    // grid's fields array, such that they cannot be shown by the user or programmatically.
    // <P>
    // To override the visibility of individual fields, use the standard
    // +link{listGrid.showField()}, +link{listGrid.hideField()} and +link{listGridField.showIf}
    // APIs, for example, set showIf:"true" to show a detail field initially.
    // 
    // @group databinding
    // @visibility external
    //<
    showDetailFields:true,
    
    //> @attr ListGrid.titleField (string : see below : IRW)
    // Best field to use for a user-visible title for an individual record from this grid.
    // If +link{ListGrid.dataSource} is non null, this property may be specified on the 
    // dataSource instead.
    // <p>
    // If not explicitly set, titleField looks for fields named "title", "name", and "id" 
    // in that order.  If a field exists with one of those names, it becomes the titleField.  
    // If not, then the first field is designated as the titleField.
    //  @visibility external
    //<

    
    //> @attr listGrid.dataProperties (object : null :IRWA)
    // For databound ListGrids, this attribute can be used to customize the +link{ResultSet}
    // object created for this grid when data is fetched
    // @group databinding
    // @visibility external
    //<
    
    // Grouping
    // ---------------------------------------------------------------------------------------
    
    //> @object groupNode
    //
    // An auto-generated subclass of +link{TreeNode} representing the group nodes
    // in a grouped listgrid. 
    //
    // @see listGrid.groupBy()
    // @treeLocation Client Reference/Grids/ListGrid
    // @group grouping
    // @visibility external    
    //< 
    
    //> @attr groupNode.groupMembers (Array of ListGridRecord or GroupNode : see below : R)
    // Array of ListGridRecord that belong to this group, or, for multi-field grouping, array
    // of groupNodes of subgroups under this groupNode.
    // 
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    
    //> @attr groupNode.groupTitle (HTML : see below : R)
    // The computed title for the group, which results from +link{listGridField.getGroupTitle()}
    // 
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    
    //> @attr groupNode.groupValue (any : see below : R)
    // The value from which groups are computed for a field, 
    // which results from +link{listGridField.getGroupValue()}
    // 
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    
    //> @attr listGrid.originalData (object : null : R)
    // When grouped, a copy of the original ungrouped data.
    // 
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    
    //> @attr listGrid.groupTree (AutoChild : null : R)
    // The data tree that results from a call to  +link{listGrid.groupBy()}.
    // This will be a +link{ResultTree} if +link{listGrid.dataSource} is
    // present, otherwise it will be a +link{Tree}.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    
    //> @type GroupStartOpen 
    // Possible values for the state of ListGrid groups when groupBy is called
    //
    // @value "all" open all groups
    // @value "first" open the first group
    // @value "none" start with all groups closed
    // @visibility external
    //<
    
    //> @attr listGrid.groupStartOpen (GroupStartOpen || Array : "first" : IRW)
    // Describes the default state of ListGrid groups when groupBy is called. 
    //
    // Possible values are:
    // <ul>
    // <li>"all": open all groups
    // <li>"first": open the first group
    // <li>"none": start with all groups closed 
    // <li>Array of values that should be opened
    // </ul>
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    groupStartOpen:"first",
    
    //> @attr listGrid.canCollapseGroup (Boolean : true : IR)
    // Can a group be collapsed/expanded? When true a collapse/expand icon is shown
    // (+link{groupIcon,groupIcon}) and clicking the icon or double-clicking the group title
    // will collapse or expand the group.
    //
    // When false the group icon is not shown and double-clicking on the title does
    // not change group state. Additionally +link{groupStartOpen,groupStartOpen} is 
    // initialized to "all".
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    canCollapseGroup:true,

    //> @attr listGrid.groupTitleField (String : null : IR)
    // When a list grid is +link{listGrid.groupBy(),grouped}, each group shows
    // under an auto generated header node. By default the title of the group will be
    // shown, with a hanging indent in this node, and will span all columns in the grid.
    // Setting this property causes the titles of auto-generated group nodes to appear as
    // though they were values of the designated field instead of spanning all columns
    // and record values in the designated groupTitleField will appear indented under 
    // the group title in a manner similar to how a TreeGrid shows a Tree.
    // <P>
    // Note if +link{listGrid.showGroupSummaryInHeader} is true, the header nodes will not show
    // a single spanning title value by default - instead they will show the summary values for
    // each field. In this case, if groupTitleField is unset, a 
    // +link{listGrid.showGroupTitleColumn,groupTitleColumn} can be automatically generated to
    // show the title for each group.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    
    getGroupTitleField : function () {
        return this.groupTitleField;
    },
    
    //> @attr listGrid.showGroupTitleColumn (boolean : true : IR)
    // If this grid is +link{listGrid.groupBy(),grouped} and +link{listGrid.showGroupSummaryInHeader}
    // is true, instead of group header nodes showing up with a single cell value spanning the full
    // set of columns, summaries for each field will show up in the appropriate columns of the 
    // header node.
    // <P>
    // In this case there are 2 options for where the group title will show up. Developers may
    // specify an existing field to put the title values into via +link{listGrid.groupTitleField}.
    // If no groupTitleField is specified, this property may be set to <code>true</code>
    // which causes a <code>groupTitleColumn</code> to be automatically generated.
    // Each group header will show the group title in this column (records within the group will
    // not show a value for this column). The column appears in the leftmost position within the
    // grid (unless +link{listGrid.showRowNumbers} is true, in which case this column shows up
    // in the second-leftmost position), and by default will auto-fit to its data.
    // <P>
    // To customize this field, developers may modify 
    // +link{listGrid.groupTitleColumnProperties} 
    // <var class="SmartClient">or
    // +link{listGrid.groupTitleColumnDefaults} at the class level.</var>
    // @visibility external
    //<
    showGroupTitleColumn:true,
   
    //> @attr listGrid.groupTitleColumnProperties (ListGridField properties : null : IR)
    // Custom properties for the automatically generated <code>groupTitleColumn</code>.
    // <P>
    // See +link{listGrid.showGroupTitleColumn} for an overview of the groupTitleColumn.
    // @visibility external
    //<
    //groupTitleColumnProperties:null,
    
    //> @attr listGrid.groupTitleColumnDefaults (ListGridField properties : object : IR)
    // Default properties for the automatically generated <code>groupTitleColumn</code>.
    // Default object includes properties to enable autoFitWidth to group title values.
    // <P>
    // To modify the behavior or appearance of this column, developers may set
    // +link{listGrid.groupTitleColumnProperties} at the instance level, or override this
    // object at the class level. If overriding this object, we recommend using
    // +link{class.changeDefaults()} rather than replacing this object entirely.
    // <P>
    // See +link{listGrid.showGroupTitleColumn} for an overview of the groupTitleColumn.
    // @visibility external
    //<
    groupTitleColumnDefaults:{
        canEdit:false,
        canFilter:false,
        canHide:false,
        canReorder:false,
        showDefaultContextMenu:false,
        showHeaderContextMenuButton:false,
        autoFreeze:true,
        
        sortNormalizer:function (recordObject,fieldName,context) {
            return recordObject.groupTitle;
        },
        
        autoFitWidth:true,
        autoFitWidthApproach:"value",
        title:"&nbsp;"
    },
    
    
    // We actually show the special group title column iff
    // - we're showing the group summary in the header
    // - we have no explicitly specified group title field
    // - the showGroupTitleColumn flag is true
    showingGroupTitleColumn : function () {
        return (this.isGrouped && this.showGroupSummary && this.showGroupSummaryInHeader 
                && this.showGroupTitleColumn && this.getGroupTitleField() == null);
    },
    
    // groupTitleColumnName: This could be modified to display an actual field within the
    // grid data, but the developer might as well use groupTitleField instead. 
    // Leaving unexposed for now.
    groupTitleColumnName:"groupTitle",
    
    getGroupTitleColumn : function () {
        var grid = this;
        var groupTitleColumn = isc.addProperties(
            {   _isGroupTitleColumn:true,
                // 'grid' available through closure
                getAutoFreezePosition: function () { return grid.getGroupTitleColumnPosition() }
            }, 
            this.groupTitleColumnDefaults, 
            this.groupTitleColumnProperties
        );
     
        if (groupTitleColumn.name == null) {
            groupTitleColumn.name = this.groupTitleColumnName;
        }
        return groupTitleColumn;
    },
    
    getGroupTitleColumnPosition : function () {
        // This is really just a sanity check - we don't expect to be calling this method when
        // we're not showing the special groupTitleColumn
        if (!this.showingGroupTitleColumn()) return -1;

        
        return (this.showRowNumbers ? 1 : 0);
    },
    
    singleCellGroupHeaders : function () {
        if (this.getGroupTitleField() != null) return false;
        if (this.showGroupSummary && this.showGroupSummaryInHeader) return false;
        
        return true
    },
    
    //> @attr listGrid.showGroupSummaryInHeader (boolean : false : IRW)
    // If this grid is +link{listGrid.groupBy(),grouped}, and +link{listGrid.showGroupSummary}
    // is true, setting this property causes field summary values for each group to be displayed
    // directly in the group header node, rather than showing up at the bottom of each
    // expanded group.
    // <P>
    // Note that this means the group header node will be showing multiple field values
    // rather than the default display of a single cell spanning all columns containing the
    // group title. Developers may specify an explicit +link{listGrid.groupTitleField}, or
    // rely on the automatically generated +link{listGrid.showGroupTitleColumn,groupTitleColumn}
    // to have group titles be visible in as well as the summary values.
    // <P>
    // Also note that multi-line group summaries are not supported when showing
    // the group summary in the group header. If multiple 
    // +link{listGridField.summaryFunction,field summary functions} are defined for some field
    // only the first will be displayed when this property is set to true.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    showGroupSummaryInHeader:false,
    
    //> @method listGridField.getGroupValue()
    // Return the value which records should be grouped by.  
    // <P>
    // All records for which getGroupValue() returns the same value appear in the same
    // group.  Default is the result of +link{listGrid.getCellValue}.
    // <P>
    // While any type of value may be returned, avoiding the use of string values may
    // result in improved performance. In this case, +link{listGridField.getGroupTitle()}
    // may be implemented to map a numeric group value into a legible string.
    //
    // @param   value (any)   raw value for the cell, from the record for the row
    // @param   record (ListGridRecord) 
    //   Record object for the cell. Note: If this is a new row that has not been saved, in an 
    //   editable grid, it has no associated record object. In this case the edit values will
    //   be passed in as this parameter (see +link{listGrid.getEditValues()})
    // @param   field (Object)    Field object for which to get group value
    // @param   fieldName (String)    The name of the field
    // @param   grid (ListGrid) the ListGrid displaying the cell
    // @return (any) Group value to which this record belongs
    //
    // @see listGrid.groupBy()
    // @see listGridField.getGroupTitle()
    // @group grouping
    // @visibility external
    // @example customGrouping
    //<
    
    //> @method listGridField.getGroupTitle()
    // Return the title that should be shown to the user for the group with the
    // <code>groupValue</code> passed as a parameter.
    // <P>
    // Default title is the groupValue itself. 
    //
    // @group grouping
    //
    // @param   groupValue (any)   the value from the group is created, the result of 
    //  +link{listGridField.getGroupValue()}
    // @param   groupNode (groupNode) the node in the grid containing the group. 
    // @param   field (Object)    Field object for which to get group value
    // @param   fieldName (String)    The name of the field
    // @param   grid (ListGrid) the ListGrid displaying the cell
    // @return (any) Group value to which this record belongs
    //
    // @see listGrid.groupBy()
    // @see listGridField.getGroupValue()
    // @visibility external
    // @example customGrouping
    //<

    //> @attr listGridField.groupingModes (ValueMap : null : IR)
    // If set, provides several possible styles of grouping that are valid for this field.  For
    // example, a field of type:"date" might be able to be grouped by week, month, or by day of
    // week.
    // <P>
    // If <code>groupingModes</code> are present and
    // +link{listGrid.canGroupBy,grouping is enabled}, the menu for this field includes a
    // submenu of possible grouping modes generated from the <code>groupingModes</code> valueMap.
    // When the user selects a particular grouping mode,
    // +link{listGridField.groupingMode,field.groupingMode} is set to the user's chosen mode,
    // and this choice can be detected via the <code>field</code> parameter to
    // +link{listGridField.getGroupValue()} in order to provide different modes of grouping.
    // <P>
    // The user may also choose to group records without specifying a grouping mode, in this case,
    // the +link{listGridField.defaultGroupingMode} is used.
    // <P>
    // Note that <code>getGroupValue</code>, <code>groupingModes</code> et al can be specified on
    // +link{SimpleType} declaration, and the different grouping modes that are offered
    // automatically for various common types are defined this way.
    //
    // @group grouping
    // @visibility external
    //<

    //> @attr listGridField.groupingMode (identifier : null : IR)
    // For a field that allows multiple +link{listGridField.groupingModes,grouping modes}, 
    // the current grouping mode.
    // <P>
    // This property is set when a user chooses a particular grouping mode, and may be set on
    // ListGrid creation to affect the initial grouping.
    //
    // @group grouping
    // @visibility external
    //<

    //> @attr listGridField.defaultGroupingMode (identifier : null : IR)
    // Default groupingMode used when the user does not specify a mode or grouping is triggered
    // programmatically and +link{listGridField.groupingMode,field.groupingMode} is unset.
    // See +link{listGridField.groupingModes,field.groupingModes}.
    //
    // @group grouping
    // @visibility external
    //<

    //> @attr listGridField.groupPrecision (integer : null : IR)
    // For fields of type:"float" or derived from float, number of digits after the decimal point
    // to consider when grouping.
    // <P>
    // For example, <code>groupPrecision:2</code> indicates that 45.238 and 45.231 group together,
    // but 45.22 and 45.27 are separate.
    // <P>
    // See also +link{listGridField.groupGranularity,groupGranularity} for grouping by broader
    // ranges.
    //
    // @group grouping
    // @visibility external
    //<

    //> @attr listGridField.groupGranularity (integer : null : IR)
    // Granularity of grouping for numeric fields.
    // <P>
    // Groups will be formed based on ranges of values of size <code>groupGranularity</code>.  For
    // example, if groupGranularity were 1000, groups would be 0-1000, 1000-2000, etc.
    //
    // @group grouping
    // @visibility external
    //<

    //> @attr listGridField.canHilite (boolean : null : IRW)
    // Determines whether this field can be hilited.  Set to false to prevent this
    // field from appearing in HiliteEditor.
    // 
    // @visibility external
    //<

    //> @attr listGridField.canGroupBy (Boolean : true : IRW)
    // Determines whether this field will be groupable in the header context menu.
    // 
    // @see listGrid.groupBy()
    // @visibility external
    //<

    //> @attr listGridField.canSortClientOnly (Boolean : false : IRW)
    // When true, this field can only be used for sorting if the data is entirely client-side.
    // 
    // @visibility external
    //<

    //> @attr listGridField.showDefaultContextMenu (Boolean : true : IRW)
    // When set to false, this field will not show a context menu in it's header.
    // 
    // @visibility external
    //<

    //> @attr listGridField.filterOperator (OperatorId : null : IR)
    // With the +link{listGrid.showFilterEditor,FilterEditor} showing, the +link{Operator} to 
    // use when matching values for this field.
    // <P>
    // Note that you can set all FilterEditor fields to default to either substring or exact 
    // match via +link{listGrid.autoFetchAsFilter}, but if you want a mix of exact vs substring 
    // match on different fields, you need to use this property, and your ListGrid will produce
    // +link{AdvancedCriteria} rather than the simpler +link{Criteria} format.  This is
    // automatically and transparently handled by the SmartClient Server's SQLDataSource and
    // HibernateDataSource in Power Edition or above, but if you have your own filtering
    // implementation, it will need to be able to handle AdvancedCriteria.
    // @visibility external
    //<

    //> @attr listGridField.canExport (Boolean : null : IR)
    //	Dictates whether the data in this field be exported.  Explicitly set this
    //  to false to prevent exporting.  Has no effect if the underlying 
    //  +link{dataSourceField.canExport, dataSourceField} is explicitly set to 
    //  canExport: false.
    //
    // @visibility external
    //<

    //> @attr listGridField.exportRawValues (Boolean : null : IR)
    //	Dictates whether the data in this field should be exported raw by 
    // +link{DataBoundComponent.exportClientData, exportClientData()}.  If set to true for a 
    // field, the values in the field-formatters will not be executed for data in this field.
    // Decreases the time taken for large exports.
    //
    // @visibility external
    //<
    
    //> @attr listGrid.groupNodeStyle (String : "groupNode" : IRW)
    // The CSS style that group rows will have
    //
    // @group grouping
    // @see group:grouping
    // @visibility external
    //<
    groupNodeStyle: "groupNode",
	
    //> @attr listGrid.groupIcon (SCImgURL : "[SKINIMG]/TreeGrid/opener.gif" : IRW)
    // The URL of the base icon for the group icons in this treegrid.
    //
    // @group grouping
    // @see group:grouping
    // @visibility external
    //<
    groupIcon: "[SKINIMG]/TreeGrid/opener.gif",
    
    //> @attr listGrid.groupIconSize (Number : 16 : IRW)
    // Default width and height of group icons for this ListGrid.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    groupIconSize: 16,
    
    //> @attr listGrid.groupIndentSize (Number : 20 : IRW)
    // Default number of pixels by which to indent subgroups relative to parent group.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @see listGrid.getGroupNodeHTML
    // @visibility external
    //<
    groupIndentSize: 20,
    
    //> @attr listGrid.groupLeadingIndent (Number : 10 : IRW)
    // Default number of pixels by which to indent all groups.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @see listGrid.getGroupNodeHTML
    // @visibility external
    //<
    groupLeadingIndent: 10,

    //> @attr listGrid.canGroupBy (Boolean : true : IRW)
    // If false, grouping via context menu will be disabled.
    //
    // @group grouping
    // @see listGrid.groupBy()
    // @visibility external
    //<
    canGroupBy: true,

    //> @attr listGrid.groupByMaxRecords (Number : 1000 : IRW)
    // Maximum number of records to which a groupBy can be applied. If there are more records,
    // grouping will not be available via the default header context menu, and calls to 
    // +link{listGrid.groupBy()} will be ignored.
    // 
    // @group grouping
    // @visibility external
    // @see     groupBy
    //<
    groupByMaxRecords: 1000,
    
    //> @attr listGrid.isGrouped (Boolean : false : R)
    // True if this listgrid is grouped, false otherwise
    //
    // @group grouping
    // @visibility external
    // @see     groupBy
    //<
    
    //> @attr listGrid.nullGroupTitle (String : '-none-' : IRW)
    // Default alias to use for groups with no value
    //
    // @group grouping
    // @visibility external
    // @see     groupBy
    //<
    nullGroupTitle: "-none-",
    
    //> @attr listGrid.groupByField ( String or Array of String : see below : IR)
    // List of fields to group grid records. If only a single field is used, that field
    // may be specified as a string. After initialization, use +link{listGrid.groupBy()} 
    // to update the grouping field list, instead of modifying groupByField directly.
    // @group grouping
    // @visibility external
    // @see     groupBy
    // @example dynamicGrouping
    //<
    

    // ----------------------
    // Value icons
    // The valueIcons object is a mapping between values and image URLs - when specified
    // we show the valueIcon image either next to, or instead of the normal cell value.
    
    //> @attr listGridField.valueIcons (object : null : IRW)
    // This property is a mapping between data values to +link{SCImgURL, SCImgURLs} for icons to display
    // for those data values.  For example, given a field named "status" with possible values
    // "Normal", "Slow", "Offline", the follow definition would show various icons for that
    // field:
    // <P>
    // <pre>
    // fields : [
    //     { name:"status", 
    //       valueIcons: {
    //           Normal : "greenIcon.png",
    //           Slow : "yellowIcon.png",
    //           Offline : "redIcon.png"
    //       }
    //     },
    //     ... other fields ...
    // ]
    // </pre>
    // <P>
    // <code>valueIcons</code> can either be displayed alongside the normal value or can
    // replace the normal field value so that only the icon is shown.  See
    // +link{listGridField.showValueIconOnly}.
    // <P>
    // If inline editing is enabled for this field, editors displayed for this field will also
    // show valueIcons.  This may be overridden by explicitly setting
    // +link{listGridField.editorValueIcons}.
    // <P>
    // Note that the following attributes related to valueIcon styling will also be picked up
    // by the editor from the ListGridField object unless explicitly specified via the
    // equivalent <code>editor_</code> attributes:<br>
    // +link{listGridField.valueIconWidth}<br>
    // +link{listGridField.valueIconHeight}<br>
    // +link{listGridField.valueIconSize}<br>
    // +link{listGridField.valueIconLeftPadding}<br>
    // +link{listGridField.valueIconRightPadding}<br>            
    // +link{listGridField.imageURLPrefix}<br>    
    // +link{listGridField.imageURLSuffix}
    //
    // @group imageColumns
    // @visibility external
    //<
    
    //> @attr listGrid.valueIconSize (number : 16 : IRW)
    // Default width and height of value icons for this ListGrid.
    // Can be overridden at the listGrid level via explicit +link{ListGrid.valueIconWidth} and
    // +link{ListGrid.valueIconHeight}, or at the field level via +link{ListGridField.valueIconSize},
    // +link{ListGridField.valueIconWidth} and {ListGridField.valueIconHeight}
    // @visibility external
    // @group imageColumns
    // @see ListGrid.valueIconWidth
    // @see ListGrid.valueIconHeight
    // @see ListGridField.valueIconSize
    //<
    valueIconSize:16,
    
    
    //> @attr listGrid.valueIconWidth (number : null : IRW)
    // Width for value icons for this listGrid.
    // Overrides +link{ListGrid.valueIconSize}.
    // Can be overridden at the field level
    // @group imageColumns
    // @visibility external
    //<
    
    //> @attr listGrid.valueIconHeight (number : null : IRW)
    // Height for value icons for this listGrid.
    // Overrides +link{ListGrid.valueIconSize}.
    // Can be overridden at the field level
    // @group imageColumns
    // @visibility external
    //<
    
    //> @attr listGridField.valueIconSize (number : null : IRW)
    // Default width and height of value icons in this field.
    // Takes precedence over valueIconWidth, valueIconHeight and valueIconSize specified at
    // the ListGrid level.
    // Can be overridden via +link{ListGridField.valueIconWidth} and {ListGridField.valueIconHeight}
    // @visibility external
    // @group imageColumns
    // @see ListGrid.valueIconSize
    // @see ListGridField.valueIconWidth
    // @see ListGridField.valueIconHeight
    //<
    
    //> @attr listGridField.valueIconWidth (number : null : IRW)
    // Width for value icons for this listGrid field.
    // Overrides +link{ListGrid.valueIconSize}, +link{ListGrid.valueIconWidth}, and
    // +link{ListGridField.valueIconSize}.
    // @group imageColumns
    // @visibility external
    //<
    
    //> @attr listGridField.valueIconHeight (number : null : IRW)
    // Height for value icons for this listGrid field.
    // Overrides +link{ListGrid.valueIconSize}, +link{ListGrid.valueIconHeight}, and
    // +link{ListGridField.valueIconSize}.
    // @group imageColumns
    // @visibility external
    //<
    
    //> @attr   listGridField.valueIconLeftPadding (number : null : IRW)
    // How much padding should there be on the left of valueIcons for this field
    // Overrides +link{listGrid.valueIconLeftPadding}
    // @group imageColumns
    // @see ListGridField.valueIcons
    // @visibility external
    //<
    
    //> @attr   listGridField.valueIconRightPadding (number : null : IRW)
    // How much padding should there be on the right of valueIcons for this field
    // Overrides +link{listGrid.valueIconRightPadding}
    // @group imageColumns
    // @see ListGridField.valueIcons
    // @visibility external
    //<
    
    //> @attr listGridField.editorValueIcons (object : null : IRW)
    // When some cell in this field is being edited, setting this property will specify the
    // value icons to display in the cell's editor. If unset, the editor's valueIcons
    // will be determined in the same way as it would be for a static cell.
    // @group imageColumns
    // @visibility external
    //<

    //> @attr listGridField.editorValueIconWidth (number : null : IRW)
    // When some cell in this field is being edited, setting this property will specify the
    // width for value icons in the cell's editor. If unset, the editor's valueIcon width and
    // height will be determined in the same way as it would be for a static cell.
    // @group imageColumns
    // @visibility external
    //<
    
    //> @attr listGridField.editorValueIconHeight (number : null : IRW)
    // When some cell in this field is being edited, setting this property will specify the
    // height for value icons in the cell's editor. If unset, the editor's valueIcon width and
    // height will be determined in the same way as it would be for a static cell.
    // @group imageColumns
    // @visibility external
    //<    
    
    //> @attr listGridField.showValueIconOnly (boolean : null : IRW)
    // If this field has a valueIcons property specified, setting this property causes 
    // the valueIcon for each value to be displayed in the cell without also showing the
    // record's value for the field.
    // <P>
    // If unset the default behavior is to show the icon only if an explicit valueMap is
    // specified as well in addition to a valueIcons map, otherwise show both the valueIcon and
    // value for the cell.
    // <P>
    // Note that if this field is editable +link{FormItem.showValueIconOnly} will be passed
    // through to editors displayed in this field.
    //
    // @group imageColumns
    // @see listGridField.valueIcons
    // @see listGridField.suppressValueIcon
    // @visibility external
    //<
    
    // NOTE: showValueIconOnly: the use cases are:
    // - represent a value as an icon only to minimize space
    // - show text, but add an icon as decoration, either to all values, or to emphasize some
    //   values for quicker scanning
    // The property 'showValueIconOnly' allows the developer to explicitly show the valueIcon
    // with or without text.  If showValueIconOnly is unset, we make the assumption that:
    // - if the field is *not* constrained to a fixed set of values (has no valueMap), there's 
    //   no way to have icons for all the values, so the purpose of the icons is to add
    //   emphasis to certain values [so we show both text and images]
    // - otherwise the developer has an icon for every possible value, so there is no need for
    //   the value to also be displayed - we size the field large enough to accommodate the icon
    //   only, and suppress the text.
        
    //> @attr   listGridField.suppressValueIcon (boolean : null : IRW)
    // If this field has a valueIcons property specified, setting this property to true will 
    // prevent the valueIcon being written out into this field's cells.
    //
    // @group imageColumns
    // @see listGridField.valueIcons
    // @see listGridField.showValueIconOnly
    // @visibility external
    //<
    
    //> @attr   listGridField.valueIconOrientation (string : null : IRW)
    // If we're showing a valueIcon for this field should it appear to the left or the right
    // of the text?  By default the icon will appear to the left of the textual value - 
    // set this to "right" to show the icon on the right of the text.
    // Has no effect if +link{listGridField.showValueIconOnly} is true
    // @visibility external
    // @group imageColumns
    //<
    
    
    //> @attr   listGrid.valueIconLeftPadding (number : 2 : IRW)
    // How much padding should there be on the left of valueIcons by default
    // Can be overridden at the field level
    // @group imageColumns
    // @see ListGridField.valueIcons
    // @visibility external
    //<
    valueIconLeftPadding:2,
    
    //> @attr   listGrid.valueIconRightPadding (number : 2 : IRW)
    // How much padding should there be on the right of valueIcons by default
    // @group imageColumns
    // Can be overridden at the field level    
    // @see ListGridField.valueIcons
    // @visibility external
    //<
    valueIconRightPadding:2,
    
    
    
    //> @attr   ListGridField.imageURLPrefix (string : null : IRWA)
    // If this field has type [+link{type:ListGridFieldType}] set to <code>"image"</code>
    // and the URL for the image displayed is not absolute, the path of the URL will be relative 
    // to this string<br>
    // Alternatively, if this field displays any valueIcons, this prefix will be applied to
    // the beginning of any +link{ListGridField.valueIcons} when determining the
    // URL for the image.
    // @group imageColumns
    // @visibility external
    // @example imageType
    //<
    
    //> @attr   ListGridField.imageURLSuffix (string : null : IRWA)
    // If any cells in this field are showing a value icon (see: +link{ListGridField.valueIcons})
    // or this is has +link{type:ListGridFieldType} set to <code>"image"</code>, this the value
    // of this property will be appended to the end of the URL for the icon displayed.<br>
    // Typical usage might be to append a file type such as <code>".gif"</code> to the
    // filename of the image.<br>
    // For editable fields, this property will also be passed through to any editors as
    // +link{FormItem.imageURLSuffix}.
    // @group imageColumns
    // @visibility external
    // @example imageType
    //<    
    
    //> @attr   ListGridField.editorImageURLPrefix (string : null : IRWA)
    // When some cell in this field is being edited, this property can be used to apply 
    // an explicit +link{FormItem.imageURLPrefix} to the editor in question. 
    // This can be used to modify the valueIcons within the editor.<br>
    // If unset, but +link{ListGridField.imageURLPrefix} is specified, that will be used
    // instead.
    // @group editing
    // @visibility external
    //<
    
    //> @attr   ListGridField.editorImageURLSuffix (string : null : IRWA)
    // When some cell in this field is being edited, this property can be used to apply 
    // an explicit +link{FormItem.imageURLSuffix} to the editor in question. 
    // This can be used to modify the valueIcons within the editor.<br>
    // If unset, but +link{ListGridField.imageURLPrefix} is specified, that will be used
    // instead.
    // @group editing
    // @visibility external
    //<

  	//>	@attr	listGrid.imageSize (number : 16 : IRW)
    // Default size of thumbnails shown for fieldTypes image and imageFile.  Overrideable on a
    // per-field basis via +link{attr:ListGridField.imageSize} or
    // +link{attr:ListGridField.imageWidth}/+link{attr:ListGridField.imageHeight}
    // 
    // @group imageColumns
    // @visibility external
	//<
    imageSize: 16,

  	//>	@attr	listGridField.imageSize (number : 16 : IRW)
    // Size of images shown for fieldTypes image and imageFile in this field.
    // This setting overrides the global ListGrid default +link{attr:ListGrid.imageSize}.
    // <P>
    // If set to a String, assumed to be a property on each record that specifies the image
    // height.  For example, if <code>field.imageSize</code> is "logoSize",
    // <code>record.logoSize</code> will control the size of the image.
    //
    // @see attr:ListGridField.imageWidth
    // @see attr:ListGridField.imageHeight
    // 
    // @group imageColumns
    // @visibility external
	//<

  	//>	@attr	listGridField.imageWidth (number : 16 : IRW)
    // Width of images shown for fieldTypes image and imageFile in this field.
    // <P>
    // If set to a String, assumed to be a property on each record that specifies the image
    // width.  For example, if <code>field.imageWidth</code> is "logoWidth",
    // <code>record.logoWidth</code> will control the width of the image.
    //
    // @see attr:ListGrid.imageSize
    // @see attr:ListGridField.imageSize
    // @see attr:ListGridField.imageHeight
    // 
    // @group imageColumns
    // @visibility external
	//<

  	//>	@attr	listGridField.imageHeight (number : 16 : IRW)
    // Height of image shown for fieldTypes image and imageFile in this field.
    // <P>
    // If set to a String, assumed to be a property on each record that specifies the image
    // height.  For example, if <code>field.imageHeight</code> is "logoHeight",
    // <code>record.logoHeight</code> will control the height of the image.
    //
    // @see attr:ListGrid.imageSize
    // @see attr:ListGridField.imageSize
    // @see attr:ListGridField.imageWidth
    // 
    // @group imageColumns
    // @visibility external
	//<

    // ListGridField
    // ---------------------------------------------------------------------------------------

	//  -- Define the 'listGridField' pseudo class for doc

	//>	@object ListGridField
    // An ordinary JavaScript object containing properties that configures the display of
    // and interaction with the columns of a +link{ListGrid}.
	//
	// @see ListGrid.fields
	// @see ListGrid.setFields
	// @treeLocation Client Reference/Grids/ListGrid
	// @visibility external    
	//<

    //> @type ListGridFieldType
    // ListGrids format data for viewing and editing based on the <i>type</i> attribute of the
    // field.  This table describes how the ListGrid deals with the various built-in types.
    //
    // @value "text"    Simple text rendering for view.  For editing a text entry field is shown.
    // If the length of the field (as specified by the +link{attr:dataSourceField.length}
    // attribute) is larger than the value specified by +link{attr:listGrid.longTextEditorThreshold}, a
    // text input icon is shown that, when clicked on (or field is focused in) opens a larger
    // editor that expands outside the boundaries of the cell (textarea by default, but
    // overrideable via +link{ListGrid.longTextEditorType}).
    //
    // @value "boolean" For viewing and editing a checkbox is shown with a check mark for the
    // <code>true</code> value and no check mark for the <code>false</code> value.  See
    // +link{ListGrid.booleanTrueImage} for customization.
    //
    // @value "integer" Same as <code>text</code>.  Consider setting
    // +link{listGridField.editorType,editorType} to use a +link{SpinnerItem}.
    //
    // @value "float" Same as <code>text</code>.  Consider setting
    // +link{listGridField.editorType,editorType} to use a +link{SpinnerItem}.
    //
    // @value "date"
    // Expected to contain <code>Date</code> type data. Dates will be formatted using
    // +link{listGridField.dateFormatter} if specified, otherwise +link{ListGrid.dateFormatter}.
    // If both these attributes are unset, dates are formatted using the standard 
    // +link{Date.setShortDisplayFormat(),short display format} for dates.
    // <P>
    // For editing, by default a +link{DateItem} is used with +link{DateItem.useTextField} set
    // to true, providing textual date entry plus a pop-up date picker. The
    // +link{DateItem.dateFormatter, dateFormatter} and +link{DateItem.inputFormat, inputFormat}
    // for the editor will be picked up from the ListGridField, if specified.
    //
    // @value "time"
    // Expected to contain Time data encoded in javascript <code>Date</code> objects.
    // Times will be formatted using +link{listGridField.timeFormatter} if specified, 
    // otherwise +link{ListGrid.timeFormatter}.
    // <P>
    // If both these attributes are unset, times are formatted using the standard 
    // +link{Time.shortDisplayFormat,short display format} for times.
    // <P>
    // For editing, by default a +link{TimeItem} is used. The
    // +link{TimeItem.timeFormatter, timeFormatter} for the editor will be picked up from 
    // the ListGridField, if specified.
    //
    // @value "sequence" Same as <code>text</code>
    //
    // @value "link"     Renders a clickable html link (using an HTML anchor tag: &lt;A&gt;).
    // The target URL is the value of the field, which is also the default display value.  You
    // can override the display value by setting +link{attr:listGridRecord.linkText} or
    // +link{attr:listGridField.linkText}.
    // <P>
    // Clicking the link opens the URL in a new window by default.  To change this behavior,
    // you can set <code>field.target</code>, which works identically to the "target"
    // attribute on an HTML anchor (&lt;A&gt;) tag.  See +link{listGridField.target} for more
    // information.
    // <P>
    // In inline edit mode, this type works like a text field.
    // <P>
    // To create a link not covered by this feature, consider using
    // +link{listGridField.formatCellValue()} along with +link{Canvas.linkHTML()}, or simply
    // +link{listGrid.getCellStyle,styling the field} to look like a link, and providing
    // interactivity via +link{listGridField.recordClick,field.recordClick()}.
    //
    // @value "image"   Renders a different image in each row based on the value of the field.  If
    // this URL is not absolute, it is assumed to be relative to
    // +link{ListGridField.imageURLPrefix} if specified. The size of the image is controlled by
    // +link{attr:listGridField.imageSize}, +link{attr:listGridField.imageWidth},
    // +link{attr:listGridField.imageHeight} (and by the similarly-named global default
    // attributes on the ListGrid itself).
    // <P>
    // You can also specify the following attributes on the field: <code>activeAreaHTML</code>, and
    // <code>extraStuff</code> - these are passed to +link{method:canvas.imgHTML} to generate the
    // final URL.
    // <P>
    // Note if you want to display icons <b>in addition to</b> the normal cell value, you
    // can use +link{listGridField.valueIcons,valueIcons} instead.
    // 
    // @value "icon" Shows +link{listGridField.icon,field.icon} in every cell, and also in the
    // header.  Useful for a field that is used as a button, for example, launches a detail
    // window or removes a row.  Implement a +link{listGridField.recordClick,field.recordClick}
    // to define a behavior for the button.
    // <P>
    // NOTE: for a field that shows different icons depending on the field value, see
    // +link{listGridField.valueIcons}.
    // <P>
    // <code>type:"icon"</code> also defaults to a small field width, accommodating just the icon
    // with padding, and to a blank header title, so that the header shows the icon only.  
    // <P>
    // +link{listGridField.iconWidth,field.iconWidth} and related properties configure
    // the size of the icon both in the header and in body cells.
    // <P> 
    // If you want the icon to appear only in body cells and not in the header, set
    // +link{listGridField.cellIcon,field.cellIcon} instead, leaving field.icon null.
    // 
    // @value "binary"  For viewing, the grid renders a 'view' icon (looking glass) followed by a
    // 'download' icon and then the name of the file is displayed in text.  If the user clicks the
    // 'view' icon, a new browser window is opened and the file is streamed to that browser
    // instance, using +link{dataSource.viewFile()}.  For images and other file types with
    // known handlers, the content is typically displayed inline - otherwise the browser will
    // ask the user how to handle the content.  If the download icon is clicked,
    // +link{dataSource.downloadFile()} is used to cause the browser to show a "save" dialog.
    // There is no inline editing mode for this field type.
    //
    // @value "imageFile"   Same as <code>binary</code>
    //
    // @value "summary" Show a calculated summary based on other field values within the
    //  current record. See +link{listGridField.recordSummaryFunction} for more information
    // 
    // 
    // @see attr:listGridField.type
    // @see type:FieldType
    // @visibility external
    // @example gridsDataTypes
    //<    

	//> @attr listGridField.type (ListGridFieldType : "text" : [IR])
    //  ListGrids picks a renderer for the view and edit mode of a field based on this attribute.
    //
    //  @see type:ListGridFieldType
    //  @see type:FieldType
	//  @group  appearance
	//  @visibility external
	//<
    
	//> @attr listGridField.name (string : null : [IR])
	// Name of this field.  Must be unique within this ListGrid.
    // <P>
    // The name of field is also the property in each record which holds the value for that
    // field.
    // <P>
    // If a +link{listGrid.dataSource} is specified and the DataSource has a field with the
    // same name, the ListGridField and DataSourceField are merged so that properties on the
    // ListGridField
    // 
	// @group data
	// @visibility external
	//<
	
	//> @attr listGridField.dataPath (string : null : [IRA])
	// dataPath for this field. This property allows the grid to display details of nested data
	// structures in a flat list of columns. 
	// @group data
	// @visibility external
	//<

	//> @attr listGridField.title (string : null : [IRW])
	// A title for this field, to display in the header of the listGrid object.  Alternately you can
    // specify a +link{getFieldTitle()} method on the field to return the HTML for the field title.
    //
	// @group  appearance
    // @see method:listGridField.getFieldTitle()
	// @visibility external
	//<

    //> @method listGridField.getFieldTitle()
    // If your derivation of the field title is more complex than specifying a static string,
    // you can specify a getFieldTitle() method on your field to return the title string.
    // Otherwise you can use the +link{title} attribute on the field to specify the title.
    // <P>
    // You can use +link{listGrid.setFieldProperties,setFieldProperties()} to dynamically
    // update the title.
    //
    // @param viewer (ListGrid) pointer back to the ListGrid
    // @param fieldNum (number) index of this field in the grid's fields array.
    // @group appearance
    // @see attr:listGridField.title
    // @visibility external
    //<

    //> @attr listGridField.target (string : "_blank" : IRW)
    // By default, clicking a link rendered by this item opens it in a new browser window.  You 
    // can alter this behavior by setting this property.  The value of this property will be 
    // passed as the value to the <code>target</code> attribute of the anchor tag used to render 
    // the link.
    // <P>
    // If you set listGridField.target to "javascript", the default behavior is to catch and 
    // consume mouse-clicks that would result in the link being followed.  Instead, the
    // +link{listGrid.cellClick()} event is fired for the containing cell.
    // 
    // @visibility external
    //<
    
	//> @method listGridField.showIf()
	// An optional +link{group:stringMethods,stringMethod} which if provided, is evaluated to
    // conditionally determine whether this field should be displayed. 
    // Evaluated on initial draw, then reevaluated on explicit 
	// calls to <code>listGrid.refreshFields()</code> or <code>listGrid.setFields()</code>. 
    // <P>
    // Use 'showIf:"false"' to set a ListGrid field to initially hidden.
    // <P>
    // Note that explicit calls to +link{listGrid.showField,grid.showField()} or hideField()
    // will wipe out the <code>showIf</code> expression, as will the end user showing and
    // hiding columns via the +link{listGrid.showHeaderContextMenu,header contextMenu}.
    // <P>
    // Also note that fields marked as +link{DataSourceField.detail,detail:true} will be hidden by
    // default even if +link{ListGrid.showDetailFields} is <code>true</code>. To show detail fields
    // inherited from a DataSource, include an explicit field definition for the field and
    // set this property to return <code>true</code>.
    //
    // @param list (ListGrid) A pointer to the listGrid containing the field
    // @param field (ListGridField) the ListGridField object
    // @param fieldNum (integer) the index of the field
    // @return (boolean) whether the field should be shown
    // 
	// @group appearance
	// @see method:ListGrid.refreshFields
	// @visibility external
	//<

    //> @attr listGridField.frozen (boolean : null : IR)
    // Whether this field should be "frozen" for the purposes of horizontal scrolling.  See
    // +link{group:frozenFields}.
    // @group frozenFields
    // @visibility external
    //<

    //> @attr listGridField.canFreeze (boolean : null : IR)
    // Whether this field should display freezing/unfreezing options in its header context menu. 
    // See +link{group:frozenFields}.
    // @see method:listGrid.getHeaderContextMenuItems()
    // @group frozenFields
    // @visibility external
    //<

    //> @attr listGridField.autoFreeze (boolean : null : IR)
    // Whether this field should be automatically frozen when other fields are frozen.  When
    // true, the field will be automatically frozen to the extreme of the grid.  The
    // automatically generated +link{listGrid.checkboxField, checkbox},
    // +link{listGrid.expansionField, expansion} and 
    // +link{listGrid.rowNumberField, rowNumber} fields are examples of fields that specify
    // <code>autoFreeze: true</code>.
    // <P>
    // You can control the position of this field in the array of frozen fields by providing a
    // +link{listGridField.getAutoFreezePosition} implementation.
    // @group frozenFields
    // @visibility external
    //<

    //> @method listGridField.getAutoFreezePosition()
    // When a field has +link{listGridField.autoFreeze, autoFreeze} set to true, developers can
    // implement this method to indicate where in the frozen-fields array this field should 
    // appear.
    // <P>
    // Some automatically generated fields, such as 
    // +link{listGrid.rowNumberField, rowNumberField}, 
    // +link{listGrid.expansionField, expansionField} and
    // +link{listGrid.checkboxField, checkboxField}, provide default implementations of this 
    // method.
    // @return (number) the index at which this autoFreeze field should appear in the frozen body
    // @group frozenFields
    //<

    //> @attr listGridField.canHide (boolean : null : IR)
    // Whether this field can be hidden via the header context menu. 
    //
    // @see method:listGrid.getHeaderContextMenuItems()
    // @group appearance
    // @visibility external
    //<

    //> @attr listGridField.canDragResize (boolean : null : IR)
    // Whether this field can be dragResized using the mouse.  If unset, the default behavior
    // is governed by +link{listGrid.canResizeFields}.
    // @visibility external
    //<

    //> @attr listGridField.canReorder (boolean : null : IR)
    // Whether this field can be reordered using the mouse.  If unset, the default behavior is
    // governed by +link{listGrid.canReorderFields}.  Note that setting this property to 
    // <code>false</code> will lock this field from being moved - that is, the user is 
    // prevented from moving this field directly by dragging with the mouse, or by dropping another 
    // field onto this field.  However, unless this field is at the extreme edge of the grid, 
    // or all fields between it and the extreme edge of the grid are also
    // <code>canReorder: false</code>, (ie, if there are unlocked fields on either side of this
    // field), then it is possible that this locked field may be 
    // reordered automatically, as a result of the user dragging one unlocked field onto another 
    // unlocked field.
    // @visibility external
    //<

    // Grid, Group and Record-level summaries
    // ---------------------------------------------------------------------------------------

    //> @attr listGridField.showGridSummary (boolean : null : IR)
    // If +link{listGrid.showGridSummary} is true, should this field show a summary value.
    // If unset, this field will show a summary value in the summary row if an
    // explicit +link{listGridField.summaryFunction} is specified or if a
    // +link{SimpleType.getDefaultSummaryFunction(),default summary function} is defined
    // for the specified field type.
    // @visibility external
    //<
    
    //> @attr listGridField.showGroupSummary (boolean : null : IR)
    // If +link{listGrid.showGroupSummary} is true, should this field show a summary value
    // in a summary row when the grid is grouped?
    // If unset, this field will show a summary value in the summary row if an
    // explicit +link{listGridField.summaryFunction} is specified or if a
    // +link{SimpleType.getDefaultSummaryFunction(),default summary function} is defined
    // for the specified field type.
    // @visibility external
    //<
    
    //> @attr listGridField.summaryFunction (SummaryFunction  or Array of SummaryFunction : null : IR)
    // If +link{listGrid.showGridSummary} or +link{listGrid.showGroupSummary} is true, 
    // this attribute can be used to specify
    // an explicit +link{type:SummaryFunction} for calculating the summary value to
    // display.
    // <P>
    // If an array of summaryFunctions is specified, they will be executed in turn and the
    // grid will show multiple summary rows at the grid or group level (or both)
    // containing the resulting values.
    // @visibility external
    //<
    
    //> @attr listGridField.summaryValueTitle (String : null : IR)
    // If +link{listGrid.showGridSummary} or +link{listGrid.showGroupSummary} is true and the
    // +link{listGridField.summaryFunction} is set to <code>"title"</code>, this attribute may be
    // set to a string to display in the group and/or grid summary. If unspecified the
    // +link{listGridField.title} will show up in the summary.
    // @visibility external
    //<
    
    //> @method listGridField.getGridSummary() [A]
    // If +link{listGrid.showGridSummary} is true, and this method is specified it will be
    // called to generate the summary value to be displayed in the grid summary row. Note that
    // this is called instead of making use of the +link{listGridField.summaryFunction}.
    // <P>
    // As with +link{listGrid.getGridSummary()} this method may return an array of results - 
    // in this case each result will show up in a separate row in the +link{listGrid.summaryRow}
    // grid.
    // <P>
    // If this grid is grouped, and +link{listGrid.showGroupSummary} is true, this method
    // will be passed a third parameter - an array of group-level summaries.
    // @param records (Array of ListGridRecord) records for which a summary is being generated
    // @param field (ListGridField) pointer to the field for which summary value is being generated
    // @param [groupSummaries] (Array of objects) If this grid is grouped and
    //  +link{listGrid.showGridSummary} is specified, this parameter contains an array of already-
    //  calculated summary values for each group in the grid. Each element in this array will
    //  be an object containing calculated summary values for each field in the grid, as well as
    //  a specified groupValue and groupName, allowing the developer to determine which group this
    //  summary value comes from
    // @return (any) summary value to display.
    // @visibility external
    //<
    
    //> @attr listGridField.formatGridSummary (stringMethod : null : IR)
    // Optional stringMethod to format the summary value displayed
    // in the +link{listGrid.showGridSummary,grid summary}.
    // Takes a single parameter <code>value</code> and should return the formatted version
    // of that value
    // <P>
    // Note that for fields with a specified summary function of "count", if this method is 
    // undefined we default to formatting the count value by appending 
    // <code>field.pluralTitle</code> if defined, otherwise <code>field.title</code> to the
    // numeric count value returned by the standard count function. To change this behavior for
    // such fields, specify an explicit 'formatGridSummary' and/or 'formatGroupSummary' method
    // @visibility external
    //<
    
    //> @method listGridField.getGroupSummary() [A]
    // If +link{listGrid.showGroupSummary} is true, and this method is specified it will be
    // called to generate the field summary value to be displayed for each group level summary row.
    // Note that this is called instead of making use of the +link{listGridField.summaryFunction}.
    // <P>
    // This method may return an array of results - in this case the group will show multiple summary
    // rows, with each entry in the array showing up in a different record.
    //
    // @param records (Array of ListGridRecord) records for which a summary is being generated
    //  (so all records in the group).
    // @param field (ListGridField) pointer to the field for which summary value is being generated
    // @param [groupNode] (object) object with specified groupValue and groupName for this group
    // @return (any) summary value to display
    // @visibility external
    //<
    
    //> @attr listGridField.formatGroupSummary (stringMethod : null : IR)
    // Optional stringMethod to format the group level summary values for this field displayed via
    // +link{listGrid.showGroupSummary}.
    // Takes a single parameter <code>value</code> and should return the formatted version
    // of that value
    // <P>
    // Note that for fields with a specified summary function of "count", if this method is 
    // undefined we default to formatting the count value by appending 
    // <code>field.pluralTitle</code> if defined, otherwise <code>field.title</code> to the
    // numeric count value returned by the standard count function. To change this behavior for
    // such fields, specify an explicit 'formatGridSummary' and/or 'formatGroupSummary' method
    // @visibility external
    //<

    //> @method listGridField.getRecordSummary() [A]
    // Only applies to +link{listGridFieldType,summary type} fields. If specified , this
    // method will be called to generated the record summary value to be displayed for each row
    // in this field.
    // <P>
    // Note that if specified, this is called instead of making use of the
    // +link{listGridField.recordSummaryFunction}.
    // <P>
    // If +link{listGrid.showGridSummary} or +link{listGrid.showGroupSummary} is true, this
    // field's value in the summary row[s] will still be calculated by calling this method.
    // In this case, the record object passed in will contain summary values for each field.
    // If custom handling is required for this case, it may be detected by checking the
    // record object's +link{listGridRecord.isGroupSummary} and +link{listGridRecord.isGridSummary}
    // attributes.
    // @param record (ListGridRecord) record for which a summary is being generated
    // @param field (ListGridField) pointer to the summary type field
    // @return (any) summary value to display
    // @visibility external
    //<
    
    //> @attr listGridField.recordSummaryFunction (RecordSummaryFunction : null : IR)
    // Only applies to fields of type <code>"summary"</code>.
    // Fields of this type will display a calculated value based on the other field values
    // within the current record.
    // <P>
    // This attribute specifies how the summary field value will be calculated. See 
    // +link{type:RecordSummaryFunction} for valid options.
    // <P>
    // A subset of the ListGrid's fields will be passed to the RecordSummaryFunction. 
    // Which fields to include is determined based on +link{listGridField.includeInRecordSummary}
    // <P>
    // If +link{listGrid.showGridSummary} or +link{listGrid.showGroupSummary} is true, this
    // field's value in the summary row[s] will still be calculated by calling this method.
    // In this case, the record object passed in will contain summary values for each field.
    // If custom handling is required for this case, it may be detected by checking the
    // record object's +link{listGridRecord.isGroupSummary} and +link{listGridRecord.isGridSummary}
    // attributes.
    // @visibility external
    //<
    
    //> @attr listGridField.partialSummary (boolean : null : IR)
    // Only applies to fields of type <code>"summary"</code>.
    // This attribute is set on a summary field, when calculating the summary value from
    // some record, the summary function will only be passed the fields before this summary field.
    // This may be useful for displaying running totals across a record.
    // <P>
    // Note that this feature would typically be used with
    // +link{listGrid.canReorderFields,canReorderFields:false}
    // @visibility external
    //<
    
    //> @attr listGridField.includeInRecordSummary (boolean : null : IR)
    // If a listGrid is showing a field of type summary, should this field be passed to the
    // recordSummaryFunction when calculating the summary value to display.
    // If unset, fields are included if they are of type "integer" or "float" only (since most
    // summary functions perform numeric calculations). See also
    // +link{listGridField.includeInRecordSummaryFields}.
    // @visibility external
    //<
    
    //> @attr listGridField.includeInRecordSummaryFields (array of fieldNames : null : IR)
    // If this listGrid has any fields of type <code>"summary"</code> and
    // this field will be +link{listGridField.includeInRecordSummary,included} in summary calculations
    // by default, this attribute provides an opportunity to explicitly specify which summary fields
    // the record should be displayed in.
    // <P>
    // Specified as an array of fieldNames. If set, this field value will only be included for
    // record summary value calculations for summary fields whos name is included in this array.
    // @visibility external
    //<
    
    // Header button icons
    // ---------------------------------------------------------------------------------------
    // Include all relevant docs from StatefulCanvas
    
    //> @attr listGridField.icon (SCImgURL: null : [IR])
    // Optional icon to show next to the title for this field. 
    // Should be set to a URL to an image. Relative paths will be evaluated starting at
    // the imgDir of this component. This URL is partial - it may be updated to indicate
    // the current disabled (etc) state of the field.
    // <P>
    // If +link{listGridField.type,field.type} is set to "icon", this icon will also be shown
    // in every cell of this field - see also +link{listGridField.cellIcon,field.cellIcon}.
    // 
    // @visibility external
    //<

    //> @attr listGridField.iconSize (integer : null : [IR])
    // If +link{listGridField.icon} is specified, this property can be used to specify the 
    // size of the icon to be displayed in the ListGrid header button. 
    // (See +link{StatefulCanvas.iconSize})
    // @see listGridField.icon
    // @visibility external
    //<
    
    //> @attr listGridField.iconWidth (integer : null : [IR])
    // If +link{listGridField.icon} is specified, this property can be used to specify the 
    // width of the icon to be displayed in the ListGrid header button. 
    // (See +link{StatefulCanvas.iconWidth})<br>
    // If this field is editable, and +link{ListGridField.editorIconWidth} is unset, this 
    // property will be passed onto the editors for this field as +link{FormItem.iconWidth}, 
    // which will effect the default size for +link{ListGridField.icons, icons} displayed 
    // in the editor. 
    // @see listGridField.icon
    // @see listGridField.icons
    // @visibility external
    //<

    //> @attr listGridField.iconHeight (integer : null : [IR])
    // If +link{listGridField.icon} is specified, this property can be used to specify the 
    // height of the icon to be displayed in the ListGrid header button. 
    // (See +link{StatefulCanvas.iconHeight})<br>
    // If this field is editable, and +link{ListGridField.editorIconHeight} is unset, this 
    // property will be passed onto the editors for this field as +link{FormItem.iconWidth}, 
    // which will effect the default size for +link{ListGridField.icons, icons} displayed 
    // in the editor. 
    // @see listGridField.icon
    // @see listGridField.icons
    // @visibility external
    //<

    //> @attr listGridField.iconOrientation (string : "left" : [IR])
    // If this field is showing an icon, should it appear to the left or right of the title?<br>
    // Valid options are <code>"left"</code> or <code>"right"</code>
    // @see listGridField.icon    
    // @visibility external
    //<
    // iconOrientation JS doc not included from statefulCanvas as that refers to 
    // setIconOrientation(), and we don't have an exposed way to get at the ListGrid field
    // header button at runtime.

    //> @attr listGridField.iconSpacing (integer : 6 : [IR])
    // @include statefulCanvas.iconSpacing   
    // @see listGridField.icon        
    // @visibility external
    //<

    //> @attr listGridField.showDisabledIcon (boolean : true : [IR])
    // @include statefulCanvas.showDisabledIcon
    // @see listGridField.icon        
    // @visibility external
    //<

    //> @attr listGridField.showRollOverIcon (boolean : false : [IR])
    // @include statefulCanvas.showRollOverIcon
    // @see listGridField.icon        
    // @visibility external
    //<

    //> @attr listGridField.showFocusedIcon (boolean : false : [IR])
    // @include statefulCanvas.showFocusedIcon
    // @see listGridField.icon        
    // @visibility external
    //<
    
    //> @attr listGridField.showDownIcon (boolean : false : [IR])
    // @include statefulCanvas.showDownIcon
    // @see listGridField.icon        
    // @visibility external
    //<

    //> @attr listGridField.showSelectedIcon (boolean : false : [IR])
    // @include statefulCanvas.showSelectedIcon   
    // @see listGridField.icon        
    //  @visibility external
    //<

    //> @attr listGridField.cellIcon (SCImgURL : null : [IR])
    // For a field of type:"icon" only, set the icon that appears in body cells.  Unless
    // setting +link{listGridField.icon,field.icon}, setting field.cellIcon will not show an
    // icon in the header.
    // 
    // @visibility external
    //<

    //> @attr listGridField.showFileInline (boolean : null : [IR])
    // For a field of type:"imageFile", indicates whether to stream the image and display it
    // inline or to display the View and Download icons.
    // 
    // @visibility external
    //<

    // FormItem icons
    // ---------------------------------------------------------------------------------------
    
    //> @attr listGridField.icons (Array of FormItemIcon Properties: null : [IRA]) 
    // If this field is editable, this property can be used to specify 
    // +link{FormItem.icons, icons} to be displayed in the editors displayed for this field
    // @group editing
    // @visibility external
    //<
    
    //> @attr listGridField.editorIconWidth (number : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.iconWidth}.<br>
    // If this property unset, the iconWidth property from the editor can be picked up from 
    // +link{listGridField.iconWidth} instead.
    // @see listGridField.icons
    // @group editing    
    // @visibility external
    //<

    //> @attr listGridField.editorIconHeight (number : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.iconHeight}.<br>
    // If this property unset, the iconHeight property from the editor can be picked up from 
    // +link{listGridField.iconHeight} instead.
    // @see listGridField.icons
    // @group editing    
    // @visibility external
    //<
    
    //> @attr listGridField.defaultIconSrc (string : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.defaultIconSrc}.
    // @see listGridField.icons
    // @group editing    
    // @visibility external
    //<
    
    //> @attr listGridField.iconPrompt (string : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.iconPrompt}.
    // @see listGridField.icons
    // @group editing        
    // @visibility internal
    //<

    //> @attr listGridField.iconHSpace (string : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.iconHSpace}.
    // @see listGridField.icons
    // @group editing        
    // @visibility internal
    //<    
    
    //> @attr listGridField.iconVAlign (string : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.iconVAlign}.
    // @see listGridField.icons
    // @group editing        
    // @visibility external
    //<    

    // editor picker icon
    
    //> @attr listGridField.showPickerIcon (boolean : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.showPickerIcon}.
    // @group editing    
    // @visibility pickerIcon
    //<
    
    //> @attr listGridField.pickerIconSrc (string : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.pickerIconSrc}.
    // @group editing        
    // @visibility pickerIcon
    //<
    
    //> @attr listGridField.pickerIconWidth (integer : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.pickerIconWidth}.
    // @group editing        
    // @visibility pickerIcon
    //<
    
    //> @attr listGridField.pickerIconHeight (integer : null : [IRA])
    // If this field is editable, this property will be passed to editors displayed for
    // cells within this field as +link{FormItem.pickerIconHeight}.
    // @group editing        
    // @visibility pickerIcon
    //<

    // Summary Title
    // ---------------------------------------------------------------------------------------
    
    //> @attr listGridField.summaryTitle (string : null : [IRWA])
    // Optional long summary title for this field, provided in addition to 
    // +link{listGridField.title}. This gives the developer an option to use a very short,
    // or empty title for the ListGrid column (where space may be a factor), but have a longer 
    // value available to be used elsewhere.<br>
    // By default this value will be used for the title of the context-menu item
    // for showing/hiding the listGrid field when the user right-clicks on the ListGrid header.
    //
    // @group appearance
    // @see attr:listGridField.title
    // @visibility external
    //<

    //> @method listGridField.getSummaryTitle() [A]
    // Optional string method to return a long summary title for this field, if a dynamic
    // summary title is required for this field.  
    //
    // @param viewer (listGrid) pointer back to the ListGrid
    // @param field (listGridField) pointer to the field object
    // @group appearance
    // @see attr:listGridField.summaryTitle
    // @see attr:listGridField.title
    // @visibility external
    //<

    // Header Appearance
    // ---------------------------------------------------------------------------------------
    
	//> @attr listGridField.width (Number or String : "*" : [IRW])
	// The width of this field, specified as either an absolute number of pixels,
	// a percentage of the remaining space like "25%", or "*" to split remaining space among
    // all fields which have "*".
    // <P>
    // See also +link{listGrid.minFieldWidth} to ensure no field goes below a minimum size.
    // <P>
    // Use +link{listGrid.resizeField} to programmatically change field width after creation.
    // <P>
    // Use +link{listGrid.getFieldWidth} to access the rendered field width after
    // the ListGrid is drawn.
    //
	// @group appearance
	// @visibility external
	//<

    //> @attr   listGridField.align (Alignment : null : [IRW])
    // Horizontal alignment for field's column header: "left", "right"
    // or "center". Applied to the column header title and cells by default. A separate
    // alignment for cells can be specified via +link{listGridField.cellAlign}.<br>
    // If null, values are left-aligned. If this field is editable, the
    // alignment of cells in the body will also be reflected in any editors for the field.
    //  @group  appearance
    //  @visibility external
    //<

    //> @attr listGridField.headerBaseStyle (cssClass : null : [IRW])
    // Custom base style to apply to this field's header button instead of 
    // +link{listGrid.headerBaseStyle}.<br>
    // Note that depending on the header button constructor, you may have to override
    // +link{listGridField.headerTitleStyle} as well.
    // @group appearance
    // @visibility external
    //<
    
    //> @attr listGridField.headerTitleStyle (cssClass : null : [IRW])
    // Custom titleStyle to apply to this field's header button instead of 
    // +link{listGrid.headerTitleStyle}.<br>
    // Note that this will typically only have an effect if 
    // +link{listGrid.headerButtonConstructor} is set to +link{class:StretchImgButton} or a subclass 
    // thereof.
    // @group appearance
    // @visibility external
    //<
    

    // Header Spans
    // ---------------------------------------------------------------------------------------
    // - known limitations
    //   - can't reorder a column to before or after a spanned set of columns, if the spanned
    //   columns are at the start or end of the visible fields.
    //   - several uses of this.Super(), instead of the faster this.invokeSuper() approach.
    //   Attempt to use invokeSuper() failed, likely because the header is not a discrete
    //   class, but an instance of Toolbar, and my guess (Alex) is that Class.invokeSuper()
    //   doesn't handle this particular case.


    //> @attr listGrid.headerSpans (Array of HeaderSpan : null : IR)
    // Header spans are a second level of headers that appear above the normal ListGrid headers,
    // spanning one or more listGrid fields in a manner similar to a column-spanning cell in an 
    // HTML table.
    // <P>
    // A header span can be created by simply naming the fields the header should span.  The
    // example below creates a headerSpan that spans the first two fields of the ListGrid.
    // <pre>
    //    isc.ListGrid.create({
    //        headerHeight:40,
    //        fields : [
    //            { name:"field1" },
    //            { name:"field2" },
    //            { name:"field3" }
    //        ],
    //        headerSpans : [
    //            { 
    //                fields: ["field1", "field2"],
    //                title: "Field 1 and 2"
    //            }
    //        ]
    //    });
    // </pre>
    // Header spans will automatically react to resizing of the headers they span, and will be
    // hidden automatically when all of the spanned fields are hidden.
    // <P>
    // Header spans appear in the +link{listGrid.header,header} area of the ListGrid, sharing space
    // with the existing headers, so it's typical to set +link{listGrid.headerHeight} to
    // approximately double its normal height when using headerSpans.
    // <P>
    // See +link{headerSpan} for many properties that allow the control of the appearance of
    // headerSpans.  Note that headerSpans are created via the +link{AutoChild} pattern, hence
    // you can change the SmartClient component being used, or any of it's properties.
    // <P>
    // Neither headerSpans themselves nor the fields within them may be drag reordered, but other
    // unspanned headers may be.
    // <P>
    // Note that headerSpans primarily provide a visual cue for grouping multiple headers 
    // together.  If you have an OLAP, data "cube" or multi-dimensional data model, the
    // +link{CubeGrid} component is the right choice.
    //
    // @group headerSpan
    // @visibility external
    //<    

    //> @attr listGrid.headerSpanHeight (integer : null : IR)
    // Default height for a +link{listGrid.headerSpans,headerSpan} with no height specified.
    // <P>
    // If <code>headerSpanHeight</code> is not specified (the default), headerSpans will be 1/2
    // of +link{listGrid.headerHeight}.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr listGrid.headerSpanVAlign (vAlign : "center" : IR)
    // Default alignment for +link{listGrid.headerSpans,headerSpans} with no
    // +link{headerSpan.valign} specified.
    //
    // @group headerSpan
    // @visibility external
    //<
    headerSpanVAlign: "center",

    //> @attr listGrid.unspannedHeaderVAlign (vAlign : null : IR) 
    // When +link{listGrid.headerSpans,headerSpans} are in use, this property sets the default
    // vertical alignment for for fields which do <b>not</b> have a headerSpan.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr listGrid.headerSpanConstructor (SCClassName : null : IR)
    // +link{SCClassName,SmartClient Class} to use for headerSpans.  Typically a +link{Button} or
    // +link{StretchImgButton} subclass.
    // <P>
    // If unset, headerSpans will be created using the +link{listGrid.headerButtonConstructor}.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr listGrid.headerSpan (AutoChild : null : IR)
    // +link{listGrid.headerSpans,headerSpans} are created via the +link{AutoChild} pattern, hence
    // <code>headerSpanConstructor</code>, <code>headerSpanDefaults</code> and
    // <code>headerSpanProperties</code> are valid.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @object HeaderSpan
    // A header span appears as a second level of headers in a ListGrid, spanning one or more
    // ListGrid columns and their associated headers.
    // <P>
    // See +link{listGrid.headerSpans}.
    // <P>
    // In addition to the properties documented here, all other properties specified on the
    // headerSpan object will be passed to the +link{Class.create,create()} method of the
    // +link{listGrid.headerSpanConstructor}.  This allows you to set properties such as
    // +link{button.baseStyle} or +link{stretchImgButton.src} directly in a
    // <code>headerSpan</code>.  
    //
    // @group headerSpan
    // @treeLocation Client Reference/Grids/ListGrid
    // @visibility external
    //<

    //> @attr headerSpan.name (identifier : null : IR)
    // Name for this headerSpan, for use in APIs like +link{listGrid.setHeaderSpanTitle()}.
    // <P>
    // Name is optional, but if specified, must be unique for this ListGrid (but not globally
    // unique).
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr headerSpan.fields (Array of String : null : IR)
    // List of fields that this header spans.  Fields should be identified by their value for
    // +link{listGridField.name}.
    //
    // @group headerSpan
    // @visibility external
    //< 

    //> @attr headerSpan.title (String : null : IR)
    // Title for this headerSpan.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr headerSpan.height (integer : null : IR)
    // Height of this headerSpan.  Defaults to +link{listGrid.headerSpanHeight}.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr headerSpan.valign (VerticalAlignment: null : IR)
    // Vertical alignment of the title of this headerSpan.
    // <P>
    // Defaults to listGrid.headerSpanVAlign if unset.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr headerSpan.align (Alignment: "center" : IR)
    // Horizontal alignment of the title of this headerSpan.
    //
    // @group headerSpan
    // @visibility external
    //<

    //> @attr listGrid.showTreeColumnPicker (boolean : true : IR)
    // When +link{listGrid.headerSpans} are in use, whether to show a hierarchical column picker
    // that includes both headerSpans and normal headers, with normal headers indented under
    // headerSpans similarly to how a +link{TreeGrid} displays a Tree.
    // <P>
    // If <code>showTreeColumnPicker</code> is false, no column picker will be shown on the
    // headerSpan itself, and the column picker for a clicked on a normal field header will include
    // only normal fields.
    //
    // @group headerSpan
    // @visibility external
    //<
    showTreeColumnPicker: true,


    // Cell Styling
    // ---------------------------------------------------------------------------------------
    
    //> @attr listGridField.cellAlign (Alignment : null : [IRW])
    // Horizontal alignment for cells in this field's column: "left", "right"
	// or "center".<br>
    // If null, alignment is derived from +link{ListGridField.align}. If this field is editable, 
    // the alignment of cells in the body will also be reflected in any editors for the field.
	//  @group  appearance
	//  @visibility external
	//<
    
    //> @attr   listGridField.baseStyle (cssClass : null : [IRW])
    //  Custom base style to apply to all cells in this field instead of +link{ListGrid.baseStyle}
    //  To override the baseStyle at the row level, use 
    //  +link{ListGrid.recordBaseStyleProperty, record[listGrid.recordBaseStyleProperty]}
    //  instead.
    // @see ListGrid.recordBaseStyleProperty    
    // @group appearance
    // @visibility external
    // @example gridCells
    //<
    
    // Sorting (per field)   
    // ---------------------------------------------------------------------------------------

	//> @attr   listGridField.canSort (boolean : true : [IRW])
	//  Enables or disables sorting by this column. If false, neither interactive
	//  nor scripted (via the sort() method) instructions will sort the listGrid by this
	//  column.
	//  @group  sorting
	//  @see    method:ListGrid.sort
	//  @see    attr:ListGrid.canSort
	//  @visibility external
	//<
    
    //> @attr listGridField.sortDirection (SortDirection : null : [IRW])
    // Specifies the default sorting direction for this column. If specified on the
    // +link{listGrid.sortField,default sort field} for the listGrid, sorting occurs
    // automatically, otherwise this will be the default direction when the user clicks the
    // field header, or calls +link{ListGrid.sort()} without specifying an explicit sort direction.
    // <P>
    // Overrides ListGrid.sortDirection
	//  @group  sorting
	//  @see type:SortDirection
	//  @visibility external
	//<
    
	//> @method listGridField.sortNormalizer() (A)
	// Optional function to return the value that should be used when sorting this field.
    // <P>
    // Note that, if the dataset exceeds +link{ListGrid.dataPageSize} and hence paging is
    // introduced, the grid relies on the server to provide sorting, and the sortNormalizer
    // will no longer be called.
    //
	// @param recordObject    (object)    record to normalize
	// @param fieldName       (string)    name of the field on which sorting occurred
    // @param context (ListGrid) A pointer back to the list grid displaying this field will
    //   be available as the <code>context</code> argument. Note that you can also get a pointer
    //   to the field definition object by calling <code>context.getField(fieldName)</code>
	//  @return (any)   normalized value for sorting
	//  @group  sorting
	//  @visibility external
    //  @example dataTypes
	//<

	// Editing (per field)
	// ----------------------------------------------------------------------------------------

	//> @attr listGridField.canEdit (boolean : null : [IRW])
	// Can this field be edited?  May be overridden by setting the 'canEdit' property at the
    // listGrid level.
    // <p>
    // For more dynamic control over whether fields can be edited, see
    // +link{listGrid.canEditCell}.
    //
	// @group  editing
	// @see attr:listGrid.canEdit
    // @see attr:listGrid.recordEditProperty
    // @see method:listGrid.canEditCell
	// @visibility external
    // @example disableEditing
	//<

    
    //> @attr listGridField.alwaysShowEditors (boolean : null : [IRW])
    // When this attribute is set, editors will be rendered into every row of the grid for
    // this field, rather than showing up in a single record at a time.
    // This attribute is only valid when +link{listGrid.editByCell} is false
    // @group editing
    //<
    

    
    //> @attr listGridField.defaultValue (any : null : [IRW])
    // If this field +link{listGridField.canEdit, can be edited}, this property can be used to
    // specify a default value for this field's editor when adding new rows to the grid.
    // @see listGrid.startEditingNew()
    // @group editing
    // @visibility external
    //<
    
    //> @method listGridField.defaultDynamicValue()
    // If this field +link{listGridField.canEdit, can be edited}, this property can be used to
    // set a dynamic default value which will show up in editors for this field. 
    // Will be applied to the editor for the field as +link{FormItem.defaultDynamicValue}
    //
    // @param	item	(FormItem)  The editor for the cell itself (also available as "this").
    //                              Note that in addition to the standard FormItem APIs available
    //                              on the editor, it also has:<br>
    //                              - a pointer back to the containing listGrid 
    //                              [<code>item.grid</code>]<br>
    //                              - the colNum being edited [<code>item.colNum</code>]<br>
    //                              - the rowNum being edited [<code>item.rowNum</code>]
    // @param	form    (DynamicForm) the managing DynamicForm instance
    // @param   values  (Object)      the current set of values for the form as a whole
    // @group editing
    // @visibility external
    //<
    
	//> @attr   listGridField.enterKeyEditAction (EnterKeyEditAction : "done" : [IRW])
	// What to do when a user hits enter while editing this field?<br>
	// Overrides the <code>enterKeyEditAction</code> as specified at the listGrid level while 
	// focus is in this field.
	//  @group  editing
	//  @visibility external
	//<
    
	//> @attr   listGridField.escapeKeyEditAction (EscapeKeyEditAction : "cancel" : [IRW])
	// What to do when a user hits escape while editing this field?<br>
	// Overrides the <code>escapeKeyEditAction</code> as specified at the listGrid level while 
	// focus is in this field.
	//  @group  editing
	//  @visibility external
	//<

    
	//> @attr   listGridField.nextTabColNum (number : null : [IRWA])
	// If specified, when the user hits tab while editing this field, editing will move to the 
	// specified colNum in the next row (or the first editable field after it), rather than the 
	// next editable field in this row.
	//  @group  editing
	//  @visibility advancedInlineEdit
	//<
    
	//> @attr   listGridField.previousTabColNum (number : null : [IRWA])
	// If specified, when the user hits shift+tab while editing this field, editing will move 
	// to the specified colNum in the previous row (or the first editable field before it), 
	// rather than the previous editable field in this row.
	//  @group  editing
	//  @visibility advancedInlineEdit
	//<
    
	//> @attr   listGridField.editorType (FormItem className : null : [IRWA])
	//      Name of form item class to use for the form item created to edit this field. 
	//      (Only used if this field is editable).<br>
	//      Note: If this is not specified, the edit-form item type may be derived from the
	//      <code>editorType</code> property, typically inherited from datasource fields, or 
	//      from the <code>type</code> of the field (showing the appropriate form item for 
	//      the data-type). See the +link{group:editing} overview for more on editing ListGrid
    //      fields.
	//  @group  editing
	//  @see attr:listGrid.canEdit
	//  @visibility external
    //  @example customEditors
	//<
    // link to editing group documentation included as that describes the additional
    // "rowNum", "colNum" and "grid" properties stored on the editor.
    
	//> @attr   listGridField.editorProperties (FormItem properties : null : [IRWA])
	// Properties to apply the the form item created to edit this field. (Only used if
	// this field is editable).
    // <P>
    // For example, if you have a field "shoeSize" with +link{dataSourceField.editorType} set
    // to "SpinnerItem" in order to use a SpinnerItem as your field editor, and you want to pass the
    // +link{spinnerItem.step} property to the created SpinnerItem:
    // <pre>
    //    fields : [
    //        { name:"shoeSize", editorType:"SpinnerItem", 
    //          editorProperties : { step:0.5 } },
    //        ... other fields ...
    //    ]
    // </pre>
    //
	// @group editing
	// @visibility external
    // @example customEditors
	//<
    
	//> @attr   listGrid.modalEditing (boolean : null : [IRWA])
	//      If this property is true, any mouse click outside of the open cell editors
    //      will end editing mode, hiding the cell editors and saving any changes to those
    //      cell values.
	// @group  editing
    // @visibility external
    // @example modalEditing
	//<

	//> @method listGridField.editorEnter (A)
	// Callback fired when the user first starts editing a cell.
    // <P>
    // This callback is typically used to establish dynamic default values via
    // +link{listGrid.setEditValue()} or +link{listGrid.setEditValues()}.
	//
	// @param record (ListGridRecord) record for the cell being edited.  <b>Will be null</b>
    //                                for a new, unsaved record.
	// @param value (any) value for the cell being edited
	// @param rowNum (integer)	row number for the cell
	// @param colNum (integer)	column number of the cell
    // @param grid (ListGrid) ListGrid to which this field belongs
	// @group editing
	// @visibility external
	//<

	//> @method listGridField.editorExit (A)
	// Callback fired when the user attempts to navigate away from the current edit cell, 
	// or complete the current edit.<br>
	// Return false from this method to cancel the default behavior (Saving / cancelling the
	// current edit / moving to the next edit cell)
	//
	// @param   editCompletionEvent (EditCompletionEvent)  What interaction triggered this
	//                                                          edit cell exit
	// @param	record     (object)	record for the cell being edited
	// @param	newValue   (any)    new value for the cell being edited
	// @param	rowNum	   (number)	row number for the cell
	// @param	colNum	   (number)	column number of the cell
    // @param   grid    (ListGrid)  ListGrid to which this field belongs    
    // @return  (boolean)   Returning false from this method will cancel the default behavior
    //                      (for example saving the row) and leave the editor visible and focus
    //                      in this edit cell.
	//  @group  editing
	//  @see listGrid.editorExit
	// @visibility external
	//<
        
    //> @method listGridField.cellChanged()
	// Callback fired when field changes value as the result of a cell edit.  Fired only on
    // successful save of edit, when the new value doesn't match the value before editing.<br>
    // <p>
    // Same signature as +link{method:listGrid.cellChanged()}, but defined on a per-field
    // basis.
    //
	// @param	record     (ListGridRecord)	record for the cell being changed
	// @param	newValue   (any)    new value for the cell
	// @param	oldValue   (any)    old value for the cell
	// @param	rowNum	   (number)	row number for the cell
	// @param	colNum	   (number)	column number of the cell
	// @param	grid       (ListGrid)	grid where cell was changed.
    //
	// @group  editing
	// @see method:listGrid.cellChanged()
	// @visibility external
	//<
    
    //> @attr listGridField.validators (array of Validator : null : [IRW])
    // Array of +link{class:Validator} objects for this field.  When the user edits cells in
    // this field, these validators will be applied to the edited value.<br>
    // Note: for databound listGrids, this property may be specified on the 
    // +link{class:DataSourceField}, enabling both client and server side validation.
    // @see class:Validator
    // @see listGridField.required
    // @group gridValidation 
    // @visibility external
    // @example dataValidation
    //<
    
    //> @attr listGridField.validateOnChange (boolean : null : [IRW])
    // If set to true, any +link{listGridField.validators} for this field will be run whenever
    // the value of the field is changed.
    // <P>
    // Analogous to the +link{FormItem.validateOnChange} property.
    // @group gridValidation
    // @visibility external
    //<
    
    //> @attr listGridField.required (boolean : null : [IRW])
    // When the user edits cells in this field, is this value required to be non-empty 
    // in order for validation to pass.<br>
    // Note: for databound listGrids, this property may be specified on the 
    // +link{class:DataSourceField}, enabling both client and server side validation.
    //
    // @see listGridField.validators
    // @group gridValidation
    // @visibility external
    //<
    
    //> @attr listGridField.displayFormat (varies : null : [IRWA])
    // For fields of type <code>"date"</code>, set this property to a valid
    // +link{dateDisplayFormat} to specify how the date should be formatted.<br>
    // For fields of type <code>"time"</code>, set this property to a valid 
    // +link{type:TimeDisplayFormat, TimeDisplayFormat} to specify how the time should be formatted.<br>
    // If unset, display format may be set at the listGrid level via +link{ListGrid.dateFormatter}
    // or +link{ListGrid.timeFormatter}.
    // <p>
    // If this field is editable the displayFormat will also be passed to the editor created
    // to edit this field.  For dates you may also need to set +link{listGridField.inputFormat}.
    //
    // @see listGridField.inputFormat
    // @see listGrid.dateFormatter
    // @see listGrid.timeFormatter
    // @deprecated Use +link{listGridField.dateFormatter} and +link{listGridField.timeFormatter} 
    //  instead.
    // @visibility external
    //<
    
    //> @attr listGridField.dateFormatter (DateDisplayFormat : null : [IRW])
    // Display format to use for date type values within this field. 
    // <P>
    // The +link{listGridField.timeFormatter} may also be used to format underlying Date values as
    // times (ommitting the date part entirely). If both <code>dateFormatter</code> and
    // <code>timeFormatter</code> are specified on a field, for
    // fields specified as +link{listGridField.type,type "time"} the
    // <code>timeFormatter</code> will be used, otherwise the <code>dateFormatter</code>
    // <P>
    // If <code>field.dateFormatter</code> and <code>field.timeFormatter</code> is unspecified,
    // date display format may be defined at the component level via
    // +link{ListGrid.dateFormatter}, or for fields of type <code>"datetime"</code>
    // +link{ListGrid.datetimeFormatter}. Otherwise the
    // default is to use the system-wide default short date format, configured via
    // +link{Date.setShortDisplayFormat()}.  Specify any valid +link{type:DateDisplayFormat} to 
    // change the format used by this item.
    // <P>
    // If this field is editable the dateFormatter will also be passed to the editor created
    // to edit this field. Note that you can also specify an explicit +link{listGridField.inputFormat}
    // which will be passed through to the editor as well, though this is not typically required
    // as the input format should be automatically derived by the SmartClient system
    // for standard DateDisplayFormats.
    //
    // @see listGrid.dateFormatter
    // @see listGrid.datetimeFormatter
    // @see listGridField.timeFormatter
    // @visibility external
    //<
    
    //>	@attr listGridField.timeFormatter (TimeDisplayFormat : null : [IRWA])
    // Time-format to apply to date type values within this field.  If specified, any
    // dates displayed in this field will be formatted as times using the appropriate format.
    // This is most commonly only applied to fields specified as type <code>"time"</code> though
    // if no explicit +link{listGridField.dateFormatter} is specified it will be respected for other 
    // fields as well.
    // <P>
    // If unspecified, a timeFormatter may be defined 
    // +link{ListGrid.timeFormatter,at the component level} and will be respected by fields
    // of type <code>"time"</code>.
    // <P>
    // If this field is editable, the timeFormatter will also be passed to the editor created to
    // edit this field as +link{formItem.timeFormatter}.
    //
	// @group appearance
    // @visibility external
	//<
    //timeFormatter:null
    
    //> @attr listGridField.inputFormat (DateInputFormat : null : [IRWA])
    // For fields of type <code>"date"</code> or <code>"datetime"</code>, if this is an editable 
    // listGrid, this property allows you to specify the +link{DateItem.inputFormat, inputFormat}
    // applied to the editor for this field.
    // @see listGridField.dateFormatter
    // @visibility external
    //<
    
    //> @attr listGridField.isRemoveField (boolean : null : [IRA])
    // If set to true and +link{listGrid.canRemoveRecords} is true, this field will be rendered
    // as the remove-field for this grid. In most common usage scenarios this field will essentially
    // be a placeholder indicating where the remove field should be rendered, meaning properties 
    // other than <code>isRemoveField</code>, such as <code>name</code> or <code>title</code>, may
    // be left unset.
    // @see listGrid.canRemoveRecords
    // @visibility external
    //<

    //> @method listGridField.recordClick()
    //
    // Executed when this field is clicked on.  Note that if +link{ListGrid.recordClick()} is
    // also defined, it will be fired for fields that define a recordClick handler if the
    // field-level handler returns true. Return false to prevent the grid-level handler from firing.
    //
    // @param	viewer		(ListGrid)	the listGrid that contains the click event
    // @param	record		(ListGridRecord)	the record that was clicked on
    // @param	recordNum	(number)	number of the record clicked on in the current set of
    //                                  displayed records (starts with 0)
    // @param	field		(ListGridField)	the field that was clicked on (field definition)
    // @param	fieldNum	(number)	number of the field clicked on in the listGrid.fields
    //                                  array
    // @param	value       (object)    value of the cell (after valueMap, etc. applied)
    // @param	rawValue	(object)	raw value of the cell (before valueMap, etc applied)
    // @return	(boolean)	false to stop event bubbling
    //
    // @group	events
    //
	// @see method:listGrid.recordClick()
	// @visibility external
    // @example recordClicks
	//<

    //> @method listGridField.recordDoubleClick()
    //
    // Executed when this field is double-clicked.  Note that if
    // +link{ListGrid.recordDoubleClick()} is also defined, it will be fired for fields that define
    // a recordDoubleClick handler if the field-level handler returns true. Return false to prevent
    // the grid-level handler from firing.
    //
    //
    // @param	viewer		(ListGrid)	the listGrid that contains doubleclick event
    // @param	record		(ListGridRecord)	the record that was double-clicked
    // @param	recordNum	(number)	number of the record clicked on in the current set of
    //                                  displayed records (starts with 0)
    // @param	field		(ListGridField)	the field that was clicked on (field definition)
    // @param	fieldNum	(number)	number of the field clicked on in the listGrid.fields
    //                                  array
    // @param	value       (object)    value of the cell (after valueMap, etc. applied)
    // @param	rawValue	(object)	raw value of the cell (before valueMap, etc applied)
    // @return	(boolean)	false to stop event bubbling
    //
    // @group	events
    //
	// @see method:listGrid.recordClick()
	// @visibility external
    // @example recordClicks
	//<

    // Filtering
    // ---------------------------------------------------------------------------------------

	//> @attr   listGridField.canFilter (boolean : null : [IRW])
	//      If showing a filter row for this ListGrid, should the filter criteria for this 
	//      field be editable
	//  @group  filterEditor
	//  @visibility external
    //  @example disableFilter
	//<

	//> @attr listGridField.filterEditorValueMap (object : null : [IRW])
	//  If this listGrid is showing a filter row, this property can be used to specify a
	//  mapping of internal data to/from display values to be in the appropriate filter
	//  row form item.
	//  @visibility external
	//  @group filterEditor
	//<
    
	//> @attr   listGridField.filterEditorType (FormItem className : null : [IRWA])
	//      If this ListGrid is showing a filter row, this property can be used to
	//      specify the form item class to use for the filter form item associated with this 
	//      field
	//      (Only used if this field is not canFilter:false).<br>
	//      Note: If this is not specified, the edit-form item type may be derived from the
	//      'editorType' property, typically inherited from datasource fields, or from the
	//      'type' of the field (showing the appropriate form item for the data-type).
	//  @group  filterEditor
	//  @visibility external
	//<
    
    //> @attr   listGridField.defaultFilterValue (any : null : [IRWA])
    // If this ListGrid is showing a filter row, this property can be used to apply a default
    // value to show in the filter editor for this field.
    // @group filterEditor
    // @visibility external
    //<
    

	//> @attr   listGridField.filterEditorProperties (FormItem properties : null : [IRWA])
	// If this ListGrid is showing a filter row
    // (+link{listGrid.showFilterEditor,showFilterEditor}:true), this property
    // can be used to specify properties for the appropriate filter form item.
	// @group filterEditor
    // @visibility external
	//<
        
	//>	@attr     listGridField.filterOnKeypress (boolean : null : [IRWA])
	// If we're showing the filterEditor (listGrid.showFilterEditor is true), this property 
	// determines whether this list should be filtered every time the user edits the value of
    // the filter editor for this field.
	// @group filterEditor
	// @visibility external
	//<
    
    //> @attr listGrid.fetchDelay (number : 300 : IRWA)
	// If we're showing the filterEditor (+link{listGrid.showFilterEditor} is true), and we're
	// re-filtering on every keypress (+link{listGrid.filterOnKeypress} is true), this 
    // property is the delay in milliseconds between the user changing the filter and the 
    // filter request being kicked off. If multiple changes are made to the filter 
    // within this fetch delay, only the most recent will actually cause a re-filter
    // @group filterEditor
    // @visibility external
    //<
    fetchDelay:300,

	//>	@attr listGridField.shouldPrint (boolean : null : IRW)
    // Whether this field should be included in the printable representation of the grid.
	//
    // @group printing
	// @visibility external
	//<

    // AutoComplete
    // ---------------------------------------------------------------------------------------

	//>	@attr listGridField.autoComplete (AutoComplete : null : IRW)
	// Whether to do inline autoComplete when editing this field.
	// <p>
	// If unset, defaults to listGrid.autoComplete
	//
	// @see listGrid.autoComplete
	// @visibility autoComplete
	//<

	//>	@attr listGridField.uniqueMatch (boolean : null : IRW)
	// When autoComplete is enabled, whether to offer only unique matches to the user.
	// <p>
	// If unset, defaults to listGrid.uniqueMatch.
	//
	// @see listGrid.uniqueMatch
	// @visibility autoComplete
	//<
    uniqueMatch:true,
    

	// Formatting (per field)
	// --------------------------------------------------------------------------------------------

	//> @method listGridField.getCellValue()    
	// A stringMethod which returns the cell value to display for this field for some record.
	// If defined, called by ListGrid.getCellValue().  Called in the scope of the field object.
    //
    // Deprecated as of Jan 12 05 in favor of field.formatCellValue, because 'getCellValue()'
    // is a lower-level API which handles (for example) returning the HTML for editors within
    // the cell.
	//  
	// @param  viewer  (ListGrid)  the ListGrid for which we're returning a cellValue
	// @param  record  (object)    the current record object
	// @param  recordNum   (nubmer)    row-index of the current record
	// @param  field   (listGridField) current field object
	// @param  fieldNum    (number)    column-index of the current field
	// @param  value   (any)   unformatted value for this field, determined via 
	//                          ListGrid.getRawCellValue()
	// @see    method:listGrid.getCellValue
	// @see method:listGridField.formatCellValue
	// @group  display_values
	// @visibility external
	// @return (any)   value to display in the ListGrid cell
	// @deprecated As of SmartClient 5.5, use +link{listGridField.formatCellValue}.
	//<
    
	// We provide separate formatters for the raw value displayed in a static cell, and the
	// value displayed in an editor.
	// This makes sense because:
	// - developers are likely to want to apply different formats - for example including some
	//   raw HTML in the static value, but not in the value displayed in a text based editor.
	// - the more common 'formatCellValue()' needs no parser to convert from the formatted value
	//   back to the raw value
	// If a developer wishes to apply the same formatter in both cases, the suggested approach
	// would be to write a single formatter function and have it be called from both methods.
    
	//> @method listGridField.formatCellValue()
	// Return the HTML to display in cells of this field.
    // <P>
    // Given the raw value for this field as taken from the record Formatter to apply to the
    // static values displayed in cells for this field.
    // <P>
	// <i>Example usage</i>: formatting a currency value stored in cents (so "100" to "$1.00")<br>
	// The value passed to this method is the raw value for the cell.<br>
	// Takes precedence over <code>formatCellValue</code> defined at the grid level for cells
	// in this field.
    // <P>
	// Note: this formatter will not be applied to the values displayed in cells being edited.
	// The +link{listGridField.formatEditorValue,formatEditorValue()} is provided for that purpose.
	//
	// @group display_values
	//
	// @param   value (any)   raw value for the cell, from the record for the row
	// @param   record   (ListGridRecord) 
    //   Record object for the cell. Note: If this is a new row that has not been saved, in an 
    //   editable grid, it has no associated record object. In this case the edit values will
    //   be passed in as this parameter (see +link{listGrid.getEditValues()})
	// @param   rowNum  (number)    row number for the cell
	// @param   colNum  (number)    column number for the cell.
	// @param   grid    (ListGrid) the ListGrid displaying the cell
	// @return (String) HTML to display in the cell
	//
	// @see listGrid.formatCellValue()
	// @see listGridField.formatEditorValue()
	// @visibility external
    // @example formatValues
	//<
	
	//> @attr listGridField.escapeHTML (boolean : null : IRW)
	// By default HTML values in ListGrid cells will be interpreted by the browser.
    // Setting this flag to true will causes HTML characters to be escaped, meaning the
    // raw value of the field (for example <code>"&lt;b&gt;AAA&lt;/b&gt;"</code>) is displayed
    // to the user rather than the interpreted HTML (for example <code>"<b>AAA</b>"</code>)
    // @visibility external
    //<
    
	//> @attr listGridField.linkText (String : null : IRW)
    // The HTML to display in cells of this field if the fieldType is set to link. 
    // <P>
    // This property sets linkText that will be the same for all records.  You can set linkText
    // on a per-record basis via +link{attr:listGridRecord.linkText}.
    //
    //  @see type:ListGridFieldType
    //  @see type:FieldType
    //  @see attr:listGridRecord.linkText
    //  @see attr:listGrid.linkTextProperty
    //  @see attr:listGridField.linkTextProperty
	//  @group  display_values
	//  @visibility external
    //  @example linkImage
	//<
	
	//> @attr listGridField.linkTextProperty (string : null : IRW)
    // Name of the property in a ListGridRecord that holds the HTML to display in cells of this
    // field if the fieldType is set to "link".
    //
    //  @see type:ListGridFieldType
    //  @see type:FieldType
    //  @see attr:listGridRecord.linkText
    //  @see attr:listGridField.linkText
    //  @see attr:listGrid.linkTextProperty
	//  @group  display_values
	//  @visibility external    
	//<
	
	//> @attr listGridField.linkURLPrefix (string : null : IRWA)
    // If this field has type [+link{type:ListGridFieldType}] set to <code>"link"</code>,
    // setting this property will apply a standard prefix to the link URL for cells in this field.
    // @visibility external
    //<
    
    
	//> @attr listGridField.linkURLSuffix (string : null : IRWA)
    // If this field has type [+link{type:ListGridFieldType}] set to <code>"link"</code>,
    // setting this property will apply a standard suffix to the link URL for cells in this field.
    // @visibility external
    //<
    
    // --------------------
    // Editing 
    
	//> @method listGridField.formatEditorValue
	// Return the value to display in cells of this field which are being edited.
    // <P>
	// <i>Example usage</i>: converting a stored value in cents (100) to a dollar-and-cents 
	// value in the editor (1.00)
    // <P>
	// The value passed to this method is the raw value for the cell.
    // <P>
	// <code>formatEditorValue</code> takes precedence over +link{listGrid.formatEditorValue()}
    // defined at the grid level for cells in this field.
    // <P>
	// To convert the formatted value displayed within an editor back to a raw value, implement
	// +link{listGridField.parseEditorValue} as well.
	//
	// @group editing
	//
	// @param   value (any)   raw value for the cell being edited
	// @param   record   (ListGridRecord) 
    //   Record object for the cell. Note: If this is a new row that has not been saved, in an 
    //   editable grid, it has no associated record object. In this case the edit values will
    //   be passed in as this parameter.
	// @param   rowNum  (number)    row number for the cell
	// @param   colNum  (number)    column number for the cell.
	// @param   grid    (ListGrid instance) A pointer to the ListGrid displaying the cell
	// @return (any) formatted value to display in the editor
	//
	// @see listGridField.formatCellValue()
	// @see listGrid.formatEditorValue()
	// @see listGridField.parseEditorValue()
	//
	// @visibility external
	//<
    
	//> @method listGridField.parseEditorValue
	// Method used to convert the value displayed in an editor for some cell in this field into
	// a raw value for saving.<br>
	// Takes precedence over <code>parseEditorValue</code> defined at the grid level.
	//
	// @group editing
	//
	// @param   value (any)   value displayed in the editor for the cell
	// @param   record (object) record object for the row being edited. May be null if this
	//                          is a new row being added to the end of the list.
	// @param   rowNum  (number)    row number for the cell
	// @param   colNum  (number)    column number for the cell.
	// @param   grid    (ListGrid instance) A pointer to the ListGrid displaying the cell
	// @return (any) raw value for the field derived from formatted value in editor
	// @see listGrid.parseEditorValue()
	// @see listGridField.formatEditorValue()
	// @visibility external    
	//<

	//> @attr listGridField.valueMap (Array of Object : null : IRW)
    // Array of legal values for this field, or an Object where each property maps a stored
    // value to a user-displayable value.<br>
    // Note that if this field is editable (see +link{listGrid.canEdit}, 
    // +link{listGridField.canEdit}), editors displayed for this field will pick up their
    // valueMap either from this value or from +link{listGridField.editorValueMap}.
    // <P>
    // See also +link{dataSourceField.valueMap}.
    //
	// @group display_values
    // @see ListGrid.setValueMap()
    // @see ListGrid.getDisplayValue()
	// @visibility external
    // @example listType
	//<
    
    //> @attr listGridField.editorValueMap (Array or Object : null : IRW)
    // A valueMap to use for editors shown for this field.  By default if this is not
    // specified +link{listGridField.valueMap,field.valueMap} will be used instead.
    // <P>
    // Dynamic valueMaps can be provided by implementing +link{listGrid.getEditorValueMap()}.
    // 
    // @group editing
    // @visibility external
    // @see listGrid.getEditorValueMap()
    // @see listGrid.setEditorValueMap()
    //<
    
    //> @method listGridField.change()
    // If this field is editable, any +link{formItem.change, change} handler specified
    // on the ListGridField will be passed onto the editors for this field.
    // <P>
    // Note that if +link{listGridField.canToggle} is true, the user may change the value of
    // a boolean field without going into edit mode by single clicking on the field. In this 
    // case the +link{listGridField.change()} and +link{listGridField.changed()} handlers will
    // fire but the <code>form</code> and <code>item</code> parameters will be null.
    //
    // @param	form    (DynamicForm) the managing DynamicForm instance
    // @param	item	(FormItem)    the editor (form item) itself (also available as "this").
    //                              Note that in addition to the standard FormItem APIs available
    //                              on the editor, it also has:<br>
    //                              - a pointer back to the containing listGrid 
    //                              [<code>item.grid</code>]<br>
    //                              - the colNum being edited [<code>item.colNum</code>]<br>
    //                              - the rowNum being edited [<code>item.rowNum</code>]
    // @param   value   (any)         The new value of the form item
    // @param   oldValue    (any)     The previous value of the form item
    // @return (Boolean) The change may be cancelled <var class="SmartClient">by returning false</var>
    // @see listGridField.changed()
    // @see listGrid.cellChanged()
    // @group editing
    // @visibility external
    //<
    
    //> @method listGridField.changed()
    // If this field is editable, any +link{formItem.changed, changed} handler specified
    // on the ListGridField will be passed onto the editors for this field.
    // Note that if +link{listGridField.canToggle} is true, the user may change the value of
    // a boolean field without going into edit mode by single clicking on the field. In this 
    // case the +link{listGridField.change()} and +link{listGridField.changed()} handlers will
    // fire but the <code>form</code> and <code>item</code> parameters will be null.
    //
    // @param	form    (DynamicForm) the managing DynamicForm instance
    // @param	item	(FormItem)    the editor (form item) itself (also available as "this").
    //                              Note that in addition to the standard FormItem APIs available
    //                              on the editor, it also has:<br>
    //                              - a pointer back to the containing listGrid 
    //                              [<code>item.grid</code>]<br>
    //                              - the colNum being edited [<code>item.colNum</code>]<br>
    //                              - the rowNum being edited [<code>item.rowNum</code>]
    // @param   value   (any)         The current value (after the change).
    // @see listGridField.change()
    // @see listGrid.cellChanged()
    // @group editing
    // @visibility external
    //<
    
	//> @attr listGridField.emptyCellValue (HTMLString : "&amp;nbsp;" : IRW)
	// The value to display for a cell whose value is null or the empty
	// string after applying formatCellValue and valueMap (if any).
    // <p>
    // This is the field-specific attribute.  You may also set the emptyCellValue at the grid
    // level to define the emptyCellValue for all empty fields in the grid.
    // 
	// @group display_values
    // @see listGrid.emptyCellValue
	// @visibility external
    // @example emptyValues
	//<
    
	// Field.optionDataSource
	// --------------------------------------------------------------------------------------------

	//> @attr listGridField.autoFetchDisplayMap (boolean : null : [IRW])
	// If true, automatically fetches records and derives a valueMap from
    // +link{listGridField.optionDataSource}.
    // <p>
    // Same as +link{listGrid.autoFetchDisplayMap}, but defined on a per-field basis.
    //
	// @group display_values
	// @see listGrid.autoFetchDisplayMap
    // @visibility external
	//<
     
    //> @attr listGridField.optionTextMatchStyle (TextMatchStyle : null : [IR])
    // For fields with an +link{listGridField.optionDataSource}, where
    // +link{listGridField.autoFetchDisplayMap} is true, this property will govern
    // the <code>textMatchStyle</code> attribute of the +link{DSRequest} parameter passed to 
    // +link{DataSource.fetchData()} when retrieving the remote data set to be used as 
    // a basis for this field's valueMap.
    // @group display_values
    // @visibility external
    //<
    
    //> @attr listGridField.optionFilterContext (DSRequest Properties : null : [IR])
    // If this field has an optionDataSource specified and 
    // +link{listGridField.autoFetchDisplayMap,autoFetchDisplayMap} is set, this attribute
    // provides a way to customize the dataSource request issued to fetch the display map from
    // the option dataSource.
    // @group display_values
    // @visibility external
    //<
    
    //> @attr listGridField.optionOperationId (string : null : [IR])
    // If this field has an optionDataSource specified and 
    // +link{listGridField.autoFetchDisplayMap,autoFetchDisplayMap} is set, this attribute
    // provides a way to customize the +link{DSRequest.operationId} passed to 
    // <code>dataSource.fetchData()</code> when retrieving the display map from the option 
    // dataSource.
    // @group display_values
    // @visibility external
    //<
    
	//> @attr listGridField.optionDataSource (String : null : [IRW])
	// Derive a +link{valueMap} by fetching records from another DataSource and extracting
    // the +link{listGridField.valueField,valueField} and 
    // +link{listGridField.displayField,displayField} in the loaded records, to derive one
    // valueMap entry per record loaded from the optionDataSource.
    // <P>
    // Unlike the similar use of +link{pickList.optionDataSource} for +link{PickList,pickLists}
    // used during editing or filtering, <code>listGridField.optionDataSource</code> causes the
    // entire set of records from the optionDataSource to be fetched, without paging.   Hence
    // listGridField.optionDataSource is appropriate only for smaller valueMaps.  For very
    // large valueMap situations, such as an accountId field that should be displayed as an
    // accountName where there are thousands of accounts, the recommended approach is:
    // <ul>
    // <li> do not set listGridField.optionDataSource
    // <li> declare two fields in the DataSource, eg "accountId" and "accountName".
    // <li> Set the +link{ListGridField.displayField} attribute on the data field to the
    //      name of the display field.
    // <li> When fetching records for display in a grid, have your server send back values for 
    //      both fields, but show only the data field ("accountId") in the grid.
    // </ul>
    // In this case the cells in the accountId field will show the record value from the 
    // accountName field.  This approach means the valueMap will never be loaded in its
    // entirety, instead, each loaded record contains the valueMapping for that one record, as
    // a pair of fields within the record.  +explorerExample{largeValueMapSQL,This sample}
    // illustrates this approach achieved via a server-side SQL join.
    // <P>
    // Note that it is key that the server return <b>both</b> the underlying stored value
    // <b>and</b> the display value, as suggested above, because this approach allows the
    // +link{pickList.optionDataSource} property to be used to provide paged valueMaps during
    // inline editing and +link{ListGrid.showFilterEditor,inline filtering}.  Note that
    // pickList.optionDataSource is a distinct setting from listGridField.optionDataSource,
    // settable via +link{listGridField.editorProperties} (for editing) and 
    // +link{listGridField.filterEditorProperties,field.filterEditorProperties} (for
    // filtering).
    //
	// @group display_values
	// @visibility external
	//<

	//> @attr listGridField.valueField (string : null : [IRW])
	// Specifies the +link{listGridField.optionDataSource} field used to retrieve the stored
    // values that are to be mapped to the display values (specified by
    // +link{listGridField.displayField}). Note that if this field is editable this will also
    // be applied to this field's editors.
    //
	// @group display_values
	// @visibility external
	//<

	//> @attr listGridField.displayField (string : null : [IRW])
	// Specifies the +link{listGridField.optionDataSource} field used to retrieve the display
    // values that are to be mapped from the internal values specified by +link{valueField}.
    // <P>
    // If no <code>optionDataSource</code> is defined for the field, the cell will display
    // the displayField value for the current record instead of the underlying value for
    // this field.  This approach can be used for situations where field values need a stored
    // value to displayed value mapping, but the set of all possible values is too large to
    // load as a +link{ValueMap} - see +link{listGridField.optionDataSource} for more details
    // on this approach.  Note that if this field is editable this will also be applied to this
    // field's editors.  +explorerExample{largeValueMapSQL,This sample} illustrates this
    // approach achieved via a server-side SQL join.
    // <p>
    // The display value for a record with a specified <code>displayField</code> can be 
    // picked up via +link{ListGrid.getDisplayValue()}.
    //
	// @group display_values
	// @visibility external
	//<
	
    //> @attr listGridField.sortByDisplayField (boolean : null : [IRW])
    // For a field with <code>displayField</code> configured, should client-side sorting
    // be performed on the display field value? Unless explicitly set to <code>false</code>
    // the display field value is used.
    //
    // @group display_values
    // @visibility external
    //<
	
	//> @attr listGridField.optionCriteria (Criteria : null : [IRW])
	// If +link{optionDataSource} is set for this ListGridField, criteria specified in this
	// attribute will be passed to the dataSource when performing the fetch operation to
	// determine data-value to display-value mappings
	// @group display_values
	// @visibility external
	//<
	
	
	// ---------
	
	//> @attr listGridField.includeFrom (String : null : [IR])
	// Indicates this field's values should be fetched from another, related DataSource.  
	// The individual field will inherit settings such as +link{listGridField.type,field.type}
	// and +link{listGridField.title,field.title} from the related DataSource just like
	// fields from the primary DataSource.
	// <P>
	// When +link{fetchData()} is called, the automatically created +link{DSRequest} will 
	// specify +link{dsRequest.outputs} requesting the field, and any +link{Criteria} 
	// generated by the component will likewise refer to the field from the related DataSource.
	// <P>
	// It's an error to use this property if the ListGrid does not have a DataSource at all.
	// The related DataSource must be loaded or a warning will be logged and the field 
	// definition ignored.
	// <P>
	// This value is expected to be set to the following format 
	// <code>dataSourceID.fieldName</code> where <i>dataSourceID</i> is the ID of the
	// related dataSource and <i>fieldName</i> is the name of the field from that dataSource
	// from which you wish to retrieve values. Note that if this property is set and
	// +link{field.name} is not explicitly specified, this field's <code>name</code> will
	// default to the <i>fieldName</i> value from this property.
	//
	// @group display_values
	// @visibility crossDS
	//<
	
	// ----------------------------------------------------------------------------------------
	// Don't show scrollbars -- scrolling occurs in the body
    overflow:isc.Canvas.HIDDEN,	

	//>	@attr	listGrid.backgroundColor		(string : "white" : IRW)
	//		@group	appearance
	//<
	backgroundColor:"white",
    
	//>	@attr	listGrid.minHeight		(number : 50 : IRW)
	// Minimum height for the entire list (smaller than this doesn't tend to work very well).
	//		@group	sizing
	//<
    minHeight:50,							

    defaultWidth:200,

	//>	@attr listGrid.fieldIdProperty (string : "name" : IRWA)
	//  Property to be used as field identifier on listGridField objects.
	//  The ID of the field is also the property in each record which holds the value 
	//  for that field.
	//		@group	data
	//<
    // defaulted on Canvas
    

	// GridRenderer properties
	// ---------------------------------------------------------------------------------------

	//>	@attr listGrid.showAllRecords (boolean : false : [IRW])
	// @include gridRenderer.showAllRows
    // @example autofitRows
	//<
	//showAllRecords:false,

	//>	@attr listGrid.showAllColumns (boolean : false : IR)
	// @include gridRenderer.showAllColumns
	//<
	//showAllColumns:false,

	//>	@attr listGrid.drawAllMaxCells (integer : 250 : IRWA)
	// @include gridRenderer.drawAllMaxCells
    // @group performance
    // @visibility external
	//<
	drawAllMaxCells:250,


	//>	@attr listGrid.drawAheadRatio (float : 1.3 : IRW)
	// @include gridRenderer.drawAheadRatio
    // @group performance
    // @example databoundFetch
	//<
    drawAheadRatio:1.3,

    //> @attr listGrid.quickDrawAheadRatio (float : 1.0 : IRW)
    // @include gridRenderer.quickDrawAheadRatio
    // @group performance
    //<
    quickDrawAheadRatio:1.0,

    //> @attr listGrid.instantScrollTrackRedraw (boolean : true : IRW)
    // @include gridRenderer.instantScrollTrackRedraw
    // @group performance
    // @visibility external
    //<

    //> @attr listGrid.scrollRedrawDelay (integer : 75 : IRW)
    // @include gridRenderer.scrollRedrawDelay
    // @group performance
    // @visibility external
    //<
    scrollRedrawDelay:75,

    //>	@attr listGrid.virtualScrolling (boolean : null : [IRA])
    // When incremental rendering is switched on and there are variable record heights, the virtual
    // scrolling mechanism manages the differences in scroll height calculations due to the
    // unknown sizes of unrendered rows to make the scrollbar and viewport appear correctly.
    // <P>
    // virtualScrolling is switched on automatically when fixedRecordHeights is false and when
    // using the +link{listGrid.showRecordComponents,recordComponents subsystem}, as
    // recordComponents expand the rows that contain them. This flag should be manually enabled
    // when calling +link{listGrid.addEmbeddedComponent}(...) if embedded components can
    // cause record sizes to expand beyond specified cellHeight.
    // @visibility external
    //<

    //> @attr listGrid.dataPageSize (integer : 75 : IRW)
    // @include dataBoundComponent.dataPageSize
    // @group performance
    // @visibility external
    // @example databoundFetch
    //<
    
    //> @attr listGrid.dataFetchMode (FetchMode : "paged" : IRW)
    // @include dataBoundComponent.dataFetchMode
    //<

    // configures ResultSet.fetchDelay, delay in MS before fetches are triggered
    
    dataFetchDelay : 300,

    //> @attr listGrid.body (AutoChild : null : R)
    // GridRenderer used to render the dataset.
    // @visibility external
    //<

    bodyConstructor:"GridBody",

	//>	@attr listGrid.bodyOverflow (Overflow : isc.Canvas.AUTO : [IRWA])
	// Overflow setting for the "body", that is, the area of the grid where data values are
    // rendered.
    // <P>
    // By setting both the grid itself and the body to overflow:visible, it is possible to
    // "auto-fit" to the rendered height or width of the rows.  Note that in this case
    // <code>grid.width</code> and <code>grid.height</code> act as minimums.
    //
	//      @visibility external
	//		@group	sizing
    //      @example autofitRows
	//<
    bodyOverflow:isc.Canvas.AUTO,
    

	//>	@attr listGrid.bodyBackgroundColor (string : "white" : IRW)
    // Background color applied to the ListGrid body (that is, the area of the grid where
    // data values are rendered).<br>
    // Note that this will typically not be visible to the user unless there are few enough
    // rows that there is visible space in the body below the last row. To style data cells,
    // override +link{ListGrid.baseStyle} instead.
	//		@group	appearance
    // @visibility external
	//<
    bodyBackgroundColor:"white",			
    
	//>	@attr listGrid.bodyStyleName (CSSStyleName : null : IRW)
	// CSS style used for the body of this grid.  If applying a background-color to the body
    // via a CSS style applied using this property, be sure to set 
    // +link{ListGrid.bodyBackgroundColor} to <code>null</code>.
    //		@group	appearance
    // @visibility external
	//<
	//bodyStyleName:null,
    
    // whether to allow the body and header to have different border sizes and automatically
    // adjust the body column sizes to compensate such that column boundaries line up.
    allowMismatchedHeaderBodyBorder : true,

	//>	@attr listGrid.emptyCellValue (HTMLString : "&nbsp;" : IRW)
	// The value to display for cells whose value is null or the empty string after applying
    // formatCellValue and valueMap (if any).
    // <p>
    // This is the grid-wide attribute.  You may also set the emptyCellValue on a per-field basis.
    //
	// @group cellStyling
    // @see listGridField.emptyCellValue
    // @visibility external
    // @example emptyValues
	//<
	emptyCellValue:"&nbsp;",
    
	//>	@attr listGrid.cellHeight (number : 20 : [IRW])
	// @include gridRenderer.cellHeight
    // @example multilineValues
	//<
    cellHeight:20,
    
    //> @attr listGrid.normalCellHeight (number : 20 : [IRWA])
    // If +link{listGrid.baseStyle} is unset, base style will be derived from 
    // +link{listGrid.normalBaseStyle} if this grid has fixed row heights and 
    // the specified +link{listGrid.cellHeight} matches this value. Otherwise
    // +link{listGrid.tallBaseStyle} will be used.
    // @visibility external
    //<
    normalCellHeight:20,

	//>	@attr listGrid.fixedRecordHeights (boolean : true : IRWA)
    // Should we vertically clip cell contents, or allow rows to expand vertically to show all
    // contents?
    // <P>
    // If we allow rows to expand, the row height as derived from
    // +link{gridRenderer.getRowHeight(),getRowHeight()} or the
    // default +link{cellHeight} is treated as a minimum.
    // <P>
    // <b>NOTE:</b> by default, for performance reasons, clipping is not enforced for some
    // kinds of content (such as images) on all browsers.  Set
    // +link{enforceVClipping,enforceVClipping:true} to enforce clipping for all types of
    // content on all browsers.
    //
	// @include gridRenderer.fixedRowHeights
    // @example autofitValues
	//<
	fixedRecordHeights:true,				

    //> @attr listGrid.enforceVClipping (boolean : false : IRW)
    // For performance reasons, even when +link{fixedRecordHeights} is set, vertical clipping
    // is not enforced by default for some kinds of content (such as images) on all browsers.
    // Set +link{enforceVClipping,enforceVClipping:true} to enforce clipping for all types of
    // content on all browsers.
    // <P>
    // This additional setting is likely to be phased out as browsers improve.
    //
    // @visibility external
    //<

	//>	@attr listGrid.fixedFieldWidths (boolean : true : IRWA)
	// Should we horizontally clip cell contents, or allow columns to expand horizontally to
	// show all contents?
	// <P>
	// If we allow columns to expand, the column width is treated as a minimum.
	// <P>
	// NOTE: the header does not automatically respond to expanded field widths
    // @group cellStyling
    // @visibility external
	//<
	// NOTE: doc is duplicated here because in the ListGrid we need to discuss the header.
	fixedFieldWidths:true,

    // autoFit attributes
    
    //> @attr listGrid.autoFitData (string : null : IRW)
    // Should this ListGrid automatically expand to accommodate the size of records and fields?
    // <P>
    // Valid settings are
    // <ul><li><code>"vertical"</code>: expand vertically to accommodate records.</li>
    //     <li><code>"horizontal"</code>: expand horizontally to accommodate fields.</li>
    //     <li><code>"both"</code>: expand horizontally and vertically to accommodate content.</li>
    // </ul>
    // How far the ListGrid will expand may be limited via the following properties:
    // +link{ListGrid.autoFitMaxHeight}, +link{ListGrid.autoFitMaxRecords},
    // +link{ListGrid.autoFitMaxWidth}, +link{ListGrid.autoFitMaxColumns}.
    // <P>
    // Note that this property causes the grid as a whole to expand to fit records or fields.
    // To have the fields or records themselves expand to fit cell contents, see 
    // +link{listGrid.autoFitFieldWidths} and +link{listGrid.fixedRecordHeights}.
    // @group autoFitData
    // @visibility external
    //<
    
    //> @attr listGrid.autoFitMaxHeight (number : null : IRW) 
    // If +link{listGrid.autoFitData} is set to <code>"vertical"</code> or <code>"both"</code> this
    // property provides an upper limit on how far the ListGrid will expand vertically to accommodate
    // its content. If content exceeds this height, scrollbars will be introduced as usual.  
    // In addition to this property, +link{ListGrid.autoFitMaxRecords} allows you to limit vertical
    // expansion based on the number of rows to be rendered.
    // @group autoFitData
    // @visibility external
    //<

    //> @attr ListGrid.autoFitMaxRecords (number : 50 : IRW)
    // If +link{listGrid.autoFitData} is set to <code>"vertical"</code> or <code>"both"</code> this
    // property provides the maximum number of records for which the ListGrid will expand. If more
    // records are present, scrolling will be introduced to reach them as normal.
    // If unset, by default the ListGrid will expand to accommodate as many records as are present.
    // @group autoFitData
    // @visibility external
    //<
    autoFitMaxRecords:50,

    //> @attr ListGrid.autoFitExtraRecords (number : null : IRW)
    // If +link{listGrid.autoFitData} is set to <code>"vertical"</code> or <code>"both"</code> 
    // this property specifies the number of additional records for which the grid 
    // will expand. If more
    // records are present, scrolling will be introduced to reach them as normal.
    // @group autoFitData
    // @visibility external
    //<
//    autoFitExtraRecords:0,

    //> @attr listGrid.autoFitMaxWidth (number : null : IRW) 
    // If +link{listGrid.autoFitData} is set to <code>"horizontal"</code> or <code>"both"</code>
    // this property provides an upper limit on how far the ListGrid will expand horizontally to
    // accommodate its content. 
    // If content exceeds this width, scrollbars will be introduced as usual.  
    // In addition to this property, +link{ListGrid.autoFitMaxColumns} allows you to limit 
    // horizontal expansion based on the number of columns to be rendered.
    // @group autoFitData
    // @visibility external
    //<
    
    //> @attr ListGrid.autoFitMaxColumns (number : 50 : IRW)
    // If +link{listGrid.autoFitData} is set to <code>"horizontal"</code> or <code>"both"</code>
    // this property provides the maximum number of columns for which the ListGrid will expand.
    // If more columns are present, scrolling will be introduced to reach them as normal.
    // If unset the ListGrid will expand to accommodate as many columns as are defined for the
    // grid.
    // @group autoFitData
    // @visibility external
    //<    
    autoFitMaxColumns:50,
  
    
    //> @attr listGrid.canAutoFitFields (boolean : true : IRW)
    // Whether the user able to autofit specific columns
    // to their data and/or title via a context menu item or +link{listgrid.headerAutoFitEvent}.
    // <P>
    // Autofitting can also be programmatically enabled by setting +link{field.autoFitWidth}.
    // @visibility external
    // @group autoFitFields
    //<
    canAutoFitFields:true,
    
    //>	@type AutoFitEvent
    // Event on a listGrid header to trigger auto-fit of the listgrid field.
    // @value "doubleClick" React to a double click on the listGrid header.
    // @value "click" React to a click on the listGrid header.
    // @value "none" No event will trigger auto-fit.
    // @group autoFitFields
    // @visibility external
    //<
        
    //> @attr listGrid.headerAutoFitEvent (AutoFitEvent : "doubleClick" : IR)
    // Event on a ListGrid header that triggers auto fitting to data and/or title.
    // <P>
    // Note that if sorting is enabled for the field and the headerAutoFitEvent is "click", both
    // sorting and autofit occur on a click.
    // @visibility external
    // @group autoFitFields
    //<
    headerAutoFitEvent:"doubleClick",
    
    //> @attr listGridField.autoFitWidth (boolean : null : IR)
    // Enables autofitting to values or titles for this field.. This overrides
    // the +link{listGrid.autoFitFieldWidths} attribute on a per-field basis.
    //
    // @visibility external
    // @group autoFitFields
    //<
    
    //> @attr listGrid.autoFitFieldWidths (boolean : null : IR)
    // Enables autofitting of fields to values or titles. This property may be overridden
    // on a per-field basis via +link{listGridField.autoFitWidth}.<br>
    // The +link{listGrid.autoFitWidthApproach} controls whether fitting is to values, titles 
    // or both. This property may also be overridden on a per field basis.
    // <P>
    // If +link{field.width} is also set on the field, it will be taken as a minimum width.
    // +link{ListGrid.minFieldWith} will also be respected.
    // <P>
    // Autofitting will be performed:
    // <ul>
    //  <li> whenever the dataset is completely changed or rows are added or removed
    //  <li> whenever a field which is autofitting is changed
    //  <li> on a manual call to +link{listGrid.autoFitField()} or
    //       +link{listGrid.autoFitFields()}
    // </ul>
    // Autofitting behavior continues until the user resizes the field manually, at which
    // point it stops. The user can also perform a one-time auto-fit of fields via
    // the header context menu if +link{listGrid.canAutoFitFields} is enabled.
    // <P>
    // When autofitting to column values, +link{listGrid.getDefaultFieldWidth()} will be
    // called to determine the space required for a field's values. This method
    // uses values from the rendered set of rows to
    // calculate the required column width. The values used not match the complete set of data
    // for the grid when rendering rows incrementally. See +link{listGrid.showAllRecords} and
    // +link{listGrid.drawAheadRatio}) to control incremental rendering of rows.
    // <P>
    // Note that for <code>icon</code> type fields, the +link{listGrid.autoFitIconFields} 
    // property setting may turn on auto-fit-width behavior for specific fields by default,
    // even if <code>autoFitFieldWidths</code> is false for the grid as a whole.
    // <P>
    // Using this feature has a performance penalty roughly comparable to always rendering 
    // one additional field per field where autofitting is enabled.  Specifically, enabling it
    // for all fields would be comparable to <i>both</i> doubling the number of fields
    // <i>and</i> disabling +link{listGrid.showAllColumns,horizontal incremental rendering}.
    // In a grid where only half the fields are normally visible and hence only half are
    // normally rendered, this would be roughly 4 times slower overall.<br>
    // This performance penalty is a result of +link{getDefaultFieldWidth()} having to
    // render out the data set offscreen and measure the rendered content - it does not apply
    // for cases where this method can return a simple fixed values (as with icon fields).
    // <P>
    // Which fields are currently autofitting is saved as part of the 
    // +link{getViewState,view state} of the ListGrid.
    //
    // @visibility external
    // @group autoFitFields
    //<

    
    //> @attr listGrid.autoFitClipFields (Array of String : null : IR)
    // If +link{listGrid.autoFitFieldWidths} is enabled and the calculated field sizes
    // are wide enough that horizontal scrolling would be introduced, this attribute may be
    // set to an array of fieldNames, causing those fields to be clipped rather than
    // forcing horizontal scrollbars to appear.
    // <P>
    // Note: If any +link{ListGridField.frozen,frozen columns} are included in this list they
    // will not be clipped.
    // @group autoFitFields
    // @visibility external
    //<
    
    //> @attr listGrid.autoFitFieldsFillViewport (boolean : true : IR) 
    // If +link{listGrid.autoFitFieldWidths} is enabled, and auto-fitting all field widths
    // will not take up all the horizontal space available in the viewport, should a field be
    // expanded wider than it's calculated auto-fit-width to fill the available space
    // and avoid leaving a gap.
    // <P>
    // If true, the field to expand may be specified via +link{autoFitExpandField}
    // <P>
    // Note this logic will not expand a +link{ListGridField.frozen,frozen column}.
    // 
    // @group autoFitFields
    // @visibility external
    //<
    autoFitFieldsFillViewport:true,
    
    //> @attr listGrid.autoFitExpandField (String : null : IR)
    // The field to expand if +link{listGrid.autoFitFieldWidths} and 
    // +link{autoFitFieldsFillViewport} are enabled and
    // auto-fitting will not fill all available horizontal space.
    // <P>
    // If unset, will default to the text field with the longest 
    // +link{dataSourceField.length} if length is set, otherwise, the first text
    // field with no width specified.
    // <P>
    // Note that expanding +link{ListGridField.frozen,frozen columns} is not supported.
    // @group autoFitFields
    // @visibility external
    //<
    
    //> @type AutoFitWidthApproach
    // How should field width be determined when +link{listGridField.autoFitWidth} is true?
    // @value "value" Size field to fit to the data value(s) contained in the field.
    // @value "title" Size field to fit the field title
    // @value "both" Size field to fit either the field title or the data values in the field
    //  (whichever requires more space).
    //
    // @group autoFitFields
    // @visibility external
    //<

    //> @attr listGrid.autoFitWidthApproach (AutoFitWidthApproach : "value" : [IRW])
    // When a user requests column autofitting via the 
    // +link{getHeaderContextMenuItems,header contextmenu} or via a 
    // +link{headerAutoFitEvent,mouse gesture}, what autofit approach is used.
    // @group autoFitFields
    // @visibility external
    //<
    autoFitWidthApproach:"value",
    
    //> @attr listGridField.autoFitWidthApproach (AutoFitWidthApproach : null : [IRW])
    // When a user requests column autofitting via the 
    // +link{getHeaderContextMenuItems,header contextmenu} or via a 
    // +link{headerAutoFitEvent,mouse gesture}, what autofit approach is used. If set, this
    // setting overrides the autoFitWidthApproach specified at the ListGrid level.
    // @group autoFitFields
    // @visibility external
    //<
    
    //> @type AutoFitIconFieldType
    // How should fields of +link{listGridFieldType,type:"icon"} be sized by default?
    // @value "none" Apply no special sizing to icon fields - treat them like any other 
    //   field in the grid
    // @value "iconWidth" size the field to accommodate the width of the icon
    // @value "title" size the field to accommodate the title (or the width of the icon if
    //   it exceeds the width of the title.
    //
    // @group autoFitFields
    // @visibility external
    //<
    
    //> @attr listGrid.autoFitIconFields (AutoFitIconFieldType : "title" : [IRW])
    // SmartClient listGrids have special logic to automatically size fields that
    // are displayed as an icon - that is fields with 
    // +link{listGridFieldType,type:"icon"}, fields displaying only
    // +link{listGridField.showValueIconOnly,value icons}, and boolean fields (which
    // are rendered as a checkmark type icon by default.
    // <P>
    // This attribute controls this behavior - governing whether icon fields should
    // be sized to fit their content (icon), title, or whether to disable this 
    // behavior. Setting this value to <code>"title"</code> or <code>"iconWidth"</code>
    // will cause +link{listGridField.autoFitWidth} to be enabled by default for all 
    // icon fields with the +link{listGridField.autoFitWidthApproach} set to 
    // <code>"value"</code> or <code>"both"</code> as appropriate. Note that the
    // width required for the icons is calculated by +link{listGrid.getDefaultFieldWidth()}
    // which performs a simple calculation based on the specified icon width for these
    // types of fields.
    // <P>
    // This setting governs default behavior for icon fields - for specific fields within
    // a grid, this default behavior can be overridden by setting an explicit
    // +link{listGridField.width} or
    // explicitly enabling +link{listGridField.autoFitWidth} and setting
    // +link{listGridField.autoFitWidthApproach} on the field in question.
    // @see listGrid.autoFitFieldWidths
    // @group autoFitFields
    // @visibility external
    //<
    autoFitIconFields:"title",
    
    
    // By default, fields that show only an icon, including +link{listGridFieldType,listGridFieldType:"icon"}) are automatically assigned a width that 
    // accommodates the icon.  This may cause the title to be clipped - if so, consider using
    // an abbreviated title, using an icon as the title, enabling autoFitting to titles,
    // setting an explicit width on the field, or finally, disabling this setting if it's 
    // inconvenient.
    //
    // @visibility external
    //<
    
    //> @attr listGrid.leaveScrollbarGap (boolean : true : IRW)
	// Whether to leave a gap for the vertical scrollbar, even when it's not present.
    // <P>
    // Note that if leaveScrollbarGap is false and vertical scrolling is introduced, fields
    // will be resized to fit the smaller body area if possible, in order to avoid horizontal
    // scrolling also being required.
    //
    // @group appearance
    // @visibility external
    // @example autofitRows
    //<
    
	leaveScrollbarGap:true,

	// if leaveScrollbarGap is false, whether to resize fields when vscrolling is introduced
    resizeFieldsForScrollbar:true,

	//>	@attr listGrid.autoFit (boolean : false : IRWA)
	// If true, make columns only wide enough to fit content, ignoring any widths specified.
	// Overrides fixedFieldWidths.
	// <P>
	// NOTE: the header does not automatically respond to expanded field widths
	//	@group	sizing
	//<
	//autoFit:false,

	//>	@attr listGrid.wrapCells (boolean : false : IRWA)
    // Should content within cells be allowed to wrap?
    // <P>
    // Even if content is allowed to wrap, if +link{fixedRecordHeights} is set, the content
    // will be clipped off at the cell boundary.  Either set a larger, fixed +link{cellHeight}
    // to reveal more content, or set +link{fixedRecordHeights} to false to allow auto-sizing.
    // 
    // @example autofitValues
    // @visibility external
	//<
	//wrapCells:false,

	//>	@attr listGrid.cellSpacing (number : 0 : [IRW])
	// @include gridRenderer.cellSpacing
    // @visibility internal
	//<
    
	cellSpacing:0,

	//>	@attr listGrid.cellPadding (number : 2 : [IRW])
	// @include gridRenderer.cellPadding
	//<
	cellPadding:2,

    //> @attr listGrid.dateFormatter (DateDisplayFormat : null : [IRW])
    // How should Date type values be displayed in this ListGrid by default?
    // <P>
    // This property specifies the default DateDisplayFormat to apply to Date values
    // displayed in this grid for all fields except those of +link{listGridField.type,type "time"}
    // (See also +link{listGrid.timeFormatter}).<br>
    // If +link{listGrid.datetimeFormatter} is specified, that will be applied by default
    // to fields of type <code>"datetime"</code>.
    // <P>
    // Note that if +link{listGridField.dateFormatter} or +link{listGridField.timeFormatter} are
    // specified those properties will take precedence over the component level settings.
    // <P>
    // If unset, date values will be formatted according to the system wide 
    // +link{Date.setShortDisplayFormat(),short display format} or 
    // +link{Date.setShortDatetimeDisplayFormat(),short datetime display format} for datetime type
    // fields.
    // <P>
    // If this field is editable the dateFormatter will also be passed to the editor created
    // to edit this field as +link{DateItem.dateFormatter, dateFormatter}.
    // In this case you may also need to set +link{listGrid.dateInputFormat}.
    //
    // @visibility external
    //<
    //dateFormatter:null,
    
    //> @attr listGrid.datetimeFormatter (DateDisplayFormat : null : [IRW])
    // Display format to use for fields specified as type 'datetime'.  Default is to use the
    // system-wide default date time format, configured via
    // +link{Date.setShortDatetimeDisplayFormat()}.  Specify any
    // valid +link{type:DateDisplayFormat} to change the display format for datetimes used by this grid.
    // <var class="smartclient">
    // May be specified as a function. If specified as  a function, this function will be executed in the scope of the Date
    // and should return the formatted string.
    // </var>
    // <P>
    // May also be specified at the field level via
    // +link{listGridField.dateFormatter}
    // <P>
    // If this field is editable the dateFormatter will also be passed to the editor created
    // to edit this field as +link{DateItem.dateFormatter, dateFormatter}.
    // In this case you may also need to set +link{listGrid.dateInputFormat}.
    // 
    // @see listGridField.dateFormatter
	// @group appearance
    // @visibility external
    //<
    
    //> @attr listGrid.dateInputFormat (DateInputFormat : null : [IRWA])
    // If this is an editable listGrid, this property will specify the 
    // +link{DateItem.inputFormat, inputFormat} applied to editors for fields of type 
    // <code>"date"</code>. May be overridden per field via +link{listGridField.inputFormat}.
    // @see listGrid.dateFormatter
    // @visibility external
    //<

	// function to call appropriate date formatter
	// Note: this is executed in the scope of a field object - see 'applyFieldDefaults'
    _formatDateCellValue : function (value, field, grid, record, rowNum, colNum) {
    
        if (isc.isA.Date(value)) {
            // A developer may force a "date" or "datetime" type field value to be displayed as time
            // by specifying a timeFormatter and no dateFormatter on the field.
            if (grid._formatAsTime(field)) {
                var formatter = grid._getTimeFormatter(field);
                var isLogicalTime = isc.SimpleType.inheritsFrom(field.type, "time");
                return isc.Time.toTime(value, formatter, isLogicalTime);
            }

            var isDatetime = field && isc.SimpleType.inheritsFrom(field.type, "datetime"),
                isLogicalDate = !isDatetime && isc.SimpleType.inheritsFrom(field.type, "date"),
                formatter = grid._getDateFormatter(field);
            
            // rely on date.toShortDateTime() / toShortDate() to handle applying the
            // custom formatter if specified, otherwise picking up the appropriate system-wide
            // default for the data type.
            // The second parameter to toShortDateTime() explicitly causes the date to be displayed
            // in the custom timezone set up in Time.setDefaultDisplayTimezone
            if (isDatetime) return value.toShortDateTime(formatter, true);
            return value.toShortDate(formatter, !isLogicalDate);
        }
        return value;
    },
    
    // Date formatting helpers:
    // Called from formatDateCellValue() / formatTimeCellValue() [which are type-specific formatters
    // applied to fields as part of field init], and also as a catch-all for
    // Date type values in fields of some other specified data-type.
    // We use a consistent pattern across DataBoundComponents:
    // - developer can specify explicit dateFormatter / timeFormatter per field and they'll be used.
    //   if both are specified dateFormatter takes precedence except in "time" type fields.
    // - developer can specify dateFormatter, datetimeFormatter and timeFormatter per component and
    //   they'll be used if no per-field settings are found.

    // If a field has a JS Date value, should we format it as a time? True for "time" type fields
    // or fields with an explicit time formatter only.
    _formatAsTime : function (field) {
        if (field == null) return false;
        
        // If at the field level the timeFormatter is defined *(and there's no date formatter)
        // respect it.
        if (field.timeFormatter != null && field.dateFormatter == null) return true;
        // timeFormatter null and dateFormatter non-null --> format as date
        
        if (field.dateFormatter != null && field.timeFormatter == null) return false;
        return isc.SimpleType.inheritsFrom(field.type, "time");
    },
    
    _getDateFormatter : function (field) {
        
        if (field == null) return this.dateFormatter;
        
        if (field.dateFormatter != null) return field.dateFormatter;
        // displayFormat is back-compat at this point - only applies to fields of type
        // date or datetime (or subtypes thereof)
        if (field.displayFormat != null && isc.SimpleType.inheritsFrom(field.type, "date")) {
            return field.displayFormat;
        }
        
        if (this.datetimeFormatter != null && isc.SimpleType.inheritsFrom(field.type, "datetime")) {
            return this.datetimeFormatter;
        }
        
        return this.dateFormatter;
    },
    
    // This picks up *explicit* dateInputFormat to pass through to the edit-item. No need to
    // include logic to derive from the display format if no explicit input format was specified,
    // that'll be handled by the FormItem code.
    _getDateInputFormat : function (field) {
        var inputFormat;
        if (field) inputFormat = field.inputFormat
        if (!inputFormat) inputFormat = this.dateInputFormat;
        return inputFormat;
    },

    // function to call appropriate number formatter
    // If no number formatter is defined, the default formatter will be used [standard 'toString']
    // is the default
    _formatNumberCellValue : function (value, field, grid, record, rowNum, colNum) {
        if (isc.isA.Number(value)) {
            var formatter = (field.numberFormatter || field.formatter || grid.numberFormatter);
            return value.toFormattedString(formatter);
        }
        // If passed a non-number just return it
        return value;
    },
    
	//>	@attr listGrid.timeFormatter (TimeDisplayFormat : "toShortPaddedTime" : [IRW])
	// Display format to use for fields specified as type 'time'.  May also be specified at 
    // the field level via +link{listGridField.timeFormatter}.<br>
    // If unset, time fields will be formatted based on the system wide 
    // +link{Time.shortDisplayFormat}.<br>
    // If this field is editable, the timeFormatter will also be passed to the editor
    // created to edit any time type fields as +link{TimeItem.timeFormatter}
	// @group appearance
    // @visibility external
	//<
    timeFormatter:"toShortPaddedTime",
    
    _getTimeFormatter : function (field) {
        if (field != null) {
            if (field.timeFormatter != null) return field.timeFormatter;
            if (field.displayFormat != null && isc.SimpleType.inheritsFrom(field.type, "time")) {
                return field.displayFormat;
            }
        }
        return this.timeFormatter;
    },

	// function to call appropriate time formatter
	// Note: this is executed in the scope of a field object - see 'applyFieldDefaults'
    _formatTimeCellValue : function (value, field, grid, record, rowNum, colNum) {
        var time = value;
        if (isc.isA.String(time)) {
            // Pass in the 'validTime' param - If we're given a string which doesn't 
            // parse to a time we don't want to display "12:00 am"
            time = isc.Time.parseInput(time, true);
        }
        if (isc.isA.Date(time)) {
            // If dateFormatter is set on the field, and timeFormatter is not, we respect it
            // even for fields of explicit type "time"
            if (!grid._formatAsTime(field)) {
                
                return time.toShortDate(grid._getDateFormatter(field), true);
            }
            var formatter = grid._getTimeFormatter(field);
            
            // If we're passed an invalid formatter
            return isc.Time.toTime(time, formatter, true);
        }
        return value;
    },
    
    _formatBinaryCellValue : function (value, field, grid, record, rowNum, colNum) {
        
        if (isc.isA.String(value)) return value;
        if (record == null) return null;

        var fieldName = field.name,
            nativeName = field.nativeName || fieldName,
            fileName = record[nativeName + "_filename"],
            value;

        if (field.type=="imageFile" && field.showFileInline == true) {
            var urlProperty = fieldName + "_imgURL";
            
            if (!record[urlProperty]) {
                var dimensions = isc.Canvas.getFieldImageDimensions(field, record), 
                    image = grid.getDataSource().streamFile(record, field.name);
                    
                dimensions.width = dimensions.width || grid.imageSize;
                dimensions.height = dimensions.height || grid.imageSize;
                value = record[urlProperty] = 
                    isc.Canvas.imgHTML(image, dimensions.width, dimensions.height);
            } else 
                value = record[urlProperty];
        } else {
            if (field.showFileInline == true) { // non-imageFile field
	            this.logWarn("_formatBinaryCellValue(): Unsupported field-type for showFileInline: "+field.type);
            }
            
            if (fileName == null || isc.isA.emptyString(fileName)) return this.emptyCellValue;
            var viewIconHTML = isc.Canvas.imgHTML("[SKIN]actions/view.png", 16, 16, null,
                            "style='cursor:"+isc.Canvas.HAND+"' onclick='"+grid.getID()+".viewRow("+rowNum+")'");
            var downloadIconHTML = isc.Canvas.imgHTML("[SKIN]actions/download.png", 16, 16, null,
                            "style='cursor:"+isc.Canvas.HAND+"' onclick='"+grid.getID()+".downloadRow("+rowNum+")'");

            value = viewIconHTML + "&nbsp;" + downloadIconHTML + "&nbsp;" + fileName;
        }
        
        return value;
    },

    // value is rendered as an anchor - the href and name of the anchor is the value.  By default
    // opens in a new browser window - this can be overridden by setting the 'target' property on
    // the record.
    //
    // The name of the link can be overridden by setting the 'linkName' property on the record.  By
    // default we use the value.
    _$linkTemplate:[
        "<a href='",
        ,   // 1: HREF
        "' target='",
        ,   // 3: name of target window
        // onclick handler enables us to prevent popping a window if (EG) we're masked.        
        //                      5: ID              
        "' onclick='if(window.",     ,") return ",
                //  7:ID                         9:rowNum,     11:colNum        
                         ,"._linkClicked(event,",        ,",",          ,");'>",
        ,   // 13: link text
        "</a>"
    ],
    _$doubleEscapedQuote:"\\'",
    _$_blank:"_blank",
    
    _formatLinkCellValue : function (value, field, grid, record, rowNum, colNum) {
        if (value == null || isc.is.emptyString(value)) return value;

        // target window
        var target = field.target ? 
                        field.target.replaceAll(grid._$singleQuote, grid._$doubleEscapedQuote) : 
                        grid._$_blank;
        // get the linkText property. If defined on the field, use that, otherwise 
        // use the linkTextProperty from the grid.
        var linkTextProp = field.linkTextProperty ? field.linkTextProperty : grid.linkTextProperty;
        var linkText = (record && record[linkTextProp]) ? record[linkTextProp] 
                                                            : field.linkText || value;

        // link URL
        var href = value;

        if (field.linkURLPrefix) href = field.linkURLPrefix + href;
        if (field.linkURLSuffix) href = href + field.linkURLSuffix;

        href = href.replaceAll(grid._$singleQuote, grid._$doubleEscapedQuote);

        if (target == "javascript") {
            // target is "javascript" - make the link inert and have the cellClick event fired 
            // instead
            href = "javascript:void";
        }

        // combine
        var template = grid._$linkTemplate;
        template[1] = href;
        template[3] = target;
        var ID = grid.getID();
        template[5] = ID;
        template[7] = ID;
        template[9] = rowNum;
        template[11] = colNum;
        template[13] = linkText;

        return template.join(isc.emptyString);
    },
    
    _linkClicked : function (event, rowNum, colNum) {
        // don't allow the click if the cell should not be interactive.
        var mustCancel = (this.destroyed || !this.isDrawn() || !this.isVisible() ||
                          isc.EH.targetIsMasked(this.body) || !this.recordIsEnabled(rowNum, colNum));

        var record = this.getRecord(rowNum),
            field = this.getField(colNum);

        if (event.target == "javascript" || field.target == "javascript") {
            mustCancel=true;
            this.cellClick(record, rowNum, colNum);
        }

        if (mustCancel) {
            
            if (!isc.Browser.isIE) {
                event.preventDefault();
            }
            
            return false;
        }
        return true;
    },
    
    //> @attr listGrid.linkTextProperty (string : "linkText" : [IRW])
	// Property name on a record that will hold the link text for that record.
	// <P>
	// This property is configurable to avoid possible collision with data values in the
    // record.
    // <P>
    // Use +link{listGridField.linkTextProperty} if you have more than one link field and 
	//
	// @see type:ListGridFieldType
    // @see type:FieldType
    // @see attr:listGridField.linkText
    // @see attr:listGridField.linkTextProperty
	// @group  display_values
	// @visibility external
	//<
    linkTextProperty : "linkText",
    
    // value is a URL to an image
    _formatImageCellValue : function (value, field, grid, record, rowNum, colNum) {
        // if no value is stored, just return an empty string so we don't render a broken image
        if (value == null || value == isc.emptyString) return isc.emptyString;

        // if any of field.imageWidth/Height/Size are set as strings, assume they are property
        // names on the record
        
        var dimensions = isc.Canvas.getFieldImageDimensions(field, record);

        dimensions.width = dimensions.width || grid.imageSize;
        dimensions.height = dimensions.height || grid.imageSize;

        
        var src = value,
            prefix = field.imageURLPrefix || field.baseURL || field.imgDir;

        // If imageURLSuffix is specified, apply it to the value
        if (field.imageURLSuffix != null) src += field.imageURLSuffix;            
         
        

        return isc.Canvas.imgHTML(src, dimensions.width, dimensions.height, null, field.extraStuff, 
                                  prefix, field.activeAreaHTML);
    },

    // show field.icon in the cell
    _formatIconCellValue : function (value, field, grid, record, rowNum, colNum) {
        // prevent an icon from being shown in the filter editor if the field has canFilter
        if (isc.isA.RecordEditor(grid) && grid.isAFilterEditor() && field.canFilter == false) return null;
        
        if (field._iconHTML) return field._iconHTML;
        
        field._iconHTML = isc.Canvas.imgHTML(field.cellIcon || field.icon, 
                                             field.iconWidth || field.iconSize || grid.imageSize,
                                             field.iconHeight || field.iconSize || grid.imageSize);
        return field._iconHTML;
    },

	// CSS styles
	// --------------------------------------------------------------------------------------------

    //> @attr listGrid.fastCellUpdates (boolean: true : I)
    // @include gridRenderer.fastCellUpdates
    // @group performance
    //<
    // explicitly set fastCellUpdates at the LG level 
    // this will be passed through to our body and allows us to check
    // this.fastCellUpdates directly rather than looking at the attribute on the body
    
    fastCellUpdates:isc.Browser.isIE && !isc.Browser.isIE9,
    
    //> @method listGrid.setFastCellUpdates()
    // @include gridRenderer.setFastCellUpdates()
    // @visibility external
    //<
    // explicit implementation keeps this.fastCellUpdates in synch with the version in the
    // body so we can check it directly in this.getBaseStyle
    setFastCellUpdates : function (fcu) {
        if (this.body != null) {
            this.body.setFastCellUpdates(fcu);
            // if the body refused to set to the specified 
            // value, respect that.
            fcu = this.body.fastCellUpdates;
        }
        if (this.frozenBody != null) {
            this.frozenBody.setFastCellUpdates(fcu);
        }
        this.fastCellUpdates = fcu;
    },
	
	//>	@attr listGrid.baseStyle (CSSStyleName : null : [IR])
	// +link{gridRenderer.baseStyle,base cell style} for this listGrid.
	// If this property is unset, base style may be derived from +link{listGrid.normalBaseStyle}
	// or +link{listGrid.tallBaseStyle} as described in
	// +link{listGrid.getBaseStyle()}.
    // @visibility external
    // @group appearance
	//<
	
	//>	@attr listGrid.normalBaseStyle (CSSStyleName : "cell" : [IR])
	// "Normal" baseStyle for this listGrid. Only applies if +link{listGrid.baseStyle} is 
	// set to null.
	// <P>
	// If <code>baseStyle</code> is unset, this
	// property will be used as a base cell style if the grid is showing fixed height rows, and
	// the specified cellHeight matches +link{listGrid.normalCellHeight} (and in Internet Explorer,
	// +link{listGrid.fastCellUpdates} is false). Otherwise +link{listGrid.tallBaseStyle} will
	// be used.
	// <P>
	// Having separate styles defined for fixed vs. variable height rows allows the developer
	// to specify css which is designed to render at a specific height (typically using
	// background images, which won't scale), without breaking support for styling rows
	// of variable height.
	//
	// @see listGrid.getBaseStyle()
	// @visibility external
	//<
	normalBaseStyle:"cell",

	//>	@attr listGrid.tallBaseStyle (CSSStyleName : "cell" : [IR])
	// "Tall" baseStyle for this listGrid. Only applies if +link{listGrid.baseStyle} is 
	// set to null.
	// <P>
	// If <code>baseStyle</code> is unset, this
	// property will be used as a base cell style unless the grid is showing fixed height
	// rows with a specified cellHeight that matches +link{listGrid.normalCellHeight}, in
	// which case +link{listGrid.normalBaseStyle} will be used. Note that in Internet Explorer
	// if +link{listGrid.fastCellUpdates} is true, <code>tallBaseStyle</code> will also be
	// used even if the cellHeight matches the specified <code>normalCellHeight</code> for the
	// grid.
	//
	// @see listGrid.getBaseStyle()
	// @visibility external
	//<
	tallBaseStyle:"cell",

	
    
    //>	@attr listGrid.editFailedBaseStyle (CSSStyleName : null : [IRWA])
	//  A base name for the CSS class applied to cells when editing has failed.<br>
	//  If this listGrid is editable, this style will be applied to any edited cells for which
	//  validation failed.<br>
	//  As with the default 'baseStyle' property, this style will have "Dark", "Over", "Selected", 
	//  or "Disabled" appended to it according to the state of the cell.<br>
    // If null, cells for which editing has failed will be rendered using the normal base style
    // classNames, but with custom CSSText applied as derived from <code>this.editFailedCSSText</code>
	// @visibility external
	// @group	appearance
	// @see baseStyle
    // @see editFailedCSSText
	//<
    editFailedBaseStyle:null,   //"cellEditFailed",
    
	//>	@attr listGrid.editFailedCSSText (string : "color:red;border:1px solid red;" : [IRWA])
	//  Custom CSS text to be applied to cells when editing has failed.<br>
	//  If this listGrid is editable, this css text will be applied to any edited cells for which
	//  validation failed, on top of the base style for the cell.<br>
    // For further customization of styling for cells that failed editing validation, use
    // <code>this.editFailedBaseStyle</code> instead.
	// @visibility external
	// @group	appearance
    // @see editFailedBaseStyle
	//<
    editFailedCSSText:"color:red;border:1px solid red;",
    
	//>	@attr listGrid.editPendingBaseStyle (CSSStyleName : null : [RA])
	//  A base name for the CSS class applied to cells containing pending (unsaved) edits<br>
	//  As with the default 'baseStyle' property, this style will have "Dark", "Over", "Selected", 
	//  or "Disabled" appended to it according to the state of the cell.
    //  If this property is null, cells with pending edits will pick up custom css text to
    //  be applied on top of the normal base style from <code>this.editPendingCSSText</code>
	//      @visibility external
	//		@group	appearance
	//      @see baseStyle
	//<    
    editPendingBaseStyle:null, //"cellEditPending",

	//>	@attr listGrid.editPendingCSSText (string : "color:#0066CC;" : [IRWA])
	//  Custom CSS text to be applied to cells with pending edits that have not yet been
    //  submitted.<br>
    // For further customization of styling for cells with pending edits use
    // <code>this.editPendingBaseStyle</code> instead.
	// @visibility external
	// @group	appearance
    // @see editFailedBaseStyle
	//<
    editPendingCSSText:"color:#0066CC;",

    //> @attr listGrid.recordCustomStyleProperty (String : "customStyle" : IRW)
    // @include GridRenderer.recordCustomStyleProperty
    // @visibility external
    // @see listGrid.getCellStyle()
    // @see listGrid.recordBaseStyleProperty
    //<
    recordCustomStyleProperty:"customStyle",
    
    //> @attr listGrid.recordBaseStyleProperty (string : "_baseStyle" : [IRWA])
    // This attribute allows custom base styles to be displayed on a per-record basis.
    // To specify a custom base-style for some record set 
    // <code>record[listGrid.recordBaseStyleProperty]</code> to the desired base style name - 
    // for example if <code>recordBaseStyleProperty</code> is <code>"_baseStyle"</code>, set
    // <code>record._baseStyle</code> to the custom base style name.
    // 
    // @visibility external
    // @group appearance
    // @see listGrid.baseStyle
    //<
    recordBaseStyleProperty:"_baseStyle",
    
    //> @attr listGrid.frozenBaseStyle (string : null : [IRW])
    // If this listGrid contains any frozen fields, this property can be used to apply a custom
    // baseStyle to all cells in those frozen fields. If unset, the standard base style will be
    // used for both frozen and unfrozen cells.
    // @visibility external
    // @group appearance, frozenFields
    // @see listGrid.baseStyle
    // @see listGridField.frozen
    //<
    
    //> @attr listGrid.shrinkForFreeze (boolean : false : IRWA)
    // If this list grid is showing any +link{listGridField.frozen,frozen} fields, and a horizontal
    // scrollbar is visible at the bottom of the liquid columns, should an equivalent scrollbar gap
    // be left visible below the frozen columns?<br>
    // Note that if set to <code>true</code> any backgroundColor or border applied to the ListGrid
    // will show up below the bottom row of the frozen column(s).
    // @group frozenFields
    // @visibility external
    //<
    shrinkForFreeze:false,
    
    //> @attr listGrid.alternateRecordStyles (boolean : false : [IRW])
    // @include gridRenderer.alternateRowStyles
	// @group cellStyling
    // @example gridCells
    //<
	//alternateRecordStyles:false,
    
    //> @attr listGrid.alternateBodyStyleName (CSSStyleName : null : [IRWA])
    // Optional css style to apply to the body if +link{listGrid.alternateRecordStyles} is true 
    // for this grid. If unset +link{listGrid.bodyStyleName} will be used to style the body
    // regardless of the +link{listGrid.alternateRecordStyles,alternateRecordStyles} setting.
    // @visibility external
    //<

	//>	@attr listGrid.alternateRecordFrequency (number : 1 : [IRW])
	// @include gridRenderer.alternateRowFrequency
	// @group cellStyling
	//<
	alternateRecordFrequency:1,

    // property you can set per-record to add custom CSSText
    recordCSSTextProperty : "cssText",

	//>	@attr listGrid.showHiliteInCells (boolean : false : IRWA)
	// When cell styling is being updated (updateCellStyle()), should the HTML content of the
	// cell also be updated?  If false, only the cell's CSS styling will be updated. 
	//
	// You should turn this on if you've implemented a <code>formatCellValue</code> function 
	// that adds styling cues to a cell (like an inline image), which need be updated as the 
	// cell switches states.
	// (eg, if you would use different HTML for a selected cell's contents).
	//		@group	hiliting, drawing
	//<
	//showHiliteInCells:false,				

    //> @attr listGrid.showSelectedStyle ( boolean : true : IRW )
    // @include gridRenderer.showSelectedStyle
    //<
    showSelectedStyle : true,

    // Keyboard handling
    // ---------------------------------------------------------------------------------------

    //> @attr listGrid.generateClickOnSpace (boolean : true : IRWA)
    // If true, when the user navigates to a cell using arrow keys and hits space, 
    // the cell will respond to a click event.
    // @visibility external
    //<
    generateClickOnSpace : true,
		
    //> @attr listGrid.generateClickOnEnter (boolean : false : IRWA)
    // If true, when the user navigates to a cell using arrow keys and hits Enter, 
    // the cell will respond to a click event.
    // @visibility external
    //<
    //generateClickOnEnter : false,

    //> @attr listGrid.generateDoubleClickOnSpace (boolean : false : IRWA)
    // If true, when the user navigates to a cell using arrow keys and hits Space, 
    // the cell will respond to a double click event.
    // @visibility external
    //<
    // generateDoubleClickOnSpace : false,
    
    //> @attr listGrid.generateDoubleClickOnEnter (boolean : true : IRWA)
    // If true, when the user navigates to a cell using arrow keys and hits Enter, 
    // the cell will respond to a double click event.
    // @visibility external
    //<
    generateDoubleClickOnEnter : true,

    //> @attr listGrid.arrowKeyAction (string : "select" : IRWA)
    // Action to perform when the listGrid has keyboard focus (but not editing focus) and a user
    // presses the up or down arrow key. Possible values are:
    // <ul>
    // <li><code>select</code> : select the next row in the list (calls <code>recordClick</code> handler)</li>
    // <li><code>focus</code> : move focus to the next row in the list without changing the selection</li>
    // <li><code>activate</code> : select and activate the next row in the list (calls
    //  <code>recordDoubleClick</code> handler)</li>
    // <li><code>none</code> : no action</li>
    // </ul>
    // @group events
    // @visibility external
    //<
    arrowKeyAction:"select",

    //> @attr listGrid.showRecordComponents (boolean : null : IRWA)
    // If set to true, this listGrid should create and show an embedded component in every
    // row of the grid.
    // <P>
    // Developers using this feature should implement the 
    // +link{listGrid.createRecordComponent()} and +link{listGrid.updateRecordComponent()}
    // methods.
    // <P>
    // createRecordComponent() will be called by the grid as rows are rendered to create the
    // record components to use per record or per cell.  Your implementation should return a
    // component to embed in the record passed in. Note that this method should create and
    // return a new component each time it is called.
    // <P>
    // This feature also supports reusing components in different rows in the grid. If 
    // +link{listGrid.recordComponentPoolingMode} is set to <code>"recycle"</code>,
    // components created by the <code>createRecordComponent</code>
    // method will become available for reuse when they are no longer associated with a record.
    // The system will automatically store these in a pool. When a record with no
    // associated component is rendered, if there are any recordComponents in this pool, 
    // the system will call +link{listGrid.updateRecordComponent()}, and pass in the component.
    // This allows the developer to apply record-specific attributes to an already created
    // component and render it out into the new record. This greatly improves performance for 
    // large grids as it allows a small number of components to be created and reused rather 
    // than maintaining potentially one record component for every cell in the grid.
    // <P>
    // Record components are refreshed according to the recordComponentPooling mode. If set to 
    // <code>data</code> components will be maintained as long as their associated record remains
    // present in the data set (but this can require a component to be created for every record so
    // is not desirable for large data sets). Otherwise record components are refreshed as they
    // are scrolled into view. Note that you can explicitly refresh record components via
    // +link{listGrid.invalidateRecordComponents()} and +link{listGrid.refreshRecordComponent()}
    // <P>
    // NOTE: recordComponents can have an impact on row height and therefore may require
    // +link{listGrid.virtualScrolling}.
    // This is not supported in conjunction with +link{ListGridField.frozen,frozen fields}.
    // If you are using recordComponents in a listGrid with frozenFields, you can specify an
    // explicit +link{listGrid.recordComponentHeight} to ensure every row in the grid renders
    // tall enough to accommodate the recordComponents, and as such virtual scrolling is not
    // required.
    //
    // @see recordComponentPosition
    // @see showRecordComponentsByCell
    // @see recordComponentPoolingMode
    // @see showRecordComponent()
    // @see createRecordComponent()
    // @see updateRecordComponent()
    //
    // @visibility external
    //<
    
    //> @type EmbeddedPosition
    // How a component should be embedded within its record or cell
    // @value "expand" component should be placed underneath normal record or cell content,
    //        expanding the records.  Expanding records can result in variable height rows,
    //        in which case +link{listGrid.virtualScrolling,virtualScrolling} should be
    //        enabled.
    // @value "within" component should be placed within the normal area of the record or cell.
    //        Percentage sizes will be treated as percentages of the record and
    //        +link{Canvas.snapTo} positioning settings are also allowed and refer to the
    //        rectangle of the record or cell. Note that for components embedded within cells,
    //        cell align and vAlign will be used if snapTo is unset (so top / left alignment
    //        of cell content will map to snapTo of "TL", etc).
    // @visibility external
    //<

    //> @attr listGrid.recordComponentPosition (EmbeddedPosition : null : IRW)
    // if +link{listGrid.showRecordComponents} is true, how should the component appear within
    // the cell. Valid options are 
    // <ul><li><code>"within"</code>: the component will be rendered inside the record / cell.
    //  +link{canvas.snapTo} may be set to specify where the component should render within
    //  the row or cell, and +link{canvas.snapOffsetTop} / +link{canvas.snapOffsetLeft} may
    //  be set to indent recordComponents within their parent cells.
    //  Note that if unset, the component will show up at the top/left edge
    //  for components embedded within an entire row, or for per-cell components, cell
    //  align and valign will be respected.  Note also that, when rendering components "within"
    //  cells, specified component heights will be respected and will change the height of the 
    //  row.  However, if you want components to completely fill a cell at it's default height, 
    //  set height: "100%" or rows will render at the default height of the component. </li> 
    // <li><code>"expand"</code>: the component will be written into the cell below the
    //  normal cell content, causing the cell to expand vertically to accommodate it.
    // <li><code>null</code>: If this attribute is unset, we will default to showing
    //  recordComponents with position <code>"within"</code> if 
    //  +link{showRecordComponentsByCell} is true, otherwise using <code>"expand"</code>
    //  logic.
    // </ul>
    // @see showRecordComponents
    // @visibility external
    //<
    
//    recordComponentPosition:"expand",
    
    //> @attr listGrid.showRecordComponentsByCell (boolean : null : IRWA)
    // If true, shows +link{listGrid.showRecordComponents, recordComponents} in cells, rather 
    // than just in records.
    // @visibility external
    //<

    //> @type RecordComponentPoolingMode
    // The method of component-pooling to employ for +link{showRecordComponents,recordComponents}.
    // <P>
    // @value "viewport" components are destroyed when the record is not being rendered.  Best 
    //        for large datasets where embedded components differ greatly per record.
    // @value "data" components are cleared when not in the viewport, but stay with a record 
    //        until the record is dropped from cache.  Best for guaranteed small datasets.
    // @value "recycle" components are pooled and will be passed to getEmbeddedComponent() with 
    //       "recordChanged" set to true.  Best for large datasets where embedded components 
    //       are uniform across different records and can be efficiently reconfigured to work 
    //       with a new record
    // @visibility external
    //<

    //> @attr listGrid.recordComponentPoolingMode (RecordComponentPoolingMode : "viewport" : IRWA)
    // The method of +link{type:RecordComponentPoolingMode, component-pooling} to employ for 
    // +link{showRecordComponents,recordComponents}.
    // <P>
    // The default mode is "viewport", which means that recordComponents are destroyed as soon 
    // their record leaves the viewport.  
    // <P>
    // For the most efficient implementation, switch to "recycle" mode, which pools components 
    // when records leave the viewport and re-uses them in other records.  In this mode, you 
    // should implement +link{listGrid.updateRecordComponent, updateRecordComponent()} to 
    // apply any changes to make reused components applicable to the new record they appear in, 
    // if necessary.  If you have components of different types in different columns and still
    // want to take advantage of component recycling, you can 
    // set +link{listGrid.poolComponentsPerColumn} to ensure that components intended for one 
    // column are not recycled for use in another column that should have a different component.
    // <P>
    // Note that, if different records have distinctly different components embedded 
    // in them, or multiple columns in each record embed different components, you should 
    // leave the recordComponentPoolingMode at "viewport" if your dataset is very large or 
    // use "data" otherwise.
    // @visibility external
    //<
    recordComponentPoolingMode:"viewport",

    //> @attr listGrid.poolComponentsPerColumn (boolean : null : IRW)
    // When +link{listGrid.recordComponentPoolingMode} is "recycle" and you have components of 
    // different types in different columns, set this property to true to ensure that 
    // components intended for one column are not recycled for use in another column that 
    // should have a different component.
    // <P>
    // If no components applicable to a particular column are available in the pool, the system
    // calls +link{listGrid.createRecordComponent, createRecordComponent}.
    // 
    // @visibility external
    //<

	// Rollover
	// --------------------------------------------------------------------------------------------
	//>	@attr listGrid.showRollOver (boolean : true : IRW)
	// Should we show different styling for the cell the mouse is over?
    // <br>
    // If true, the cell style will have the suffix "Over" appended.
	// @group appearance
    // @visibility external
	//<
    
	showRollOver:!isc.Browser.isTouch,
    
    //> @attr listGrid.showRollOverCanvas (boolean : null : IRWA)
    // If +link{listGrid.showRollOver} is true, this property provides an option to show roll over
    // styling with the +link{listGrid.rollOverCanvas} and +link{listGrid.rollUnderCanvas}
    // rather than using css styling.
    // @group rowEffects
    // @visibility external
    //<
    //showRollOverCanvas:null,
    
    //> @attr listGrid.rollOverCanvas (AutoChild : null : RA)
    // AutoChild created and embedded in the body if +link{listGrid.showRollOver} is true, and
    // +link{listGrid.showRollOverCanvas} is true.  This autoChild canvas will be created and 
    // displayed above the current rollOver row.
    // <P>
    // Note that the rollOverCanvas has the following read-only attributes set:<br>
    // - <code>this.grid</code> - a pointer to the ListGrid showing the rollOverCanvas<br>
    // - <code>this.record</code> - a pointer to the current roll over record object in the grid.
    // 
    // @group rowEffects
    // @visibility external
    //<

	//> @attr listGrid.rollUnderCanvas (AutoChild : null : RA)
    // AutoChild created and embedded in the body if +link{listGrid.showRollOver} is true, and
    // +link{listGrid.showRollOverCanvas} is true.  This autoChild canvas will be created and 
    // displayed behind the current rollOver cell in the page's z-order, meaning it will only be
    // visible if the cell styling is transparent. 
    // <P>
    // Note that the rollUnderCanvas has the following read-only attributes set:<br>
    // - <code>this.grid</code> - a pointer to the ListGrid showing the rollUnderCanvas<br>
    // - <code>this.record</code> - a pointer to the current roll over record object in the grid.
    // @group rowEffects
    // @visibility external
    //<

    //> @attr listgrid.showBackgroundComponent (boolean : false : IRW)
    // If <code>true</code> this grid will create and show per-row backgroundComponents 
    // as detailed +link{listGrid.backgroundComponent,here}.
    // @visibility external
    //<
    
	//> @attr listGrid.backgroundComponent (Canvas : null : IR)
	// Has no effect unless +link{listGrid.showBackgroundComponent} is <code>true</code>.
	// <P>
    // Canvas created and embedded in the body behind a given record.   When 
    // +link{listGridRecord.backgroundComponent} is set, this autoChild canvas 
    // will be constructed (if listGridRecord.backgroundComponent is not already a Canvas) and 
    // it's properties combined with those of listGridRecord.backgroundComponent and then 
    // displayed behind a specific record in the page's z-order, meaning 
    // it will only be visible if the cell styling is transparent.
    // @group rowEffects
    // @visibility external
    //<
    backgroundComponentDefaults: {
        snapTo:"TL",
        autoDraw: false,
        opacity: "50%"
    },

    //>Animation
    
    //> @attr listGrid.animateRollOver (boolean : false : IRWA)
    // If +link{listGrid.showRollOverCanvas} is <code>true</code> setting this property to true
    // ensures that when the rollOver canvas is displayed it animates into view via an
    // +link{Canvas.animateShow()}. Note that the animation effect may be customized via the
    // standard +link{Canvas.animateShowEffect}, +link{Canvas.animateShowTime} and 
    // +link{Canvas.animateShowAcceleration}.
    // @group rowEffects
    // @visibility external
    //<
    
    //> @attr listGrid.animateRollUnder (boolean : false : IRWA)
    // If +link{listGrid.showRollOverCanvas} is <code>true</code> setting this property to true
    // ensures that when the rollUnder canvas is displayed it animates into view via an
    // +link{Canvas.animateShow()}. Note that the animation effect may be customized via the
    // standard +link{Canvas.animateShowEffect}, +link{Canvas.animateShowTime} and 
    // +link{Canvas.animateShowAcceleration}.
    // @group rowEffects
    // @visibility external
    //<
    
    //<Animation
    
    //> @attr listGrid.useCellRollOvers (boolean : false : IRW)
    // Are rollovers cell-level or row-level?
    // @visibility external
    //<
	//useCellRollOvers:false,
	
	// Hover
	// --------------------------------------------------------------------------------------------

	//>	@attr listGrid.canHover (boolean : null : [IRW])
	// @include gridRenderer.canHover
    // @group hovers
    // @see attr:listGrid.showHover
    // @see attr:listGridField.showHover
    // @example valueHoverTips
	//<
	// are hover events and hover popups enabled?
	//canHover:false,							

	//>	@attr listGrid.showHover (boolean : true : [IRW])
	// @include gridRenderer.showHover
    // @group hovers
	//<
	// if canHover:true, should we show hover popups?
	showHover: true,							

	//>	@attr listGridField.showHover (boolean : null : IRW)
	// Whether to show hovers for this field.  The default hover will be the contents of the
    // cell the user is hovering over, and can be customized via
    // +link{listGridField.hoverHTML,field.hoverHTML()}.
    // <P>
    // +link{ListGrid.canHover} can be set to true to cause hovers to be shown for all fields
    // by default.  In this case, <code>field.showHover</code> can be set to false to suppress
    // hovers for an individual field.
    // <P>
    // All hovers can be disabled, regardless of other settings, by setting
    // +link{ListGrid.showHover} to false.
    // @visibility external
    // @example valueHoverTips
	//<
    
	// can be set to false to cause hover to be per-row instead of per-cell
	//hoverByCell:true,
	
	// if canHover:true, should an active hover remain active until we leave the listGrid?
	// default behavior is to clear/deactivate the hover on each cellOut/rowOut
	//keepHoverActive:false,	

	// the space between the borders of the cell and the hover, in pixels
    cellHoverOutset:5,
    
    // Note: hoverWidth, hoverStyle, et al will be picked up by the grid renderer when showing 
    // cell hovers (handled by GridRenderer class)

    //> @attr listGrid.hoverStyle (CSSStyleName : "gridHover" : [IRWA])
    // Style to apply to hovers shown over this grid.
    // @see listGrid.showHover
    // @group hovers
    // @visibility external
    //<
    hoverStyle:"gridHover",

	//>	@attr listGridField.prompt (String : null : IR)
    // Causes a tooltip hover to appear on the header generated for this field (effectively
    // sets +link{canvas.prompt} for the header).
    //
    // @visibility external
    //<

	// Selection
	// --------------------------------------------------------------------------------------------

	//>	@attr listGrid.selection (Selection : null : [RA])
	// The Selection object associated with the listGrid.
	// @group  selection
	// @visibility external
	//<

    //> @attr listGrid.selectionAppearance (SelectionAppearance : "rowStyle" : IRW)
    // How selection of rows should be presented to the user.
    // <P>
    // For <code>selectionAppearance:"checkbox"</code> with multiple selection
    // allowed, you would typically use +link{listGrid.selectionType}:"simple" (the default).  Because 
    // <code>selectionType</code> and <code>selectionAppearance</code> are unrelated, 
    // the combination of <code>selectionAppearance:"checkbox"</code> and
    // <code>selectionType:"multiple"</code> results in a grid where multiple selection can
    // only be achieved via shift-click or ctrl-click.  
    // <P>
    // If using <code>"checkbox"</code> for a +link{listGrid}, see also
    // +link{listGrid.checkboxField} for customization APIs.
    // <P>
    // If using <code>"checkbox"</code> for a +link{treeGrid}, an extra icon,
    // +link{treeGrid.getExtraIcon} is not supported. Additionally only
    // +link{listGrid.selectionType}:"simple" and "single" are supported.
    // You can also toggle the display of a disabled checkbox on a treegrid, displayed 
    // when the node can't be selected, via +link{TreeGrid.showDisabledSelectionCheckbox}.
    // @group selection
    // @visibility external
    //<
    selectionAppearance: "rowStyle",

    //>	@attr listGrid.canSelectAll (boolean : null : [IRW])
    // Controls whether a checkbox for selecting all records appears in the header with 
    // +link{listGrid.selectionAppearance, selectionAppearance} set to "checkbox"
    // 
    // @group selection
    // @visibility external
    //<
    
    //>	@attr listGrid.selectionType (SelectionStyle : null : [IRW])
	// Defines a listGrid's clickable-selection behavior.  
    // <P>
    // The default selection appearance is governed by +link{listGrid.selectionAppearance}: if
    // selectionAppearance is "checkbox", this will be "simple", otherwise, this will be
    // "multiple".
	// 
    // @group	selection, appearance
	// @see type:SelectionStyle
	//      @visibility external
    // @example multipleSelect
	//<
    
	//>	@attr listGrid.selectionProperty (string : null : IRA)
	// If specified, the selection object for this list will use this property to mark records
	// as selected.  In other words, if this attribute were set to <code>"isSelected"</code>
	// any records in the listGrid data where <code>"isSelected"</code> is <code>true</code>
	// will show up as selected in the grid. Similarly if records are selected within the grid
	// after the grid has been created, this property will be set to true on the selected
	// records.
	//
	// @group  selection, appearance
	// @visibility external
	//<    
	//selectionProperty:null,	
	
    //> @attr listGrid.recordCanSelectProperty (String : "canSelect" : IRA)
    // If set to false on a record, selection of that record is disallowed.
    // @visibility external
    //< 
	recordCanSelectProperty:"canSelect",

	//>	@attr listGrid.canSelectCells (boolean : false : IRW)
	//  @group	selection
	//  If this property is true, users can select individual cells rather than entire rows
	//  at once.
	//<
	//canSelectCells:false,

	//>	@attr listGrid.canDragSelect (boolean : false : IRW)
	//  If this property is true, users can drag the mouse to select several rows or cells.
	//  This is mutually exclusive with rearranging rows or cells by dragging.
	//  @group	selection
	//  @visibility	external
    //  @example dragListSelect
	//<
	//canDragSelect:false,

	//>	@attr listGrid.canDragSelectText (boolean : false : IRW)
	// If this property is true, users can drag the mouse to select text within grid rows.
	// This is mutually exclusive with 
    // +link{canDragReorder,rearranging rows or cells by dragging}, and with 
    // +link{canDragSelect,drag selection of rows}.
    //
    // @group selection
	// @visibility external
	//<
	//canDragSelect:false,
    
    //> @attr listGrid.showSelectionCanvas (boolean : null : IRWA)
    // If +link{listGrid.selectionType} is set to <code>"single"</code>, setting this property to
    // true means selection will be displayed to the user with the +link{listGrid.selectionCanvas} 
    // and +link{listGrid.selectionUnderCanvas} rather than using css styling.
    // @group rowEffects
    // @visibility external
    //<
    //showSelectionCanvas:null,
    
    //> @attr listGrid.selectionCanvas (autoChild : null : RA)
    // AutoChild created and embedded in the body if +link{listGrid.showSelectionCanvas} is true,
    // and +link{listGrid.selectionType} is <code>"single"</code>.
    // This autoChild canvas will be created and displayed above the selected record on every 
    // selection change.
    // <P>
    // Note that the selectionCanvas has the following read-only attributes set:<br>
    // - <code>this.grid</code> - a pointer to the ListGrid showing the selectionCanvas<br>
    // - <code>this.record</code> - a pointer to the currently selected record object in the grid.
    // @group rowEffects
    // @visibility external
    //<
    
	//> @attr listGrid.selectionUnderCanvas (autoChild : null : RA)
    // AutoChild created and embedded in the body if +link{listGrid.showSelectionCanvas} is true,
    // and +link{listGrid.selectionType} is <code>"single"</code>.
    // displayed behind the current selected cell in the page's z-order, meaning it will only be
    // visible if the cell css styling is transparent. 
    // <P>
    // Note that the selectionUnderCanvas has the following read-only attributes set:<br>
    // - <code>this.grid</code> - a pointer to the ListGrid showing the selectionUnderCanvas<br>
    // - <code>this.record</code> - a pointer to the current selected record object in the grid.
    // @group rowEffects
    // @visibility external
    //<
    
    //> @attr listGrid.checkboxField (AutoChild : null : IR)
    // Returns the specially generated checkbox field used when +link{selectionAppearance} is
    // "checkbox".  Created via the +link{AutoChild} pattern so that
    // <code>checkboxFieldDefaults</code> and <code>checkboxFieldProperties</code> are available
    // for skinning purposes. Note that +link{listGridField.shouldPrint} is <code>false</code>
    // for the checkboxField by default - if you want this column to show up in the grid's print
    // view, use <code>checkboxFieldProperties</code> to set this property to true.
    // <P>
    // This field will render an icon to indicate the selected state of each row, which, when
    // clicked will toggle the selection state. The icon src may be configured using 
    // +link{listGrid.checkboxFieldTrueImage} and +link{listGrid.checkboxFieldFalseImage}, as
    // well as +link{listGrid.checkboxFieldImageWidth} and +link{listGrid.checkboxFieldImageHeight}.
    // <P>
    // The checkboxField can be detected by calling +link{listGrid.isCheckboxField()} on any
    // ListGridField object.
    //
    // @group checkboxField
    // @visibility external
    //<
    
    //> @attr listGrid.checkboxFieldTrueImage (SCImgURL : null :IRWA)
    // If +link{listGrid.selectionAppearance} is set to <code>"checkbox"</code> this property
    // determines the image to display in the checkbox field for a selected row.
    // If unset, the +link{listGrid.booleanTrueImage} will be used.
    // @see listGrid.checkboxFieldFalseImage
    // @see listGrid.checkboxFieldImageWidth
    // @see listGrid.checkboxFieldImageHeight
    // @group checkboxField
    // @visibility external
    //<
    
    //> @attr listGrid.checkboxFieldFalseImage (SCImgURL : null :IRWA)
    // If +link{listGrid.selectionAppearance} is set to <code>"checkbox"</code> this property
    // determines the image to display in the checkbox field for an unselected row.
    // If unset, the +link{listGrid.booleanFalseImage} will be used.
    // @see listGrid.checkboxFieldTrueImage
    // @see listGrid.checkboxFieldImageWidth
    // @see listGrid.checkboxFieldImageHeight
    // @group checkboxField
    // @visibility external
    //<
    
    //> @attr listGrid.checkboxFieldPartialImage (SCImgURL : null :IRWA)
    // If +link{listGrid.selectionAppearance} is set to <code>"checkbox"</code> this property
    // determines the image to display in the checkbox field for a partially selected row.
    // If unset, the +link{listGrid.booleanPartialImage} will be used.
    // @see listGrid.checkboxFieldTrueImage
    // @see listGrid.checkboxFieldImageWidth
    // @see listGrid.checkboxFieldImageHeight
    // @group checkboxField
    // @visibility external
    //<
    
    //> @attr listGrid.checkboxFieldImageWidth (integer : null : IR) 
    // If +link{listGrid.selectionAppearance} is set to <code>"checkbox"</code> this property
    // may be set to govern the width of the checkbox image displayed to indicate whether a row is
    // selected. If unset, the checkboxField image will be sized to match the
    // +link{listGrid.booleanImageWidth} for this grid.
    // @group checkboxField
    // @visibility external
    //<
    
    //> @attr listGrid.checkboxFieldImageHeight (integer : null : IR) 
    // If +link{listGrid.selectionAppearance} is set to <code>"checkbox"</code> this property
    // may be set to govern the height of the checkbox image displayed to indicate whether a row is
    // selected. If unset, the checkboxField image will be sized to match the
    // +link{listGrid.booleanImageHeight} for this grid.
    // @group checkboxField
    // @visibility external
    //<
          
    //>Animation
    
    //> @attr listGrid.animateSelection (boolean : false : IRWA)
    // If +link{listGrid.showSelectionCanvas} is <code>true</code> setting this property to true
    // ensures that when the selection canvas is displayed it animates into view via an
    // +link{Canvas.animateShow()}.  Note that the animation effect may be customized via the
    // standard +link{Canvas.animateShowEffect}, +link{Canvas.animateShowTime} and 
    // +link{Canvas.animateShowAcceleration}.
    // @group rowEffects
    // @visibility external
    //<
    
    //> @attr listGrid.animateSelectionUnder (boolean : false : IRWA)
    // If +link{listGrid.showSelectionCanvas} is <code>true</code> setting this property to true
    // ensures that when the selectionUnder canvas is displayed it animates into view via an
    // +link{Canvas.animateShow()}. Note that the animation effect may be customized via the
    // standard +link{Canvas.animateShowEffect}, +link{Canvas.animateShowTime} and 
    // +link{Canvas.animateShowAcceleration}.
    // @group rowEffects
    // @visibility external
    //<
    
    //<Animation

	// Empty and loading messages
	// --------------------------------------------------------------------------------------------

    //>	@attr listGrid.showEmptyMessage (boolean : true : [IRW])
    // @include gridRenderer.showEmptyMessage
    // @example emptyGrid
    //<
	showEmptyMessage:true,
    
	//>	@attr listGrid.emptyMessage (HTMLString : "No items to show." : [IRW])
    // @include gridRenderer.emptyMessage
    // @example emptyGrid
	//<
	emptyMessage:"No items to show.",

	//>	@attr listGrid.emptyMessageStyle (CSSStyleName : "emptyMessage" : [IRW])
    // The CSS style name applied to the +link{emptyMessage} if displayed.
    // @group emptyMessage
    // @visibility external
	//<
	emptyMessageStyle:"emptyMessage",

    //> @attr listGrid.filterButtonPrompt (String : "Filter" : [IR])
    // The prompt to show when the mouse hovers over the Filter button in the FilterEditor.
    // @group i18nMessages
    // @visibility external
	//<
    filterButtonPrompt : "Filter",

	//>	@attr listGrid.loadingDataMessage (HTMLString : "\${loadingImage}&amp;nbsp;Loading data..." : [IRW])
    // The string to display in the body of a listGrid while data is being loaded.
    // Use <code>"\${loadingImage}"</code> to include +link{Canvas.loadingImageSrc,a loading image}.
    // @see loadingDataMessageStyle
    // @group emptyMessage, i18nMessages
    // @visibility external
	//<
    loadingDataMessage : "${loadingImage}&nbsp;Loading data...",

	//>	@attr listGrid.loadingDataMessageStyle (CSSStyleName : "loadingDataMessage" : [IRW])
    // The CSS style name applied to the loadingDataMessage string if displayed.
    // @group emptyMessage
    // @visibility external    
	//<
    loadingDataMessageStyle: "loadingDataMessage",

	//>	@attr listGrid.loadingMessage (string : "&amp;nbsp;" : IR)
    // If you have a databound listGrid and you scroll out of the currently loaded dataset, by
    // default you will see blank rows until the server returns the data for those rows.  The
    // loadingMessage attribute allows you to specify arbitrary html that will be shown in each
    // such "blank" record while the data for that record is loading.
    //
	// @group emptyMessage, i18nMessages
    // @visibility external
	//<
	loadingMessage:"&nbsp;",	

    // Separator / Single Cell rows
    // ---------------------------------------------------------------------------------------

    //> @attr listGrid.singleCellValueProperty (string : "singleCellValue" : IRW)
    // If <code>record[this.singleCellValueProperty]</code> is set for some record, the 
    // record will be displayed as a single cell spanning every column in the grid, with 
    // contents set to the value of <code>record[this.singleCellValueProperty]</code>.
    // @visibility external
    //<
    singleCellValueProperty:"singleCellValue",

    //> @attr listGrid.isSeparatorProperty (string : "isSeparator" : IRW)
    // If <code>record[this.isSeparatorProperty]</code> is set for some record, the 
    // record will be displayed as a simple separator row.
    // @visibility external
    //<
    isSeparatorProperty:"isSeparator",
 
    // Filter Editor (aka QBE)   
    // ---------------------------------------------------------------------------------------

	//> @attr listGrid.showFilterEditor (boolean : null : IRW)
	// Should this listGrid display a filter row.  If true, this ListGrid
	// will be drawn with a single editable row, (separate from the body) with a filter button.
    // <P>
    // Values entered into this row are used as filter criteria to filter this List's data on
    // enter-keypress or filter button click. +link{listGrid.autoFetchTextMatchStyle} determines
    // the textMatchStyle for the request passed to +link{listGrid.fetchData()}.
    // <P>
    // Note that if +link{listGrid.filterData()} or +link{listGrid.fetchData()} is called directly
    // while the filter editor is showing, the filter editor values will be updated to reflect the
    // new set of criteria. If you wish to retain the user entered filter criteria and 
    // programmatically modify a subset of field values programmatically this can be achieved by
    // deriving new criteria by copying the existing set of criteria and adding other changes - 
    // something like this:
    // <pre><code>
    //   var newCriteria = myListGrid.getFilterEditorCriteria();
    //   isc.addProperties(newCriteria, {
    //      field1:"new value1",
    //      field2:"new value2"
    //   });
    //   myListGrid.setCriteria(newCriteria);
    // </code></pre>
    // In this example code we're using +link{listGrid.getFilterEditorCriteria()} rather than 
    // +link{listGrid.getCriteria()} - this ensures that if the user has typed a new value into
    // the filter editor, but not yet clicked the filter button, we pick up the value the user
    // entered.
    // <P>
    // If you call <code>filterData()</code> and pass in criteria for dataSource 
    // fields that are not present in the ListGrid, these criteria will continue to be applied along
    // with the user visible criteria.
    // <P>
    // <b>filterEditor and advanced criteria</b>: If a developer calls <code>filterData()</code>
    // on a ListGrid and passes in +link{AdvancedCriteria}, expected behavior of the filter 
    // editor becomes ambiguous, as  AdvancedCriteria supports far more complex filter 
    // expressions than the ordinary filterEditor is capable of expressing.
    // <br>
    // The above example code assumes simple criteria, but if we wanted to apply advanced 
    // criteria to the grid we could call +link{dataSource.combineCriteria()} rather than doing
    // a simple addProperties() on the criteria object.
    // <P>
    // Default behavior for AdvancedCriteria will combine the AdvancedCriteria with the values 
    // in the filter editor as follows:
    // <ul>
    // <li>If the top level criteria has operator of type "and":<br>
    //  Each field in the top level
    //  criteria array for which a 'canFilter' true field is shown in the listGrid will show up
    //  if the specified operator matches the default filter behavior 
    //  (based on the +link{listGrid.autoFetchTextMatchStyle}).<br>
    //  If the user enters values in the filter editor, these will be combined with the
    //  existing AdvancedCriteria by either replacing or adding field level criteria at the top 
    //  level.</li>
    // <li>If the top level criteria is a single field-criteria:<br>
    //  If the field shows up in the listGrid and is canFilter:true, it will be displayed to
    //  the user (if the operator matches the default filter behavior for the field).<br>
    //  If the user enters new filter criteria in the filterEditor, they will be combined with
    //  this existing criterion via a top level "and" operator, or if the user modifies the
    //  field for which the criterion already existed, it will be replaced.</li>
    // <li>Otherwise, if there are multiple top level criteria combined with an "or" operator,
    //  these will not be shown in the filter editor. Any filter parameters the user enters will
    //  be added to the existing criteria via an additional top level "and" operator, meaning
    //  the user will essentially filter a subset of the existing criteria</li>
    // </ul>
    //  @group filterEditor
	//  @visibility external
    //  @example filter
	//<
	
	//showFilterEditor:null
    
    
    filterEditorDefaults : { shouldPrint:false }, 

    
    //> @attr listGrid.filterButtonProperties (Button properties : null : IR)
    // If +link{listGrid.showFilterEditor} is true, this attribute may be used to customize the
    // filter button shown to the right of the filterEditor row.
    // @visibility external
    //<

	//> @attr listGrid.filterEditorHeight (number : 22 : IRW)
    // Height for the filterEditor, if shown.
	//    
	//  @group filterEditor
	//  @visibility external
	//<
    filterEditorHeight:22,
    
    //> @attr listGrid.autoFetchAsFilter (boolean : null : IR)
    // Determines whether +link{filterData()} or +link{fetchData()} should be called when this 
    // grid is initially filtered via +link{autoFetchData}, or filtered by the user via the
    // +link{showFilterEditor,filterEditor}. 
    // @group dataBinding
    // @deprecated in favor of listGrid.autoFetchTextMatchStyle
    //<
    // Note: This was exposed in the 7.0 release candidate builds only.
    // Leaving supported (deprecated) but not publically documented
    // If set, at init time, it will set textMatchStyle to exact or substring (handled in Canvas.js)
    
    //> @attr listGrid.autoFetchTextMatchStyle (TextMatchStyle : "substring" : IR)
    // When this grid is initially filtered via +link{autoFetchData}, or filtered by the user 
    // via the +link{showFilterEditor,filterEditor}, this attribute can be used to set the
    // <code>textMatchStyle</code> on the dsRequest passed to <code>fetchData()</code>.
    // @group dataBinding
    // @visibility external
    //<
    // implemented in canvas.getInitialFetchContext() and recordEditor.performAction()
    autoFetchTextMatchStyle:"substring",
    
	// Editing
	// --------------------------------------------------------------------------------------------
	//> @attr listGrid.canEdit (boolean : null : [IRW])
	//      Can the user edit cells in this listGrid? Can be set for the listGrid, and overridden for 
	//      individual fields.<br>
	//      If 'canEdit' is false at the listGrid level, fields can never be edited - in this case
	//      the canEdit property on individual fields will be ignored.<br>
	//      If 'canEdit' is set to true at the listGrid level, setting the 'canEdit' property to
	//      false at the field level will prevent the field from being edited inline.<br>
	//      If 'canEdit' is not set at the listGrid level, setting 'canEdit' to true at the field 
	//      level enables the field to be edited inline.
	//      @visibility external
	//      @group  editing
	//      @see    startEditing()
    //      @see listGridField.canEdit
    //      @see listGrid.recordEditProperty
    //      @see listGrid.canEditCell()
	//      @see    fields
    //      @example editByRow
	//<
	//canEdit:null,

    //> @attr listGrid.canEditNew (boolean : null : [IRW])
	//  Can the user add new rows?
	//  @group  editing
	//  @see attr:listGrid.canEdit
    //  @see attr:listGrid.recordEditProperty
	//  @visibility internal
	//<
    canEditNew: false,

	//> @attr listGrid.recordEditProperty (string : "_canEdit" : [IRWA])
	// Property name on a record that should be checked to determine whether the record may be
	// edited.
	// <br>
	// This property is configurable to avoid possible collision with data values in record.
	// With the default setting of "_canEdit", a record can be set non-editable by ensuring
	// record._canEdit == false.
	// <br>
	// For controlling editability for the entire grid or for a field, set grid.canEdit or
	// field.canEdit.
	//
	//  @group  editing
	//  @see attr:listGrid.canEdit
	//  @see attr:listGridField.canEdit
    //  @see method:listGrid.canEditCell
	//  @visibility external
	//<
    recordEditProperty:"_canEdit",
    
	// Name for property used by internal '_testRowEditData' method to track whether
	// records have been compared to edit-data in order to map rowNums to edit values.
	// Customizable, in case of collision with record data - but unlikely to be overridden.
    editValuesTestedProperty:"_editValuesTested",

    //> @attr listGrid.alwaysShowEditors (boolean : null : [IR])
    // When this attribute is set, editors will be rendered into every row of the grid rather than
    // showing up in a single record at a time.
    // This attribute is only valid when +link{listGrid.editByCell} is false
    // @group editing
    // @visibility external
    //<

	//>	@attr listGrid.editByCell (boolean : null : [IRW])
	// Determines whether when the user edits a cell in this listGrid the entire row becomes
	// editable, or just the cell that received the edit event.
    // <P>
	// No effect if this.canEdit is false or null.
	//  @group  editing
	//  @visibility external
	//  @see listGrid.canEdit
    //  @example editByCell
	//<

	//>	@attr listGrid.saveByCell (boolean : null : [IRW])
	// Whether edits should be saved whenever the user moves between cells in the current edit
	// row.
	// <P>
	// If unset, defaults to +link{editByCell,this.editByCell}.
	// <P>
	// To avoid automatic saving entirely, set +link{autoSaveEdits}:false.
	//
	//  @group  editing
	//  @visibility external
	//  @see listGrid.editByCell
	//<
    
	//>	@attr listGrid.validateByCell (boolean : null : [IRW])
	// Whether client-side validation checks should be performed when the user moves between
	// cells in the current edit row.  If unset, defaults to +link{listGrid.editByCell}.
    // <P>
    // Note that validation always occurs when a row is to be saved, so setting
    // +link{saveByCell}:true forces validation on cell transitions.  To completely disable
    // automatic validation, set +link{neverValidate}:true.
    //
    // @group gridValidation
    // @visibility external
    // @see group:editing
    //<
    

    // autoValidate will disable validation on row-transitions, so validation will only
    // occur on save attempts.  Not currently exposed.
    autoValidate:true,
    
    //> @attr listGrid.validateOnChange (boolean : null : [IRW])
    // If true, validation will be performed on each edited cell when each editor's 
    // "change" handler is fired.
    //
    // @see ListGridField.validateOnChange
    // @group gridValidation
    // @visibility external
    //<
    //validateOnChange:null
    
    //> @attr listGrid.neverValidate (boolean : null : [IRWA])
	// If true, validation will not occur as a result of cell editing for this grid.
	//	@group	gridValidation
    // @visibility external
	//<
    //neverValidate:null,
    
    
    //> @attr listGrid.canRemoveRecords (boolean : false : IR)
    // If set, provide UI for the user to remove records from the grid. This is achieved by
    // rendering an additional field in the listGrid which, when clicked, will remove the
    // record associated with the clicked row via a call to +link{listGrid.removeData()}.
    // <P>
    // If +link{listGrid.animateRemoveRecord} is true, the removed record will appear to shrink
    // out of view when it is removed.
    // <P>
    // By default the field will display the +link{listGrid.removeIcon} next to each record, and
    // will be rendered as the leftmost column. Two mechanisms exist to further modify this field:
    // <ul>
    // <li>To change the position of the remove-field, include an explicitly specified field with
    //     the attribute +link{listGridField.isRemoveField,isRemoveField:true} set. This will then
    //     be used as the remove field instead of adding a field to the beginning of the set of
    //     columns.</li>
    // <li>Additional direct configuration of the remove field may be achieved by modifying
    //     +link{listGrid.removeFieldProperties}.</li>
    // </ul>
    // @group databinding
    // @visibility external
    //<
    
    shouldShowRemoveField : function () {
        if (this.fieldSourceGrid != null) return this.fieldSourceGrid.shouldShowRemoveField();
        return this.canRemoveRecords;
    },

    //> @attr listGrid.removeIcon (SCImgURL : "[SKIN]/actions/remove.png" : IR)
    // When +link{ListGrid.canRemoveRecords} is enabled, default icon to show in
    // the auto-generated field that allows removing records.
    // @visibility external
    //<
    removeIcon:"[SKIN]/actions/remove.png",
    
    //> @attr listGrid.removeIconSize (Number : 16 : IRW)
    // Default width and height of +link{removeIcon,remove icons} for this ListGrid.
    //
    // @visibility external
    //<
    removeIconSize: 16,
    
    //> @attr listGrid.animateRemoveRecord (boolean : true : IRW)
    // When +link{ListGrid.canRemoveRecords} is enabled, should records be animated out of view
    // when they are removed by the user? 
    // @visibility external
    //<
    // When showing alternate records styles, the styles will essentially be reassigned after the
    // animation completes which means we finish our smooth animation with what looks a little like
    // a jump - not clear how to avoid this, but we could warn about this in the attribute
    // description.
    animateRemoveRecord:true,
    
    //> @attr listGrid.animateRemoveTime (number : 100 : IRW)
    // When animating record removal 
    // +link{listGrid.animateRemoveRecord,(see animateRemoveRecord)}, if 
    // +link{listGrid.animateRemoveSpeed} is not
    // set, this property designates the duration of the animation in ms.
    // @group animation
    // @visibility animation
    // @see listGrid.animateRemoveRecord
    //<
    animateRemoveTime:100,

    //> @attr listGrid.animateRemoveSpeed (number : 200 : IRW)
    // When +link{listGrid.animateRemoveRecord, animating record removal}, this property 
    // designates the speed of the animation in pixels per second. Takes precedence over the
    // +link{listGrid.animateRemoveTime} property, which allows the developer to specify a
    // duration for the animation rather than a speed.
    // @group animation
    // @visibility animation
    // @see listGrid.animateRemoveRecord
    //<    
    animateRemoveSpeed:200,

    //> @attr listGrid.removeFieldTitle (String : "[Remove record]" : IRW)
    // The title to use for the +link{listGrid.removeFieldDefaults, remove field}.  Note that this
    // text will appear in the column-picker but the not in the field-header because the removeField
    // has +link{listGridField.showTitle, showTitle} set to false by default.  This can be
    // changed via +link{listGrid.removeFieldProperties}.
    // @group i18nMessages
    // @visibility external
    //<
    removeFieldTitle: "[Remove record]",

    //> @attr listGrid.removeFieldDefaults (ListGridField properties : {...} : IR)
    // Default configuration properties for the "remove field" 
    // displayed when +link{ListGrid.canRemoveRecords} is enabled.
    // +link{classMethod:class.changeDefaults()} should be used when modifying this object. 
    // <P>
    // The default configuration includes a +link{listGridField.recordClick()} handler which
    // calls +link{listGrid.removeData()} to actually perform the data removal.
    // @visibility external
    //<
    removeFieldDefaults:{
        type:"icon",
        width:21,
        showHeaderContextMenuButton:false,
        showDefaultContextMenu:false,
        showTitle: false,
        canEdit:false,
        canSort:false,
        canGroupBy:false,
        canFilter:false,
        // cellIcon will be set up on the fly to match listGrid.removeIcon
        recordClick : function (viewer, record, rowNum, field, colNum) {
            var editRows = viewer.getAllEditRows(),
                isEditing = editRows.contains(rowNum);

            if (isEditing) viewer.discardEdits(rowNum, colNum);
            viewer._removeRecordClick(rowNum, record);
            return false;
        }
    },
    
    //> @attr listGrid.removeFieldProperties (ListGridField properties : null : IR)
    // Configuration properties for the "remove field" displayed when
    // +link{ListGrid.canRemoveRecords} is enabled. These configuration settings will be overlaid
    // on top of the +link{listGrid.removeFieldDefaults}.
    // @visibility external
    //<

	//>	@attr listGrid.filterByCell (boolean : null : [IRWA])
	// If we're showing the filterEditor (this.showFilterEditor is true), this property 
	// determines whether this list should be filtered every time the user puts focus in
	// a different field in the filter editor.
	// @group filterEditor
	// @visibility external
	//<
	
	//>	@attr listGrid.filterOnKeypress (boolean : null : [IRWA])
	// If we're showing the filterEditor (this.showFilterEditor is true), this property 
	// determines whether this list should be filtered every time the user modifies the value
    // in a field of the filter-editor. Can also be set at the field level.
	// @group filterEditor
	// @visibility external
	//<
    
	//> @attr listGrid.waitForSave (boolean : false : [IRWA])
	// If this is an editable listGrid, this property determines whether the user will be
	// able to dismiss the edit form, or navigate to another cell while the save is in 
	// process (before the asynchronous server response returns).
	//	@group	editing
	// @visibility external
	//<
	//waitForSave:false,

	//> @attr listGrid.stopOnErrors (boolean : false : [IRWA])
	// If this is an editable listGrid, this property determines how failure to save due to 
	// validation errors should be displayed to the user.
	// <P>
	// If this property is true, when validation errors occur the errors will be displayed
	// to the user in an alert, and focus will be returned to the first cell to fail validation.
	// <P>
	// If false, this the cells that failed validation will be silently styled with the 
	// editFailedBaseStyle.<br>
	// <b>Note:</b> stopOnErrors being set to true implies that 'waitForSave' is also true.
	// We will not dismiss the editor until save has completed if stopOnErrors is true.
	//
	//  @visibility external
	//	@group	editing
	//  @see    waitForSave
	//<
	//stopOnErrors:false,

    //> @attr listGrid.autoSaveEdits (boolean : true : [IRWA])
    // If this ListGrid is editable, should edits be saved out when the user finishes editing
    // a row (or a cell if +link{ListGrid.saveByCell} is true).
    // <P>
    // The default of <code>true</code> indicates that edits will be
    // +link{saveByCell,automatically saved} as the
    // user navigates through the grid and/or +link{enterKeyEditAction,hits 'Enter'} to end
    // editing.  See the +link{group:editing,Grid Editing} overview for details. 
    // <P>
    // Setting <code>autoSaveEdits</code> false creates a "mass update" / "mass delete"
    // interaction where edits will be retained for all edited cells (across rows if
    // appropriate) until +link{saveEdits()} is called to save a particular row, or
    // +link{saveAllEdits()} is called to save all changes in a batch.
    //
    // @group editing
    // @visibility external
    //<
    autoSaveEdits:true,
    
    // ListGrid validation error icon. Very similar API to the FormItem class validation error 
    // icon.
    
    //> @attr ListGrid.showErrorIcons (boolean : true : IRW)
    //  If this grid is editable, and an edit has caused validation failure for some cell,
    //  should we show an icon to indicate validation failure?
    //  @group  errorIcon
    //  @visibility internal
    //<
    showErrorIcons : true,
    
    //> @attr ListGrid.errorIconHeight (number : 16 : IRW)
    //      Height of the error icon, if we're showing icons when validation errors occur.
    //  @group  errorIcon
    //  @visibility internal
    //<
    errorIconHeight : 16,
    
    //> @attr ListGrid.errorIconWidth (number : 16 : IRW)
    //      Height of the error icon, if we're showing icons when validation errors occur.
    //  @group  errorIcon
    //  @visibility internal
    //<    
    errorIconWidth : 16,
    
    //> @attr ListGrid.errorIconSrc (SCImgURL : "[SKIN]/ListGrid/validation_error_icon.png" : IRW)
    //      Src of the image to show as an error icon, if we're showing icons when validation
    //      errors occur.
    //  @group  errorIcon
    //  @visibility internal
    //<    
    errorIconSrc : "[SKIN]/validation_error_icon.png",    
        
	//> @attr listGrid.confirmCancelEditing (boolean : false : [IRW])
	// If this is an editable listGrid, when the user attempts to cancel an edit, should we
    // display a confirmation prompt before discarding the edited values for the record?
	//
	//  @visibility external
	//	@group	editing
	//<
	//confirmCancelEditing:false,

	//> @attr listGrid.cancelEditingConfirmationMessage (string : Cancelling this edit will discard unsaved changes for this record. Continue? : [IRW])
	// If this is an editable listGrid, and <code>this.confirmCancelEditing</code> is true
    // this property is used as the message to display in the confirmation dismissal prompt.
    //
	//  @visibility external
	//	@group	editing, i18nMessages
	//<
	cancelEditingConfirmationMessage:"Cancelling this edit will discard unsaved changes for this record. Continue?",

    //> @attr listGrid.confirmDiscardEdits (boolean : true : [IRW])
	// For editable listGrids, outstanding unsaved edits when the user performs a new filter
    // or sort will be discarded. This flag determines whether we should display a confirmation
    // dialog with options to save or discard the edits, or cancel the action in this case.
	//
	//  @visibility external
	//	@group	editing
	//<
	confirmDiscardEdits:true,
    
	//> @attr listGrid.confirmDiscardEditsMessage (string : "This action will discard unsaved changes for this list.": [IRW])
	// If <code>this.confirmDiscardEdits</code> is true, this property can be used to customize the
    // error message string displayed to the user in a dialog with options to 
    // cancel the action, or save or discard pending edits in response to sort/filter actions
    // that would otherwise drop unsaved edit values.
	// @visibility external
	// @group editing, i18nMessages
	//<
	confirmDiscardEditsMessage:"This action will discard all unsaved changes for this list.",

    //> @attr listGrid.discardEditsSaveButtonTitle (string :"Save" : IRW)
    // If +link{listGrid.confirmDiscardEdits} is true this is the title for the save button
    // appearing in the lost edits confirmation dialog. Override this for localization if necessary.
    // @visibility external
    // @group editing, i18nMessages
    //<
    discardEditsSaveButtonTitle:"Save",
    
	//> @attr listGrid.addNewBeforeEditing (boolean : false : IRWA)
	// When creating a new edit record via 'startEditingNew()' [or tabbing beyond the end
	// of the last editable field in a list], should we contact the server to create a
	// server-side record before editing begins?
	// @group	editing
	// @visibility advancedInlineEdit
	//<
	//addNewBeforeEditing:false,

	//> @attr listGrid.rowEndEditAction (RowEndEditAction : null : IRW)
	// If the user is editing a record in this listGrid, and attempts to navigate to a field
	// beyond the end of the row, via tab (or shift-tab off the first editable field), this 
	// property determines what action to take:<ul>
	// <li>"next": start editing the next (or previous) record in the list
	// <li>"same": put focus back into the first editable field of the same record.
	// <li>"done": hide the editor
	// <li>"stop": leave focus in the cell being edited
	// <li>"none": no action
	// </ul>
	//	@group	editing
	// @visibility external
	//<                                            
    //rowEndEditAction:"next",

	//> @attr listGrid.listEndEditAction (RowEndEditAction : null : IRW)
	// If the user is editing the last record in this listGrid, and attempts to navigate 
	// beyond the last row either by tabbing off the last editable field, or using the down
	// arrow key, this property determines what action to take:<ul>
	// <li>"next": start editing a new record at the end of the list.
	// <li>"done": hide the editor
	// <li>"stop": leave focus in the cell being edited
	// <li>"none": no action
	// </ul>
	//	@group	editing
	// @visibility external
    // @example enterNewRows
	//<     
	
    //listEndEditAction:"stop",
    
    //> @attr listGrid.showNewRecordRow (boolean    : null  : [IRW])
    // If this is an editable ListGrid, this property determines whether an extra row should
    // be displayed below the last record, which can be used to add a new record to the dataset
    //<
    
	//>	@attr listGrid.newRecordRowMessage (string : "-- Add New Row --" : IR)
    // If this listGrid is showing the 'newRecordRow' (used for adding new rows to the end
    // of the data), this property determines what message should be displayed in this row.
    // @group editing, i18nMessages
    //<
    newRecordRowMessage:"-- Add New Row --",
    
	//> @attr listGrid.enterKeyEditAction (EnterKeyEditAction : "done" : IRW)
	// What to do when a user hits enter while editing a cell:
    // <ul>
	// <li>"nextCell": start editing the next editable cell in this record (or the first
	//     editable cell in the next record if focus is in the last editable cell in the row)
	// <li>"nextRow": start editing the same field in the next row (skipping any rows where
	//      that would be a non-editable cell.
	// <li>"nextRowStart": start editing the first editable cell in the next row.
	// <li>"done": hide the editor (editing is complete)
    // </ul>
    // Note that if this.autoSaveEdits is true, this may cause a save of the current edit values
	// @group	editing
	// @visibility external
	//<
    enterKeyEditAction:"done",
    
    //> @attr listGrid.escapeKeyEditAction (EscapeKeyEditAction : "cancel" : IRW)
    // What to do when a user hits escape while editing a cell:<ul>
    // <li>"cancel": close the editor and discard the current set of edit values
    // <li>"done": just close the editor (the edit is complete, but the edited values are retained).
    // </ul>
    // Note that if +link{autoSaveEdits} is true, this may cause a save of the current edit values
    // @group editing
    // @visibility external
    //<
    escapeKeyEditAction:"cancel",
    
	//> @type ListGridEditEvent	
    // Event that will trigger inline editing.
    //
    // @value "click"       A single mouse click triggers inline editing
    // @value "doubleClick" A double click triggers inline editing
    // @value "none"        No mouse event will trigger editing.  Editing must be
    //                      programmatically started via +link{listGrid.startEditing()}
    //                      (perhaps from an external button) or may be triggered by 
    //                      keyboard navigation if +link{listGrid.editOnFocus} is set.
    //
	// @group editing
	// @visibility external
    //<

	//> @attr listGrid.editEvent (ListGridEditEvent : "doubleClick" : IRW)
    // Event that will trigger inline editing, see +link{type:ListGridEditEvent} for options.
    // <P>
    // Note this setting has no effect unless +link{canEdit} has been set to enable editing.
    // <P>
    // See also +link{editOnFocus} and +link{startEditing}.
    //
	// @group editing
	// @visibility external
    // @example editByRow
	//<
	editEvent:isc.EH.DOUBLE_CLICK,

	//> @attr listGrid.editOnFocus (boolean : null : [IRWA])
	// Should we start editing when this widget receives focus (if this ListGrid supports
	// editing)?
    // <P>
	// Note that this property being set to true will cause editing to occur on a single
	// click, even if +link{editEvent} is <code>"doubleClick"</code>, because single clicking
    // the grid will place keyboard focus there automatically.
    //
	// @group editing
	// @visibility external
	//<
	// Note - editOnFocus behavior is slightly more complicated than might be imagined. Actual
	// behavior:
	// - focus must go to the body (not the header) to start editing.
	// - if we are currently editing, getting focus will not trigger a new edit.
	// - if the focus is a result of clicking on the listGrid body, we will only start editing
	//   if the user clicked on an editable cell -- this is the same behavior as with 
	//   editEvent:"click"
	// - otherwise when this widget receives focus, the first editable row will become editable.
    
    //> @attr listGrid.editOnF2Keypress (boolean : true : [IRWA])
    // Should we start editing when the widget has focus and the user presses the "f2" key
    // (if this ListGrid supports editing)?
    // <P>
    // Note that if +link{listGrid.editEvent} is set to <code>"click"</code> or
    // <code>"doubleClick"</code>, the <code>Space</code> or <code>Enter</code> key may
    // also be used to start editing, depending on the value for +link{generateClickOnSpace},
    // +link{generateDoubleClickOnSpace}, +link{generateClickOnEnter} and 
    // +link{generateDoubleClickOnEnter}.
    // <P>
    // If +link{listGrid.canEdit} is false, or +link{listGrid.editEvent} is set to "none" this
    // property has no effect.
    //
    // @group editing
    // @visibility external
    //<
    // Tested on:
    // FF 4.0.1 (Mac)
    // Chrome 11.0.696.77 (Mac)
    // Safari 5.0.5 (Mac)
    // IE 8 (XP)
    editOnF2Keypress:true,
    
    
	//> @attr listGrid.editOnKeyPress (boolean : null : [IRWA])
	// If set to true, when this grid has focus, if the user starts typing character keys
    // we'll start editing the focused cell.
    // @group editing
	// @visibility internal
	//<
    
    //> @attr listGrid.moveEditorOnArrow (boolean : null : [IRWA])
	// If +link{listGrid.editOnKeyPress, editOnKeyPress} is true, once the user starts editing
    // a cell by typing while focused in it, should left / right arrow keys cause the 
    // edit cell to shift horizontally?
    // @group editing
	// @visibility internal
	//< 
    
    	        
    //> @attr listGrid.selectOnEdit (boolean : true : [IRWA])
    //  When the user starts editing a row, should the row also be selected?
    //  <P>
    //  Note that  when this attribute is set to <code>true</code>, other all other rows in the grid
    //  will be deselected when a record is selected due to editing. 
    // @group editing
    // @visibility external
    //<
    selectOnEdit : true,
    
    //> @attr listGridField.canToggle (boolean : varies : IRWA)
    // Allows a boolean or +link{valueMap,valueMapped} field to be edited without going into
    // edit mode. When this attribute is set, clicking on the field will change the value - for
    // boolean fields toggling between <code>true</code> and <code>false</code>, and for valueMapped
    // fields, advancing the value to the next option in the valueMap.
    // <P>
    // To enable this feature, +link{listGrid.canEdit} must be set to true.
    // For boolean type fields <code>canToggle</code> is true by default, meaning setting
    // <code>canEdit</code> to true implies the user can toggle the value via a single click
    // without going into edit mode. You can disable this by explicitly setting
    // <code>canToggle</code> to false for a boolean field.<br>
    // Note that you can enable toggling only (without allowing the user to edit other fields)
    // by just setting +link{listGrid.editEvent,grid.editEvent:"none"}.
    // <P>
    // If +link{listGrid.editEvent} is set to "click", when the user clicks on the field, 
    // the value will be toggled, and inline editing will be triggered as usual.
    // Otherwise the toggled value will be saved immediately to the server, or if 
    // +link{listGrid.autoSaveEdits} has been set to false, will be stored as an edit value
    // for the record.
    //
    // @group editing
    // @visibility external
    //<
    
    //> @attr listGrid.enumCriteriaAsInitialValues (boolean : true : IR)
    // In a ListGrid that has a DataSource and has filter criteria that include values for
    // fields declared as +link{DSFieldType,type "enum"} in the DataSource, by default a newly
    // edited row will use those filter criteria as initial values.
    // <P>
    // For example, if a ListGrid is showing all Accounts that have status:"Active" and a new row
    // is created, the new row will default to status:"Active" unless this flag is set to false.
    //
    // @group editing
    // @visibility external
    //<
    enumCriteriaAsInitialValues:true,
	
	//> @attr listGrid.application (application : null : IRW)
	//		Application to use when saving edited values in a databound ListGrid
	//	@group	editing
	//<

	//> @type AutoComplete
	// Possible autoComplete modes.
	//
	// @value   "smart"    Enable SmartClient autoComplete feature.  Suppresses browser's
	//                     built-in autoComplete feature where applicable.
	// @value   "native"   Allow browser's built-in autoComplete feature to function
	// @value   "none"     Suppress brower's built-in autoComplete feature
	// @group autoComplete
	// @visibility autoComplete
	//<

	//>	@attr listGrid.autoComplete (AutoComplete : null : IRW)
	// Whether to do inline autoComplete in text fields during inline editing<br>
    // Overridden by +link{ListGridField.autoComplete} if specified.
    // If unset picks up the default from the appropriate editor class (subclass of FormItem).
    // 
	// @see listGridField.autoComplete
	// @group autoComplete
	// @visibility autoComplete
	//<
    //autoComplete:null,

	//>	@attr listGrid.uniqueMatch (boolean : true : IRW)
	// When SmartClient autoComplete is enabled, whether to offer only unique matches to the
	// user.
	// <p>
	// Can be individually enabled per TextItem, or if set for the grid as a whole, can
	// be set differently for individual fields.
	//
	// @see listGridField.uniqueMatch
	// @group autoComplete
	// @visibility autoComplete
	//<

    // autoSelectEditors - if true when the user starts editing a cell always select the content
    // of the cell
    autoSelectEditors:true,
	// defaults for the form used for inline editing
    editFormDefaults: {
    	
        canSelectText:true,
        autoDraw:false,
        
        // show error icons on the left by default
        errorOrientation:"left",
        showErrorText:false,
        showErrorStyle:false,
        
        itemKeyPress:function (item, keyName, characterValue) {
            return this.grid.editorKeyPress(item, keyName, characterValue);
        }
    }, 
    
    //> @attr listGrid.longTextEditorThreshold (integer : 255 : IRW) 
    // When the length of the field specified by +link{attr:dataSourceField.length} exceeds this
    // value, the ListGrid shows an edit field of type +link{attr:listGrid.longTextEditorType}
    // rather than the standard text field when the field enters inline edit mode.
    //
    // @group editing
    // @visibility external
    //<
    longTextEditorThreshold : 255,
    
    //> @attr listGrid.longTextEditorType (string : "PopUpTextAreaItem" : IRW) 
    // When the length of the field specified by +link{attr:dataSourceField.length} exceeds 
    // <code>this.longTextEditorThreshold</code> show an edit field of this type
    // rather than the standard text field when the field enters inline edit mode.
    //
    // @group editing
    // @visibility external
    //<
    longTextEditorType : "PopUpTextAreaItem",
    
	// functions installed into FormItems used for inline editing.  
	// Note - these will be executed in the scope of the form item (not in the scope of the
	// ListGrid).
	//
	// Set up keyboard handling on form items to handle user navigation via "Enter", "Escape",
	// "Tab" keypresses, etc.
    //
	// NOTE: in Moz, if the user is holding down tab and we are cancelling the field change
	// *while logging to an open Log window*, we'll be stuck in the field indefinitely.  This
	// is probably a non-bug but can look like a freeze.
	//
	// row editing: Cancel tab to prevent focus cycling through the visible form items, since
	// for the first/last editable or visible item, we want to place focus in newly drawn
	// editors (on next/previous row, or just in newly drawn area that we scrolled into)
    editorKeyDown : function (item, keyName) {
        
        if (isc.Browser.isMoz && item && item.multiple && 
            isc.isA.NativeSelectItem(item) && keyName == "Enter") 
        {
            item._selectedOnEnterKeydown = item.getValue();
        }
        // In Safari 3.1 on Windows and Mac (both 525.13), to cancel navigation on tab you have to
        // return false from keyDown not keyPress.
        // This does not prevent keyPress from firing.
        // Therefore in this browser always return false from keyDown if it's a tab keypress
        if (isc.Browser.isSafari && isc.Browser.safariVersion >= 525.13 && keyName == "Tab") {
            return false;
        }
    },

    editorKeyPress : function (item, keyName, characterValue) {

    	// We will return false to cancel native behavior on any key event for the keys
    	// used for navigating around the edit cells (arrow keys, tab, etc.)
        var EH = isc.EH,
            returnValue,
            editEvent;        
        
        var isMultiLineEditor =  //>PopUpTextAreaItem
                    isc.isA.PopUpTextAreaItem(item) ||     //<PopUpTextAreaItem
                    (isc.RichTextItem && isc.isA.RichTextItem(item)) ||
                    isc.isA.TextAreaItem(item);

        if (keyName == "Tab") {
        	// Always cancel the native event that would take focus from this item.
        	// This is appropriate as we will always focus in the next item programmatically, 
        	// even if we are currently showing the entire edit row.
        	
            var shift = EH.shiftKeyDown();
                
            // If this is a container item, with sub items, or has a number of focusable
            // icons, we may be moving focus within the item, rather than going to another
            // edit cell.  This is handled by _moveFocusWithinItem().
            if (this.ns.isA.ContainerItem(item) || 
                (item.icons != null && item.icons.length > 0 
                    // If the event occurred on a PopUpTextAreaItem, native focus is either 
                    // on the icon, or within the TextArea in another form.
                    // In this case we always navigate to the next cell
                    && !this.ns.isA.PopUpTextAreaItem(item))
               ) 
            {
                if (!this._moveFocusWithinItem(item, shift)) {
                    return false;
                }
            }
            
            editEvent = shift ? isc.ListGrid.SHIFT_TAB_KEYPRESS 
                                            : isc.ListGrid.TAB_KEYPRESS;
            returnValue = false;
                    
        } else if (keyName == "Enter") {
        	// If the event occurred over an icon, we don't want to interfere with it, as
        	// enter will activate the link (for accessibility)
            if (item.getFocusIconIndex() != null) return;
            
            // allow enter to work normally for text areas.  Alt + Enter overrides.
            if (isMultiLineEditor && isc.EH.altKeyDown() == false) {
                return returnValue;
            }
            if (item._selectedOnEnterKeydown != null) {
                var oldVal = item._selectedOnEnterKeydown;
                delete item._selectedOnEnterKeydown;
                item.setValue(oldVal);
            }
            editEvent = isc.ListGrid.ENTER_KEYPRESS;
            returnValue = false;

        } else if (keyName == "Escape") {
            editEvent = isc.ListGrid.ESCAPE_KEYPRESS;
            returnValue = false;

        // By default move to a new row in response to
        //  - arrow keypress (up / down)
        //  - ctrl + arrow keypress (up / down), but not if the shift key is also down
        // Exceptions:
        //  - Text area use alt+arrowKeypress only
        //  - explicitly avoid row change on alt+arrow key in SelectItem, where this is used
        //    to show / hide the pickList
        //
        
        } else if (keyName == "Arrow_Up") {
            var textArea = isMultiLineEditor;
            if (textArea && !isc.EH.altKeyDown()) return returnValue;
            
            if (isc.isA.SelectItem(item) && !isc.EH.ctrlKeyDown()) return returnValue;
            
            if (isc.EH.ctrlKeyDown() && isc.EH.shiftKeyDown()) return returnValue;
            editEvent = isc.ListGrid.UP_ARROW_KEYPRESS;
            returnValue = false;

        } else if (keyName == "Arrow_Down") {

            var textArea =  isMultiLineEditor;
            if (textArea && !isc.EH.altKeyDown()) return returnValue;
            
            if (isc.isA.SelectItem(item) && !isc.EH.ctrlKeyDown()) return returnValue;
            
            if (isc.EH.ctrlKeyDown && isc.EH.shiftKeyDown()) return returnValue;
            
            editEvent = isc.ListGrid.DOWN_ARROW_KEYPRESS;
            returnValue = false;
    
        // if the user started editing via editOnKeyPress and the 'moveEditorOnArrow' flag is true
        // we shift cells on left/right arrow
        } else if (this.moveEditorOnArrow && this._editSessionFromKeyPress) {
            if (keyName == "Arrow_Left") {
                editEvent = isc.ListGrid.LEFT_ARROW_KEYPRESS;
                returnValue = false;
            } else if (keyName == "Arrow_Right") {
                editEvent = isc.ListGrid.RIGHT_ARROW_KEYPRESS;
                returnValue = false;            
            }
        }
        
        if (editEvent != null) {
            
            if (isc.EH.clickMaskUp()) {
                isc.EH.setMaskedFocusCanvas(null, isc.EH.clickMaskRegistry.last());
            }
    
            // Fire cellEditEnd to handle saving out the value / moving to the next cell as 
            // appropriate
            this.cellEditEnd(editEvent);
        }
        
        return returnValue;            
    },

    // _moveFocusWithinItem() Helper method fired when the user hits tab / shift+tab while
    // focused in some edit item. This will move focus to the appropriate icon or sub item
    // if necessary.  A return value of false indicates focus was moved within the item, so
    // should not move to another edit cell.
    _moveFocusWithinItem : function (item, shift) {
        
        if (!item) return true;
        return (!item._moveFocusWithinItem(!shift));
    },

	// Override elementFocus on the form items:
	// If we're editing the whole row, and the user clicks in a new field to focus in it,
	// call 'editCellEnd' method to perform validation / saving on the previous
	// edit field (if required).
	
    _editFormItem_elementFocus : function (suppressHandlers) {
    	

        var form = this.form,
            lg = form.grid;
    	    
        
        var rowNum, colNum, fieldName, fieldChanged;
        if (lg._editorShowing) {
            rowNum = lg._editRowNum;
            if (!lg.editByCell) {
                rowNum = lg._editRowNum;
                fieldName = this.getFieldName();
                colNum = lg.fields.findIndex(lg.fieldIdProperty, fieldName);
                
                if (!suppressHandlers) {
                    fieldChanged = (lg._editColNum != colNum);
                    // If the user has clicked in another field in the edit form,
                    // fire editField on the appropriate field
                    if (fieldChanged) {
                	// store the new edit cell
                        lg.setNewEditCell(rowNum, colNum);
                	// fire 'cellEditEnd' to save / validate before moving to the new cell
                        lg.cellEditEnd(isc.ListGrid.EDIT_FIELD_CHANGE);
                    
                	// Note - if we could cancel the focus here, it might make sense, as the 
                	// cellEditEnd callback method will re-focus in the new focus cell, but we can't
                	// cancel the focus by simply returning false from this method.
                	// Therefore allow the focus to proceed, and fall through to the super 
                	// implementation of this method which will fire focus handlers, show any
                	// 'showOnFocus' icons, etc.
                    }
                }
            } else {
                colNum = lg._editColNum;         
            }
        }
        this.Super("elementFocus", arguments);
        
        if (lg._editorShowing) {
        	// If this is the current edit field, and hasn't yet fired its 'editorEnter' handlers
        	// fire them now.
            var rowEnter = this._rowEnterOnFocus,
                cellEnter = this._cellEnterOnFocus;
            // Note: we're clearing out the flags before we fire the handlers. If the
            // handler trips a change of edit row, etc., we want editorExit to fire.
            delete this._rowEnterOnFocus;
            delete this._cellEnterOnFocus;
            
            
            var editVals = isc.addProperties(
                {}, 
                lg.getCellRecord(rowNum,colNum),
                lg._getEditValues(rowNum,colNum)
            );
            
            if (cellEnter) {
                var fieldName = lg.getFieldName(colNum);
                lg._handleEditorEnter(this, rowNum, colNum, editVals[fieldName]);
            }
            if (rowEnter) lg._handleRowEditorEnter(this, rowNum, editVals);

        } else {
            lg.logWarn("suppressing editorEnter handlers on focus as listGrid._editorShowing is null");
        }
    },
    

	// Header
	// ----------------------------------------------------------------------------------------

    //> @groupDef gridHeader
    // Properties and methods related to the ListGrid header. ListGrid headers are implemented
    // as a +link{class:Toolbar} of buttons shown at the top of the ListGrid 
    // (one button per column).<br>
    // The toolbar header provides UI for interacting with the ListGrid fields directly (sorting,
    // resizing, reordering columns, etc).
    // @visibility external
    //<
        
    //> @attr listGrid.header (AutoChild : null : R)
    // A Toolbar used to manager the headers shown for each column of the grid.
    // @group gridHeader
    // @visibility external
    //<
    
	//>	@attr listGrid.canTabToHeader (boolean : null : IR)
	// Should the header be included in the tab-order for the page? If not explicitly specified,
	// the header will be included in the tab order for the page if +link{isc.setScreenReaderMode()} is
    // called.
	// @group accessibility
	// @visibility external
	//<
	// See comments by _setTabIndex() for how listGrids handle tab-index manipulation
	//canTabToHeader:null,
    
	//>	@attr listGrid.headerHeight (number : 22 : [IRW])
	//          The height of this listGrid's header, in pixels.
    //      @setter listGrid.setHeaderHeight()
	//      @visibility external
	//      @group  gridHeader, sizing
	//<
    // Note: we treat a headerHeight of zero as an equivalent of showHeader:false
	headerHeight:22,

    //> @attr listGrid.minFieldWidth (int : 15 : IRW)
    // Minimum size, in pixels, for ListGrid headers.
    // @visibility external
    //<
    minFieldWidth:15,

	//>	@attr listGrid.showHeader (boolean: true : [IRW])
	// Should we show the header for this ListGrid?
    // @group gridHeader, appearance
	// @visibility external
	//<
	showHeader:true,

	//>	@attr listGrid.headerBarStyle (CSSStyleName : null : IR)
	// Set the CSS style used for the header as a whole.
	// @group	gridHeader, appearance
    // @visibility external
	//<
    //headerBarStyle:null,
    
    //>	@attr listGrid.headerBackgroundColor (color : "#CCCCCC" : IRW)
	// BackgroundColor for the header toolbar. Typically this is set to match the color
    // of the header buttons.
	//		@group	gridHeader, appearance
    // @visibility external
    //<
	headerBackgroundColor:"#CCCCCC",		
 
    headerDefaults : {
        // immediately reposition headers during drag resizing, don't delay
        instantRelayout:true,
        // when the user resizes buttons, don't try to fit them into the available space -
        // allow the user to introduce hscrolling
        enforcePolicy:false, 

        // when the header is clicked, have it call headerClick() on us
		itemClick : function (button, buttonNum) {
			this.Super("itemClick",arguments);
		    this.grid._headerClick(buttonNum, this);
		},
		
		itemDoubleClick : function (button, buttonNum) {
            this.Super("itemDoubleClick", arguments);
            this.grid._headerDoubleClick(buttonNum, this);
        },

        showContextMenu : function () {
            return this.grid.headerBarContextClick(this);
        },
        
		backgroundRepeat:isc.Canvas.NO_REPEAT,
        
        // don't print the header, we handle this as part of the body instead, to ensure column
        // alignment
        shouldPrint:false
    },
    
    //> @attr listGrid.headerButtonConstructor (Class : null : IR)
    // Widget class for this ListGrid's header buttons. If unset, constructor will be 
    // picked up directly from the standard +link{class:Toolbar} button constructor.
	// @group	gridHeader, appearance
    // @visibility external
    //<
    
    //>	@attr listGrid.headerBaseStyle (CSSStyleName : null : IR)
	// +link{Button.baseStyle} to apply to the buttons in the header, and the sorter, for 
    // this ListGrid.
    // Note that, depending on the +link{listGrid.headerButtonConstructor, Class} of the header
    // buttons, you may also need to set +link{listGrid.headerTitleStyle}.
	// @group	gridHeader, appearance
    // @visibility external
	//<
    
    //> @attr listGrid.headerTitleStyle (CSSStyleName : null : IR) 
    // +link{StretchImgButton.titleStyle} to apply to the buttons in the header, and the sorter,
    // for this ListGrid.
    // Note that this will typically only have an effect if 
    // +link{listGrid.headerButtonConstructor} is set to +link{class:StretchImgButton} or a subclass 
    // thereof.
	// @group	gridHeader, appearance
    // @visibility external
	//<
    
	//>	@attr listGrid.frozenHeaderBaseStyle (CSSStyleName : null : IR)
    // If this listGrid contains any frozen fields, this property can be used to apply a custom
    // headerBaseStyle to the frozen set of fields. If unset, the standard headerBaseStyle
    // will be used for both frozen and unfrozen cells.
    // @visibility external
    // @group gridHeader, appearance, frozenFields
    // @see listGrid.headerBaseStyle
    // @see listGridField.frozen
    //<
    
    //>	@attr listGrid.frozenHeaderTitleStyle (CSSStyleName : null : IR)
    // If this listGrid contains any frozen fields, this property can be used to apply a custom
    // headerTitleStyle to the frozen set of fields. If unset, the standard headerTitleStyle
    // will be used for both frozen and unfrozen cells.
    // @visibility external
    // @group gridHeader, appearance, frozenFields
    // @see listGrid.headerTitleStyle
    // @see listGridField.frozen
    //<
    

    //>	@attr listGrid.headerButtonDefaults (Button Properties: {...} : IRA)
	// Defaults to apply to all header buttons. To modify this object, 
    // use +link{class.changeDefaults(), ListGrid.changeDefaults()} 
    // rather than replacing with an entirely new object.
	// @group	gridHeader, appearance
    // @visibility external
	//<
	headerButtonDefaults:{
        
        // override getCurrentCursor to show default (non pointer) for canSort:false fields
        getCurrentCursor : function () {
            var grid = this.parentElement ? this.parentElement.grid : null;
            var currentCursor;
            if (grid && this.masterIndex != null) {
                var field = grid.getField(this.masterIndex),
                    canSort = (grid.canSort != false && grid._canSort(field) != false);
                if (canSort) currentCursor = isc.Canvas.HAND;
                else currentCursor = isc.Canvas.DEFAULT;
            } else {
                currentCursor = this.getClass().getPrototype().cursor;
            }
            this.cursor = currentCursor;
            return this.Super("getCurrentCursor", arguments);
        },
        
        dragScrollType:"parentsOnly",
		minWidth:20
	},										

	//>	@attr listGrid.headerButtonProperties (Button Properties: null : IRA)
	// Properties to apply to all header buttons.
    // Overrides defaults applied via  +link{ListGrid.headerButtonDefaults}.
    // @group	gridHeader, appearance
    // @visibility external
	//<
    
    //> @attr listGrid.sorterConstructor (Class : Button : IR)
    // Widget class for the corner sort button, if showing. For consistent appearance, this
    // is usually set to match +link{listGrid.headerButtonConstructor}
    // @group	gridHeader, appearance
    // @visibility external
    //<
    sorterConstructor:isc.Button,
    
    
    //> @attr listGrid.sorterDefaults (object : {...} : IRA)
    // Defaults to apply to the corner sort button. To modify this object, use
    // +link{Class.changeDefaults(), ListGrid.changeDefaults()} rather than replacing with an
    // entirely new object.
    // @group gridHeader, appearance
    // @visibility external
    //<
    sorterDefaults:{
        _redrawWithParent:false,
        getTitle : function () {return this.parentElement.getSortArrowImage() },
        click : function () { return this.parentElement.sorterClick() },
        showContextMenu : function() { return this.parentElement.sorterContextClick() },
        canFocus:false  // no need to focus, since we allow focus-ing on the header buttons
    },
    
	//>	@attr listGrid.sorterProperties (Button Properties: null : IRA)
	// Properties to apply to the sorter button. Overrides defaults applied via 
    // +link{ListGrid.sorterDefaults}.
    // @group	gridHeader, appearance
    // @visibility external
	//<
    
	// Sorting
	// --------------------------------------------------------------------------------------------

	//>	@attr listGrid.canSort (boolean : true : [IRW])
	// Enables or disables interactive sorting behavior for this listGrid. Does not
	// affect sorting by direct calls to the sort method.
	// @visibility external
	// @group  sorting
	//<
	canSort:true,

	//>	@attr listGrid.canUnsort (boolean : false : [IRW])
    // When set to true, the third click on a column header removes the sort indicator
    // from the field.
    //
	// @visibility internal
	// @group  sorting
	//<
	canUnsort:false,

	//>	@attr listGrid.invalidateCacheOnUnsort (boolean : false : [IRW])
    // If true, and +link{listGrid.canUnsort} is also true and the grid is databound to a
    // +link{ResultSet}, then unsort causes the grid to drop the current client-side
    // data cache and refetch the current range of records from the server.
    //
	// @visibility internal
	// @group  sorting
	//<
    invalidateCacheOnUnsort: false,

	//>	@attr listGrid.sortFieldNum (number : null : [IRW])
	// Specifies the number of the field by which to sort this listGrid. Column numbers
	// start at 0 for the left-most column.
	// @visibility external
	// @group  sorting
    // @example sort
    // @deprecated as of version 7.0 in favor of +link{listGrid.sortField}
	//<
	//sortFieldNum:null,

    //> @attr listGrid.sortField (String or integer : null : IR)
    // Specifies the field by which this grid should be initially sorted. Can be set to either a
    // +link{listGridField.name,field name} or the index of the field in the fields Array.
    // <P>
    // ListGrids also support being initialized with multiple-field sort via 
    // +link{listGrid.initialSort}. If initialSort is specified, it will be used in preference
    // to this property.
    // <P>
    // To sort the grid after it has been initialized, use +link{ListGrid.sort()} or 
    // +link{listGrid.setSort()}. Details about the current sort of a live grid can be
    // retrieved by calling +link{listGrid.getSortField()} or +link{listGrid.getSort()}
    //
    // @group sorting
    // @example sort
    // @visibility external
    //<

	//>	@attr listGrid.keyboardClickField (string|number : null : [IRW])
	// When simulating click events listGrid rows as a result of keyboard events
	// (navigating using the arrow keys, space, enter for doubleClick), which column
	// should the event be generated upon?
	// Should be set to the name or index of the desired column.
	// If null, defaults to the first column.
	// @group  events
	//<

	//>	@attr listGrid.sortDirection (SortDirection : Array.ASCENDING : [IRW])
    // Sorting direction of this ListGrid. If specified when the ListGrid is initialized,
    // this property will be the default sorting direction for the +link{listGrid.sortField}.
    // May be overridden by specifying +link{ListGridField.sortDirection}.
    // <P>
    // After initialization, this property will be updated on +link{ListGrid.sort()} or
    // +link{ListGrid.setSort()} to reflect the current sort direction of the grid. When
    // this grid is sorted by multiple fields, the grid's sortDirection reflects the
    // sort direction of the primary sort field.
    // @group  sorting
    // @see type:SortDirection
    // @example sort
    // @setter listGrid.sort()
	// @visibility external
	//<
   	
	sortDirection:Array.ASCENDING,

	//>	@attr listGrid.showSortArrow (SortArrow : null : [IRW])
	// Indicates whether a sorting arrow should appear for the listGrid, and its
	// location. See +link{SortArrow} for details.
    // <P>
	// Clicking the sort arrow reverses the direction of sorting for the current sort
	// column (if any), or sorts the listGrid by its first sortable column. The arrow
	// image on the button indicates the current direction of sorting.
    // If undefined, the sort arrow will show up in the sorted field, and the
    // corner sort button will be displayed if a vertical scrollbar is being displayed
    //
	// @group sorting, appearance
	// @visibility external
	//<
	
    //showSortArrow:null,

    //> @attr listGrid.canPickFields (Boolean : true : [IRW])
    // Indicates whether the field picker item and submenu should be present in the header
    // context menu. This menu allows the user to hide visible fields and show hidden fields.
    // By default only fields explicitly included in the +link{listGrid.fields} array will
    // be available in this menu, unless +link{listGrid.canPickOmittedFields} is set to true for
    // a databound grid.
    //
    // @visibility external
    //<
    canPickFields: true,
    
    //> @attr listGrid.canPickOmittedFields (Boolean : false : [IR])
    // If this grid has a specified +link{listGrid.dataSource}, and
    // +link{listGrid.useAllDataSourceFields} is false, setting this property to true
    // will cause all dataSource fields not included in the specified set of fields to
    // show up in the +link{listGrid.canPickFields,field picker menu item}.
    // <P>
    // Has no effect if +link{listGrid.fields} is unset (as in this case all dataSource fields
    // will be displayed by default), or if +link{listGrid.canPickFields} is false.
    // @visibility external
    //<
    canPickOmittedFields:false,
    
    // Frozen Fields (aka Frozen Columns)
    // ---------------------------------------------------------------------------------------
    
    //> @groupDef frozenFields
    // Frozen fields are fields that do not scroll horizontally with other fields, remaining on
    // the screen while other fields may be scrolled off.  This feature is typically used to
    // allow basic identifying information (like an "accountId") to remain on screen while the
    // user scrolls through a large number of related fields.
    // <P>
    // Fields can be programmatically frozen via setting
    // +link{listGridField.frozen,field.frozen} to true when the grid is created, or
    // dynamically frozen and unfrozen via +link{listGrid.freezeField,freezeField()} and
    // +link{listGrid.unfreezeField,unfreezeField()}.
    // The setting +link{listGrid.canFreezeFields,canFreezeFields} enables a user interface to
    // allow end users to dynamically freeze and unfreeze fields.
    // <P>
    // The frozen fields feature is not compatible with the following features:
    // <ul>
    // <li> variable height auto-sizing records
    //      (+link{listGrid.fixedRecordHeights,fixedRecordHeights:false})
    // <li> +link{autoFitData}:"horizontal", as well as headers that autoFit to titles
    //      (normally enabled via <code>field.overflow:"visible"</code>)
    // <li> the +link{CubeGrid} subclass of ListGrid
    // <li> nested grids
    // </ul>
    // The frozen fields feature <b>is</b> compatible with column resize and reorder, selection
    // and multi-selection, loading data on demand, inline editing, drag and drop and reorder
    // of records, the +link{TreeGrid} subclass of ListGrid, and all dynamic styling-related and
    // formatting-related features.
    // 
    // @title Frozen Fields
    // @visibility external
    //<

    //> @attr listGrid.canFreezeFields (boolean : null : IRW)
    // Whether an interface should be shown to allow user is allowed to dynamically "freeze" or
    // "unfreeze" columns with respect to horizontally scrolling. If unset, this property defaults
    // to <code>true</code> unless:<ul>
    // <li>+link{listGrid.fixedRecordHeights,this.fixedRecordHeights} is <code>false</code></li>
    // <li>+link{listGrid.bodyOverflow,this.bodyOverflow} is <code>"visible"</code></li>
    // <li>+link{listGrid.autoFitData,this.autoFitData} is set to <code>"horizontal"</code> or 
    // <code>"both"</code></li>
    // <li>Any field has overflow set to <code>"visible"</code></li></ul>
    // <P>
    // Note that the <code>canFreezeFields</code> setting enables or disables the user
    // interface for freezing and unfreezing fields only.  Fields can be programmatically
    // frozen via setting +link{listGridField.frozen,field.frozen} to true when the grid is
    // created, or dynamically frozen and unfrozen via +link{freezeField()} and
    // +link{unfreezeField()}.
    //
    // @group frozenFields
    // @visibility external
    //<
    // Note that fixedColumnWidths:false will also disable canFreezeFields but this
    // is not currently public.
    
    // -------------------------
    // Formula / summary fields (picked up from databoundcomponent)
    
    //> @attr listGrid.canAddFormulaFields (boolean : false : IRW)
    // @include dataBoundComponent.canAddFormulaFields
    // @visibility external
    //<
    
    //> @method listGrid.getFormulaFieldValue()
    // @include dataBoundComponent.getFormulaFieldValue()
    // @visibility external
    //<
    
    //> @attr listGrid.canAddSummaryFields (boolean : false : IRW)
    // @include dataBoundComponent.canAddSummaryFields
    // @visibility external
    //<
    
    //> @method listGrid.getSummaryFieldValue()
    // @include dataBoundComponent.getSummaryFieldValue()
    // @visibility external
    //<
    
    
    
	// Context Menus
	// --------------------------------------------------------------------------------------------

	//>	@attr listGrid.showCellContextMenus (boolean : false : [IRW])	
	// Whether to show a context menu with standard items for all context clicks on rows in the
	// body.
    // @visibility external
	//<
	//showCellContextMenus:false,
    
    //> @attr listGrid.openRecordEditorContextMenuItemTitle (string : "Edit" : [IRW])
    // If +link{listGrid.canOpenRecordEditor} is true and +link{listGrid.showCellContextMenus}
    // is true, this property specifies the title for the context menu item shown allowing the
    // user to perfom editing on a row via an embedded form.
    // @group i18nMessages
    // @visibility nextedGrid
    //<
    openRecordEditorContextMenuItemTitle:"Edit",

    //> @attr listGrid.dismissEmbeddedComponentContextMenuItemTitle (string : "Dismiss" : IRW)
    // If +link{listGrid.showCellContextMenus} is true, and we are currently showing either
    // an embedded editor (see +link{listGrid.canOpenRecordEditor}) or an embedded
    // detail grid (see +link{listGrid.canOpenRecordDetailGrid}, this property
    // specifies the title for the context menu item shown allowing the user to dismiss the
    // embedded component.
    // @group i18nMessages
    // @visibility nextedGrid
    //<
    dismissEmbeddedComponentContextMenuItemTitle:"Dismiss",

    //> @attr listGrid.deleteRecordContextMenuItemTitle (string : "Delete" : IRW)
    // If +link{listGrid.showCellContextMenus} is true, this property
    // specifies the title for the context menu item shown allowing the user to delete the
    // record on which the contextMenu was shown.
    // @group i18nMessages
    // @visibility experimental
    //<    
    deleteRecordContextMenuItemTitle:"Delete",


	//> @attr listGrid.canOpenRecordDetailGrid (boolean : true : [IRW])
	// Whether context menu items will be shown for viewing records from related DataSources in
	// grids embedded in the record.
	// <P>
	// Valid only when <code>showCellContextMenus</code> is true.
	// @visibility nestedGrid
	//<
    canOpenRecordDetailGrid:true,

	//> @attr listGrid.recordDetailGridProperties (Object : null : [IR])
	// Properties for detail grids shown embedded inside rows.
	// @visibility nestedGrid
	//<

	//> @attr listGrid.canOpenRecordEditor (boolean : true : [IRW])
	// Whether a context menu item will be shown for editing records with a form embedded in
	// the record.
	// <P>
	// Valid only when <code>showCellContextMenus</code> is true.
	// @visibility nestedGrid
	//<
    canOpenRecordEditor:true,

	//> @attr listGrid.recordEditorProperties (Object : null : [IR])
	// Properties for editor forms shown embedded inside rows.
    // @see listGrid.canOpenRecordEditor
	// @visibility nestedGrid
	//<
    
    //> @attr listGrid.recordEditorSaveButtonTitle (string : "Save" : [IRW])
    // Title for the Save button shown in the editor form embedded inside rows if 
    // +link{listGrid.canOpenRecordEditor} is true.
    // @see listGrid.canOpenRecordEditor
    // @group i18nMessages
    // @visibility nestedGrid
    //<
    recordEditorSaveButtonTitle:"Save", 

    //> @attr listGrid.recordEditorCancelButtonTitle (string : "Cancel" : [IRW])
    // Title for the Cancel button shown in the editor form embedded inside rows if 
    // +link{listGrid.canOpenRecordEditor} is true.
    // @see listGrid.canOpenRecordEditor
    // @group i18nMessages
    // @visibility nestedGrid
    //<
    recordEditorCancelButtonTitle:"Cancel",


	//>!BackCompat 2007.02.02
    // showCornerContextMenu was never externally documented and we have no in-code comments
    // about having ever exposed this property, so it may be safe to get rid of this
    // back-compat

	//>	@attr listGrid.showCornerContextMenu (boolean : null : [IR])	
	// Whether to allow a context menu on the sorter with standard items for showing and hiding
	// fields.
    // @deprecated as of 5.6 in favor of +link{attr:listGrid.showHeaderContextMenu}
	//<
    //<!BackCompat
    
	//>	@attr listGrid.showHeaderContextMenu (boolean : true : [IR])	
	// Whether to show a context menu on the header with standard items for showing and hiding
	// fields.
    // @group gridHeader
    // @see method:listGrid.displayHeaderContextMenu()
    // @see method:listGrid.getHeaderContextMenuItems()
    // @visibility external
	//<
    // NOTE: avoid crashing if Menu class isn't loaded by defaulting to false.
	// when we load the Menu class, we override this default.
	//showHeaderContextMenu:false,
    
    // headerMenuButton
    // ----------------------------
    //>	@attr listGrid.showHeaderMenuButton (boolean : false : [IR])	
    // If set to true and +link{listGrid.showHeaderContextMenu,showHeaderContextMenu} is true, the
    // +link{listgrid.headerMenuButton} will be displayed when the user rolls
    // over the header buttons in this grid.
    // @group headerMenuButton    
    // @visibility external
	//<
    // As with showHeaderContextMenu, this default should not be set to true until we know
    // for sure that Menu has been loaded (see Menu.js)
    //showHeaderMenuButton:true,
    
    //> @attr listGrid.headerMenuButtonConstructor (className : null : [IRA])
    // Constructor for the  +link{listGrid.headerMenuButton}. If unset a standard "Button" will
    // be rendered out. Note that this property may be overridden by different skins.
    // @group headerMenuButton
    // @visibility external
    //<
    //headerMenuButtonConstructor: "StretchImgButton",
    
    //> @attr listGrid.headerMenuButton (AutoChild : null : [RA])
    // If +link{showHeaderMenuButton} is true, when the user rolls over the header buttons in this
    // grid the headerMenuButton will be shown over the header button in question. When clicked
    // this button will display the standard header context menu (see
    // +link{listGrid.displayHeaderContextMenu}).
    // <P>
    // +link{group:headerMenuButton,Several properties} exist to customize the appearance of the
    // headerMenuButton. Also see the +link{type:AutoChild} documentation for information on how 
    // to make freeform modifications to autoChild widgets
    // @group headerMenuButton
    // @visibility external
    //<
    
    //>	@attr listGrid.headerMenuButtonIcon (URL : "[SKIN]/ListGrid/sort_descending.gif" : [IRA])	
	// If +link{listGrid.showHeaderMenuButton} is true, this property governs the icon shown on the
    // auto-generated <code>headerMenuButton</code>
    // @group headerMenuButton
    // @visibility external
	//<
    headerMenuButtonIcon:"[SKIN]/ListGrid/headerMenuButton_icon.gif",
    
    //>	@attr listGrid.headerMenuButtonIconWidth (number : 7 : [IRA])	
	// If +link{listGrid.showHeaderMenuButton} is true, this property governs the width of the icon
    // shown on the auto-generated <code>headerMenuButton</code>
    // @group headerMenuButton
    // @visibility external
	//<
    headerMenuButtonIconWidth:7,
    
    //>	@attr listGrid.headerMenuButtonIconHeight (number : 7 : [IRA])	
	// If +link{listGrid.showHeaderMenuButton} is true, this property governs the height of the icon
    // shown on the auto-generated <code>headerMenuButton</code>
    // @group headerMenuButton
    // @visibility external
	//<
    headerMenuButtonIconHeight:7,
    
    //>	@attr listGrid.headerMenuButtonWidth (number : 16 : [IRA])	
	// If +link{listGrid.showHeaderMenuButton} is true, this property governs the width of the 
    // auto-generated <code>headerMenuButton</code>
    // @group headerMenuButton
    // @visibility external
	//<
    headerMenuButtonWidth:16,

    //>	@attr listGrid.headerMenuButtonHeight (measure : "100%" : [IRA])	
	// If +link{listGrid.showHeaderMenuButton} is true, this property governs the height of the 
    // auto-generated <code>headerMenuButton</code>
    // @group headerMenuButton
    // @visibility external
	//<
    headerMenuButtonHeight:"100%",
    

	// Drag Resize / Reorder / Drag and Drop
	// --------------------------------------------------------------------------------------------

	//>	@attr listGrid.canDragRecordsOut (boolean : false : [IRW])	
	// Indicates whether records can be dragged from this listGrid and dropped elsewhere.
	// @visibility external
	// @group  dragging
    // @see ListGridRecord.canDrag
    // @see ListGridRecord.canAcceptDrop
    // @example dragListMove
	//<
	canDragRecordsOut:false,
    
	//>	@attr listGrid.canAcceptDroppedRecords (boolean : false : [IRW])
	// Indicates whether records can be dropped into this listGrid.
	// @visibility external
	// @group  dragging
    // @see ListGridRecord.canDrag
    // @see ListGridRecord.canAcceptDrop    
    // @example dragListMove
	//<
	//canAcceptDroppedRecords:false,
    
	//>	@attr listGrid.canReorderRecords (boolean : false : [IRW])
	// Indicates whether records can be reordered by dragging within this listGrid.
	// @visibility external
    // @group  dragging
    // @see ListGridRecord.canDrag
    // @see ListGridRecord.canAcceptDrop    
    // @example dragListMove
    // @example gridsDragReorder
	//<
	//canReorderRecords:false,
    
	//>	@attr listGrid.canReorderFields (boolean : true : [IRW])
	// Indicates whether fields in this listGrid can be reordered by dragging and
	// dropping header fields.
	// @visibility external
	// @group  dragging
    // @example columnOrder
	//<
	canReorderFields:true,
    
	//>	@attr listGrid.canResizeFields (boolean : true : [IRW])
	// Indicates whether fields in this listGrid can be resized by dragging header
	// fields.
	// @visibility external
	// @group  dragging
    // @example columnSize
	//<
	canResizeFields:true,

	// for dragging records out, use the drag tracker
    dragAppearance:isc.EH.TRACKER,
    
    //> @type DragTrackerMode
    // When records are being dragged from within a ListGrid, what sort of drag-tracker
    // should be displayed?
    // @value "none" Don't display a drag tracker at all
    // @value "icon" Display an icon to represent the record(s) being dragged. Icon src is
    //              derived from +link{ListGrid.getDragTrackerIcon()}
    // @value "title" Display a title for the record being dragged. Title derived from
    //              +link{ListGrid.getDragTrackerTitle()}
    // @value "record" Display the entire record being dragged
    // @group dragTracker
    // @visibility external
    //<
    
    //> @attr listGrid.dragTrackerMode (DragTrackerMode : "icon" : [IRA])
    // When records are being dragged from within a ListGrid, what sort of drag-tracker
    // should be displayed?<br>
    // Note that if multiple records are being dragged the displayed tracker will be
    // based on the first selected record.
    // @group dragTracker
    // @visibility external
    //<
    dragTrackerMode:"title",
    
    
	//>	@attr listGrid.resizeFieldsInRealTime (boolean : isc.Browser.isIE && isc.Browser.isWin : IRWA)
	//		True == we redraw the list viewer in real time as fields are being resized.
	//		This can be slow with a large list and/or on some platforms.
    //
	//		@group	dragging
    //      @visibility external
	//<
	resizeFieldsInRealTime: (isc.Browser.isIE && isc.Browser.isWin) 
                            || (isc.Browser.isFirefox && isc.Browser.geckoVersion >= 20080529)
                            // Safari 3.0+, Google Chrome
                            || (isc.Browser.isSafari && isc.Browser.safariVersion >= 500), 
	
    //>	@attr listGrid.dragDataAction		
    // @include dataBoundComponent.dragDataAction
    //<
    
	// Embedded Components
	// --------------------------------------------------------------------------------------------
    embeddedComponentIndent: 25,

	// Nested Master-Detail
	// --------------------------------------------------------------------------------------------
    nestedGridDefaults : {
        height:150
    },

	// Skinning
	// --------------------------------------------------------------------------------------------
	//>	@attr listGrid.skinImgDir (URL : "images/ListGrid/" : IRWA)
	// Where do 'skin' images (those provided with the class) live?
	// @group appearance, images
    // @visibility external
	//<
	skinImgDir:"images/ListGrid/",		

	//>	@attr listGrid.sortAscendingImage (ImgProperties : {...} : IRWA)
	// Image to show when sorting ascending. See +link{class:ImgProperties} for format.
	// @group appearance
    // @visibility external
	//<
	sortAscendingImage:{src:"[SKIN]sort_ascending.gif", width:7, height:7},

	//>	@attr listGrid.sortDescendingImage (ImgProperties : {...} : IRWA)
	// Image to show when sorting descending. See +link{class:ImgProperties} for format.
	// @group appearance
    // @visibility external
	//<
	sortDescendingImage:{src:"[SKIN]sort_descending.gif", width:7, height:7},

	//>	@attr listGrid.trackerImage (ImgProperties : {...} : IRWA)
	// Default image to use for the dragTracker when things are dragged within or out of this 
    // list. See +link{class:ImgProperties} for format.
    //
	// @group dragTracker
    // @see listGrid.dragTrackerMode
    // @see listGrid.getDragTrackerIcon()
    // @visibility external
	//<	
    trackerImage:{src:"[SKIN]tracker.gif", width:16, height:16},
    
    //> @attr listGrid.booleanTrueImage (SCImgURL : null : IRWA)
    // Image to display for a true value in a boolean field.
    // <P>
    // To turn this off explicitly set +link{listGridField.suppressValueIcon} to true.
    // <P>
    // If this, +link{listGrid.booleanFalseImage} and +link{listGrid.booleanPartialImage}
    // are undefined, this will be set to +link{checkboxItem.checkedImage}.
    // @see listGrid.booleanFalseImage
    // @see listGrid.booleanPartialImage
    // @group imageColumns
    // @visibility external
    //<
    booleanTrueImage:null,
    
    //> @attr listGrid.booleanFalseImage (SCImgURL : null : IRWA)
    // Image to display for a false value in a boolean field. Default <code>null</code> value
    // means no image will be displayed
    // <P>
    // To turn this off explicitly set +link{listGridField.suppressValueIcon} to true
    // <P>
    // If this, +link{listGrid.booleanTrueImage} and +link{listGrid.booleanPartialImage}
    // are undefined, this will be set to +link{checkboxItem.uncheckedImage}.
    // @group imageColumns
    // @see listGrid.booleanTrueImage
    // @see listGrid.booleanPartialImage
    // @visibility external
    //<
    booleanFalseImage:null,
    
    //> @attr listGrid.booleanPartialImage (SCImgURL : null : IRWA)
    // Image to display for a partially true value in a boolean field (typically selection).
    // <P>
    // To turn this off explicitly set +link{listGridField.suppressValueIcon} to true.
    // <P>
    // If this, +link{listGrid.booleanTrueImage} and +link{listGrid.booleanFalseImage}
    // are undefined, this will be set to +link{checkboxItem.partialSelectedImage}.
    // @see listGrid.booleanTrueImage
    // @see listGrid.booleanFalseImage
    // @group imageColumns
    // @visibility external
    //<
    booleanPartialImage:null,

    //> @attr listGrid.booleanImageWidth (number : 16 : IRWA)
    // Width for the +link{listGrid.booleanTrueImage}, +link{listGrid.booleanFalseImage}
    // and +link{listGrid.booleanPartialImage}.
    // Note: If +link{listGrid.booleanTrueImage} is unset, the +link{checkboxItem.checkedImage}
    // will be used to indicate a true value in a boolean field. In this case this property is
    // ignored in favor of +link{checkboxItem.valueIconWidth}.
    // @group imageColumns
    // @visibility external
    //<
    
    booleanImageWidth:16,
    
    //> @attr listGrid.booleanImageHeight (number : 16 : IRWA)
    // Height for the +link{listGrid.booleanTrueImage}, +link{listGrid.booleanFalseImage}
    // and +link{listGrid.booleanPartialImage}.
    // Note: If +link{listGrid.booleanTrueImage} is unset, the +link{checkboxItem.checkedImage}
    // will be used to indicate a true value in a boolean field. In this case this property is
    // ignored in favor of +link{checkboxItem.valueIconHeight}.
    // @group imageColumns
    // @visibility external
    //<
    booleanImageHeight:16,
    
    
    //> @attr listGrid.mozBodyOutlineColor (string : "white" : IRWA)
    // If we're in Moz Firefox 1.5 or above, and showing a header, what color should the 
    // dotted focus outline show around the body. Must be a color that contrasts with the 
    // header of the ListGrid.
    // @visibility internal
    //<
    mozBodyOutlineColor:"white",
    //> @attr listGrid.mozBodyNoHeaderOutlineColor (string : "red" : IRWA)
    // If we're in Moz Firefox 1.5 or above, and we're not showing a header, what color 
    // should the dotted focus outline show around the body. Must be a color that contrasts 
    // with the header of the ListGrid.
    // @visibility internal
    //<
    mozBodyNoHeaderOutlineColor:"red",

    
    
// -----------------------------------------------------------------------------------------
// row numbers
//

    //> @attr listGrid.showRowNumbers (boolean : null : IRWA)
    // When set to true, shows an additional field at the beginning of the field-list 
    // (respecting RTL) that displays the current rowNum for each record.
    // @group rowNumberField
    // @visibility external
    //<

    //> @attr listGrid.rowNumberStyle (CSSStyleName : "cellDark" : IRWA)
    // The CSS Style name for the +link{listGrid.rowNumberField}.
    // @group rowNumberField
    // @visibility external
    //<
    rowNumberStyle: "specialCol",

    //> @attr listGrid.rowNumberStart (number : 1 : IRWA)
    // The number to start the row-count from - default value is 1.
    // @group rowNumberField
    // @visibility external
    //<
    rowNumberStart: 1,

    //> @attr listGrid.rowNumberField (AutoChild ListGridField : null : IRWA)
    // An automatically generated field that displays the current row number when 
    // +link{listGrid.showRowNumbers, showRowNumbers} is true.
    // @group rowNumberField
    // @visibility external
    //<
    rowNumberFieldDefaults: {
        name: "_rowNumberField",
        excludeFromState:true,
        canEdit: false,
        canFilter:false,
        canGroupBy: false,
        canSort: false,
        canExport: false,
        canHide: false,
        canReorder: false,
        canDragResize: false,
        // make this special field canHilite false so we don't see it in HiliteEditors by default
        canHilite: false,
        showAlternateStyle: false,
        _isRowNumberField: true,
        showHeaderContextMenuButton: false,
        showDefaultContextMenu: false,
        keyboardFiresRecordClick: false,
        showGroupSummary:false,
        showGridSummary:false,
        formatCellValue : function (value, record, rowNum, colNum, grid) {
            if (grid.isGrouped) {
                if (record == null || record._isGroup) return "&nbsp;";
                
                var groupedRowNum = grid.getGroupedRecordIndex(record);
                // skip any records we can't find in the group tree (EG summary rows)
                if (groupedRowNum == -1) return null;
                return (grid.rowNumberStart + groupedRowNum);
            } else {
                return this.rowNumberStart + rowNum;
            }
        },
        autoFreeze: true
    },

    // helper method to get index of the group in which a record exists
    getParentGroupIndex : function (record) {
        // bail if we're not grouped (return group 0)
        if (!this.isGrouped) return 0;

        // find out which group this record is in
        var tree = this.groupTree,
            parentNode = tree.getParent(record),
            rootChildren = tree.getChildren(tree.getParent(parentNode)),
            groupCount=0
        ;

        for (var i=0; i<rootChildren.length; i++) {
            var child = rootChildren[i];
            if (child.groupValue == parentNode.groupValue) {
                groupCount = i;
                break;
            }
        }

        return groupCount;
    },

    // helper method to get the true row-number for a grouped record, whether or not prior groups are closed
    getGroupedRecordIndex : function (record) {
        // bail if we're not grouped
        if (!this.isGrouped) return -1;
        // find the true index of this record in a grouped grid - indexOf doesn't cater for 
        // closed groups
        var tree = this.groupTree,
            
            parentNode = tree == null ? null : tree.getParent(record);
            
        // Checking for parentNode == null allows us to skip group-summary nodes which
        // shouldn't be counted anyway.
        if (parentNode == null) return -1;

        var rootChildren = tree.getChildren(tree.getParent(parentNode)),
            groupCount=0,
            trueIndex=0
        ;

        for (var i=0; i<rootChildren.length; i++) {
            var child = rootChildren[i];

            if (child.groupValue == parentNode.groupValue) {
                var siblings = tree.getChildren(child);
                for (var j=0; j<siblings.length; j++) {
                    if (this.objectsAreEqual(siblings[j], record)) {
                        return trueIndex + j;
                    }
                }
            }
            var prevSiblings = tree.getChildren(child),
                length = prevSiblings.length;
            // Don't count group summary rows - these show up at the end of the
            // group [and we support an arbitrary number of them]
            if (this.showGroupSummary && !this.showGroupSummaryInHeader) {
                for (var ii = length-1; ii >=0; ii--) {
                    if (prevSiblings[ii].groupSummaryRecordProperty) length-=1;
                    else break;
                }
            }
            trueIndex += length;
        }

        return trueIndex;
    },

    // helper method to compare the properties on two objects
    objectsAreEqual : function (object1, object2) {
        for (var key in object1) {
            if (object1[key] != object2[key]) return false;
        }
        return true;
    },

    _rowNumberFieldWidth: 30,
    getRowNumberField : function () {
        var grid = this,
            rnField = {
                // default the width 
                width:this._rowNumberFieldWidth,
                baseStyle: this.rowNumberStyle,
                rowNumberStart: this.rowNumberStart,
                getAutoFreezePosition: function () { return grid.getRowNumberFieldPosition() }
            }
        ;
        isc.addProperties(rnField, this.rowNumberFieldDefaults, this.rowNumberFieldProperties);

        rnField.title = "&nbsp;";
        
        return rnField;
    },

    getCurrentRowNumberField : function () {
        var fields = this.getFields(),
            rnFields = fields.find(this.fieldIdProperty, "_rowNumberField");
        return !rnFields ? null : isc.isAn.Array(rnFields) ? rnFields[0] : rnFields;
    },

    //> @method listGrid.isRowNumberField()
    // Identifies whether the passed-in field is the specially generated
    // +link{listGrid.rowNumberField, rowNumberField} used when +link{showRowNumbers} is
    // true.  Use this method in your custom event handlers to avoid inappropriately
    // performing actions when the rowNumberField is clicked on.
    // 
    // @param field (ListGridField) field to test
    // @return (boolean) whether the provided field is the rowNumberField
    // @group rowNumberField
    // @visibility external
    //<
    isRowNumberField : function (field) {
        if (!field || !field._isRowNumberField) return false;
        else return true;
    },

    // helper function to get the rowNumber field position
    // Appears at the far left of the grid, to the left of the other special fields which are
    // - group summary title field
    // - expansion component icon field
    // - checkbox selection field
    getRowNumberFieldPosition : function () {
        if (this.fieldSourceGrid) return this.fieldSourceGrid.getRowNumberFieldPosition();
        if (!this.showRowNumbers) return -1;
        return 0;
    },

    shouldShowRowNumberField : function () {
        // fieldSourceGrid: for cases like the summaryRow / filterEditor where we
        // share field objects across grids (and don't necessarily percolate settings like 
        // 'showRowNumbers').
        return this.fieldSourceGrid ? this.fieldSourceGrid.shouldShowRowNumberField() 
                                    : (this.showRowNumbers == true);
    },

    //> @attr listGrid.exportRawValues (Boolean : null : IR)
    // Dictates whether the data in this grid should be exported raw by 
    // +link{DataBoundComponent.exportClientData, exportClientData()}.  If set to true,  
    // data will not be processed by field-formatters during exports.
    // Decreases the time taken for large exports.  This property can also be set at the
    // +link{listGridField.exportRawValues, field level}.
    //
    // @visibility external
    //<

// -----------------------------------------------------------------------------------------
// Expando Rows
//

    //> @attr listGrid.canExpandRecords (boolean : false : IRWA)
    // When set to true, shows an additional field at the beginning of the field-list 
    // (respecting RTL) to allow users to expand and collapse individual records.
    // See +link{listGrid.expandRecord()} and +link{listGrid.expansionMode} for details
    // on record expansion.
    // <P>
    // If expanded records will be variable height,
    // you should switch on +link{listGrid.virtualScrolling, virtualScrolling}.
    // <P>
    // Note that expanded records are not currently supported in conjunction 
    // with +link{listGridField.frozen,frozen fields}.
    
    // @group expansionField
    // @visibility external
    //<

    //> @attr listGrid.canExpandMultipleRecords (boolean : true : IRWA)
    // When +link{listGrid.canExpandRecords} is true, this property indicates whether multiple
    // records can be expanded simultaneously.  If set to false, expanding a record will
    // automatically collapse any record which is already expanded.  The default value is 
    // <code>true</code>.
    // 
    // @group expansionField
    // @visibility external
    //<
    canExpandMultipleRecords: true,

    //> @attr listGrid.maxExpandedRecords (integer : null : IRWA)
    // When +link{listGrid.canExpandRecords} and +link{listGrid.canExpandMultipleRecords} are
    // both true, this property dictates the number of
    // records which can be expanded simultaneously.  If the expanded record count hits the
    // value of this property, further attempts to expand records will result in a popup
    // warning (see +link{listGrid.maxExpandedRecordsPrompt}) and expansion will be cancelled.
    // <P>
    // The default value is null, meaning there is no limit on the number of expanded records.
    // 
    // @group expansionField
    // @visibility external
    //<

    //> @attr listGrid.maxExpandedRecordsPrompt (String : "This grid is limited to \${count} simultaneously expanded records.  Please collapse some expanded records and retry." : IR)
    // This is a dynamic string - text within <code>\${...}</code> will be evaluated as JS code
    // when the message is displayed. Note that the local variable <code>count</code> will be
    // available and set to this.maxExpandedRecords. The string will be executed in the scope of the
    // ListGrid so <code>this</code> may also be used to determine other information about this grid.
    // <P>
    // Default value returns <P>
    // <code>
    // <i>This grid is limited to <code>[+link{listGrid.maxExpandedRecords}]</code> simultaneously 
    // expanded records.  Please collapse some expanded records and retry.</i>
    // </code>
    // @visibility external
    // @group i18nMessages
    //<
    maxExpandedRecordsPrompt: "This grid is limited to \${count} simultaneously expanded records.  Please collapse some expanded records and retry.",

    //> @type ExpansionMode
    // When +link{ListGrid.canExpandRecords, canExpandRecords} is true, ExpansionMode
    // dictates the type of UI to be displayed in the expanded portion of the row.
    // <P>
    // There are a number of builtin ExpansionModes and you can override 
    // +link{listGrid.getExpansionComponent, getExpansionComponent()} to create your own
    // expansion behaviors.
    //  
    //  @value  "detailField"  Show a single field's value in an +link{class:HtmlFlow}. Field 
    //      to use is +link{listGrid.detailField}.
    //  @value  "details"   Show a +link{class:DetailViewer} displaying those fields from the 
    //      record which are not already displayed in the grid.
    //  @value  "related"    Show a separate +link{class:ListGrid} containing related-records.
    //      See +link{ListGridRecord.detailDS} and +link{ListGrid.recordDetailDSProperty} for 
    //      more information.
    //  @value  "editor"    Show a +link{class:DynamicForm} to edit those fields from the record
    //      which not already present in the grid.
    //  @value  "detailRelated"    Show a +link{class:DetailViewer} displaying those fields
    //      from the record not already displayed in the grid, together with a separate
    //      +link{class:ListGrid} containing related-records.
    // @group expansionField
    // @visibility external
	//<

    //> @attr listGrid.detailField (String : null : IRWA)
    // The field whose contents to show in the expanded portion of a record when 
    // +link{listGrid.canExpandRecords, canExpandRecords} is <code>true</code> and 
    // +link{type:ExpansionMode, listGrid.expansionMode} is <code>detailField</code>.
    // @group expansionField
    // @visibility external
    //<
    
    //> @attr listGrid.expansionMode (ExpansionMode : null : IRWA)
    // The +link{ExpansionMode} for records in this grid.
    // Default <code>null</code> value means no expansion.
    // @group expansionField
    // @visibility external
    //<
    
    //> @attr listGrid.expansionField (AutoChild ListGridField : null : IRWA)
    // The field providing the facility to expand and collapse rows.
    // @group expansionField
    // @visibility external
    //<
    expansionFieldDefaults: {
        name: "_expansionField",
        canEdit: false,
        canFilter:false,
        canGroupBy: false,
        canSort: false,
        canExport: false,
        canHide: false,
        canReorder: false,
        canDragResize: false,
        // make this special field canHilite false so we don't see it in HiliteEditors
        canHilite: false,
        _isExpansionField: true,
        showHeaderContextMenuButton: false,
        showDefaultContextMenu: false,
        keyboardFiresRecordClick: false,
        cellAlign: "center",
        recordClick: function (viewer, record, recordNum, field, fieldNum, value, rawValue) {
            if (!viewer.canExpandRecords || !field._isExpansionField) return;
            if (!viewer._canExpandRecord(record, recordNum)) return;
            if (record.expanded) viewer.collapseRecord(record);
            else viewer.expandRecord(record);
        },
        formatCellValue : function (value, record, rowNum, colNum, grid) {
            // This ensures that if we're looking at an edit row for a new record we
            // don't show the expansion icon
            record = grid.getCellRecord(rowNum, colNum);
            if (record == null || record._isGroup) return null;
            if (!grid._canExpandRecord(record, rowNum)) return null;
            return grid.getValueIconHTML(
                record.expanded ? grid.expansionFieldTrueImage : grid.expansionFieldFalseImage, 
                this
            );
        },
        autoFreeze: true
    },

    // Helper method - should this grid show the special expansion field when canExpandRecords is true
    shouldShowExpansionField : function () {
        return this.fieldSourceGrid ? this.fieldSourceGrid.shouldShowExpansionField() 
                                    : this.canExpandRecords == true;
    },

    //> @attr listGrid.expansionFieldTrueImage (SCImgURL : null :IRWA)
    // If +link{listGrid.canExpandRecords} is set to <code>true</code>, this property
    // determines the image to display in the expansion field for expanded rows.
    // If unset, the +link{listGrid.booleanTrueImage} will be used.
    // @see listGrid.expansionFieldFalseImage
    // @see listGrid.expansionFieldImageWidth
    // @see listGrid.expansionFieldImageHeight
    // @group expansionField
    // @visibility external
    //<
    expansionFieldTrueImage: "[SKINIMG]/ListGrid/group_opened.gif",

    //> @attr listGrid.expansionFieldFalseImage (SCImgURL : null :IRWA)
    // If +link{listGrid.canExpandRecords} is set to <code>true</code>, this property
    // determines the image to display in the expansion field for collapsed rows.
    // If unset, the +link{listGrid.booleanFalseImage} will be used.
    // @see listGrid.expansionFieldTrueImage
    // @see listGrid.expansionFieldImageWidth
    // @see listGrid.expansionFieldImageHeight
    // @group expansionField
    // @visibility external
    //<
    expansionFieldFalseImage: "[SKINIMG]/ListGrid/group_opening.gif",

    //> @attr listGrid.expansionFieldImageWidth (integer : null : IR) 
    // If +link{listGrid.canExpandRecords} is set to <code>true</code>, this property
    // may be set to govern the width of the expansion image displayed to indicate whether a row 
    // is expanded. If unset, the expansionField image will be sized to match the
    // +link{listGrid.booleanImageWidth} for this grid.
    // @group expansionField
    // @visibility external
    //<
    
    //> @attr listGrid.expansionFieldImageHeight (integer : null : IR) 
    // If +link{listGrid.canExpandRecords} is set to <code>true</code>, this property
    // may be set to govern the height of the expansion image displayed to indicate whether a 
    // row is expanded. If unset, the expansionField image will be sized to match the
    // +link{listGrid.booleanImageHeight} for this grid.
    // @group expansionField
    // @visibility external
    //<

    //> @method listGrid.getExpansionField()
    // Returns the specially generated expansion field used when +link{canExpandRecords} is
    // true.
    // <P>
    // Called during +link{setFields()}, this method can be overridden to add advanced dynamic
    // defaults to the expansion field (call Super, modify the default field returned by Super,
    // return the modified field).  Normal customization can be handled by just setting
    // +link{AutoChild} properties, as mentioned under the docs for +link{listGrid.expansionField}.
    //
    // @return (ListGridField)
    // @group expansionField
    // @visibility external
    //<
    // the amount to add to the icon width to get the expansion field width
    expansionFieldExtraWidth: 16,
    getExpansionField : function () {
        var grid = this,
            expField = {
                excludeFromState:true,
                // default the width to the width of the icon plus an arbitrary buffer
                width: this._getExpansionFieldImageWidth() + this.expansionFieldExtraWidth,
                getAutoFreezePosition: function () { return grid.getExpansionFieldPosition() }
            }
        ;

        // if expansionFieldImageWidth/Height are set on this grid, pass them through to the field
        expField.valueIconWidth = this._getExpansionFieldImageWidth();
        expField.valueIconHeight = this._getExpansionFieldImageHeight();

        // combine the properties for the field using the autoChild pattern
        isc.addProperties(expField, this.expansionFieldDefaults, this.expansionFieldProperties);

        expField.title = "&nbsp;";

        return expField;
    },

    getCurrentExpansionField : function () {
        var fields = this.getFields(),
            expFields = fields.find(this.fieldIdProperty, "_expansionField");
        return !expFields ? null : isc.isAn.Array(expFields) ? expFields[0] : expFields;
    },

    _getExpansionFieldImageWidth : function () {
        return this.expansionFieldImageWidth || this.booleanImageWidth ||
                (isc.CheckboxItem ? isc.CheckboxItem.getInstanceProperty("valueIconWidth") : null);
    },
    _getExpansionFieldImageHeight : function () {
        return this.expansionFieldImageHeight || this.booleanImageHeight ||
                (isc.CheckboxItem ? isc.CheckboxItem.getInstanceProperty("valueIconHeight") : null);
    },

    //> @method listGrid.isExpansionField()
    // Identifies whether the passed-in field is the specially generated
    // +link{listGrid.expansionField,expansionField} used when +link{canExpandRecords} is
    // true.  Use this method in your custom event handlers to avoid inappropriately
    // performing actions when the expansionField is clicked on.
    // 
    // @param field (ListGridField) field to test
    // @return (boolean) whether the provided field is the expansion field
    // @group expansionField
    // @visibility external
    //<
    isExpansionField : function (field) {
        if (!field || !field._isExpansionField) return false;
        else return true;
    },

    // helper function to get the expansion field position
    // This is one of the fields that appears "stuck" to the left of the grid - these are
    // - row number field
    // - group summary title field
    // - expansion field
    // - checkbox selection field
    getExpansionFieldPosition : function () {
        if (this.fieldSourceGrid != null) return this.fieldSourceGrid.getExpansionFieldPosition();
        
        if (!this.canExpandRecords) return -1;
        
        var position = 0;
        if (this.showRowNumbers) position +=1;
        if (this.showingGroupTitleColumn()) position += 1;
        return position;
    },

    _canExpandRecord:function (record,rowNum) {
        if (record == null) record = this.getRecord(rowNum);
        if (record == null) return false;
        return this.canExpandRecord(record,rowNum);
    },
    //> @method listGrid.canExpandRecord()
    // Indicates whether a given record or rowNum can be expanded.  The default implementation
    // checks the value of +link{listGrid.canExpandRecords} and 
    // <code>record[+link{listGrid.canExpandRecordProperty}]</code>.
    // <P>
    // Override this method for more specific control over individual record expansion.
    // <P>
    // <b>Note:</b> Rows with no underlying record in the data array - for example newly 
    // added edit rows that have not yet been saved - cannot be expanded.
    // 
    // @param record (ListGridRecord) record to work with
    // @param rowNum (Number) rowNum of the record to work with
    // @return (boolean) true if the record can be expanded
    // @group expansionField
    // @visibility external
    //<
    canExpandRecord : function (record, rowNum) {
        return record[this.canExpandRecordProperty] == false ? false : 
            true && (this.canExpandRecords != false);
    },

    //> @method listGrid.expandRecord()
    // Expands a given +link{ListGridRecord, record} by creating a subcomponent and inserting it 
    // in to the record's grid-row.  A number of built-in +link{ExpansionMode, expansionModes} 
    // are supported by the default implementation of 
    // +link{listGrid.getExpansionComponent, getExpansionComponent()} and you can override
    // that method to provide your own expansion behavior.
    // <P>
    // Once a record has been expanded, the currently visible expansion component may be
    // retrieved via +link{getCurrentExpansionComponent()}.
    // 
    // @param record (ListGridRecord) record to expand
    // @group expansionField
    // @visibility external
    //<
    _expandedRecordCount: 0,
    expandRecord : function (record) {
        var component,
            rowNum = this.getRecordIndex(record)
        ;
        
        if (!record.expanded) {

            // notification method / cancellation point
            
            if (this.onExpandRecord != null && !this.onExpandRecord(record)) return;

            if (!this.canExpandMultipleRecords) {
                // can only expand one record - if one's expanded already, collapse it now
                if (this._currentExpandedRecord)
                    this.collapseRecord(this._currentExpandedRecord);
                this._currentExpandedRecord = record;
            } else if (this.maxExpandedRecords) {
                // limited number of expanded records allowed - if we've hit that number, show the
                // maxExpandedRecordsPrompt and return
                if (this._expandedRecordCount == this.maxExpandedRecords) {
                    var message = this.maxExpandedRecordsPrompt.evalDynamicString(this, { 
                        count: this.maxExpandedRecords 
                    });
                    isc.say(message);
                    return;
                }
            }
            
            // create an appropriate subcomponent and bind it
            component = this.getExpansionComponent(record);
            var layout = this.createAutoChild("expansionLayout", 
                {
                    layoutLeftMargin: this.embeddedComponentIndent,
                    members: [ component ]
                }
            );
            layout.isExpansionComponent = true;
            this.addEmbeddedComponent(layout, record, this.data.indexOf(record));
            record.expanded = true;
            record.hasExpansionComponent = true;
            this._expandedRecordCount++;
            if (!this.canExpandMultipleRecords) this._currentExpandedRecord = record;
        }

        this.delayCall("markForRedraw", ["Expanded Record"]);
        this.refreshRow(rowNum);
    },

    //> @method listGrid.collapseRecord()
    // Collapses a given +link{ListGridRecord, record} which has been previously expanded using
    // +link{listGrid.expandRecord}.
    // 
    // @param record (ListGridRecord) record to collapse
    // @group expansionField
    // @visibility external
    //<
    collapseRecord : function (record) {
        var component = 
                (record && record._embeddedComponents) ? 
                        record._embeddedComponents.find("isExpansionComponent", true) : null
        ;

        if (isc.isA.Layout(component)) {
            var member = component.getMember(0);
            if (isc.isA.DynamicForm(member) && member.valuesHaveChanged()) {
                if (this.autoSaveEdits == true) {
                    var _this = this;

                    if (this.expansionEditorShowSaveDialog) {
                        isc.confirm(this.expansionEditorSaveDialogPrompt,
                            function (yes) {
                                if (yes) {
                                    _this.saveExpansionDetail(member, component, record);
                                } else {
                                    _this._collapseRecord(record, component);
                                }
                            }
                        );
                    } else {
                        this.saveExpansionDetail(member, component, record);
                    }
                    return;
                } else {
                    var values = member.getChangedValues(),
                        rowNum = this.getRecordIndex(record),
                        _this = this
                    ;
                    if (this.expansionEditorShowSaveDialog) {
                        isc.confirm("You have unsaved changes - do you want to save them now?",
                            function (yes) {
                                if (yes) {
                                    for (var key in values) {
                                        //alert(key);
                                        _this.setEditValue(rowNum, key, values[key]);
                                    }
                                    _this._collapseRecord(record, component);
                                } else {
                                    _this._collapseRecord(record, component);
                                }
                            }
                        );
                        return;
                    } else {
                        for (var key in values) {
                            this.setEditValue(rowNum, key, values[key]);
                        }
                        this._collapseRecord(record, component);
                        return;
                    }
                }
            }
        }

        this._collapseRecord(record, component);
    },

    saveExpansionDetail : function (member, component, record) {
        var _this = this;
        member.saveData(
            function (dsResponse, data, dsRequest) {
                if (data) {
                    record = data;
                    _this._collapseRecord(record, component);
                }
            }, { showPrompt: true, promptStyle: "cursor" }
        );
    },
    
    _collapseRecord : function (record, component) {
        component = component || 
            (record && record._embeddedComponents ? 
                    record._embeddedComponents.find("isExpansionComponent", true) : null);

        if (record.expanded) {
            // notification method / cancellation point
            
            if (this.onCollapseRecord != null && !this.onCollapseRecord(record)) return;

            if (this._currentExpandedRecord && this._currentExpandedRecord == record)
                delete this._currentExpandedRecord;

            this.removeEmbeddedComponent(record, component ? component : this.frozenFields ? this.frozenFields.length : 0);
            this._expandedRecordCount--;
        }
        record.expanded = false;
        this._remapEmbeddedComponents();
        this.redraw();
    },

    //> @attr listGrid.expansionDetailField (AutoChild : null : RA)
    // Automatically generated +link{class:HTMLFlow} for displaying the contents of 
    // +link{listGrid.detailField, a specified field} in a record's expanded section when
    // +link{type:ExpansionMode, listGrid.expansionMode} is <code>detailField</code>.
    // <P>
    // This component is an +link{type:AutoChild} and as such may be customized via 
    // <code>listGrid.expansionDetailFieldProperties</code> and 
    // <code>listGrid.expansionDetailFieldDefaults</code>.
    // <P>
    // Note, however, that this is a multi-instance component (potentially one per record), 
    // so it is created using +link{Class.createAutoChild, createAutoChild()} not 
    // +link{Class.addAutoChild, addAutoChild()}, and no default single instance is created by 
    // name on the grid.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionDetailFieldDefaults: {
        _constructor: isc.HTMLFlow,
        autoDraw: false,
        width: "100%",
        height: "100%"
    },
    //> @attr listGrid.expansionDetails (AutoChild : null : RA)
    // Automatically generated +link{class:DetailViewer} for displaying the details of a record
    // in it's expanded section when
    // +link{type:ExpansionMode, listGrid.expansionMode} is <code>details</code>.  Note that 
    // only those fields
    // which do not already appear in the grid are displayed in the expanded section.
    // <P>
    // This component is an +link{type:AutoChild} and as such may be customized via 
    // <code>listGrid.expansionDetailsProperties</code> and 
    // <code>listGrid.expansionDetailsDefaults</code>.
    // <P>
    // Note, however, that this is a multi-instance component (potentially one per record), 
    // so it is created using +link{Class.createAutoChild, createAutoChild()} not 
    // +link{Class.addAutoChild, addAutoChild()}, and no default single instance is created by 
    // name on the grid.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionDetailsDefaults: {
        _constructor: isc.DetailViewer,
        autoDraw: false,
        width: "100%"
    },
    //> @attr listGrid.expansionRelated (AutoChild : null : RA)
    // Automatically generated +link{class:ListGrid} for displaying data related to a record
    // in it's expanded section when
    // +link{type:ExpansionMode, listGrid.expansionMode} is <code>related</code>.  
    // The +link{class:DataSource} containing the related data is provided by 
    // +link{listGrid.getRelatedDataSource, getRelatedDataSource()} which, by default,
    // returns the DataSource referred to in +link{listGridRecord.detailDS}.
    // <P>
    // This component is an +link{type:AutoChild} and as such may be customized via 
    // <code>listGrid.expansionRelatedProperties</code> and 
    // <code>listGrid.expansionRelatedDefaults</code>.
    // <P>
    // Note, however, that this is a multi-instance component (potentially one per record), 
    // so it is created using +link{Class.createAutoChild, createAutoChild()} not 
    // +link{Class.addAutoChild, addAutoChild()}, and no default single instance is created by 
    // name on the grid.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionRelatedDefaults: {
        _constructor: isc.ListGrid,
        autoDraw: false,
        width: "100%",
        autoFitData: "vertical",
        autoFitMaxRecords: 4
    },

    //> @attr listGrid.expansionEditorSaveDialogPrompt (string : "You have unsaved changes - do you want to save them now?" : IR)
    // When +link{listGrid.canExpandRecords, canExpandRecords} is true and
    // +link{listGrid.expansionMode, expansionMode} is <i>editor</i>, the prompt to display
    // when an expanded row is collapsed while it's nested editor has changed values.
    //
    // @group i18nMessages
    // @visibility external
    //<
    expansionEditorSaveDialogPrompt: "You have unsaved changes - do you want to save them now?",
    expansionEditorShowSaveDialog: false,

    //> @attr listGrid.expansionEditor (AutoChild : null : RA)
    // Automatically generated +link{class:DynamicForm} for editing the details of a record
    // in it's expanded section when
    // +link{type:ExpansionMode, listGrid.expansionMode} is <code>editor</code>.  Note that only 
    // those fields which do not already appear in the grid will appear in the expanded section.
    // <P>
    // This component is an +link{type:AutoChild} and as such may be customized via 
    // <code>listGrid.expansionEditorProperties</code> and 
    // <code>listGrid.expansionEditorDefaults</code>.
    // <P>
    // Note, however, that this is a multi-instance component (potentially one per record), 
    // so it is created using +link{Class.createAutoChild, createAutoChild()} not 
    // +link{Class.addAutoChild, addAutoChild()}, and no default single instance is created by 
    // name on the grid.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionEditorDefaults: {
        _constructor: "DynamicForm",
        autoDraw: false,
        numCols: 4,
        colWidths: ["*", "*", "*", "*"],
        width: "100%",
        saveOperationType: "update"
    },
    //> @attr listGrid.expansionDetailRelated (AutoChild : null : RA)
    // Automatically generated +link{class:HLayout} appearing in a record's expanded section
    // when +link{type:ExpansionMode, listGrid.expansionMode} is <code>detailRelated</code>.  
    // This component contains two other autoChild components,
    // a +link{class:DetailViewer} for viewing fields from the record which are not already 
    // present in the grid and a separate embedded +link{class:ListGrid} for displaying other 
    // data related to this record via record.detailDS.  See +link{listGrid.expansionDetails} 
    // and +link{listGrid.expansionRelated} for more information.
    // <P>
    // This component is an +link{type:AutoChild} and as such may be customized via 
    // <code>listGrid.expansionDetailRelatedProperties</code> and 
    // <code>listGrid.expansionDetailRelatedDefaults</code>.
    // <P>
    // Note, however, that this is a multi-instance component (potentially one per record), 
    // so it is created using +link{Class.createAutoChild, createAutoChild()} not 
    // +link{Class.addAutoChild, addAutoChild()}, and no default single instance is created by
    // name on the grid.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionDetailRelatedDefaults: {
        _constructor: isc.HLayout,
        autoDraw: false,
        width: "100%",
        height: "100%"
    },

    //> @attr listGrid.expansionLayout (AutoChild : null : RA)
    // Automatically generated +link{class:VLayout} which fills a record's expanded section
    // and contains other builtin +link{ExpansionMode, expansion-components}.  You can also
    // override +link{listGrid.getExpansionComponent, getExpansionComponent()} to provide
    // components of your own specification.
    // <P>
    // This component is an +link{type:AutoChild} and as such may be customized via 
    // <code>listGrid.expansionLayoutProperties</code> and 
    // <code>listGrid.expansionLayoutDefaults</code>.
    // 
    // @group expansionField
    // @visibility external
    //<
    expansionLayoutDefaults: {
        _constructor: isc.VLayout,
        autoDraw: false,
        width: "100%",
        height: 10,
        overflow: "visible"
    },
    
    //> @method listGrid.getCurrentExpansionComponent()
    // Returns the expansion component derived from +link{listGrid.getExpansionComponent()}
    // currently visible in some record, or null if the specified record is not showing
    // an expansion component.
    //
    // @param record (Integer Or ListGridRecord) rowNum or record to get the expansionComponent for
    // @return (Canvas) the currently visible expansion component for the expanded row.
    // @group expansionField
    // @visibility external
    //<
    getCurrentExpansionComponent : function (record) {
        if (isc.isA.Number(record)) record = this.getRecord(record);
        if (!record.hasExpansionComponent) return null;
        
        // we actually hang 'isExpansionComponent' onto the layout containing the generated
        // expansion component so return its (only) member.
        var component = record._embeddedComponents ? 
                    record._embeddedComponents.find("isExpansionComponent", true) : null;
        if (component) {
            return component.members[0];
        }
        return null;
    },

    //> @method listGrid.getExpansionComponent()
    // Gets the embedded-component to show as a given record's expansionComponent.  This 
    // component is then housed in +link{listGrid.expansionLayout, a VLayout} and embedded 
    // into a record's row.
    // <P>
    // By default, this method returns one of a set of builtin components, according to the 
    // value of +link{type:ExpansionMode, listGrid.expansionMode}.  You can override this method 
    // to return any component you wish to provide as an expansionComponent.
    // <P>
    // As long as the record is expanded, this component may be retrieved via a call to
    // +link{getCurrentExpansionComponent()}.
    // 
    // @param record (ListGridRecord) record to get the expansionComponent for
    // @return (Canvas | Canvas Properties) the component to embed
    // @group expansionField
    // @visibility external
    //<
    // override getExpansionComponent() to do exactly that, returning some component other than
    // those provided by default
    getExpansionComponent : function (record) {
        return this._getStockEmbeddedComponent(record, true, false);
    },

    //> @attr listGridRecord.backgroundComponent (Canvas : null : IR)
    // Has no effect unless +link{listGrid.showBackgroundComponent} is <code>true</code>.
	// <P>
    // Canvas created and embedded in the body behind a given record.   When set, either as
    // a Canvas or Canvas Properties, will be constructed if necessary, combined with the 
    // autoChild properties specified for +link{listGrid.backgroundComponent} and displayed 
    // behind this record in the page's z-order, meaning 
    // it will only be visible if the cell styling is transparent.
    // @group rowEffects
    // @visibility external
    //<

// Expando Rows - Record implementation

    //> @attr listGrid.detailDS (String : null : IRWA)
    // If +link{canExpandRecords} is true and +link{type:ExpansionMode,listGrid.expansionMode}
    // is <code>"related"</code>, this property specifies the dataSource for the 
    // related records grid to be shown embedded in expanded records.
    // <P>
    // This property may also be specified on a per-record basis - see 
    // +link{recordDetailDSProperty}
    // @visibility external
    //<

    //> @attr listGrid.recordDetailDSProperty (String : "detailDS" : IRWA)
    // The name of the ListGridRecord property that specifies the DataSource to use when 
    // +link{type:ExpansionMode, listGrid.expansionMode} is "related".  The default is 
    // +link{ListGridRecord.detailDS}. Note that you can set the +link{detailDS} at the
    // grid level instead if the same dataSource is to be used for all records.
    // 
    // @visibility external
    //<
    recordDetailDSProperty: "detailDS",

    //> @attr listGridRecord.detailDS (DataSource : null : IRWA)
    // The default value of +link{listGrid.recordDetailDSProperty}.
    // 
    // @visibility external
    //<
    
    //>	@method	listGrid.getRelatedDataSource() 
    // Returns the +link{class:DataSource} containing data related to the passed record.  Used
    // when +link{ListGrid.canExpandRecords} is true and +link{ExpansionMode} is "related". The
    // default implementation returns the DataSource specified in
    // +link{listGridRecord.detailDS} if set, otherwise +link{listGrid.detailDS}.
    // 
    // @param   record  (ListGridRecord)    The record to get the Related dataSource for.
    // @return (DataSource) The related DataSource for the "record" param
    // 
    // @visibility external
    //<
    getRelatedDataSource : function (record) {
        return isc.DS.getDataSource(record[this.recordDetailDSProperty]) ||
                    isc.DS.get(this.detailDS);
    },
    

    //> @attr listGrid.childExpansionMode (ExpansionMode : null : IRWA)
    // For +link{ExpansionMode, expansionModes} that show another grid or tree, what the 
    // child's expansionMode should be.
    // <P>Default value <code>null</code> means no further expansion.
    //
    // @group expansionField
    // @visibility external
    //<

    //> @attr listGrid.expansionCanEdit (Boolean : false : IRWA)
    // For +link{ExpansionMode, expansionModes} that show another grid or tree, is that 
    // component editable?
    // <P>The default value for this property is <code>false</code>.
    //
    // @group expansionField
    // @visibility external
    //<
    expansionCanEdit: false

});



isc.ListGrid.addMethods({

// warnOnReusedFields -- will log a warning at init time if this.fields is set to an array of 
// field objects which are already displayed as fields in another grid.
// Internal - we only disable this for a couple of special cases where we really do want to share
// field objects across grids, like the record editor
warnOnReusedFields:true,

//>	@method	ListGrid.initWidget()	(A)
// Initialize the canvas and call listGrid.setData().
//		@param	[all arguments]	(object)	objects with properties to override from default
//<
initWidget : function () {
    
    // call the setter on 'showRecordComponents()' - this disables the
    // drawAllMaxCells logic for the grid.
    if (this.showRecordComponents) {
        delete this.showRecordComponents;
        this.setShowRecordComponents(true);
    }
    
    // we want to avoid a fields array object getting re-used across multiple ListGrids.
    // This can happen if the developer does something like assigning a fields object to the
    // instance prototype.
    // To handle this we have another attribute 'defaultFields' which may be set on the instance
    // prototype and differs only from initializing with a fields block in that array will be
    // shallow cloned and assigned to this.fields rather than sharing the same object across
    // ListGrid instances
    
    if (this.fields == null && this.defaultFields != null) {
        this.fields = isc.shallowClone(this.defaultFields);
        //this.logWarn("fields picked up from this.defaultFields -- now set to:" + this.fields); 
    } else if (this.warnOnReusedFields && this.fields != null) {
        if (this.fields._initializedFieldsArray) {
            this.logWarn("ListGrid initialized with this.fields attribute set to an array " +
                "which is already being displayed in another ListGrid instance. To reuse " +
                "standard field configuration across multiple ListGrids, use " +
                "listGrid.defaultFields rather than assigning directly to listGrid.fields.");

        // Also catch the case where the developer re-uses individual fields within separate
        // arrays across grids
        } else {

            var dupedFields;
            if (isc.isAn.Array(this.fields)) {
                dupedFields = this.fields.findIndex("_initializedFieldObject", true) != -1;
            } else if (isc.isAn.Object(this.fields)) {
                for (var fieldName in this.fields) {
                    var field = this.fields[fieldName];
                    if (field && field._initializedFieldObject) {
                        dupedFields = true;
                        break;
                    }
                }
            }
            if (dupedFields) {
                 this.logWarn("ListGrid initialized with this.fields attribute set to an array " +
                "containing fields which are already being displayed in another ListGrid " +
                "instance. To reuse standard field configuration across multiple ListGrids, use " +
                "listGrid.defaultFields rather than assigning directly to listGrid.fields.");
            }
        }
    }
    
    if (this.canEditNew) this.listEndEditAction = this.rowEndEditAction = "next";

    if (this.alwaysShowEditors) {
        this.editByCell = false;
        
        // disable selection
        
        this.selectionType = "none";
        this.selectOnEdit = false;
        
        if (this.canGroup != false) {
            this.logInfo("grouping functionality is not supported when alwaysShowEditors is true." +
                    " Explicitly disabling this.canGroup", "inactiveEditorHTML");
            this.canGroup = false;
        }
        if (this.modalEditing) {
            this.logInfo("modalEditing is not supported when alwaysShowEditors is true." + 
                " Explicitly setting this.modalEditing to fales", "inactiveEditorHTML");
            this.modalEditing = false;
        }
        
        // enforce editEvent:"click" - this means that if the user clicks on a cell which
        // isn't showing an inactive editor (for example where showEditorPlaceholder is true),
        // we still start editing that row/cell
        this.editEvent = "click";
        
        // If canEdit is unset, and we have no fields explicitly marked as canEdit:true,
        // flip the default to true.
        // This gives us the most intuitive behavior - if the developer specifies per-field
        // editablility we'll respect it, otherwise we'll default to canEdit true
        if (!this.isEditable()) {
            this.logInfo("alwaysShowEditors has been set for this grid but canEdit is unset and " +
              "no fields are explicitly marked as editable. Defaulting this.canEdit to true. " +
              "Note that to avoid this override developers can explicitly specify canEdit " +
              "at the grid or field level", "inactiveEditorHTML");
            this.canEdit = true;
        }
        
    }

    // if we have variable record heights and virtualScrolling is unset, switch it on
    
    if (this.canExpandRecords ||
        (this.fixedRecordHeights == false && this.virtualScrolling == null)) 
    {
        // the _specifiedFixedRecordHeights flag is used by shouldShowAllColumns - we know
        // that even though a row may exceed its cell height this isn't due to any cell's content
        // in the row, so we don't need to render out every column to get correctly sized rows.
        if (this.fixedRecordHeights) this._specifiedFixedRecordHeights = this.fixedRecordHeights;
        this.fixedRecordHeights = false;
        this.virtualScrolling = true;
    }

    // disable canAddFormulaField if the required component isn't present
    if (this.canAddFormulaFields && isc.FormulaBuilder == null) {
        this.logInfo("Required modules for adding formula fields not present - setting " +
                    "canAddFormulaFields to false.");
        this.canAddFormulaFields = false;
    }
    if (this.canAddSummaryFields && isc.SummaryBuilder == null) {
        this.logInfo("Required modules for adding summary fields not present - setting " +
                    "canAddSummaryFields to false.");
        this.canAddSummaryFields = false;
    }
    
	// force loading rows to contain at least &nbsp; otherwise row height may be reported as less
	// than the actual height.
    if (this.loadingMessage == null || this.loadingMessage == isc.emptyString) 
        this.loadingMessage = "&nbsp;";
    
    // default our overflow to "visible" if autoFitData is set
    if (this.autoFitData != null) {
        this._specifiedOverflow = this.overflow;
        this.setOverflow("visible");
    }

    // default our groupStartOpen to "all" if canCollapseGroup is false
    if (this.canCollapseGroup == false) this.groupStartOpen = "all";

    // store off the initial sortDirection - we'll revert to this when unsorting
    this._baseSortDirection = this.sortDirection;

	// initialize the data object, setting it to an empty array if it hasn't been defined
	this.setData(this.data ? null : this.getDefaultData());

    // set up selectionType dynamic default
    this.setSelectionAppearance(this.selectionAppearance, true);
    
    this._setUpDragProperties(); 
},

getDefaultData : function () { return []; },

_setUpDragProperties : function () {
	// set up our specific drag-and-drop properties
	this.canDrag = !this.canDragSelectText &&
                    (this.canDrag || this.canDragRecordsOut || this.canReorderRecords || this.canDragSelect);
	this.canDrop = (this.canDrop || this.canDragRecordsOut || this.canReorderRecords);
	this.canAcceptDrop = (this.canAcceptDrop || this.canAcceptDroppedRecords || this.canReorderRecords);

},

getEmptyMessage : function () { 
    if (isc.ResultSet && isc.isA.ResultSet(this.data) && !this.data.lengthIsKnown()) {
        if (isc.Offline && isc.Offline.isOffline()) {
            return this.offlineMessage;
        }
        return this.loadingDataMessage == null ? "&nbsp;" : 
            this.loadingDataMessage.evalDynamicString(this, {
                loadingImage: this.imgHTML(isc.Canvas.loadingImageSrc, 
                                           isc.Canvas.loadingImageSize, 
                                           isc.Canvas.loadingImageSize)
            });
    }
    if (this.isOffline()) {
        return this.offlineMessage;
    }
    return this.emptyMessage == null ? "&nbsp;" : this.emptyMessage.evalDynamicString(this, {
        loadingImage: this.imgHTML(isc.Canvas.loadingImageSrc, 
                                   isc.Canvas.loadingImageSize, 
                                   isc.Canvas.loadingImageSize)
    });
},

isEmpty : function () {
    if (!this.data) return true;
    
    // treat having no fields as being empty so we don't attempt to write out and manipulate
    // an empty table 
    if (!this.fields || this.fields.length == 0) return true;
    
    
    if (isc.ResultSet && isc.isA.ResultSet(this.data)) {

        if (this.data.isPaged()) {
            if (!this.data.isEmpty()) return false;
            
            
            var editRows = this.getAllEditRows();
            if (editRows && editRows.length > 0) {
                for (var i = 0; i < editRows.length; i++) {
                    if (editRows[i] >= 0) return false;
                }
            }
            return true;
        } else {
            // If our length is not known we must be in the process of loading, so return the
            // loading message.
            if (this.data.lengthIsKnown()) return this.getTotalRows() <= 0;
            else return true;
        }
    } else {
        return (this.getTotalRows() <= 0);
    }
},

//>	@attr listGrid.preserveEditsOnSetData (boolean : null : IRWA)
// By default any edit values in an editable ListGrid are dropped when 'setData()' is called, 
// as previous edits are almost certainly obsoleted by the new data-set.
// This flag allows the developer to suppress this default behavior.
// @visibility internal
// @group data
//<
// Leave this internal for now - no known use cases for it, but seems like something that
// could come up.
//preserveEditsOnSetData : null,

//>	@method	listGrid.setData()    ([])
// Initialize the data object with the given array. Observes methods of the data object
// so that when the data changes, the listGrid will redraw automatically.
//      @visibility external
//		@group	data
//
//		@param	newData		(List of ListGridRecord)	data to show in the list
//<
setData : function (newData) {
	// if the current data and the newData are the same, bail
	//	(this also handles the case that both are null)
	if (this.data == newData) return;
	
    if (!this.preserveEditsOnSetData) this.discardAllEdits();
    
    // drop "lastHiliteRow" -no sense in hanging onto it
    this.clearLastHilite();
    
	// if we are currently pointing to data, stop observing it
	if (this.data) {
        this._ignoreData(this.data);
        // if the data was autoCreated, destroy it to clean up RS<->DS links
        if (this.data._autoCreated && isc.isA.Function(this.data.destroy))
            this.data.destroy();
    }

	// if newData was passed in, remember it
	if (newData) this.data = newData;
	
	// if data is not set, bail
	if (!this.data) return;

	// observe the data so we will update automatically when it changes
	this._observeData(this.data);
	
	// if we can regroup, do so. 
    
	this.regroup(true);
	
	// Don't redraw / refresh grid summary - this happens explicitly later.
	this.calculateRecordSummaries(null, true);
    
	// create a new selection if we don't have one or if we receive new data
    // Do this after grouping - if we created a new groupTree object we'll have already set up
    // an appropriate selection object
	if (!this.selection || (this.data != this.selection.data)) {
        this.createSelectionModel();
	}

    // Call this._remapEditRows() if we're hanging onto edit values
    // as we know they're now out of date.
    
	if (this.preserveEditsOnSetData) this._remapEditRows();
    // if this.alwaysShowEditors is set, and we have data, and we're not currently showing
    // editors, show them now.
    
    var fetching = isc.ResultSet && isc.isA.ResultSet(this.data) && !this.data.lengthIsKnown();
    if (!fetching && this._alwaysShowEditors() && !this._editorShowing) {
        this.startEditing(null,null,true,null,true);
    }

    // if data doesn't support setSort (resultSet and Array do), disable multiSort
    if (!this.data.setSort) 
        this.canMultiSort = false;
    else if (this.getDataSource() && this.canMultiSort != false) 
        this.canMultiSort = this.getDataSource().canMultiSort && this.canSort;
    
    var sortSpecifiers = this.getSort();
    if (sortSpecifiers) this.setSort(sortSpecifiers);

    
    if (isc.ResultSet && isc.isA.ResultSet(this.data) && 
        this.body && this.body.overflow == "visible") 
    {
        this.body.showAllRows = false;
    }
    
    // Force a recalculation of the grid summaries since this method doesn't fire 
    // dataChanged() which is normally what causes this to occur.
    if (this.summaryRow && this.showGridSummary) {
        this.summaryRow._recalculateSummaries();
    }

    // clear out the current expanded row count, along with the current expanded record,
    // if there is one, since the expansionComponents have been dropped, along with the
    // records they were in
    this._expandedRecordCount=0;
    if (!this.canExpandMultipleRecords && this._currentExpandedRecord) 
        delete this._currentExpandedRecord;

    if (this._lastStoredSelectedState) {
        this.setSelectedState(this._lastStoredSelectedState);
        delete this._lastStoredSelectedState;
    }
    
    // if any fields are marked as autoFitWidth, recalculate their sizes
    this.updateFieldWidthsForAutoFitValue("setData called.");
    
	// mark us as dirty so we'll be redrawn if necessary
    this._markBodyForRedraw("setData");
},


// Determine which field to expand when autoFitFieldWidths is true and the 
// fields don't take up the full set of space available.
// If +link{listGrid.autoFitExpandField} is explicitly set use it.
// Otherwise, basically we want to find the first field that will not look odd to expand
// Implementation:
// - if a field is not of type text, ignore (don't want to expand images, dates, booleans, numbers)
// - If field has showValueIconsOnly set, ignore (even though it has text content it shouldn't
//   expand beyond the value icons width)
// - field.length:
//      - if all fields have field.length specified, expand the largest one
//      - if no fields have this property set, just expand the first text field
//      - if *some* fields have field.length set - the length may indicate a small field
//        (say 2 characters) which we don't want to expand.
//        Expand fields with no specified length in preference to those with
//        a small specified length.
autoFitExpandLengthThreshold : 10,
getAutoFitExpandField : function () {
    if (!this.autoFitFieldsFillViewport) return null;
    
    if (this.autoFitExpandField != null) {
        var field = this.getField(this.autoFitExpandField);
        // We don't support auto-expanding a frozen field at this time
        
        if (field != null && this.fields && this.fields.contains(field) && 
            (!this.frozenFields || !this.frozenFields.contains(field)))
        {
            return field;
        }
    }
    var fields = [], lengthFields = [];
    
    if (this.fields) {
        for (var i = 0; i < this.fields.length; i++) {
            var field = this.fields[i];
            if (!field.showValueIconOnly && 
               (field.type == "text" || field.type == null)) 
            {
                if (field.frozen) continue;
                fields.add(field);
                if (fields[i] != null && fields[i].length != null) {
                   lengthFields.add(fields[i]);
                }
            
            }
        }
    }
    
    if (lengthFields.length > 0) {
        lengthFields.sortByProperty("length", Array.DESCENDING);
        if (lengthFields.last().length >= this.autoFitExpandLengthThreshold ||
            lengthFields.length == fields.length) 
        {
            return lengthFields[0];
        }
    }
    if (fields.length > 0) {
        var i = 0;
            field = fields[i]
        while (field != null && field.length != null &&
                    field.length < this.autoFitExpandLengthThreshold)
        {
            i++;
            field = fields[i];
        }
        return field;
    }

    // Note that this could still return null - no text fields etc.
    return null;
},

// updateFieldWidthsForAutoFitValue()
// If this.autoFitWidthApproach means we size to
// column data and any fields are marked as autoFitWidth:true,
// this method will set the _fieldWidthsDirty flag on the gridRenderer, causing
// _updateFieldWidths() to be called on the next body redraw,
// Calling code should call this before causing a body redraw.
updateFieldWidthsForAutoFitValue : function (reason) {
    if (!this.body || this.body._fieldWidthsDirty) {
        return;
    }
    
    var fields = this.fields || [];
    for (var i = 0; i < fields.length; i++) {
        var shouldAutoFit = this.shouldAutoFitField(fields[i]);
        if (!shouldAutoFit) continue;
        var approach = this.getAutoFitWidthApproach(fields[i]);
        
        if (approach == "value" || approach == "both") {
            this.fields._appliedInitialAutoFitWidth = false;
            this.body._fieldWidthsDirty = "Updating field widths for field auto-fit" + 
                                      (reason ? (":" + reason) : ".");
            break;
        }
    }
},


invalidateCache : function () {
    // make sure that header checkbox is unchecked after refreshing cache
    if (this.getCheckboxFieldPosition() != -1) {
        this.deselectAllRecords();
        this._setCheckboxHeaderState(false);
    }

    // if we have a stored drawArea, delete it now so that the 
    // redraw recalculates it and reapplies recordComponents
    if (this.body && this.body._oldDrawArea) delete this.body._oldDrawArea;

    return this.Super("invalidateCache", arguments);
},

// use this rather than field.canSort to handle canSortClientOnly fields
_canSort : function (field) {

    var canSort = (field.canSort == false) ? false : true;
    if (!field.canSortClientOnly || isc.isAn.Array(this.data)) return canSort ;

    if (isc.isA.ResultSet(this.data)) {
        if (!this.data.lengthIsKnown() || !this.data.canSortOnClient()) {
            return false;
        } else return canSort;
    } 
    return canSort;
},

// helper method to get an appropriate sortDirection for a field
_getFieldSortDirection : function (field) {
    var direction;
    
    var fieldDir = field.sortDirection;
    if (fieldDir != null) {
        direction = Array.shouldSortAscending(fieldDir) ? "ascending" : "descending";
    } else if (this.sortDirection != null) {
        direction = Array.shouldSortAscending(this.sortDirection) ? "ascending" : "descending";
    } else {
        direction = "ascending";
    }
    return direction;
},

// Override createSelectionModel, from DataBoundComponent, to set the body's selection object
// with our selection object. Our body GridRenderer will then observe selection.setSelected.
createSelectionModel : function () {

    
    this.invokeSuper(isc.ListGrid, "createSelectionModel", arguments);
    if (isc.isA.Canvas(this.body)) {
        this.body.setSelection(this.selection);
        if (this.frozenBody) this.frozenBody.setSelection(this.selection);
    }
    
},

destroySelectionModel : function () {
    if (this.body) this.body.clearSelection();
    if (this.frozenBody) this.frozenBody.clearSelection();
    return this.Super("destroySelectionModel", arguments);
},

//>	@method	listGrid.setSelectionType()	[A]
// Changes selectionType on the fly.
// @param	selectionType (SelectionStyle)	New selection style. 
// @visibility external
//<
setSelectionType : function (selectionType, ignoreCheckbox) {
    // NOTE: this is sufficient because the Selection object dynamically inspects this property
    // on it's grid
    this.selectionType = selectionType;
    if (this.body) this.body.selectionType = selectionType;
},

//>	@method	listGrid.setSelectionAppearance()	
// Changes selectionAppearance on the fly.
// @param	selectionAppearance (String)	new selection appearance
// @visibility external
//<
setSelectionAppearance : function (selectionAppearance, isInit) {
    if (this.selectionAppearance == selectionAppearance && !isInit) return;
    
    this.selectionAppearance = selectionAppearance;

    // at initialization time, if selectionType wasn't explicitly set, default selectionType to
    // "simple" for "checkbox" appearance, otherwise "multiple"
    if (isInit && this.selectionType == null) {
        this.selectionType = (selectionAppearance == "checkbox" ? isc.Selection.SIMPLE : 
                                                                  isc.Selection.MULTIPLE);
    }
    
    // If this.completeFields is set (IE setFields has run already), we need to either add or
    // remove the checkboxField
    // Call setFields() with a duplicate of the current set of fields, less the
    // checkboxField (if there was one)
    // This will create a new checkboxField if necessary otherwise just clear the existing one
    if (this.completeFields != null) {
        var completeFields = [];
        for (var i = 0; i < this.completeFields.length; i++) {
            var field = this.completeFields[i];
            if (this.isCheckboxField(field)) continue;
            completeFields.add(field);
        }
        this.setFields(completeFields);
    }
},

//> @method listGrid.setBodyOverflow()  ([A])
// Update the +link{listGrid.bodyOverflow, bodyOverflow} for this listGrid.
// @param overflow (Overflow) new overflow setting for the body
// @visibility external
//<
setBodyOverflow : function (newOverflow) {
    this.bodyOverflow = newOverflow;
    if (this.body) this.body.setOverflow(this.bodyOverflow);
},

//> @method listGrid.setBodyStyleName()
// Update the +link{listGrid.bodyStyleName,bodyStyleName} for this listGrid.
// @param styleName (CSSStyleName) new body style name
// @visibility external
//<
setBodyStyleName : function (styleName) {
    this.bodyStyleName = styleName;
    if (this.body && (!this.alternateBodyStyleName || !this.alternateRecordStyles)) {
        this.body.setStyleName(styleName);
    }
},

//> @method listGrid.setAlternateBodyStyleName()
// Update the +link{listGrid.alternateBodyStyleName,alternateBodyStyleName} for this listGrid.
// @param styleName (CSSStyleName) new body style name when showing alternateRecordStyles
// @visibility external
//<
setAlternateBodyStyleName : function (styleName) {
    this.alternateBodyStyleName = styleName;
    if (this.body && this.alternateRecordStyles) {
        // if passed 'null', reset to this.bodyStyleName
        this.body.setStyleName(styleName || this.bodyStyleName);
    }
},

//> @method listGrid.setAlternateRecordStyles()
// Setter for +link{listGrid.alternateRecordStyles}
// @param alternateStyles (boolean) New value for <code>this.alternateRecordStyles</code>
// @visibility external
//<
setAlternateRecordStyles : function (alternateStyles) {
    if (this.alternateRecordStyles == alternateStyles) return;
    this.alternateRecordStyles = alternateStyles;
    
    if (this.body && (this.alternateBodyStyleName != null)) {
        if (alternateStyles) this.body.setStyleName(this.alternateBodyStyleName);
        else this.body.setStyleName(this.bodyStyleName);
    }
    
},

// Override hasInherentHeight / width: If we're autoFitting to our data, advertise inherent height
// This means that a layout will not expand us to fit the available space.

hasInherentHeight : function (a,b,c,d) {
    if (this.inherentHeight != null) return this.inherentHeight;
    if (this.autoFitData == isc.Canvas.VERTICAL || this.autoFitData == isc.Canvas.BOTH) {
        return true;
    }
    return this.invokeSuper(isc.ListGrid, "hasInherentHeight", a,b,c,d);
},

hasInherentWidth : function (a,b,c,d) {
    if (this.inherentWidth != null) return this.inherentWidth;
    if (this.autoFitData == isc.Canvas.HORIZONTAL || this.autoFitData == isc.Canvas.BOTH) {
        return true;
    }
    return this.invokeSuper(isc.ListGrid, "hasInherentWidth", a,b,c,d);
},

//> @method listGrid.setAutoFitData()
// Setter for +link{listGrid.autoFitData}.
// @param autoFitData (string) One of <code>"vertical"</code>, <code>"horizontal"</code>
//  or <code>"both"</code>. To disable auto fit behavior, pass in <code>null</code>.
// @group autoFitData
// @visibility external
//<
setAutoFitData : function (autoFitData) {
    this.autoFitData = autoFitData;
    
    if (this._autoDerivedCanFreeze && (autoFitData == "both" || autoFitData == "horizontal"
                                 
                                       || autoFitData == "vertical")) {
        delete this._autoDerivedCanFreeze;
        delete this.canFreezeFields;
    }
    if (autoFitData == null && this._specifiedOverflow) {
        this.setOverflow(this._specifiedOverflow);
    } else if (this.overflow != "visible") {
        this._specifiedOverflow = this.overflow;
        this.setOverflow("visible");
    }
    if (this.body) {
        this.body.autoFitData = this.autoFitData;
        this.body.adjustOverflow();
    }
},

//> @method listGrid.setAutoFitExtraRecords()
// Setter for +link{listGrid.autoFitExtraRecords}.
// @param extraRecords (integer) Number of extra rows beyond the data-size we'll expand to 
// accommodate if +link{listGrid.autoFitData,auto fit} is enabled vertically.
// @group autoFitData
// @visibility external
//<
setAutoFitExtraRecords : function (extraRecords) {
    this.autoFitExtraRecords = extraRecords;
    if (this.body) {
        this.body.autoFitExtraRecords = extraRecords;
        this.body.adjustOverflow();
    }
},


//> @method listGrid.setAutoFitMaxRecords()
// Setter for +link{listGrid.autoFitMaxRecords}.
// @param maxRecords (integer) Maximum number of rows we'll expand to accommodate if 
// +link{listGrid.autoFitData,auto fit} is enabled vertically.
// @group autoFitData
// @visibility external
//<
setAutoFitMaxRecords : function (maxRecords) {
    this.autoFitMaxRecords = maxRecords;
    if (this.body) {
        this.body.autoFitMaxRecords = maxRecords;
        this.body.adjustOverflow();
    }
},

//> @method listGrid.setAutoFitMaxHeight()
// Setter for +link{listGrid.autoFitMaxHeight}.
// @param height (integer) Maximum height in px we'll expand to accommodate if 
// +link{listGrid.autoFitData,auto fit} is enabled vertically.
// @group autoFitData
// @visibility external
//<
setAutoFitMaxHeight : function (height) {
    this.autoFitMaxHeight = height;
    if (this.body) {
        this.body.adjustOverflow();
    }
},
getAutoFitMaxBodyHeight : function () {
    if (this.autoFitMaxHeight == null) return null;
    var offset = this.getVBorderPad();
    if (this.showHeader) offset += this.headerHeight;
    if (this.showFilterEditor) offset += this.filterEditorHeight;
    return this.autoFitMaxHeight - offset;
    
},

// When auto-fitting vertically, specified height for the grid acts as a minimum
getAutoFitMinBodyHeight : function () {
    var minHeight = this.getHeight(),
        offset = this.getVBorderPad();
    if (this.showHeader) offset += this.headerHeight;
    if (this.showFilterEditor) offset += this.filterEditorHeight;
    return (minHeight - offset);
},

//> @method listGrid.setAutoFitMaxColumns()
// Setter for +link{listGrid.autoFitMaxColumns}.
// @param maxColumns (integer) Maximum number of fields we'll expand to accommodate if 
// +link{listGrid.autoFitData,auto fit} is enabled horizontally.
// @group autoFitData
// @visibility external
//<
setAutoFitMaxColumns : function (maxColumns) {
    this.autoFitMaxColumns = maxColumns;
    if (this.body) {
        this.body.autoFitMaxColumns = maxColumns;
        this.body.adjustOverflow();
    }
},

//> @method listGrid.setAutoFitMaxWidth()
// Setter for +link{listGrid.autoFitMaxWidth}.
// @param width (integer) Width in px we'll expand to accommodate if 
// +link{listGrid.autoFitData,auto fit} is enabled horizontally.
// @group autoFitData
// @visibility external
//<
setAutoFitMaxWidth : function (width) {
    this.autoFitMaxWidth = width;
    if (this.body) {
        this.body.autoFitMaxWidth = width;
        this.body.adjustOverflow();
    }
},

// --------------------------------
// AutoFitFields

//> @method listGrid.autoFitField()
// Programmatically cause a field to auto-fit horizontally to it's contents or title.
// <P>
// Does not establish permanent auto-fitting - use +link{listGrid.setAutoFitWidth()} to do so.
// @param fieldName (string)
// @return (integer) new width in pixels
//
// @group autoFitFields
// @visibility external
//<
autoFitField : function (fieldName, scrollIntoView) {
    var field = this.getField(fieldName),
        fieldNum = this.getFieldNum(field);
        
    // avoid attempting to autofit hidden fields, or fields where
    // autoFitWidth is already set (should just happen dynamically!)
    if (field == null || fieldNum == -1) return;
    
    var width = this.getFieldAutoFitWidth(field);
    if (width == null || width == this._fieldWidths[fieldNum]) return;
    
    // resize the field (as if the user drag-resized to the auto-fit size)
    var dontStoreWidth = this.shouldAutoFitField(field);
    this.resizeField(fieldName, width, !dontStoreWidth);
    if (dontStoreWidth) field._calculatedAutoFitWidth = this._fieldWidths[fieldNum];
    
    if (scrollIntoView) this.scrollColumnIntoView(fieldNum, false);
    return width;
},

//> @method listGrid.autoFitFields()
// Perform a one-time horizontal auto-fit of the fields passed. Fields will be sized
// to match their contents or title (as specified in +link{listGrid.autoFitWidthApproach})
// Does not establish permanent auto-fitting - use +link{listGrid.setAutoFitWidth()} to do so.
// @param [fields] (Array of ListGridField) Array of fields to auto fit. If this parameter
//  is not passed, autoFitting will occur on all visible fields.
//
// @group autoFitFields
// @visibility external
//<

autoFitFields : function (fields) {
    if (fields == null) fields = this.fields;
    for (var i = 0; i < fields.length; i++) {
        this.autoFitField(fields[i])
    }
},

shouldAutoFitField : function (field) {
    if (field.autoFitWidth != null) return field.autoFitWidth;
    return this.autoFitFieldWidths;
},


getFieldAutoFitWidth : function (field, minWidth) {
    if (this.body == null) return;
    var approach = this.getAutoFitWidthApproach(field),
        
        checkHeader = approach != "value",
        checkBody = approach != "title",
        
        headerWidth, bodyColWidth,
        
        width;
    var colNum = this.getColNum(field);
    if (checkHeader) {
        
        var header = this.getFieldHeaderButton(colNum);
            
        if (header != null) {
            var origWidth = header.getWidth(), origOF = header.getOverflow();

            header.setWidth(minWidth || this.minFieldWidth || 1);
            header.setOverflow("visible");
            
            
            if (header.label && header.label.isDirty()) header.label.redraw();
            
            header.parentElement.reflow();
            headerWidth = header.getVisibleWidth();
            // reset to original width
            header.setWidth(origWidth);
            
            width = headerWidth;
        }
    }
    
    if (checkBody) {
        // getAutoFitValueWidths can take an array of fields to check. 
        // Returns a sparse array of widths, indexed by colNum
        var bodyColWidthArr = this.getAutoFitValueWidths([this.fields[colNum]]),
            bodyColWidth = bodyColWidthArr ? bodyColWidthArr[colNum] : null;
        
        if (bodyColWidth != null && 
            (width == null || width < bodyColWidth)) width = bodyColWidth;
    }
    
    
    if (width != null) {
        if (minWidth == null || minWidth < this.minFieldWidth) minWidth = this.minFieldWidth;
        if (minWidth != null && width < minWidth) width = minWidth;
    }
    return width;    
},

//> @method listGrid.setAutoFitWidth()
// Setter for +link{field.autoFitWidth}. Enables or disables dynamic autoFitWidth behavior
// on the specified field. Note if the field is currently autoFitWidth:true, and this method is
// disabling autoFit, the field will not be resized by default - if you wish to resize to an
// explicit width, use +link{listGrid.resizeField()}.
//
// @param fieldName (string) field to auto-fit
// @param autoFit (boolean) Should autoFitWidth be enabled or disabled? 
// @group autoFitFields
// @visibility external
//<
setAutoFitWidth : function (fieldName, autoFit) {
    var field = this.getField(fieldName);
    if (field == null) return;
    if (field.autoFitWidth == autoFit) return;
    
    field.autoFitWidth = autoFit;

    if (autoFit) this.autoFitField(field);
    // If we're turning off auto-fit, don't bother to resize
},

//> @method listGrid.setAutoFitFieldWidths()
// Setter for +link{listGrid.autoFitFieldWidths}. Modifies the default auto-fit-width behavior
// for fields in this grid. Note that this may be overridden at the field level via 
// +link{listGridField.autoFitWidth}.
// @param autoFit (boolean) New value for autoFitFieldWidths
// @param [dontResetWidths] (boolean) If autoFitFieldWidths was true, and is being set to false, 
//  should fields be resized to their originally specified size? 
//  Pass in this parameter to suppress this behavior.
// @visibility external
//<
setAutoFitFieldWidths : function (autoFit, dontResetWidths) {
    if (autoFit == this.autoFitFieldWidths) return;
    this.autoFitFieldWidths = autoFit
    if (autoFit) {
        this._updateFieldWidths("autoFitFieldWidths enabled");
    } else if (!dontResetWidths) {
        // If we're showing a header, we use it to handle converting specified
        // field widths into real sizes - running updateHeader will rebuild it and
        // perform this initial calculation for us.
        if (this.showHeader && this.headerHeight > 0) this.updateHeader();
        // Clear the flag indicating we've run through auto-fit logic and re-run
        // _updateFieldWidths() to revert the fields to specified sizes.
        this.fields._appliedInitialAutoFitWidth = false;
        this._updateFieldWidths("autoFitFieldWidths disabled");
    }
},

//> @method listGrid.setAutoFitWidthApproach()
// Setter for the +link{listGrid.autoFitWidthApproach}.
// @param approach (AutoFitWidthApproach) new AutoFitWidth approach
// @visibility external
//<
setAutoFitWidthApproach : function (approach) {
    if (this.autoFitWidthApproach == approach) return;
    this.autoFitWidthApproach = approach;

    // If we're showing a header, we use it to handle converting specified
    // field widths into real sizes - running updateHeader will rebuild it and
    // perform this initial calculation for us.
    
    if (this.showHeader && this.headerHeight > 0) this.updateHeader();
    // Clear the flag indicating we've run through auto-fit logic and re-run
    // _updateFieldWidths() to revert the fields to specified sizes.
    this.fields._appliedInitialAutoFitWidth = false;
    this._updateFieldWidths("autoFitFieldWidthApproach changed");
},

// mark the body for redraw, or if the body doesn't exist, the widget as a whole
_markBodyForRedraw : function (reason) {
	if (this.bodies) {
        this.bodies.map("markForRedraw", reason);
    } else {
        this.markForRedraw(reason);
    }        
},

redraw : function (a, b, c, d) {
    
    if (this.body) {
        
        if (this.body._scrollbarChangeDuringAnimation) {
            this._updateFieldWidths("scrollbar change during animation");
            delete this.body._scrollbarChangeDuringAnimation;
        }
    }

    this.invokeSuper(isc.ListGrid, "redraw", a, b, c, d);
    
},

//>	@method listGrid._observeData() (A)
//		observe methods on the data so we redraw automatically when data changes
//		called automatically by setData
//	@param	data	(object)		new data to be observed
//<
_observeData : function (data) {
	// redraw if the data changed
    
    if (!this.isObserving(data, "dataChanged")) {
        this.observe(data, "dataChanged",
                        "observer.dataChanged(" + 
                            (isc.ResultSet && isc.isA.ResultSet(data) ?
                            "arguments[0],arguments[1],arguments[2],arguments[3],arguments[4])" : 
                            ")")
        );
    }
 
    if (!this.isObserving(data, "dataArrived")) {
        if (isc.ResultSet && isc.isA.ResultSet(data)) {
            this.observe(data, "dataArrived", "observer._dataArrived(arguments[0],arguments[1])");
        } else if (isc.ResultTree && isc.isA.ResultTree(data)) {
            this.observe(data, "dataArrived", "observer._dataArrived(arguments[0])");
        }
    }

	// Note - we must check for data being a tree as if it is not defined, the inherited 
    // ListGrid.init() code will set it to an empty array, in which case this observation will 
    // fail.
    if (isc.isA.Tree(data)) {
        // update view in response to folders opening / closing
    	this.observe(data, "changeDataVisibility", "observer._folderToggleObservation()");
    } 
},
//> @method listGrid.groupTreeChanged()
// Callback fired from group tree +link{listGrid.groupTree} dataChanged().
// <p>
// Handles remapping edit rows and forcing a redraw if necessary.
// @group grouping
//< 
// The groupTree may be a ResultTree or a Tree object.
// When a listGrid is grouped we still observe dataChanged on the underlying data object and react
// to it by updating or rebuilding the groupTree as required, and marking for redraw etc.
// Therefore if this method is fired from standard dataChanged() we typically need to take no
// action.
// Note that we explicitly disable databound cache-synch for the ResultTree and instead manage
// updating the ResultTree cache directly as part of ListGrid.dataChanged. This is appropriate since
// The ResultTree code for cache sync is organized around node ids and parent ids whereas the
// groupTree is a dynamic grouping based on records have the same values for a field.
// 
// This will actually fire in response to listGrid sort or direct manipulation of the groupTree
//
// Note that this method is only fired when an existing groupTree changes - not when regroup()
// is run, creating a new groupTree.
_$dataChanged:"dataChanged",
groupTreeChanged : function () {
    
    // If the groupTree was updated from underlying data change, no need to
    // redraw etc (already handled in dataChanged())
    if (this._handlingDataChanged) return;
    
    if (!this._savingEdits && !this.suppressEditRowRemap) this._remapEditRows();
    var lastRow = this.getTotalRows()-1;
    if (this.body) {
        
        if (this.body.lastOverRow > lastRow) delete this.body.lastOverRow;
        if (this.body.lastMouseOverRow > lastRow) delete this.body.lastMouseOverRow;
        if (this.body._lastHiliteRow > lastRow) delete this.body._lastHiliteRow;
    }
    if (this._lastRecordClicked > lastRow) delete this._lastRecordClicked;
    
    if (this.hilites) this.applyHilites();
    if (!this._suppressRedrawOnDataChanged) this._markBodyForRedraw(this._$dataChanged);
    
},
//>	@method listGrid._observeGroupData() (A)
//      observe methods on the group tree object, so that changes to the group layout
//      can be detected
//	@param	data	(object)		new group tree to be observed
//  @visibility internal
//<
_observeGroupData : function (data) {
	// redraw if the data changed 
    this.observe(data, "dataChanged", "observer.groupTreeChanged()");
    this.observe(data, "changeDataVisibility", "observer._folderToggleObservation()");
},

// METHODS MOVED FROM TREEGRID
// The following methods were moved from treegrid to allow the listgrid to support the tree
// as a data model for grouping. They will continue to be doc'd on treegrid for now

// Helper method - fired when folders open/close within the tree
_folderToggleObservation : function () {
    //>Animation
    // During animated folder open/close we suppress redraw in response to the folder toggling
    
    if (this._suppressFolderToggleRedraw) {        
        this._remapEditRows();
        return;
    }
    // Cut short any currently running animated folder open / close 
    // Just call finishAnimation - this will no op if no animation is running
    
    if (this.body) this.body.finishRowAnimation();
    //<Animation
    
    // Length changes so we need to remap edit rows.
    this._remapEditRows();

    // redraw to display the updated folder
    this._markBodyForRedraw('folderToggled');
},

//> @method treeGrid.toggleFolder()   ([])
//          Opens the folder specified by node if it's closed, and closes it if it's open.
//          TreeGrid will redraw if there's a change in the folder's open/closed state.
//
//      @visibility external
//      @param  node        (TreeNode)      node to toggle
//<
toggleFolder : function (node) {
    if (this.data.isOpen(node)) {
        this.closeFolder(node);
    } else {
        this.openFolder(node);
        
        if (this.frozenBody) this.frozenBody.markForRedraw();
    }
},

//> @method treeGrid.openFolder() ([A])
// Opens a folder.
// <p>
// Executed when a folder node receives a 'doubleClick' event. This handler must be
// specified as a function, whose single parameter is a reference to the relevant folder
// node in the tree's data.<br>
// See the ListGrid Widget Class for inherited recordClick and recordDoubleClick events.
//
// @param   node        (TreeNode)      node to open
// @see closeFolder()
// @see folderOpened()
// @see class:ListGrid
// @visibility external
//<
openFolder : function (node) {
    // CALLBACK API:  available variables:  "node"
    // Convert a string callback to a function
    if (this.folderOpened != null) {
        this.convertToMethod("folderOpened");
        if (this.folderOpened(node) == false) return false;
    }
    
    
    if (this.animateFolders) {
        this.animateOpen(node);
    } else {
        this.data.openFolder(node);
    }

},   

//> @method treeGrid.animateOpen()
// Animates a folder opening to display its children (which grow into view).
// Automatically triggered from <code>treeGrid.folderOpen()</code> if 
// <code>this.animateFolders</code> is true.
// @group animation
// @param folder (node) node to open
// @visibility animation_advanced
//<
animateOpen : function (folder) {
    var data = this.data;
    if (data.isOpen(folder)) return;

    // Open the data, but don't redraw with the new data visible (we'll handle redrawing 
    // when the animation completes).
    this._suppressFolderToggleRedraw = true;
    data.openFolder(folder);
    delete this._suppressFolderToggleRedraw;

    // parent may be null if we're looking at the root node
    var parent = data.getParent(folder);
    if (parent && !data.isOpen(parent)) return;
     
    if (data.getLoadState(folder) != isc.Tree.LOADED) {
        //this.logWarn("animation for LOD folder");
        // wait for dataChanged() to fire
        this._pendingFolderAnim = folder;
        return;
    }
   
    this._startFolderAnimation(folder);
},

//> @method treeGrid.closeFolder()
// Closes a folder.
//
// @param   node        (TreeNode)      node to close
// @see openFolder()
// @see folderClosed()
// @visibility external
//<
closeFolder : function (node) {
    // CALLBACK API:  available variables:  "node"
    // Convert a string callback to a function
    if (this.folderClosed != null) {
        this.convertToMethod("folderClosed");
        if (this.folderClosed(node) == false) return false;
    }

    // cancel editing of any nodes under this one
    if (this.getEditRow() != null) {
        var editRecord = this.getRecord(this.getEditRow());
        if (this.data.isDescendantOf(editRecord, node)) this.endEditing();
    }
    // now tell the data to close the folder
    if (this.shouldAnimateFolder(node)) 
        this.animateClose(node);
    else 
        this.data.closeFolder(node);
},

//> @method treeGrid.animateClose()
// Animates a folder closing to hide its children (which shrink out of view).
// Automatically triggered from <code>treeGrid.folderOpen()</code> if 
// <code>this.animateFolders</code> is true.
// @param folder (node) node to open
// @group animation
// @visibility animation_advanced
//<
animateClose : function (folder) {
    if (!this.data.isOpen(folder)) return;

    var parent = this.data.getParent(folder);
    if (parent && !this.data.isOpen(parent)) {
        return this.closeFolder(folder);
    }
    
    var data = this.data,
        folderIndex = data.indexOf(folder),
        numChildren = data.getOpenList(folder).getLength()-1;

    
    
    this.startRowAnimation( false, 
                            folderIndex+1, 
                            folderIndex + numChildren + 1, 
                            {target:this, methodName:"redraw"}, 
                            this.animateFolderSpeed, 
                            this.animateFolderTime,
                            this.animateFolderEffect,
                            true
                          );

    this._suppressFolderToggleRedraw = true;
    this.data.closeFolder(folder);
    delete this._suppressFolderToggleRedraw
    
    if (this.body && this.body._delayedRowAnimation != null) {
        this.body._openFolder = folder;
    }

},

_startFolderAnimation : function (folder) {
    // At this point we know we have all the children for the folder loaded - verify
    // that we actually should animate the folder into view - if we have too many children
    // we may not want to -- in this case just redraw.
    if (!this.shouldAnimateFolder(folder)) {
        this.markForRedraw();
        return;
    }
    var data = this.data,
        folderIndex = data.indexOf(folder),
        numChildren = data.getOpenList(folder).getLength()-1;

    // don't try to animate empty folders
    if (folderIndex < 0 || numChildren <= 0) return;
    
    this.startRowAnimation( true, 
                            folderIndex+1, 
                            (folderIndex + numChildren+1), 
                            {target:(this.bodyLayout || this.body), methodName:"redraw"}, 
                            this.animateFolderSpeed, 
                            this.animateFolderTime,
                            this.animateFolderEffect,
                            true
                          );
},

// Used to store open folder state in the groupTree
// (Also used by TreeGrid.getOpenState())

_addNodeToOpenState : function (tree, node, openState, isGroupTree) {
    if (!tree.isOpen(node) || !tree.isLoaded(node)) return false;
    var children = tree.getFolders(node),
        hasOpenChildren = false;
    if (children != null) {
        for (var i = 0; i < children.length; i++) {
            hasOpenChildren = this._addNodeToOpenState(tree, children[i], openState, isGroupTree) 
                              || hasOpenChildren;
        }
    }
    if (isGroupTree) {
        var folderInfo = {};
        folderInfo[node.groupName] = node.groupValue;
        openState.add(folderInfo);
    } else {
        openState[openState.length] = tree.getPath(node);
    }
    return true;
},

// END METHODS MOVED FROM TREEGRID


dataChanged : function (type, originalRecord, rowNum, updateData, filterChanged) {

    if (isc._traceMarkers) arguments.__this = this;
    
    // set a flag so we know we're handing dataChanged
    // This prevents us from causing unnecessary additional redraws from dataChanged on the
    // groupTree if we're currently grouped by any field(s) 
    this._handlingDataChanged = true;
   
    // DataChanged fires in some cases where we don't want to reset autoFieldWidths
    // For example, scrolling through a paged resultset where the columns resizing on scroll
    // would be ugly.
    // Use ResultSet parameters to test for cases to react to:
    // - crud operations which will effect the data displayed
    // - filter changing (including invalidateCache calls)
    // Note that we have equivalent logic inline in removeData, cellChanged to handle
    // non-databound grids' data being changed through standard grid APIs
    var resetAutoFitWidths = filterChanged || type == "add" || type == "remove";
    if (!resetAutoFitWidths && type == "replace") {
        if (originalRecord == null || rowNum == null) resetAutoFitWidths = true;
        else {
            var updatedRecord = currData.get(rowNum);
            if (updatedRecord == null) resetAutoFitWidths = true;
            else {
                for (var i = 0; i < this.fields.length; i++) {
                    if (this.shouldAutoFitField(this.fields[i])) {
                        var fieldName = this.getFieldName(this.fields[i]);
                        if (updatedRecord[fieldName] != updateData[fieldName]) {
                            resetAutoFitWidths = true;
                            break;
                        }
                    }
                }
            }
        }
    }    
    
    // If we have any summary fields where we need to store summaries on the records, update
    // them now
    // Suppress redraw and grid summary update -- we explicitly handle these below.
    this.calculateRecordSummaries(null, false);

    // if a change was made to the groupBy field of a record, regroup
    var groupByFields = this.getGroupByFields();
    if (groupByFields != null && !this._markForRegroup) {

        // fully regroup for add/remove, or for an update where dataChanged is not passed the
        // originalRecord to figure out if the groupField was changed
        if (type == "add" || type == "remove" || type == "replace" ||
                (type == "update" && (originalRecord == null || rowNum == null)))
        {
            this._markForRegroup = true;
        // otherwise for other updates see if we need to regroup
        } else if (type == "update") { 
            var currData = this.data;
            if (this.data.isGroupedOutput && this.originalData) currData = this.originalData;
            
            var updatedRecord = currData.get(rowNum);
            
            // regroup fully if updatedRecord does not exist (was deleted)
            if (updatedRecord == null) this._markForRegroup = true;

            // if primary keys differ, the record was deleted via filtering. regroup
            if (!this._markForRegroup) {
                var pks = (this.dataSource != null 
                        ? this.getDataSource().getPrimaryKeyFieldNames()
                        : []);
                for (var i=0; i<pks.length; i++) {
                    if (originalRecord[pks[i]] != updatedRecord[pks[i]]) {
                        this._markForRegroup = true;
                        break;
                    }
                }
            }

            // regroup if group field values differ
            if (!this._markForRegroup) for (var i = 0; i < groupByFields.length; i++) {
                var undef, fieldName = groupByFields[i];
                if (originalRecord[fieldName] !== undef && !this.fieldValuesAreEqual(
                        fieldName, originalRecord[fieldName], updatedRecord[fieldName])) {
                    // XXX incrementalRegroup can handle this case, but more testing is 
                    // necessary. change this post 7.0.
                    // Note, originalRecord must be changed to what node is below:
                    // this.data.find(pk) etc
                    // this._incrementalRegroup(updatedRecord, originalRecord, rowNum, 
                    //      updateData);
                    this._markForRegroup = true;
                    break;
                }
            }
            
            if (!this._markForRegroup) {
                var pk = this.getDataSource().getPrimaryKeyFieldNames()[0];
                var node = this.data.find(pk, updatedRecord[pk]);
                
                // apply all modified fields to the node.
                isc.addProperties(node, updatedRecord);
                
            }
        }
    }
    
    
    if (this._markForRegroup && !this._savingEdits) {
        this._markForRegroup = false;
        this._lastStoredSelectedState = this.getSelectedState(true);
        this.regroup();
        if (this._lastStoredSelectedState) {
            this.setSelectedState(this._lastStoredSelectedState);
            delete this._lastStoredSelectedState;
        }
    }
    
    //>Animation
    // Call finishRowAnimation - will kill any show/hide row animations.
    // These animations assume the data remains constant for the duration
    // of the animation.
    // (No-ops if appropriate)
    if (this.body) this.body.finishRowAnimation();
    //<Animation
    
    // Call _remapEditRows() to ensure that editValues are associated with the (possibly
    // modified) rowNumbers using pointers between record primary key and edit values 
    
    if (!this._savingEdits && !this.suppressEditRowRemap) this._remapEditRows();

    // re-associate embeddedComponents with records which were not previously present in the 
    // cache but are now - set grid._shouldRetainEmbeddedComponents to false to have components
    // removed when their associated records are no longer in the cache.
    this._remapEmbeddedComponents();

    // if this.alwaysShowEditors is set, and we have data, and we're not currently showing
    // editors, show them now.
    // This handles us getting new data (from a fetch for example)
    if (this._alwaysShowEditors() && !this._editorShowing) {
        this.startEditing(null,null,true,null,true);
    }

    
    var lastRow = this.getTotalRows()-1;
    if (this.body) {
        if (this.body.lastOverRow > lastRow) delete this.body.lastOverRow;
        if (this.body.lastMouseOverRow > lastRow) delete this.body.lastMouseOverRow;
        if (this.body._lastHiliteRow > lastRow) delete this.body._lastHiliteRow;
    }
    if (this._lastRecordClicked > lastRow) delete this._lastRecordClicked;
    
    if (this.hilites) this.applyHilites();
    
    
    if (!this._suppressRedrawOnDataChanged) {
        // recalculate autoFitWidth field widths to fit the new data
        if (resetAutoFitWidths) this.updateFieldWidthsForAutoFitValue(this._$dataChanged);
        
        this._markBodyForRedraw(this._$dataChanged);
        
        // recalculate grid summaries
        
        if (this.summaryRow && this.showGridSummary) this.summaryRow._recalculateSummaries();
        
    }
 
    // restore the selected state after a grouping operation
    if (this._lastStoredSelectedState) {
        this.setSelectedState(this._lastStoredSelectedState);
        delete this._lastStoredSelectedState;
    }

    // clear the _handlingDataChanged flag
    delete this._handlingDataChanged;
    
},

// wrap the call out to the dataArrived override point and handle sorter according to
// canSortClientOnly value and current data-state
_dataArrived : function (startRow, endRow) {
    var sortField = this._getSortFieldNum();
    if (sortField != null && sortField != -1) {
        var fieldNum = this.getFieldNum(sortField),
            field = this.getField(fieldNum);
        if (field && field.canSortClientOnly && !this._canSort(field)) {
            this._setSortFieldNum(null);

            // tell that toolbar button to unselect / get rid of sort arrow
            if (sortField != null && this.header && isc.isA.Toolbar(this.header)) {
                this.header.deselectButton(sortField);
                var button = this.header.getButton(sortField);
                if (button) button.setTitle(this.getHeaderButtonTitle(button));
            }

            // Get rid of the sort arrow in the sorter
            if (this.sorter) this.sorter.setTitle(this.sorter.getTitle());
        }
    }
    
    if (this.getCurrentCheckboxField()) {
        var cbPos = this.getCheckboxFieldPosition(),
            field = this.getField(cbPos),
            falseImage = this.checkboxFieldFalseImage || this.booleanFalseImage;
        // if we are showing a checkbox select-all header, and we don't have 
        // a full cache, disable the checkbox header and show a hover prompt
        if (isc.ResultSet && isc.isA.ResultSet(this.data) 
            && !this.data.allMatchingRowsCached()) 
        {                                    
            var props = {
                disabled: true,
                showHover: true,
                prompt: this.selection.selectionRangeNotLoadedMessage,
                title: this.canSelectAll == false ? "&nbsp;" : 
                        this.getValueIconHTML(falseImage.replace("." , "_Disabled."), field)
            }
            this.setFieldProperties(cbPos, props);
        // if we now have a full cache, enable the checkbox selectAll header
        } else {
            var props = {
                disabled: false,
                showHover: false,
                prompt: null,
                title: this.canSelectAll == false ? "&nbsp;" : 
                        this.getValueIconHTML(falseImage, field)
            }
            this.setFieldProperties(cbPos, props);
        }
    }

    
    if (isc.screenReader && this.body != null) {
        if (isc.isA.Tree(this.data)) {
            // in this case we're passed a single param, the parent node.
            var node = startRow;
            if (this.data.isOpen(node) && this.data.hasChildren(node)) {
                var children = this.data.getChildren(node);
                if (children && children.length > 0) node = children[0]
            }
            var rowNum = this.data.indexOf(node);
            // If we don't currently have focus, just remember the native focus row - this means
            // if we're showing a modal prompt / redrawing we should refocus on the right native
            // element...
            this.body._putNativeFocusInRow(rowNum, !this.hasFocus);
        }
    }

    this.dataArrived(startRow, endRow);
},


// doc'd in registerStringMethods block
dataArrived : function (startRow, endRow) {},

//>	@method listGrid._ignoreData() (A)
//		stop observing methods on data when it goes out of scope
//		called automatically by setData
//	@param	data	(object)		old data to be ignored
//<
_ignoreData : function (data) {
    //>Animation
    // Call finishRowAnimation - will kill any show/hide row animations
    // These animations assume the data remains constant for the duration
    // of the animation.    
    // (No-ops if appropriate)
    if (this.body) this.body.finishRowAnimation();
    //<Animation
	
    if (isc.isA.Tree(this.data)) this.ignore(data, "changeDataVisibility");

	this.ignore(data, "dataChanged");
    
    if (this.isObserving(data, "dataArrived")) {
        this.ignore(data, "dataArrived");
    }
    
	
	if (this.selection) this.selection.deselectAll();
	// NOTE: we don't ignore this.selection.setSelected because
	//			we're re-using the same selection object
},

//>	@method	listGrid.applyFieldDefaults()
//		@group	data
//         Derive default field sizes and formatters where possible, based on schema information.
//<
_generatedFieldNameCount:0,
applyFieldDefaults : function (fields) {
    if (fields == null) return;

	// apply ListGrid-specific defaults, like using toShortDate() for Date fields
    for (var i = 0; i < fields.length; i++) {
        var field = fields[i];

        if (field == null) continue;

        // In general we can support un-named fields in dataBoundComponents if there is a dataPath
        if (!this.allowNamelessFields && field[this.fieldIdProperty] == null) {
            if (field.dataPath == null) {
                this.logWarn("unable to process field with no name / dataPath:" +
                             this.echo(field));
                continue;
            }
            // apply an arbitrary name - this gives us a straightforward way to map the
            // field object any generated editor item, etc.
            
            field[this.fieldIdProperty] = "field" + this._generatedFieldNameCount++;
        }
        // default the alignment of each field to "left" if not specified
        var defaultAlign = this.isRTL() ? isc.Canvas.RIGHT : isc.Canvas.LEFT;
                                                              
        var type = field.type,
            baseType = (type != null ? isc.SimpleType.getBaseType(type) : null);

        
            
        // note: needs to be first, as "image" type technically inherits from text
        if (isc.SimpleType.inheritsFrom(field.type, "image")) {
            field._typeFormatter = this._formatImageCellValue; 

    	// Attempt to size columns to fit their content
        } else if (baseType == this._$text) { 

            if (field.width == null && field.length != null) {
                if (field.length < 15 && !field.valueMap) {
                	// use minimal space for small text fields with no value map
                    field.width = field.length * 7;
                }
            } 
 
        } else if (baseType == "integer" || baseType == "float") {
            // align numbers right by default to line up decimal places       
            defaultAlign = isc.Canvas.RIGHT
            field._typeFormatter = this._formatNumberCellValue;

    	// by default size date columns fields to match the default shortDate format applied
        // to date fields
    	
        } else if (baseType == "date") {
            var canEdit = (this.canEdit == true && field.canEdit != false) ||
                          (this.canEdit != false && field.canEdit == true);
            // If the field has unspecified size, size to accommodate formatted date
            // (or editor if the field is editable)
            field.width = field.width || (canEdit ? 100 : 80);
            // right alignment lines up years if day/month values are numeric and not padded
            defaultAlign = isc.Canvas.RIGHT;
            field._typeFormatter = this._formatDateCellValue;
                        
    	// by default size time columns fields to match the default format applied to time 
        // fields
        } else if (baseType == "time") {
            field.width = field.width || 80;
            field._typeFormatter = this._formatTimeCellValue;
            defaultAlign = isc.Canvas.RIGHT;
                        
        } else if (type == "binary" || type == "blob" || type == "upload" || type == "imageFile") {
            field._typeFormatter = this._formatBinaryCellValue;
        } else if (type == "link") {
            field._typeFormatter = this._formatLinkCellValue;        
        } else if (type == "icon") {
            if (field.width == null && field.autoFitWidth == null) {
                if (this.autoFitIconFields != "none") {
                    field.autoFitWidth = true;
                    field.autoFitWidthApproach = 
                        (this.autoFitIconFields == "title") ? "both" : "value";
                }
            }
            
            // check autoFitIconFields -- if set, set min width to accommodate the
            // icons and set autoFitWidth:true / autoFitWidthApproach such that
            // it'll expand to accommodate the title if appropriate
            if (field.width == null && field.autoFitWidth == null) {
                if (this.autoFitIconFields != "none") {
                    field.autoFitWidth = true;
                    field.autoFitWidthApproach = 
                        (this.autoFitIconFields == "title") ? "both" : "value";
                    // set the default width (min width) to the 
                    // calculated default width
                    
                    field.width = this.getDefaultFieldWidth(field);
                }
            }
            field.align = field.align || "center";

            // install a formatter that will put button.icon into the cell
            field._typeFormatter = this._formatIconCellValue;

            // default title so that icon appears alone (otherwise would default to field name
            // if title was unset) 
            field.title = field.title || "&nbsp;";
            
        // turn on 'canToggle' for all boolean fields.
        // If 'canEdit' is also set to true these fields will be editable via a single
        // click
        } else if (type == "boolean" || type== "checkbox") {
            if (field.canToggle == null) field.canToggle = true;
        }
         
        // For boolean fields we show checkbox images by default
        // this is handled via the valueIcon system - see getValueIcon(), getValueIconWidth() and
        // showValueIconOnly()
        
        
        // If formatCellValue was passed to us as a string, convert it to a method
        if (field.formatCellValue != null && !isc.isA.Function(field.formatCellValue)) 
            isc.Func.replaceWithMethod(field, "formatCellValue", "value,record,rowNum,colNum,grid");
        
        if (this.showValueIconOnly(field)) {
            defaultAlign = isc.Canvas.CENTER;
            
            // apply the "icon" field logic to fields that show valueIcons - 
            // respect autoFitIconFields here too
            if (field.width == null && field.autoFitWidth == null) {
                if (this.autoFitIconFields != "none") {
                    field.autoFitWidth = true;
                    field.autoFitWidthApproach = 
                        (this.autoFitIconFields == "title") ? "both" : "value";
                    // set the default width (min width) to the 
                    // calculated default width
                    field.width = this.getDefaultFieldWidth(field);
                }
            }
        }
        // TODO: numeric quantities with range validators could be given specific sizes
        
        if (!field.align) field.align = defaultAlign;
        
        // For fields marked as multiple:true, set the "validateEachItem" flag.
        // This ensures that when validators run in an editable grid we will
        // validate each selected value
        if (field.multiple && field.validateEachItem == null) field.validateEachItem = true;
    }
},


// Helper method called on boolean fields to determine whether we should use the
// booleanTrueImage/booleanFalseImage and related settings, or fall back to the general
// valueIcons system, which may show a combination of text and icons
_$boolean:"boolean",
_formatBooleanFieldAsImages : function (field) {
    // If booleanTrue/FalseImage have been set null, always back off to showing text / valueIcons
    if (this.booleanTrueImage == null && this.booleanFalseImage == null) return false;
    
    var type = field.type,
        baseType = (type != null ? isc.SimpleType.getBaseType(type) : null);
    if (baseType != this._$boolean) return false;
   
    // read as: user has not tried to set valueIcon-specific flags.
    // Also note: it's commonly necessary to set a valueMap with a boolean field in order to
    // allow stored values like YES/NO/null to be mapped to boolean true/false, so a valueMap
    // doesn't indicate an intent to use valueIcons.  If you have a valueMap and there are more
    // values than true/false/unset, you shouldn't declare the field boolean, it should be
    // enum.
    return (!field.suppressValueIcons && field.showValueIconOnly == null &&
             field.valueIcons == null && field.formatCellValue == null);    
},


//>	@method	listGrid.setFieldProperties()
// Set properties for a particular field, such as the title.
// <P>
// NOTE: to resize a field, use resizeField() instead.
//
// @param	fieldNum (number or String) name of the field, or index.
// @param	properties (Button Properties) properties to apply to the header
// @visibility external
//<
// NOTE: little testing has been done on which properties can actually be set this way
setFieldProperties : function (fieldNum, properties) {
    var field, allFields = this.getAllFields();
    var origField = fieldNum;
    if (isc.isA.Number(fieldNum)) {
        // if an index was passed, use the visible fields
        field = this.getField(fieldNum);
    } else {
        // if a key was passed, use the complete fields to handle hidden columns
        var globalFieldNum = isc.Class.getArrayItemIndex(
                                fieldNum, allFields, this.fieldIdProperty);
        field = allFields[globalFieldNum];
        // map back to the fieldNum within this.fields (not within this.completeFields)
        fieldNum = this.getFieldNum(field);
    }    
    if (!field) return;
    isc.addProperties(field, properties);

    if (this.header != null && this.header.isDrawn()) {
        // getFieldHeader / getLocalFieldNum will account for frozen fields
        var header = this.getFieldHeader(fieldNum),
            headerButton = header.getMember(this.getLocalFieldNum(fieldNum));            
        if (headerButton) headerButton.setProperties(properties);
    }
},

//> @method listGrid.setFieldTitle()
// Change the title of a field after the grid is created.
//
// @param fieldNum (integer or String) name of the field, or index.
// @param title (String) new title
// @visibility external
//<
setFieldTitle : function (fieldNum, title) {
    this.setFieldProperties(fieldNum, {title:title});
},


// AutoComplete
// --------------------------------------------------------------------------------------------

//> @method listGrid.setAutoComplete()
// Change the autoCompletion mode for the grid as a whole.
//
// @param   newSetting (AutoComplete)  new setting
// @group autoComplete
// @visibility autoComplete
//<
setAutoComplete : function (newSetting) {
    this.autoComplete = newSetting;
},

//> @method listGrid.setFieldAutoComplete()
// Change the autoCompletion mode for an individual field.
//
// @param   newSetting (AutoComplete)  new setting
// @group autoComplete
// @visibility autoComplete
//<
setFieldAutoComplete : function (field, newSetting) {
    field = this.getField(field);
    if (field) field.autoComplete = newSetting;
},

// --------------------------------------------------------------------------------------------

//>	@method	listGrid.showFields()
// Force an array of fields to be shown. This method does not add new fields to the grid,
// it simply changes field visibility. If a field.showIf expression exists, it will be
// destroyed.
// <P>
// Note: for showing multiple fields it is more efficient to call this method than to call
// +link{showField()} repeatedly.
//
// @param	field           (Array of String or Array of ListGridField)	Fields to show.
// @param   [suppressRelayout] (boolean) If passed, don't resize non-explicitly sized columns
//                                       to fill the available space.
// @visibility external
// @example columnOrder
//<
// Actually this is a synonym for showField() - separated for ease of documentation
showFields : function (fields, suppressRelayout) {
    return this.showField(fields,suppressRelayout);
},

//>	@method	listGrid.showField()
// Force a field to be shown. This method does not add new fields to the grid,
// it simply changes field visibility. If a field.showIf expression exists, it will be
// destroyed.
// <P>
// Note: for showing multiple fields it is more efficient to call +link{showFields()} than
// to call this method repeatedly.
//
// @param	field           (field name or ListGridField)	field to show
// @param   [suppressRelayout] (boolean) If passed, don't resize non-explicitly sized columns
//                                       to fill the available space.
// @visibility external
// @example columnOrder
//<
showField : function (fields, suppressRelayout) {

    arguments.__this = this;
    
    if (!isc.isAn.Array(fields)) {
        fields = [fields];
    }
    
    var noFields = true,
        allVisible = true;
    
    var setFieldsCalled = this.completeFields != null,
        mustSetFields = (!setFieldsCalled || this.frozenFields || this._suppressedFrozenFields)
    ;

    for (var i = 0; i < fields.length; i++) {
    
        var field = fields[i],
            fieldObj = field;
        
        // Use getSpecifiedField() to retrieve the fieldObject from the fields / completeFields
        // array.
        // Note that this returns null for an invalid field object / ID
        fieldObj = this.getSpecifiedField(fieldObj);
        
        if (fieldObj == null) {
            fields[i] = null;
            this.logWarn("showField(): unable to find field object for field: " + field
                         + ". Taking no action. Call setFields() to add new fields.")
            continue;
        }
        
        noFields = false;
        
        // -- We always want to clear out any showIf property on the field, as even if the field is
        //    currently being shown, we want the field to continue to be shown from this point on
        //    regardless of any conditions in a showIf property
        if (fieldObj.showIf != null) fieldObj.showIf = null;
        
        if (fieldObj.frozen) mustSetFields = true;
        
        // if this field is in a headerSpan, we need to call setFields() to rebuild it
        if (this.spanMap && this.spanMap[fieldObj.name] != null) mustSetFields = true;

        if (mustSetFields) continue;
        
        // If this.fields. contains the object, we can assume it's already visible if we're drawn
        // and will show up when we get drawn otherwise.
        if (this.fields.contains(fieldObj)) {
            fields[i] = null;
            continue;
        }

        // At this point we know we have at least one field that needs to be added
        // to the fields array (was previously hidden)
        allVisible = false;
        // hang onto the "live" fieldObj in the array
        fields[i] = fieldObj;
    }
    
    if (mustSetFields) {
         // Frozen fields: with frozen fields, the partial rebuild attempted below doesn't work.
         // This hasn't been looked into in detail yet.
         this.setFields(this.completeFields || this.fields);
         this.handleFieldStateChanged();
         return;
    }
    
    if (noFields || allVisible) return;
    
    // update this.fields
    this.deriveVisibleFields();
    
    // Empty slots may be present due to already visible fields
    fields.removeEmpty();
    
    var shownFieldNums = [],
        foundFields = 0;
        
    // Determine the position of the newly shown fields in our fields array.    
    for (var i = 0; i < this.fields.length; i++) {
        var index = fields.indexOf(this.fields[i]);
        if (index != -1) {
            shownFieldNums[index] = i;
            
            foundFields++;
            // stop when we've figured out the positon of all fields passed in.
            if (foundFields == fields.length) break;
        }
    }
    
    var header = this.header;
    // Update any UI to display the new field, if necessary
    

    // create the header button for the new column
    if (header != null) {
        if (!suppressRelayout) this.header.hPolicy = "fill";
        // undocumented feature - addButtons handles being pased an array of field indices
        // as well as an array of buttons
        this.header.addButtons(fields.duplicate(), shownFieldNums);
    }
    
    // tell the body about the new field
    if (this.body) {
    
        // If we're showing an editor in the new field we need to 
        // - create the form items for it and slot it into the edit form
        // - shift the colNum on other items, and the _editColNum of the grid as a whole.
        if (this._editorShowing) {
       
            var editRowNum = this.getEditRow(),
                record = this.getRecord(editRowNum),
    
                
                editedRecord = this.getEditedRecord(editRowNum),
                
                adjustedEditColNum = false,
                items = this.getEditForm().items,
                liveItemIndex = items.length-1,
                liveItem = items[liveItemIndex],
                itemColNum = liveItem.colNum;
        
            // sort the shownFieldNums
            // this allows us to update the live edit item colNum values easily.
            // Note that it means the shownFieldNums will no longer match up to the fields
            // entries in the shown fields array.
            shownFieldNums.sort();
            
            for (var i = shownFieldNums.length-1; i >= 0; i--) {
                var offset = i+1,
                    fieldNum = shownFieldNums[i],
                    threshold = (fieldNum - i);
                    
                if (!adjustedEditColNum && this._editColNum >= fieldNum) {
                    this._editColNum += offset;
                }
                
                // Update the edit form
                var fieldObj = this.fields[fieldNum],
                    width = this.getEditFormItemFieldWidths(record)[fieldNum],
                    item;
                
                while (liveItem != null && itemColNum >= threshold) {
                    liveItem.colNum += offset;
                    
                    liveItemIndex --;
                    liveItem = (liveItemIndex >= 0) ? items[liveItemIndex] : null;
                    itemColNum = (liveItem != null) ? liveItem.colNum : null;
                }
                
                // at this point we've updated any items with a colNum >= ours
                // Create an item for the newly shown field and slot it into the edit form
                //
                // liveItemIndex will now be set to the slot before the one we're interested in
                // (index of the item that previously matched our colNum, minus 1)
                //
                // Note if we're editing by cell this won't apply since if we're showing an editor
                // the field its showing on must already be visible.
                var drawnRange = this.body.getDrawArea();
                if (!this.editByCell && fieldNum >= drawnRange[2] && fieldNum <= drawnRange[3]) {
                    item = this.getEditItem(fieldObj, record, editedRecord, editRowNum, fieldNum, width);
                }
                if (item != null) {
                    this._editRowForm.addItems([item], liveItemIndex+1);
                }
            }
        }

        this.body.fields = this.normalFields || this.fields;
        this.setBodyFieldWidths(this.getFieldWidths());

        this._remapEmbeddedComponentColumns(this.body);

        // instant redraw rather than markForRedraw because we have to avoid dropping
        // values
        if (this.body.isDrawn()) this.body.redraw("show field");
    }
    
    // reset the sortFieldNum - we may have added or repositioned the primary sort field
    if (this.sortField != null) {
        this.sortFieldNum = null;
        this.sortFieldNum = this._getSortFieldNum()
    }
    
    // If we have a filterEditor showing, update its fields too
    if (this.filterEditor != null) this.filterEditor.showField(fields, suppressRelayout);

    // update recordSummaries as well as grid/group ones
    this.recalculateSummaries();
    if (this.summaryRow != null && this.showGridSummary) {
        this.summaryRow.showField(fields, suppressRelayout);
    }
    this.markForRedraw("showField");

    this.handleFieldStateChanged();
},

//>	@method	listGrid.hideFields()
// Force an array of fields to be hidden.
// <P>
// NOTE: If a field.showIf expression exists, it will be destroyed.
// <P>
// When hiding multiple fields, this method should be called rather than
// calling +link{listGrid.hideField()} repeatedly for each field to hide.
//
// @param fields (Array of String or Array of ListGridField) fields to hide
// @param [suppressRelayout] (boolean) if passed, don't relayout non-explicit sized fields 
//                                      to fit the available space
// @visibility external
//<
hideFields : function (fields, suppressRelayout) {
    return this.hideField(fields, suppressRelayout);
},


//>	@method	listGrid.hideField()
// Force a field to be hidden.<br><br>
//
// NOTE: If a field.showIf expression exists, it will be destroyed.
// <P>
// Note also that if multiple fields are to be hidden it is more efficient to 
// call +link{hideFields()} passing in the array of fields to hide rather than to call 
// this method repeatedly.
// 
// @param	field           (field name or ListGridField)	field to hide
// @param [suppressRelayout] (boolean) if passed, don't relayout non-explicit sized fields 
//                                      to fit the available space
// @visibility external
// @example columnOrder
//<
hideField : function (fields, suppressRelayout) {
    arguments.__this = this;
    
    var noFields = true,
        allHidden = true;

    if (!isc.isAn.Array(fields)) {
        fields = [fields];
    }
    // we shuffle some stored colNums around on edit items - store out the index of each field
    // being hidden in an array to make this easier.
    var hiddenFieldNums = [];
    
    // If we need to update the UI, it's more efficient to do it here directly than
    // just calling setFields().
    // However if setFields has never been called we will have to call it
    //
    // Frozen fields: with frozen fields, the partial rebuild attempted below doesn't work.
    // This hasn't been looked into in detail yet.
    

    var mustSetFields = (this.completeFields == null || this.frozenFields);
       
    // If we're showing an editor we need to make certain changes to get rid of the
    // form item, etc.
    var editorShowing = this._editorShowing,
        editRow = editorShowing ? this.getEditRow() : null,
        editCol = editorShowing ? this.getEditCol() : null,
        hidEditCell = false;
    for (var i = 0; i < fields.length; i++) {
        var field = fields[i],
            fieldObj = field;

        // Use getSpecifiedField() to pick up the field object from the completeFields array
        // (or if setFields has never been called, from the fields array).
        // Note - if we're passed an invalid field object, this method returns null.
        fieldObj = this.getSpecifiedField(fieldObj); 
        if (fieldObj == null) {
            this.logWarn("hideField(): unable to find field object for field: " + field
                         + ". Taking no action. To add this field use the setFields() method.");
            fields[i] = null;
            continue;
        }
        noFields = false;
            
        // -- Set showIf to always evaluate to false.
        fieldObj.showIf = this._$false;
        
        // If the field is not currently present in this.fields, we can safely assume it's already
        // hidden. No need to proceed in this case
        if (!this.fields.contains(fieldObj)) {
            fields[i] = null;
            continue;
        }
        allHidden = false;
        
        
        // If we're going to call setFields, we're done - we'll just call that
        // method outside this for-loop to update the completeFields array, edit values, etc.
        if (mustSetFields) continue;
        
        var fieldNum = this.fields.indexOf(fieldObj),
            fieldName = this.getFieldName(fieldNum);
        
        hiddenFieldNums.add(fieldNum);
        
        if (editorShowing) {
            if (editCol == fieldNum) hidEditCell = true;

            // Whatever happens we're going to clear the edit value from the cell being hidden
            // pass the additional 3rd parameter to avoid re-displaying the record's value
            // in the (about to be cleared) cell.
            var focusItem = this.getEditFormItem(fieldName);
            // (Item may not exist due to incremental rendering, non editable fields, editByCell)
            if (focusItem && focusItem.hasFocus) focusItem.blurItem();
            this.clearEditValue(editRow, fieldNum, true);
        }
    }
    // If we were passed an empty array, or an array containing already hidden fields
    // we can bail here.
    if (noFields || allHidden) return;

    // if we have to call setFields, simply call it and allow that method to do the work
    // of updating the fields array etc.
    if (mustSetFields) {
        this.setFields(this.completeFields || this.fields);
        this.handleFieldStateChanged();
        return;
    }
    
    fields.removeEmpty();
    if (editorShowing) {
        if (hidEditCell) {
            // If we're editing by cell, and hiding the current edit cell just kill the edit.
            if (this.editByCell) {
                this.cancelEditing(isc.ListGrid.PROGRAMMATIC);
                editorShowing = false;
            } else {
                // If possible we want to shift edit focus to an adjacent field
                // findNextEditCell() is unaware of the cells we've just hidden so
                // reimplement the relevent part of this method to find the next editable and
                // still visible field. Check for closest field to the left first, then to the right
                var newEditCol = editCol-1,
                    foundEditCol = false;
                while (newEditCol >= 0) {
                    if (!hiddenFieldNums.contains(newEditCol) &&
                        this.canEditCell(editRow, newEditCol) &&
                        this._canFocusInEditor(editRow, newEditCol))
                    {
                        foundEditCol = true;
                        break;
                    }
                    newEditCol--;
                }
                if (!foundEditCol) {
                    newEditCol = editCol +1;
                    while (newEditCol < this.fields.length) {
                        if (!hiddenFieldNums.contains(newEditCol) &&
                            this.canEditCell(editRow, newEditCol) &&
                            this._canFocusInEditor(editRow, newEditCol))
                        {
                            foundEditCol = true;
                            break;
                        }
                        newEditCol++;
                    }
                }
                
                // If we don't have any other editable cells in the row, just cancel the edit
                if (!foundEditCol) {
                    this.cancelEditing(isc.ListGrid.PROGRAMMATIC);
                    editorShowing = false;
                } else {
                    // focus in the adjacent field.
                    
                    this._startEditing(editRow, newEditCol,
                                            !this.getEditForm().hasFocus);
                }
            }
        }
    }
	// update this.fields
    this.deriveVisibleFields();
	// destroy the header button
    var header = this.header;
    if (header != null) {
        // Setting the hPolicy to "fill" will cause the header to relay it's buttons out to
        // fill the available space.
        if (!suppressRelayout) this.header.hPolicy = "fill";
        var buttons = [];
        for (var i = 0; i < hiddenFieldNums.length; i++) {
            var fieldNum = hiddenFieldNums[i];
        
            var button = this.header.getButton(fieldNum);
            buttons[buttons.length] = button;
            if (this.headerMenuButton && this.headerMenuButton.masterElement == button) {
                this.headerMenuButton.depeer();
            }
        }
        // removeButtons actually effects the display passed in, so duplicate it
        this.header.removeButtons(buttons.duplicate());
        buttons.map("destroy");
    }
	// If we're currently showing any edit form items for subsequent columns,
	// we must decrement their 'colNum' properties.
	// do this *before* we redraw the body, as the body redraw relies on these values being
	// accurate to create new items for fields that get shifted into view.
	// Also update the _editColNum if necessary. 
	
	
    var itemsToClear = [];
    if (editorShowing) {
        hiddenFieldNums.sort();
        
        var form = this._editRowForm,
            items = form.getItems(),
            
            itemIndex = items.length-1,
            item = items[itemIndex],
            itemColNum = item.colNum,
            
            adjustedEditColNum = false;
        
        for (var i = hiddenFieldNums.length-1; i >= 0; i--) {
        
            var offset = i+1,
                threshold = hiddenFieldNums[i];

            if (!adjustedEditColNum && this._editColNum > threshold) {
                this._editColNum -= offset;
                adjustedEditColNum = true;
            }
            
            while (item != null && itemColNum >= threshold) {
                if (itemColNum == threshold) itemsToClear.add(item);
                else item.colNum -= offset;
                
                itemIndex --;
                item = (itemIndex >= 0) ? items[itemIndex] : null;
                itemColNum = (item != null) ? item.colNum : null;
            }
        }
    }
	// tell the body about the removed fields
    if (this.body) {
        this.body.fields = this.normalFields || this.fields;
        this.setBodyFieldWidths(this.getFieldWidths());

        this._remapEmbeddedComponentColumns(this.body);        
    	// instant redraw rather than markForRedraw because we have to avoid dropping
    	// values
        if (this.body.isDrawn()) this.body.redraw("hide field");
    }

    if (editorShowing && itemsToClear.length > 0) {
    	// If we're currently showing an edit form item for this field, remove it now (already
    	// been cleared from the DOM), and had the edit values cleared.
    	for (var i = 0; i < itemsToClear.length; i++) {
        	// Hide the actual item if there is one.
            var item = itemsToClear[i];
            this._editRowForm.removeItems([item]);
        }
    }
    // reset the sortFieldNum - we may have removed or repositioned the primary sort field
    if (this.sortField != null) {
        this.sortFieldNum = null;
        this.sortFieldNum = this._getSortFieldNum()
    }


	// If we have a filterEditor showing, update its fields too
    if (this.filterEditor != null) this.filterEditor.hideField(fields, suppressRelayout);
    if (this.summaryRow && this.showGridSummary) {
        this.summaryRow._recalculateSummaries();
        this.summaryRow.hideField(fields, suppressRelayout);
    }
    
    this._remapEmbeddedComponents();
    this.handleFieldStateChanged();
},

//>	@method	listGrid.fieldIsVisible()
// Check whether a field is currently visible
//
// @param	field           (field name or field object)	field to be checked
// @return (boolean) true if the field is currently visible, false otherwise.
// @visibility external
//<
fieldIsVisible : function (field) {
    var fieldObj = field;
	// If passed a field ID, look for it in the completeFields array rather than the fieldsArray
	// as it is may be defined, but not visible    
    if (!isc.isAn.Object(fieldObj)) fieldObj = this.getSpecifiedField(field);

    return this.fields.contains(fieldObj);
},

// ----------------------------------------------------------------------------
// panelHeader related methods

showActionInPanel : function (action) {
    // specifically add the "editNew" action, which is not added by default
    if (action.name == "editNew") return true;
    return this.Super("showActionInPanel", arguments);
},

//> @method listGrid.getTitleField()
// @include dataBoundComponent.getTitleField() 
//<

//>	@method	listGrid.setFields()  ([A])
// Sets the fields array and/or field widths to newFields and sizes, respectively.<br><br>
//
// If newFields is specified, it is assumed that the new fields may have nothing in common with
// the old fields, and the component is substantially rebuilt.  Consider the following methods
// for more efficient, more incremental changes: resizeField, reorderField, showField,
// hideField, setFieldProperty.
//
// @visibility external
//
// @param	[newFields]		(List of ListGridField)	array of fields to draw
//<
_$setFields:"set fields",
setFields : function (newFields) {
    if (isc._traceMarkers) arguments.__this = this;
    
    // If called with new fields (more than just an update of existing field state), reset
    // the flag indicating that we've used specified width as a minimum for autoFitWidth fields
    if (newFields != null && this.fields != null &&
        this.fields != newFields && this.completeFields != newFields) 
    {
        delete this.fields._appliedInitialAutoFitWidth;
    }

    if (!newFields && this.getDataSource() && !this.getDataSource().hasFields()) {
        this.logWarn("ListGrid.setFields() : neither this ListGrid nor its dataSource have fields");    
    }
    // set boolean images to default checkboxItem images if unset
    if (!this.booleanFalseImage && !this.booleanTrueImage && !this.booleanPartialImage) { 
        this.booleanTrueImage = isc.CheckboxItem ? 
                        isc.CheckboxItem.getInstanceProperty("checkedImage") : null;
        this.booleanFalseImage = isc.CheckboxItem ? 
                        isc.CheckboxItem.getInstanceProperty("uncheckedImage") : null;
        this.booleanPartialImage = isc.CheckboxItem ? 
                        isc.CheckboxItem.getInstanceProperty("partialSelectedImage") : null;
        // set imageWidth from checkboxItem.valueIconWidth to avoid images having different sizes
        // when editing and not editing
        
        this.booleanImageWidth = isc.CheckboxItem ? 
                        isc.CheckboxItem.getInstanceProperty("valueIconWidth") : null;
        this.booleanImageHeight = isc.CheckboxItem ? 
                        isc.CheckboxItem.getInstanceProperty("valueIconHeight") : null;
                    
    }

    // if there is a sortFieldNum set, get a pointer to that field
	// we'll check later to reset the sort if we need to
	var sortFieldNum = this._getSortFieldNum(),
        oldSortField = (sortFieldNum != null && this.fields
                        ? this.fields[sortFieldNum] : null);
    
	// Interaction of setFields() with editing:
	// - If we have editValues for any row(s) [Pending unsaved edits], it is possible that
	//   some of the fields for which we have edit values will go away - in this case we need
	//   to drop those edit values, and any validation errors for those fields.
	// - If we are currently showing an editor for some row, we may also need to update the
	//   fields in the edit form.
	// * We'll drop the removed fields' edit values silently, [log at info level only] - this
	//   is expected / acceptable behavior when fields are removed.
    var editorShowing = this._editorShowing,
        editRowNum = this.getEditRow(),
        editColNum = this.getEditCol(),
        editForm = this._editRowForm,
        oldEditFieldName = this.fields ? this.getEditFieldName() : null
    ;
    
	// listGrid.completeFields and listGrid.fields are set to arrays which contain pointers
	// to the same set of "field" objects.
	// - Any fields with a showIf property that evaluates to false will be removed from
	//   this.fields, making it an array of only the list of *visible* fields.
	// - on a reorder or similar manipulation of the set of fields, you should manipulate 
	//   this.completeFields and pass the result to setFields().  Otherwise, you will lose any
	//   fields that aren't visible at the time.
	
    
	

	// on the first setFields() ever, create a new array for the visible fields
    if (this.completeFields == null) this.fields = [];

	// bind the passed-in fields to the DataSource and store
	// canPickOmittedFields - if true we'll pick up all dataSource fields but have them 
	// marked as showIf:false if useAllDataSourceFields is false
    this.completeFields = this.bindToDataSource(newFields, this.canPickOmittedFields);
    if (this.completeFields == null) this.completeFields = [];

    // rowNumberField displaying the current rowNum of each record
    if (this.shouldShowRowNumberField()) {
        var rnField = this.getCurrentRowNumberField(),
            rnPos = this.getRowNumberFieldPosition(),
            shouldAdd = !rnField
        ;
        
        // Exception: If the completeFields passed in (the new fields) contains the
        // special field already, no need to add it.
        
        if (shouldAdd && 
            newFields && newFields.find(this.fieldIdProperty, "_rowNumberField") != null) 
        {
            shouldAdd = false;
        }

        if (shouldAdd) rnField = this.getRowNumberField();

        if (shouldAdd) this.completeFields.addAt(rnField, rnPos);
        else this.completeFields.slideList([rnField], rnPos);
    } else {
        var rnField = this.getCurrentRowNumberField();
        if (rnField) this.completeFields.remove(rnField);
    }

    // expansionField for selection
    if (this.shouldShowExpansionField()) {
        var expField = this.getCurrentExpansionField(),
            expPos = this.getExpansionFieldPosition(),
            shouldAdd = !expField
        ;
        if (shouldAdd && newFields && 
            newFields.find(this.fieldIdProperty, "_expansionField") != null) 
        {
            shouldAdd = false;
        }

        if (shouldAdd) expField = this.getExpansionField();
        // make expansionField frozen if we have any frozen fields - note autoFreeze: true does this now
        
        if (shouldAdd) this.completeFields.addAt(expField, expPos);
        else this.completeFields.slideList([expField], expPos);
    } else {
        var expField = this.getCurrentExpansionField();
        if (expField) this.completeFields.remove(expField);
    }

    // checkboxField for selection
    if (this.shouldShowCheckboxField()) {
        var cbField = this.getCurrentCheckboxField(),
            cbPos = this.getCheckboxFieldPosition(),
            shouldAdd = !cbField
        ;
        if (shouldAdd && newFields && newFields.find(this.fieldIdProperty, "_checkboxField")) {
            shouldAdd = false;
        }

        if (shouldAdd) cbField = this.getCheckboxField();
        // make checkboxField frozen if we have any other frozen fields - note autoFreeze: true does this now

        if (shouldAdd) this.completeFields.addAt(cbField, cbPos);
        else this.completeFields.slideList([cbField], cbPos);
    } else {
        var cbField = this.getCurrentCheckboxField();
        if (cbField) this.completeFields.remove(cbField);
    }
    
    // Add / update the removeField if this.canRemoveRecords is true
    if (this.shouldShowRemoveField()) {
        var removeFieldNum = this.completeFields.findIndex("isRemoveField", true),
            removeField = (removeFieldNum >= 0) ? this.completeFields[removeFieldNum] 
                                                : {excludeFromState:true, isRemoveField:true};
        
        if (removeFieldNum == -1 && newFields) {
            removeFieldNum = newFields.findIndex("isRemoveField", true);
        }
        if (!removeField._removeFieldInitialized) {
            
            isc.addProperties(removeField, this.removeFieldDefaults, this.removeFieldProperties);
            if (removeField.name == null) removeField.name = "_removeField";
            if (removeField.title == null) removeField.title = this.removeFieldTitle;
            if (removeField.cellIcon == null) removeField.cellIcon = this.removeIcon;
            if (removeField.iconSize == null) removeField.iconSize = this.removeIconSize;
            if (removeFieldNum == -1) {
                this.completeFields.add(removeField);
            }
            
            removeField._removeFieldInitialized = true;
        }
    }
    // sets things up to look up display maps when LG is drawn
    if (isc.DataSource) this._setOptionDataSources();    

    // set field state if necessary
    if (this.fieldState != null) this.setFieldState();
    
	// apply various type-based defaults
    this.applyFieldDefaults(this.completeFields);
    
    // Set a flag on the array and the individual field objects - this allows us catch the case 
    // where a developer reuses a standard fields array across multiple grids rather than using
    // the defaultFields attribute
    this.completeFields._initializedFieldsArray = true;
    this.completeFields.setProperty("_initializedFieldObject", true);
    
    
	// determine which fields should be shown, and add them to the visible fields array
	// (this.fields)
    this.deriveVisibleFields();
    
    // warn about all fields being hidden.  Note it's valid to have no fields if you've created
    // a grid that will be bound to a DataSource or provided fields later.
    if (this.fields.length == 0 && this.completeFields.length > 0) {
        this.logWarn("All specified fields for this component are hidden. Note that fields " +
                     "may be hidden via 'showIf' or 'detail' attribute values. " +
                     "In order to display data this grid must have some visible fields.");
    }

	// wipe out the cached fieldWidths, if any
    this._fieldWidths = null;
    
    var newEditColNum, editFieldStillPresent, hadFocus;
	
    if (editorShowing) {
    	// make sure we get the latest value of the field being edited
        // (Not relevant if we're not drawn since the user can't have updated)
        this.storeUpdatedEditorValue();

    	// assume we should continue editing at the field with the same id
        newEditColNum = this.fields.findIndex(this.fieldIdProperty, oldEditFieldName);
        if (newEditColNum != -1 && !this.canEditCell(editRowNum, newEditColNum)) 
            newEditColNum = -1;
        
    	// if the field with same id isn't editable, find the next editable cell
        if (newEditColNum == -1) {
            var newEditCell;
            // extra param to suppress checking past this row            
            if (!this.editByCell) newEditCell = this.findNextEditCell(editRowNum, 0, 1, 
                                                                      true,true, false, true);
            if (newEditCell != null && newEditCell[0] == editRowNum) {
                newEditColNum = newEditCell[1];
            }

        	// Kill the edit if we're editing by cell, or can't find another editable field
        	// in the edit row.
            if (newEditColNum == -1) {
                this.cancelEditing(isc.ListGrid.PROGRAMMATIC);
                editorShowing = false;
            }
        } else {
        	// field with the same name present and editable: blur and refocus after redraw
            var item = editForm.getItem(oldEditFieldName);
            if (item) {
                hadFocus = item.hasFocus;
                if (hadFocus) editForm._blurFocusItemWithoutHandler();
            }
            editFieldStillPresent = true;
        }
        
    	// Hide the editor if still editing.
    	// Note that this will fire a blur handler unless we have already blurred without 
    	// the handler
    	
        if (editorShowing) this.hideInlineEditor(false, true);
    }
    
    var autoCanFreeze = (this.canFreezeFields == null || this._autoDerivedCanFreeze) &&
                        this.fixedRecordHeights != false && this.fixedFieldWidths != false &&
                        this.autoFitData != "horizontal" && this.autoFitData != "both" &&
                        this.bodyOverflow != "visible";
    if (autoCanFreeze) {
        if (this.completeFields.getProperty("overflow").contains("visible")) autoCanFreeze = false;
    }
    if (autoCanFreeze) {
        this._autoDerivedCanFreeze = true;
        this.canFreezeFields = true;
    } else if (this._autoDerivedCanFreeze) {
        delete this._autoDerivedCanFreeze;
        this.canFreezeFields = null;
    }
    
	// if we're working with a cellSelection, it needs to know how many columns are visible
    if (this.canSelectCells) this.selection.numCols = this.fields.length;

	// if there is an oldSortField, try to find it in the new fields array
	if (oldSortField) {
        var newSortField = this.fields.indexOf(oldSortField);
        if (newSortField == -1) newSortField = null;
        this._setSortFieldNum(newSortField);
	}

	// if we are showing the header, rebuild it with the new set of fields
    if (this.showHeader && this.headerHeight > 0 && this.header != null) { 
    	// this method will actually update the header to match this.fields
        this.updateHeader();

    	// sync scroll position of new header with current body scroll position
        if (this.body != null) {
            this.syncHeaderScrolling(this.body.getScrollLeft());
        }
    }
    
    this.updateBody();
  
    // If we have a filterEditor, update its fields
    
    if (this.filterEditor) {
         // update the ds on the filterEditor (and the edit-form) - we need a ds to support some 
        // expression-parsing features, like "matches other field" detecting fields by title as
        // well as name
        this.filterEditor.updateDataSource(this.getDataSource());
        var completeFields = this.completeFields || [];
        this.filterEditor.setFields(completeFields.duplicate());
    }
    
	// if we've rebuilt the header, this will place it and draw it.
	// this will also determine new field widths for the new fields (whether from the header or
	// otherwise)
    this.layoutChildren(this._$setFields);
    
	// Now the fields have been set, update the edit values field:
	// Update our editValues, and validation errors to account for any fields that have 
	// gone from the list.
    if (this._editSessions != null) {
        var fieldsDropped = {};
        
        for (var i in this._editSessions) {  
            if (this._editSessions[i] == null) continue;
            var vals = this._editSessions[i]._editValues,
            	// We want to hang onto primary key values, even if they're not shown in the
            	// ListGrid
                pks = (this.dataSource != null 
                    ? this.getDataSource().getPrimaryKeyFieldNames()
                    : [])
            ;

            for (var currentFieldName in vals) {
            	
                if (!this.fields.containsProperty(this.fieldIdProperty, currentFieldName) &&
                    !pks.contains(currentFieldName)) {
                	// track which fields were dropped so we can inform the user / developer
                    fieldsDropped[currentFieldName] = true;
                	// clearEditValue will clear the editValue and any validation errors for 
                	// the field
                	// Pass the additional 3rd parameter to avoid refreshing the updated cells
                    
                    this.clearEditValue(vals, currentFieldName, true);
                }
            }
        }
        
        fieldsDropped = isc.getKeys(fieldsDropped);
        if (fieldsDropped.length > 0) {
            this.logInfo("'setFields()' removed the following fields which had pending edit " +
                         "values for some row[s]: '" + fieldsDropped.join("', '") + "'.\n" +
                         "Edit values for these fields have been dropped.", "gridEdit");
        }
    }

	// re-show the editor, with the new set of fields
    if (editorShowing) {
    	// if the previous edit field is still showing, just re-show the editor
    	// and focus silently if appropriate.
        if (editFieldStillPresent) {
        	// 2nd and 3rd parameters are both false - even though this field may be in a
            // new position, it's the same logical field, so we don't want to fire
            // editorEnter()
            this.showInlineEditor(editRowNum, newEditColNum, false, false, true);
            if (hadFocus) editForm._focusInItemWithoutHandler(newEditFieldName);
        } else {
        	// If we've killed the previous edit field, but still want to be editing, use
        	// _startEditing() to start editing the new cell to ensure editValues get set up
            // if necessary
            this._startEditing(editRowNum, newEditColNum, !hadFocus);
        }
        
    	// Refocus without firing the handler if the old edit field is still around and had
    	// focus before the setFields.
        if (hadFocus) {
            var newEditFieldName = this.getEditFieldName();
            if (newEditFieldName == oldEditFieldName) {
                editForm._focusInItemWithoutHandler(newEditFieldName);
            } else {
                editForm.focusInItem(newEditFieldName)
            }
        }
    }

	// if we've already been sorted and we can sort, sort the data
	if (this._getSortFieldNum() != null && this.canSort) this.resort();

    if (this.summaryRow && this.showGridSummary) {
        this.summaryRow.setFields(this.completeFields.duplicate());
        this.summaryRow._recalculateSummaries();
    }
},

// override add/removeField to account for us having 'this.completeFields' which
// dataBoundComponent doesn't expect
addField : function (field, index) {
    return this.Super("addField", [field, index, this.completeFields], arguments);
},
removeField : function (field) {
    return this.Super("removeField", [field, this.completeFields], arguments);
},

// Helper method - should this grid show the special checkbox field for 
// selectionAppearance:"checkbox". TreeGrid's show the checkbox in the
// special tree field so we don't show this field.
_$checkbox:"checkbox",
shouldShowCheckboxField : function () {
    if (this.fieldSourceGrid) return this.fieldSourceGrid.shouldShowCheckboxField();
    return (this.selectionAppearance == this._$checkbox &&
        this.selectionType != this._$none &&
        !isc.isA.TreeGrid(this));
},

//> @method listGrid.focusInFilterEditor()
// If the filter editor (+link{listGrid.showFilterEditor}) is visible for this grid, 
// this method will explicitly put focus into the specified field in the filter editor.
// @group filterEditor
// @visibility external
// @param [fieldName] (String) Name of the field to put focus into. If unspecified focus will go
//                             to the first field in the editor
//<
focusInFilterEditor : function (fieldName) {
    if (this.filterEditor == null) return;
    var fieldNum = fieldName != null ? this.getColNum(fieldName) : null;
    this.filterEditor.startEditing(0, fieldNum);
},

//> @method listGrid.filterByEditor()
// If the filter editor (+link{listGrid.showFilterEditor}) is visible for this grid, 
// this method will perform a filter based on the current values in the editor.
// @group filterEditor
// @visibility external
//<
filterByEditor : function () {
    if (this.filterEditor != null) this.filterEditor.performAction();
},


// Override bindToDataSource. Unlike other data-bound widgets if this.showDetailFields is true
// we want to default our detailFields to be hidden (but accessable via the headerContextMenu)
// Note: We do this at init (setFields) time by setting the showIf property on the field,
// rather than overriding fieldShouldBeVisible() to return false for detail fields, so that
// when showField() is called on the field, that method can return true and allow the field
// to show.
bindToDataSource : function (fields, componentIsDetail, a,b,c,d) {
    var numericFieldName = false;

    var completeFields = this.invokeSuper(isc.ListGrid, "bindToDataSource", 
                                          fields, componentIsDetail, a,b,c,d);
    if (this.showDetailFields && completeFields != null) {
        for (var i = 0; i < completeFields.length; i++) {
            var field = completeFields[i];
            if (field.showIf == null && field.detail == true) {
                field.showIf = this._$false;
            }
            
            if (isc.isA.Number(parseInt(field.name)) && 
                parseInt(field.name).toString() == field.name) 
            {
                numericFieldName = true;
            }
            field._isFieldObject = true;
        }
    }
    this._noNumericFields = !numericFieldName;
    return completeFields;
},

// Field State
// --------------------------------------------------------------------------------------------
// The fieldState is an object capturing presentation information about the fields - 
// expected to be used as a way for developers to save the current presentation (EG in cookies)
// and re-load that presentation when the page is reloaded.
// fieldState is an opaque format. 


  
//> @type listGridFieldState  
// An object containing the stored presentation information for the fields of a listGrid.
// Information contained in a <code>listGridFieldState</code> object includes the 
// visibility and widths of the listGrid's fields.<br>
// Note that this object is a JavaScript string, and may be stored (for example) as a blob 
// on the server for state persistence across sessions.
// 
// @group viewState
// @visibility external
//<

//>	@method	listGrid.getFieldState() 
// Returns a snapshot of the current presentation of this listGrid's fields as 
// a +link{type:listGridFieldState} object.
// <P>
// This object can later be passed to +link{listGrid.setFieldState()} to reset this grid's
// fields to the current state.
// <P>
// Note that the information stored includes the current width and visibility of each of this 
// grid's fields, as well as any +link{listGrid.canAddFormulaFields,formula} or
// +link{listGrid.canAddSummaryFields,summary fields} added by the user.
//
// @group viewState
// @see listGrid.setFieldState();
// @visibility external
// @return (listGridFieldState) current state of this grid's fields.
//<
// LG.getFieldState() moved up to DBC

//>	@method	listGrid.setFieldState() 
// Sets some presentation properties (visibility, width, userFormula and userSummary) of the 
// listGrid fields based on the +link{type:listGridFieldState} object passed in.<br>
// Used to restore previous state retrieved from the grid by a call to +link{listGrid.getFieldState()}.
//
// @group viewState
// @param fieldState (listGridFieldState) state to apply to the listGrid's fields.
// @visibility external
// @see listGrid.getFieldState()
//<
setFieldState : function (fieldState) {
    //!OBFUSCATEOK
    if (this.completeFields == null) this.setFields(this.fields);
    
    if (fieldState == null && this.fieldState != null) {
        if (isc.isA.String(this.fieldState)) {
            fieldState = this.evalViewState(this.fieldState, "fieldState")
        }
        this.completeFields = this._setFieldState(this.fieldState);
        // fieldState is init-only property, so null after use
        this.fieldState = null;
        return;
    }

    fieldState = this.evalViewState(fieldState, "fieldState")
    if (fieldState) {
        this.completeFields = this._setFieldState(fieldState);
        this.refreshFields();
    }
},

handleFieldStateChanged : function () {
    this.fieldStateChanged();
    this.handleViewStateChanged();
},
// fieldStateChanged implemented in DBC

// ---------------------------------------------------------------------------------------

//> @type SelectionAppearance
// How data selection should be presented to the user.
// @value "rowStyle" selected rows should be shown with different appearance - see
//                   +link{ListGrid.getCellStyle()} and optionally
//                   +link{ListGrid.selectionCanvas}.
// @value "checkbox" an extra, non-data column should be automatically added to the ListGrid,
//                   showing checkboxes that can be toggled to select rows.  
//                   See +link{listGrid.getCheckboxField()}.
// @visibility external
//<

// defaults for checkbox field
checkboxFieldDefaults: {
    name: "_checkboxField",
    excludeFromState:true,
    canEdit: false,
    shouldPrint:false,
    // Note we could probably allow filtering by this field for client-side grids - which would
    // allow you to view all selected / unselected rows easily, (nice feature) but it wouldn't
    // work for databound grids
    canFilter:false,
    canGroupBy: false,
    canSort: false,
    canExport: false,
    canHide: false,
    canReorder: false,
    canDragResize: false,
    // make this special field canHilite false so we don't see it in HiliteEditors by default
    canHilite: false,
    _isCheckboxField: true,
    type:"boolean",
    showDefaultContextMenu: false,
    showHeaderContextMenuButton: false,
    hoverHTML: "return null;", // supress hover at row level, only hovering for header
    autoFreeze: true
},

//> @method listGrid.getCheckboxField()
// Returns the specially generated checkbox field used when +link{selectionAppearance} is
// "checkbox".
// <P>
// Called during +link{setFields()}, this method can be overridden to add advanced dynamic
// defaults to the checkbox field (call Super, modify the default field returned by Super,
// return the modified field).  Normal customization can be handled by just setting
// +link{AutoChild} properties, as mentioned under the docs for +link{listGrid.checkboxField}.
//
// @return (ListGridField)
// @group checkboxField
// @visibility external
//<
// the amount to add to the icon width to get the checkbox field width
_checkboxFieldWidth: 15,
getCheckboxField : function () {
    var grid = this,
        cbField = {
        // default the width to the width of the icon plus an arbitrary buffer
            width:this._checkboxFieldWidth + this._getCheckboxFieldImageWidth(),
            getAutoFreezePosition: function () { return grid.getCheckboxFieldPosition() }
        }
    ;
    isc.addProperties(cbField, this.checkboxFieldDefaults, this.checkboxFieldProperties);

    var falseImage = this.checkboxFieldFalseImage || this.booleanFalseImage;
    
    cbField.title = (this.canSelectAll == false || this.selectionType == "single" ? "&nbsp;" : 
        this.getValueIconHTML(falseImage, cbField));

    return cbField;
},

getCurrentCheckboxField : function () {
    var fields = this.getFields();
    if (!fields) return null;
    var cbFields = fields.find(this.fieldIdProperty, "_checkboxField");
    return !cbFields ? null : isc.isAn.Array(cbFields) ? cbFields[0] : cbFields;
},

_getCheckboxFieldImageWidth : function () {
    return this.checkboxFieldImageWidth || this.booleanImageWidth ||
            (isc.CheckboxItem ? isc.CheckboxItem.getInstanceProperty("valueIconWidth") : null);
},
_getCheckboxFieldImageHeight : function () {
    return this.checkboxFieldImageHeight || this.booleanImageHeight ||
            (isc.CheckboxItem ? isc.CheckboxItem.getInstanceProperty("valueIconWidth") : null);
},

//> @method listGrid.isCheckboxField()
// Identifies whether the passed-in field is the specially generated
// +link{listGrid.checkboxField,checkboxField} used when +link{selectionAppearance} is
// "checkbox".  Use this method in your custom event handlers to avoid inappropriately
// performing actions when the checkboxField is clicked on.
// 
// @param field (ListGridField) field to test
// @return (boolean) whether the provided field is the checkbox field
// @group checkboxField
// @visibility external
//<
isCheckboxField : function (field) {
    if (!field || !field._isCheckboxField) return false;
    else return true;
},

// helper function to get the checkbox field position
// This is a special field which appears "stuck to" the left edge of the grid.
// We have a few other special fields:
// - rowNumbers column
// - groupTitle column when showing group summaries in header
// - record expansion icon column
getCheckboxFieldPosition : function () {
    if (this.fieldSourceGrid) return this.fieldSourceGrid.getCheckboxFieldPosition();
    if (this.selectionAppearance != "checkbox" || isc.isA.TreeGrid(this)) return -1;
    
    var pos = 0;
    if (this.showRowNumbers) pos += 1;
    if (this.showingGroupTitleColumn()) pos += 1;
    if (this.canExpandRecords) pos += 1;
    return pos;
},

//> @type listGridSelectedState  
// An object containing the stored selection information for a listGrid.
// Note that this object is not intended to be interrogated directly, but may be stored 
// (for example) as a blob on the server for state persistence across sessions.
// 
// @group viewState
// @visibility external
//<
// listGridSelectedState object is implemented as an array of primaryKeys indicating the 
// selected set of records.

//>	@method	listGrid.getSelectedState() 
// Returns a snapshot of the current selection within this listGrid as 
// a +link{type:listGridSelectedState} object.<br>
// This object can be passed to +link{listGrid.setSelectedState()} to reset this grid's selection
// the current state (assuming the same data is present in the grid).<br>
// @group viewState
// @see listGrid.setSelectedState();
// @visibility external
// @return (listGridSelectedState) current state of this grid's selection
//<
getSelectedState : function (supressWarnings) {
    if (!this.selection) return null;
    if (!this.dataSource || 
        isc.isAn.emptyObject(this.getDataSource().getPrimaryKeyFields())) 
    {
        if (!supressWarnings) {
            
            this.logWarn("can't getSelectedState without a DataSource " +
                         "with a primary key field set");
        }
        return null;
    }

    var selection = this.selection.getSelection() || [],
        selectedState = [];

    // store primary keys only.  Works only with a DataSource
    for (var i = 0; i < selection.length; i++) {
        selectedState[i] = this.getPrimaryKeys(selection[i]);
    }
    
    return isc.Comm.serialize(selectedState,false);
},

//>	@method	listGrid.setSelectedState() 
// Reset this grid's selection to match the +link{type:listGridSelectedState} object passed in.<br>
// Used to restore previous state retrieved from the grid by a call to 
// +link{listGrid.getSelectedState()}.
//
// @group viewState
// @param selectedState (listGridSelectedState) Object describing the desired selection state of
//                                              the grid
// @see listGrid.getSelectedState()
// @visibility external
//<
setSelectedState : function (selectedState) {
    
    selectedState = this.evalViewState(selectedState, "selectedState")
    if (!selectedState) {
        this.selection.deselectAll();
        return;
    }
    var selection = this.selection, 
        data = this.originalData || this.data;

    if (data && selection) {
        selection.deselectAll();
        var records = [];
        for (var i = 0; i < selectedState.length; i++) {
            var item = selectedState[i];
            // resultSet.indexOf() looks up by matching PK values
            var index = data.findByKeys(selectedState[i], this.getDataSource());
            // record may have been removed
            if (index != -1) records.add(data.get(index));
        }
        this.selection.selectList(records);
    }
},

//> @type listGridSortState  
// An object containing the stored sort information for a listGrid.
// Note that this object is not intended to be interrogated directly, but may be stored 
// (for example) as a blob on the server for state persistence across sessions.
// 
// @group viewState
// @visibility external
//<
// listGridSortState object is implemented as a simple JS object containing fieldName and sortDir
// attributes - also now supports the multiSorting subsystem by storing the grid's list of 
// sortSpecifiers


//>	@method	listGrid.getSortState() 
// Returns a snapshot of the current sort state within this listGrid as 
// a +link{type:listGridSortState} object.<br>
// This object can be passed to +link{listGrid.setSortState()} to reset this grid's sort to
// the current state (assuming the same fields are present in the grid).<br>
// @group viewState
// @see listGrid.setSortState();
// @visibility external
// @return (listGridSortState) current sort state for the grid.
//<
getSortState : function () {

    if (this.logIsInfoEnabled("sorting")) {
        this.logInfo("\n"+
            "grid.sortFieldNum is: "+this.sortFieldNum+"\n"+
            "grid.sortField is: "+this.sortField+"\n"+
            "grid.getField(grid.sortFieldNum) is:\n"+
                isc.echoAll(this.getField(this.sortFieldNum))+"\n"+
            "-----------------------------------------\n"+
            "grid._getSortFieldNum() is: "+this._getSortFieldNum()+"\n"+
            "grid.getField(grid._getSortFieldNum()) is:\n"+
                isc.echoAll(this.getField(this._getSortFieldNum()))+"\n"+
            "", "sorting"
        )
    }

    var sortFieldNum = this._getSortFieldNum(),
        sortField = (sortFieldNum != null ? this.getField(sortFieldNum) : null),
        sortFieldName = sortField != null ? this.getFieldName(sortField) : null,
        sortDir = sortField && sortField.sortDirection != null ? 
                sortField.sortDirection : this.sortDirection,
        sortState = {fieldName:sortFieldName, sortDir:sortDir}
    ;

    if (this._sortSpecifiers && this._sortSpecifiers.length > 0) {
        var specifiers = isc.shallowClone(this._sortSpecifiers);
        // remove some props added during sorting - keeps the output neat and they'll be 
        // re-applied on setSort()
        specifiers.clearProperty("primarySort");
        specifiers.clearProperty("sortIndex");
        specifiers.clearProperty("normalizer");
        specifiers.clearProperty("context");
        sortState.sortSpecifiers = specifiers;
    }

    // eval() of a string containing object literal text will js error - enclose in "(" ... ")" to 
    // avoid this.
    return "(" + isc.Comm.serialize(sortState,false) + ")";
},


//>	@method	listGrid.setSortState() 
// Reset this grid's sort state (sort field and direction or list of 
// +link{sortSpecifier}s) to match the 
// +link{type:listGridSortState} object passed in.<br>
// Used to restore previous state retrieved from the grid by a call to 
// +link{listGrid.getSortState()}.
//
// @param sortState (listGridSortState) Object describing the desired sort state for the grid.
// @group viewState
// @see listGrid.getSortState()
// @visibility external
//<
setSortState : function (state) {
    state = this.evalViewState(state, "sortState")
    if (!state) {
        this.clearSort();
        return;
    }

    if (state.sortSpecifiers) {
        // multisort
        this.setSort(isc.shallowClone(state.sortSpecifiers));
    } else if (state.fieldName == null) {
        this.clearSort();
    } else {
        // will only get here now if the user has legacy state or creates their own state 
        // object which doesn't include sortSpecifiers - a sortSpecifier will be created 
        // anyway and further calls to getSortState() will include it
        var fieldNum = this.getFieldNum(state.fieldName)
        if (fieldNum != -1) this.sort(fieldNum, state.sortDir);
    }
},

//> @type listGridViewState  
// An object containing the "view state" information for a listGrid.<br>
// This object contains state information reflecting<br>
// - +link{type:listGridFieldState}<br>
// - +link{type:listGridSortState}<br>
// - +link{type:listGridSelectedState}<br>
// for the grid.<br>
// Note that this object is not intended to be interrogated directly, but may be stored 
// (for example) as a blob on the server for view state persistence across sessions.
// 
// @group viewState
// @visibility external
//<
// listGridViewState object is implemented as a simple JS object containing the following 
// fields:
// - selected [a listGridSelectedState object]
// - field [a listGridFieldState object]
// - sort [a listGridSortState object]

//>	@method	listGrid.getViewState() 
// Returns a snapshot of the current view state of this ListGrid.<br>
// This includes the field state, sort state and selected state of the grid, returned as a
// +link{type:listGridViewState} object.<br>
// This object can be passed to +link{listGrid.setViewState()} to reset this grid's vew state
// to the current state (assuming the same data / fields are present in the grid).<br>
// @group viewState
// @see type:listGridViewState
// @see listGrid.setViewState();
// @visibility external
// @return (listGridViewState) current view state for the grid.
//<    
getViewState : function (returnObject) {
    var state = {
        selected:this.getSelectedState(true),
        field:this.getFieldState(),
        sort:this.getSortState(),
        hilite:this.getHiliteState(),
        group: this.getGroupState()
    };

    // Available so TG can call Super() and get an object back
    if (returnObject) return state;
    return "(" + isc.Comm.serialize(state,false) + ")";
},

getGroupState : function () {
    var state = this.getGroupByFields();
    if (state == null) state = "";
    else state = state.join(",");
    return state;
},
setGroupState : function (state) {
    if (state) this.groupBy(state.split(","));
    else this.ungroup();
},

//>	@method	listGrid.setViewState() 
// Reset this grid's view state to match the +link{type:listGridViewState} object passed in.<br>
// Used to restore previous state retrieved from the grid by a call to 
// +link{listGrid.getViewState()}.
//
// @param viewState (listGridViewState) Object describing the desired view state for the grid
// @group viewState
// @see listGrid.getViewState()
// @visibility external
//<
setViewState : function (state) {
    state = this.evalViewState(state, "viewState")
    if (!state) return;

    // Order is somewhat important - for example show fields before potentially sorting 
    // by them, etc
    if (state.field) this.setFieldState(state.field);
    this.setSortState(state.sort);
    this.setGroupState(state.group);
    this.setHiliteState(state.hilite);
    this.setSelectedState(state.selected);
},

// Fire the viewStateChanged notification. This is documented in registerStringMethods()

handleViewStateChanged : function () {
    this.fireOnPause("viewStateChangeNotification", {target:this,methodName:"viewStateChanged"}, 0);
},
getViewStateChangedFunction : function () {
    if (this._viewStateChangedFunction == null) {
        var grid = this;
        this._viewStateChangedFunction = function () {
            if (grid.destroyed) return;
            grid.viewStateChanged();
        };
    }
    return this._viewStateChangedFunction;
},
viewStateChanged : function() {
},

//> @method listGrid.setDataSource()
// @include dataBoundComponent.setDataSource()
// @visibility external
//<
// Override setDataSource() - we need to reset the stored filter criteria in case we are
// showing the filterEditor.
setDataSource : function (dataSource, fields) {
    // If we have a specified groupByField, ungroup when changing from one DataSource to another.
    // *Don't ungroup if we didn't have a dataSource before this method ran - this is likely to
    // happen on init when a developer can validly set groupBy before setting dataSource.
    
    var currentDS = this.getDataSource();
    if (currentDS != null && currentDS != dataSource && currentDS.ID != dataSource) {
        var groupByFields = this.getGroupByFields();
        if (groupByFields != null && groupByFields.length > 0 && groupByFields[0] != null &&
            this.originalData != null)
        {
            this.ungroup();
        }
    }

    this.Super("setDataSource", arguments);
    this.clearFilterValues();
    // discard edits as they don't apply to the new DataSource records
    
    this.discardAllEdits();
    // if ds.canMultiSort is specified, set this on the grid also
    if (!this.data.setSort) {
        this.canMultiSort = false;
    } else if (this.canMultiSort != false) {
        dataSource = this.getDataSource();
        this.canMultiSort = (dataSource && dataSource.canMultiSort != null ? 
            dataSource.canMultiSort : true) && this.canSort;
    }
},


// determine which fields should be shown, and add them to the visible fields array.
// (Used as an internal helper - developers should call 'refreshFields' instead)
deriveVisibleFields : function () {
	// NOTE: we use setArray() so that this.fields remains the same array instance.
    this.fields.setArray(this.getVisibleFields(this.completeFields));
    this.deriveFrozenFields();
    this.refreshMasterIndex();
},


//> @method ListGrid.refreshFields
// Re-evaluates +link{ListGridField.showIf} for each field, dynamically showing and 
// hiding the appropriate set of fields
// @visibility external
//<
refreshFields : function () {
    // Just fall through to 'setFields()' for now
	
    this.setFields(this.completeFields);
},

getFieldWidths : function (reason) {
    // o appropriate time to get field widths:
	// LG w/ header:
	// - initially: right after we draw the header
	// - after setFields() header rebuilt: right after we rebuild the header
	// LG w/o header:
	// - initially: anytime before the body is drawn
	// - after setFields(): anytime before the body is redrawn

	var sizes = this._getCalculatedFieldWidths();
	
	// When autoFitFieldWidths is true, we may need to tweak these values to either
	// fill the available space, or clip certain fields if we're overflowing the
	// available space.
    if (this.autoFitFieldWidths && !this._calculatingAutoFitFieldWidths) {
        this._calculatingAutoFitFieldWidths = true;
        var unfrozenWidths = sizes.duplicate(),
            frozenWidths = null;
            
        if (this.frozenFields != null) {
            var left = this.freezeLeft;
            if (left) {
                frozenWidths = unfrozenWidths.slice(0, this.frozenFields.length);
                unfrozenWidths = unfrozenWidths.slice(this.frozenFields.length);
            } else {
                frozenWidths = unfrozenWidths.slice(this.frozenFields.length);
                unfrozenWidths = unfrozenWidths.slice(0, this.frozenFields.length);
            }
        }
        
        var availableSpace = this.getAvailableFieldWidth(),
            totalSize = unfrozenWidths.sum();

        var unfrozenSpace = availableSpace;
        if (frozenWidths != null) unfrozenSpace -= frozenWidths.sum();

                
        
        // Case 1: the fields don't fill the available space.
        // Expand the autoFitExpandField to fill the available space in the body.
        // Note: We don't auto-expand frozen fields - that would require resizing the
        // frozen body as well. The getAutoFitExpandField() method already handles not
        // returning frozen fields.
        // NOTE: If we're fitting to data, but data is currently loading, don't expand a field
        // now - wait until we redraw with loaded data. Otherwise we don't really know the
        // rendered sizes of all fields, so we won't know how much to expand the expansion field
        // by. Then when data arrives and the other fields all resize, we end up rendering the
        // expansion field potentially too wide since the other fields may now overflow available
        // space.
        
        var validData = true;
        if (this.autoFitWidthApproach != "title") {
            var dA = this.getDrawArea();
            if (!this.data || Array.isLoading(this.data.get(dA[0]))) {
                validData = false;
            }
        }
        if (totalSize < unfrozenSpace && validData) {
            var expandField = this.getAutoFitExpandField();
            if (expandField) {
                // we want to update the sizes array (includes both frozen and
                // unfrozen fields) so get the global fieldNum for the expand field
                // and update that value.
                var expandFieldNum = this.getFieldNum(expandField);
                var diff = unfrozenSpace - totalSize;
                sizes[expandFieldNum] += diff;
                
                // If we're showing a header for the field we have to resize
                // the button too so it stays in synch with the sizes array we
                // return (which will be applied to the body)
                var button = this.getFieldHeaderButton(expandFieldNum);
                if (button && button.isDrawn()) {
                    button.setWidth(sizes[expandFieldNum]);
                }
            }
        // case 2: the auto-fit fields are overflowing the available space, clip them
        // if appropriate.
        // Note: we don't clip frozen fields - that would require resizing the actual bodies.
        } else if (totalSize > unfrozenSpace && this.autoFitClipFields != null) {
            
            // If any autoFitFields are marked as clippable, and we're now overflowing
            // horizontally, we want to re-run stretchResize logic ignoring 
            // calculated autoFit for those fields -- just let them take up the available
            // space.
            // If we're not showing a header this is easy
            // If we are (and autoFitWidthApproach includes the header title), we need to
            // have the header re-run it's logic so it takes account of other (non clippable)
            // fields' title-widths
            // This means reset it's policy to fill, resize our fields to "*", then reflow,
            // reset the policy back to whatever it was before (probably "none")
            var policy;
            if (this.header && this.header.isDrawn()) {
                 policy = this.header.hPolicy;
                 this.header.hPolicy = "fill";
            }
            var clipFields = this.autoFitClipFields;
            for (var i = 0; i < clipFields.length; i++) {
                var field = this.getField(clipFields[i]),
                    fieldNum = this.getFieldNum(field);
                if (field == null || fieldNum < 0) continue;
                
                // Don't attempt to clip frozen fields since that would require resizing
               // the frozen body.
                if (field.frozen) {
                    this.logInfo("auto-fitting field:" + field.name + 
                        " is present in the autoFitClipFields array for this grid, but is" +
                        " currently frozen. This is not supported - the field will not be clipped.",
                        "frozenFields");
                    continue;
                }

                
                // deleting the calculated autoFitWidth ensures that when 
                // _getCalcualtedFieldWidths runs stretchResizePolicy will simply resize
                // the row to fit if possible, or if a header is showing, the header reflow
                //.will achieve the same result.
                delete field._calculatedAutoFitWidth;
                var header = this.getFieldHeader(fieldNum);
                if (header && header.isDrawn()) {
                    button = header.getMember(this.getLocalFieldNum(fieldNum));
                    
                    button.setWidth(field.width || "*");
                    
                    button.setOverflow("hidden");
                }
            }
            if (this.header && this.header.isDrawn()) {
                var reflowReason = this._$gettingFieldWidths;
                if (reason != null) reflowReason += reason;
                this.header.reflowNow(reflowReason);
                this.header.hPolicy = policy;
                if (this.frozenHeader) {
                    this.frozenHeader.hPolicy = "fill";
                    this.frozenHeader.reflowNow(reflowReason);
                    this.frozenHeader.hPolicy = policy;
                }
            }
//            this.logWarn("after reflow..." + this.header.getMemberSizes());
            
            // rerun the method to get the stretch-resize calculated widths
            // (will run the appropriate logic based on whether or not we're showing
            // headers
            sizes = this._getCalculatedFieldWidths();
        }
        this._calculatingAutoFitFieldWidths = false;
    }
    
//    this.logWarn("getFieldWidths() ultimately gave sizes:" + sizes);
    
    return sizes;
    
},
_$gettingFieldWidths:"Getting listGrid fieldWidths. ",

// helper for getFieldWidths() - returns the stretch-resize calculated widths
// (based on the header if appropriate).
_getCalculatedFieldWidths : function () {
    
	
    

    var header = this.header;
    if (isc.isA.Layout(header) && header.isDrawn()) {
//         this.logWarn("using header-based field widths");
        
        // Force an immediate redraw of any dirty buttons.
        // This is required to ensure sizes are correct -- if redrawOnResize is true for
        // the button or label with overflow:"visible", and setWidth() is called on it
        // the redraw isn't immediate - we need to force a redraw now if it hasn't occurred
        // so getVisibleWidth() returns the new size.
        var buttons = header.members;
        for (var i = 0; i < buttons.length; i++) {
            if (buttons[i].isDirty()) buttons[i].redraw();
            if (buttons[i].label != null && buttons[i].label.isDirty()) buttons[i].label.redraw();
        }

    	// derive field widths from header sizes
        var sizes = header.getMemberSizes();
        if (sizes.length > 0) {
            if (this.allowMismatchedHeaderBodyBorder) {
                
                var leftHeaderBorder = header.getLeftBorderSize() + header.getLeftMargin(),
                    rightHeaderBorder = header.getRightBorderSize() + header.getRightMargin();
                if (leftHeaderBorder != 0) {
                    sizes[0] += leftHeaderBorder;
                }
            }

            var totalSize = sizes.sum(),
                availableSpace = header.getInnerWidth();
           
            if (this.allowMismatchedHeaderBodyBorder) {
                var overflowAmount = totalSize - availableSpace;
                if (overflowAmount > 0) {
                    sizes[sizes.length-1] += Math.min(overflowAmount, rightHeaderBorder);
                }
            }
        }
        if (this.frozenFields) {
            var frozenWidths = this.frozenHeader.getMemberSizes();
            sizes.addListAt(frozenWidths, this.freezeLeft() ? 0 : sizes.length);
        }
        
        
    } else {
//        this.logWarn("using stretchResize-based field widths");
        var sizes = this.getStretchResizeWidths();
     
    }
    return sizes;
    
},

getStretchResizeWidths : function () {
    if (this.fields == null) return [];
    
    var widths = this.fields.getProperty("width"),
        autoFitWidths = this.fields.getProperty("_calculatedAutoFitWidth");
    for (var i = 0; i < widths.length; i++) {
        if (autoFitWidths[i] != null) widths[i] = autoFitWidths[i];
    }

    // no header: derive field widths via stretch resize policy on widths set in fields
    return isc.Canvas.applyStretchResizePolicy(
                widths,
                (this.innerWidth != null ? this.innerWidth : this.getAvailableFieldWidth()),
                this.minFieldSize
                );
},

getAvailableFieldWidth : function () {

    var width = ((this.autoFitData == "both" || this.autoFitData == "horizontal") 
                    ? this.getVisibleWidth() : this.getWidth()) - this.getHMarginBorder();
	// leave a gap for the scrollbar if vertical scrolling is on or if we're configured to
	// always leave a gap
    var leaveGap = this._shouldLeaveScrollbarGap();
    
    if (leaveGap) {
        width -= this.body ? this.body.getScrollbarSize() : this.getScrollbarSize();
    }
    return width;
},



//> @method ListGrid.getFieldWidth()
// Returns a numeric value for the width of some field within this listGrid.
// @param fieldNum (Number | String) Index or fieldName of field for which width is to be determined.
// @return (Number) width of the field in px, or null if the width can't be determined
// @visibility external
//<
getFieldWidth : function (fieldNum) {
    fieldNum = this.getFieldNum(fieldNum);
    if (fieldNum == -1 || !this.fields || fieldNum >= this.fields.length) return null;
    
    if (this.body != null) return this.getColumnWidth(fieldNum);
    return this.getFieldWidths()[fieldNum];
},

_adjustFieldSizesForBodyStyling : function (sizes, vertical) {
    if (vertical == null) vertical = false;
    
    if (sizes == null || sizes.length == 0) return sizes;
    
    
    if (!this.body) {
        
        return sizes;
    }
    
    // Adjust sizes of first and last field to account for styling on the body.
    
    sizes[sizes.length -1] = 
        this._adjustLastFieldForBodyStyling(sizes[sizes.length-1], sizes.sum(), vertical);
    sizes[0] = this._adjustFirstFieldForBodyStyling(sizes[0], vertical);
    
    return sizes;
},

_adjustFirstFieldForBodyStyling : function (size, vertical) {
    if (!this.body) return size;
    
    // always knock the left (or top) border off the first field, otherwise everything starts
    // shifted too far right (or down)
    var bodyStartAdjust = (vertical ? this.body.getTopBorderSize() + this.body.getTopMargin()
                                    : this.body.getLeftBorderSize() + this.body.getLeftMargin());
    if (bodyStartAdjust != 0) size -= bodyStartAdjust;
    
    return Math.max(0, size);
},

_adjustLastFieldForBodyStyling : function (size, totalFieldsWidth, vertical) {
    if (!this.body) return size;

    // Figure out whether all the sizes will expand as far or further than the bottom or right
    // edge. If so we want to knock off the end border from the last col or row.
    var bodyEndAdjust = (vertical ? this.body.getBottomBorderSize() + this.body.getBottomMargin()
                                  : this.body.getRightBorderSize() + this.body.getRightMargin());
    
    if (bodyEndAdjust != 0) {
        
        var overflowAmount = totalFieldsWidth - 
            (vertical ? this.body.getInnerHeight() : this.body.getInnerWidth());
        if (overflowAmount > 0) {
            size -= Math.min(overflowAmount, bodyEndAdjust);
        }
    }
    return Math.max(size,1);
},

setBodyFieldWidths : function (sizes) {
	// set the _fieldWidths array to the list passed in
    this._fieldWidths = sizes;
    
    // adjust the first / last column width for any left/right border or margin on the body.
    
    var origSizes = sizes;
    if (this.allowMismatchedHeaderBodyBorder) {
        if (isc.isAn.Array(sizes)) {
            sizes = this._adjustFieldSizesForBodyStyling(sizes.duplicate());
        }
    }

    // if we have frozenFields, separate out the sizes for those columns
    var frozenFields = this.frozenFields;
    if (frozenFields) {
        var frozenWidths = this.getFrozenSlots(sizes);
        sizes = this.getUnfrozenSlots(sizes);
        //this.logWarn("frozen widths: " + frozenWidths + ", remaining widths: " + sizes);
        this.frozenBody.setColumnWidths(frozenWidths);

        var freezeWidth = frozenWidths.sum();
        // this will automatically cause the main body to size to fill remaining space
        this.frozenBody.setWidth(freezeWidth);
         
        this.frozenBody._userWidth = freezeWidth;
        if (this.frozenHeader) this.frozenHeader.setWidth(freezeWidth);
    }

	// give the GridRenderer new fieldWidths
	// will mark the body for redraw.
    if (this.body != null) this.body.setColumnWidths(sizes);
	// If we're showing the edit form, update the widths of the form items
	
    if (this._editorShowing) {
        var items = this._editRowForm.getItems(),
            record = this.getRecord(this.getEditRow()),

        	// This method gives the widths of form items for every column - since
        	// we only create a subset of form items, we won't use every one.
            
            completeFormFieldWidths = this.getEditFormItemFieldWidths(record);

        for (var i = 0; i < items.length; i++) {
            var colNum = items[i].colNum;

            if (items[i].width != completeFormFieldWidths[colNum]) 
                items[i].setWidth(completeFormFieldWidths[colNum]);
        }
    }
},


// createChildren fired from 'prepareForDraw()'

createChildren : function () {
	// create the header and body
	if (this.showHeader && this.headerHeight > 0 && !this.header) this.makeHeader();
    if (this.showFilterEditor && !this.filterEditor) this.makeFilterEditor();
    this.createBodies();

    
    if (this.header) this.body.moveAbove(this.header);
    if (isc.Browser.isMoz && isc.Browser.geckoVersion >= 20051111) {
        if (this.header) {
            this.body.mozOutlineOffset = "0px";
            if (this.body.mozOutlineColor == null) 
                this.body.mozOutlineColor = this.mozBodyOutlineColor;
        } else {
            if (this.body.mozOutlineColor == null) 
                this.body.mozOutlineColor = this.mozBodyNoHeaderOutlineColor;
            this.body.mozOutlineOffset = "-1px";
        }
    }


    if (this.showGridSummary && (!this.summaryRow || this.summaryRow.masterElement != this))
        this.showSummaryRow();
    
    
    this._useNativeTabIndex = false;
},

layoutChildren : function (reason,b,c) {

    
    isc.Canvas._instancePrototype.layoutChildren.call(this, reason,b,c);    
    //this.Super("layoutChildren", arguments);
    if (this.body != null) {
        // This method handles resizing the body if necessary
        this._updateFieldWidths(reason, b,c);
        // We suppress autoDraw when we add the body to ourselves as a child as part of
        // createBodies.
        // Now that we've sized it, etc - actually draw it.
        if (this.isDrawn() && !this.body.isDrawn()) this.body.draw();
        // Every time we layout children (may be due to body resize or body scroll change), if
        // we're showing the frozen body, ensure it's the correct (explicit) height
        if (this.frozenBody) {
            var adjustForHScroll = this.shrinkForFreeze && this.body.hscrollOn;
            var height = this.bodyLayout.getInnerHeight();
            if (adjustForHScroll) height -= this.body.getScrollbarSize();
            this.frozenBody.setHeight(height);
            if (this.frozenBody._needRowRedraw()) this.frozenBody.markForRedraw("height changed");
            // this will avoid the layout from scrapping this specified height on resize etc
            this.frozenBody._userHeight = height;
        }
    }
},


_updateFieldWidths : function (reason, b,c) {
	// don't do anything until we've created our children (eg resized before draw())
    if (this.body == null) return;
    
	

    // wipe out fieldWidths on resize so they'll be recalculated.  
	
    this._fieldWidths = null;
    // If any fields has autoFitWidth set to true, calculate the auto-fit size for the
    // column apply it to the field
    // Note that we only care about the cases where we're fitting to the body content - if
    // we're fitting to the header this is handled by simply setting overflow to "visible"
    // on the header button.
    
    
    if (this.fields && !this.fields._appliedInitialAutoFitWidth && !this.skipAutoFitWidths) {
        var autoFitFieldWidths = this.getAutoFitValueWidths(null, true);
        if (autoFitFieldWidths == null) {
            this.fields.setProperty("_calculatedAutoFitWidth", null);
        } else {
            for (var i = 0; i < this.fields.length; i++) {
                var field = this.fields[i];
                if (autoFitFieldWidths[i] == null) {
                    field._calculatedAutoFitWidth = null;
                    continue;
                }
                var minWidth = field.width;
                if (!isc.isA.Number(minWidth)) minWidth = this.minFieldWidth;
                if (!isc.isA.Number(minWidth)) minWidth = 1;
                if (minWidth < autoFitFieldWidths[i]) {
                    field._calculatedAutoFitWidth = autoFitFieldWidths[i];
                    // update the header if there is one
                    // Note: If autoFitWidthApproach is "both", the header title can
                    // still overflow this new specified size (giving us the desired behavior
                    // of fitting to the larger of the 2 drawn sizes)
                    var headerButton = this.getFieldHeaderButton(i);
                    if (headerButton != null) {
                        headerButton.setWidth(autoFitFieldWidths[i]);
                        headerButton.parentElement.reflow();
                    }
                } else if (field._calculatedAutoFitWidth != null) {
                    field._calculatedAutoFitWidth = null;
                    
                    var headerButton = this.getFieldHeaderButton(i);
                    if (headerButton != null) {
                        headerButton.setWidth(minWidth);
                        headerButton.parentElement.reflow();
                    }
                }
                    
            }
            // Hang a flag on the array to avoid re-calculating the width every time we
            // run stretchResizePolicy, etc
            this.fields._appliedInitialAutoFitWidth = true; 
            
        }
        
    }
    
    var innerWidth = this.getAvailableFieldWidth(),
        innerWidthChanged = (innerWidth != this.innerWidth);
    this.innerWidth = innerWidth;

    //this.logWarn("total columns width: " + innerWidth + 
    //             (this.body ? ", vscrollon: " + this.body.vscrollOn : ""));
    
    var header = this.header,
        headerHeight = (this.showHeader ? this.headerHeight : 0);
    if (header != null) {
        // place the header
        // in RTL, scrollbar is on left
        var left = (this.isRTL() && this._shouldLeaveScrollbarGap() ? this.body.getScrollbarSize() : 0);            

    	
        header.hPolicy = "fill";
        if (this.frozenHeader) this.frozenHeader.hPolicy = "fill";
        var headerWidth = this.innerWidth;
        
        if (!this.leaveScrollbarGap && header.isDrawn() && 
            headerWidth != header.getWidth() && reason == "body scroll changed") {
        	
        	//this.logWarn("header changing size" + this.getStackTrace());
            if (this._settingBodyFieldWidths || !this.resizeFieldsForScrollbar) {                
                header.hPolicy = "none";
            }
        }
        var headerLayout = this.headerLayout || header;
        headerLayout.setRect(left, 0, headerWidth, headerHeight);
    	// if we're in the middle of the initial drawing process, draw the header now so we can
    	// get fieldWidths from it to give to the body before it draws.  Also draw the header
    	// if we're completely drawn and it's undrawn because it was just recreated.
        if (!header.isDrawn() && (reason == "initial draw" || this.isDrawn())) {
            if (!this.frozenFields) {
                header.draw();
            } else {
                // running getFieldWidths before the header is drawn
                // to determine how much space will be required for the frozen header
                // (see _getCalculatedFieldWidths for explanation)
                var fieldWidths = this.getFieldWidths(reason),
                    frozenWidths = this.getFrozenSlots(fieldWidths);
                this.frozenHeader.setWidth(frozenWidths.sum());
                this.headerLayout.draw()
            }
        }
        header.hPolicy = "none";
        if (this.frozenHeader) this.frozenHeader.hPolicy = "none";

        if (this.sorter) {
            // move the sorter button over if necessary    
            this.sorter.setLeft(this.getSorterLeft());
            // If this.showSort is unset, the sort button's visibility is determined by
            // leaveScrollbarGap, and whether the body has a vertical scrollbar.
            // This may have changed, so ensure the sorter is visible or hidden as appropriate.
            var showSorter = this._showSortButton();
            this.sorter.setVisibility(
                this._showSortButton() ? isc.Canvas.INHERIT : isc.Canvas.HIDDEN
            );
        }
    }

	// if we haven't figure out field widths, this is the time to get them, since the header
	// has been drawn if we have one, and the body is yet to be drawn.  
	// NOTE: if we are clear()d and then redrawn, and we have a header, we don't want to get
	// sizes from the header while it's undrawn, as it will not run the layout policy while
	// undrawn.
    if ((!this._fieldWidths || innerWidthChanged) &&
        ((header && header.isDrawn()) || headerHeight == 0)) 
    {
        var fieldWidths = this.getFieldWidths(reason);
        this.setBodyFieldWidths(fieldWidths);
        

        // update the filter editor field widths too [only need to worry about the body]
        if (this.filterEditor && this.filterEditor.body != null) {
            // Duplicate the fieldWidths - we don't want the modifications to this item's array
            // to directly impact the record editor's field widths array
            this.filterEditor.setBodyFieldWidths(fieldWidths.duplicate());
        }
        
        // update the summaryRow (if visible)
        if (this.summaryRow && this.showGridSummary && this.summaryRow.body != null) {
            this.summaryRow.setBodyFieldWidths(fieldWidths.duplicate());
        }
        
    	if (this.logIsDebugEnabled("layout")) {
            this.logDebug("new field widths: " + this._fieldWidths, "layout"); 
        }
    }
    
    // We rely on this method to resize the body to fit the available space, but a body resize is 
    // not actually required every time this method runs - frequently the desired body size is
    // unchanged.
    // 
    // If autoFitData is enabled, the body may already be sized larger than the calculated size
    // below (if the data overflows the specified space). In this case resizing back to the
    // specified size, then allowing adjustOverflow to re-fit to data will cause a flash.
    // If the size and space taken up by the data is unchanged, suppress this resize so we
    // don't get this flash (since we'll end up back at the same size anyway).
    //
      
    if (!this._bodyResizeReasons) {
        this._bodyResizeReasons = {}
        this._bodyResizeReasons[this._$resized] = true;
        
        this._bodyResizeReasons[this._$toggleFrozen] = true;
        this._bodyResizeReasons[this._$setFields] = true;
        this._bodyResizeReasons[this._$headerHeightChanged] = true;
        this._bodyResizeReasons[this._$headerVisibilityChanged] = true;
    }
    
    // if this is the initial rendering of the body we always need to size it out
    // otherwise only resize if we know the size needs to be updated
    var shouldResizeBody = this.autoFitData == null ||
                            !this.body.isDrawn() || 
                            this._bodyResizeReasons[reason];

    if (shouldResizeBody) {
        // how big should we make the body?
        this.bodyHeight = Math.max(1, (this.getInnerHeight() - headerHeight));
        this.bodyWidth = Math.max(1, this.getInnerWidth());
    
        var bodyLayout = this.bodyLayout || this.body;        
        bodyLayout.setRect(0, headerHeight,
                           this.bodyWidth, this.bodyHeight);
        // force an immediate redraw of the body if it's dirty.
        // This allows overflow:visible bodies to resize immediately
        if (this.body.isDirty() && !this.body._redrawing && !bodyLayout._redrawing) {
        
            bodyLayout.redraw();
        }
    }
        
},

// getAutoFitValueWidths: Calculates the auto-size width based on rendered field values
// Returns a sparse array of field sizes indexed by fieldNum in this.fields.
// By default expects no params and returns only auto-size width of the fields marked with
// autoFitWidth:true.
// If fields are passed in directly we'll calculate sizes for those fields only (regardless of
// any per-field autoFitWidth settings)
getAutoFitValueWidths : function (fields, checkApproach) {
    
    if (this.data == null) return;
    if (isc.isA.ResultSet(this.data)) {
        var drawArea = this.body.getDrawArea();
        if (!this.data.rangeIsLoaded(drawArea[0], drawArea[1])) {
            // If we're autofitting all fields, and data is loading, mark for resize
            // again - will occur on redraw from dataChanged/dataArrived
            if (fields == null) {
                this.updateFieldWidthsForAutoFitValue("Delayed resize pending data load");
                
            }
            return;
        }
    }
    
    var noAutoFit = (fields == null);
    if (fields == null) {
        fields = [];
        for (var i = 0; i < this.fields.length; i++) {
            var field = this.fields[i];
            if (this.shouldAutoFitField(field)) {
                
                // checkApproach - if passed don't collect valueWidths unless the field's
                // autoFitWidthApproach is 'value' or 'both'
                if (!checkApproach || this.getAutoFitWidthApproach(field) != "title") {
                    
                    fields.add(field);
                    noAutoFit = false;
                }
            }
        }
    }
    if (noAutoFit) return;
    
        
    var frozenAutoFit = [],
        unfrozenAutoFit = [];
    for (var i = 0; i < fields.length; i++) {
        var field = fields[i],
            fieldNum = this.getFieldNum(field);
            
        if (field.frozen) {
            frozenAutoFit.add(fieldNum);
        } else {
            unfrozenAutoFit.add(fieldNum);
        }
    }
    
    
    // widths will be a sparse array containing just the calculated field widths
    var widths = [];
    for (var i = 0; i < this.fields.length; i++) {
        if (fields.contains(this.fields[i])) {
            widths[i] = this.getDefaultFieldWidth(this.fields[i]);
        }
    }
    return widths;
},

//> @method listGrid.getDefaultFieldWidth()
// Method to calculate and return the default width of a field. This method is called
// to calculate the size of each field's content as part of the 
// +link{listGrid.autoFitFieldWidths,field auto fit} behavior.
// <P>
// The default implementation varies by +link{ListGridFieldType,field type}. 
// For fields of type <code>"icon"</code>, or fields which show only a 
// +link{ListGridField.valueIcons,valueIcon} as a value, and for boolean fields which
// show a checkbox value, the width will be calculated based on the icon size and
// +link{listGrid.iconFieldPadding}. For other fields the width will be calculated
// based on the rendered width of content. Note that for 
// <code>"image"</code> type fields, this method will rely on the +link{listGridField.imageWidth}
// being specified.
//
// @param field (ListGridField) Field for which the size should be determined
// @return (Integer) default size required for the field's content.
//
// @visibility external
//<
 
iconPadding:2,
getDefaultFieldWidth : function (field) {

    // special cases where we can avoid writing out and measuring content    
    if (field.type == "icon") {
        return (field.iconWidth || field.iconSize) + 2*this.cellPadding + 2*this.iconPadding;
    } else if (this.showValueIconOnly(field)) {
        return this.getValueIconWidth(field) + (2* this.cellPadding) +
                              this.getValueIconRightPadding(field) + 
                              this.getValueIconLeftPadding(field);
    }

    
    var localFieldNum = this.getLocalFieldNum(this.getFieldNum(field));
    var width;
    if (field.frozen) {
        
        width = this.frozenBody.getColumnAutoSize(localFieldNum);
    } else {
        width = this.body.getColumnAutoSize(localFieldNum);
    }
    if (width == null) width = field.width;
    return width;
},

getAutoFitWidthApproach : function (field) {
    if (field.autoFitWidthApproach != null) return field.autoFitWidthApproach;
    return this.autoFitWidthApproach;
},
    
    

draw : function (a,b,c,d) {
    if (isc._traceMarkers) arguments.__this = this;
	if (!this.readyToDraw()) return this;

    // create children and set up fields if not already set up
    this.prepareForDraw();

	// call the superclass draw routine to do the actual drawing
	this.invokeSuper(isc.ListGrid, "draw", a,b,c,d);

	// get the actual rendered sizes of the columns
	this.body.getColumnSizes();

    
    for (var i = 0; i < this.bodies.length; i++) {
        var body = this.bodies[i];
        if (body._embeddedComponents) this.markForRedraw();
    }

	// if the sortFieldNum is specified, tell the header about it
    var sortFieldNum = this._getSortFieldNum();
	if (this.header && sortFieldNum != null) {
        var header = this.getFieldHeader(sortFieldNum),
            sortButton = this.getFieldHeaderButton(sortFieldNum);
        header.selectButton(sortButton);
    }

	// scroll the header along with the list (necessary for textDirection == RTL)
    this.bodyScrolled();
    
    return this;
},

// should we show inactive editors for every record - returns true if we have any data and
// alwaysShowEditors is true at the grid or field level [can check specific fields]
_alwaysShowEditors : function (field, ignoreTotalRows) {
    // ignoreTotalRows -- useful to test whether we would ever show editors in every cell
    // as opposed to whether we currently do.
    
    // if we have no data don't show any edit cells
    if (!ignoreTotalRows && this.getTotalRows() == 0) return false;
    
    if (this.alwaysShowEditors) return true;
    var fields = field ? [field] : this.fields;
    if (fields) {
        for (var i = 0; i < fields.length; i++) {
            if (fields[i].alwaysShowEditors) {
                return true;
            }
        }
    }
    return false;
},

prepareForDraw : function () {
	

	// call setFields() for the first time, if it hasn't already been done 
    if (this.completeFields == null) this.setFields(this.fields);

    // if alwaysShowEditors is true, ensure we are editing
    if (this.getEditRow() == null) {
        var forceEditing = this._alwaysShowEditors();
        if (forceEditing) this.startEditing(null,null,true,null,true);
    }

    // if a grouping is already set, apply it with groupBy
    if (this.groupByField) {
        var fields;
        if (isc.isA.Array(this.groupByField)) {
            fields = this.groupByField;
        } else {
            fields = [ this.groupByField ];
        }            
        this.groupByField = null;

        // if we have no fields don't call groupBy yet - should occur on setFields
        //if (this.fields != null) 
        this.groupBy(fields);
    }

	// create the header and body. 
    this.createChildren();
},

//> @method listGrid.getGroupByFields()
// Get the current grouping of this listGrid as an array of fieldNames.
// <P>
// This method returns an arry containing the names of the field(s) by which this
// grid is grouped (either from +link{listGrid.groupByField} having been explicitly set or
// from a call to +link{listGrid.groupBy()}). If this grid is not currently grouped, this method
// will return null.
// @return (Array of String) Current grouping for this grid. If grouped by a single field an array
//  with a single element will be returned.
// @visibility external
//<

getGroupByFields : function () {
    var fields = this.groupByField;
    if (fields != null && !isc.isAn.Array(fields)) {
        fields = [fields];
    }
    return fields;
},

destroy : function (indirectDestroy) {
    if (this._dragLine) {
        this._dragLine.destroy();
        this._dragLine = null;
    }
    if (this._cornerMenu) this._cornerMenu.destroy();
    if (this._spanContextMenu) this._spanContextMenu.destroy();
    if (this.cellContextMenu) this.cellContextMenu.destroy();
    if (this._editRowForm) {
        this._editRowForm.destroy();
        delete this._editRowForm;
        delete this._editorShowing;
    }

	if (this.data){
        // if the data was autoCreated, destroy it to clean up RS<->DS links
        if (this.data._autoCreated && isc.isA.Function(this.data.destroy)) {
            this.data.destroy();
        } else {
            // ignore so we don't leak memory from the observation references
            this._ignoreData(this.data);
            delete this.data;
        }
    }
    if (this.selection) {
        this.destroySelectionModel();
    }
    if (this.selectionCanvas) this.selectionCanvas.destroy();
    if (this.selectionUnderCanvas) this.selectionUnderCanvas.destroy();
    if (this.rollOverCanvas) this.rollOverCanvas.destroy();
    if (this.rollUnderCanvas) this.rollUnderCanvas.destroy();
    
    this._dropODSData();

    this.Super("destroy", arguments);
},

//>	@method	listGrid.redrawHeader()	(A)
//			redraw the header
//		@group	drawing
//<
redrawHeader : function () {
	if (this.header) this.header.markForRedraw();
},

//>	@method	listGrid.getBaseStyle()
// Return the base stylename for this cell.  Has the following implementation by default:
// <ul>
// <li>If +link{listGrid.editFailedBaseStyle, this.editFailedBaseStyle} is defined, and the
//     cell is displaying a validation error return this value.</li>
// <li>If +link{listGrid.editPendingBaseStyle, this.editFailedPendingStyle} is defined, and
//     the cell is displaying an edit value that has not yet been saved (see 
//     +link{ListGrid.autoSaveEdits}) return this value.</li>
// <li>Otherwise return +link{ListGrid.recordBaseStyleProperty, record[listGrid.recordBaseStyleProperty]},
//     if defined, otherwise +link{ListGridField.baseStyle, field.baseStyle}.</li>
// </ul>
// If no custom style is found for the cell as described above, the default baseStyle will be
// returned. If +link{listGrid.baseStyle} is specified this will be used. Otherwise for
// grids showing fixed height rows which match +link{listGrid.normalCellHeight} 
// +link{listGrid.normalBaseStyle} will be used. For grids with variable, or modified
// cell heights, +link{listGrid.tallBaseStyle} will be used. 
// <P>
// Note also that
// enabling +link{listGrid.fastCellUpdates} will cause the <code>tallBaseStyle</code> to be
// used rather than +link{listGrid.normalBaseStyle}.
//
// @see getCellStyle()
//
// @param	record  (ListGridRecord)	Record associated with this cell. May be <code>null</code>
//                               for a new edit row at the end of this grid's data set.
// @param	rowNum	(number)	row number for the cell
// @param	colNum	(number)	column number of the cell
// @return	(CSSStyleName)	CSS class for this cell
// @visibility external
// @example replaceStyle
//<
getBaseStyle : function (record, rowNum, colNum) {

    
    if (this.canEdit == true && !this.isPrinting) {
        if (this.editFailedBaseStyle && this.cellHasErrors(rowNum, colNum))
            return this.editFailedBaseStyle;
    
        if (this.editPendingBaseStyle && this.cellHasChanges(rowNum, colNum, false))
            return this.editPendingBaseStyle;
    }
    
    if (record && this.recordBaseStyleProperty && record[this.recordBaseStyleProperty]) 
        return record[this.recordBaseStyleProperty];
    
    var field = this.getField(colNum);
    if (field && field.baseStyle) return field.baseStyle;
    
    if (field && field.type == "summary" && this.recordSummaryBaseStyle) 
        return this.recordSummaryBaseStyle;
    
    if (field && field.frozen && this.frozenBaseStyle) return this.frozenBaseStyle;
    
    if (this.isPrinting && (this.printBaseStyle != null)) return this.printBaseStyle;
    
    var baseStyle = this.baseStyle;
    if (baseStyle == null) {
        if (this.cellHeight != this.normalCellHeight || 
            this.fastCellUpdates || 
            !this.shouldFixRowHeight(record, rowNum) ||
            
            (record != null && record._embeddedComponents))
        {
            baseStyle = this.tallBaseStyle;
        } else {
            baseStyle = this.normalBaseStyle;
        }
    }
    return baseStyle;
},

getCellCSSText : function (record, rowNum, colNum) {
    // per-record cssText
    if (record) {
        var cssText = record[this.recordCSSTextProperty];
        if (cssText != null) return cssText;
    }

    var cssText;

    if (this.isEditable()) {
        if (this.editFailedBaseStyle == null && this.editFailedCSSText && 
            this.cellHasErrors(rowNum, colNum)) 
        {
            cssText = this.editFailedCSSText;
        } else if (this.editPendingBaseStyle == null && this.editPendingCSSText && 
            this.cellHasChanges(rowNum, colNum, false)) 
        {
            cssText = this.editPendingCSSText;
        }
    } 
    
    cssText = this.getRecordHiliteCSSText(record, cssText, this.getField(colNum));
    return cssText;

},

//>	@method	listGrid.getRawCellValue()
//		@group	data
//			return the raw data for one particular cell in the list
//
//		@param	record		(object)	a record in the data
//		@param	recordNum	(number)	number of that record (in case it's important for the output)
//		@param	fieldNum	(number)	number of the field to display
//
//		@return	(string)	raw value for this cell
//<

getRawCellValue : function (record, recordNum, fieldNum, isFieldName) {
    var field,
        fieldName,
        dataPath;

    if (isFieldName) {
        fieldName = fieldNum
        // when fieldName specified, search completeFields if available
        field = this.completeFields ? isc.Class.getArrayItem(fieldName, this.completeFields, this.fieldIdProperty) : this.getField(fieldName);
    } else {
        field = this.fields[fieldNum];
        fieldName = field[this.fieldIdProperty];
    }
     
    dataPath = field ? field.dataPath : null;

    if (fieldName == null && dataPath == null) return this.emptyCellValue;
    // Note even if fieldName is set, field is not guaranteed to be present - this could
    // be a displayField which is being rendered in a different field's cell

    var editValue, value, undef;
    if (this.rowEditNotComplete(recordNum)) {
        editValue = this._getEditValue(recordNum, fieldNum);
    }

    if (editValue !== undef) {
        // This ensures that the value for the cell as it is currently stored for local editing
        // is displayed, rather than the saved value of the cell.
        
        return editValue;
    } else {
        // record can be null if there's no record for this cell
        if (record == null) return this.emptyCellValue;

        // let the dataset return a value if it supports doing so
        
        if (this.data.getFieldValue && field) {       
            value = this.data.getFieldValue(record, fieldName, field);
           
        } else if (record.ownerDocument && field) {
            // if working with XML, fetch values via XPath selectors if so configured.  (Needed
            // for viewing XML elements when not dataBound, but could be moved to be a built-in
            // feature of both List and native Array)
            
            value = isc.xml.getFieldValue(record, fieldName, field);
        } else {  
            if (dataPath != null) {
                value = isc.Canvas._getFieldValue(this._trimDataPath(dataPath), record, 
                                                                this, true);
            } else {
                value = record[fieldName];
            }
        }
    }
    // show the result of getRecordSummary for summary fields
    // Note that if shouldApplyRecordSummaryToRecord is true it's already stored on the record
    // object so we've already picked it up
    if (this.shouldShowRecordSummary(field,record) && 
        !this.shouldApplyRecordSummaryToRecord(field)) 
    {
        value = this.getRecordSummary(recordNum, field);
    }
    
    // if the field has a 'getRawCellValue' attribute
    if (field && field.getRawCellValue) {
        // CALLBACK API:  available variables:  "viewer,record,recordNum,field,fieldNum,value"
        // Convert a string callback to a function
        isc.Func.replaceWithMethod(field, "getRawCellValue", "viewer,record,recordNum,field,fieldNum,value");

        // call it as a function (returns string | null)
        value = field.getRawCellValue(this, record, recordNum, field, fieldNum, value);
	}

    // handle formula and summary fields
    if (field) {
        if (field.userFormula) value = this.getFormulaFieldValue(field, record);
        // note: actual value computed later (after formatters have been applied
        // to raw value)
        if (field.userSummary) this.getSummaryFunction(field);
    }
    return value;
},

// This method trims the component's dataPath off of the field, if the field dataPath reiterates
// it.  This puts the field dataPath in the correct context, since this ListGrid's data member
// will be the List of records corresponding to a multiple:true subfield of the overall data
// structure, NOT the overall data structure itself
_trimDataPath : function (dataPathParam) {
    var dataPath = dataPathParam.trim(isc.Canvas._$slash);
    if (!dataPath.contains(isc.Canvas._$slash)) return dataPathParam;
    var compDataPath = this.getFullDataPath();
    if (compDataPath == null || compDataPath == "") return dataPath;
    compDataPath = compDataPath.trim(isc.Canvas._$slash);
    var left = compDataPath.split(isc.Canvas._$slash);
    var right = dataPath.split(isc.Canvas._$slash);
    for (var i = 0; i < left.length; i++) {
        if (left[i] != right[i]) {
            break;
        }
    }
    
    if (i == 0) return dataPathParam;
    
    var trimmedDataPath = "";
    for (var j = i; j < right.length; j++) {
        trimmedDataPath += right[j];
        if (j != left.length) trimmedDataPath += "/";
    }
    return trimmedDataPath;
},

// shouldShowRecordSummary - returns true if this is a summary field unless
// this is a summary row and there's an explicit summaryFunction (which takes precedence over
// any record summary function).

_$summary:"summary",
shouldShowRecordSummary : function (field, record) {
    if (field && field.type == this._$summary) {
        if (record[this.groupSummaryRecordProperty]) {
            return (field.summaryFunction == null && field.getGroupSummary == null);
        } else if (record[this.gridSummaryRecordProperty]) {
            return (field.summaryFunction == null && field.getGridSummary == null);
        }
        return true;
    }
    return false;
},

//>	@method	listGrid.getCellValue()   ([A])
//          Obtains the display value for a specific cell according to the given input 
//          parameters.<br>
//          To format the value displayed in the cell, make use of the 
//          <code>formatCellValue</code> methods rather than overriding this method directly.
//      @visibility external
//		@group	data
//
//		@param	record		(object)	the current record object
//		@param	recordNum	(number)	number of the record in the current set of displayed 
//                                      record (e.g. 0 for the first displayed record)
//		@param	fieldNum	(number)	number of the field in the listGrid.fields array
//
//      @see    method:ListGrid.formatCellValue
//		@return	(string)	display value for this cell
//<
_$HR:"<HR>",
getCellValue : function (record, recordNum, fieldNum, gridBody) {
	// If we're handed an empty record, we may be looking at a new edit row - if so get
	// the edit values instead
    if (record == null) {
        if (this.showNewRecordRow && this._isNewRecordRow(recordNum)) {
            return this.getNewRecordRowCellValue();
        }
        record = this._getEditValues(recordNum, fieldNum);
    } else {
        // Special cases:
        // if it's a separator row, return a horizontal rule
        
        if (record[this.isSeparatorProperty]) return this._$HR;
    
        // group controller node - write out the groupNodeHTML
        if (record._isGroup) {
            
            var currentField = this.fields[fieldNum],
                groupTitleField = this.getGroupTitleField(),
                isGroupTitleField;
                
            if (groupTitleField == null) {
                isGroupTitleField = currentField._isGroupTitleColumn;
            } else {
                isGroupTitleField = currentField.name == groupTitleField;
            }
            
            // If we're showing multiple fields and we're not showing a summary in the
            // header, explicitly empty all fields except the groupTitleField
            if (this.singleCellGroupHeaders() || isGroupTitleField) {
                return this.getGroupNodeHTML(record, gridBody);
            } else if (!this.showGroupSummaryInHeader) {
                return "&nbsp;";    
            }
        }
    }

	// get the actual data record
	var field = this.fields[fieldNum],
		value = null;

	// if field is null, we've shrunk the number of columns, so forget it
	if (field == null) return "";

    // If this is the checkboxField, write out the checkbox HTML
    
    if (this.isCheckboxField(field)) {
        var icon;
        if (!this.body.canSelectRecord(record)) {           
            // record cannot be selected but we want the space allocated for the checkbox anyway.
            icon = "[SKINIMG]/blank.gif";
        } else {
            // checked if selected, otherwise unchecked
            var isSel = this.selection.isSelected(record) ? true : false;
            icon = isSel ? (this.checkboxFieldTrueImage || this.booleanTrueImage)
                         : (this.checkboxFieldFalseImage || this.booleanFalseImage);
        }
        // if the record is disabled, make the checkbox image disabled as well
        if (record[this.recordEnabledProperty] == false) {
            icon = icon.replace("." , "_Disabled.");  
        }
        
        var html =  this.getValueIconHTML(icon, field);
       
        return html; 
      
    }
    
    // Determine whether we should be showing JUST a valueIcon
    // Do this before we figure out other HTML to avoid calculating stuff we're not going to 
    // actually use.
    var icon,
        iconOnly = this.showValueIconOnly(field),
        isEditCell;
	if (record != null) {
        
        
		if (record[this.singleCellValueProperty] != null) {
            return record[this.singleCellValueProperty];
        }
        
		if (Array.isLoading(record)) {
            
            if (!isc.Browser.isSafari || fieldNum == 0) {
                return this.loadingMessage;
            }
            return "&nbsp;";
        }
        
    	// If we are currently editing there are three possibilities for what we should write
    	// out for each cell:
    	// - This cell is being edited -- write out the edit form's element HTML
    	// - This cell has pending edit values, but is not currently being edited.
    	//   (this can happen if we're editing another cell in this row, or if an edit was not
    	//   successfully saved for this cell) -- display the edit values.
    	// - This cell should display the value for the record (default behavior)
    	// We catch the first case here, and write out the edit-form item into the cell if
    	// appropriate.
    	// The other two cases will be handled by 'getRawCellValue()', which will check for
    	// the presence of editValues on a cell.
    	
        isEditCell = (this._editorShowing && this._editRowNum == recordNum && 
                      (!this.editByCell || this._editColNum == fieldNum) &&
                      this.canEditCell(recordNum, fieldNum));
        if (isEditCell) {
            value = this.getEditItemCellValue(record, recordNum, fieldNum);
        // Checking for this._editorShowing would mean that when changing edit cell, 
        // hideInlineEditor would render the row with no inactive editors, and they
        // wouldn't get shown again on showEditForm for the new cell without another
        // row-refresh
        } else if (this._showInactiveEditor(fieldNum) && this.canEditCell(recordNum,fieldNum)) {
            value = this.getInactiveEditorCellValue(record, recordNum, fieldNum);
            // set isEditCell -- this will suppress the standard 'valueIcon' stuff
            isEditCell = true;
        } else {

        	// get the value according to the field specification
            var field = this.fields[fieldNum],
                displayFieldValue,
                pickUpDisplayFieldValue;
            
            // If a displayField was specified on the field, and no separate optionDataSource,
            // we should display the value for the record from the displayField rather than the
            // data field.
            if (field.displayField != null) {
                pickUpDisplayFieldValue = !field.valueMap && 
                                          !field.getCellValue && 
                                          this._useDisplayFieldValue(field);
                                          
                if (pickUpDisplayFieldValue) {
                    var displayField = field.displayField;
                    displayFieldValue = this.getRawCellValue(record, recordNum, displayField, true);
                }
                // Note: we still need to proceed through the rest of the logic, since the
                // underlying data value drives the valueIcon
            }
            value = this.getRawCellValue(record, recordNum, fieldNum);
     		// if the field has a 'getCellValue' attribute
        	// NOTE: this is deprecated - overriding 'getCellValue()' at the Grid level is very
        	// advanced, so for simple formatting (which is likely to be done at either the Field
        	// or Grid level) we provide 'formatCellValue()' methods instead.)
        	// Leave this code in place for Back-compat only.
    
        	//>!BackCompat 2005.1.12
    		if (field.getCellValue) {
    			// CALLBACK API:  available variables:  "viewer,record,recordNum,field,fieldNum,value"
    			// Convert a string callback to a function
    			isc.Func.replaceWithMethod(field, "getCellValue",
                                                 "viewer,record,recordNum,field,fieldNum,value");
    			value = field.getCellValue(this, record, recordNum, field, fieldNum, value);
    		}
        	//<!BackCompat

            
            icon = this.getValueIcon(field, value, record, recordNum);
          
            // if we're ONLY showing an icon we don't really need to figure out the text from
            // the valueMap.
            if (!iconOnly) {

                // if the field has an 'valueMap' parameter, treat the value as a key in the map
        		var valueMap = field.valueMap;
                if (valueMap) {
                    // NOTE: this can be really expensive, since we may eval the function that
                    // returns the value map for each record!!!
                    if (isc.isA.String(valueMap)) valueMap = this.getGlobalReference(valueMap);
                    // NOTE: don't look up displayed values in Array valueMaps, which just list legal
                    // values, and don't provide stored->displayed value mappings.
                    if (!isc.isAn.Array(valueMap)) {
                        if (isc.isAn.Array(value)) {
                            var displayValue = [];
                            for (var i = 0; i < value.length; i++) {
                                var dataVal = isc.getValueForKey(value[i], valueMap, value[i]);   
                                displayValue[displayValue.length] = dataVal;
                            }
                            value = displayValue;
                        } else {
                            value = isc.getValueForKey(value, valueMap);
                        }
                    }
                }
            }
            
            if (pickUpDisplayFieldValue) value = displayFieldValue;
        }
	}
    
    if (!isEditCell) {
        // If this cell isn't showing an editor we may need to write out an icon
        
        var iconHTML = null;
        
        if (icon != null) {                 
            iconHTML = this.getValueIconHTML(icon, field);
        }

        if (iconOnly) {
            // If there's no icon write out the empty cell value. This avoids us having
            // un-styled cells.
            if (!iconHTML || isc.isAn.emptyString(iconHTML)) iconHTML = this.emptyCellValue;
            value = iconHTML;
        } else {
            // use formatCellValue() to perform any additional formatting
            value = this._formatCellValue(value, record, field, recordNum, fieldNum);
            // apply hilites to capture htmlBefore/after
            var hilites = this.getFieldHilites(record, field);
            if (hilites) value = this.applyHiliteHTML(hilites, value);
            
            if (iconHTML) {
                if (field.valueIconOrientation != isc.Canvas.RIGHT)
                    value = iconHTML + value;
                else
                    value = value + iconHTML;
            }
        }
        
        // Only show error icon HTML if we're not showing an editor for the cell - otherwise
        // we'd get doubled error icons.
        if (this.isEditable() && this.showErrorIcons && this.cellHasErrors(recordNum, fieldNum)) {
            value = this.getErrorIconHTML(recordNum, fieldNum) + value;
        }        
    }
    // if current field is the groupTitleField, add indentation to the cell value
    var gtf = this.isGrouped ? this.getGroupTitleField() : null;
    if (gtf && this.fields[fieldNum].name == gtf) {
        var groupIndent = 
            isc.Canvas.spacerHTML(this.groupIndentSize + this.groupLeadingIndent, 1);
        value = groupIndent + value;    
    }
    return value;
},

// Formula/summary -related overrides from DBC
getTitleFieldValue : function (record) {
    var titleField = this.getDataSource().getTitleField(),
        title = this.getCellValue(record, this.getRecordIndex(record), 
            this.getFieldNum(titleField), this.body
    );

    if (!title || title == "") {
    	title = this.getRawCellValue(record, this.getRecordIndex(record), titleField, true);
	}
	return title;
},
getRawValue : function (record, fieldName) {
    var recordIndex = this.getRecordIndex(record),
        field = this.getSpecifiedField(fieldName);

    if (recordIndex < 0) return record[fieldName];
        
    if (record[field[this.fieldIdProperty]]!= null)return record[field[this.fieldIdProperty]];
	else return this.getRawCellValue(record, recordIndex, fieldName, true);
},
getFormattedValue : function (record, fieldName, value) {
    var recordIndex = this.getRecordIndex(record),
        field = this.getSpecifiedField(fieldName),
        fieldNum = this.getFieldNum(fieldName);

    if (recordIndex < 0) return value == null ? record[fieldName] : value;

    if (this.fieldIsVisible(field) && value == null)
        return this.getCellValue(record, recordIndex, fieldNum, this.body);
    
	if (value == null) value = this.getRawValue(record, fieldName);
	
	// if the field has an 'valueMap' parameter, treat the value as a key in the map
    var valueMap = field ? field.valueMap : null;
    if (valueMap) {
        // NOTE: this can be really expensive, since we may eval the function that
        // returns the value map for each record!!!
        if (isc.isA.String(valueMap)) valueMap = this.getGlobalReference(valueMap);
        // NOTE: don't look up displayed values in Array valueMaps, which just list legal
        // values, and don't provide stored->displayed value mappings.
        if (!isc.isAn.Array(valueMap)) {
            if (isc.isAn.Array(value)) {
                var displayValue = [];
                for (var i = 0; i < value.length; i++) {
                    var dataVal = isc.getValueForKey(value[i], valueMap, value[i]);   
                    displayValue[displayValue.length] = dataVal;
                }
                value = displayValue;
            } else {
                value = isc.getValueForKey(value, valueMap);
            }
        }
    }

    var ret = this._formatCellValue(value, record, field, recordIndex, fieldNum);
    return ret;

},

// get the width of the specified field or null
getSpecifiedFieldWidth : function (field) {
    field = this.getField(field);
    var fieldName = field[this.fieldIdProperty],
        members = this.header ? this.header.members : null,
        frozenMembers = this.frozenFields && this.frozenHeader ? this.frozenHeader.members : null,
        width;

    if (members || frozenMembers) {
        var member;
        
        if (members) member = members.find(this.fieldIdProperty, fieldName);
        if (!member && frozenMembers) {
            member = frozenMembers.find(this.fieldIdProperty, fieldName);
        }
        if (member && member._userWidth && isc.isA.Number(member._userWidth)) {
            width = member._userWidth;
        }
    }
    return width;
},

getValueIconHTML : function (icon, field) {
    
    var prefix = field.imageURLPrefix || field.baseURL || field.imgDir,
        suffix = field.imageURLSuffix,
        width = this.getValueIconWidth(field),
        height = this.getValueIconHeight(field),
        leftPad = this.getValueIconLeftPadding(field),
        rightPad = this.getValueIconRightPadding(field);
    
    if (suffix != null) icon += suffix;
                    
    var iconHTML = isc.Canvas._getValueIconHTML(icon, prefix, width, height,
                                      leftPad, rightPad,
                                      // no need for an ID
                                      null,
                                      // pass in the LG as an instance - required
                                      // for generating the path of the valueIcon src
                                      this);
    return iconHTML;
},

//>	@method	listGrid.getCellAlign()
// Return the horizontal alignment for cell contents. Default implementation returns 
// +link{listGridField.cellAlign} if specified, otherwise +link{listGridField.align}.
//
// @see getCellStyle()
//
// @param   record (listGridRecord) this cell's record
// @param	rowNum	(number)	row number for the cell
// @param	colNum	(number)	column number of the cell
// @return	(Alignment)     Horizontal alignment of cell contents: 'right', 'center', or 'left'	
// @visibility external
//<
getCellAlign : function (record, rowNum, colNum) {
    // single cells are aligned according to RTL - skip checkboxField (which needs center align)
    if (record && record[this.singleCellValueProperty] != null &&
        (!this.showSingleCellCheckboxField(record) || 
         !this.isCheckboxField(this.getField(colNum)))
       )
    {
        return this.isRTL() ? isc.Canvas.RIGHT : isc.Canvas.LEFT;
    }
    
    var fieldNum = colNum;
    var field = this.fields[fieldNum];
    if (!field) return isc.Canvas.LEFT;
    
    // formula fields are always right aligned - hard-code this in here
    if (field.userFormula || field.userSummary) {
        return this.isRTL() ? isc.Canvas.LEFT : isc.Canvas.RIGHT;
    }
        
    return (field.cellAlign || field.align);
    
},

// Helper method
// If we're showing a singleCellValue for a record, should we show the checkboxField to the
// left of the single value
showSingleCellCheckboxField : function (record) {
    return (this.getCurrentCheckboxField() != null) && record &&
            !record._isGroup && !record[this.isSeparatorProperty];
},

//>	@method	listGrid.getCellVAlign()
// Return the vertical alignment for cell contents.
// Expected values are: 'top', 'center', or 'bottom'
//
// @see getCellStyle()
//
// @param   record (listgridRecord) this cell's record
// @param	rowNum	(number)	row number for the cell
// @param	colNum	(number)	column number of the cell
// @return	(Alignment)     Vertical alignment of cell contents: 'right', 'center', or 'left'	
// @visibility external
//<
// Unset by default


// Helper method: 
// If a displayField was specified on some field, and no separate optionDataSource,
// we should display the value for the record from the displayField rather than the data field.
_useDisplayFieldValue : function (field) {
    return field && field.displayField && (field.displayField != field.name) &&
           (!field.optionDataSource || 
            isc.DS.get(field.optionDataSource) == this.getDataSource());
},                                            

// Value Icons
// ---------------------------------------------------------------------------------------

_valueIconStyleTemplate:[
    "style='margin-left:",  // [0]
    ,                       // [1] - icon padding
    "px;margin-right:",     // [2]
    ,                       // [3] - icon padding
    "px;'"
],

//> @method ListGrid.getValueIcon()
// Returns the appropriate valueIcon for a cell based on the field and the data value for the 
// cell. Default implementation returns null if +link{ListGridField.suppressValueIcon} is true
// otherwise looks at +link{ListGridField.valueIcons}.
// @param field (ListGridField) field associated with the cell
// @param value (any) data value for the cell's record in this field.
// @param record (ListGridRecord) record associated with this cell
// @return (SCImgURL) url for the icon
// @group imageColumns
// @visibility external
//<
// We need the rowNum for checking whether the cell is editable
// calculate this from the record object via findRowNum(), but also support being passed an
// explicit rowNum for critical path code 
getValueIcon : function (field, value, record, rowNum) {
    if (!field.valueIcons || field.suppressValueIcon) {
        if (this._formatBooleanFieldAsImages(field)) {
            var img = (value ? this.booleanTrueImage : this.booleanFalseImage);
            // if the cell can't be edited and can be toggled, make sure it displays the 
            // disabled checkbox icon
            rowNum = (rowNum != null) ? rowNum : this.findRowNum(record);
            var colNum = field.masterIndex;
            if (!this.canEditCell(rowNum, colNum) && field.canToggle) {
                img = isc.Img.urlForState(img, false, false, "Disabled");
            }
            // If no image was specified, still write out a blank gif into the slot - this
            // allows us to recognize events over the (invisible) icon for canToggle behavior
            if (img == null) img = isc.Canvas.getImgURL(isc.Canvas._blankImgURL);
            return img;
        }
        return null;
    }
    var icon = field.valueIcons[value];
    return icon;
},

// Helpers to get padding on each side of a field's valueIcon
getValueIconLeftPadding : function (field) {
    return (field && field.valueIconLeftPadding != null ? field.valueIconLeftPadding 
            : this.valueIconLeftPadding || 0);
},

getValueIconRightPadding : function (field) {
    return (field && field.valueIconRightPadding != null ? field.valueIconRightPadding 
            : this.valueIconRightPadding || 0);
},

// showValueIconOnly - returns true if the valueIcon should be displayed without any 
// text value for some field.
_$boolean:"boolean",
showValueIconOnly : function (field) {
    if (field.showValueIconOnly != null) return field.showValueIconOnly;
    // See discussion near showValueIconOnly docs.
    // If we have a valueIcons map, with no vmap, return text AND icon
    // if we have both valueIcons and a vmap, return just the icon
    // if we have no icon map, obviously return text (and icon, which is null)
    if (field.valueIcons != null && field.valueMap != null) return true;
    
    // If we are looking at a boolean field for which we want to show just the checkbox images
    // return true - otherwise false
    return this._formatBooleanFieldAsImages(field);

},

//> @method ListGrid.getValueIconWidth()
// If some field is showing valueIcons, this method returns the width those items should render
// Default implementation derives this from the first specified of 
// +link{ListGridField.valueIconWidth}, +link{ListGridField.valueIconSize}, 
// +link{ListGrid.valueIconWidth}, or +link{ListGrid.valueIconSize}
// @param field (object) field for which we're retrieving the valueIcon width
// @return (number) width for the icon
// @group imageColumns
// @visibility internal
//<
getValueIconWidth : function (field) {
    
    if (this.isCheckboxField(field)) return this._getCheckboxFieldImageWidth();
    if (this._formatBooleanFieldAsImages(field)) return this.booleanImageWidth;

    return (field.valueIconWidth != null ? field.valueIconWidth  :
                (field.valueIconSize != null ? field.valueIconSize : 
                    (this.valueIconWidth != null ? this.valueIconWidth : this.valueIconSize)));
},

//> @method ListGrid.getValueIconHeight()
// If some field is showing valueIcons, this method returns the height those items should render
// Default implementation derives this from the first specified of 
// +link{ListGridField.valueIconHeight}, +link{ListGridField.valueIconSize}, 
// +link{ListGrid.valueIconHeight}, or +link{ListGrid.valueIconSize}
// @param field (object) field for which we're retrieving the valueIcon height
// @return (number) height for the icon
// @group imageColumns
// @visibility internal
//<
getValueIconHeight : function (field) {
    if (this.isCheckboxField(field)) return this._getCheckboxFieldImageHeight();
    if (this.isExpansionField(field)) return this._getExpansionFieldImageHeight();
    if (this._formatBooleanFieldAsImages(field)) return this.booleanImageHeight;
    
    return (field.valueIconHeight != null ? field.valueIconHeight  :
                (field.valueIconSize != null ? field.valueIconSize : 
                    (this.valueIconHeight != null ? this.valueIconHeight : this.valueIconSize)));
},

// New record row: optional special row added to encourage new record entry
// ---------------------------------------------------------------------------------------

_isNewRecordRow : function (recordNum) {
    return this.showNewRecordRow && (recordNum == this.getTotalRows()-1);
},

//> @method ListGrid.getNewRecordCellValue()
//  Returns the contents to display in the new record row.
//  Note that this row displays a single cell spanning every column.
//  @return (string)    Value to display in new record row. Default 
//                      implementation returns <code>"-- Add New Row --"</code>
//<
getNewRecordRowCellValue : function () {
    return '<div align="center">' + this.newRecordRowMessage + '</div>';
},

// ---------------------------------------------------------------------------------------
getErrorIconHTML : function (rowNum, colNum) {
    var errors = this.getCellErrors(rowNum, colNum);
    if (errors == null) return isc._emptyString;

    var HTML = this.imgHTML(this.errorIconSrc, this.errorIconWidth, this.errorIconHeight, null, 
                            " isErrorIcon='true'");

    // Since we're writing the icon out to the left of our content, write a spacer out to the
    // right of the image to give us some padding between the image and the cell content
    
    if (isc.Browser.isIE && this._editorShowing && this.getEditRow() == rowNum && 
        (!this.editByCell || this.getEditCol() == colNum)) {
        HTML += " ";
    } else {
        
        HTML += isc.Canvas.spacerHTML(this.cellPadding, "auto");
    }
    return HTML;

},

// Handle events over the error icon by showing a hover containing the
// error text
_handleErrorIconOver : function (rowNum, colNum) {
    this._overErrorIcon = [rowNum,colNum];
    
    // This will fire the method on a delay unless the hover is already up
    // in which case it'll update it immediately.
    isc.Hover.setAction(this, this._handleErrorIconHover);
    

},
_handleErrorIconHover : function () {
    if (this._overErrorIcon == null) return;

    var rowNum = this._overErrorIcon[0],
        colNum = this._overErrorIcon[1];
        
    var errors = this.getCellErrors(rowNum, colNum);
    if (errors == null) return;
    
    var promptString = isc.FormItem.getErrorPromptString(errors);
    isc.Hover.show(promptString, this._getHoverProperties());
},

_handleErrorIconOut : function () {
    delete this._overErrorIcon;
    
    if (isc.Hover.isActive) {
        isc.Hover.clear();
    }
},

// _formatCellValue: Helper method to format the static cell value using developer defined
// formatCellValue() methods.
_$text:"text",
_formatCellValue : function (value, record, field, rowNum, colNum) { 

    if (field && field.userSummary) {
        value = this.getSummaryFieldValue(field, record);
    
    } else {
        if (field && field.formatCellValue != null) {
            value = field.formatCellValue(value,record,rowNum,colNum,this);
        } else if (field && field.cellValueTemplate) {
            // NOTE: 
            // - probably don't need grid.cellValueTemplate, as this would be rare
            // - not exposed publicly yet
            // - might want XSLT option
            value = field.cellValueTemplate.evalDynamicString(this, {
                        value:value, record:record, field:field});
        // listGrid-wide formatter
        } else if (this.formatCellValue != null) {
            value = this.formatCellValue(value, record, rowNum, colNum);
        // check for formatter defined on a SimpleType definition
        } else if (field && field._shortDisplayFormatter != null) {
            value = field._simpleType.shortDisplayFormatter(value, field, this, record, rowNum, colNum);
        // We apply some standard (default) formatters to fields with particular data types.
        // NOTE: these should be moved to the built-in SimpleType definitions
        } else if (field && field._typeFormatter != null) {
            value = field._typeFormatter(value, field, this, record, rowNum, colNum);
        }
    }
    
	// Further conversions to ensure that the value displays correctly in the cell:
    
	// For "empty" values, write out the emptyCellValue
    
	if (value == null || isc.is.emptyString(value)) {
		// if the field specifies a 'emptyCellValue' 
		if (field.emptyCellValue != null) {
			// return the field-specific value
			value = field.emptyCellValue;
		} else if (field.type == "summary") {
		    value = this.invalidSummaryValue;
   
		} else {
			// otherwise return the emptyCellValue for the entire list
			value = this.emptyCellValue;
		}
        
    // In IE, an element containing only whitespace characters (space or enter) will not show css
    // styling properly.
    
    } else if (this._emptyCellValues[value] == true) {
        value = this._$nbsp;

	// convert the value to a string if it's not already        
	} else if (!isc.isA.String(value)) {
        // For date type values we want to respect dateFormatter / timeFormatter specified at the
        // field or component level.
        // If the specified fieldType is "date", "datetime" or "time" this is already handled by the
        // _typeFormatter function (which calls formatDateCellValue / formatTimeCellValue for 
        // these fields)
        
        if (isc.isA.Date(value)) {
        
            // We already know the "type" of this field is not "time" (as formatTimeCellValue)
            // would have run and converted it to a string already.
            // So we'll actually only format as time if the field has ane explicit timeFormatter
            // and no explicit dateFormatter.
            if (this._formatAsTime(field)) {
                var formatter = this._getTimeFormatter(field);
                var isLogicalTime = isc.SimpleType.inheritsFrom(field.type, "time");
                value = isc.Time.toTime(value, formatter, isLogicalTime);
            } else {
    
                var isDatetime = field && isc.SimpleType.inheritsFrom(field.type, "datetime"),
                    isLogicalDate = !isDatetime && field && isc.SimpleType.inheritsFrom(field.type, "date"),
                    formatter = this._getDateFormatter(field);
                if (isDatetime) value = value.toShortDateTime(formatter, true);
                else value = value.toShortDate(formatter, !isLogicalDate);
            }
        } else {
            value = isc.iscToLocaleString(value);
        }
    }
    // hook for final processing of the display value that is applied to the actual display
    // value as derived by the various other hooks
    if (this.formatDisplayValue) {
        value = this.formatDisplayValue(value, record, rowNum, colNum);
    }
    
    // support escapeHTML flag per field
    if (field && field.escapeHTML) {
        if (isc.isA.String(value) && value != this._$nbsp) {
            value = value.asHTML();
        }
    }

	return value;
},
// these Strings can be considered to be "empty" cells, causing bad styling.  Replace with
// '&nbsp;'

_emptyCellValues:{" ":true, "\n":true, "\r":true, "\r\n":true},
_$nbsp:"&nbsp;",

//>	@method listGrid.getEditItemCellValue()   ([IA])
//          Returns the HTML for a cell within a row that is being edited (as a result of a call
//          to 'editRow')<br>
//          Will <i>not</i> call 'updateEditRow()' to update the values displayed in the edit
//          row - this must be handled by calling methods, once we know the form element has
//          been written into the DOM.
//      @visibility internal
//		@group	editing
//
//      @param  record      (object)    the current record object
//      @param  rowNum      (number)    index of the record containing this cell
//      @param  colNum      (number)    index of the field containing this cell
//
//		@return	(string)	display value for this cell
//<
getEditItemCellValue : function (record, rowNum, colNum) {
    var itemName = this.getEditorName(rowNum, colNum);
    
    // If the editRowForm or item haven't yet been created, no op - implies this
    // is being called at an invalid time
    
    if (!this._editRowForm || !this._editRowForm.getItem(itemName)) {
        return "&nbsp;"
    }

	// Write a form item out into this cell.
	// We have already created the '_editRowForm' - a dynamic form with an appropriate set of
	// form items (see 'editRow').
	// Make use of the appropriate item's getStandaloneHTML() to write out the form element
    
    // If we have any errors for the field, set them on the form item too so the error icon
    // gets rendered out
    var errors = this.getCellErrors(rowNum, colNum);
    if (errors) {
        this._editRowForm.setFieldErrors(itemName, errors);
    }

	// get the HTML for the form item
	// Relies on the form item being present - this is fine as long as our logic to create
	// and remove edit form items for incremental rendering stays in synch with the set of 
	// cells being written out.    
    var item = this._editRowForm.getItem(itemName),    
        HTML = item.getStandaloneItemHTML(item.getValue(), false, true);

    // once we've retrieve the HTML, clear the errors so if we re-render the form for another item
    // etc, we don't end up with errors hanging around
    if (errors) {
        this._editRowForm.setFieldErrors(itemName, null);
    }
    
	
    var body = item.containerWidget;
    if (!body._drawnEditItems) body._drawnEditItems = [];   
    if (!body._drawnEditItems.contains(item)) {
        body._drawnEditItems.add(item);
    }
    
    return HTML;
},

// should we show inactive version of the edit form item for some cell?
_showInactiveEditor : function (fieldNum) {
    return this._alwaysShowEditors(fieldNum);
},


// getInactiveEditorCellValue()
// If this.alwaysShowEditors is true, we write inactive versions of our edit form items in every
// row (with appropriate value displayed for the record in question)
getInactiveEditorCellValue : function (record, rowNum, colNum) {
    var itemName = this.getEditorName(rowNum, colNum);

	// get the HTML for the form item
	// Relies on the form item being present - this is fine as long as our logic to create
	// and remove edit form items for incremental rendering stays in synch with the set of 
	// cells being written out.    
    var item = this._editRowForm.getItem(itemName),   
        editorType = isc.DynamicForm.getEditorType(item,this._editRowForm),
        value = this.getRawCellValue(record, rowNum, colNum), 
        HTML;
        
    if (this.shouldShowEditorPlaceholder(this.getField(colNum),editorType)) {
        HTML = this.getEditorPlaceholderHTML(editorType, value, record,rowNum,colNum);
    } else {
        var context = {grid:this.getID(), record:record, rowNum:rowNum, colNum:colNum}
        HTML = item.getInactiveEditorHTML(value, false, true, context);
        // the inactiveEditorContext ID gets stored directly on the cell
        // When we refresh the cell we'll throw away this inactive context (and potentially create
        // another one)
        // We want to be able to map from row+colNum to contextID so we don't have
        // to iterate through all the inactiveContexts on all the form items
        if (!this._inactiveEditorContextMap) this._inactiveEditorContextMap = {};
        var row_col = "_" + rowNum + "_" + colNum;
        if (this._inactiveEditorContextMap[row_col]) {
            this.logWarn("creating new inactiveEditor context for a cell without having cleared " +
                "the previous one." + this.getStackTrace(), "inactiveEditorHTML");
        }
        this._inactiveEditorContextMap[row_col] = context;
    }
	
    return HTML;
},


// if we're showing editors for every record, if this method returns true, suppress showing
// inactive editorHTML and show simplified placeholder HTML instead for the field in question.
// This allows us to not have to ensure that every editorType supports inactiveHTML in an efficient
// manner.
shouldShowEditorPlaceholder : function (field, editorType) {

    if (field.showEditorPlaceholder != null) return field.showEditorPlaceholder;
    
    return !this._supportsInactiveEditorHTML[editorType];
},
// list of editorTypes that support inactiveEditorHTML for alwaysShowEditors:true

_supportsInactiveEditorHTML:{
    text:true, TextItem:true,
    select:true, SelectItem:true,
    combobox:true, ComboBoxItem:true, comboBox:true,
    checkbox:true, CheckboxItem:true,
    date:true, DateItem:true,
    spinner:true, SpinnerItem:true,
    popUpTextArea:true, PopUpTextAreaItem:true
},

// getEditorPlaceholderHTML()
// If this.alwaysShowEditors is true, return the placeholder HTML to show in place of
// inactiveEditorHTML for cells where shouldShowEditorPlaceholder returns true.
getEditorPlaceholderHTML : function (editorType, value, record, rowNum, colNum) {
    // for now just return the value.
    return value;
},

// _editItemsDrawingNotification - function to notify the edit row form items when they are
// drawn, cleared or redrawn.

_editItemsDrawingNotification : function (item, fireMoved, gr) {
    var items;
    if (item) items = [item];
    else {
        items = [];
        var allItems = this._editRowForm.getItems();
        for (var i = 0; i < allItems.length; i++) {
            if (allItems[i].containerWidget == gr) items.add(allItems[i]);
        }
    }   
    var newlyDrawnItems = gr._drawnEditItems;

    for (var i = 0; i < items.length; i++) {
        var currentItem = items[i],
            wasDrawn = currentItem.isDrawn(),
            isDrawn = newlyDrawnItems ? newlyDrawnItems.contains(currentItem) : false,
            isCanvasItem = isc.CanvasItem && isc.isA.CanvasItem(currentItem);
        if (wasDrawn) {
            if (isDrawn) {
                currentItem.redrawn();
                // Redraw of the body will frequently result in repositioning the edit cells.
                // Fire the 'moved' handler on any visible form items that were present before
                // the redraw to notify them of being moved.
                
                if (fireMoved) currentItem.moved();
            }
            else {
                currentItem.cleared();
            }
            
        } else if (isDrawn) {
            if (isCanvasItem) currentItem.placeCanvas();
            currentItem.drawn();
        }
    }
    
    
            
    // Get rid of the _drawnEditItems. Next time this method is run we only want to catch 
    // items which have subsequently been drawn
    delete gr._drawnEditItems;
},

// clearingInactiveEditorHTML
// Called when we're about to clear the generated inactiveEditorHTML due to cell refresh
// (so clearing a single row), or redraw (clearing all).
// Use this to clean up the generated inactive context objects so we don't fill up lots of memory on
// every redraw etc.
_clearingInactiveEditorHTML : function (rowNum,colNum) {

    // Don't crash if we have no inactiveEditorContextMap set.
    
    if (this._inactiveEditorContextMap == null) return;
    
    // have cells be rowNum/colNum pairs allowing us to clear just a single row, or
    // if we find we need more intelligent incremental update handling, single col, etc
    if (!this._editRowForm ||
        !this._alwaysShowEditors((colNum != null ? this.getField(colNum) : null),true)) return;
    
    if (rowNum != null) {
        var row_col = "_" + rowNum + "_" + colNum,
            context = this._inactiveEditorContextMap[row_col];
        if (context) {
            context.formItem.clearInactiveEditorContext(context);
            delete this._inactiveEditorContextMap[row_col];
        }
    } else {
        var editForm = this.getEditForm(),
            items = editForm.getItems();

        for (var i = 0; i < items.length; i++) {
            items[i].clearAllInactiveEditorContexts();
        }
        
        delete this._inactiveEditorContextMap;
    }
},

// setRecordValues()
// Method to update client-side data in place
// This is called directly by DynamicForms when saving values if this is acting as the selection
// component for a form.
setRecordValues : function (pks, values) {
    if (!this.shouldSaveLocally()) {
        this.logWarn("setRecordValues() not supported for grids where saveLocally is false");
        return;
        
    }
    if (!this.data) return;
    
    var rowNum = this.data.indexOf(pks),
        record = this.data.get(rowNum);
        
    // Incoming values will be in a structured form matching the dataPath, which is no good to
    // us because the records inside this component only contain fields from a specific 
    // sub-hierarchy.  For example, if this grid is bound to dataPath "/items" it will have
    // flat-named fields called "a" and "b" that are actually at dataPaths "/items/a" and 
    // "/items/b".  New values will come into this method in that complete form, so unless 
    // we do something, our record will up looking like this:
    // {
    //   a: "something",
    //   b: 10,
    //   items: {
    //     a: "the new value",
    //     b: 100
    //   }
    // }
    //
    // So, we just create a new values object by pulling out whatever is at this component's 
    // dataPath in the passed-in values (it will be the same as the form's dataPath, because 
    // the only way we get here is for selectionComponents, and a selectionComponent implicitly
    // shares a dataPath with the forms for which it is a selectionComponent)
    
    var newValues = isc.Canvas._getFieldValue(this.dataPath, values, this);
        
    isc.combineObjects(record, newValues);
    this.calculateRecordSummaries([record]);
    if (this.useCellRecords) {
        rowNum = this.findRowNum(record);
        var colNum = this.findColNum(record);
        this.refreshCell(rowNum,colNum);
    } else {
        this.refreshRow(rowNum);
    }
    
    if (this.valuesManager != null) {
        // _updateMultipleMemberValue handles combining a data-index into a path
        // 2nd param unnecessary since we're modifying the whole record, not a field within it.
        this.valuesManager._updateMultipleMemberValue(rowNum, null, record, this);
    }
    
},


//>	@method	listGrid.setRawCellValue()
//		@group	data
//			Set the raw data for one particular cell in the list.
//
//		@param	record		(object)	record in question
//		@param	recordNum	(number)	number of that record
//		@param	fieldNum	(number)	number of the field to display
//		@param	newValue	(any)		new value
//
//<
// Overridden at the cubeGrid level to handle being passed an entire record rather than a single
// field value for the cell
setRawCellValue : function (record, recordNum, fieldNum, newValue) {

	var field = this.fields[fieldNum];
	// if record or field is null, we're dealing with an invalid column, so forget it
	if (!record || !field) return;
	// if the field has a 'setRawCellValue' attribute
	// We'll assume this DIRECTLY sets the value on the LG's data object, so we're basically done 
	// at this point (other than firing dataChanged below to ensure everything gets updated 
	// visually)
	if (field.setRawCellValue) {
		// CALLBACK API:  available variables:  "viewer,record,recordNum,field,fieldNum,value"
		// Convert a string callback to a function
		isc.Func.replaceWithMethod(field, "setRawCellValue", "viewer,record,recordNum,field,fieldNum,value");

		// call it as a function (returns string | null)
		field.setRawCellValue(this, record, recordNum, field, fieldNum, newValue);
        
	// otherwise just use the cells value in the normal way
	} else {
        
        if (field.dataPath) {
            isc.Canvas._saveFieldValue(field.dataPath, newValue, record, this);
        } else {
            // set the value according to the field specification
            record[field[this.fieldIdProperty]] = newValue;
        }
    }    

	// HACK: fire dataChanged() by hand.  Really, we need an interface to update fields on
	// objects in the List so the List can recognize the change and fire dataChanged() itself
    this.data.dataChanged();
},

//>	@method	listGrid.getCellBooleanProperty()	(A)
//  Given a property name, and a cell, check for the value of that property (assumed to be a 
//  boolean) on the ListGrid, and the cell's field, returning false if the value is false at 
//  either level.
//  If true at the ListGrid and Field level, check the value of the the second "recordProperty"
//  on the record object for the cell.
//  (If recordProperty is not passed, the record object will not be examined).
//
//	@param	property	(string)	Name of the property to look for.
//	@param	rowNum	(number)	Row number of the cell.
//	@param	colNum	(string)	Field number for the cell.
//  @param  [recordProperty]    (string)    Name of the equivalent property to check on the
//                                          record object 
//
//	@return	(boolean)	true == at least one is true and none are false
//
//<

_$false:"false", _$true:"true",
getCellBooleanProperty : function (property, recordNum, fieldNum, recordProperty) {
	var trueFound = false,
		listValue = this[property]
	;
	if (listValue == false || listValue == this._$false) return false;

	var fieldValue = this.fields[fieldNum][property];
	if (fieldValue == false || fieldValue == this._$false) return false;
    
    if (recordProperty != null) {

    	var record = this.getRecord(recordNum, fieldNum),
	    	recordValue = (record != null ? record[recordProperty] : false)
    	;
    	if (recordValue == false || recordValue == this._$false) return false;
        if (recordValue == true || recordValue == this._$true) return true;
    }
	
	// At this point we know none of the values was an explicit false - but we only want to 
	// return true if the value was specified as true (rather than undefined) at some level.
	// We've already checked at the record level (if necessary)
	return (listValue == true) || (fieldValue == true) ||
            (listValue == this._$true) || (fieldValue == this._$true);
},


// ---------------------------------------------------------------------------------------

//> @method listGrid.setShowRecordComponents()
// Setter for the +link{listGrid.showRecordComponents} attribute
// @param showRecordComponents (boolean) new value for <code>this.showRecordComponents</code>
// @visibility external
//<
setShowRecordComponents : function (showRC) {
    if (this.showRecordComponents == showRC) return;
    
    
    if (showRC) {
        if (this.animateFolders) {
            this._animateFoldersWithoutRC = true
            this.animateFolders = false;
        }
    } else {
        if (this._animateFoldersWithoutRC) {
            this.animateFolders = true;
            delete this._animateFoldersWithoutRC;
        }
    }
    
    this.showRecordComponents = showRC;
    
    // Update virutalScrolling if necessary.
    // We'll also update virtual scrolling in createBodies() - this handles the case where
    // we're already showing recordComponents and a grid is frozen at runtime.
    this._updateVirtualScrollingForRecordComponents();
    
    // suppress 'drawAllMaxCells' type behavior - we don't want to render out potentially
    // hundreds of canvases.
    if (showRC) {
        this._oldDrawAllMaxCells = this.drawAllMaxCells;
        this.drawAllMaxCells = 0;
        if (this.body != null) this.body.drawAllMaxCells = 0;
      
    } else {
        if (this._oldDrawAllMaxCells != null) {
            this.drawAllMaxCells = this._oldDrawAllMaxCells;
            if (this.body != null) this.body.drawAllMaxCells = this._oldDrawAllMaxCells;
            delete this._oldDrawAllMaxCells;
        }
        
    }
    this.invalidateRecordComponents();

    
},

_updateVirtualScrollingForRecordComponents : function () {
    if (!this.showRecordComponents) {
        if (this._rcVScroll) {
            delete this.virtualScrolling;
            delete this._rcVScroll;
        }
    
    // Virtual scrolling:
    // Embedded componenents can make row heights unpredictable
    // (may not show for every row, may be 'position:"expand"', or exceed this.cellHeight 
    // etc)
    // Because of this we typically have to enable virtual scrolling for record components.
    // However we don't currently support virtual scrolling with frozen fields, so
    // don't enable it if we have frozen fields.
    // This *may* lead to unpredictable behavior. Cases where it's ok:
    // - if recordComonentPosition is 'within' and 
    //   the recordComponentPosition < this.cellHeight
    // - if recordComponentHeight is set (and truly is not exceeded by embedded components)
    
    } else {
        if (this.virtualScrolling == null || this._rcVScroll) {
            if (this.frozenFields == null) {
                this.virtualScrolling = true;
                this._rcVScroll = true;
                
            } else {

                if (this.recordComponentHeight == null) {
                    this.logWarn("This grid has frozen fields and is showing " +
                        "recordComponents. This may lead to unpredictable row heights which " +
                        "are not supported with frozen fields. Setting " +
                        "listGrid.recordComponentHeight will avoid this issue.",
                        "recordComponents"); 
                }
                if (this._rcVScroll) {
                    delete this.virtualScrolling;
                    delete this._rcVScroll;
                }
            }
        }
    }
    
    if (this.body && this.virtualScrolling != this.body.virtualScrolling) {
        this.body.virtualScrolling = this.virtualScrolling;
        if (this.frozenBody) {
            this.frozenBody.virtualScrolling = this.virtualScrolling;
        }
    }
    // No need to call 'redraw' on the body -- calling code is expected to handle this.
},



//>	@method	listGrid.getDrawArea()	(A)
// Returns the extents of the rows and columns current visible in this grid's viewport.
// <P>
// Note: if there are any +link{listGridField.frozen,frozen fields}, they are not included
// in the draw area range returned by this method. Frozen fields are assumed to never be 
// scrolled out of view.  The column coordinates returned by this method will only include
// unfrozen columns.
//
// @return	(Array of Number)	The row/col co-ordinates currently visible in the viewport as
//    [startRow, endRow, startCol, endCol].
// @visibility external
//<

getDrawArea : function () {
    if (this.body) {
        var drawArea = this.body.getDrawArea();
        if (this.frozenFields && this.freezeLeft()) {
            drawArea[2] += this.frozenFields.length;
            drawArea[3] += this.frozenFields.length;
        }
        return drawArea;
    }
    
    return null;
},

// _drawAreaChanged() - notification fired on GridRenderer.redraw() when the
// previous draw area doesn't match the new draw area

_drawAreaChanged : function (oldStartRow, oldEndRow, oldStartCol, oldEndCol, body) {
    if (this.frozenFields && this.freezeLeft()) {
        oldStartCol += this.frozenFields.length;
        oldEndCol += this.frozenFields.length;
    }
    var oldDrawArea = [oldStartRow, oldEndRow, oldStartCol, oldEndCol];
    if (oldDrawArea.equals(this.getDrawArea())) return;
    this.drawAreaChanged(oldStartRow,oldEndRow,oldStartCol,oldEndCol);
},

// documented in registerStringMethods
drawAreaChanged:function () {},


// updateRecordComponents() - fired from redraw on grid body (or frozen body).
// This method essentially iterates through our current draw area and ensures that if
// showRecordComponents is true, we're showing recordComponents for each row (or cell),
// calling 'createRecordComponent()' or 'updateRecordComponent()' as necessary to create
// new components, and discarding (or pooling) record components that we previously created
// which are no longer visible.
// This method should not need to be called by developers directly.
// To force an explicit invalidation and refresh of recordComponents, use 
// invalidateRecordComponents()
// *Note: This method also handles updating backgroundComponents if specified

updateRecordComponents : function () {
    
    // Sanity check to avoid infinite loops if adding embedded components trips redraw of
    // body for example
    var debugLog = this.logIsDebugEnabled("recordComponents");
    if (this._updatingRecordComponents) {
        if (debugLog) {
            this.logDebug("updateRecordComponents called recursively - returning",
                "recordComponents");
        }
        return;
    }
    
     // If we're performing a show/hide row height animation, bail.
    // In this case the HTML in the body won't match the set of records in our data set
    // so we can't update / place embedded components properly
    var body = this.body,
        frozenBody = this.frozenBody;
    if (body == null) return;
    
    if (body._animatedShowStartRow !=  null) {
        return;
    }
    
    // draw() fires this method before calculating _fieldWidths, in which case we can't
    // yet size/position our embedded components. This can occur when we rebuildForFreeze.
    // Catch this case and return.
    if ((body._fieldWidths == null || 
        (body._fieldWidths.length == 0 && body.fields.length > 0)) ||
        (frozenBody && 
            (frozenBody._fieldWidths == null || 
            (frozenBody._fieldWidths.length == 0 && frozenBody.fields.length > 0)))) return;
    
    this._updatingRecordComponents = true;
    
    // Implementation overview: The concept here is that if showRecordComponents is true,
    // we call a method 'createRecordComponent()' [or potentially 'updateRecordComponent']
    // for every visible row, or if showing by cell, every visible cell, lazily as its rendered
    // out.
    // When new cells are rendered out we want to
    // - call createRecordComponent() [or updateRC] for newly rendered cells
    // - *not* call createRC for cells that were visible and still are (regardless of whether
    //   createRecordComponent returned an actual component or just null)
    // - for cells that are no longer visible, clear up the created record components, 
    //   clearing them, destroying them or adding them to our 'recordComponentPool' depending on
    //   the recordComponentPoolingMode.
    // Rather than trying to achieve this by tracking viewports (which has the major disadvantage
    // of being fragile on data change or field config change), we take this approach:
    // We store all generated record components in 2 places:
    // - on the records themselves, indexed by fieldName, under record._recordComponents
    // - on the ListGrid in both an array and an object mapping componentIDs to true
    // If the createRecordComponent method returned null for any cell, we store a special
    // nullMarker object on the record._recordComponents object as well.
    //
    // When this method runs we can then iterate through all visible records / fields
    // - determine if we have visible record components already present, or null markers,
    //   in which case we leave the component alone
    // - otherwise run the method to create a new record component / get one from the pool and
    //   apply it to the cell.
    // Once we've gone through all visible cells we iterate through all the recordComponents
    // we previously created and wipe out (either clear, destroy or recycle) any that weren't
    // noted as being attached to a visible cell in the previous step.
    
    // _liveRecordComponents / _liveRecordComponentsObj is the full set of recordComponents
    // generated last time this method was run.
    var oldRecordCompArr = this._liveRecordComponents || [],
        oldRecordCompObj = this._liveRecordComponentsObj || {};
    
    if (debugLog) {
        this.logDebug("updateRecordComponents - old record components before refreshing:" + 
            this.echo(oldRecordCompArr),
            "recordComponents");
    }
    
    this._liveRecordComponentsObj = {};
    this._liveRecordComponents = [];
    
    
    // If showRecordComponents is false we can skip all logic to create
    // new recordComponents. If we had any previously created recordComponents we'll clear
    // them below. This will handle the showRecordComponents setting being changed dynamically.
    if (this.showRecordComponents || this.showBackgroundComponents) {
        
        // Determine what our current draw area is - set of drawn fields and rows.
        // This method is being called as part of redraw, before the render has occurred, so
        // we can't just look at body._firstDrawnRow / _lastDrawnRow etc - we need
        // to call the getDrawArea() APIs on the body to actually calculate the new values
        
        var drawArea = this.body.getDrawArea(),
            cellComponents = this.showRecordComponentsByCell,
            bodyID = this.body.getID(),
            frozenBodyID = this.frozenBody ? this.frozenBody.getID() : null;
            
        if (debugLog) {
            this.logDebug("updating to potentially show recordComponents for drawArea:" 
                + drawArea, "recordComponents");
        }
        for (var rowNum = drawArea[0]; rowNum <= drawArea[1]; rowNum++) {
            var record = this.getRecord(rowNum);
            
            if (record == null || Array.isLoading(record)) continue;
            
            if (this.showRecordComponents) {
                
                // If we don't have cell components we will add components to the (unfrozen) body,
                // one per row.
                if (!cellComponents) {
                     
                    var shouldShowRecordComponent = this.shouldShowRecordComponent(record),
                        liveComp = null;
                    if (shouldShowRecordComponent) {
                        // getLiveRecordComponent() will pick up the record component we've already
                        // applied to the record/field.
                        // NOTE: If createRecordComponent ran and returned null we store a special
                        // null-marker object which we'll get back here as well. This means we
                        // don't re-run createRecordComponent() unless we actually want to.
                        liveComp = this._getLiveRecordComponent(record, null, bodyID);
                        if (liveComp != null) {
                            if (liveComp.isNullMarker) {
                                liveComp = null;
                            } else {
                                var ID = liveComp.getID();
                                oldRecordCompObj[ID] = null;
                            }
                        } else {
                            liveComp = this._applyNewRecordComponent(record, null, this.body, rowNum);
                        }
                    }
                    
                    // Store pointers to both the component and its ID
                    
                    if (liveComp != null) {
                        var ID = liveComp.getID();
                        this._liveRecordComponentsObj[ID] = true;
                        this._liveRecordComponents[this._liveRecordComponents.length] = liveComp;
                    }
                    
                // same logic as above, but applied per cell to both the frozen and
                // unfrozen body.
                } else {
                
                    if (this.frozenBody != null) {
                        for (var fieldNum = 0; fieldNum < this.frozenBody.fields.length; fieldNum++) {
                            var field = this.frozenBody.fields[fieldNum],
                                fieldName = field.name;
                            
                            var shouldShowRecordComponent = this.shouldShowRecordComponent(record, field.masterIndex),
                                liveComp = null;
                                
                            if (shouldShowRecordComponent) {
                                liveComp = this._getLiveRecordComponent(record, fieldName, frozenBodyID);
                                if (liveComp != null) {
                                    if (!liveComp.isNullMarker) {
                                        var ID = liveComp.getID();
                                        oldRecordCompObj[ID] = null;
                                    } else {
                                        liveComp = null;
                                    }
                                } else {
                                    liveComp = this._applyNewRecordComponent(record, fieldName, this.frozenBody,
                                                    rowNum, fieldNum);
                                }
                            }
                            if (liveComp != null) {
                                var ID = liveComp.getID();
                                this._liveRecordComponentsObj[ID] = true;
                                this._liveRecordComponents[this._liveRecordComponents.length] = liveComp;
                            }
                        }
                    }
                    for (var bodyCol = drawArea[2]; bodyCol <= drawArea[3]; bodyCol++) {
                        var field = this.body.fields[bodyCol],
                            fieldName = field.name;
                            
                        var shouldShowRecordComponent = this.shouldShowRecordComponent(record, field.masterIndex),
                            liveComp = null;
                        if (shouldShowRecordComponent) {
                            var liveComp = this._getLiveRecordComponent(record, fieldName, bodyID);
                            if (liveComp != null) {
                                if (!liveComp.isNullMarker) {
                                     var ID = liveComp.getID();
                                     oldRecordCompObj[ID] = null;
                                } else {
                                    liveComp = null;
                                }
                            } else {
                                liveComp = this._applyNewRecordComponent(record, fieldName, this.body,
                                            rowNum, bodyCol);
                            }
                        }
                        
                        if (liveComp != null) {
                            var ID = liveComp.getID();
                            
                            this._liveRecordComponentsObj[ID] = true;
                            this._liveRecordComponents[this._liveRecordComponents.length] = liveComp;
                        }
                    }
                }
            }
            
            
            if (this.showBackgroundComponents) {
                if (record && record.backgroundComponent) {
                    var component = record._embeddedComponents ? 
                            record._embeddedComponents.find("isBackgroundComponent", true) : null;
        
                    if (!component) {
                        // should be showing a backgroundComponent but it's not present yet - add it now
                        if (isc.isA.Canvas(record.backgroundComponent)) {
                            // backgroundComponent is specified as a canvas
                            var comp = record.backgroundComponent.addProperties(
                                this.backgroundComponentProperties,
                                { isBackgroundComponent: true }
                            );
                        } else {
                            // backgroundComponent is specified as properties
                            var props = isc.addProperties({ isBackgroundComponent: true },
                                this.backgroundComponentProperties,
                                record.backgroundComponent);
                            var comp = this.createAutoChild("backgroundComponent", props);
                        }
        
                        var tableIndex = body.getTableZIndex();
                        comp.setZIndex(tableIndex - 49);
                        comp.setWidth("100%");
                        comp.setHeight("100%");
                        comp.setOverflow("hidden");
                        
                        
                        this.addEmbeddedComponent(record.backgroundComponent, record, rowNum, null, "within");
                        
                        // This should stick with the record until it's wiped due to data change
                        // or similar (EG remapEmbeddedComponents)
                        // At that point, if this method runs again it'll be cleared
                    }
                }
            }
        }
    }
    
    if (this.logIsInfoEnabled("recordComponents")) {
        this.logInfo("updateRecordComponents - new recordComponents:" + 
            this.echo(this._liveRecordComponentsObj) +
            ", old record components (will be cleaned up if value is 'true'):" + 
            this.echo(oldRecordCompObj), "recordComponents");
    }
    
    // At this point we've iterated through our draw area (or showRecordComponents is false,
    // in which case we want to drop all pre existant Record Components).
    // Any pre-existant recordComponents that are still visible have been removed from
    // the 'oldRecordCompObj'.
    // Iterate through pre-existant record components that are left and clear them up
    // (remove from DOM if necessary, destroy / pool if necessary)
    for (var i = 0; i < oldRecordCompArr.length; i++) {
        // if it's been cleared from the oldRecordCompObj we know its still visible / being used
        var ID = oldRecordCompArr[i].getID();
        if (oldRecordCompObj[ID] != true) {
            continue;
        }
        if (debugLog) {
            this.logDebug("cleaning up RecordComponent:" + oldRecordCompArr[i], "recordComponents");
        }
        this._cleanUpRecordComponent(oldRecordCompArr[i]);
        oldRecordCompObj[ID] = null;
    }
    delete this._updatingRecordComponents;
},

// _applyNewRecordComponent()
// This method will run 'createRecordComponent()' or 'updateRecordComponent()' to
// get the recordComponent for some record or cell.
_applyNewRecordComponent : function (record, fieldName, body, rowNum, bodyCol) {
    
    if (this.logIsDebugEnabled("recordComponents")) {
        this.logDebug("getting record component for row/field:" + [rowNum,fieldName],
            "recordComponents");
    }
    
    var bodyID = body.getID();
    
    var pool = this.recordComponentPoolingMode == "recycle",
        component,
        // same row variable - only used if we're picking up a pooled component
        sameRow,
        colNum = fieldName == null ? null : this.getColNum(fieldName);
        
    if (pool) {
        var compConfig = this.getFromRecordComponentPool(record, fieldName);
        component = compConfig ? compConfig[0] : null;
        sameRow = compConfig ? compConfig[1] : null;
    }
    
    if (!component) {
        if (this.createRecordComponent && isc.isA.Function(this.createRecordComponent)) {
            component = this.createRecordComponent(record, this.getColNum(fieldName));
            if (component != null) component.isRecordComponent = true;
            this.logDebug("created new record component:" + component, "recordComponents");
        }
    } else {
        if (this.updateRecordComponent && isc.isA.Function(this.updateRecordComponent)) {

            component = this.updateRecordComponent(record, colNum, component, !sameRow);

            // component may well be null
            if (component == null) {
                if (this.logIsInfoEnabled("recordComponents")) {
                    this.logInfo("showRecordComponents: updateRecordComponent() method " +
                        "failed to return an updated component.", "recordComponents");
                }
            }
            this.logDebug("updated record component from pool:" + component, "recordComponents");
        }
    }
    
    
    var addNullMarker = component == null;
    if (addNullMarker) {
        component = {
            isNullMarker:true,
            _embedBody:bodyID,
            _recordComponentBatch:this._recordComponentSequence
        }
    }
    if (record._recordComponents == null) {
        record._recordComponents = {};
    }
    if (fieldName == null) fieldName = this._$noFieldString;
    
    
    record._recordComponents[fieldName] = component;
    // We're applying a "currentFieldName" / "currentRecord" flag in addition
    // to the _currentFieldName applied by the embeddedComponents code. This is
    // intentional - we use these flags in pooling mode to pick up the component that
    // matched the previous record (if possible) and previous field if
    // poolComponentsPerColumn is true. Don't want to rely on the flags that are set up and
    // potentially cleared by the standard embeddedComponent subsystem.
    if (pool && !addNullMarker) {
        component.currentFieldName = fieldName;
        component.currentRecord = record;
    }
    if (!addNullMarker) {
        //this.logWarn("created component:" + component + ", adding to:" + [rowNum,fieldName]);
        return body.addEmbeddedComponent(component, record, rowNum, bodyCol, this.getRecordComponentPosition());
    }
},

// fired when a recordComponent's cell is no longer visible. Behavior depends on 
// recordComponentPoolingMode.
_cleanUpRecordComponent : function (component, forceDestroy) {
    if (this.logIsDebugEnabled("recordComponents")) {
        this.logDebug("cleaning up recordComponent:" + component,
            "recordComponents");
    }

    var poolingMode = this.recordComponentPoolingMode;
    // If passed the forceDestroy parameter, behave in 'viewport' mode regardless of
    // the actual pooling mode - this means we'll destroy the component passed in.
    if (forceDestroy) poolingMode = "viewport";
    
    if (poolingMode == "data") {
        
        // Nothing to do here:
        // If the record is still around, placeEmbeddedComponent() will have cleared it and
        // it'll simply re-render when scrolled back into view, etc.
        //
        // If the record is no longer present in the data array, remapEmbeddedComponents()
        // will have already marked it for destruction, and updateRecordComponents() has
        // already rebuilt the array of _liveRecordComponents so we won't be hanging onto
        // a pointer to it anymore.

    } else {
        var ID = component.ID,
            body = isc.Canvas.getById(component._embedBody),
            record = component.embeddedRecord,
            fieldName = component._currentFieldName;

        // wipe the component from the components-cache on the record [IFF it hasn't been updated
        // to point at a new component]
        
        if (record._recordComponents[fieldName] == ID) {
            delete record._recordComponents[fieldName];
        }
        
        // this component may have already been removed from the body, for example by
        // _remapEmbeddedComponentColumns(). In this case _embedBody will have been null
        // so we can detect this by the body var being unset here.
        if (body != null) {
            body.removeEmbeddedComponent(component.embeddedRecord, component);
        }
        
        if (poolingMode == "viewport") {
            component.markForDestroy();
        } else {
            
            if (component.destroying || component.destroyed || component._pendingDestroy) return;
            this.addToRecordComponentPool(component);
        }
    }
},

// Helper method - look at a record and see if we currently have a recordComponent for it.

_$noFieldString:"_noField",
_recordComponentSequence:0,

getLiveRecordComponent : function (record, fieldName, bodyID) {
    if (!record) return null;
    if (isc.isA.Number(record)) record = this.getRecord(record);
    if (!bodyID) bodyID = this.body.getID();
    return this._getLiveRecordComponent(record, fieldName, bodyID);
},
_getLiveRecordComponent : function (record, fieldName, bodyID) {
    
    if (fieldName == null) fieldName = this._$noFieldString;
    
    var recordComponents = record._recordComponents;
    
    if (recordComponents == null || recordComponents[fieldName] == null) return null;
    
    var component = recordComponents[fieldName];
    
    if (component._embedBody != bodyID) {
        return null;
    }
    if (component.isNullMarker && component._recordComponentBatch != this._recordComponenSequence) {
        return null;
    }
    
    // We should never see this but if a component gets destroyed without first being
    // cleared out of the record._recordComponents block, wipe it out now. This will force
    // creation of a new recordComponent in calling code.
    if (component.destroyed || component.destroying || component._pendingDestroy) {
        this.logWarn("Destroyed or Destroying record component:" + component + 
            " present on record. Ignoring", "recordComponents");
        recordComponents[fieldName] = null;
        return null;
    }
    
    return component;
},

//> @method listGrid.invalidateRecordComponents()
// Invalidates the currently visible set of +link{listGrid.showRecordComponents,recordComponents}
// and gets fresh ones for the visible rows in the grid according to the 
// +link{listGrid.recordComponentPoolingMode}
// <P>
// See also +link{listGrid.refreshRecordComponent()} which allows you to refresh a specific
// recordComponent
//
// @visibility external
//<
invalidateRecordComponents : function () {
    
    // force destruction of the visible recordComponents - otherwise this
    // method would have no visible effect in 'data' pooling mode.
    this.dropRecordComponents(true);
    
    if (this.showRecordComponents && this.isDrawn()) {
        this.updateRecordComponents();
    }
},

dropRecordComponents : function (forceDestroy) {
    
    // up the recordComponentSequence count. This is used to identify our special null markers
    // and essentially invalidates them, meaning we'll re-run the createRecordComponent logic
    // for records with null markers we've already set at this point.
    
    this._recordComponentSequence++;
    
    var oldRecordCompArr = this._liveRecordComponents || [];
        
    delete this._liveRecordComponents;
    delete this._liveRecordComponentsObj;
    
    for (var i = 0; i < oldRecordCompArr.length; i++) {
        this._cleanUpRecordComponent(oldRecordCompArr[i], forceDestroy);
    }
},

//> @method listGrid.refreshRecordComponent()
// Discards any currently visible set of +link{listGrid.showRecordComponents,recordComponent}
// and gets a fresh one for the specified record (or cell) according to the 
// +link{listGrid.recordComponentPoolingMode}
// <P>
// See also +link{listGrid.invalidateRecordComponents()} which allows you to refresh all record
// components that are currently visible in the grid.
//
// @param rowNum (integer) Row to refresh
// @param [colNum] (integer) Column to refresh. This parameter should be passed 
//      if +link{showRecordComponentsByCell} is true.
// @visibility external
//<
refreshRecordComponent : function (rowNum, colNum) {
    if (!this.showRecordComponents || rowNum == null || this.body == null) return;
    
    if (this.showRecordComponentsByCell && colNum == null) {
        this.logWarn("refreshRecordComponent() called with no colNum. This parameter is required when " +
            "showRecordComponentsByCell is true. Taking no action.");
        return;
    }
    
    var record = this.getRecord(rowNum);
    if (record == null || Array.isLoading(record)) return;
    
    var body = this.body,
        bodyColNum = null,
        fieldName = null;
        
    if (this.showRecordComponentsByCell) {
        body = this.getFieldBody(colNum);
        bodyColNum = this.getLocalFieldNum(colNum);
        fieldName = this.getFieldName(colNum);
    }
  
    var prevComp = this._getLiveRecordComponent(record, fieldName, body.getID());
    if (prevComp != null && prevComp.isNullMarker) prevComp = null;
    
    if (prevComp != null) {
        // This will destroy the component, or add to the pool.
        this._cleanUpRecordComponent(prevComp, (this.recordComponentPoolingMode != "recycle"));
    }
    
    var liveComp;
    if (this.shouldShowRecordComponent(record)) {
        liveComp = this._applyNewRecordComponent(record, fieldName, body, rowNum, bodyColNum);
        if (liveComp && liveComp.isNullMarker) liveComp = null;
    }
    
    // Update this._liveRecordComponents and this._liveRecordComponentsObj
    var index = this._liveRecordComponents.length;
    if (prevComp) {
        var ID = prevComp.getID();
        this._liveRecordComponentsObj[prevComp.getID()] = null;
        // Null the slot in the live recordComponents array and reuse it if possible for
        // efficiency.
        if (liveComp != null) {
            index = this._liveRecordComponents.indexOf(prevComp);
            this._liveRecordComponents[index] = null;
        } else {
            this._liveRecordComponents.remove(prevComp);
        }
    }
        
    if (liveComp != null) {
        var ID = liveComp.getID();
        this._liveRecordComponentsObj[ID] = true;
        this._liveRecordComponents[index] = liveComp;   
    }
},

getRecordComponentPosition : function () {
    if (this.recordComponentPosition != null) return this.recordComponentPosition;
    return (this.showRecordComponentsByCell ? "within" : "expand");
},

getRecordComponentPool : function () {
    if (!this._recordComponentPool) this._recordComponentPool = [];
    return this._recordComponentPool;
},

// we want to indicate whether the record changed. Handle this by returning a 2 element array
// - the component and a boolean.
getFromRecordComponentPool : function (record, fieldName) {
    var components = this.getRecordComponentPool(),
        subList = [],
        component;
    
    if (!components || components.length == 0) return null;
    
    if (this.poolComponentsPerColumn == true) {
        subList = components.findAll("currentFieldName", fieldName);
    } else {
        subList = components;
    }
    if (!subList || subList.length == 0) return null;
    
    for (var i = 0; i < subList.length; i++) {
        component = subList[i];
        var prevRecord = component.currentRecord;
        if (this.comparePrimaryKeys(prevRecord, record)) {
            components.remove(component);
            return [component,true];
        }
    }
    // we didn't find a component that previously sat in this record, just return an arbitrary
    // one - the last one on the list.
    components.length -= 1;
    return [component,false];
},

addToRecordComponentPool : function (component) {
    var components = this.getRecordComponentPool();

    components.add(component);
},

// Should we show a recordComponent for this record/col?
// checks for various records we want to skip, like the separator rows, and
// fires the public 'showRecordComponent()' method to allow custom suppression of RCs for
// certain rows or cells.
shouldShowRecordComponent : function (record, colNum) {
    
    if (record == null || record._isGroup || record[this.isSeparatorProperty]
        || Array.isLoading(record))
    {
        return false
    }
    return this.showRecordComponent(record,colNum);
},

// Override point documented in registerStringMethods().

showRecordComponent : function () {
    return true;
},

// notification from each body when getInnerHTML is called.
bodyDrawing : function (body) {

    if (isc._traceMarkers) arguments.__this = this;

    // don't fetch valueMap in response to draw on both the frozen and liquid body - just
    // on the normal (liquid) one
    if (body != this.body) return;
    var startedQueue;

    if (this._fetchValueMap) {
        // fetch valueMaps for fields where optionDataSource is set
        startedQueue = !isc.RPCManager.startQueue();
        this._fetchValueMapData();
    }
    
    this.requestVisibleRows();
    
    if (startedQueue) isc.RPCManager.sendQueue();
    this._fetchValueMap = null;
},


//> @attr listGrid.recordComponentHeight (Integer : null : IRWA)
// If +link{listGrid.showRecordComponents} is true, this attribute may be used to
// specify a standard height for record components.
// If specified every row in the grid will be sized tall enough to accommodate a recordComponent
// of this size.
// <P>
// Note that if this property is unset, the grid will not be able to know row heights in
// advance, and +link{listGridField.frozen,freezing of columns} is not currently supported in
// this case.
// @visibility external
//<


//> @method listGrid.setRecordComponentHeight()
// Setter for the +link{listGrid.recordComponentHeight}
// @param height (integer) recordComponent height
// @visibility external
//<
setRecordComponentHeight : function (height) {
    this.recordComponentHeight = height;
    if (this.isDrawn()) this.markForRedraw();
},

// Override 'getAvgRowHeight()' - if recordComponentHeight is specified and we're showing
// recordComponents, make use of it.
getAvgRowHeight : function (body) {
    
    if (this.showRecordComponents && this.recordComponentHeight != null) {
        return this.getRecordComponentRowHeight();
    }
    // standard behavior
    return body.fixedRowHeights ? body.cellHeight : Math.max(body.cellHeight,body.avgRowHeight);
    
},

_$expand:"expand",
getRecordComponentRowHeight : function () {
    if (this.recordComponentHeight == null) return null;
    var pos = this.getRecordComponentPosition();
    if (pos == this._$expand) return this.cellHeight + this.recordComponentHeight;
    else return Math.max(this.recordComponentHeight, this.cellHeight);
},

// ListGridField.optionDataSource handling
// ---------------------------------------------------------------------------------------

// This logic handles fields where an optionDataSource acts as a server-side valueMap
//
// For fields with a specified optionDataSource separate from this grid's dataSource, with
// displayField and valueField specified, we need to perform a fetch against the server to
// get a map from data values to display values

// _setOptionDataSources() (called from setFields())
// iterate through the list of fields and take note of optionDataSource property to look up the
// display map later when the LG is drawn or redrawn (see bodyDrawing)
_setOptionDataSources : function () {

    
    // _fetchValueMap - one time flag to kick off a fetch in bodyDrawing
    this._fetchValueMap = null;
    
    var oldODSs = this._optionDataSources;
    this._optionDataSources = [];
    var gridDS = this.getDataSource();
    
    for (var i = 0; i < this.completeFields.length; i++) {
        var field = this.completeFields[i];
        
        if (field == null) {
            this.logWarn("Fields array contains an empty entry");
            continue;
        }
        
        var displayField = field.displayField || field[this.fieldIdProperty],
            valueField = field.valueField || field[this.fieldIdProperty];
            
        // autoFetchDisplayMap can be set at the field or LG level
        if (field.optionDataSource == null ||
            // If the display field matches the valueField don't bother to do a fetch since
            // we don't need to map the static raw cell value to a display value.
            // If the field is editable this means that we'll rely on the editor to perform a fetch
            // based on the optionDataSource rather than being passed an explicit valueMap.
            field.displayField == null || (field.displayField == field.valueField) ||
            (field.valueField == null && (field.displayField == field[this.fieldIdProperty])) ||
            (field.autoFetchDisplayMap == false) ||
            (this.autoFetchDisplayMap == false && field.autoFetchDisplayMap == null))
        {
            continue;
        }
        
        var optionDS = isc.DS.get(field.optionDataSource);
        if (optionDS == null) {
            this.logWarn(field.optionDataSource + " dataSource not found, check value of " +
                         "optionDataSource property on the " + field[this.fieldIdProperty] +
                         " field");
            continue;
        
        
        }
        
        var optionDSID = optionDS.ID,
            fieldName = field[this.fieldIdProperty],
            optionCriteria = field.optionCriteria;
            
        // have we already set up an optionDataSource config object from our previous set of fields?
        // which matches this ODS / criteria?

        // If so re-use it - we may not even need to re-fetch data!
        var addedToConfig = false;
        // Never try to fold fields with explicitly specified optionFetchContext into 
        // a single responses
        if (oldODSs && !field.optionFilterContext) {
            for (var ii = 0; ii < oldODSs.length; ii++) {
                var ODSConfig = oldODSs[ii];
                if (ODSConfig == null) continue;
                if (this._fieldMatchesODSConfig(field, ODSConfig)) {
                    ODSConfig._fields = [field];
                    this._optionDataSources.add(ODSConfig);
                    oldODSs[ii] = null;
                    addedToConfig = true;
                    /*
                    this.logWarn("setOptionDataSources() field:"+ field.name + 
                        " bound to dataSource ID:"+ optionDSID + 
                        ", [crit:"+ this.echo(optionCriteria) +
                        "] matched existing optionDataSource config block from previous fields " +
                        "so added field to config.");
                    */
                    break;
                } 
            }
        }
        
        if (!addedToConfig && !field.optionFilterContext) {
            for (var ii = 0; ii < this._optionDataSources.length; ii++) {
                var ODSConfig = this._optionDataSources[ii];
                if (this._fieldMatchesODSConfig(field, ODSConfig)) {
                    ODSConfig._fields.add(field);
                    addedToConfig = true;
                    /*
                    this.logWarn("setOptionDataSources() field:"+ field.name + 
                        " bound to dataSource ID:"+ optionDSID + 
                        ", [crit:"+ this.echo(optionCriteria) +
                        "] matched existing optionDataSource config block so added to field.");
                    */
                    break;
                }
            }
        }
        
        if (!addedToConfig) {
            this._optionDataSources.add({
                _dsID:optionDSID,
                _fields:[field],
                _textMatchStyle:field.optionTextMatchStyle,
                _criteria:optionCriteria,
                _optionContext:field.optionFilterContext,
                _optionOperationId:field.optionOperationId
            });
            
            // -- is this the only case where we need to fetch?
            //    Assumption is that in other cases we either already kicked off a fetch
            //    or picked up the field from our 'old ds's', and so already have a resultSet
            
            // keep going for a bit...
            
            
            // in this case we need to kick off a fetch for this dataSource
            // Set the flag to handle this when the body gets redrawn to reflect the new set of 
            // fields
            /*
            this.logWarn("setOptionDataSources() field:"+ field.name + 
                " bound to dataSource ID:"+ optionDSID + 
                ", [crit:"+ this.echo(optionCriteria) +
                "] doesn't match fetch conditions for any other field with an option dataSource, " +
                "so performing a new fetch for it.");
            */
            this._fetchValueMap = true;
        }
    }
    
    // update the valueMaps for new fields bound to optionDataSources for which we already have
    // a resultSet
    for (var i = 0; i < this._optionDataSources.length; i++) {
        if (this._optionDataSources[i]._data != null) {
            this._updateValueMapFromODS(this._optionDataSources[i]);
        }
    }
    
    // If we had any previous optionDataSource config type options which are no longer required,
    // clean them up now by calling 'detroy()' on the resultSet (data object) and letting everything
    // go out of scope at the end of the method.
    if (oldODSs != null) {
        for (var i = 0; i < oldODSs.length; i++) {
            if (oldODSs[i] && oldODSs[i]._data) oldODSs[i]._data.destroy();
        }
    }
},

// little helper method - does a field with an optionDataSource specified match an existing
// optionDataSource config object.

_fieldMatchesODSConfig : function (field, config) {
        
    return (
        // same dataSource
        config._dsID == field.optionDataSource &&
        // optionTextMatchStyle should match
        
        (field.optionTextMatchStyle == config._textMatchStyle) &&
        // if optionOperationId is specified on a field it must also match
        (field.optionOperationId == config._optionOperationId) && 
        // criteria unset or match
        (isc.DataSource.getDataSource(config._dsID).compareCriteria(
            field.optionCriteria || {}, config._criteria || {}) == 0)
    );
    
},

// Given a resultSet bound to a dataSource, update the valueMap for any fields bound to this
// optionDataSource.
_updateValueMapFromODS : function (ODSConfig) {
    var odsID = ODSConfig._dsID,
        fields = ODSConfig._fields;
    
    if (fields == null || fields.length == 0) {
        this.logWarn("_updateValueMapFromODS fired for dataSource:"+ odsID + 
                    " which no longer applies to any fields in this ListGrid");
        return;
    }

    var resultSet = ODSConfig._data;
 
    // avoid trying to get a valueMap if we are called when the cache has just been invalidated
    // or before the ResultSet has been created
    if (resultSet == null || 
        (isc.isA.ResultSet(resultSet) && !resultSet.lengthIsKnown())) return;   

    for (var i = 0; i < fields.length; i++) {
        var field = fields[i],
            valueField = (field.valueField || field[this.fieldIdProperty]),
            map = resultSet.getValueMap(valueField, field.displayField);
            
        // If we haven't yet integrated the field into our fields array, store the
        // valueMap on it directly rather than calling this.setValueMap() -- this way it'll get
        // picked up when setFields() completes
        if (this.getField(field[this.fieldIdProperty]) == null) {
            field.valueMap = map;
        } else {
            this.setValueMap(field[this.fieldIdProperty], map);
        }
    }
},


// _fetchValueMapData()
// for fields with an optionDataSource, kick off a fetch to get a valueMap for the field
_fetchValueMapData : function () {
    
    var shouldSendQueue;
    
    // For each field with an optionDataSource, kick off a fetch if we have no data, otherwise
    // update the valueMap with the existing data
    for (var i = 0; i < this._optionDataSources.length; i++) {
        var ODSConfig = this._optionDataSources[i];
        if (ODSConfig._data == null) {
            if (shouldSendQueue == null) {
                shouldSendQueue = !isc.RPCManager.startQueue();
            }

            var optionDSID = ODSConfig._dsID;
            
            // add component context to request properties for rpc history tree in dev console
            // as a comma delimited list of fields bound to the ODS in question
            var compContext,
                optFields = ODSConfig._fields;
            if (optFields && optFields.length > 0) {
                compContext = "";
                for (var ii = 0; ii < optFields.length; ii++) {
                    compContext += optFields[ii].name;
                    if (ii < optFields.length - 1) compContext += ",";
                }
            }
            
            var context = ODSConfig._optionContext || {};
            isc.addProperties(
                context,
                {showPrompt:false, clientContext:{ODSConfig:ODSConfig},
                 componentContext: compContext,
                 textMatchStyle:ODSConfig._textMatchStyle
                }
            );
            if (ODSConfig._optionOperationId != null) {
                context.operationId = ODSConfig._optionOperationId;
            }
            isc.DataSource.getDataSource(optionDSID).fetchData(
                ODSConfig._criteria, 
                {target:this, methodName:"_fetchValueMapCallback"},
                context
            );
        }
    }

    if (shouldSendQueue) isc.RPCManager.sendQueue();
},


// _fetchValueMapCallback()
// Callback from request to get all data from some field's optionDS.
// Creates a valueMap so we show the displayField value in the field
_fetchValueMapCallback : function (dsResponse, data, dsRequest) {

    var optionDataSourceConfig = dsRequest.clientContext.ODSConfig;
    // if the fields were changed before the fetch returned
    // OR we kicked off 2 fetches against the same dataSource and have already 
    // set up our resultSet data, bail
    if (!optionDataSourceConfig || (optionDataSourceConfig._data != null) ||
        !this._optionDataSources || !this._optionDataSources.contains(optionDataSourceConfig)) 
    {        
        return;
    }
    
    optionDataSourceConfig._data = isc.ResultSet.create({
        dataSource:optionDataSourceConfig._dsID,
        ODSConfig:optionDataSourceConfig,
        targetGrid:this,
        dataChanged:"this.targetGrid._updateValueMapFromODS(this.ODSConfig)",
        initialData:data
    })
    
    // Force an update now
    this._updateValueMapFromODS(optionDataSourceConfig);

    return true;
},


// Helper to destroy() the resultSet(s) we create to handle valueMaps from the server
_dropODSData : function () {
    if (this._optionDataSources == null) return;
    for (var i = 0; i < this._optionDataSources.length; i++) {
        var data = this._optionDataSources[i]._data;
        if (data) data.destroy();
    }
    delete this._optionDataSources;
},

requestVisibleRows : function () {
	// ask for all the rows we're about to render.  This enables better predictive fetching
    // relative to asking for data one row at a time while we render.
	
    if (isc.ResultSet && isc.isA.ResultSet(this.data)) {

        if (this.body == null) return this.data.getRange(0, this.dataPageSize);
        // if the data set is empty no need to fetch
        if (this.data.lengthIsKnown() && this.data.getLength() == 0) return;

        
        if (!this.data.lengthIsKnown()) {
            this.body.showAllRows = false;
        } else {
            // NOTE: this check is necessary because the body itself forces showAllRows to true
            // on init if overflow:visible.  It would probably be more robust to pass a
            // one-time flag to getTableHTML() so that we don't clobber other changes to
            // showAllRows
            this.body.showAllRows = (this.body.overflow == isc.Canvas.VISIBLE ? 
                                     true : this.showAllRecords);
        }

        var drawRect = this.body.getDrawArea();
        
        if (this._scrollCell && isc.isAn.Array(this._scrollCell)) {
            // if scrolling was applied before draw(), move the drawRect to the requested row
            var diff = drawRect[1]-drawRect[0];
            drawRect[0] = this._scrollCell[0];
            drawRect[1] = drawRect[0]+diff;
        }

        // force all rows to be grabbed if we're grouping. (We'll need them anyway.)
        if (this.isGrouped) {
            return this.data.getRange(0, this.groupByMaxRecords);
        } else {            
            return this.data.getRange(drawRect[0], drawRect[1]);
        }
    }
    return null;
},

// Printing
// --------------------------------------------------------------------------------------------
                  
//>	@attr listGrid.printAutoFit (boolean : true : IRW)
// Whether cell contents should wrap during printing.  Equivalent to +link{autoFit}, but
// specific to printed output.
// @group printing
// @visibility external
//<
printAutoFit:true,

//>	@attr listGrid.printWrapCells (boolean : true : IRW)
// Whether cell contents should wrap during printing.  Equivalent to +link{wrapCells}, but
// specific to printed output.
// @group printing
// @visibility external
//<
printWrapCells:true,

//>	@attr listGrid.printHeaderStyle (CSSStyleName : "printHeader" : IRW)
// Style for header cells in printed output.  Defaults to +link{headerBaseStyle} if null.
// @group printing
// @visibility external
//<
printHeaderStyle:"printHeader",			

//>	@attr listGrid.printBaseStyle (CSSStyleName : null : IRW)
// Style for non-header cells in printed output.  Defaults to +link{baseStyle} if null.
// @group printing
// @visibility external
//<			

getPrintHeaders : function (startCol, endCol) {
    var output = isc.SB.create();
    
    var defaultAlign = (this.isRTL() ? isc.Canvas.LEFT : isc.Canvas.RIGHT);

    var currentSpan, spanFieldCount,
        spanHTML = (this.headerSpans == null ? null : []),
        headerHTML = [];
        
    var cellStartHTML = ["<TD CLASS=", (this.printHeaderStyle || this.headerBaseStyle),
                         " ALIGN="].join("");
                         
    // If we're writing out header spans, we'll write two rows of cells
    // Just iterate through the fields once, then assemble the HTML and return it.
	for (var colNum = startCol; colNum < endCol; colNum++) {
	    var field = this.body.fields[colNum];
        if (this.headerSpans != null) {
            if (currentSpan == null) {
                currentSpan = this.headerSpans[0];
                spanFieldCount = 1;
            } else {
                if (!currentSpan.fields.contains(field[this.fieldIdProperty])) {
                    spanHTML.addList([cellStartHTML, "center colspan=", spanFieldCount, ">",
                                        currentSpan.title, "</TD>"]);
                    currentSpan = this.headerSpans[this.headerSpans.indexOf(currentSpan)+1];
                    spanFieldCount = 1;
                } else {
                    spanFieldCount++;
                }
            }
        }
        
	    var align = field.align || defaultAlign;
        headerHTML.addList([cellStartHTML, align, ">",
                            this.getHeaderButtonTitle(field.masterIndex), "</TD>"]);
    }
    
    // if we're showing header spans we need to generate the actual HTML for the last
    // spanning cell and slot in a row for all the spanning cells.
    if (currentSpan != null) {
         spanHTML.addList([cellStartHTML, "center colspan=", spanFieldCount, ">",
                            currentSpan.title, "</TD>"]);
         output.append("<TR>", spanHTML.join(""), "</TR>");
    }
    
    // Output the standard header row
    output.append("<TR>", headerHTML.join(""), "</TR>");
    
    return output.toString();
},

getPrintHTML : function (printProperties, callback) {

    

    var body = this.body;
    // we may have getPrintHTML called while we're undrawn - if so, we'll need to set up our 
    // children here
    if (body == null) {
        this.createChildren();
        body = this.body;
    }
    
    // if the body is pending a redraw, force it now
    if (this.isDirty() || body.isDirty()) {
        this.redraw("updating HTML for printing");
    }
    
    var printProps = isc.addProperties({}, printProperties, this.printProperties);

    // with a partial cache, print only the contiguous cache around the currently visible rows
    
    var startRow, endRow;
    if (isc.isA.ResultSet(this.data) && !this.data.allMatchingRowsCached()) {            
        var visRows = this.body.getVisibleRows(),
            firstVisibleRow = visRows ? visRows[0] : null,
            cachedRange = this.data.getCachedRange(firstVisibleRow);            
        if (cachedRange != null) {
            startRow = cachedRange[0];
            endRow = cachedRange[1];
        }        
    }
    
    // Printing of the body can "go asynchronous" in 2 ways
    // If we have embedded components, retriving their print HTML may be asynchronous
    // If we're attempting to print more than <this.printMaxRows> rows at a time we force
    // the HTML creation to be asynchronous to avoid any script-running-slowly notifications.
    
    
    
    var printWidths = isc.Canvas.applyStretchResizePolicy(this.fields.getProperty("width"), 
                                                printProps.width || isc.Page.getWidth());
    
    // getTablePrintHTML() - implemented at the GridBody level.
    // If it goes asynchronous it'll fire the callback and return null - otherwise it'll
    // return print HTML
    return body.getTablePrintHTML({startRow:startRow, endRow:endRow, callback:callback,
                                     printWidths:printWidths, printProps:printProps});

},

// This is run before getting the body tableHTML for printing
// If printing HTML in chunks it'll be run repeatedly for each chunk!
_prepareForPrinting : function (printWidths, printProperties) {
    
    this.isPrinting = this.body.isPrinting = true;
    this.currentPrintProperties = printProperties;
    
    var body = this.body,
        // don't print editors?
        oldEditorShowing = this._editorShowing;
    
    // properties to store off and restore
    var origProps = isc.getProperties(body, ["autoFit", "wrapCells", "showAllRows",
                                "showAllColumns", "fixedRowHeights", "_fieldWidths", "fields"
                                
                                
                    ]);                                             

    body.showAllRows = true;
    body.showAllColumns = true;
        
    this._editorShowing = false; // never show editors
    body.autoFit = this.printAutoFit;
    body.wrapCells = this.printWrapCells;
    body.fixedRowHeights = !this.printWrapCells;

    var fields = this.fields.duplicate();
    // suppress rendering out the shouldPrint fields
    fields.removeAll(fields.findAll("shouldPrint", false));
    body.fields = fields;
    
    // mark ourselves as unfrozen so we avoid logic to (for example) offset body col-num
    // with frozen col num)
    var oldFrozenFields = this.frozenFields;
    delete this.frozenFields;
    
    body._fieldWidths = printWidths;
    return {oldEditorShowing:oldEditorShowing, oldFrozenFields:oldFrozenFields,
            origProps:origProps};                                                

},

_donePrinting : function (context) {    
    var body = this.body,
        origProps = context.origProps,
        oldEditorShowing = context.oldEditorShowing;
        
    isc.addProperties(body, origProps);
    if (origProps.showAllRows == null) body.showAllRows = null;
    this._editorShowing = oldEditorShowing;
    this.frozenFields = context.oldFrozenFields;
    
    delete this.currentPrintProperties;
    

    this.isPrinting = this.body.isPrinting = false;
    // Force a redraw - this will recalculate current drawn coordinates.
    this.body.redraw();
},

//> @attr ListGrid.printMaxRows (integer : 100 : IRWA)
// Advanced property - when generating printHTML for a large ListGrid, rows are printed in
// batches in order to avoid triggering a native "script is running slowly" browser dialog.
// <P>
// For grids with exceptional numbers of columns or complex formatting logic, this number
// might need to be adjusted downward.
//
// @group printing
// @visibility external
//<
// Note that this means getPrintHTML() is frequently asynchronous for ListGrids
printMaxRows:100,


// Event Handling
// --------------------------------------------------------------------------------------------

//>	@method	listGrid.rowClick()	(A)
//
// Event handler for when rows in the body are clicked upon. The default implementation handles
// firing +link{ListGrid.startEditing()} if appropriate, and fires
// +link{ListGridField.recordClick()} and/or +link{ListGrid.recordClick()} if set. Developers
// should typically implement recordClick rather than overriding this method.
//
//      @param  record      (ListGridRecord)    record object returned from getCellRecord()
//		@param	recordNum   (number)	index of the row where the click occurred
//		@param	fieldNum	(number)	index of the col where the click occurred
//      @param  [keyboardGenerated]   (boolean) indicates whether this was a synthesized record
//                                              click in response to a keyboard event
//		@group	events
//      @see    recordClick()
//
//      @group  events
//		@return	(boolean)	
//      @visibility external
//<
rowClick : function (record, recordNum, fieldNum, keyboardGenerated) {

	// record the last record clicked (used for keyboard navigation)
    this._lastRecordClicked = recordNum;
 
	var record = this.getCellRecord(recordNum, fieldNum),
		field = this.fields[fieldNum];

    // don't fire recordClick on loading rows
    if (Array.isLoading(record)) return;

	// if the field has a 'recordClick' method, call that
	var value = this.getCellValue(record, recordNum, fieldNum);
	var rawValue = this.getRawCellValue(record, recordNum, fieldNum);

	// if the record is a group header, expand/collapse the group
	if (record != null && record._isGroup) {
	    if (this.canCollapseGroup == false) return;
	    // if the user navigates through the grid using arrow up / down keys, don't toggle
	    // the group when the user hits the group header (Except on explicit space keypress which is
	    // more of a deliberate 'pseudo-click' type interaction).
	    
	    if (keyboardGenerated) {
	        var key = isc.EH.getKey();
	        if (key != "Space" && key != "Enter") return;
	    }

        // row indices are invalidated after folder toggle, so flush all edits first
        var mythis = this, myrecord=record;
        if (this.getEditRow() != null) this.saveAllEdits(null, function () {
            mythis.toggleFolder(myrecord);
        });
        else this.toggleFolder(record);
		return;
	}

	if (field.recordClick && !(keyboardGenerated && field.keyboardFiresRecordClick == false)) {
		// CALLBACK API:  available variables:  "viewer,record,recordNum,field,fieldNum,value,rawValue"
		// Convert a string callback to a function
		isc.Func.replaceWithMethod(field, "recordClick", 
                                         "viewer,record,recordNum,field,fieldNum,value,rawValue");
        var returnVal = field.recordClick(this, record, recordNum, field, fieldNum, value, rawValue);
        if (returnVal == false) return false;
	}

	// Note - there is also a (legacy) recordClick handler on the GridRenderer class, with a
	// different signature.
	// 'recordClick()' is not one of the GridAPIs - so won't be fired automatically from the
	// GridRenderer click handling code.  If the method was defined on both the ListGrid and
	// the Body, both methods would be fired.
	// CALLBACK API:  available variables:  "viewer,record,recordNum,field,fieldNum,value,rawValue"
	if (this.recordClick) {
        var returnVal = this.recordClick(this, record, recordNum, field, fieldNum, value, rawValue);
        if (returnVal == false) return false;
    }
    
	// if the cell is editable on click, edit it now
	// We also start editing if editOnFocus is true.
    var editOnClick = this.isEditable() && (this.editEvent == isc.EH.CLICK || this.editOnFocus);

    // one-click toggling of boolean/valueMapped fields.
    // Note: also allows entering of editing if editEvent is click.
    if (field.canToggle && this.canEditCell(recordNum, fieldNum) && this.shouldToggle(field)) {
        var valueMap = this.getEditorValueMap(field, this.getEditedRecord(recordNum,fieldNum));
        // autocreate a valueMap for boolean
        if (valueMap == null && isc.SimpleType.getBaseType(field.type) == this._$boolean) {
            valueMap = [true,false];
        }
        if (valueMap != null) {
            if (!isc.isAn.Array(valueMap)) valueMap = isc.getKeys(valueMap);
            if (valueMap.length > 1) {

                var fieldName = this.getFieldName(fieldNum),
                    editValue = this.getEditedCell(recordNum, fieldNum),
                    index = valueMap.indexOf(editValue);
                index += 1;
                if (index >= valueMap.length) index = 0;
                var oldValue = editValue;
                
                editValue = valueMap[index];
                
                var cancelChange;
                
                if (field.change != null) {
                    this.logInfo("canToggle firing specified field.change() event directly", "gridEdit");
                    cancelChange = this.fireCallback(
                            field.change, 
                            "form,item,value,oldValue",
                            [null,null,editValue,oldValue]
                    ) == false;
                }
                
                if (!cancelChange) {
    
                    // autoSaveEdits: Usually if canToggle is true we instantly commit
                    // however if we're also jumping into edit mode it makes more sense to
                    // just toggle the edit value and save when the user dismisses the editor
                    // as usual.
                    if (!editOnClick && this.autoSaveEdits) {
                        this.setEditValue(recordNum, fieldNum, editValue, true, false);
                            
                    } else {
                        this.setEditValue(recordNum, fieldNum, editValue);
                    }
                    if (field.changed != null) {
                        this.logInfo("canToggle firing specified field.changed() event directly", "gridEdit");
                        this.fireCallback(
                            field.changed,
                            "form,item,value",
                            [null,null,editValue]
                        );
                    }
                    
                    if (this.autoSaveEdits) this.saveEdits(null, null, recordNum, fieldNum);
                }
            }
        }
    }

	if (editOnClick) {
        if (this.handleEditCellEvent(recordNum, fieldNum, isc.ListGrid.CLICK) == true) {
            return true;    
        }
    	// If this was a keyboard event, and the keyboard click field is not editable, iterate 
    	// through the other fields, and edit the first editable one we find
        if (keyboardGenerated) {
            for (var i = 0; i< this.fields.length; i++) {
                if (i == fieldNum) continue;
                if (this.handleEditCellEvent(recordNum, i, isc.ListGrid.CLICK) == true) {
                    return true;
                }
            }
        }
    } 
},

// By default we only toggle if the event occurred over a field's value-icon
// Exception: If _formatBooleanFieldAsImage returns false we didn't write out an icon for 
// the cell
shouldToggle : function (field) {
    // Note: no need to check 'canToggle' - this method is only called for fields where
    // canToggle is true.
    if (!this._formatBooleanFieldAsImages(field)) return true;
    var part = this.getEventPart();
    return (part && part.part == "valueicon");
},

//>	@method	listGrid.rowDoubleClick()	(A)
// Event handler for when a body record is double-clicked.
// <P>
// Default implementation fires 'editCell' if appropriate, and handles firing
// 'recordDoubleClick' stringMethod if defined at the field or LG level (That method has a
// different signature from this one)
//
//      @param  record      (ListGridRecord)    record object returned from getCellRecord()
//		@param	recordNum   (number)	index of the row where the click occurred
//		@param	fieldNum	(number)	index of the col where the click occurred
//      @param  [keyboardGenerated]   (boolean) indicates whether this was a synthesized record
//                                              doubleclick in response to a keyboard event
//
//      @see    recordDoubleClick()
//		@group	events
//		@return	(boolean)	false if first click not on same record; true otherwise
//      @visibility external
//<
rowDoubleClick : function (record, recordNum, fieldNum, keyboardGenerated) {

	var field = this.fields[fieldNum], 
        value = this.getCellValue(record, recordNum, fieldNum),
        rawValue = this.getRawCellValue(record, recordNum, fieldNum);
    // suppress user-defined handlers on the group header node
    if (record != null && record._isGroup) return;

	if (field.recordDoubleClick) {
		// CALLBACK API:  available variables:  "viewer,record,recordNum,field,fieldNum,value,rawValue"
		// Convert a string callback to a function
		isc.Func.replaceWithMethod(field, "recordDoubleClick", 
                                         "viewer,record,recordNum,field,fieldNum,value,rawValue");
        var returnVal = field.recordDoubleClick(this, record, recordNum, field, fieldNum, value, rawValue);
        if (returnVal == false) return returnVal;
	}
    if (this.recordDoubleClick != null) {
		// CALLBACK API:  available variables: "viewer,record,recordNum,field,fieldNum,value,rawValue"
        var returnVal = this.recordDoubleClick(this, record, recordNum, field, fieldNum, value, rawValue);
        if (returnVal == false) return returnVal;
    }

	// if the cell is editable, edit it now
	//	(editCell will return true if we've brought up the cell editor)
	if (this.isEditable() && this.editEvent == isc.EH.DOUBLE_CLICK) {
        if (this.handleEditCellEvent(recordNum, fieldNum, isc.ListGrid.DOUBLE_CLICK) == true) return true;    
    	// If this was a keyboard event, and the keyboard click field is not editable, iterate 
    	// through the other fields, and edit the first editable one we find
        if (keyboardGenerated) {
            for (var i = 0; i< this.fields.length; i++) {
                if (i == fieldNum) continue;
                if (this.handleEditCellEvent(recordNum, i, isc.ListGrid.DOUBLE_CLICK) == true) return true;
            }
        }
    }

},

// Body Context Menu
// --------------------------------------------------------------------------------------------
// This will fire 'ListGrid.cellContextClick' if defined.
// Otherwise implements default behavior of showing context menu for the cell.
// enhancement: check for contextMenu,cellContextClick on cell, row, field?
_cellContextClick : function (record, rowNum, colNum) {

	// clear any previous context menu items
	this.cellContextItems = null;

	// Call handler if defined; return false to cancel context menu if handler returns false.  The
	// handler can call lv.makeCellContextItems(record,recordNum,fieldNum), modify the items, and
	// set lv.cellContextItems to customize the context menu.
	if (this.cellContextClick) {
        var record = this.getCellRecord(rowNum, colNum);
		if (this.cellContextClick(record, rowNum, colNum) == false) return false;
    }

	// show cell context menus, or generic context menu?
	if (this.showCellContextMenus) {
	
		// create the cellContextMenu if necessary
		if (!this.cellContextMenu) this.cellContextMenu = this.getMenuConstructor().create(this.contextMenuProperties);

		// get standard menu items if the handler above did not set custom items
		if (!this.cellContextItems) {
			this.cellContextItems = this.makeCellContextItems(
                this.getCellRecord(rowNum, colNum), rowNum, colNum);
		}

		// if there are any menu items, set and show the menu
		if (isc.isAn.Array(this.cellContextItems) && this.cellContextItems.length > 0) {
			this.cellContextMenu.setData(this.cellContextItems);
			this.cellContextMenu.showContextMenu(this);
		}
		
		// return false to kill the standard context menu
		return false;
				
	} else {
    	// do normal Canvas context menu handling. Will fall through to this.showContextMenu.
    	return true;
	}
},

//> @method ListGrid.getShowChildDataSourceContextMenuItemTitle() [A]
// If +link{ListGrid.canOpenRecordDetailGrid} is true and +link{ListGrid.showCellContextMenus}
// is true, we will show menu items to drill into the child datasources in this grid's context
// menu. This method returns the title for that menu item. Override for localization.
// @param ds (DataSource) child datasource to be drilled into
// @return (string) By default returns <code>"Show " + ds.getPluralTitle()</code>
// @group i18nMessages
// @visibility nestedGrid
//<
getShowChildDataSourceContextMenuItemTitle : function (ds) {
    return "Show " + ds.getPluralTitle();
},

// generate standard cell context menu items
makeCellContextItems : function (record, rowNum, colNum) {
    if (this.dataSource != null) {
        var menuItems = [];

    	// menu items to drill into a child DataSource via a nested grid
        if (this.canOpenRecordDetailGrid) {
            var recordDS = isc.DS.get(this.getRecordDataSource(record)),
                childDataSources = recordDS.getChildDataSources();
            if (childDataSources != null) {
                for (var i = 0; i < childDataSources.length; i++) {
                    var ds = childDataSources[i];
                    menuItems.add({
                        title : this.getShowChildDataSourceContextMenuItemTitle(ds),
                        record : record,
                        dataSource : ds,
                        click : "target.openRecordDetailGrid(item.record, item.dataSource)"
                    });
                }
            }
        }

    	// menu item to edit with an embedded form
        if (this.canOpenRecordEditor) {
            
            menuItems.add({
                title : this.openRecordEditorContextMenuItemTitle,
                record : record,
                click : "target.endEditing();target.openRecordEditor(item.record)"
            });
        }

    	// if we are currently showing anything inside the row offer to dismiss it
        if (record != null && this._openRecord == record) {
            menuItems.add({
                title : this.dismissEmbeddedComponentContextMenuItemTitle,
                click : "target.closeRecord()"
            });
        }

    	// menu item to delete a record
        menuItems.add({
            title : this.deleteRecordContextMenuItemTitle,
            click : "target.removeSelectedData()"
        });

        return (menuItems.length > 0 ? menuItems : null);
    }
	return null;
},

// Hover
// ---------------------------------------------------------------------------------------

// override getCanHover.
// If this.canHover is explicitly set to true or false, respect it,
// Otherwise - if any fields are going to show hovers, return true

getCanHover : function () {
    if (this.canHover != null) return this.canHover;
    var fields = this.getFields();
    if (fields != null) {
        for (var i = 0; i < fields.length; i++) {
            if (fields[i].showHover) return true;
        }
    }
    // Either null or false
    return this.canHover;
},

// NOTE: JSDoc imported from GR
cellHoverHTML : function (record, rowNum, colNum) {
    // If we're showing an editor in the cell suppress the standard cell hover.
    // Exception - if the cell itself is not editable, but the rest of the row is, continue to
    // show hovers over the 
    if (this._editorShowing && this.getEditRow() == rowNum && this.canEditCell(rowNum,colNum) &&
        (!this.editByCell || this.getEditCol() == colNum)) {
            return null;
        }
    var field = this.getField(colNum);
    if (field.showHover == false) return null;
    if (field.showHover == null && !this.canHover) return null;
    
    var value = this.getCellValue(record, rowNum, colNum);

    if (field.hoverHTML) {
        isc.Func.replaceWithMethod(field, "hoverHTML",
                                         "record,value,rowNum,colNum,grid");
        return field.hoverHTML(record,value,rowNum,colNum,this);
    }

    if (value != null && !isc.isAn.emptyString(value) && value != this.emptyCellValue) {    
        return value;
    }
},

//> @method listGridField.hoverHTML()
// StringMethod override point for returning HTML to be shown in hovers over cells in the
// column described by this field.
// <P>
// Called only when +link{listGrid.canHover,canHover} and +link{listGrid.showHover,showHover}
// are both true.
// <P>
// The value of "this" within the method will by the +link{ListGridField,field definition}.
//
// @param record (ListGridRecord) record being hovered over
// @param value  (any) value of the cell being hovered over
// @param rowNum (number) row number where hover occurred
// @param colNum (number) column number where hover occurred
// @param grid   (ListGrid) ListGrid this field is a part of
// @return (HTML) HTML to show in the hover
// @group hovers
// @visibility external
// @example valueHoverTips
//<


//> @attr listGrid.showHoverComponents (boolean : false : IRWA)
// When set to true and canHover is also true, shows a widget hovering at the mouse point.
// <P>
// A number of builtin modes are provided - see +link{type:HoverMode}.
// @group hoverComponents
// @visibility external
//<


//> @type HoverMode
// When +link{ListGrid.canHover, canHover} and 
// +link{ListGrid.showHoverComponents, showHoverComponents} are both true, HoverMode
// dictates the type of UI to be displayed when a user hovers over a row or cell.
// <P>
// There are a number of builtin HoverModes and you can override 
// +link{listGrid.getCellHoverComponent, getCellHoverComponent()} to create your own
// hover behaviors.
//  
//  @value  "detailField"  Show a single field's value in an +link{class:HtmlFlow}. Field 
//      to use is +link{listGrid.detailField}.
//  @value  "details"   Show a +link{class:DetailViewer} displaying those fields from the 
//      record which are not already displayed in the grid.
//  @value  "related"    Show a separate +link{class:ListGrid} containing related-records.
//      See +link{ListGridRecord.detailDS} and +link{ListGrid.recordDetailDSProperty} for 
//      more information.
//  @value  "detailRelated"    Show a +link{class:DetailViewer} displaying those fields
//      from the record not already displayed in the grid, together with a separate
//      +link{class:ListGrid} containing related-records.
// @group hoverComponents
// @visibility external
//<

//> @attr listGrid.hoverMode (HoverMode : null : IRWA)
// When +link{ListGrid.showHoverComponents, showHoverComponents()} is true, the builtin mode
// to use when automatically creating a hover component for rows in this grid.
// <P>
// A number of builtin modes are provided - see +link{type:HoverMode}.  You can also override
// +link{ListGrid.getCellHoverComponent, getCellHoverComponent()} to provide a custom hover
// widget - in that case, this attribute is ignored.
// @group hoverComponents
// @visibility external
//<

_getCellHoverComponent : function (record, rowNum, colNum) {
    if (!this.showHoverComponents) return null;
    // If we're showing an editor in the cell suppress the standard cell hover.
    if (this._editorShowing && this.getEditRow() == rowNum && 
        (!this.editByCell || this.getEditCol() == colNum)) return null;
    var field = this.getField(colNum);
    if (field.showHover == false) return null;
    if (field.showHover == null && !this.canHover) return null;

    if (this.getCellHoverComponent && isc.isA.Function(this.getCellHoverComponent)) {
        return this.getCellHoverComponent(record, rowNum, colNum);
    } else return null;
},

//> @method listGrid.getCellHoverComponent()
// When +link{showHoverComponents} is set, this method is called to get the component to show
// as a hover for the current cell.
// <P>
// By default, this method returns one of a set of builtin components, according to the 
// value of +link{type:HoverMode, listGrid.hoverMode}.  You can override this method 
// to return any component you wish to provide as a hoverComponent, or invoke the superclass
// method to have the default hover component generated, then further customize it.
// <P>
// By default, components returned by <code>getCellHoverComponent()</code> will be
// automatically destroyed when the hover is hidden.  To prevent this, set
// +link{canvas.hoverAutoDestroy} to false on the returned component.
//
// @param record (ListGridRecord) record to get the hoverComponent for
// @return (Canvas | Canvas Properties) the component to show as a hover
// @group hoverComponents
// @visibility external
//<
defaultCellHoverComponentWidth: 300,
defaultCellHoverComponentHeight: 150,
getCellHoverComponent : function (record, rowNum, colNum) {
    return this._getStockEmbeddedComponent(record, false, true, rowNum, colNum);
},

_getStockEmbeddedComponent : function (record, isExpansion, isHover, rowNum, colNum) {
    var gridFields = this.getFields(),
        dsFields = this.dataSource ? isc.getValues(this.getDataSource().getFields()) : 
            gridFields,
        defWidth = (!isHover ? null :
            this.hoverWidth || this.hoverSize || this.defaultCellHoverComponentWidth),
        defHeight = (!isHover ? null :
            this.hoverHeight || this.hoverSize || this.defaultCellHoverComponentHeight),
        remainingFields = [],
        component
    ;

    for (var i=0; i<dsFields.length; i++) {
        var field = dsFields.get(i);
        if (this.dataSource) {
            if (!gridFields.find("name", field.name)) {
                remainingFields.add(field);
            }
        } else {
            // show all fields when not databound
            if (!this.isExpansionField(field)) {
                remainingFields.add(field);
            }
        }
    } 

    var mode = (isHover ? this.hoverMode : (isExpansion ? this.expansionMode : null));

    var props;
    
    // create an appropriate subcomponent and bind it
    if (mode == "detailField") {
        component = this.createAutoChild("expansionDetailField", 
            {contents: record[this.detailField]}
        );
        props = {
            width: (isHover ? defWidth : "100%"),
            height: (isHover ? defHeight : "100%"),
            members: [component]
        };
        
        if (isHover) {
            props = isc.addProperties(props, {
                hoverAutoDestroy: this.hoverAutoDestroy,
                overflow: "auto"
            });
        }

        component = isc.VLayout.create(props);
    } else if (mode == "details") {
        props = { dataSource: this.dataSource, fields: remainingFields };
        
        if (isHover) {
            props = isc.addProperties(props, {
                width: defWidth,
                height: defHeight,
                hoverAutoDestroy: this.hoverAutoDestroy
            });
        }
        component = this.createAutoChild("expansionDetails", props);
        component.setData(record);

    } else if (mode == "related") {
        props = { dataSource: this.getRelatedDataSource(record) };
    
        if (isHover) {
            props = isc.addProperties(props, {
                canEdit: false,
                width: defWidth,
                height: defHeight,
                dataProperties: { context: { showPrompt: false } },
                hoverAutoDestroy: this.hoverAutoDestroy
            });
        }

        if (isExpansion) {
            props = isc.addProperties(props, {
                canExpandRecords: this.childExpansionMode ? true : false,
                expansionMode: this.childExpansionMode,
                canEdit: this.expansionCanEdit
            });
        }

        component = this.createAutoChild("expansionRelated", props);

        // if editing is allowed in the sub-grid, set autoSaveEdits: true
        if (this.expansionCanEdit) component.autoSaveEdits = true;
        component.delayCall("fetchRelatedData", [record, this.dataSource]);

    } else if (mode == "detailRelated") {
        props = { dataSource: this.dataSource, fields: remainingFields };
        
        if (isHover) {
            props = isc.addProperties(props, {
                dataProperties: { context: { showPrompt: false } },
                hoverAutoDestroy: this.hoverAutoDestroy
            });
        }
        
        var detail = this.createAutoChild("expansionDetails", props)        
        detail.setData(record);

        props = { dataSource: this.getRelatedDataSource(record), height: "100%",
            canEdit: (isHover ? false : (isExpansion ? this.expansionCanEdit : null))
        };
        
        if (isHover) {
            props = isc.addProperties(props, {
                dataProperties: { context: { showPrompt: false } },
                hoverAutoDestroy: this.hoverAutoDestroy
            });
        }
        if (isExpansion) {
            props = isc.addProperties(props, {
                canExpandRecords: this.childExpansionMode ? true : false,
                expansionMode: this.childExpansionMode
            });
        }
        
        var related = this.createAutoChild("expansionRelated", props);

        props = { members:[detail, related] };
        if (isHover) {
            props = isc.addProperties(props, {
                width: defWidth,
                height: defHeight,
                hoverAutoDestroy: this.hoverAutoDestroy
            });
        }

        component = this.createAutoChild("expansionDetailRelated", props);

    } else if (mode == "editor") {
        component = this.createAutoChild("expansionEditor", {
            dataSource: this.dataSource,
            fields: remainingFields
        });
        component.editRecord(this.getEditedRecord(rowNum) || record);
    }

    return component;
},

//> @attr listGridField.showAlternateStyle (boolean : null : IRWA)
// When set to false, don't apply alternate-row styling to this field.
// @visibility external
//<

// Selection
// --------------------------------------------------------------------------------------------

// Simple helper methods to avoid having to refer directly to this.selection
// Genericized up to DataBoundComponent, July 2008


//> @method listGrid.selectRecord()
// Select/deselect a +link{Record} passed in explicitly, or by index.
// <P>
// Note that this method selects records unconditionally, allowing multiple selected 
// records, even when +link{listGrid.selectionType} is "single".  To enforce mutually-exclusive 
// record-selection, use +link{listGrid.selectSingleRecord}.
// @include dataBoundComponent.selectRecord()
//<

//> @method listGrid.deselectRecord()
// @include dataBoundComponent.deselectRecord()
//<

//> @method listGrid.selectRecords()
// @include dataBoundComponent.selectRecords()
//<

//> @method listGrid.deselectRecords()
// @include dataBoundComponent.deselectRecords()
//<

//> @method listGrid.selectAllRecords()
// @include dataBoundComponent.selectAllRecords()
//<
selectAllRecords : function () {
    this._dontRefreshSelection = true;
    this.selection.selectAll();
    this._dontRefreshSelection = null;
    this._markBodyForRedraw("select all");
    if (this.getCurrentCheckboxField() != null) this._setCheckboxHeaderState(true);
    this.fireSelectionUpdated();
},

//> @method listGrid.deselectAllRecords()
// @include dataBoundComponent.deselectAllRecords()
//<
deselectAllRecords : function () {
    this._dontRefreshSelection = true;
    this.selection.deselectAll();
    this._dontRefreshSelection = null;
    this._markBodyForRedraw("deselect all");
    if (this.getCurrentCheckboxField() != null) this._setCheckboxHeaderState(false);
    this.fireSelectionUpdated();
},

//> @method listGrid.anySelected()
// @include dataBoundComponent.anySelected()
//<

//> @method listGrid.selectSingleRecord()
// @include dataBoundComponent.selectSingleRecord
// @visibility external
//<

// Keyboard Navigation
// --------------------------------------------------------------------------------------------

//> @method listGrid.keyPress()
// Handle a keyPress event on the ListGrid as a whole.  
// <P>
// Note that the majority of keyboard handling for a ListGrid is performed by
// +link{bodyKeyPress()} and most overrides are better performed there.
//
// @return (boolean) return false to cancel
// @visibility external
//<

//> @method listGrid.bodyKeyPress()
// Handle a keyPress event on the body.  
// <P>
// Default implementation handles navigating between records with arrow keys, and activating
// records with space and enter.
//
// @return (boolean) return false to cancel
//
// @visibility external
//<

_$ArrowUp:"Arrow_Up", _$ArrowDown:"Arrow_Down",
_$Space:"Space", _$Enter:"Enter",
_$f2:"f2",
bodyKeyPress : function (event, eventInfo) {
    
    if (this._editorShowing) {
        var target = event.keyTarget,
            canvasItem;
        while (canvasItem == null && target != this && target != null) {
            canvasItem = target.canvasItem;
            target = target.parentElement;
        }
        if (canvasItem != null && canvasItem.form == this.getEditForm()) {
            var returnVal = this.editorKeyPress(canvasItem, isc.EH.getKey(),
                                isc.EH.getKeyEventCharacterValue());
            return (returnVal == null ? isc.EH.STOP_BUBBLING : returnVal);
        }
    }
    

    if (this.data.getLength() > 0) {

        // if we start editing on keypress, return false to kill the event (avoiding
        // page navigation on backspace keypress, etc)
        var EH = isc.EventHandler,
            keyName = event.keyName;  
        var editOnKeyPress = this.editOnKeyPress && this.isEditable();
        if (editOnKeyPress && this._editOnKeyPress(event, eventInfo)) return false;
        
    	// for arrow keys, navigate to the appropriate record
        if (keyName == this._$ArrowUp) {
            return this._navigateToNextRecord(-1);
        } else if (keyName == this._$ArrowDown) {
            return this._navigateToNextRecord(1);

    	// Generate a click on the current focus record when the user hits Space            
        } else if (keyName == this._$Space) {
            
            if (this.generateClickOnSpace) 
                if (this._generateFocusRecordClick() == false) return false;
                
            if (this.generateDoubleClickOnSpace)
                return this._generateFocusRecordDoubleClick();

    	// Generate a doubleClick on the current focus record when the user hits Enter    
        } else if (keyName == this._$Enter) {
            if (this.generateClickOnEnter)  
                if (this._generateFocusRecordClick() == false) return false;

            if (this.generateDoubleClickOnEnter) 
                return this._generateFocusRecordDoubleClick();
        
        // Start editing on f2 keypress if editOnF2Keypress is true.
        } else if (keyName == this._$f2 && this.editOnF2Keypress &&
                    this.isEditable() && this.editEvent != "none") 
        {
            var rowNum = this.getFocusRow();
            if (rowNum == null) rowNum = 0;
            this.startEditing(rowNum);
            
            return false;
        }
    }
    return true
},

// editOnKeyPress behavior
// - modelled on spreadsheet style application editing
// - normal record selection and navigation occurs on click / arrow keypress when not currently
//   editing the grid
// - on character keypress, start editing (respecting the character typed)

_$f2:"f2",
_$Escape:"Escape",
_$Backspace:"Backspace",
_$Delete:"Delete",
_$keyPress:"keyPress",
_editOnKeyPress : function (event, eventInfo) {

    var keyName = eventInfo.keyName,
        charVal = isc.EH.getKeyEventCharacter(event);  
    // We don't want to start editing if the user hit a non character key, such as a function key
    // or escape, etc   
        
    if (keyName != this._$f2 && keyName != this._$Delete && keyName != this._$Backspace && 
        (keyName == this._$Escape || 
            isc.EH._nonCharacterKeyMap[keyName] || charVal == null || charVal == isc.emptyString)) 
    {
        return false;
    }

    var cell = this.getFocusCell(),
        row = cell[0] || 0,
        col = cell[1] || 0;
    // If we're already showing an editor just bail
    if (this._editorShowing) return false;
    
    // on Enter / f2 keypress don't modify the value in the cell
    var undef;
    if (keyName == this._$Enter || keyName == this._$f2) charVal = null;

    var editVal;
    if (charVal != null) {
        if (keyName == this._$Delete || keyName == this._$Backspace) {
            editVal = null;
        } else if (this.autoSelectEditors) {
            editVal = charVal;
        } else {
            editVal = this.getEditedCell(row,col) + charVal;
        }
        // this flag ensures that when we focus in the item we put selection at the end, rather
        // than selecting the entire value.
        this._editorCursorAtEnd = true;
    }
    return this.handleEditCellEvent(cell[0], cell[1], this._$keyPress, editVal);
    
},

// getArrowKeyAction() - used by _navigateToNextRecord() to determine how the record
// should be hilighted.

// Strings used in navigation styles
_$none:"none", _$focus:"focus", _$select:"select", _$activate:"activate",
getArrowKeyAction : function () {

    var action = this.arrowKeyAction;

    // No action at all trumps everything
    if (action == this._$none) return this._$none;

    // if ctrl key is down always just hilite / focus
    if (isc.EH.ctrlKeyDown()) return this._$focus;
    
    return action;
},

// _navigateToNextRecord()
// Called from this.bodyKeyPress() on arrow keys, to handle navigating around the listGrid.
// If step == +1, we want to navigate to the next record in the list, if step is -1, we want to
// navigate to the previous record.
// Determines which record to navigate to, and falls through to _generateRecordClick() or 
// _hiliteRecord() depending on the result of this.getArrowKeyAction()
_navigateToNextRecord : function (step) {
	// Are we going to simulate a click on the next record or just hilight it?
    var navStyle = this.getArrowKeyAction();
    
	// If keyboard navigation is disabled return true to allow processing to continue.
    if (navStyle == this._$none) return true;

	// Note: we are either going forward or backward one record - assume jumping over multiple 
	// records is not supported
	//
	// Default to selecting the next record
    if (step == null) step = 1;
    
	// Determine which record was last hilighted or clicked
    var newSelectionIndex;

	// By default we want the last row that had keyboard focus
    newSelectionIndex = this.getFocusRow(step > 0);
    
    // Otherwise, get the last record clicked
    if (newSelectionIndex == null) newSelectionIndex = this._lastRecordClicked;
    
    var originalSelection = isc.isA.Number(newSelectionIndex) ? newSelectionIndex : 0;
    
    if (isc.isA.Number(newSelectionIndex)) newSelectionIndex += step;
	// Default to the first record
    else newSelectionIndex = 0;
    
    var lastRow = this.getTotalRows() -1;
	// if we are trying to navigate past the ends just ensure the focus row is selected
    if (newSelectionIndex < 0 || newSelectionIndex > lastRow) {
        // bail if there were no records
        if (lastRow < 0) return true;
        
        // Ensure the original record is selected / focused
        
        newSelectionIndex = originalSelection;
        
    }
	// At this point we are sure that newSelectionIndex is a number.
	// If the number is beyond the end of the list in either direction, or 
	// the record is not enabled, recordIsEnabled() will return false.
	// Try the next record in the step direction, and so on until we find an enabled record or
	// hit the end of the list.

	// If the record is disabled, find the first non-disabled record (in the appropriate
	// direction)
    while (!this.recordIsEnabled(newSelectionIndex, 0)) {
        newSelectionIndex += step;  
    	// if we are trying to navigate past the ends of the list, bail
        if (newSelectionIndex < 0 || newSelectionIndex > lastRow) {
            newSelectionIndex = originalSelection;
            break;
        }
    }
    // move native focus to the selected row so that screen readers will read it
    if (isc.screenReader) {
        this.body._putNativeFocusInRow(newSelectionIndex);
    }
    
    //this.logWarn("navStyle: " + navStyle + ", target index: " + newSelectionIndex);

    if (navStyle == this._$focus) this._hiliteRecord(newSelectionIndex);
    else {
        // if the user hit up arrow on the first row or down arrow on the last row, don't
        // actually force a click handler to fire on the row.
        // This leads to odd interactions with ListGrid editing on click as arrow down on the
        // last row will hide the editor, then arrow down again will show it on the same
        // row but the first editable col.
        // If the user does want to force a click on the current row via the keyboard
        // they can always hit space or enter.
        
        if (newSelectionIndex == originalSelection) {
            var colNum = this._getKeyboardClickNum();
            if (this.body.selectionEnabled() &&
                this.recordIsEnabled(newSelectionIndex, colNum)) 
            {
                this.selection.selectOnMouseDown(this, newSelectionIndex, colNum);
                this.selection.selectOnMouseUp(this, newSelectionIndex, colNum);
                
            }
        }
        else if (navStyle == this._$select) this._generateRecordClick(newSelectionIndex);
        else if (navStyle == this._$activate) this._generateRecordDoubleClick(newSelectionIndex);
    }
    this.scrollRecordIntoView(newSelectionIndex)
    
	// Don't allow the keypress event handling to continue here.
    return false;
},

_getKeyboardClickNum : function () {

	// If this.keyboardClickField was specified, return the appropriate colNum
	// Note - can be specified as a field number or field name...
    var kcf = this.keyboardClickField;
    if (kcf == null) return 0;
    
    if (isc.isA.Number(kcf) && kcf > 0 && kcf < this.fields.length) return kcf;
    
    var kcCol = this.fields.find(this.fieldIdProperty, kcf),
        colNum = (kcCol ? this.fields.indexOf(kcCol) : 0);

    return colNum;        

},

_generateRecordClick : function (recordNum) {    

    this.clearLastHilite();

	// if passed a record, resolve it to an index!
    if (isc.isAn.Object(recordNum)) recordNum = this.getRecordIndex(recordNum);

	// Make sure we're not trying to select a record beyond the ends of the list.
    if (!isc.isA.Number(recordNum) || recordNum < 0) recordNum = 0;
    if (recordNum >= this.data.getLength()) recordNum = this.data.getLength() -1;

	// remember we artificially selected this record from a keyboard event
    this.body._lastHiliteRow = recordNum;
    
    var colNum = this._getKeyboardClickNum();
    
	// Trigger the methods to perform the selection (selection.selectOnMouseDown AND 
	// selection.selectOnMouseUp)
	
    var performSelection = 
        (this.body.selectionEnabled() && this.recordIsEnabled(recordNum, colNum)) ;
	
    if (performSelection) this.body.selectOnMouseDown(this, recordNum, colNum);
    
	// explicitly fire this 'rowClick' method, passing in the additional method flagging this
	// as a keyboard generated click
    this.rowClick(this.getCellRecord(recordNum, colNum), recordNum, colNum, true);
    if (performSelection) this.body.selectOnMouseUp(this, recordNum, colNum);
    
	// Stop event propogation
    return false;

},

//> @method listGrid.getFocusRow() [A]
// Get the row that currently has keyboard focus.  Arrow key navigation moves relative to this
// row.
// 
// @return (Number) rowNum of the current focus row
// @visibility external
//<
// @param last (boolean) if multiple rows are selected, should we return the last row in the
//  selection (rather than the first?
getFocusRow : function (last) {
    
	// We want the last record hilighted by the keyboard.
	// Note: If the last keyboard hilite type event was a generated record click, the
	// lastHiliteRow will match the lastRecordclicked property for this widget.
	// If the last keyboard hilite type event was a hilite (rollover style) event, the 
	// lastHiliteRow will match the lastOverRow for the body.
	// If neither of these are true, we can assume a subsequent mouse event has occurred over
	// a different row, effectively invalidating the _lastHiliteRow property, so should be
    // ignored and deleted
    if (this.body._lastHiliteRow != null && 
        ((this.body._lastHiliteRow == this.body.lastOverRow) || 
           (this.body._lastHiliteRow == this._lastRecordClicked)) )
    {        
        return this.body._lastHiliteRow;
    }
    delete this.body._lastHiliteRow;
    
    // If there is no valid keyboard hilite row, return a record from the end of the selection
    // We use 'direction' param to indicate whether it's more appropriate to return the
    // first or last selected record of a multiple selection
	var selection = this.getSelection();    
    if (selection.length == 0) return null;
    selection = selection[(last ? selection.length -1 : 0)]
    return this.getRecordIndex(selection);

},

getFocusCell : function () {
    return [this.getFocusRow(), this._getKeyboardClickNum()]
},

_generateFocusRecordClick : function () {

    var currentRecord = this.getFocusRow();

    if (currentRecord != null) {
        this._generateRecordClick(currentRecord);
        return false;
    }
    
	// allow event processing to continue..
    return true;
},

_generateRecordDoubleClick : function (rowNum) {

    // determine the appropriate col
    var colNum = this._getKeyboardClickNum();
    // generate a double click, on the appropriate record (and field), passing in the
	// parameter flagging this as a keyboard synthesized click event.
	//this._handleRecordDoubleClick(currentRecord, colNum);
    this.rowDoubleClick(this.getCellRecord(rowNum, colNum), rowNum, colNum, true);
},

_generateFocusRecordDoubleClick : function () {

    // determine the appropriate col
    var colNum = this._getKeyboardClickNum(),
        currentRecord = this.getFocusRow();

    if (currentRecord != null) {
        
        this._generateRecordDoubleClick(currentRecord);
        return false;
    }
    
	// allow event processing to continue..
    return true;

},

// Scrolling
// --------------------------------------------------------------------------------------------

//>	@method	listGrid.scrollRecordToTop()	(A)
// Scroll the listGrid body such that the specified row is visible at the top of the viewport.
//		@group	scrolling
//		@param	rowNum  (number)    Index of the row to scroll into view
//<
scrollRecordToTop : function (rowNum) { return this.scrollRecordIntoView(rowNum, false); },

//>	@method	listGrid.scrollRecordIntoView()	(A)
// Scroll the listGrid body such that the specified row is visible close to the
// center of the viewport.
//		@group	scrolling
//		@param	rowNum  (number)    Index of the row to scroll into view
//<

scrollRecordIntoView : function (rowNum, center) {
    return this.scrollCellIntoView(rowNum, null, center);
},

//>	@method	listGrid.scrollColumnIntoView()	(A)
//			Will scroll the listGrid body such that the specified column is visible close to the
//          centre of the viewport.
//		@group	scrolling
//		@param	colNum  (number)    Index of the column to scroll into view
//<
// Currently unused by the libraries, included for completeness
scrollColumnIntoView : function (colNum, center) {
    return this.scrollCellIntoView(null, colNum, center);
},

//> @method listGrid.scrollToRow()
// Scroll the grid to specified row, loading data if neccessary
// @group scrolling
// @param	rowNum  (number)    Row index of the cell to scroll into view
//<
scrollToRow : function (rowNum) {
    this.scrollCellIntoView(rowNum, 0);    
    // allow for chaining other function calls after scrollToRow()
    return this;
},

// helper callback to weed out calls to scrollToRow that should be ignored. 
// we only want to perform the last call to scrollToRow() that occurs before data arrives.
scrollCellCallback : function (rowNum, colNum, center, alwaysCenter, stamp) {    
    if (stamp == this._currentScrollCall) {
        // reset counter
        this._currentScrollCall = null;
        this.scrollCellIntoView(rowNum, colNum, center, alwaysCenter);
    }
},

//>	@method	listGrid.scrollCellIntoView()	(A)
//			Will scroll the listGrid body such that the specified cell is visible close to the
//          centre of the viewport.
//		@group	scrolling
//		@param	rowNum  (number)    Row index of the cell to scroll into view
//		@param	colNum  (number)    Column index of the cell to scroll into view
//<
// NOTE: as of 2006.11.28, the 'scrollIntoView' method this eventually falls through to supports
// individual x and y positions of top/left, center, bottom/right.
// Therefore easy to modify this method to take x and y positions rather than a single boolean 
// "center" for either centered or displayed at top and left.
// alwaysCenter: scroll even if the cell is already in view (center it or place at left/top
//               according to "center" parameter
scrollCellIntoView : function (rowNum, colNum, center, alwaysCenter) {
    if ((isc.isAn.Array(this.data) && this.data.length == 0 && this.dataSource)
        || (isc.ResultSet && isc.isA.ResultSet(this.data) && !this.data.lengthIsKnown())) {
        // keep track of conscective calls to scrollToRow() so we can only perform the most
        // recent one, before data arrives.        
        if (!this._currentScrollCall) this._currentScrollCall = 1;
        else this._currentScrollCall += 1;
        var stamp = this._currentScrollCall;
        isc.Page.waitFor(this, "dataArrived", 
            {method: this.scrollCellCallback, args: [rowNum, colNum, center, alwaysCenter, stamp],
            target:this}
        );
                
        return;
    }
     
	// if the body isn't drawn, we can't scroll the cell into view - set a flag to scroll the
	// body when it gets drawn
    if (!this.body || !this.body.isDrawn()) {
        this.logInfo("scrollCellIntoView() called before the body has been drawn.  Cell " +
                     rowNum + "," + colNum + " will scrolled into view on draw().");
        this._scrollCell = [rowNum, colNum];
        return;

    }
    
    if (center == null) center = true;

    var x, y, width, height, body = this.body;
    if (rowNum != null) {
        if (!body._isVirtualScrolling) {
            y = body.getRowTop(rowNum);
            height = body.getRowSize(rowNum);
        } else {
            // If the row is already in the viewport, don't call scrollToTargetRow()
            // as this will shift it about which can be confusing if the user is 
            // navigating through rows with the keyboard.
            var undrawn = body._firstDrawnRow == null || rowNum < body._firstDrawnRow || 
                rowNum > body._lastDrawnRow,
                inViewport = !undrawn;
            if (inViewport) {
                var rowTop = body.getRowTop(rowNum),
                    rowHeight = body.getRowHeight ? 
                                    body.getRowHeight(this.getCellRecord(rowNum), rowNum) :
                                    body.cellHeight,
                    scrollTop = body.getScrollTop();
                if (scrollTop > rowTop || 
                    ((body.getViewportHeight() + scrollTop) < (rowTop + rowHeight))) 
                {
                    inViewport = false;
                }
            }
            if (!inViewport) {
                // scrolling to a particular coordinate would be meaningless with unknown row
                // heights
                body._targetRow = rowNum;
                if (center) {
                    body._rowOffset = -1 * ((body.getViewportHeight() / 2) - body.cellHeight);
                } else {
                    body._rowOffset = 0;
                }
                body._scrollToTargetRow();
            }
            return;
        }
    }
    if (colNum != null) {
        x = this.getColumnLeft(colNum);
        width = this.getColumnWidth(colNum);
    }
    
    // Catch the case where we're dirty, and the row being scrolled into view is not yet present
    // in our HTML (so we can't scroll into view until the redraw occurs)
    if (this.isDirty() || this.body.isDirty()) {
        var mustRedraw;
        if (rowNum != null) {
            var scrollHeight = body.getScrollHeight();
            if (y+height > scrollHeight) mustRedraw = true;
        }
        if (!mustRedraw && colNum != null) {
            var scrollWidth = body.getScrollWidth();
            if (x+width > scrollWidth) mustRedraw = true;
        }
        if (mustRedraw) this.redraw("scrollIntoView");
    }
    
	//this.logWarn("ScrollIntoView passed: " + [rowNum, colNum] + 
	//             ", calculated target cell position:" + [x,y] + ", size:" + [width,height]);

    body.scrollIntoView(x,y,width,height, (center ? "center" : "left"), (center ? "center" : "top"), 
                        null, null, alwaysCenter)

},

// Header/Body Scroll Sync
// --------------------------------------------------------------------------------------------
// Note - we keep the body / header's horizontal scroll position in synch by firing 
// syncHeaderScrolling when the body is scrolled, and syncBodyScrolling where the header is
// scrolled.
// We have to have these no-op if the header / body are already at the same place to avoid an
// infinite loop.
bodyScrolled : function (left, top, isFrozen) {
    
    // Assertion
    // the frozen body should never be clipping horizontally so if this was a
    // scroll from the frozen body, just synch up the vertical scroll position of the
    // unfrozen body and we're done.
    // NOTE: There's no obvious way for the user to scroll just the frozen body but this
    // could probably happen from interactions like keyboard events
    
    
    if (isFrozen) {
        this.body._noScrollObservation = true;
        var frozenBody = this.frozenBody;
        if (frozenBody._literalScroll) {
            
            this.body._targetRow = frozenBody._targetRow;
            this.body._rowOffset = frozenBody._rowOffset;
            this.body._scrollRatio = frozenBody._scrollRatio;
            this.body._scrollToTargetRow();
        } else {
            this.body.scrollTo(null, top, "scrollSync");
        }
        delete this.body._noScrollObservation
        
        return;
    }
    
    
    if (this.frozenBody != null) {
        this.frozenBody._noScrollObservation = true;
        var body = this.body,
            frozenBody = this.frozenBody;
        
        // virtual scrolling: The frozen body's virtual scrolling logic
        // (draw area etc) is all driven of the unfrozen body since that can detect
        // cases like "quick drag scrolling" of the scrollbars.
        // If the body is currently doing a "scrollToTargetRow", pick up the
        // stored targetRow info from the body and scrollToTargetRow ourselves
        
        if (body._literalScroll) {
            
            frozenBody._targetRow = body._targetRow;
            frozenBody._rowOffset = body._rowOffset;
            frozenBody._scrollRatio = body._scrollRatio;
            frozenBody._scrollToTargetRow();
        } else {
            frozenBody.scrollTo(null, top, "bodyScrollSync");
        }
        delete this.frozenBody._noScrollObservation;
    }

    
    this.syncHeaderScrolling(left, top);
    this.syncFilterEditorScrolling(left, top);
    this.syncSummaryRowScrolling(left,top);

    // If we took focus from the edit form as part of a redraw and haven't restored it yet
    // restore it now
    
    if (this._editorShowing && this._editorSelection) {
        var editForm = this.getEditForm(), 
            editRow = this.getEditRow(), 
            editColNum = this.getEditCol(),
            editItem = editForm.getItem(this.getEditorName(editRow, editColNum));

        if (editItem) {
            if (!editItem.hasFocus && 
                (editForm.hasFocus || isc.EH.getFocusCanvas() == null))
            {
                this._restoreFocusAfterRedraw(editColNum);
            } else {
                delete this._editorSelection;
            }
        }
    }

},


syncHeaderScrolling : function (left, top) {
	if (left != null && this.header) {
        if (!this.isRTL()) {
            if (left != this.header.getScrollLeft()) this.header.scrollTo(left, null, "headerScrollSync");
        } else {
        	
            var header = this.header,
                body = this.body,
                headerMaxScroll = header.getScrollWidth() - header.getViewportWidth(),
                headerScrollPos = headerMaxScroll - header.getScrollLeft(),
                bodyMaxScroll = body.getScrollWidth() - body.getViewportWidth(),
                bodyScrollPos = bodyMaxScroll - left;
            /*
            this.logWarn("scroll sync: body new left: " + left + 
                         ", body max: " + bodyMaxScroll +
                         ", body pos: " + bodyScrollPos +
                         ", header current left: " + header.getScrollLeft() +
                         ", header max: " + headerMaxScroll +
                         ", header pos: " + headerScrollPos +
                         ", will scroll header to: " + (headerMaxScroll - bodyScrollPos));
            */
                     
            if (bodyScrollPos != headerScrollPos) {
                header.scrollTo(headerMaxScroll - bodyScrollPos, null, "scrollSync");
            }
        }
    }
},

// when the header is scrolled, keep the body scrolled in synch with it!
headerScrolled : function () {
    
    if (!this._delayingBodyScrolling) {
        this._delayingBodyScrolling = this.delayCall("syncBodyScrolling");
    }
},


syncBodyScrolling : function () {
    delete this._delayingBodyScrolling;
    var left = this.header.getScrollLeft();
    if (this.body) { 
        if (!this.isRTL()) {
            if (left != this.body.getScrollLeft()) this.body.scrollTo(left, null, "scrollSync");
            
        } else {
            var header = this.header,
                body = this.body,
                headerMaxScroll = header.getScrollWidth() - header.getViewportWidth(),
                headerScrollPos = headerMaxScroll - header.getScrollLeft(),
                bodyMaxScroll = body.getScrollWidth() - body.getViewportWidth(),
                bodyScrollPos = bodyMaxScroll - left;

            if (bodyScrollPos != headerScrollPos) {
                body.scrollTo(bodyMaxScroll - headerScrollPos, null, "scrollSync");
            }
        }
    }
},

// if we are showing a filter editor we must keep that horizontally scrolled to the same 
// position as the body
syncFilterEditorScrolling : function (left, top) {
    if (this.filterEditor != null && this.filterEditor.body != null &&
    	// No op if they are already in synch to avoid an infinite loop
        this.filterEditor.body.getScrollLeft() != left) 
    {
        this.filterEditor.body.scrollTo(left, null, "scrollSync");
    }
},

syncSummaryRowScrolling : function (left,top) {
    if (this.summaryRow != null && this.showGridSummary && this.summaryRow.body != null &&
    	this.summaryRow.body.getScrollLeft() != left) 
    {
        this.summaryRow.body.scrollTo(left, null, "scrollSync");
    }
},

// RollOver
// --------------------------------------------------------------------------------------------

_hiliteRecord : function (recordNum) {

    if (!isc.isA.Number(recordNum)) {
        recordNum = this.getRecordIndex(recordNum);
    }

	// Make sure we're not trying to select a record beyond the ends of the list.
    if (!isc.isA.Number(recordNum) || recordNum < 0) recordNum = 0;
    if (recordNum >= this.data.getLength()) recordNum = this.data.getLength() -1;

    this.clearLastHilite();
    
	// note the row number hilighted by keyboard navigation
    this.body._lastHiliteRow = recordNum;

	// set this.body.lastOverRow, so the recordStyle will be updated to the mouseOver style
	
    this.body.lastOverRow = recordNum;
    this.body.lastOverCol = 0;  // required to make the GR believe the mouse was over a real cell
    
	// no need to calculate the style - setRowStyle will achieve that
    this.bodies.map("setRowStyle", recordNum); 
    
},

//>	@method	listGrid.clearLastHilite()	(A)
// Unhilites the last hilited item.
//		@group	events, hiliting
//<
clearLastHilite : function () {
    if (!this.body) return;

	// clear the pointer to the last row hilited via keyboard navigation
    this.body._lastHiliteRow = null;

    var rowToClear = this.body.lastOverRow;
	if (isc.isA.Number(rowToClear)) {
        delete this.body.lastOverRow;
    	// no need to calculate new styleName here - let setRowStyle determine that
        if (this.showRollOver) this.body.updateRollOver(rowToClear);
	}

},

// ListGrid tab-index management.
// --------------------------------------------------------------------------------------------


// Note that we basically use the body like a focusProxy - when focus() is called, focus
// will go to the body.
// o Set _useFocusProxy to false - we don't want the grid to ever have native focus 
// o Set _useNativeTabIndex to false - this is done AFTER creating the body so the body can 
//   pick up the '_useNativeTabIndex' as explicitly specified on the ListGrid before that 
//   property gets overridden.
_useFocusProxy:false,

// Override _setTabIndex to also set the tab index of the header and body
_setTabIndex : function (index, auto,a,b) {
    this.invokeSuper(isc.ListGrid, "_setTabIndex", index,auto,a,b);
    
	// update the tab index of the header and body too!
	// Note: we're marking these tabIndices as auto-allocate:false, as we don't want the 
	// standard canvas tabIndex management to be triggered during addChild, etc.
	var canTabToHeader = this.canTabToHeader;
	if (canTabToHeader == null) canTabToHeader = isc.screenReader;
    if (this.header != null && canTabToHeader) this.header._setTabIndex(index, false);
    
    if (this.body != null) this.body._setTabIndex(index, false);
},

// Override setAccessKey to set the accessKey on the body rather than on the listGrid
setAccessKey : function (accessKey) {
	// call Super - will remember this.accessKey, (though it won't actually set it on the LV handle)
    this.Super("setAccessKey", arguments)
    if (this.body != null) this.body.setAccessKey(accessKey);
},

// Override setFocus to focus on the body rather than the ListGrid
setFocus : function (newfocus) {
    if (this.body != null) this.body.setFocus(newfocus);
},

// Override _canFocus() - we are focusable if the body is focusable
// Note that the body already picks up the 'canFocus' attribute from the ListGrid, if specified
_canFocus : function () {
    if (this.body) return this.body._canFocus();
    return false;
},

// Body Clicks
// --------------------------------------------------------------------------------------------

//>	@method	listGrid.recordClick()    
// Executed when the listGrid receives a 'click' event on an enabled, non-separator
// record. The default implementation does nothing -- override to perform some action
// when any record or field is clicked.<br>
// A record event handler can be specified either as
// a function to execute, or as a string of script to evaluate. If the handler is defined
// as a string of script, all the parameters below will be available as variables for use
// in the script.<br>
//	To do something specific if a particular field is clicked, add a recordClick
//	method or string of script to that field (same parameters) when you're setting up
//	the list.<br>
//	<b>Notes:</b><ul>
//  <li>This will not be called if the click is below the last item of the list.</li>
//  <li>This method is called from the default implementation of
//  +link{method:listGrid.rowClick}, so if that method is overridden
//  this method may not be fired.</li></ul>
//		@group	events
//
// @param	viewer		(listGrid)	the listGrid that contains the click event
// @param	record		(ListGridRecord)	the record that was clicked on
// @param	recordNum	(number)	number of the record clicked on in the current set of
//                                  displayed records (starts with 0)
// @param	field		(ListGridField)	the field that was clicked on (field definition)
// @param	fieldNum	(number)	number of the field clicked on in the listGrid.fields
//                                  array
// @param	value       (object)    value of the cell (after valueMap, etc. applied)
// @param	rawValue	(object)	raw value of the cell (before valueMap, etc applied)
//
// @see    rowClick()
//
// @visibility external
//<
// NOTE: params not needed for default no-op implementation
recordClick : function () {},


//>	@method	listGrid.recordDoubleClick()  
// Executed when the listGrid receives a 'doubleClick' event on an enabled, non-separator
// record. The default implementation does nothing -- override to perform
// some action when any record or field is double clicked.<br>
// A record event handler can be specified either as a function to execute, or as a string
// of script to evaluate. If the handler is defined as a string of script, all the
// parameters below will be available as variables for use in the script.<br>
// To do something specific if a particular field is double clicked, add a
// recordDoubleClick method or string of script to that field (same parameters) when you're
// setting up the list.<br>
// <b>Notes:</b><ul>
// <li>This will not be called if the click is below the last item of the list.</li>
// <li>This method is called from the default implementation of +link{method:listGrid.rowDoubleClick},
// so if that method is overridden this method may not be fired.</li></ul>
//		@group	events
//
// @param	viewer		(listGrid)	the listGrid that contains the doubleclick event
// @param	record		(ListGridRecord)	the record that was double-clicked
// @param	recordNum	(number)	number of the record clicked on in the current set of
//                                  displayed records (starts with 0)
// @param	field		(ListGridField)	the field that was clicked on (field definition)
// @param	fieldNum	(number)	number of the field clicked on in the listGrid.fields
//                                  array
// @param	value       (object)	value of the cell (after valueMap, etc. applied)
// @param	rawValue	(object)	raw value of the cell (before valueMap, etc applied)
//
// @see    rowDoubleClick()
//
// @visibility external
//<
// NOTE: params not needed for default no-op implementation
recordDoubleClick : function () {},

// --------------------------------------------------------------------------------------------
// Summary row
// --------------------------------------------------------------------------------------------

// If grid.showGridSummary is true, generate a 'summaryRow' auto child to show summaries at
// the bottom of the list grid. 
// Default behavior is to show totals for numeric fields

//> @attr listGrid.showGridSummary (boolean : false : IRW)
// Should this ListGrid show a summary row beneath the last record of the grid. This summary
// row will contain per-field summary information. See +link{listGridField.showGridSummary} and
// +link{listGrid.getGridSummaryFunction()} for details on how the summary value to be displayed
// for each column will be calculated.
// <P>
// Note that the +link{listGrid.summaryRow,summaryRow autoChild} will be created to actually
// display the summary row.
// @visibility external
//<
showGridSummary:false,

//> @attr listGrid.invalidSummaryValue (string : "&amp;nbsp;" : IRWA)
// Value to display to the user if showing summary values (through +link{listGrid.showGridSummary},
// +link{listGrid.showGroupSummary} or +link{listGridFieldType,listGridFieldType:"summary"}), and
// the summary function returns <code>"null"</code> (implying it was unable to calculate a
// valid summary value).
// @visibility external
//<
invalidSummaryValue:"&nbsp;",

//> @attr listGrid.includeInSummaryProperty (string : "includeInSummary" : IRW)
// Property name on a record that will be checked to determine whether a record should
// be included when calculating totals for the +link{listGrid.showGridSummary,grid summary}.
// @visibility external
//<
includeInSummaryProperty:"includeInSummary",

//> @attr listGridRecord.includeInSummary (boolean : null : IRW)
// If specified as false this record should be ignored when calculating summary totals 
// to be shown in the +link{listGrid.showGridSummary,summary row} for this grid.
// <P>
// Note that <code>includeInSummary</code> is the default property name for this attribute,
// but it may be modified via +link{listGrid.includeInSummaryProperty}.
// @visibility external
//<


//> @attr listGrid.gridSummaryRecordProperty (string : "isGridSummary" : IRW)
// If +link{listGrid.showGridSummary} is true, this attribute will be set to true on the
// record object representing the grid summary row.
// @visibility external
//<
gridSummaryRecordProperty:"isGridSummary",

//> @attr listGridRecord.isGridSummary (boolean : null : IRW)
// This attribute will automatically be set to true for the record representing the 
// grid-level summary row shown if +link{listGrid.showGridSummary} is true.
// <P>
// Note that <code>isGridSummary</code> is the default property name for this attribute but
// it may be modified by setting +link{listGrid.gridSummaryRecordProperty}
// @visibility external
//<

//> @attr listGrid.groupSummaryRecordProperty (string : "isGroupSummary" : IRW)
// If +link{listGrid.showGroupSummary} is true, this attribute will be set to true on each
// record object representing a group-level summary row.
// @visibility external
//<
groupSummaryRecordProperty:"isGroupSummary",

//> @attr listGridRecord.isGroupSummary (boolean : null : IRW)
// This attribute will automatically be set to true for records representing  
// group-level summary rows shown if +link{listGrid.showGroupSummary} is true.
// <P>
// Note that <code>isGroupSummary</code> is the default property name for this attribute but
// it may be modified by setting +link{listGrid.groupSummaryRecordProperty}
// @visibility external
//<

//> @method listGrid.setShowGridSummary()
// Setter for the +link{listGrid.showGridSummary} attribute
// @param showGridSummary (boolean) new value for this.showGridSummary
// @visibility external
//<
setShowGridSummary : function (showGridSummary) {
    if (this.showGridSummary == showGridSummary) return;
    this.showGridSummary = showGridSummary;
    if (this.showGridSummary) {
        this.showSummaryRow();
    } else {
        this.clearSummaryRow();
    }
},

//> @method listGrid.recalculateSummaries()
// Recalculates values for fields with 
// +link{listGridField.recordSummaryFunction, summary-functions} defined and for values 
// displayed in the +link{listGrid.showGridSummary,grid summary} and
// +link{listGrid.showGroupSummary,group summary rows}.
// @param [records] (Array of Record) Optional array of records to recalculate summaries for, 
//                  or null for all records
// @visibility external
//<
recalculateSummaries : function (records, suppressRedraw) {
    suppressRedraw = (suppressRedraw != null ? suppressRedraw : true);
    this.calculateRecordSummaries(records, suppressRedraw);
    
    this._recalculateSummaries(!suppressRedraw);
},

_recalculateSummaries : function (calculateSummaryRow) {
    if (this.showGridSummary && this.summaryRow != null && calculateSummaryRow != false) {
        this.summaryRow._recalculateSummaries();
    }
    if (this.showGroupSummary) {
        this.refreshGroupSummary();
    }
},

// shouldShowGridSummary() - determines whether a field should show a grid summary
// If field.showGridSummary is specified, respect it, otherwise check for
// the presence of a field level summaryFunction, or a default summaryFunction for the field type
shouldShowGridSummary : function (field) {
    if (field.showGridSummary != null) return field.showGridSummary;

    return (field.getGridSummary != null || this.getGridSummaryFunction(field) != null)
},

//> @method listGrid.getGridSummaryFunction() [A]
// Determines the +link{type:SummaryFunction} to use when calculating per-field summary values
// describing multiple records in this grid.
// Used to determine the summary function to use for both +link{listGrid.showGridSummary} and
// +link{listGrid.showGroupSummary}.
// <P>. 
// Default implementation picks up +link{listGridField.summaryFunction} if explicitly specified,
// otherwise checks for a default summary function based on field type (see
// +link{SimpleType.setDefaultSummaryFunction()})
// @param field (listGridField) field to check for summary function
// @return (SummaryFunction) summary function for the field in question
// @visibility external
//<
getGridSummaryFunction : function (field) {
    if (!field) return;
    // If an explicit summaryFunction is specified, pick it up.
    // Otherwise pick up the default based on the field type.
    var summaryFunction = field.summaryFunction ||
                           isc.SimpleType.getDefaultSummaryFunction(field.type);
    
    return summaryFunction;
},

//> @method listGrid.getGridSummary() [A]
// When +link{listGrid.showGridSummary} is <code>true</code> this method is called for each field
// which will show a grid summary value (as described in +link{ListGridField.showGridSummary})
// to get the summary value to display below the relevant column.
// <P>
// The default implementation is as follows:
// <ul><li>If this is a databound grid and not all data is loaded, returns null for every field</li>
//     <li>Otherwise if +link{ListGridField.getGridSummary()} is defined, calls that method passing
//         in the current data set for the grid</li>
//     <li>If +link{ListGridField.getGridSummary()} is undefined, makes use of the
//          +link{listGrid.getGridSummaryFunction(),standard summary function} for the field to
//          calculate the summary based on the current data set</li>
// </ul>
// This method may return an array of values. This implies that the grid summary should show
// multiple rows. Note that if a field has more than one summaryFunction specified, this method
// will pick up values from each summary function and return them in an array, meaning
// these summaries will show up on multiple rows in the grid.
// <P>
// This method may be overridden to completely customize the summary value displayed for
// columns in this grid. An example use case would be when summary information is available on
// the client and does not need to be calculated directly from the data.
// <P>
// <b>Note:</b> this method will not be called if +link{listGrid.summaryRowDataSource} is
// specified.
//
// @param field (ListGridField) field for which the summary value should be returned
// @return (any) summary value to display for the specified field.
// @visibility external
//<
getGridSummary : function (field) {

    if (!field || !this.data || (isc.isA.ResultSet(this.data) && !this.data.lengthIsKnown())) 
        return;
    var data = this.getOriginalData(),
        isRS = isc.ResultSet && isc.isA.ResultSet(data),
        isTree = isc.isA.Tree(data);
        
    if (isTree) {
        data = data.getDescendants(data.getRoot());
        isRS = isc.ResultSet && isc.isA.ResultSet(data);
    }
        
    if (isRS && !data.allMatchingRowsCached()) {
        this.logWarn("Unable to show summary values - dataset not completely loaded");
        return;
    }
    
    var localData = isRS ? data.getRange(0, data.getLength()) : data;

    // If we have outstanding / unsaved edits, we want to use those in our calculations
    var editRows = this.getAllEditRows();
    if (editRows != null && editRows.length > 0) {
        // ensure we don't stomp on our live data array!
        localData = localData.duplicate();
        
        for (var i = 0; i < localData.length; i++) {
            var record = localData[i];
            var rowNum = this.getEditSessionRowNum(record);
            if (rowNum != null) localData[i] = this.getEditedRecord(rowNum, null, true);
        }
    }    
    // getGridSummary() is an override point on a listGridField which allows the developer to
    // calculate a grid summary based on the group summaries already calculated for the grid.
    // It takes an additional param -- an array of 'groupSummary' objects.
    // groupSummary contains
    // - groupName and groupValue to identify which group it belongs to
    // - for each field in the grid, the summary value for the records in the group.
    
    if (field.getGridSummary) {
        var groupSummaries;
        if (this.isGrouped && this.showGroupSummary && (this.groupTree != null)) {
            groupSummaries = this.assembleGroupSummaries();
        }        
        return field.getGridSummary(localData, field, groupSummaries);
    }

    return this.getSummaryValue(localData, field);
},

//> @attr listGrid.summaryRowDataSource (DataSource : null : IRA)
// If +link{listGrid.showGridSummary} is true, by default summary values are calculated on the 
// client based on the current data-set for the grid (see +link{ListGrid.getGridSummary()} and
// +link{ListGrid.getGridSummaryFunction()}).
// <P>
// In some cases however it may make sense to calculate summary values on the server and retrieve
// them via a dataSource fetch. If set, this property specifies a dataSource to fetch against for
// the summary row. The dataSource should return a single record with summary data for each
// field for which summary data should be shown. Note that specifying this property completely
// bypasses the standard client-side grid summary calculation logic.
// <P>
// The fetch may be further customized via +link{listGrid.summaryRowCriteria} and
// +link{listGrid.summaryRowFetchRequestProperties}
// @visibility external
//<
// summaryRowDataSource:null,
getSummaryRowDataSource : function () {
    return this.summaryRowDataSource;
},

//> @attr listGrid.summaryRowCriteria (Criteria : null : IRWA)
// If +link{listGrid.showGridSummary} is true, and a +link{listGrid.summaryRowDataSource} is specified
// this property may be used to specify fetch criteria to apply when retrieving summary data
// to show in the summary row. If unset, and any filter criteria have been specified for the
// grid, they will be used.
// @visibility external
//<
// summaryRowCriteria:null,
getSummaryRowCriteria : function () {
    if (this.summaryRowCriteria != null) return this.summaryRowCriteria;
    var data = this.getOriginalData();
    if (isc.ResultSet && isc.isA.ResultSet(data)) {
        return this.data.getCriteria();
    } 
    return this.getInitialCriteria();
},

//> @attr listGrid.summaryRowFetchRequestProperties (DSRequest Properties : null : IRWA)
// If +link{listGrid.showGridSummary} is true, and a +link{listGrid.summaryRowDataSource} is specified
// this property may be used to customize the fetch request used when retrieving summary data
// to show in the summary row. An example use case might be specifying a 
// +link{DSRequest.operationId} to perform a custom fetch operation which retrieved only summary
// values based on criteria.
//
// @visibility external
//<
//summaryRowFetchRequestProperties:null,
getSummaryRowFetchRequestConfig : function () {
    return isc.addProperties(
                    {textMatchStyle:this.autoFetchTextMatchStyle, 
                     showPrompt:false,
                     startRow:0, endRow:1}, 
                    this.summaryRowFetchRequestDefaults,
                    this.summaryRowFetchRequestProperties);
},

//> @method listGrid.getGridSummaryData()
// This method returns the data to be displayed in the +link{summaryRow} when +link{showGridSummary}
// is true.
// <P>
// By default this will call +link{getGridSummary} for each field and generate an array of records
// containing the resulting values.
// <P>
// This method may be overridden for custom grid-summary display, and may return multiple records
// if more than one summary row is desired.
// 
// @visibility external
//<
getGridSummaryData : function (recalculate) {
    
    if (this._gridSummaryData && !recalculate) return this._gridSummaryData;
      
    var fields = this.completeFields || this.fields,
        data = [];
    
    for (var i = 0; i < fields.length; i++) {
        var field = fields[i],
            fieldName = fields[i].name;
        if (!this.shouldShowGridSummary(field)) continue;
        
        var values = this.getGridSummary(field);
        // getGridSummary returns either an atomic value or an array. If an array is returned
        // we map this to multiple records - this is how multiple summary functions give us
        // a multi-line result in the grid-summary.
        
        if (!isc.isAn.Array(values)) {
            values = [values]
        }
        for (var ii = 0; ii < values.length; ii++) {
            if (data[ii] == null) {
                data[ii] = {};
                // This property is used to allow summary fields to suspend the standard
                // summary from data in this record [so a standard 'grid summary' type sum etc can
                // be performed against the live grid data].
                data[ii][this.gridSummaryRecordProperty] = true;
                // this prevents the checkbox select column from showing a checkbox
                // for the summary row
                
                data[ii][this.recordCanSelectProperty] = false;
            }
            data[ii][fieldName] = values[ii];
        }
    }
    
    this._gridSummaryData = data;
    return data;
},

//> @method listGrid.getGroupSummaryData()
// If this grid is +link{listGrid.groupByField,grouped}, and +link{showGroupSummary} is true, 
// this method will be called for each group to return the group summary data to display at the
// end of the group.
// <P>
// By default this will call +link{listGridField.getGroupSummary} if defined for each field and 
// generate an array of records containing the resulting values. If no explicit per-field
// getGroupSummary method is present, this method will fall back to calling the appropriate
// +link{listGridField.summaryFunction}.
// <P>
// This method may be overridden for custom group-summary display, and may return multiple records
// if more than one summary row per group is desired.
//
// @param records (Array of Records) the records in the group, for which the summary values are
//   being calculated
// @param groupNode (object) object with specified groupValue and groupName for this group
//
// @visibility external
//<

getGroupSummaryData : function (records, groupNode) {

    var summaryData = [];
    for (var i = 0; i < this.fields.getLength(); i++) {
        var field = this.getField(i),
            fieldName = field.name,
            summaryValue;
        if (this.shouldShowGroupSummary(field)) {
            summaryValue = this.getGroupSummary(records, field, groupNode);
            
            // handle being passed an atomic value or an array. If an array
            // is returned we show multiple summary rows.
            if (!isc.isAn.Array(summaryValue)) summaryValue = [summaryValue];
            for (var ii = 0; ii < summaryValue.length; ii++) {
                var summaryRecord = summaryData[ii];
                if (summaryRecord == null) {
                    summaryRecord = summaryData[ii] = {};
                    
                    summaryRecord.customStyle = this.groupSummaryStyle;
                    // mark as disabled so we don't respond to clicks / attempts to select etc.
                    // should have no effect on styling since customStyle is static
                    summaryRecord[this.recordEnabledProperty] = false;
                    // don't include this record in the overall grid summary row!
                    summaryRecord[this.includeInSummaryProperty] = false;
                    
                    // flag the record as a groupSummary record
                    // This allows custom record summary functions to treat it
                    // differently if necessary
                    summaryRecord[this.groupSummaryRecordProperty] = true;
                }
                var currentVal = summaryValue[ii];
                
                // if the summary function returns null, display the
                // invalid summary value
                if (currentVal== null) {
                    currentVal = this.invalidSummaryValue;
                } else {
                    
                    // if a group summary formatter is specified apply it now.
                    
                    if (field.formatGroupSummary) {
                        if (isc.isA.String(field.formatGroupSummary)) {
                            field.formatGroupSummary =
                                isc.Func.expressionToFunction("value",field.formatGroupSummary);
                        }
                        currentVal = field.formatGroupSummary(currentVal);
                    // if this is a "count" field with no explicitly specified grid or group
                    // summary formatter default to showing the plural title or title.
                    } else {
                        var summaryFunc = this.getGridSummaryFunction(field);
                        if (!isc.isAn.Array(summaryFunc)) summaryFunc = [summaryFunc];
                        if (summaryFunc[ii] == "count") {
                            var title = field.pluralTitle;
                            if (title == null) title = field.title;
                            if (title != null) currentVal += " " + title;
                        }
                    }
                }
                summaryRecord[fieldName] = currentVal;
            }
        }
    }
    return summaryData;
},

// This method is called to put together the group level summary values and pass them
// to the getGridSummary method, so developers can look at the summaries, rather than
// necessarily calculating grid summaries based on every data record.
assembleGroupSummaries : function (node, summaries) {
    var tree = this.groupTree;
    if (!node) node = tree.getRoot();
    if (!summaries) summaries = [];
    
    var folders = tree.getFolders(node);
    for (var i = 0; i < folders.length; i++) {
        var headerNode = folders[i];
        
        if (this.groupByFieldSummaries == null ||
            this.groupByFieldSummaries.contains(headerNode.groupName)) 
        {
        
            var summaryRecords = tree.combineWithEditVals(tree.getRecordsInGroup(headerNode));
            var summaryData = this.getGroupSummaryData(summaryRecords, headerNode);
    
            for (var ii = 0; ii < summaryData.length; ii++) {
                var groupSummary = isc.addProperties({},summaryData[ii]);
                delete groupSummary.customStyle;
                delete groupSummary[this.recordEnabledProperty];
                delete groupSummary[this.includeInSummaryProperty];
                groupSummary.groupName = headerNode.groupName;
                groupSummary.groupValue = headerNode.groupValue;
                
                summaries.add(groupSummary);
            }
        }
        
        // make this recursive
        this.assembleGroupSummaries(folders[i], summaries);
    }
    return summaries;

},

// getSummaryValue() - generic handler to take a bunch of records and a field definition and
// return the summary value from them. Used for both grid level summaries and group-level summaries
getSummaryValue : function (records, field) {
    
    // pull out any records where includeInSummary is false
    var includedRecords = [];
    for (var i = 0; i < records.length; i++) {
        var record = records[i];
        if (!record || (record[this.includeInSummaryProperty] == false)) continue;
        includedRecords[includedRecords.length] = record;
    }
    
    var summaryFunction = this.getGridSummaryFunction(field);
    if (summaryFunction != null) {
        if (!isc.isAn.Array(summaryFunction)) {
            summaryFunction = [summaryFunction]
        }
        var results = [];
        for (var i = 0; i < summaryFunction.length; i++) {

            var currentFunction = summaryFunction[i];
            if (currentFunction != null) {
                results[i] = isc.SimpleType.applySummaryFunction(includedRecords, 
                                            field, currentFunction);
            }
        }
        return results;
    }
    return null;
},

shouldShowGroupSummary : function (field) {
    if (field.showGroupSummary != null) return field.showGroupSummary;
    return (field.getGroupSummary != null || this.getGridSummaryFunction(field) != null)
},


getGroupSummary : function (records, field, groupNode) {
    var summaryValue;
    if (field.getGroupSummary != null) {
        summaryValue = field.getGroupSummary(records, field, groupNode);
    } else {
        summaryValue = this.getSummaryValue(records, field);
    }
    return summaryValue;
},

// field level summaries
getRecordSummary : function (recordNum, summaryField) {
    
    // we want to use edit vals when calculating totals
    var record = isc.isAn.Object(recordNum) ? recordNum : this.getEditedRecord(recordNum);
    
    // override point for a completely custom method based on the record without looking at other
    // fields, etc
    if (summaryField.getRecordSummary != null) {
        return summaryField.getRecordSummary(record, summaryField);
    }
    
    var fieldsToInclude = [],
        fields = this.fields;
    
    for (var i = 0; i < fields.length; i++) {
        var field = fields[i];
        
        // should a field be included in this summary calculation?
        
        // Never include a field in its own calculation!
        if (field.name == summaryField.name) {
            // partial summary - only include fields up to this one (index < this field's index)
            if (summaryField.partialSummary) break;
            continue;
        }

        var shouldInclude = field.includeInRecordSummary;
        
        // If 'includeInRecordSummary' is unset, default to including numeric non summary fields
        
        if (shouldInclude == null && (field.type == "integer" || field.type == "float")) {
            shouldInclude = true;
        }
        
        // if 'includeInRecordSummaryFields' is explicitly set, respect it as well
        if (shouldInclude && field.includeInRecordSummaryFields != null) {
            if ((isc.isA.String(field.includeInRecordSummaryFields) && 
                     field.includeInRecordSummaryFields != summaryField.name) ||
                (isc.isAn.Array(field.includeInRecordSummaryFields) &&
                     !field.includeInRecordSummaryFields.contains(summaryField.name)))
            {
                shouldInclude = false;
            }
        }
        if (shouldInclude) {
            fieldsToInclude.add(field);
        }
    }
    var summaryFunction = summaryField.recordSummaryFunction || "sum";
    var value = isc.DataSource.applyRecordSummaryFunction(summaryFunction, record, 
                                            fieldsToInclude, summaryField);
    return value;
},

// Summary fields are tricky: If we have a summary field which is also showing
// a group or grid level summary, it needs the summary value available on each
// record passed in so it can do calculations like "sum"
// Recalculate these summaries and hang them on the objects now

_$summary:"summary",
shouldApplyRecordSummaryToRecord : function (field) {
    return field && (field.type == this._$summary) && 
            (field.summaryFunction != null || field.getGroupSummary != null);
},

calculateRecordSummaries : function (records, suppressDisplay) {
    if (!this.fields) return;
    if (records == null) {
        records = this.getOriginalData();
    }
    if (records == null || (isc.isA.ResultSet(records) && !records.lengthIsKnown())) 
        return;
    
    var valuesChanged = false;
    
    var summaryFields = [];
    for (var i = 0; i < this.fields.length; i++) {
        var field = this.getField(i);
        if (this.shouldApplyRecordSummaryToRecord(field)) {
            summaryFields.add(field)
        }
    }
    if (summaryFields.length > 0) {
        for (var i = 0; i < records.getLength(); i++) {
            var record = records.get(i),
                editVals = this.getEditValues(record),
                editedRecord = editVals ? isc.addProperties({},record,editVals) : record;
            if (record == null) continue;
              
              
              
            for (var ii = 0; ii < summaryFields.length; ii++) {
                
                var field = summaryFields[ii];
                
                // Verify that we actually want to show a record summary value in
                // this cell. 
                // This method will return false for the case where this is a
                // summary row and we want to (for example) sum the calculated record summary
                // shown in this column for all rows in the grid, rather than trying to 
                // perform record-summary arithmetic on the summary row itself.
                if (!this.shouldShowRecordSummary(field,record)) continue;
                
                var oldValue = record[field.name];
              
                var summaryVal = this.getRecordSummary(editedRecord,field);
                record[field.name] = summaryVal;
                
                if (!this.fieldValuesAreEqual(field, oldValue, summaryVal)) {
                    valuesChanged = true;
                }
            }
        }
    }
    if (valuesChanged && !suppressDisplay) {
        if (!this.isDirty()) this.markForRedraw();
        if (this.showGridSummary && this.summaryRow) this.summaryRow._recalculateSummaries();
    }
},
            

// Standard field level summary functions.
// These allow a developer to specify a field of type "summary" with summaryFunction "sum" rather 
// than re-implementing the sum method on their field.
// It's basically the same list as the SimpleType registered summary functions but it
// takes different parameters - A single record and multiple fields rather than vice versa


//> @attr listGrid.recordSummaryBaseStyle (CSSStyleName : "recordSummaryCell" : IRWA)
// If showing any record summary fields (IE: fields of +link{listGridFieldType,type:"summary"}),
// this attribute specifies a custom base style to apply to cells in the summary field
// @visibility external
//<
recordSummaryBaseStyle:"recordSummaryCell",

//> @attr listGrid.summaryRow (AutoChild : null : RA)
// Automatically generated ListGrid for displaying grid summary information (see
// +link{listGrid.showGridSummary}).
// <P>
// This component is an +link{type:AutoChild} and as such may be customized via 
// <code>listGrid.summaryRowProperties</code> and <code>listGrid.summaryRowDefaults</code>
// @visibility external
//<
summaryRowConstructor:"ListGrid",

summaryRowDefaults:{
    // disable rollOver styling
    showRollOver:false

},

// clearSummaryRow() - clears (but doesn't destroy()) the summaryRow component.
// Developers will call 'setShowGridSummary' rather than calling this method directly
clearSummaryRow : function () {
    if (this.summaryRow && this.summaryRow.masterElement == this) {
        this.summaryRow.clear();
        this.removePeer(this.summaryRow);
        this._unRegisterAttachedPeer(this.summaryRow, isc.Canvas.BOTTOM);
        
        if (this.isDrawn()) {
            this.clear();
            this.draw();
        }
    }
        
},

//> @attr listGrid.summaryRowHeight (integer : 20 : IR)
// Default height for the +link{listGrid.summaryRow,summary row autoChild}. Note that this
// height is a minumum - the summary row has +link{listGrid.autoFitData} set to "vertical" so
// if multiple rows are visible in the grid summary, the summaryRow component
// will expand to accomodate them.
//
// @visibility external
//<
summaryRowHeight:20,

//> @attr listGrid.summaryRowStyle (CSSStyleName : "gridSummaryCell" : IRWA)
// +link{listGrid.baseStyle} for the +link{listGrid.summaryRow}
// @visibility external
//<
summaryRowStyle:"gridSummaryCell",

// showSummaryRow() - creates and shows the summaryRow autoChild
// not called directly -- call 'setShowGridSummary' instead
showSummaryRow : function () {
    if (this.summaryRow && this.summaryRow.masterElement == this) return;

    if (!this.summaryRow) {
        
        var dataSource = this.getSummaryRowDataSource();
        
        var initialFields;
        if (this.completeFields) initialFields = this.completeFields.duplicate();
        else if (this.fields) initialFields = this.fields.duplicate();
        // We respond to setFields(), setFieldWidths(), showField(), hideField() etc explicitly
        // in those methods so don't worry if initialFields is null - it'll get set when required

        var height = this.summaryRowHeight;              
        this.summaryRow = this.createAutoChild("summaryRow", {
            
            warnOnReusedFields:false,
            autoDraw:false,
            
            // When registered as attached peer, the master (the grid) will react to our
            // "resized" event due to overflow and refresh margins so everything is
            // correctly positioned and sized.
            // However initial draw doesn't fire resized even if visibleHeight exceeds specified
            // height (and this is the first time we're finding out)
            draw : function () {
                this.Super("draw", arguments);
                this.creator.refreshMargin();
            },
            
            width:"100%",
            snapTo:"B",
            height:height,
            
            autoFitData:"vertical",
            // autoFitMaxRows etc can be specified in summaryRowProperties if desired.
            
            // setting bodyOverflow to hidden.
            // If we don't do this and field widths exceed grid size, we end up with
            // hscrollbars on both the listgrid and the summary row.
            
            bodyOverflow:"hidden",
            
            showHeader:false,
            
            getBaseStyle:function() {return this.creator.summaryRowStyle},
            alternateRecordStyles:false,
            
            disabled:this.disabled,
            
            // Make fetchValueMapData into a no-op. We copy our fields from
            // our creator so any optionDataSources will match up meaning we can rely on
            // the creator's fetchValueMapData / 'setValueMap()' to update our valueMap
            // rather than having to do a second fetch against optionDataSources
            _fetchValueMapData : function () {
            },
            
            // The default summary functions all return number or string values, even if field.type
            // is boolean. If this method returns true we'll always display a checkbox rather than
            // the expected summary value.
            
            _formatBooleanFieldAsImages : function (field) {
                return false;
            },

            // support for databinding of summary row
            dataSource:dataSource,
            // pick up data from the gridSummaryData method
            data:dataSource == null ? this.getGridSummaryData() : null,

            _recalculateSummaries : function () {
                var grid = this.creator;
                // if we are getting data from a dataSource, simply re-fetch
                if (this.dataSource != null) {
                    var criteria = grid.getSummaryRowCriteria(),
                        forceRefresh = isc.ResultSet && isc.isA.ResultSet(this.data) &&
                                       !this.data.willFetchData(criteria);
                    this.fetchData(
                        criteria, null,
                        grid.getSummaryRowFetchRequestConfig()
                    );
                    if (forceRefresh) this.invalidateCache();
                } else {
                    this.setData(grid.getGridSummaryData(true));
                }
            },
            
            fieldSourceGrid:this,
            fields:initialFields,
            
            // We size fields according to the main grid field-widths.
            
            skipAutoFitWidths:true,
            getFieldWidths : function () {
                return this.creator.getFieldWidths();
            },
            
            _formatCellValue : function (value,record,field,rowNum,colNum) {
                var grid = this.creator;
                
                if (grid.shouldShowGridSummary(field)) {
                    if (value == null) return this.invalidSummaryValue;
                    if (field.formatGridSummary) {
                    
                        if (!isc.isA.Function(field.formatGridSummary)) {
                            field.formatGridSummary =
                                isc.Func.expressionToFunction("value", field.formatGridSummary);
                        }
                        if (isc.isA.Function(field.formatGridSummary)) {
                             return field.formatGridSummary(value);
                        }
                    
                    // if no explict formatGridSummary function exists, suffix counts with
                    // the plural title for the field.
                    } else {
                        var summaryFunction = this.creator.getGridSummaryFunction(field);
                        if (!isc.isAn.Array(summaryFunction)) {
                            summaryFunction = [summaryFunction];
                        }
                        if (summaryFunction[rowNum] == "count") {
                            var title = field.pluralTitle;
                            if (title == null) title = field.title;
                            if (title != null) value += " " + title;
                            return value;
                        }
                    }
                    
                    return this.Super("_formatCellValue", arguments);
                }
                
                
                return this.Super("_formatCellValue", arguments);
            }
        });
        this.addPeer(this.summaryRow);
    } else {
        this.summaryRow.setDataSource(this.getSummaryRowDataSource());
        this.summaryRow.setFields(this.completeFields.duplicate());
        // recalculateSummaries handles moving and resizing the summary row
        // (It has to, since the number of summary rows displayed may change, changing the
        // height of the summary row grid).
        this.summaryRow._recalculateSummaries();

        this.addPeer(this.summaryRow);
        
        
        this.summaryRow.setBodyFieldWidths(this._fieldWidths.duplicate());
        this.syncSummaryRowScrolling(this.body.getScrollLeft(), this.body.getScrollTop);
        
    }
    this.summaryRow.moveAbove(this);
    // Unlike most attachedPeers (edges etc), summary row can resize, which would
    // require a margin recalculation on the main grid. Pass in the "observeResized" param
    // to ensure this is taken care of.
    this._registerAttachedPeer(this.summaryRow, isc.Canvas.BOTTOM, null, true);
    this.refreshMargin();
},

// --------------------------------------------------------------------------------------------
// Filter editor row
// --------------------------------------------------------------------------------------------

// When 'showFilterEditor' is true on a listGrid, it will be drawn with a RecordEditor used to
// filter the List's data.

//> @method listGrid.setShowFilterEditor()
// Setter for the +link{ListGrid.showFilterEditor} property. Allows the filter editor to be
// shown or hidden at runtime.
// @param value (boolean) true if the filter editor should be shown, false if it should be hidden
// @group filterEditor
// @visibility external
//<
setShowFilterEditor : function (value) {
    // No op is required to avoid potentially setting up duplicate FEs etc.
    if (this.showFilterEditor == value) return;
    this.showFilterEditor = value;
    
    if (value) {
        // only call makeFilterEditor if we're drawn - otherwise this will happen lazily
        // in createChildren()
        if (this.isDrawn()) this.makeFilterEditor()
            
    // filterEditor may have never been defined if this widget has not yet been drawn
    } else if (this.filterEditor) {
        this.filterEditor.destroy();
        this.filterEditor = null;
    }
    // necessary because registerAttachedPeer currently only affects margins as drawn by
    // getTagStart()
    if (this.isDrawn()) {
        this.clear(); this.draw();
    }
    //this.layoutChildren();
},

// makeFilterEditor()
// Create a RecordEditor instance as this.filterEditor.
makeFilterEditor : function () {
    this.filterEditor = isc.RecordEditor.create({
        autoDraw:false,
        warnOnReusedFields:false,
        
        // Disable fetchValueMapData on the filterEditor entirely.
        // If we're showing an edit item for the field it'll have an optionDataSource
        // set on it, meaning the selectItem / comboBox etc will be responsible for issuing
        // any fetch request against the ODS if necessary.
        _fetchValueMapData:function () {
        },

        top:this.getTop() + this.getTopMargin(),
        left:this.getLeft() + this.getLeftMargin(),

        height : this.filterEditorHeight,
        // NOTE: filterEditors handle width sizing themselves
        
        disabled:this.disabled,
        
        sourceWidget:this,
        ID:this.getID() + "filterEditor",
      
        // we share field objects with the main grid.
        // This 'fieldSourceGrid' property is used to ensure we do the right thing about
        // generated fields like the selection-checkbox field etc.
        fieldSourceGrid:this,
      
        actionType:"filter",

        actionButtonPrompt: this.filterButtonPrompt,
        actionButtonProperties:this.filterButtonProperties,
        
        // XXX the register/unregisterAttachedPeer is not something we want to go with
        // long-term, but if we did, this should get factored up to Canvas.
        destroy : function () {
            this.sourceWidget._unRegisterAttachedPeer(this, isc.Canvas.TOP);
            this.Super("destroy", arguments);
        },
        
      
        fetchDelay: this.fetchDelay,

        allowFilterExpressions: this.allowFilterExpressions,
        expressionDataSource: this.getDataSource()

    }, this.filterEditorDefaults, this.filterEditorProperties);
    
	
    this.addPeer(this.filterEditor);
    this.filterEditor.moveAbove(this);
    
    this._registerAttachedPeer(this.filterEditor, isc.Canvas.TOP);

},



//>	@method listGrid.getFilterEditorValueMap()  ([A])
//
//  If we're showing the filter (query-by-example) row for this ListGrid, this method is
//  used to determine the valueMap to display in the filter row for this field.
//  Default implementation will return the field.filterEditorValueMap if specified, or
//  field.valueMap.
//
//  @group  filterEditor
//
//  @param  field   (ListGridField)    field definition field for which we need a valueMap
//  @return         (ValueMap)  ValueMap for the edit field (or null if no valueMap required)
//  @visibility external 
//<
getFilterEditorValueMap : function (field) {
    return  field.filterEditorValueMap || field.valueMap;
},

//>	@method listGrid.getFilterEditorType()  ([A])
//
//  If we're showing the filter (query-by-example) row for this ListGrid, this method is
//  used to determine the type of form item to display in the filter edit row for this field.
//  Default implementation will return the field.filterEditorType if specified, or
//  field.valueMap.
//
//  @group  filterEditor
//
//  @param  field   (ListGridField)    field definition field for which we need a valueMap
//  @return         (ValueMap)  ValueMap for the edit field (or null if no valueMap required)
//  @visibility external 
//<
getFilterEditorType : function (field) {

    // Simple case: support explicit filterEditorType on the field
    if (field.filterEditorType != null) return field.filterEditorType;

    // TODO: reimplement this once RecordEditor correctly returns AdvancedCriteria
    if (isc.SimpleType.inheritsFrom(field.type, "date") && this.getDataSource() && 
        this.getDataSource().supportsAdvancedCriteria()) 
    {
        return "MiniDateRangeItem";
    }

    // filter editor config is basically picked up from field defaults and explicit
    // field.filterEditorProperties.
    // If a a field specifies an explicit filterEditorType or a filterEditorProperties block with
    // an explicit editor type, respect it.
    // Otherwise if a field specifies an explicit editorType, respect that
    // Otherwise generate the editor type based on data type in the normal way
    // A couple of exceptions:
    // - override canEdit with canFilter, so we don't get a staticTextItem in the field
    // - clear out field.length: we don't want to show the long editor type (text area) in our
    //   filter editor
    var filterEditorConfig = isc.addProperties ({}, field,
                                                 {canEdit:field.canFilter !== false,
                                                  length:null});
    
    // the _constructor property can come from XML -> JS conversion, and matches the 
    // XML tag name for the field element.
    // Don't attempt to use this to determine DynamicForm editor type - it's likely to be
    // ListGridField or similar which shouldn't effect the generated form item type.
    if (filterEditorConfig._constructor != null) delete filterEditorConfig._constructor;
    if (field.filterEditorType != null) filterEditorConfig.editorType = field.filterEditorType;
    isc.addProperties(filterEditorConfig, field.filterEditorProperties);
    var type = isc.DynamicForm.getEditorType(filterEditorConfig, this);
    return type;
    
},

// getFilterEditorProperties - returns a block of properties to apply to the form item displayed
// in the filter row for some field.
getFilterEditorProperties : function (field) {
    var result = field.filterEditorProperties || {};
    if (field.filterOperator) result.operator = field.filterOperator;
    return result;
},

// Unexposed but publically accessible method to get a pointer to the filter editor grid

getFilterEditor : function () {
    return this.filterEditor;
},

//> @method listGrid.setFilterEditorCriteria()
// If +link{listGrid.showFilterEditor} is true, this method will update the criteria shown
// in the <code>filterEditor</code> without performing a filter.
// @param criteria (Criteria or AdvancedCriteria) New criteria to show
// @visibility external
//<
setFilterEditorCriteria : function (criteria) {
    if (this.filterEditor) {
        this.setFilterValues(criteria);
    }
},

//> @method listGrid.getFilterEditorCriteria()
// If +link{listGrid.showFilterEditor} is true, this method will return the criteria currently
// displayed in the <code>filterEditor</code>. Note that these values may differ from the
// criteria returned by +link{listGrid.getCriteria()} if the filter editor values have been modified
// without performing an actual filter.
// @param [omitHiddenFields] (boolean) By default this method will include criteria applied to
//   fields, including criteria that are not actually visible/editable in the filterEditor for the
//   grid. Pass in this parameter to get only values for visible fields returned.
// @return (Criteria or AdvancedCriteria) criteria currently displayed in the filterEditor
// @visibility external
//<
// Note: we rely on the filterEditor edit form to handle combining specified criteria with criteria
// from live items - as such calling getValuesAsCriteria() on that form will return all specified
// criteria, even for fields not visible in the grid.
// This is not always desired - it can be useful to apply additional criteria outside those
// editable by the user, and in order to repeatedly do this, we need to be able to get back a
// clean copy of *only* the criteria the user has modified. The includeHiddenFields parameter
// allows us to do this.
// - See the DynamicReporting sample for an example of this use-case:
//      isomorphic/system/reference/SmartClient_Explorer.html#dynamicReporting

getFilterEditorCriteria : function (omitHiddenFields) {
    if (this.filterEditor) {
        
        if (!omitHiddenFields || !this.filterEditor.getEditForm()) {
            return this.filterEditor.getValuesAsCriteria(this.autoFetchTextMatchStyle);
            
        } else {
            var items = this.filterEditor.getEditForm().getItems(),
                simple = true,
                criteria = {},
                advancedCriteria = {operator:"and", criteria:[]};
            for (var i = 0; i < items.length; i++) {
                if (items[i].hasAdvancedCriteria()) {
                    var crit = items[i].getCriterion();
                    if (crit != null) {
                        simple = false;
                        advancedCriteria.criteria.add(crit);
                    }
                } else {
                    var value = items[i].getValue();
                    if (value != null) {
                        criteria[items[i].getFieldName()] = value;
                    }
                }
            }
            if (!simple) {
                criteria = isc.DataSource.combineCriteria(criteria, advancedCriteria);
            }
            return criteria;
        }
    }
},

// Default setCriteria implementation will no-op if we have no data (and won't update the
// filterEditor even if we do have data) - override to ensure the FE reflects the new criteria
setCriteria : function (criteria) {
    if (this.filterEditor != null) {
        this.setFilterValues(criteria);
    }
    return this.Super("setCriteria", arguments);
},


// setFilterValues() - helper method called when this widgets filter criteria change.
// Will store the criteria locally and call the method to update the filter editor values.
setFilterValues : function (criteria) {
    
    // store this in a local var - this allows us to show and hide the filterEditor independently
    // and know what the current criteria are.
    this._filterValues = isc.addProperties({}, criteria);
    
    this.updateFilterEditor();
},


// update the filter editor's values to match the current filter criteria
// Needs to happen whenever the filter criteria change or the filter editor is shown
updateFilterEditor : function () {
    var editor = this.filterEditor;
    if (!editor) return;

    var values = this._getFilterEditorValues();
    this.filterEditor.setValuesAsCriteria(values, true);
},


// _getFilterEditorValues returns the values to be shown in our filter editor (doesn't get the
// values FROM our filter editor - see getFilterEditorCriteria() for that).
// Called when the filter editor is first created, and used by updateFilterEditorValues to update
// the filterEditor when it is already showing.
// Note that developers can customize the display value via the 'updateFilterEditorValues()'
// stringMethod
_getFilterEditorValues : function () {
    var currentCriteria = isc.addProperties({}, this._filterValues);
    
    // Allow for a completely custom display of filterEditorValues by the developer
    if (this.updateFilterEditorValues != null) {
        
        currentCriteria = this.updateFilterEditorValues(currentCriteria, this.autoFetchTextMatchStyle);
    }
    
    
    
    // If we've never performed a filter, use our default filter values. Note that if we 
    // explicitly filter with null or empty criteria we do NOT want to reset to defaults - 
    // defaults are only used if we've never filtered this datasource. Handled by the fact
    // that setFilterValues() will always store an object, never null.
    if (currentCriteria == null) {
        currentCriteria = {};
        for (var i = 0; i < this.completeFields.length; i++) {
            currentCriteria[this.completeFields[i].name] = this.completeFields[i].defaultFilterValue;
        }
    }
   
    return currentCriteria;
},


// re-set the filterEditor's values to display the defaultFilterValues.
// This will be called if the defaultFilterValues change (can happen from setFields()) [and
// no filter has yet been performed], or if we change datasources so the current filter gets
// invalidated. 

clearFilterValues : function () {
    this._filterValues = null;
    this.updateFilterEditor();
},

handleFilterEditorSubmit : function (criteria, context) {
    // notification method fired when the user modifies the criteria in the filter editor
    // and hits the filter button / enter key.
    if (this.filterEditorSubmit != null && this.filterEditorSubmit(criteria) == false) return;
    this.filterData(criteria, null, context);
},


// If we're doing an auto-fetch, apply each field's defaultFilterValue to the initialCriteria object
getInitialCriteria : function () {
    var initialCriteria = {},
        hasInitialFieldValue,
        fields = this.getFields(),
        undefined;
    for (var i = 0; i < fields.length; i++) {
        if (fields[i].defaultFilterValue !== undefined) {
            hasInitialFieldValue = true;
            var fieldName = this.getFieldName(fields[i]);
            initialCriteria[fieldName] = fields[i].defaultFilterValue;
        }
    }
    
    // allow explicitly specified initialCriteria to override the field level defaultFilterValues
    if (!hasInitialFieldValue) {
        initialCriteria = this.initialCriteria || this.getCriteria();
    } else {
        isc.addProperties(initialCriteria, this.initialCriteria || this.getCriteria());
    }
    
    return initialCriteria;
    
},


// Treat the filterEditor as a "special" peer -- keep it next to us in the page's z-order
_adjustSpecialPeers : function (newIndex) {
    if (this.filterEditor != null) this.filterEditor.setZIndex(newIndex-1);
    return this.Super("_adjustSpecialPeers", arguments);
},

//> @attr listGrid.allowFilterExpressions (boolean : null : IR)
// For use with +link{showFilterEditor}:true, allows simple search expressions to be entered
// into filter fields, as though +link{dynamicForm.allowExpressions} were true.
// <P>
// Can also be enabled or disabled on a field-by-field basis via
// +link{listGridField.filterEditorProperties}.
//
// @group advancedFilter
// @visibility external
//<

//> @attr listGrid.showFilterExpressionLegendMenuItem (boolean : null : IR)
// For use with +link{showFilterEditor}:true and +link{allowFilterExpressions}:true, adds a 
// menu item to the Filter context menu that shows the supported Expression table in a dialog.
//
// @group advancedFilter
// @visibility external
//<



// --------------------------------------------------------------------------------------------
// Inline Editing
// --------------------------------------------------------------------------------------------



// Editing kickoff
// --------------------------------------------------------------------------------------------

//>	@method listGrid.canEditCell() (A)
// Can this cell be edited?
// <P>
// The default implementation of <code>canEditCell()</code> respects the various property
// settings affecting editability: +link{listGridField.canEdit,field.canEdit} disables editing
// for a field, a record with the +link{listGrid.recordEditProperty,recordEditProperty} set to
// false is not editable, and disabled records are not editable.
// <P>
// You can override this method to control editability on a cell-by-cell basis.  For example,
// if you had a grid that allows editing of "orders", and you had a field "shipDate" that is
// normally editable, but should not be editable if the order is already "complete", you might
// implement <code>canEditCell()</code> as follows:
// <P>
// <pre>
//   isc.ListGrid.create({
//       ...
//       canEditCell : function (rowNum, colNum) {
//           var record = this.getRecord(rowNum),
//               fieldName = this.getFieldName(colNum);
//           if (fieldName == "shipDate" && 
//               record.orderStatus == "complete") 
//           {
//               return false;   
//           }
//           // use default rules for all other fields
//           return this.Super("canEditCell", arguments);
//       }
//   });
// </pre>
// <P>
// Notes on providing custom implementations:
// <ul>
// <li> In order to allow complete control over editing, <code>canEditCell()</code> is called
// very frequently.  If you see delays on row to row navigation, check that your implementation
// is efficient
// <li> If you change the editability of a cell on the fly, for example, during 
// +link{editorExit()} on another cell, call refreshCell() to show or hide the editor
// <li> If this ListGrid allows new records to be created, <code>canEditCell()</code> may be
// called when there is no record available, in which case getRecord() will return null.  The
// values input so far by the user are available via +link{getEditValues()}. 
// </ul>
//
// @param rowNum (number) row number for the cell
// @param colNum (number) column number of the cell
// @return (boolean) Whether to allow editing this cell
//
// @group editing
// @visibility external
//<
canEditCell : function (rowNum, colNum) {
    // just return if passed bad colNum
    if (colNum < 0 || colNum >= this.fields.length) return false;
    var cellRecord = this.getCellRecord(rowNum, colNum);
	// Note - we may have no cellRecord - this will occur if we're editing a new row
	if (cellRecord != null) {
        if (!this.recordIsEnabled(rowNum, colNum)) return false;

    	// Suppress editing if we're showing an embedded editor or detailGrid for this row
        if (this._openRecord == cellRecord) return false;
    }
    
    // If the field is explicitly marked as disabled, disallow editing
    
    var field = this.getField(colNum);
    if (field && field.disabled) return false;
    
    if (field && field.type == "summary") return false;

	// otherwise check the cascaded canEdit property
	if (
    	// Note: the isEditable() check is required as 'getCellBooleanProperty' will not return
    	// false if the record's "_canEdit" property is set to 'true', but canEdit is unset at
    	// the field and listGrid level, and we don't allow editing cells when canEdit is
    	// not set at both the listGrid and field level.
        !this.isEditable() ||    
        this.getCellBooleanProperty(
                "canEdit",
                rowNum,
                colNum,
                (cellRecord != null ? this.recordEditProperty : null)
        ) == false
    ) {
        return false; 
    }
    
    return true;
},

//> @method listGrid.isEditable()   (A)
//  Looks at the 'canEdit' property on both the listGrid and the set of fields to 
//  determine whether any cells in this listGrid can be edited.
//  @return (boolean)   true if any fields in this listGrid can be edited.
//<
_$true:"true",
isEditable : function () {
	// A field can be edited if:
	// listGrid.canEdit is true, and field.canEdit is not set to false
	// listGrid.canEdit is unset, and field.canEdit is set to true
	// No field can be edited if listGrid.canEdit is set to false.
    
    if (this.canEdit == false) return false;
    
    if (this.canEdit == true || this.canEdit == this._$true) {
        var fields = this.getFields() || [];
        for (var i = 0; i < fields.length; i++) {
            if (fields[i].canEdit != false) return true;
        }
        return false;
    } else {
    	// this.canEdit is null
        var fields = this.getFields() || [];
        for (var i = 0; i < fields.length; i++) {
            if (fields[i].canEdit == true) return true;
        }
        return false;
    }
},

//> @method listGrid.setCanEdit()
// Updates the canEdit property for this listGrid at runtime.
// <P>
// If setting canEdit to false, any current editing is cancelled by calling
// +link{cancelEditing}.
//
// @param (boolean) desired value of canEdit for this grid
//<
setCanEdit : function (canEdit) {
    if (canEdit == false) {
        if (this.getEditRow() != null) this.cancelEditing(isc.ListGrid.PROGRAMMATIC);
        this.canEdit = false;
    } else {
        this.canEdit = canEdit;
    }
},

//> @method listGrid.setFieldCanEdit()
//  Updates the canEdit property for some field at runtime.
//  @param  (boolean)   desired value of canEdit for this field
//<
setFieldCanEdit : function (field, canEdit) {

    if (isc.isA.String(field)) field = this.getField(field);
    if (field == null || !this.completeFields.contains(field) || field.canEdit == canEdit) return;
    
    field.canEdit = canEdit;
    if (this._editorShowing) {
        var editRow = this.getEditRow(),
            fieldName = field[this.fieldIdProperty],
            colNum = this.getColNum(field);
        
    	// If we're editing by cell and making the current edit field non editable, kill the
    	// current edit
        if (this.editByCell) {
            if (!canEdit && colNum == this.getEditCol()) {
                this.cancelEditing(isc.ListGrid.PROGRAMMATIC);
            }

    	// If we're showing editors for the entire row, we need to refresh the appropriate 
    	// field in the edit row to display / hide the editor
        } else if (colNum >= 0) {

        	// If we're hiding the current edit field, we want to shift focus to the nearest
        	// edit field instead.
        	// If there isn't one, just cancel the edit.
            if (!canEdit && colNum == this.getEditCol()) {

                var currentEditItem = this._editRowForm.getItem(fieldName),
                    fieldHasFocus = currentEditItem.hasFocus;
                    
            	// Try to put focus in an adjacent field - say the previous one
            	// (try going backwards first, then forwards)
                var newEditCell = this.findNextEditCell(editRow, colNum, -1, 
                                                        true, false, false, true);
        
                if (newEditCell == null || newEditCell[0] != editRow) 
                    newEditCell = this.findNextEditCell(editRow, colNum, 1, 
                                                        true, false, false, true);
            
            	// If there isn't another editable cell in this row, just cancel the edit
                if (newEditCell == null || newEditCell[0] != editRow) {
                    this.cancelEditing(isc.ListGrid.PROGRAMMATIC);
                    return;
                } 
            	// startEditing the new cell. This will fire the editorExit handler on the
            	// previous cell and save out the value if appropriate.
            	// Note: don't focus in the new cell unless focus was already in the cell being
            	// made un-editable.
                this.startEditing(newEditCell[0], newEditCell[1], !fieldHasFocus);
            }
        	// Refresh the cell to actually display / hide the edit form item.
            this.refreshCell(this.getEditRow(), colNum);
        }
    }
},



//>	@method listGrid.handleEditCellEvent()	(A)
// Handle an 'editCell' event - typically a click or double click on an editable ListGrid.
// Verifies that the cell passed in is a valid edit candidate before falling through to
// startEditing().
//
//	@group	editing
//
//	@param	rowNum      (number)	Row number of the cell to edit.
//	@param	colNum      (number)	Column number of the cell to edit.
//  @param  event       (editCompletionEvent)   How was this edit event triggered. If 
//                                      we shift focus to a new edit cell this event will be
//                                      passed to the editorExit handler(s) of the previous row.
//                                      Expected vals are "click", "doubleClick" or "focus"
//                                      (or null).
//  @param [newValue] (any) optional new edit value for the cell
//	@return	(boolean)	true if we are editing the cell, false if not editing for some reason
//<
handleEditCellEvent : function (rowNum, colNum, event, newValue) {
    // set a flag to notify us that we started this 'edit session' on keyPress
    // This is required for 'moveEditorOnArrow' behavior, which only applies to edit sessions
    // started on keypress
    // We can't use the _editorCursorAtEnd flag as that won't get set unless the value was 
    // modified, and we want the moveEditorOnArrow behavior even if the user started editing
    // from an f2 keypress...
    // This flag will be cleared 
    // - here if the user double clicks (etc) another cell
    // - by cellEditEnd unless the event is keyboard navigation to another cell
    if (event == this._$keyPress) this._editSessionFromKeyPress = true;
    else delete this._editSessionFromKeyPress

	// if they're trying to edit an invalid cell, return false
	if (rowNum < 0 || colNum < 0) return false;
	// can we actually edit that cell?  If not, bail.
    if (this.editByCell) {
         if (!this.canEditCell(rowNum, colNum)) return false;
    } else {
        // If the user double-clicks on a non-editable field and editByCell is 
        // false assume they want to start editing the row and put focus into the closest 
        // editable cell in that row.
        var editCell = this.findNextEditCell(rowNum, colNum, -1, true, true, false, true);
        if (editCell == null || editCell[0] != rowNum) 
            editCell = this.findNextEditCell(rowNum, colNum, 1, true, false, false, true);
            
        if (editCell == null || editCell[0] != rowNum) return false;
        // Update the colNum to reflect the closest editable cell to the one clicked if the
        // row is editable.
        colNum = editCell[1];
    }
    var undef;
    if (newValue !== undef) {
        this.setEditValue(rowNum,colNum, newValue);
    }
    
    // startEditing will save out the value in the previous edit cell if necessary...    
    return this.startEditing(rowNum, colNum, null, event);
},

// Show/Hide Inline Editor
// --------------------------------------------------------------------------------------------

//>	@method listGrid.startEditing()	(A)
// Start inline editing at the provided coordinates.
// <p>
// Invoked when a cell is editable and the <code>editEvent</code> occurs on that cell.  Can
// also be invoked explicitly.
// <P>
// If this method is called while editing is already in progress, the value from the current
// editCell will either be stored locally as a temporary edit value, or saved via 'saveEdits()'
// depending on <code>this.saveByCell</code>, and the position of the new edit cell.<br>
// Will update the UI to show the editor for the new cell, and put focus in it unless 
// explicitly suppressed by the optional <code>suppressFocus</code> parameter.
//
//	@group	editing
//
//	@param	[rowNum]      (number)	Row number of the cell to edit.  Defaults to first
//                                  editable row
//	@param	[colNum]      (number)	Column number of the cell to edit.  Defaults to first
//                                  editable column
//  @param  [suppressFocus] (boolean)   If passed this parameter suppresses the default 
//                                  behavior of focusing in the edit form item when 
//                                  the editor is shown.
//	@return	(boolean)	true if we are editing the cell, false if not editing for some reason
//
// @see canEditCell()
// @see editEvent
// @visibility external
//<

startEditing : function (rowNum, colNum, suppressFocus, eCe, suppressWarning) {


    // force editing on if it's not configured for any field, but a programmatic call is made
    if (!this.canEdit && !(this.completeFields || this.fields).getProperty("canEdit").or()) {
        this.canEdit = true;
    }

    // if setFields() has never been called, call it now.
    if (this.completeFields == null) this.setFields(this.fields);

	// Possibilities:
	// - This is an entirely new editing flow
	// - We are in an uncompleted editing flow
	// - We are currently showing an editor for another cell
    
	// What needs to happen?
	// - Showing a previous editor:
	//   *Save the new value from the edit cell locally into the temp set of editValues (for
	//     the appropriate record)
	//   *Hide the inline editor (unless it's another field in the same row and we're editing
	//     the whole row)
	// - Update edit info for this edit:
	//   *If we have no editFlowID, set up a new one
	//   *If there are no editValues stored for this record, store those values
	//   *Update the current editRowNum and editColNum, so we know which field is being edited

	// - show the editor:
	//   *Call 'showInlineEditor' to show the editor.  This will draw a new editor if required,
	//       or just focus in the appropriate field of the existing editor.

    // default to editable row / col
    var noRow = (rowNum == null),
        noCol = (colNum == null);
    if (noRow || noCol) {
        var testRow = (noRow ? 0 : rowNum),
            testCol = (noCol ? 0 : colNum);
        var newCell = this.findNextEditCell(testRow, testCol, 1, noCol, true);
        if (newCell == null) {
            this.logInfo("startEditing() passed bad cell coordinates:" + [rowNum, colNum],
                "gridEdit");
        } else {
            this.logInfo("startEditing() using derived coordinates:"+ newCell, "gridEdit");
            rowNum = newCell[0];
            colNum = newCell[1];
        }
    }
    // Legal coordinates are 
    // - any rowNum within the dataSet
    // - any rowNum for which we already have editValues (even if the record has not been saved)
    // - one rowNum past our last row (== this.getTotalRows()). In this case we'll be creating 
    //   a new row on the end of the list.
    // Bail if the coordinates are not legal.
	if (rowNum == null || rowNum < 0 || rowNum > this.getTotalRows()) {
		//>DEBUG
		// SuppressWarning param passed in when alwaysShowEditors is true and we blindly call
		// 'startEditing()' with no coordinates.
		// In this case we expect to fail to start the edit if there's no data or no visible, 
		// editable cells.
		if (!suppressWarning) {
		    this.logWarn("startEditing() passed bad cell coordinates:" + [rowNum, colNum] + 
                        ", can't edit" + this.getStackTrace(), "gridEdit");
		}
		//<DEBUG
		return false;
	}
    
    if (!this.canEditCell(rowNum, colNum)) {
		//>DEBUG
		this.logInfo("startEditing(): cell " + [rowNum, colNum] + 
                     " is non editable. Returning.", "gridEdit");
		//<DEBUG
		return false;
    }
    
    // at this point we have a valid cell to start editing.
    // If we're showing an editor, allow 'changeEditCell()' to handle saving out the
    // previous cell value, etc. - otherwise just call _startEditing() to start the edit 
    // process
    if (this._editorShowing) {
        this._changeEditCell((eCe || isc.ListGrid.PROGRAMMATIC), 
                             this.getEditRow(), this.getEditCol(), rowNum, colNum);
    } else {
    
    	// Punt it over to _startEditing to handle the actual editing
        this._startEditing(rowNum, colNum, suppressFocus);
    }

    // return true to indicate editing has begun
    
    return true;
},

// _changeEditCell()
// Internal method used by both 'startEditing' and 'cellEditEnd' to complete editing one cell
// and start editing another.
// This method will fire user event / change handlers, and save out the edit values if 
// appropriate.
// Falls through to _startEditing() to handle updating the display (hides this editor and shows
// the new one), and setting up the editValues for the new edit row.
_changeEditCell : function (editCompletionEvent, currentRowNum, 
                            currentColNum, newRowNum, newColNum) 
{

    // Note that the getEditValue() method will automatically pick up the current value
    // of the form item and store it in the edit-values
	// Note - this will fire the 'editorChange()' handler if the value has changed.
    var newValue = this.getEditValue(currentRowNum, currentColNum);

	// if we never fired 'editorEnter' avoid firing editor exit.
	
    var fieldName = this.getFieldName(currentColNum),
        editForm = this._editRowForm,
        editItem = editForm ? editForm.getItem(fieldName) : null,
        shouldFireEditorExit = editItem ? !editItem._cellEnterOnFocus : true
    ;
    if (editItem) {
        // If the edit item never got focused, the 'enterOnFocus' tags will still be present
        // just clean these up since we'll reset them if necessary (if focus goes back to the row)
        delete editItem._cellEnterOnFocus;
        delete editItem._rowEnterOnFocus;
        
        
        if (this._shouldParkFocus(editItem, currentRowNum, newRowNum, newColNum)) {
            this._parkFocus(editItem, currentColNum);
        }
    }
    
	// determine whether we need to validate or save on this cell transition
    // Use _getEditValues() - we already updated the editValue if necessary
    var leavingRow = (newRowNum != currentRowNum),
        newValues = this._getEditValues(currentRowNum, currentColNum);
                
	// Fire any developer defined handlers to fire when the user attempts to exit the edit cell
	// Stop if 'editorExit' handlers returned false.
    if (shouldFireEditorExit) {
        var editKilled = !this._handleEditorExit(editCompletionEvent, 
                                                 currentRowNum, currentColNum, newValue);
        if (leavingRow && !editKilled) {
            editKilled = !this._handleRowEditorExit(editCompletionEvent, currentRowNum, newValues);
        }
        
        if (editKilled) {
        	// If the editorExit handlers didn't already cancel this edit, or start a new one, force
        	// focus back into the current edit field.
        	// This is required as focus may not be in this form field
        	
            if (editCompletionEvent == isc.ListGrid.EDIT_FIELD_CHANGE) {
                var newFieldName = this.getFieldName(newColNum);
                if (editItem && editForm.getItem(newFieldName).hasFocus) {
                    editItem.focusInItem();
                }
            }
            return false;
        }
    }
    
    var saveNow = (this.autoSaveEdits &&
                    ((leavingRow && 
                     this.shouldSaveOnRowExit(currentRowNum, currentColNum, editCompletionEvent)) || 
                     this.shouldSaveOnCellExit(currentRowNum, currentColNum, editCompletionEvent)));
            
    // The 'neverValidate' property effectively disables validation for form items. 
    // Otherwise: 
    // - If we're saving, we avoid validating here, since saving will auto-validate for us.                        
    // - otherwise perform cell or row validation based on this.validateByCell / this.autoValidate
    
    if (!saveNow && this._validationEnabled()) {
        var validationFailed,
            validateRow = leavingRow &&
                          this.shouldValidateByRow(currentRowNum, currentColNum, editCompletionEvent);
        // Note that if we're working with cellRecords we call validateCell() directly
        // if we should validate the row
        if (validateRow && !this.usingCellRecords) {                       
            validationFailed = !this.validateRow(currentRowNum);            
        // If we're not validating the entire row, determine whether we should validate the
        // cell individually
        } else {                        
            if (validateRow || 
                this.shouldValidateByCell(currentRowNum, currentColNum, editCompletionEvent)) 
            {    
                validationFailed = !this.validateCell(currentRowNum, currentColNum);
            }
        }

        // Suppress navigation if validation failed. No need to show error to user - already
        // handled by the validation methods.
        if (this.stopOnErrors && validationFailed) return false;
    }  
    
    // At this point the old editor is still showing, but values have been updated, and 
    // handlers fired.  Fall through to _startEditing to handle hiding this editor and showing
    // the new one.
	
    if (saveNow) {    
        return this._saveAndStartEditing(newRowNum, newColNum, editCompletionEvent);
    } else {
    	// proceed immediately to next cell  
        this._startEditing(newRowNum, newColNum);
    }
    
    // If the user changed the grouped value of an edit row, regroup.
    // Exception - if we're saving, locally, we will have already saved at this point, which
    // already handles regrouping.
    if (leavingRow && this.isGrouped && (!saveNow || !this.shouldSaveLocally())) {
        this._updateGroupForEditValueChange(currentRowNum);
    }
},

// If the user edits a grouped tree, and changes a value within the groupBy field we
// may need to regroup.
// Determine this from rowNum by looking at the current editValues / record values for the
// groupBy field[s] and the current position in the group tree.
//
// We call this when the user is done editing a row, so from 
// - user moving from one edit row to another - _changeEditCell()
// - editor being dismissed - saveAndHideEditor()
//   [NB: that method doesn't actually save if autoSaveEdits is false]
// We do not call this if either of these actions tripped a local save since that
// occurs synchronously and handles regrouping automatically.


_updateGroupForEditValueChange : function (rowNum) {
    if (!this.isGrouped || !isc.isA.Tree(this.data)) return;
    
    var node = this.data.get(rowNum),
        groupNode = node,
        shouldRegroup = false;
    // passed a bad rowNum
    if (node == null) return;
    
    var groupFields = this.getGroupByFields();
    
    if (!isc.isAn.Array(groupFields)) groupFields = [groupFields];
    var curVals = this.getEditedRecord(rowNum);
    
    for (var i = groupFields.length-1; i >=0; i--) {
         var fieldName = groupFields[i],
            groupNode = this.data.getParent(groupNode);
    
        if (groupNode == null || groupNode.groupName != fieldName) {
            this.logWarn("error updating group for edit value change - unexpected group " +
                "tree structure. Regrouping.");
            shouldRegroup = true;
            break;
        }
        
        var value = curVals[fieldName];
        if (groupNode.groupValue != value) {
            shouldRegroup = true;
            break;
        }
    }
    
    if (shouldRegroup) {
        // This could probably be an incremental regroup since we know we're just updating
        // one row
        this.regroup();
        this._remapEditRows();
        this.markForRedraw();
    }
    return shouldRegroup;
},


_shouldParkFocus : function (editItem, currentRowNum, newRowNum, newColNum) {
    if (!isc.Browser.isIE) return false;

    // If current focus is in a non-text-based item we don't need to park focus - selection
    // will update as appropriate.
    var textBasedFocusItem = (isc.isA.PopUpTextAreaItem(editItem) && 
                              // in a difficult-to-reproduce case, editItem._popupForm can end up
                              // undefined when hiding columns while editing.  This check
                              // prevents a JS error, but may still cause focus to skip
                              editItem._popupForm && editItem._popUpForm.hasFocus) ||
                             (editItem.hasFocus && isc.FormItem._textBasedItem(editItem));
    //if (!textBasedFocusItem) return false;
    
    // If we will not be removing the current item from the DOM, we don't need to park focus.
    
    if (!this.editByCell && newRowNum == currentRowNum) return false;
    
    // If the new item is text-based - no need to park focus
    var newEditorType = 
        this.getEditorType(this.getField(newColNum), this.getCellRecord(newRowNum, newColNum));
    return !(newEditorType == null || isc.FormItem._textBasedItem(newEditorType, true));
    
},

// Internal method to unconditionally start editing a rowNum / colNum.
// This method will 
//  - hide the current editor (if appropriate)
//  - set up edit values for the new cell
//  - show the editor for the new cell (and focus unless suppressFocus param passed)
// o Does not check for validity of rowNum / colNum
// o Does not save / modify pending edit values for some other cell/row
_startEditing : function (rowNum, colNum, suppressFocus) {
	
    if (rowNum == "delayed") {
    	// we're firing from a timer for a delayed edit
        
    	// another call to startEditing happened while we were waiting for the timer to fire,
    	// and its params took precedence
        var params = this._delayedStartEditing;
        if (params == null) return;

        rowNum = params[0];
        colNum = params[1];
        suppressFocus = params[2];

    	
        
    } else if (this.isDrawn() &&
                (!this.body.readyToRedraw() || 
                 (this.frozenBody && !this.frozenBody.readyToRedraw())))
    {
    	// set a timer if we have not already set one, otherwise, just update the parameters
    	// for the delayed edit
        if (!this._delayedStartEditing) {
            this.delayCall("_startEditing", ['delayed'],0);
        }
        this._delayedStartEditing = [rowNum, colNum, suppressFocus];
    	
        return;
    }

	// we're starting editing now, we don't need the delayed edit params (the current
	// startEditing invocation overrides them)
    delete this._delayedStartEditing;
    
    var changingRow = this.getEditRow() != rowNum;
        
    // On a call to startEditing the current editRow / col, just put focus into the
    // field editor in question
    if (!changingRow && !suppressFocus && (this.getEditCol() == colNum)) {
        this.getEditForm().focusInItem(this.getEditorName(rowNum, colNum));
        return;
    }

	//>DEBUG
    this.logInfo("Starting editing at row " + rowNum + ", colNum " + colNum, "gridEdit");
	//<DEBUG
 
    // if we currently have an embedded editor showing, dismiss it   
    if (this._openRecord != null) this.closeRecord();
    if (this._editorShowing) {
    	// if we're changing rows, or only editing one cell at a time, hide the current editor,
    	// as it has no overlap with the old editor
        if (this.editByCell || changingRow) {
        
        	// hide the editor (but don't focus back in the body), and don't hide the CM as
        	// we're about to show another editor.
        	
            this.hideInlineEditor(false, true);
        }
    }
    
    
    
	// We're in a new edit flow if
	// - this._editingFlowID is null
	// - saveByCell is false
	// - we're editing a cell in a different row.
	// Otherwise this method is just showing the editor for a different cell
	
    var newEditFlow = (this._editingFlowID == null)  ||
                      changingRow ||
                      this.saveByCell;
    
    if (newEditFlow) {
        this._editingFlowID = this._getNextEditFlowID();
    }

    //this.logWarn("about to start editing, editValues for row: " + rowNum + " are now: " +
    //             this.echo(this.getEditValues(rowNum)));
    
    // Set up initial empty edit vals
	        
    var displayNewValues = this._editorShowing && (this.getEditRow() == rowNum)
    this.initializeEditValues(rowNum, colNum, displayNewValues);
    
    // At this point - we are definitely going to start editing the new row, and the editValues
    // have been set up.
    // If 'selectOnEdit' is true, select the row.
    
    var record = this.getCellRecord(rowNum, colNum);
    if (this.selectOnEdit && record != null) this.selectRecordForEdit(record);
    
    
    // ModalEditing (and edit event 'click') - in this case we show a click mask so won't
    // update rollovers when the user moves over other rows in the grid.
    // If they then click another record we'll start editing there, but never have cleared the
    // current 'over' row.
    // This can lead to the over styling getting left around until editing is complete.
    // Resolve this by explicitly clearing the hilite here:
    if (this.modalEditing) this.clearLastHilite();

	// If this is a new record, and 'addNewBeforeEditing' is true, we want to create the new
	// edit record BEFORE we start editing it - we do this via the standard 'saveEdits()' 
	// method - this will save out the newly created editValues.
	// The save, which may be asynchronous, may modify the values of the saved record, setting
	// up default field values, etc.  We want to reflect these changes in the edit data for
	// the record.
	// Handle this by having a method _updateNewEditRowValues() update the edit values for the
	// row with the values taken from the record.
	// We continue to show the editor on the newly created row, either now, or if wait for save
	// is true, after the save occurs.
    if (record == null && this.addNewBeforeEditing) {
    	// Hang onto the editRow / col, so we know what cell we're currently editing
    	// This is usually done in 'showInlineEditor' but we need this for the saveEdits call
    	// in this case
        this._editRowNum = rowNum;        
        this._editColNum = colNum;
        var waitForSave = this.shouldWaitForSave();
    
        
        var callback = "this._updateNewEditRowValues(" + waitForSave + "," + suppressFocus + ")" ;
        this.saveEdits(isc.ListGrid.PROGRAMMATIC, callback);
        if (waitForSave) return;
        else {
        	// The 'saveEdits()' call above may have changed the edit row's position.
            rowNum = this._editRowNum;
            colNum = this._editColNum;
        }
    }
	// - show the editor:
	//      - if the editor is showing for the same row, this will just focus
	//      - if another row, or editByCell, this will update the edit form, and show it in 
	//        the right place
    // Note that 3rd param is always true - we'll be calling showInlineEditor to show the
    // same cell in this method.
    this.showInlineEditor(rowNum, colNum, true, changingRow, suppressFocus);
    
    return true;

},

// Select the record about to be edited
selectRecordForEdit : function (record) {
    // perf: avoid updating the row we're about to draw editors into anyway
    
    if (!this.editByCell) record._ignoreStyleUpdates = true; 
    
    if (this.canSelectCells) {
          var cell = this.getRecordCellIndex(record);
          this.selection.selectSingleCell(cell[0],cell[1]);
    } else if (this.selection != null && (!this.selection.isSelected(record) || this.selection.multipleSelected())) {
        // we want a selection similar to select on mouse down: If we're using simple selection
        // select the record in addition to whatever else is selected - otherwise do a 
        // single selection
        if (this.selectionType == isc.Selection.NONE) {
            // shouldn't see this as we disable selectOnEdit when selectionType is none
            this.logInfo("selectOnEdit is true, but this.selectionType is set to 'none'." +
                " Unable to perform a selection on edit.", "gridEdit");
            
        } else if (this.selectionType == isc.Selection.SIMPLE) this.selection.select(record);
        else this.selection.selectSingle(record);
    }
    delete record._ignoreStyleUpdates;

},


// For text-based editors, update selection on focus
// The behavior we want is:
// - if the user started editing via editOnKeyPress (they started typing in the cell), put focus
//   at the end of the value so they don't wipe out what they already typed
// - otherwise respect this.autoSelectEditors
// We achieve this via a temporary flag set when we start editing a cell via 'editOnKeyPress'
_updateEditorSelection : function (item) {
    // applies only to text items (and subclasses)
    if (!isc.isA.TextItem(item) && !isc.isA.TextAreaItem(item) &&
        !(isc.isA.DateItem(item) && item.useTextField)) return;
    var element = isc.isA.DateItem(item) ? item.dateTextField.getDataElement() : item.getDataElement();
    if (!element) return;
    if (this._editorCursorAtEnd || !this.autoSelectEditors) {
        var val = item.getDataElement().value || "";
        item.setSelectionRange(val.length, val.length);
        // Assertion: we only want this special behavior of putting the cursor at the end after
        // the first 'focus()' on the item the user typed in, so clear out the flag here
        delete this._editorCursorAtEnd;
    } else {
        element.select();
    }
},

// Helper method to update the edit row values for the special case where we have saved out a
// new record before editing it due to 'addNewBeforeEditing'

_updateNewEditRowValues : function (showEditor, suppressFocus) {

    var rowNum = this._editRowNum,
        colNum = this._editColNum,
        record = this.getCellRecord(rowNum, colNum);

    if (record != null && record != "loading") {
    	
        var values = this.getEditValues(rowNum, colNum);
        for (var i in record) {
        	
            if (record[i] != null && values[i] == null) {
                values[i] = record[i];
            }
        }
    }

    if (showEditor) this.showInlineEditor(rowNum, colNum, true, true, suppressFocus);
    else this.updateEditRow(rowNum);
	// Note - no need to explicitly focus in the editor here - this will be handled by 
	// showInlineEditor if we're showing a new editor - and if we're already showing the
	// editor we don't need to modify it's focus
},

//>	@method listGrid.editField() (A)
//
//  Start editing a specific field.  This will save the current edit if appropriate.
//  
//	@group	editing
//
//	@param	fieldName   (string)    Field to start editing
//	@param	[rowNum]  (number)	Optional row to start editing - if null defaults to the current
//                              edit row.
//	@return	(boolean)	true if we are editing the cell, false if not editing for some reason
//
// @see canEditCell()
// @see editEvent
//<
editField : function (fieldName, rowNum) {
    // If this grid has not yet been drawn, this.fields may not have been set up yet
    if (this.completeFields == null) this.setFields(this.fields);

    var colNum;
    if (isc.isA.Number(fieldName)) colNum = fieldName;
    else colNum = this.fields.findIndex(this.fieldIdProperty, fieldName);
    if (rowNum == null) {
        rowNum = this.getEditRow();
        // It's legal to pass in no rowNum param, but if we're not already editing a row,
        // bail with a warning.
        if (rowNum == null) {
            this.logWarn("editField(): unable to determine which row to edit - returning.", 
                        "gridEdit");
            return;
        }
    }
    // Fall through to startEditing() to handle performing the edit.
    return this.startEditing(rowNum, colNum);
},

// create or update the editForm to reflect the editable and visible fields around the given
// coordinate, and show editors.
// Focuses in the target cell unless suppressed.
// internal: assumes we are logically set up to edit this row (editValues created)
// Application developers would call 'startEditing()' instead of this method

// this method is also responsible for firing editorEnter handlers when focus goes to the
// new edit cell - handled by setting up a flag to be checked by the form item's focus()
// handler.
showInlineEditor : function (rowNum, colNum, newCell, newRow, suppressFocus) {
    // This method is called in the following circumstances:
    // - we need to re-set focus to the current edit cell (example: setFields, redraw of body, etc)
    // - we need to move focus to a new cell that is already showing (called from startEditing())
    // - The editor is currently hidden and we need to show it (and put focus into the
    //   appropriate cell).
    // It is not expected to be called when the editor is currently showing for another
    // row (or for editByCell grids, another cell) - this method should not have to handle
    // hiding the edit form just showing it (if necessary) and assiging focus.
    //
	// NOTE: if we're doing full-row editing and the editForm is already showing, we assume the
	// current edit field's value has already been updated / saved by 'startEditing' call
	// whenever appropriate.
    if (this._editorShowing) {
    	// Catch the case where we're showing the edit form for another row
        if (rowNum != this.getEditRow() || (this.editByCell && colNum != this.getEditCol())) {
            this.logWarn("Unexpected call to 'showInlineEditor' during another edit " +
                         "- cancelling previous edit", "gridEdit");
            this.cancelEditing();
            this.startEditing(rowNum, colNum);
            return;
        }
    }
    // Set a flag to note that we're in the process of setting up the editor.
    // This will avoid infinite loops if showInlineEditor is called directly from
    // anything called by this method - EG: the redraw triggered by scrolling the
    // cell into view
    
    if (this._settingUpEditor && this._settingUpEditor[rowNum] == rowNum
        && this._settingUpEditor[colNum] == colNum) 
    {
        return;
    }
    this._settingUpEditor = [rowNum,colNum];
    
    var suppressScroll = this.suppressEditScrollIntoView || suppressFocus;

    this.logDebug("showing inline editor at: " + [rowNum, colNum] + 
                  ", will focus: " + !suppressFocus, "gridEdit");

	
    var scrollBeforeShowing = this.body && (!this.body.shouldShowAllColumns() || !this.body.showAllRows);
    if (scrollBeforeShowing && !suppressScroll) {
        // set scrollRedrawDelay to zero before scrolling into view. This ensures that the body will
        // be marked dirty instantly if a redraw is required, which in turn makes sure we set up the
        // correct set of form items
        var srd = this.body.scrollRedrawDelay;
        this.body.scrollRedrawDelay = 0;
        this.scrollCellIntoView(rowNum, colNum, false);
        this.body.scrollRedrawDelay = srd;
    }
    
    // If we're showing the rollOver canvases, clear them now
    if (this.rollOverCanvas && this.rollOverCanvas._currentRowNum == rowNum) {
        this.updateRollOverCanvas(this.rollOverCanvas._currentRowNum,
                                  this.rollOverCanvas._currentColNum, true);
    }
        
    var forceRedraw;
    if (!this._editorShowing) {
    	// create or update the editForm used to display editors for the fields.
        var rebuiltForm = this.makeEditForm(rowNum, colNum);
        if (this._alwaysShowEditors() && rebuiltForm) {
            // if the edit form items actually changed (rather than just having
            // new properties applied to them in place), we'll need a full redraw
            // since the inactiveHTML currently written into all other rows will
            // not be associated with the new set of live items 
            this.logInfo("Edit Form rebuilt with alwaysShowEditors:true, requires full redraw",
                         "inactiveEditorHTML");
            forceRedraw = true;
        }
    }    
    
	// Update the remembered editColNum
    this._editRowNum = rowNum;    
    this._editColNum = colNum;
    
    // Also update the stored "lastRecordClicked".
    // This is used for keyboard navigation when the editor isn't showing.
    // Setting this will ensure that we don't mysteriously jump to whatever was previously
    // recorded as the last record clicked once the editor is dismissed and the user hits
    // the up or down arrow key
    
    this._lastRecordClicked = rowNum;
 
    
	// write the editor form items into the DOM
    this._showEditForm(rowNum, colNum, forceRedraw);
    
    // Clear the settingUpEditor flag here
    
    this._settingUpEditor = null;

    
    // Ensure the edit cell is visible in the viewport
    
    if (this.body && !suppressScroll) {   
        var mustScroll;
        if (!scrollBeforeShowing) mustScroll = true;
        else {
            var body = this.body,
                rowTop = body.getRowTop(rowNum),
                rowHeight = body.getRowSize(rowNum),
                scrollTop = body.getScrollTop(),
                portHeight = body.getViewportHeight();                
            mustScroll = (rowTop < scrollTop) || (rowTop + rowHeight > scrollTop + portHeight);
        }
        if (mustScroll){
            this.scrollCellIntoView(rowNum, colNum, false);
        }
    }    

	// Don't show the click mask, or focus in the form item if we're not drawn.
    if (!this.isDrawn()) {
        return;
    }
    this._showEditClickMask();
    var focusItemName = this.getEditorName(rowNum, colNum),
        focusItem = this._editRowForm.getItem(focusItemName);

	// focusItem should be present since we've just scrolled it into view, but perform
	// check for safety anyway
    if (focusItem == null) {
        this.logWarn("ListGrid showing inline editor. Unable to get a pointer to the edit " +
                        "form item for field:"+ focusItemName);
    } else {
        
        // If this is a shift to a new cell, we'll want to fire editorEnter.
        // If a new row, we'll want to fire rowEditorEnter.
        // [Otherwise this method could be a refresh / refocus of current edit cell]
        
    
        
        if (newCell) focusItem._cellEnterOnFocus = true;
        if (newRow) focusItem._rowEnterOnFocus = true;
        // focus in the field being edited if appropriate
        if (!suppressFocus) {
            
            if (isc.Browser.isMoz) {
                
                var handle = this.body.getClipHandle(),
                    beforeFocus = handle.scrollTop;
            }
            
            
            var lastEvent = isc.EH.lastEvent;
            var delayFocus = (isc.Browser.isIE && lastEvent.eventType == isc.EH.MOUSE_DOWN && 
                                                  lastEvent.target != this._editRowForm);
    
            
            if (isc.Browser.isIE) {
                var focusParkForm = isc.ListGrid._focusParkForm;
                if (focusParkForm && 
                    (focusParkForm.hasFocus || focusParkForm.itemHasFocus())) delayFocus = true;
            }
            if (delayFocus) {            
                this._delayedFocusEvent = isc.Timer.setTimeout(
                    this._editRowForm.getID() + ".focusInItem('" + focusItemName + "');",
                    0
                );
            
            } else {
                
                if (isc.Browser.isMoz && this.body.overflow == isc.Canvas.VISIBLE) {
                    this.adjustOverflow();
                }
                this._editRowForm.focusInItem(focusItemName);
            }
        }
    }
},

// Write the editor form items into the cells in the ListGrid body, by redraw or refresh
// Re-evaluates canEditCell(), so may hide editors or show new editors.
// Internal: purely a helper to showInlineEditor
_showEditForm : function (rowNum, colNum, forceRedraw) {
    
    var editorWasShowing = this._editorShowing;
    this._editorShowing = true;
    if (!this.isDrawn() || !this.body) return;
    
	// The edit form has been created, and we need to show its items in the DOM.
	// 3 cases to catch:
	//  - Some edit form items are already showing for the edit row:
	//     If editByCell is false, it's possible that the 'canEditCell' criteria have changed
	//     for some currently visible edit cells.
	//     Need to refresh any cells where this is the case to show / hide form items.
	//  - No edit form items are showing
	//    - if the edit row is beyond the end of the list, we need to redraw the body to
	//      display the new edit row.
	//    - Otherwise for each cell where canEditCell is true, we must refreshCell() to write
	//      the form item into the DOM.

	// If we're showing the editor for a new edit row at the end of the list that's not currently
	// in the DOM redraw the body
    // Note: if this._editorShowing was true at the start of this method, we know we're
    // editing a new cell in the current edit row (_startEditing will hide the inline editor 
    // if we're editing a different row). In this case we only need to redraw the body if it's
    // already been marked as dirty.
    
	
    var newRow = this.isEmpty() ||
                 (!editorWasShowing &&   
                    (rowNum >= this.data.getLength()) && 
                    (this.showNewRecordRow || 
                     
                     (this.body.getTableElement(rowNum) == null)));
    
    var showInactiveEditors = this._alwaysShowEditors();
    
    // If we're showing embedded component(s) for the row force a redraw
    // This'll place them properly
    
    var record = this.getCellRecord(rowNum,colNum);
    if (record && record._embeddedComponents != null && record._embeddedComponents.length > 0) {
        forceRedraw = true;
    }
                  
    
    if (forceRedraw || newRow || this.body.isDirty() ||
        (this.frozenBody && this.frozenBody.isDirty()) ) {
        
        var body = this.bodyLayout ? this.bodyLayout : this.body;
        body.redraw("Showing editor");
        return;
    }

	// otherwise, editor is showing for the same row   
    if (this.editByCell) {
        // We need to refresh the entire row if
        // - frozenFields is non null -- this ensures the height of the row in both
        //   bodies matches
        // - we're using explicit tall vs normal base style -- ensure we switch to tall base
        //   style while the editor is showing
        var refreshRow = (this.frozenFields != null) || (this.baseStyle == null);
        
    	// editing by cell - just refresh the edit cell to show the editor.
        if (refreshRow) this.refreshRow(rowNum);
        else this.refreshCell(rowNum, colNum);
        
    } else {
    	// editing whole row - draw editors into the currently visible cells
        
        
        
        // use getDrawnFields() to retrieve the set of drawn fields (takes care of frozen fields and
        // incremental rendering)
        var fields = this.getDrawnFields();
    	// Check the 'canEdit' status of each cell in the edit row and refresh to show (or hide)
    	// editors as appropraite.
        
        for (var i = 0; i < fields.length; i++) {
            if (this.isCheckboxField(fields[i])) continue;
                
            var field = fields[i],
                fieldName = fields[i][this.fieldIdProperty],
                formItem = this._editRowForm.getItem(fieldName),
                colNum = formItem.colNum,
                
                formItemVisible = !!(editorWasShowing && formItem.isDrawn()),
                
                canEditCell = this.canEditCell(rowNum, colNum);
            
            if (formItemVisible != canEditCell) this.refreshCell(rowNum, colNum);
            // For cells we are not redrawing, we need to update the cellStyle if:
            // - This is the last over row - need to clear the current 'over' style since we 
            //   will not be showing  rollovers for the edit row.
            // - ensure the row shows up in the 'selected' state if selectOnEdit is true
            else if (this.selectOnEdit || this.lastOverRow) {
                var body = field.frozen ? this.frozenBody : this.body;
                body._updateCellStyle(this.getCellRecord(rowNum, colNum), rowNum, colNum);
            }
        }
    }
},

// show the clickmask to catch clicks outside the editors, which means we're done editing
// the row (and should save) if this.modalEditing is true.

_showEditClickMask : function () {

    //!DONTCOMBINE
	// only show the C.M. if this.modalEditing is true.
    if (!this.modalEditing) {
    	// Note - if a hover is showing, or pending on the body, clear it now.
    	
        if (this.canHover) this.stopHover();
        return;
    }
     
	// if the editRowForm CM is already up, no need to show it.
    if (!this._editRowForm.clickMaskUp()) {
        if (!this._editClickMaskFunction) 
            this._editClickMaskFunction = 
                new Function(this.getID() + "._handleClickOutsideEditor()");
        
        this._editRowForm.showClickMask(this._editClickMaskFunction, 
                                    // If this.stopOnErrors is true, always cancel the
                                    // 'mouseDown' that dismisses the clickMask.
                                    // [technically this should only be required if there 
                                    //  actually *are* errors, but since errors may come
                                    //  back from an asynch save it's appropriate to always
                                    //  cancel here].
                                    (this.stopOnErrors ? isc.EH.SOFT_CANCEL : isc.EH.SOFT), 
                                    // pass in the editForm to ensure that the form items
                                	// are not masked
                                    this._editRowForm);
    }
},

//> @method ListGrid.stopHover()
// Notification that the user is no longer hovering over some cell. Hides the current hover canvas
// if one is showing. 
// @visibility external
//<
// Implemented at the Canvas level. Gets called on mouseOut.

stopHover : function () {
    if (this._editorShowing && isc.EH.getTarget() == this.getEditForm()) return;
    return this.Super("stopHover", arguments);
    
},

// shouldWaitForSave()
// Should we block user interactions during save, or allow the user to keep editing.
// If this function returns true, we wait for a save to return successfully before moving to
// a new edit cell, or hiding the current editor.
// This depends on this.waitForSave, and this.stopOnErrors.
// Note that we can't really support stopOnErrors = true / waitForSave = false, as we will not
// know if we have validation errors from server logic until a save has completed (performing
// a server round trip), and stopOnErrors implies we want to leave the editor up (as well as 
// alerting the error messages).
shouldWaitForSave : function () {
    if (this.stopOnErrors && !this.waitForSave) {
        var message =
            "Note: ListGrid initialized with 'waitForSave' false, and 'stopOnErrors' true." +
            " In this case user input will be be blocked during save, to allow server side " +
            " errors to be determined before the editor is hidden. Setting 'waitForSave' to true.";

    	// Log this at the info level rather than the warning level unless waitForSave has 
    	// explicitly been set to 'false', as it is very likely to occur
        if (this.waitForSave == false) this.logWarn(message, "gridEdit");
        else this.logInfo(message, "gridEdit");
        
    	// actually update 'waitForSave' -- this will avoid us showing this log repeatedly
    	// when this method is run.
    	// Note - if we want to support modifying these values on the fly 
    	// ("setStopOnErrors" / "setWaitForSave") we will have to take this modification into
    	// account too.
        this.waitForSave = true;
    }
    
    
    return !!(this.waitForSave || this.stopOnErrors);
},

// Hide the editor(s) for the current edit row.  Internal

hideInlineEditor : function (focusInBody, suppressCMHide) {
    
    // focusInBody is intended to restore focus to the body after hiding the (focused) editor
    // Therefore check for whether the body or the edit form currently has focus and only
    // refocus if so
    focusInBody = focusInBody && (this.hasFocus || (this.body && this.body.hasFocus) || 
                                    (this.getEditForm() && this.getEditForm().hasFocus) ||
                                    (isc.ListGrid._focusParkForm &&
                                     isc.ListGrid._focusParkForm.hasFocus));    
                          
    // - clear out the flags marking the editor as being visible
	// - If we're showing the editor for a new temporary edit row, who's values have been 
	//   cleared, but we're still showing the row in the DOM, redraw the body to clear the row 
	//   from the DOM
	// - otherwise call 'refreshCell()' for each cell currently showing a form item to clear
	//   it from the DOM.
    
	// If we aren't currently editing a row, no-op
    if (!this._editorShowing) return false;
    
	
    this._editorShowing = null;
    
    // clear _editorSelection flag - ensures we don't inappropriately refocus in the edit form
    // after redraw if we subsequently re show the editor.    
    this._editorSelection = null;
    
    var editRow = this._editRowNum,
        editField = this._editColNum;
    this._editRowNum = this._editColNum = null;

	

	// ensure that the clickmask gets taken down after edit
	// We suppress this step when we're hiding an editor, then reshowing - for example on
	// ListGrid redraw / cell navigation.
	
    if (!suppressCMHide) this._editRowForm.hideClickMask();
    
    // At this point, if we have edit values for the row, but they match the underlying
    // data values, just drop them.
    
    if (this.getEditValues(editRow, editField) != null && 
        (!this._savingEdits || !this._savingEdits[this.getEditValuesID(editRow, editField)]) && 
        !this.recordHasChanges(editRow, editField, false))
    {
        this.logInfo("hideInlineEditor for row with no edits - dropping edit values", "gridEdit");
        // Don't hang onto the empty edit values for the row
        this._clearEditValues(editRow, editField);
    }

    if (!this.body) return true;

	// update the visible cells.  Now that editRow/Field is unset, they'll revert to normal
	// display.  NOTE: don't bother updating if we're already slated to redraw the body
	
    var editForm = this._editRowForm;

    if (editForm.hasFocus) {    
        editForm.blur();
    }
    if (isc.Browser.isIE) {
        var focusItem = editForm.getFocusSubItem(),
            unconfirmedBlur = isc.EH._unconfirmedBlur;
        if (editForm.hasFocus || 
            (unconfirmedBlur && 
                ((unconfirmedBlur == editForm) || (isc.EH._unconfirmedBlur.form == editForm)) ) ) 
        {
        	
            focusItem.elementBlur();

            // Another artefact of IE's asynchronous focus handling behavior is that
            // if 'blur()' is called on a text item, then the item is cleared from the DOM
            // before onblur fires, and 'focus()' is called on another item in the DOM, when the
            // user hits a key, focus will be pulled from the new focus item for no good reason.
            // We work around this here by putting focus in the 'focus park form'.
            
            this._parkFocus(focusItem, editField);        
        }
    }
    
    
    
    // If we're showing embedded component(s) for the row force a redraw
    // This'll place them properly
    
    var record = this.getCellRecord(editRow,editField),
        forceRedraw = false;
    if (record && record._embeddedComponents != null && record._embeddedComponents.length > 0) {
        forceRedraw = true;
    }
    
    if (!this.body.isDirty() && (!this.bodyLayout || !this.bodyLayout.isDirty()) &&
        !this.isDirty()) 
    {
        if (forceRedraw || editRow >= this.getTotalRows()) {
            var widget = this.bodyLayout || this.body;
            widget.markForRedraw("Editor Hidden");
        } else {            
            if (this.editByCell) {
                var refreshRow = (this.frozenFields != null) || (this.baseStyle == null);
                
                if (refreshRow) this.refreshRow(editRow);
                else this.refreshCell(editRow, editField);

            }
            else this.refreshRow(editRow);
            
            // force a refresh of the group-summary row if one is showing.
            
            this.refreshGroupSummary(editRow);
        }
    }
    
    if (focusInBody) {
    	// return focus to the body
    	// -- ensure we don't editOnFocus via the 'suppressEditOnFocus' flag
    	//    (will get cleared out by _focusChanged on the body)
        this._suppressEditOnFocus = true;
        
                
        this.body.focus();
    }
    return true;
},

_parkFocus : function (focusItem, editField) {
    
    if (isc.isA.TextItem(focusItem) || isc.isA.TextAreaItem(focusItem) || 
        isc.isA.PopUpTextAreaItem(focusItem)) 
    {
        var focusParkForm = isc.ListGrid._focusParkForm;
        
        // If the hidden text item we use for managing focus doesn't exist, create it
        // here.
        if (!focusParkForm) {
            focusParkForm = isc.ListGrid._focusParkForm = isc.DynamicForm.create({
                
                // Ensure that if this gets destroy'd we also clear up the pointer to it
                pointersToThis:[{object:isc.ListGrid, property:"_focusParkForm"}],
                getFocusParkItem: function () {
                    return this.getItem(0);
                },
                autoDraw:false,
                _redrawWithParent:false,
                ID:"_ListGrid_focusParkForm",
                _generated:true,
                selectOnFocus:true,
                tabIndex:-1,
                items:[
                    {name:"textItem", type:"text",
                        // Suppress all key event handling - this will avoid the user from 
                        // being able to tab out of this focus parking form (temporarily
                        // putting focus somewhere else on the page)
                        handleKeyPress:function(){return false;}
                    }
                ],
                width:1, height:1, overflow:isc.Canvas.HIDDEN,
                
                itemHasFocus : function () {
                    var item = this.getFocusParkItem();
                    if (item.hasFocus) return true;
                    if (isc.Browser.isIE && this.isDrawn() && this.isVisible() &&
                        (this.getActiveElement() == this.getFocusParkItem().getFocusElement()))
                            return true;
                    return false;
                },
                redraw : function () {
                    var forceRefocus = false;
                    if (this.itemHasFocus()) forceRefocus = true;
                    this.Super("redraw", arguments);
                    if (forceRefocus) this.focusInItem(this.getFocusParkItem());
                }
            })
        }
        // Ensure the hidden text form is drawn in our scope and float it under us.
        // As a child it will be cleared() / destroyed() if we are as well.
        // Avoids any unexpected native scrolling issues with focus change.
        if (focusParkForm.parentElement != this) {
            if (focusParkForm.isVisible()) focusParkForm.hide();
            this.addChild(focusParkForm);
        }
   
        //this.logWarn("moving focusPark form, editCol: " + editField +
        //             ", column left: " + this.getColumnLeft(editField) + 
        //             ", scrollLeft: " + this.body.getScrollLeft());
                    
        focusParkForm.moveTo(
            this.getColumnLeft(editField) - this.body.getScrollLeft(),
            Math.min(
                ((this.showHeader ? this.headerHeight : 0) + this.getRowTop(this.getEditRow()) 
                        - this.body.getScrollTop()),
                this.getScrollHeight()-1
            )
        );
    
        focusParkForm.moveBelow(this.body);
        // focusParkForm must be visible to receive focus
        focusParkForm.show();
        
        // If we're showing the edit clickMask, unmask the focusParkForm wrt that so we don't
        // dismiss the editor when it receives focus
        
        if (this._editRowForm.clickMaskUp()) focusParkForm.unmask(this._editRowForm.getID());
        
        // Put focus into the text item.
        focusParkForm.getFocusParkItem().focusInItem();
        
        // Ensure the editRowForm is aware it doesn't have focus any more 
        
        focusItem.form.hasFocus = false;
    }    
},


// Create inline editing form and items
// --------------------------------------------------------------------------------------------

// create a DynamicForm for inline editing.  This form manages FormItems embedded in GR cells,
// but does not actually draw them.
// It's up to the calling function to handle displaying these edit form items in the DOM.
updateEditorItemsInPlace:true,
makeEditForm : function (rowNum, colNum) {
    var record = this.getCellRecord(rowNum, colNum),
    	// get the values for the form
        
        values = this.getEditDisplayValues(rowNum, colNum),
        updateItemsInPlace
    ;
    // If we're already showing an edit form and the set of editable fields
    // hasn't changed we may be able to update the items in place (changing just
    // rowNum, colNum, record, valueMap, etc) rather than performing a full
    // setItems(...) with new items on the DynamicForm
    // If 'alwaysShowEditors' is true this is going to mean we can avoid a full
    // redraw, since the currently drawn inactiveEditorHTML for other records in 
    // the grid will already point to the existing form items and have appropriate
    // 'inactiveEditorContext' set allowing us to respond to events properly
    if (this.updateEditorItemsInPlace && this._editRowForm != null) {
        updateItemsInPlace = true;
        
        var editFields = this.editByCell ? [this.getField(colNum)] : this.getDrawnFields();
        if (editFields) {
            if (editFields.length != this._editRowForm.getItems().length) {
                updateItemsInPlace = false;
            } else {
                for (var i = 0; i < editFields.length; i++) {
                    var editField = editFields[i],
                        fieldName = this.getEditorName(rowNum, editField),
                        liveItem = this._editRowForm.getItem(fieldName);
                        
                    if (liveItem == null || 
                        (this.getEditorType(editField, values) != 
                         this._editRowForm.getEditorType(liveItem))) 
                    {
                        updateItemsInPlace = false;
                        break;
                    }
                }
            }
        }
        if (updateItemsInPlace && editFields) {
            var widths = this.getDrawnFieldWidths(record, editFields);
            
            for (var i = 0; i < editFields.length; i++) {
                var editField = editFields[i],
                    editColNum = this.getColNum(editField),
                    fieldName = this.getEditorName(rowNum, editField),
                    liveItem = this._editRowForm.getItem(fieldName),
                    record = this.getRecord(rowNum),
                    editedRecord = this.getEditedRecord(rowNum),
                    props = this.getEditItem(editField, record, editedRecord,
                                            rowNum, editColNum, widths[i], true);
                liveItem.setProperties(props);
                liveItem._size = null;
                if (editField.frozen) {
                    liveItem.containerWidget = this.frozenBody;
                } else {
                    liveItem.containerWidget = this.body;
                }
            }
            
        } else {
            // get currently visible items
            var items = this.getEditRowItems(record, rowNum, colNum, this.editByCell);
            // just update the items array and current values if the form already exists
            //this.logWarn("rebuilding editRowForm");// + this.getStackTrace());
            this._editRowForm.setItems(items);
        }
        this._editRowForm.setValues(values);
        
    } else {
        var items = this.getEditRowItems(record, rowNum, colNum, this.editByCell);
    	//this.logWarn("creating editRowForm..." + this.getStackTrace());
    	// create the editForm.  Done once only per grid lifetime
    	var properties = isc.addProperties({}, 
    	    this.editFormDefaults, {
            
        	// keep track of the listGrid
            grid:this,
            // for AutoTest apis
            locatorParent:this,
            
            showErrorIcons:this.showErrorIcons,
            
            // Give the form the same tabIndex - this is not relevant in most ListGrids,
            // since the clickMask / keypress handling will prevent tabbing in and out of the 
            // edit form, but we rely on this behavior in the RecordEditor class where the user
            // must be able to tab in and out of the edit row.
            tabIndex:this.getTabIndex(),
                
        	// pass it this widget's datasource too
            dataSource:this.dataSource,

            autoComplete:this.autoComplete,
            uniqueMatch:this.uniqueMatch,
        	// Avoid autoFocus - we explicitly focus and blur when appropriate.
            autoFocus:false,
    
            items:items,
            values:values
        }, this.editFormProperties);
        
        if (this.dateFormatter != null && properties.dateFormatter == null) {
            properties.dateFormatter = this.dateFormatter;
        }
        if (this.datetimeFormatter != null && properties.datetimeFormatter == null) {
            properties.datetimeFormatter = this.datetimeFormatter;
        }
        if (this.timeFormatter != null && properties.timeFormatter == null) {
            properties.timeFormatter = this.timeFormatter;
        }
        this._editRowForm = isc.DynamicForm.create(properties);
    }

    if (this.logIsDebugEnabled()) {
        this.logDebug("editRowForm created with values: " + 
                      this.echo(this._editRowForm.getValues()), "gridEdit");
    }
    
    // The return value will indicate whether we actually rebuilt the form
    // (as opposed to just modifying the existing items in place)
    // We use this to determine whether a full redraw is required in the 
    // alwaysShowEditors case
    return !updateItemsInPlace;
},

//> @method listGrid.getEditForm() [A]
// Method to retrieve the live edit form for an +link{listGrid.canEdit,editable} ListGrid.
// This is the automatically generated DynamicForm used to manage the per field edit items.
// <P>
// Note that this is an advanced method and developers should be aware of the following issues:
// <ul><li>The edit form is only present while the user is actually editing a record and this
// method will return null if no editor is currently showing.</li>
// <li>Live edit items may be retrieved by calling <code>getItem(<i>someFieldName</i>);</code>
// but items are only created for fields that are visible and being written out into the grid.
// This means with +link{showAllColumns} set to false an item may not be created until the
// user scrolls the column in question into view</li>
// <li>The editForm's values are managed by the ListGrid through the edit-values subsystem.
// If you want to change an edit value for a field, call +link{listGrid.setEditValue()} and the
// grid will handle updating the value in the live item if necessary. You should not need to call
// <code>setValue();</code> directly on the form or item and doing so will not always update the
// edit value for the grid. 
// </li></ul>
// In general - bear in mind that this is an advanced usage and if there is an equivalent API
// available on the ListGrid it is always preferable to use that.
// @return (DynamicForm) the live edit form, or null if the grid is not currently showing any editors.
// @visibility internal
//<
// currently unexposed - getEditFormItem() is much more likely to be needed.
getEditForm : function () {
    return this._editorShowing ? this._editRowForm : null;
},


// These helpers are required to allow rowNum / colNum based edit values management
// rather than fieldName based valuesManagement.


//> @method listGrid.getEditFormItem() [A]
// Method to retrieve a live edit form item for an +link{listGrid.canEdit,editable} ListGrid.
// This is the automatically generated editor displayed in a cell while editing the grid.
// <P>
// Note that this is an advanced method and developers should be aware of the following issues:
// <ul><li>Edit form items are only present while a user is actually editing a cell. This method
// will return null if the user is not editing the grid or the field in question is not editable
// or not visible. Note that due to +link{showAllColumns,incremental rendering} columns which are
// not currently scrolled into view may be unrendered, in which case they may have no associated
// edit item until the user scrolls them into view.</li>
// <li>The items' values are managed by the ListGrid through the edit-values subsystem.
// If you want to change an edit value for a field, call +link{listGrid.setEditValue()} and the
// grid will handle updating the value in the live item if necessary. You should not need to call
// <code>setValue();</code> directly on the item and doing so will not always update the
// edit value for the grid. 
// </li></ul>
// In general - bear in mind that this is an advanced usage and if there is an equivalent API
// available on the ListGrid it is always preferable to use that.
// @param field (String or Integer) fieldName or colNum to get the edit item for.
// @return (FormItem) the live edit item for the current edit row and specified field, or null if the grid is not currently showing any editors.
// @visibility external
//<
// NOTE: this differs from getEditItem() - it's retrieving an existent item in the edit form 
// for a cell -- not retrieving the properties to create a form item object
getEditFormItem : function (colNum) {
    var editForm = this.getEditForm();
    if (!editForm) return null;
    // getEditorName() already handles being pased a fieldName or colNum
    var fieldName = this.getEditorName(this.getEditRow(), colNum),
        item = editForm.getItem(fieldName);
    // sanity check - if item.colNum != the colNum passed in assume the editorName is reused for
    // multiple fields
    if (!isc.isA.Number(colNum) || (item && item.colNum == colNum)) return item;
},

getEditFormValue : function (colNum) {
    var item = this.getEditFormItem(colNum);
    return (item ? item.getValue() : null);
},

// helper: sets edit form items to latest editValues (including reverting to original record
// values if editValues have been cleared)
// Called from setEditValues()
_updateEditItemValues : function () {
    if (!this._editRowForm) return;

    var rowNum = this.getEditRow(), colNum = this.getEditCol(),
        values = this.getEditDisplayValues(rowNum, colNum);
    this._editRowForm.setValues(values);
},


// Helper method to get all the values for the edit form at once
getEditDisplayValues : function (rowNum, colNum) {
    
    var editValues = this.getEditValues(rowNum, colNum),
        record = this.getCellRecord(rowNum, colNum),
        values = {};
    
    for (var fieldName in record) {
        values[fieldName] = record[fieldName];
    }
    for (var fieldName in editValues) {
        values[fieldName] = editValues[fieldName];
    }
    return values;
},

// Method to get the horizontal space available for the form items for each column
// (Note this is not a 1:1 mapping to form items, as not every field is editable, and we
// incrementally create form items for columns as they are rendered out)
// Overridden by TreeGrid to account for indentation in tree field.
getEditFormItemFieldWidths : function (record) {
    var widths = [];
    for (var i =0; i<this.fields.length; i++) {
        var colNum = this.getLocalFieldNum(i),
            field = this.fields[i],
            body = field.frozen ? this.frozenBody : this.body;
        widths[i] = body.getInnerColumnWidth(colNum);
    }
    return widths;
},

//>	@method listGrid.getEditorValueMap()  ([A])
//
//  Returns the valueMap to display for a field when it is displayed in the editor while
//  editing some record.<br>
//  Called when a user starts to edit a field, or whenever the field valueMap is updated via
//  a call to +link{listGrid.setValueMap()} or +link{listGrid.setEditorValueMap()}.
//  Default implementation will return the <code>field.editorValueMap</code> if specified, otherwise
//  <code>field.valueMap</code> - can be overridden to provide a
//  different specific valueMap for some field based on the record/field data.
//
//  @group  editing
//
//  @param  field   (ListGridField)    field definition field for which we need a valueMap
//  @param  values  (object)    Field values for record being edited. Note that this will include
//                              the current edit values for fields that have not yet been saved.
//                              May be null, if editing a new record.
//  @return         (ValueMap)  ValueMap for the edit field (or null if no valueMap required)
//  @visibility external 
//<

//> @method listGridField.getEditorValueMap()
// Optional stringMethod to get a +link{listGridField.valueMap} for a specific field.
// If present this method will be called from +link{listGrid.getEditorValueMap()} and the resulting
// valueMap will be used instead of any static specified valueMap for the field.
// @param values (object) Field values for record being edited. Note that this will include
//                        the current edit values for fields that have not yet been saved.
//                        May be null, if editing a new record.
// @param field (ListGridField) pointer to the listGridField
// @param grid (ListGrid) pointer back to this ListGrid instance.
// @return (valueMap) ValueMap for the field (or null if no valueMap required)
// @visibility external
//<

getEditorValueMap : function (field, values) {
    if (field.getEditorValueMap != null) {
        
        isc.Func.replaceWithMethod(field, "getEditorValueMap", "values,field,grid");        
        return field.getEditorValueMap(values,field,this)
    }
    if (field.editorValueMap != null) {
        return field.editorValueMap;
    }
    if (field.editorProperties && field.editorProperties.valueMap) {
        return field.editorProperties.valueMap;
    }
    return field.valueMap;
},


//> @method listGrid.getEditorValueIcons()
// Returns the valueIcons for a field when it is displayed in the editor while editing some
// record. Default implementation will return +link{ListGridField.editorValueIcons} if specified
// otherwise +link{ListGridField.valueIcons}
// @param field (object) field definition
// @param values (object) current edit values for the record
// @return (object) valueIcons for the editor
// @visibility external
// @group imageColumns
//<
getEditorValueIcons : function (field, values) {
    return  field.editorValueIcons || field.valueIcons;
},

//> @method listGrid.getEditorValueIconWidth()
// Returns the width for any valueIcon for a field when it is displayed in the editor 
// while editing some record.<br>
// Returns +link{listGridField.editorValueIconWidth} if specified - otherwise 
// +link{listGridField.valueIconWidth} or +link{listGridField.valueIconSize}
// @param field (object) field definition
// @return (number) width for the value icon to show in the editor
// @visibility internal
// @group imageColumns
//<
// Don't return listGrid.valueiconWidth / size - this is just a default and likely to break
// form items with standard valueIconSizes, such as checkboxes.
getEditorValueIconWidth : function (field) {
    if (field.editorValueIconWidth != null) return field.editorValueIconWidth;
    return (field.valueIconWidth != null ? field.valueIconWidth : field.valueIconSize);
},

//> @method listGrid.getEditorValueIconHeight()
// Returns the height for any valueIcon for a field when it is displayed in the editor 
// while editing some record.<br>
// Returns +link{listGridField.editorValueIconHeight} if specified - otherwise 
// +link{ListGridField.valueIconHeight} or +link{listGridField.valueIconSize}
// @param field (object) field definition
// @return (number) height for the value icon to show in the editor
// @visibility internal
// @group imageColumns
//<
getEditorValueIconHeight : function (field) {
    if (field.editorValueIconHeight != null) return field.editorValueIconHeight;
    return field.valueIconHeight != null ? field.valueIconHeight : field.valueIconSize;
},

//> @method listGrid.setEditorValueMap() ([A])
//
// Set a valueMap to display for this field while editing.<br>
// This method sets the +link{ListGridField.editorValueMap, field.editorValueMap} property - 
// note that if  +link{ListGrid.getEditorValueMap()} has been overridden it may not make use 
// of this property.
// @group editing
// @param fieldID   (object | number | field name)  Field object or identifier
// @param   map     (object)    ValueMap to apply to the field
// @visibility external
//<
setEditorValueMap : function (fieldID, map) {
    var fieldNum = this.getColNum(fieldID),
        field = this.getField(fieldID),
        fieldName = field[this.fieldIdProperty];
    field.editorValueMap = map;
    if (this._editorShowing) {
        var rowNum = this.getEditRow(),
            editRecord = this.getEditedRecord(rowNum, fieldNum);
        // Apply the valueMap to the edit form field - note that we retrieve it via the 
        // getter method in case it has been overridden.
        this._editRowForm.setValueMap(fieldName, this.getEditorValueMap(field, editRecord));
    }
    
},

//>	@method listGrid.getEditorType()  ([A])
//
//  Returns the form item type (Class Name) to display for a field when it is displayed in the 
//  editor while editing some record.<br>
//  Default implementation will return field.editorType if specified.
//  If not specified, the default form item for the appropriate data type will be displayed
//  - can be overridden to provide a different specific form item type for some field based on 
//  the record/field data.
//
//  @group  editing
//
//  @param  field   (ListGridField)    field definition field for which we need a valueMap
//  @param  values  (object)    current edit values for the record (may be null, if editing a 
//                              new record)
//  @return         (string)  form item type for the edit field
//  @visibility external 
//<
getEditorType : function (field, values) {
    
    // determining type: editorProperties, being most specific, wins.  Otherwise
    // field.editorType, otherwise, you get the default editor picked
    // for field.type (which is the field's *data* type, not editor type).
    // NOTE: editorProps.type will always refer to the form item type, not the data type.
    // NOTE: "formItemType" is a legacy synonym of "editorType"
    var editorProperties = isc.addProperties({},field,field.editorProperties);
    
    // Use the static method on DynamicForm to get the editorType for this field.
    // Pass this ListGrid in as a parameter so the method can examine 
    // this.longTextEditorThreshold and this.longTextEditorType.
    return isc.DynamicForm.getEditorType(editorProperties, this);
},

// getEditorProperties() - return a block of default properties for editor form items displayed
// while editing some field
// Overridden in RecordEditor.

getEditorProperties : function (editField, editedRecord, rowNum) {
    return isc.addProperties({}, this.editorProperties, editField.editorProperties);
},

//>	@method listGrid.getEditRowItems()  (IA)
//
//      Given a record to edit, return an appropriate array of dynamicForm item init blocks
//
//  @group  editing
//
//  @param  record  (object)    Record to be edited
//  @param  rowNum  (number)    index of the row being edited
//  @param  fieldNum (number)   index of the field on which the 'startEditing' occurred (typically
//                              used as a focus field)
//  @return         (array)     Array of DynamicForm item object instantiation blocks
// @visibility internal
//<
// Note: if editByCell is false, and we're incrementally rendering, we only create form items 
// for the visible set of fields.

getEditRowItems : function (record, rowNum, fieldNum, singleCell) {
    var body = this.body;
	// if we haven't created the body yet, don't create any form items - they'll get set up
	// at draw, and updated at redraw.
    if (body == null) return [];

	// The set of fields for the form is basically this.fields, with some custom properties, such
	// as 'editorType'
    var firstEditable, lastEditable,
    	widths = [],
        items = [];
        
    var editedRecord = this.getEditedRecord(rowNum, fieldNum);

	// create an array of the fields we want to work with        
    var editFields;
    if (singleCell) editFields = [this.getField(fieldNum)]
    else editFields = this.getDrawnFields();
    
    widths = this.getDrawnFieldWidths(record, editFields);
    if (editFields != null) {
        for (var i = 0; i < editFields.length; i++) {
            
            // (Set up each item as a property-value map)
            
            
            var colNum = this.getColNum(editFields[i]);
            var item = this.getEditItem(editFields[i], record, editedRecord, rowNum, colNum, widths[i])
            if (item == null) continue;
            
            items[items.length] = item;
        }
    }    
    return items;

},

getDrawnFields : function () {
    
    if (!this.body) return null;
    
    var drawnFields = [], body = this.body;
    var ff = this.frozenFields, fLeft = this.freezeLeft();

    if (ff && fLeft) {
        // this assumes that we are not dealing with incremental column rendering of frozen
        // fields (should be true as not scrollable)
        drawnFields.addList(ff);
    }
    
    var firstVisible, lastVisible;
    
    if ((body._firstDrawnCol == null) || !body.isDrawn() || body.isDirty() || body._redrawing) {    
        var drawnRange = body.getDrawArea();
        firstVisible = drawnRange[2];
        lastVisible = drawnRange[3];
    } else {
        firstVisible = body._firstDrawnCol;
        lastVisible = body._lastDrawnCol;
    }
    if (ff && fLeft) {
        firstVisible += ff.length;
        lastVisible += ff.length;
    }    
    for (var i = firstVisible; i <= lastVisible; i++) {
        drawnFields.add(this.fields[i]);
    }
        
    if (ff && !fLeft) {
        drawnFields.addList(ff);
    }
    return drawnFields
},

getDrawnFieldWidths : function (record, fields) {
    // Allow the developer to pass in the fields to return widths for
    if (!fields) fields = this.getDrawnFields();
    
    if (!fields) return null;
    // apply the width from the fieldWidths array 
    var completeWidths = this.getEditFormItemFieldWidths(record),
        widths = [];
    for (var i = 0; i < fields.length; i++) {
        widths[i] = completeWidths[fields[i].masterIndex];
    }    
    return widths;
},

// getEditItem()
// returns an individual property block for edit-form form item.

// Helper methods to be applied to pop up text area items to specially process keydown and
// keypress events.  Fired in the scope of the pop up text area

_popUpTextAreaItemKeyPress : function (item, keyName, characterValue) {
    return this.grid.editorKeyPress(this, keyName, characterValue);
},
_popUpTextAreaItemKeyDown : function (item, keyName, characterValue) {
    return this.grid.editorKeyDown(this, keyName, characterValue);
},

// On focus in a pop up text area item, ensure that we have updated the current edit cell info.
// This handles the case where we got no elementFocus on the PUTA directly - happens if the
// PUTA has no focusable element itself.

_popUpTextAreaItemFocus : function () {
    var form = this.form, lg = this.grid,
        rowNum, colNum, fieldName, fieldChanged;
    if (lg._editorShowing) {
        rowNum = lg._editRowNum;
        // don't worry about editByCell case- for the pop up textareaitem to have got focus we
        // must have already shown it, meaning we must already know which field is being edited
        if (!lg.editByCell) {
            rowNum = lg._editRowNum;
            fieldName = this.getFieldName(),
            colNum = lg.fields.findIndex(lg.fieldIdProperty, fieldName);
                
            fieldChanged = (lg._editColNum != colNum);
            // If the user has clicked in another field in the edit form, fire editField on
            // the appropriate field
            if (fieldChanged) {
                // store the new edit cell
                lg.setNewEditCell(rowNum, colNum);
                // fire 'cellEditEnd' to save / validate before moving to the new cell
                lg.cellEditEnd(isc.ListGrid.EDIT_FIELD_CHANGE);
                    
            }
        }
    }
},

_getPopUpTextAreaTop : function () {
    var grid = this.grid,
        fieldName = this.getFieldName(),
        rowNum = grid.getEditRow(),
        style = grid.getCellStyle(grid.getRecord(rowNum), rowNum, grid.getColNum(fieldName)),
        rowTop = grid.getRowPageTop(rowNum) + 
                 isc.Element._getTopBorderSize(style) + isc.Element._getTopPadding(style);
    return rowTop;
},

_checkboxClick : function (a,b,c,d) {
    if (!this.hasFocus) {
        var lg = this.grid;
        lg.setNewEditCell(this.rowNum, this.colNum);
        lg.cellEditEnd(isc.ListGrid.EDIT_FIELD_CHANGE);
        // and force cellEnterOnFocus, so when the (delayed) focus 
        // handler fires, we get a cellEnter
        this._cellEnterOnFocus = true;
    }
    
    return this.invokeSuper("CheckboxItem", "handleClick", a,b,c,d);
},



// handleEditorChanged()
// Fired when the changed() handler fires for any of our edit items.
// Allows us to perform validation on change
handleEditorChanged : function (item) {

    var fieldName = item.getFieldName(),
        field = this.getField(fieldName),
        validateOnChange;
    
    if (field && field.validateOnChange != null) validateOnChange = field.validateOnChange;
    else validateOnChange = this.validateOnChange;
    
    
    var rowNum = this.getEditRow();
    if (rowNum == null || rowNum != item.rowNum) return;
    
    if (validateOnChange) {
        
        // Note: we don't always update the editValue for the cell when the change handler
        // fires on the edit item - instead we usually update when the user moves to a new cell.
        // However, validateCell(), like most other APIs that interact with the editValues,
        // will update the edit value for the cell if it's stale before performing validation.
        this.validateCell(rowNum, fieldName, false, true);
    
    // We support form items changing their value after they've lost focus - EG:
    // delayed "Tab" autoComplete for databound comboBoxItems.
    // In this case if we're validating or saving by cell we need to perform an update now
    // (as we won't get an editor exit event to react to).
    } else if (!item.hasFocus && !this.editByCell) {

        if (item.isDrawn() && item.isVisible()) {
            this.storeUpdatedEditorValue(false, item.colNum);
                
            if (this.saveByCell) {
                // Essentially we're saving for the same reason as if the user had
                // taken focus from the field.
                var editCompletionEvent = isc.ListGrid.EDIT_FIELD_CHANGE;
                this.saveEdits(editCompletionEvent, null, rowNum, item.colNum);
            } else if (this.validateByCell) this.validateCell(rowNum, fieldName);
            
        }
    }
    
},

// Override for the (internal) handleChanged method so we can be notified when the
// user changes an edit cell's form item value.
_editorHandleChangedOverride : function (a,b,c,d) {
    
    this.invokeSuper(this.getClassName(), "handleChanged", a,b,c,d);
    
    if (!this.destroyed) this.grid.handleEditorChanged(this);
    
},

_editorGetAutoComplete : function () {
    var grid = this.grid;
    // This would imply something like a destroyed form item - so just ignore it.
    if (!grid) return null;

    var field = grid.getField(this.getFieldName());
    if (field.autoComplete != null) return field.autoComplete;
    if (grid.autoComplete != null) return grid.autoComplete;
    return this.Super("_getAutoCompleteSetting", arguments);
},


_timeEditorTypes:{time:true, TimeItem:true},
_$time:"time",
_dateEditorTypes:{date:true, DateItem:true},
_datetimeEditorTypes:{datetime:true, dateTime:true, DatetimeItem:true, DateTimeItem:true},
_$date:"date",
_$datetime:"datetime",
_popUpTextAreaEditorTypes:{popUpTextArea:true, PopUpTextAreaItem:true},
_checkboxEditorTypes:{checkbox:true, CheckboxItem:true},
_$boolean:"boolean",
_$checkbox:"checkbox",
_$CycleItem:"CycleItem",
_selectEditorTypes:{select:true, SelectItem:true}, 


_commonEditorStringMethodsFromField:["change", "changed", "defaultDynamicValue"],
_commonEditorStringMethods:["change", "changed", "defaultDynamicValue",
                            "keyPress", "click", 
                            // Not sure if showIf / enableIf would be set on editors - more
                            // likely to set canEdit:false for the cell
                            "showIf", "enableIf"],
// getEditItem()
// returns a config block for an item to be displayed in the edit form.
// 'updateOnly' parameter implies we're going to take the returned properties and
// apply them to an existing form item, so no need to include standard handlers
// or properties that wouldn't be applied to a live item such as editorType

getEditItem : function (editField, record, editedRecord, rowNum, colNum, width, updateOnly) {


    var item = {};
    
    item.width = width;
	// Hang some properties onto the form item so keypress handers (etc.) written onto
	// the form item can readily access details about the edit:
    item.record = record;
    item.rowNum = rowNum;

    item.colNum = colNum;
    
    // the error icons will be written directly into the form item (so no need to
    // adjust the size of the edit item to account for them).
    
	// Set the "name" of the item
    var fieldName = this.getEditorName(rowNum, editField);
    item[this.fieldIdProperty] = fieldName;

    // copy the title (won't be visible, but needed for ARIA)
    if (editField.title != null) item.title = editField.title;

	// Use the accessor function to get the valueMap for the item
	// This allows override of visible options on a per cell basis
	// Note that we pass in the edit values, rather than the record's saved values - we want
	// the valueMap to update as edits are performed
    
    item.valueMap = this.getEditorValueMap(editField, editedRecord);
    
    // If the field has a specified optionDataSource, pass that through to the editor too,
    // along with the valueField / displayField properties
    
    
    if (editField.valueField != null) item.valueField = editField.valueField
    if (editField.displayField != null) item.displayField = editField.displayField
    if (editField.optionDataSource) item.optionDataSource = editField.optionDataSource
    if (editField.optionFilterContext) item.optionFilterContext = editField.optionFilterContext
    if (editField.optionCriteria) item.optionCriteria = editField.optionCriteria
    if (editField.optionOperationId != null) item.optionOperationId = editField.optionOperationId;
        
    // apply valueIcons and related properties to the editor
    // Note that we allow different value icons in the editor from the icons displayed in the
    // static cell
    item.valueIcons = this.getEditorValueIcons(editField, editedRecord);
    
    var valueIconWidth = this.getEditorValueIconWidth(editField),
        valueIconHeight = this.getEditorValueIconHeight(editField);
    if (valueIconWidth) item.valueIconWidth = valueIconWidth;
    if (valueIconHeight) item.valueIconHeight = valueIconHeight;
    
    item.imageURLPrefix = (editField.editorImageURLPrefix || editField.imageURLPrefix);
    item.imageURLSuffix = (editField.editorimageURLSuffix || editField.imageURLSuffix);
    // back compat only:
    item.baseURL = editField.baseURL;
    item.imgDir = editField.imgDir;  
    // Pick up icon-related properties from the edit field (if any set)
    // (most of these will be undef, so pick up standard defaults from item type)
    var undef;
    if (editField.icons !== undef) item.icons = editField.icons;
    if (editField.showPickerIcon !== undef) item.showPickerIcon = editField.showPickerIcon;
    if (editField.pickerIconSrc !== undef) item.pickerIconSrc = editField.pickerIconSrc;
    if (editField.pickerIconWidth !== undef) item.pickerIconWidth = editField.pickerIconWidth;
    if (editField.pickerIconHeight !== undef) item.pickerIconHeight = editField.pickerIconHeight;
    
    // Generic icon properties are unlikely to be set on a per-field basis, but handle them anyway
    if (editField.defaultIconSrc !== undef) item.defaultIconSrc = editField.defaultIconSrc;
    var iconHeight = (editField.editorIconHeight || editField.iconHeight);
    if (iconHeight !== undef) item.iconHeight = iconHeight
    var iconWidth = (editField.editorIconWidth || editField.iconWidth);
    if (iconWidth !== undef) item.iconWidth = iconWidth; 
    if (editField.iconPrompt !== undef) item.iconPrompt = editField.iconPrompt;
    if (editField.iconHSpace !== undef) item.iconHSpace = editField.iconHSpace;
    if (editField.iconVAlign !== undef) item.iconVAlign = editField.iconVAlign;
    
    // ValueIcons properties:
    // Setting showValueIconOnly before getting editor type ensures we get back a cycle item
    // if appropriate rather than a select.
    if (this.showValueIconOnly(editField)) {
        if (editField.editorProperties == null) editField.editorProperties = {};
        editField.editorProperties.showValueIconOnly = true;
    }
    
    // If pickListWidth / fields is defined on the field object, pass it through to the item:
    if (editField.pickListWidth != null) item.pickListWidth = editField.pickListWidth;
    if (editField.pickListFields != null) item.pickListFields = editField.pickListFields;
    

    // Set textAlign to match field alignment (required so text within text items etc reflects
    // horizontal alignment even though the item will be sized to take up all the space in the
    // cell).
    item.textAlign = (editField.cellAlign || editField.align);

    
    if (editField.editorProperties != null) {
        for (var i = 0; i < this._commonEditorStringMethods.length; i++) {

            var prop = this._commonEditorStringMethods[i],
                value = editField.editorProperties[prop];
            // convert both "action" objects and strings
            if (value != null && !isc.isA.Function(value)) {
                var stringMethodReg = isc.FormItem._stringMethodRegistry;
                editField.editorProperties[prop] = 
                    value = isc.Func.expressionToFunction(stringMethodReg[prop], value);        
            }
            if (value != null) editField.editorProperties[prop] = value;
        }  
    }

    // This block applies field.change, field.changed, and field.defaultDynamicValue
    for (var i = 0; i < this._commonEditorStringMethodsFromField.length; i++) {

        var prop = this._commonEditorStringMethodsFromField[i];
        if (editField.editorProperties && editField.editorProperties[prop] != null) {
            continue;
        }
        var value= editField[prop];
        if (value != null) {
            if (editField.editorProperties == null) editField.editorProperties = {};
            if (!isc.isA.Function(value)) {
                var stringMethodReg = isc.FormItem._stringMethodRegistry;
                
                value = isc.Func.expressionToFunction(stringMethodReg[prop], value);        
            }
            editField.editorProperties[prop] = value;
        }
        
    }

    item.valueIconLeftPadding = this.getValueIconLeftPadding(editField);
    item.valueIconRightPadding = this.getValueIconRightPadding(editField);
        
    // if we're updating an existing item in place we don't need to reapply standard handlers,
    // or any properties which can't be updated on the fly (like editorType)
    if (!updateOnly) {
        // Pick up autoCompletion settings from the grid in preference to any default settings
        // on the form item
        item._getAutoCompleteSetting = this._editorGetAutoComplete;
        item.autoCompleteCandidates = editField.autoCompleteCandidates;
        item.uniqueMatch = editField.uniqueMatch;
    
        // containerWidget should point to the ListGrid body (or frozen body if appropriate)
        // this ensures that 'formItem.isVisible()' tests the visibility of this widget, rather than
        // the DynamicForm managing the form's values.
        item.containerWidget = editField.frozen ? this.frozenBody : this.body;
        
        item.grid = this;
        
        // validateOnChange: validation of edits is performed by the grid, not the editForm.
        // Override the internal 'handleChanged()' method to notify us if the edit item value
        // is changed by the user. The grid then checks for validateOnChange, and if appropriate
        // performs validation of the cell.
        item.handleChanged = this._editorHandleChangedOverride;
        
        // Apply a keyDown handler to all items.
        // Allows us to kill native tab navigation in Safari
        item.keyDown = function (item, form, keyName) {
            return this.form.grid.editorKeyDown(item, keyName);
        }
        
        // Apply an inactiveEditorHTML handler for mouseDown so we can switch edit rows on mouseDown
        // on inactive editor HTML.
        // This handles switching edit rows when alwaysShowEditors is true and the user mouseDowns
        // on a row other than the current edit row.
        
        item.inactiveEditorMouseDown = function (inactiveContext, itemInfo) {
            if (inactiveContext && (inactiveContext.grid == this.form.grid.getID()) && 
                inactiveContext.rowNum != null && inactiveContext.colNum != null) 
            {
                this.form.grid.startEditing(inactiveContext.rowNum, inactiveContext.colNum);
            }
        } 

        item.type = editField.type;
        // pick a form item type appropriate for embedded editing
        
        // explicit specification
        item.editorType = this.getEditorType(editField, record);
        var eT = item.editorType; 
        
        // Pass specified "displayFormat", "dateFormatter" and "timeFormatter" through to the
        // edit item verbatim - FormItem knows how to handle these.
        if (editField.dateFormatter != null) item.dateFormatter = editField.dateFormatter;
        if (editField.timeFormatter != null) item.timeFormatter = editField.timeFormatter;
        if (editField.displayFormat != null) item.displayFormat = editField.displayFormat;
        
        var inputFormat = this._getDateInputFormat(editField);
        if (inputFormat) item.inputFormat = inputFormat;

        // for date items, use the text field rather than the 3 selects
        if (this._dateEditorTypes[eT] == true || this._datetimeEditorTypes[eT] == true ||
            (editField.type == this._$date && eT == null)) {
            item.editorType = (this._datetimeEditorTypes[eT] == true? this._$datetime : this._$date);
            item.useTextField = true;
            // This improves the appearance for this item type
            item.cellPadding = 0;
            // Don't apply a style to the sub items' cells - we don't want padding, etc.
            item.itemCellStyle = null;
            // have the picker icon butt up against the text box.
            item.pickerIconHSpace = 0;
            
        }
    
        //>PopUpTextAreaItem    
        // For pop-up textArea type editors, apply the keyPress handling code to the textArea<b></b>
        if (this._popUpTextAreaEditorTypes[eT] == true) {
            // PopUpTextAreaItems are a subclass of staticTextItems. Override the default textBoxStyle
            // to match the hack - suppress "over" styling when getting the cell style since we
            // always suppress it on the edit row once the editor is showing
            item.getTextBoxStyle = function () {
                var grid = this.grid,
                    record = grid.getCellRecord(this.rowNum,this.colNum);
                this.textBoxStyle = grid.getCellStyle(record, this.rowNum,this.colNum);
                return this.Super("getTextBoxStyle", arguments);
            }
    
            // notification when the grid's cell style changed so we can update our textBoxStyle        
            item.gridCellStyleChanged = function (record, rowNum, colNum, newStyle) {
                var textBox = this._getTextBoxElement();
                if (textBox) {
                    textBox.className = this.textBoxStyle = newStyle;
                }
            }
            
            // supppress doubled borders etc
            
            item.textBoxCellCSS = isc.Canvas._$noStyleDoublingCSS
    
            // Apply the custom keydown & keypress handlers to the pop up text area's textArea
            item.textAreaKeyDown = this._popUpTextAreaItemKeyDown;
            item.textAreaKeyPress = this._popUpTextAreaItemKeyPress;
            // Override getTextAreaTop() - rather than sticking to the top of the form item (which
            // may not line up with other items), stick to the top of the cell.
            item.getTextAreaTop = this._getPopUpTextAreaTop;        
            // default popUpOnEnter to true.
            if (item.popUpOnEnter == null) item.popUpOnEnter = true;
            
            // react to text area focus as we would to item focus for other items
            // so we know what cell the user is editing
            item.textAreaFocus = this._popUpTextAreaItemFocus;
        }
        //<PopUpTextAreaItem
        
        // Don't show label for checkboxes by default.
        if (this._checkboxEditorTypes[eT] == true || (editField.type == this._$boolean && eT == null)) {
            if (item.showLabel == null) item.showLabel = false;
            // Also verify that the item has focus on click.
            // Required for IE where focus is asynchronous and would occur after the click changed
            // the value
            
            if (item.handleClick == null) {
                item.handleClick = this._checkboxClick;
            } 
        }
        
        // Default to overflow:"auto" for richTextItem editors. Otherwise we find they typically
        // overflow into the next column
        if (eT == "RichTextItem" || eT == "richText") {
            if (item.overflow == null) item.overflow = "auto";
        }
    
        item.elementFocus = this._editFormItem_elementFocus;

        // By default set 'canTabToIcons' to false for edit fields
        // can be overridden on a per-field basis
        item.canTabToIcons = false;
    
        // override 'focusInItem' to manage selection
        // - we need more complicated behavior than the standard 'selectOnFocus'.
        //   If we started editing via editOnKeypress we will have updated the value of the item
        //   so should set the cursor at the end of the item value. Otherwise, just select on focus.
        item.focusInItem = this._editFormItem_focusInItem;
        
    }
    
	// Allow for developer specified defaults / properties for this field
	
    var propertyDefaults = this.getEditorProperties(editField, record, rowNum);
    isc.addProperties(item, propertyDefaults);
	// if this grid is databound, any other properties specified in the DS will be picked
	// up by the form during databinding
    return item;
},


// focusInItem override for edit form items.
_editFormItem_focusInItem : function () {
    this.Super("focusInItem", arguments);
    this.grid._updateEditorSelection(this);
},
    
// helper to return the editItem name for some cell
getEditorName : function (rowNum, editField, returnDataPath) {
    // accept a colNum or a field object
    editField = this.getField(editField);
    if (!editField) return null;
    if (returnDataPath && editField.dataPath) return editField.dataPath;
    return editField[this.fieldIdProperty];
},


_editItemStringMethodCache:{},

//>	@method	listGrid.refreshCell()    
//  @include    gridRenderer.refreshCell()
//  @example calculatedCellValue
//<
// override refreshCell just to pass the additional params through to refreshCellValue()
refreshCell : function (rowNum, colNum, refreshingRow, allowEditCellRefresh) {
    if (rowNum == null) {
        
        this.logInfo("ListGrid.refreshCell(): first parameter rowNum not present, returning");
        
        
        return;
    }
    if (!this.isDrawn() || !this.body) return;
    
    var body = this.getFieldBody(colNum);    

	// If the body is already marked for redraw, allow that to handle updating the cell
    if (body.isDirty()) {
        this.logDebug("refresh cell redrawing body", "gridEdit");
        body.redraw("refresh cell");
    	// return false to indicate that we did not update the cell in place, but forced a 
    	// redraw (used to make 'refreshRow' more efficient.
        return false;
    }
    
    if (this._alwaysShowEditors(this.getField(colNum),true)) {
        this._clearingInactiveEditorHTML(rowNum,colNum);
    }

    var bodyColNum = this.getLocalFieldNum(colNum);
    body.refreshCellStyle(rowNum, bodyColNum);
    // refresh the value too unless it's already been refreshed as part of styling
    if (!body.shouldRefreshCellHTML()) {
        this.refreshCellValue(rowNum, colNum, refreshingRow, allowEditCellRefresh);
    }
},

// refreshCellValue overridden to handle refreshing cells within the edit row
// If we are showing an edit form item for this cell:
// - if the form item is present in the DOM, and has focus:
//   - we must save out it's element value (may be dirty)
//   - blur the item without firing the handler
// - rewrite the HTML for the cell (including the form item)
// - inform the form item that it has been written out / redrawn in the DOM
// - reset the element value
// - if the item had focus, re-set focus (and selection).

refreshCellValue : function (rowNum, colNum, refreshingRow, allowEditCellRefresh) {
    if (!this.isDrawn() || !this.body) return;

    var body = this.getFieldBody(colNum),
        bodyColNum = this.getLocalFieldNum(colNum);

    // If we need to delay the refresh, fire again after a delay
    if (!body._readyToRefreshCell(rowNum, bodyColNum)) {
        this.delayCall("refreshCellValue", [rowNum, colNum, refreshingRow, allowEditCellRefresh]);
        return;
    }
	// Handle the case of showing an edit form field for this cell.
	 
    var editFieldName = this.getEditorName(rowNum, colNum),
        fieldName = this.getFieldName(colNum),
        form = this._editRowForm,
        editItem, cellHasFocus = false,
        cellShowingEditor, cellWillShowEditor;
    if (form) {    
        var editItem = form.getItem(editFieldName),
            rowHasEditor = (this._editorShowing && rowNum == this.getEditRow());
        // sanity check if the colNum on the edit item doesn't match the colNum of the
        // cell we're refreshing, it doesn't relate to this cell.
        // This occurs in CubeGrids where we have one record per cell
        if (editItem && editItem.colNum != colNum) editItem = null;

        if (rowHasEditor) {
            // whether there is currently an editor in the cell
            cellShowingEditor = (editItem && editItem.isDrawn());
            // whether there will be an editor in the cell after refresh
            cellWillShowEditor = this.canEditCell(rowNum, colNum);
            if (editItem != null && form.hasFocus) {
                var formFocusItem = form.getFocusSubItem();
                cellHasFocus = (formFocusItem == editItem || 
                               (editItem.items && editItem.items.contains(formFocusItem)));
            }
        // catch the case where we're clearing out a drawn item
        
        } else if (editItem && editItem.rowNum == rowNum) {
            cellWillShowEditor = false;
            cellShowingEditor = editItem.isDrawn();
        }
    }
    
	
    if (!allowEditCellRefresh && (cellHasFocus && cellShowingEditor && cellWillShowEditor))
    {  
        return;
    }
	// If there is a visible editor in this cell, update it's value and blur before redrawing
    if (cellShowingEditor) {            
    	
        this.getUpdatedEditorValue();
        if (editItem != null) {            
            if (cellHasFocus) {
                // Note - if the item will be visible after this method, silently blur and
                // refocus. Otherwise allow the blur handler to fire, since we won't be 
                // restoring focus.
                if (cellWillShowEditor) {         
                     form._blurFocusItemWithoutHandler();
                }
                else editItem.blurItem();
            }
            // If this method will redraw a form item, notify it now
            if (cellWillShowEditor) editItem.redrawing();
        }
    }
	// Call the body's method to update the HTML of the cell:    
    body.refreshCellValue(rowNum, bodyColNum);
    if (editItem && (cellShowingEditor || cellWillShowEditor)) {
        // Call our method to fire the appropriate 'drawn()' / 'redrawn()' / 'cleared()' 
        // notification on the edit item.
        this._editItemsDrawingNotification(editItem, null, body);
        if (cellWillShowEditor) {
            // restore the element value (since it's not written out with the element)
            
            editItem.setValue(this.getEditDisplayValue(rowNum, colNum));
        	// restore focus if it had focus
        	
            if (cellHasFocus) {
                form._focusInItemWithoutHandler(editItem);
            }
        }
    }
},


//>	@method	listGrid.refreshRow()
// @include gridRenderer.refreshRow()
//<
refreshRow : function (rowNum) {
    if (!this.body || !this.isDrawn()) return;

	// If the body is already dirty, allow the redraw to handle updating the row.
    var frozenFields = this.frozenFields && this.frozenFields.length > 0;
    // If deriveVisibleFields is called, and then this method fires before the grid has redrawn
    // we won't  have created the frozen body. Treat this case like frozenFields is false;
    if (frozenFields && !this.frozenBody) frozenFields = false;
   
    
    //var dirty = this.body.isDirty() || 
    //            frozenFields ? (this.frozenBody.isDirty() || this.bodyLayout.isDirty()) : false;
    var dirty = false;
    if (this.body.isDirty()) dirty = true;
    if (frozenFields && !dirty) {
        if (this.frozenBody.isDirty() || this.bodyLayout.isDirty()) dirty = true;
    }
    
    if (dirty) {
        var bodyWidget = frozenFields ? this.bodyLayout : this.body;
        return bodyWidget.redraw("refresh row");
    }
    
    if (this._editorShowing && this._editRowForm != null) {
        this.logInfo("refresh row: " + rowNum, "gridEdit");
    }
    
    // just call 'refreshCell' on all drawn cells
    if (frozenFields) {
        for (var i = 0; i < this.frozenFields.length; i++) {
            this.refreshCell(rowNum, this.getFieldNum(this.frozenFields[i]), true);
        }
    }

    var firstVisible = this.body._firstDrawnCol,
        lastVisible = this.body._lastDrawnCol;

    for (var i = firstVisible; i <= lastVisible; i++) {
        var colNum = this.getFieldNumFromLocal(i, this.body);
        this.refreshCell(rowNum, colNum, true);
    }
},

//>	@method listGrid.startEditingNew() (A)
//
// Start editing a new row, after the last pre-existing record in the current set of data.
// <P>
// This new row will be saved via the "add" +link{group:dataSourceOperations,DataSource
// operation}.
// <P>
// If editing is already underway elsewhere in the grid, startEditingNew() behaves just like
// +link{startEditing()}.
//
// @group  editing
//
// @param  [newValues] (object)  Optional initial set of properties for the new record
// @param  [suppressFocus] (boolean) Whether to suppress the default behavior of moving focus
//                                   to the newly shown editor.
// @see    startEditing()
// @visibility external
//<
startEditingNew : function (newValues, suppressFocus) {

    // force editing on if it's not configured for any field, but a programmatic call is made
    if (!this.canEdit && !(this.completeFields || this.fields).getProperty("canEdit").or()) {
        this.canEdit = true;
    }

    
    if (isc.isAn.Array(this.data) && this.data.length == 0 && 
        this.dataSource && !this.saveLocally) 
    {
        this.fetchData(null, null, {_suppressFetch:true});
        this.data.setFullLength(0);
    }    

	// The new row will be added to the end of the current set of rows
	
    
    var newRowNum = this.body ? this.body.getTotalRows() : this.getTotalRows();
    
    // If we're showing the 'edit new record' row, ensure we insert the record over that row,
    // rather than inserting after that row.
    if (this.showNewRecordRow) newRowNum -= 1;
    

    var newEditCell = this.findNextEditCell(newRowNum, 0, 1, true, true, true);
    
    // newEditCell can be null if canEditCell returned false for all fields!
    if (newEditCell == null) {
        this.logInfo("startEditingNew() failed to find any editable fields in this grid.", 
                     "gridEdit");
        return;
    }
    
	// Suppress displaying the new edit values - handled by 'startEditing', which will show
	// the edit form for the row.
    if (newValues != null) this.setEditValues(newEditCell, isc.addProperties({}, newValues), 
                                              true);

    // fall through to 'startEditing()' -- handles any current edit in another cell.
    this.startEditing(newEditCell[0], newEditCell[1], suppressFocus);
},

//>	@method listGrid.updateEditRow()  (IA)
//
// Internal method to update the set of form fields written into the ListGrid body's
// currently editable row (after a call to editRow()).
// - Updates the values of the form items
// - Focuses in the appropriate item
//
//  @group  editing
//
//  @param  rowNum      (number)    Row number to update.
//
//  @visibility internal
//<
updateEditRow : function (rowNum) {
	// if updateEditRow is called on a delay, by the time this update occurs, we may have moved
	// on to another row.
    if (this._editRowNum != rowNum || !this._editRowForm) {
    	//this.logWarn("updateEditRow bailing, update was for row: " + rowNum + 
    	//             " current editRow is " + this._editRowNum);
        return;
    }

	// update the item values of the edit form.
	
    this._editRowForm.setItemValues();
	// Clear out the _setValuesPending flag
    delete this._editRowForm._setValuesPending;
    this._editRowForm._waitingOnUpdate = false;
},

// shouldFixRowHeight()
// Internal method allowing 'fixedRecordHeights' to be overridden for individual rows in the
// GridRenderer.
// Currently only used for row-level editing.
// When row-level editing is enabled, we want to allow the row showing the embedded editor to
// expand to accommodate it's contents regardless of this.fixedRecordHeights
//

shouldFixRowHeight : function (record, rowNum) {

	// if this row is being edited, don't vertically clip
	// (Note check for != false rather than == true, as if lg.canEdit is unset, we support
	// editing on fields where canEdit is explicitly set to true)
    if (this.canEdit != false && this._editorShowing && rowNum == this._editRowNum) {
       return false;
    }
    return this.fixedRecordHeights;
},

//--- helpers for edit flow methods

// Provide unique identifiers to be used to identify the edit flow.

_getNextEditFlowID : function () {
    if (this.__lastEditFlowID == null) this.__lastEditFlowID = isc.timeStamp();
    return this.__lastEditFlowID ++;
},

// internal helper method for click outside editor - avoids us having to determine the appropriate
// cell's value in a click-mask event handler type stringMethod.
_handleClickOutsideEditor : function () {
    var editRow = this.getEditRow();
    this.cellEditEnd(isc.ListGrid.CLICK_OUTSIDE);
},

// Retrieving / updating editValues.
// We store copies of edited values locally for multiple records in the _editSessions
// object.  This allows us to hang onto edit values for more than one edited record at a time
// - required for records that have been edited but not yet saved, either because a save is
// in process, but hasn't returned from the server, and 'waitForSave' is false, or because a 
// save failed with validation errors, and stopOnErrors is false.



// value to display in a cell for which there may be edit values present.

// Note: optional 'record' parameter passed by 'getEditDisplayValues' only.
getEditDisplayValue : function (rowNum, colNum, record) {  
    
    var undef;
	// If 'record' is not passed, determine it from rowNum, colNum now.
    if (record === undef) record = this.getCellRecord(rowNum, colNum);

	// use the edit value for the cell if present
    var value = this._getEditValue(rowNum, colNum);

    if (value === undef && record != null) {
    	
        value = this.getRawCellValue(record, rowNum, colNum);
    }
	// If a formatter is defined for the editor values, apply it now
    value = this._formatEditorValue(value, record, rowNum, colNum);
    
    return value;
},

// Internal method to fire developer defined 'formatEditorValue' methods at either the Grid or
// Field level.
_formatEditorValue : function (value, record, rowNum, colNum) {
    // If this is a new row, the record passed in will be null - 
    // In this case pass the edit values to the formatter instead
    
    if (record == null) record = this._getEditValues(rowNum, colNum);
    
	// If a field-level formatter is defined, apply it.
    var field = this.fields[colNum];
    
    if (field && field.formatEditorValue != null) {
    	
        isc.Func.replaceWithMethod(field, "formatEditorValue", 
                                            "value,record,rowNum,colNum,grid");
        value = field.formatEditorValue(value, record, rowNum, colNum, this);
	// Only apply a Grid-level formatter if no formatter exists at the field level.
    } else if (this.formatEditorValue != null) {
        value = this.formatEditorValue(value, record, rowNum, colNum);
    }
    return value;
},

//>	@method listGrid.getEditValuesID() 
//
//  Given either a rowNum, a set of primary key values,
//  returns a unique identifier for the set of temporary locally stored edit values for some
//  record being edited.
//  If passed the editValuesID, it will just be returned.
//
//  @group  editing
//  @visibility advancedInlineEdit
//  @param  ID (number | object | string)    Identifier for editValues for which we need to
//                                           return the unique editValuesID.
//  @return (string)   Unique identifier for the set of editValues.
//<

getEditValuesID : function (ID) {

    if (ID == null || this._editSessions == null) return null;

    if (isc.isA.String(ID) && this._editSessions[ID] != null) return ID;
    
	// rowNum (common case)
    if (isc.isA.Number(ID)) return this._editRowMap[ID];
    
    // handle primary keys object OR the edit data object itself
    for (var i in this._editSessions) {
        var data = this._editSessions[i];
        if (data == ID) return i;
        var pk = data._primaryKeys;
        if (pk && this.comparePrimaryKeys(pk, ID)) {
            return i;
        }
    }
    
    return null;
},

// getEditSession - returns the edit data object for some row.
// This contains the editValues, the rowNum (if known), any validation errors, and primary
// keys for the edited record.
getEditSession : function (editDataID, colNum) {
	// editDataID can be a rowNum, a primary key or an editValuesID string.
    if (this._editSessions == null) return null;
    
    if (!isc.isA.String(editDataID)) editDataID = this.getEditValuesID(editDataID, colNum);
    return this._editSessions[editDataID];
},

// getEditSessionRowNum   - given an edit data object, or an ID for an edit data object, returns
// the rowNum of the record associated with the data.
// May be null if we're editing a new row on the end of the list, or with a paged result-set
// if we don't yet know the rowNum for the record.
getEditSessionRowNum : function (editDataID) {
    editDataID = this.getEditSession(editDataID);
    return (editDataID != null ? editDataID._rowNum : null)
},

getEditSessionColNum : function (editDataID) {
    editDataID = this.getEditSession(editDataID);    
    // Note that the _colNum will be unset if we have 1 record per row
    return (editDataID != null ? editDataID._colNum : null)
},

//>	@method listGrid.getAllEditRows() 
// Returns an array of every rowNum for which we have pending (unsubmitted) edits.
//  @group  editing
//  @visibility external
//  @return (array) Array of rowNums for rows with edit values pending submission.
//<

getAllEditRows : function (getIds) {
    return this.getAllEditCells(getIds, true);
},


getAllEditCells : function (getIds, rowsOnly) {
    var registry = this._editSessions,
        cells = [];
    if (!registry) return cells;

    if (getIds) return isc.getKeys(this._editSessions);

   
    for (var i in registry) {
        var rowNum = registry[i]._rowNum;
        // Convert "1" to 1, etc.
        
        rowNum = parseInt(rowNum);
        if (rowNum == null || rowNum < 0 || isNaN(rowNum)) continue;
        if (rowsOnly) cells[cells.length] = rowNum
        else {
            var colNum = registry[i]._colNum;
            // If rowNum only was stored on the cell, use the special getRowEditColNum() to
            // figure out the colNum in question
            if (colNum == null) {
                colNum = this._editorShowing && (rowNum == this.getEditRow()) 
                                    ? this.getEditCol() : this.getRowEditColNum(rowNum);
            }
            if (colNum != null && !isc.isA.Number(colNum)) colNum = parseInt(colNum);
            cells[cells.length] = [rowNum, colNum];
        }
    }
    return cells;  
},

//>	@method listGrid.getEditValues() 
// Returns the current set of unsaved edits for a given row being edited.
//
//  @param  valuesID (number | Object)  rowNum of the record being edited, or an Object
//                                      containing values for all the record's primary keys
//  @return (object)   Current editValues object for the row.  This contains the current
//                     edit values in {fieldName1:value1, fieldName2:value2} format.
//  @group  editing
//  @visibility external
//<
//  @param [colNum] (number) colNum of the record being edited. Only required if valuesID
//                           is passed in as a rowNum, and we're displaying one record per cell
//                          as in a +link{CubeGrid}

getEditValues : function (valuesID, colNum) {
    if (valuesID == null) {
        return this.logWarn("getEditValues() called with no valuesID. " 
                + (this.logIsDebugEnabled("gridEdit") ? this.getStackTrace() : ""));
    }
    // handle being passed a 2 element array [rowNum,colNum] since this is the format we
    // accept for setEditValues()
    if (colNum == null && isc.isA.Array(valuesID)) {
        colNum = valuesID[1];
        valuesID = valuesID[0];
    }
    
    // If we're showing an editor for this row, ensure that the current value in the
    // edit field is present in the editValues we return.
    
    var rowNum = (isc.isA.Number(valuesID) ? valuesID : this.getEditSessionRowNum(valuesID));
    if (this._editorShowing && (this.getEditRow() == rowNum)) {   
        this.storeUpdatedEditorValue();
    }
    
    return this._getEditValues(valuesID, colNum);
},

// Retrieve the stored edit values for some row (or null)
// Unlike the public method, this will not first update the editvals with the current value 
// from the editForm (if showing)
_getEditValues : function (valuesID, colNum) {
	// we may be passed the editValues object, in which case we're passing it back again, 
    var editSession = this.getEditSession(valuesID, colNum);
    return editSession != null ? editSession._editValues : null;
},

//> @method listGrid.getEditedRecord()
// Returns the combination of unsaved edits (if any) and original values (if any) for a given
// row being edited.
// <P>
// The returned value is never null, and can be freely modified.
//
// @param  valuesID (number | Object)  rowNum of the record being edited, or an Object
//                                      containing values for all the record's primary keys
// @return (Object) A copy of the record with unsaved edits included
// @group  editing
// @visibility external
//<

getEditedRecord : function (rowNum, colNum, suppressUpdate) {
    if (rowNum == null) return this.logWarn("getEditedRecord() called with no valuesID");

    // the valuesID allows for rowNum independent data storage - if passed a valuesID
    // determine resolve to rowNum/colNum here so we can get a pointer to the record object
    if (!isc.isA.Number(rowNum)) {
        rowNum = this.getEditSessionRowNum(rowNum);
        colNum = this.getEditSessionColNum(rowNum);
    }
    
    var record = this.getCellRecord(rowNum, colNum),
        // respect the parameter to avoid checking the edit form for updates
        editValues = suppressUpdate ? this._getEditValues(rowNum, colNum) 
                                    : this.getEditValues(rowNum, colNum);

    
    return isc.addProperties({}, record, editValues);
},

//> @method listGrid.getEditedCell()
// Returns the current value of a cell. If the cell has an outstanding edit value, this will
// be returned, otherwise the underlying value of the record will be returned.
//
// @param  record (number | Object)  rowNum of the record being edited, or an Object
//                                      containing values for all the record's primary keys
// @param field (number | string) colNum or fieldName of the cell
// @return (any) Current edit value, or underlying value for the cell 
// @group  editing
// @visibility external
//<
getEditedCell : function (record, field) {
    if (record == null || field == null) 
        return this.logWarn("getEditedCell() called with no record / field parameter");

    var editValues = this.getEditValues(record, field),
        rowNum = isc.isA.Number(record) ? record : this.getEditSessionRowNum(record),
        colNum = isc.isA.Number(field) ? field : this.getFieldNum(field),
        record = this.getCellRecord(rowNum, colNum);

    var editFieldName = this.getEditorName(rowNum, this.getField(colNum))
    var undef;
        
    if (editValues && editValues[editFieldName] !== undef) return editValues[editFieldName];        
    return record ? record[editFieldName] : null;
},

// When we attempt to save an edit remember the edit values
// We use this to perform intelligent change detection while pending a save on the server
rememberSubmittedEditValues : function (valuesID, colNum) {
    
    var editSession = this.getEditSession(valuesID);
    if (editSession != null) {
        editSession._submittedValues = isc.addProperties({}, editSession._editValues);
    }
},

// retrieve the last set of editValues submitted to the server.
getSubmittedEditValues : function (valuesID, colNum) {
    var editSession = this.getEditSession(valuesID, colNum);
    return editSession != null ? editSession._submittedValues : null;
},

// Clear the stored "submitted edit values" 
// called from editFailedCallback method - oldValues object is required so we don't clear
// the submitted editValues if they have been modified since the (failed) save was committed
// a case we can only hit if 2 overlapping saves have been kicked off.
clearSubmittedEditValues : function (valuesID, oldValues) {
    var editSession = this.getEditSession(valuesID);
    if (editSession == null) return;
    var submittedVals = editSession._submittedValues;
    if (!submittedVals) return;
    
    for (var field in oldValues) {
        if (submittedVals[field] == oldValues[field]) delete submittedVals[field];
    }
    if (isc.isA.emptyObject(submittedVals)) editSession._submittedValues = null;
},



//>	@method listGrid.createEditValues() 
//
// This method creates a new set of editValues for a row at the end of the list, and returns
// the editValuesID which can subsequently be passed to any of the following methods as a
// unique identifier (in place of the 'rowNum' parameter where appropriate):
// 'setEditValue()', 'getEditValues()', 'getEditValue()', 'getEditValues()', 'clearEditValue()'
// and 'clearEditValues()'.<br>
// The new edit values will be displayed at the end of the list.
//
//  @param  values       (any)       New values for the row
//  @visibility advancedInlineEdit
//<

createEditValues : function (values) {
    var rowNum = this.body.getTotalRows();
	// set up the edit values, and display them in the listGrid (don't pass the 
	// suppressDisplay param)
    
    this.setEditValues(rowNum, values);
    return this.getEditValuesID(rowNum);
},



// initializeEditValues() - helper method to set up empty edit vals for some
// record
initializeEditValues : function (rowNum, colNum, displayNewValues) {
    // If we don't have edit values for this record, set them up.
    // (we can use the internal method - we know we don't have outstnding edits in the
    // edit form at this point, as the editor should have been hidden)
    if (this._getEditValues(rowNum, colNum) == null) {
        this.setEditValues([rowNum, colNum], {}, displayNewValues)
        //this.logWarn("editValues for row: " + rowNum + " are now: " + this.echo(this.getEditValues(rowNum)));
    }
},
    

//>	@method listGrid.setEditValues() 
//
// This method sets up a set of editValues for some row / cell.  It differs from 
// 'setEditValue()' in that:<br>
// &nbsp;- it takes values for multiple fields<br> 
// &nbsp;- it clears out any previous edit values for the record<br>
//
//  @param rowNum (number) Row number for the record being edited
//  @param  values       (any)       New values for the row
//
// @visibility external
//<
// @param   suppressDisplay (boolean)   Additional internal parameter to suppress updating the
//                                      affected row to display the new edit values.
// Note that the first param may be a 2 element array of rowNum, colNum for one record-per-cell
// data models (documented in CubeGrid)

setEditValues : function (rowNum, editValues, suppressDisplay) {
    var colNum;
    if (isc.isAn.Array(rowNum)) {
        colNum = rowNum[1];
        rowNum = rowNum[0];
    }
    
    if (!isc.isA.Number(rowNum)) {
        this.logWarn("setEditValues() called with bad rowNum: " + this.echo(rowNum));
        return;
    }
    
    // Default to an empty set of values - if the user wants to entirely clear an editValues
	// object, they should use 'clearEditValue(s)' instead.
    if (editValues == null) editValues = {};
    
    var oldEditValues, changedFields, addedRow = true;
    if (!suppressDisplay) {
        
        var record = this.getCellRecord(rowNum, colNum);
        if (record == null) record = {};
        else addedRow = false;
        
        oldEditValues = this.getEditValues(rowNum, colNum);
        if (oldEditValues != null) addedRow = false;
        
        changedFields = isc.addProperties({}, oldEditValues);
        for (var i in changedFields) {
            changedFields[i] = record[i];
        }
        
        isc.addProperties(changedFields, editValues);
        
        // At this point changedFields will be a mapping of the new edit display values for
        // each field that has been changed.
    }

    if (this.logIsInfoEnabled("gridEdit")) {
        oldEditValues = oldEditValues || this.getEditValues(rowNum, colNum);
        if (!oldEditValues) {
            this.logInfo("establishing new edit session at row: " + rowNum +
                         (colNum != null ? ", col:" + colNum : "") +
                         (this.logIsDebugEnabled("gridEdit") ?
                         " with values: " + this.echo(editValues) :
                         ""), "gridEdit");
        }
            
    }
    
	// store the new edit value 
    this._storeEditValues(rowNum, colNum, editValues);
    
    var hasModifiedValues = !isc.isAn.emptyObject(changedFields);

    if (suppressDisplay || !this.isDrawn() || !this.body) {
        if (hasModifiedValues) {
            // If suppress display is passed we do still want to redraw the summary row if there
            // is one since the calling code never handles that
            if (this.summaryRow && this.showGridSummary) this.summaryRow._recalculateSummaries();
        }
        return;
    }
    
    if (hasModifiedValues && rowNum < this.data.getLength()) {
        // don't refresh display when recacluating summaries - we'll refresh the row and
        // the grid summary if necessary.
        this.calculateRecordSummaries([this.data.get(rowNum)], true);
    }

	// values shown in the cells for this row are now stale (whether each cell is being edited
	// or not), so refresh.
    
	// if totalRows changed, it indicates that that a new edit row is being created at the end
	// of the body, hence we need to redraw to add the row to the DOM
    var shouldRedrawBody =  (addedRow || this.body.isDirty());
        
    if (shouldRedrawBody) {
        var editorShowing = this.isEditingRecord(rowNum, colNum) && this._editRowForm != null
    
    	
        if (editorShowing) this._updateEditItemValues();

        this.body.markForRedraw(
        	
        );
        if (hasModifiedValues && this.summaryRow && this.showGridSummary) {
            this.summaryRow._recalculateSummaries();
        }
        
    } else {    
        this._displayNewEditValues(rowNum, colNum, changedFields);
    }
},

_displayNewEditValues : function (rowNum, colNum, changedFields, errors) {
    
    if (!changedFields || isc.isAn.emptyObject(changedFields)) {        
        return;
    }
    
    var editorShowing = this.isEditingRecord(rowNum, colNum) && this._editRowForm != null
    // update each cell effected by the change, either by setting the form item value or
	// refreshing the cell
    
    var hasVisibleChanges = false;
    for (var fieldName in changedFields) {
        var fieldColNum = this.getColNum(fieldName);
        
        // If we're showing an edit item for the cell, set it's value rather than
        // refreshing the cell (and rewriting the form item HTML completely)
        var editItemDrawn;
        if (editorShowing) {
            this._editRowForm.setValue(fieldName, changedFields[fieldName]);
            
            var editItem = this._editRowForm.getItem(fieldName);
            editItemDrawn = (fieldColNum >=0 && editItem && this.canEditCell(rowNum, fieldColNum));
        }
        // undrawn cell - just continue to the next field
        // [note do this after modifying edit form values] 
        if (fieldColNum == -1) continue;
        hasVisibleChanges = true;
        
        if (!editItemDrawn) {
            this.refreshCell(rowNum, fieldColNum);
        } else if (errors && errors[fieldName]) {
            this.showCellErrors(rowNum, fieldColNum);
        } 
    }    
    if (hasVisibleChanges && this.summaryRow && this.showGridSummary) {
        this.summaryRow._recalculateSummaries();
        // refresh the group summary!
        this.refreshGroupSummary(rowNum);
    }

},


// Internal helper method for 'setEditValues' - actually stores the editValues object for the
// row
// Call 'setEditValues()' rather than calling this method directly.
// NOTE: editValuesId is a *highly internal* param that allows creating of a new editSession
// with a predictable ID

_storeEditValues : function (rowNum, colNum, editValues, editValuesId) {
	// Get the record for this row.  Note that this might be null as we may be adding a new
	// edit row to the end of the list.
    var record = this.getCellRecord(rowNum, colNum);

	// Assertion - if we don't have the edit data for this rowNum, we don't have the
	// edit data for this record, since 'getRecord()' will always associate the edit data
	// for some record with the appropriate rowNum, if required.
    var editSession = this.getEditSession(rowNum, colNum) || 
            this.createEditSession(rowNum, colNum, record, editValuesId);
	// Always add the primary keys to edit values for databound lists, so that they are
	// available to identify the record in saveEditedValues().  Non-databound lists are
	// expected to use object identity.
    if (this.dataSource != null) {
        // set the 'newRecord' marker on loading rows to indicate that there's no associated
        // record (and we don't have PKs for the row
        if (record == "loading") {
            editSession._newRecord = true;
        } else if (record != null) {
            var ds = this.getDataSource(),
                pkArray = ds.getPrimaryKeyFieldNames();
        
            for (var i = 0; i < pkArray.length; i++) {
                editValues[pkArray[i]] = record[pkArray[i]];
            }
        }
    }

    // set the _newRecord flag for each row without an associated record     
    if (record == null) {
        editSession._newRecord = true;
        
        // if we have any fields with a specified default value, pick it up as a default
        // edit value.
        // (true even if the field is hidden)
        
        // If we are showing any 'enum' type fields, and 
        var fields = this.completeFields || this.fields || [],
            undef;
            
        for (var i = 0; i < fields.length; i++) {
            var field = fields[i],
                fieldName = field[this.fieldIdProperty];
            if (editValues[fieldName] === undef) {
                var defaultValue = this.getDefaultEditValue(fieldName, field);
                if (defaultValue != null) {
                    editValues[fieldName] = defaultValue;
                }
            }
        }
    }

	// clear out any stored editValues and copy the passed editValues into place, preserving
	// the same instance.
    for (var i in editSession._editValues) {
        delete editSession._editValues[i];
    }
    for (var i in editValues) {
        editSession._editValues[i] = editValues[i];
    }

	// Cache the last edit row - used by 'getTotalRows()'.  This value will also be 
	// updated by 'clearEditValues()'
    
    if (this._lastEditRow == null || rowNum >= this._lastEditRow) {
        this._lastEditRow = rowNum
    }

},

// Helper method - gets the default value to show in an editor for a field (if no explicit value
// was specified).
getDefaultEditValue : function (fieldName, field) {
    var defaultValue = field.defaultValue;
    if (defaultValue == null && this.enumCriteriaAsInitialValues &&
        field.type == "enum" && this._filterValues != null&&
        this._filterValues[fieldName] != null)
    {
        // We could check the value against this.getEditorValueMap(fieldName, null)?
        defaultValue = this._filterValues[fieldName];
    }
    return defaultValue;
},

//>	@method listGrid.createEditSession() (I)
//
//  Internal method to initially set up internal, temporary edit values (and old, pre edit values)
//  for a record.
//  This method doesn't store these editValues - use 'setEditValues()' for that
//  These get updated as the record is edited, and cleared out when the edit is complete.
//
//  @group  editing
//  @visibility internal
//<

createEditSession : function (rowNum, colNum, record, editValuesID) {

    var editSession = {};
    if (record != null && record != Array.LOADING) 
        editSession._primaryKeys = this.getPrimaryKeys(record);

	// A null primary key will imply we're editing a new record.
	// Assertion: when initializing edit data we will always be passed a rowNum
	
    editSession._rowNum = rowNum;
    if (this.useCellRecords) {
        editSession._colNum = colNum;
        
        if (this.getCellFacetValues) editSession._facetValues = this.getCellFacetValues(rowNum, colNum);
    }

    editSession._editValues = {};
    
    if (this._editSessions == null) this._editSessions = {};
    
	// generate an editValuesId if not passed one
    if (editValuesID == null) {
        if (this._currentEditValuesID == null) this._currentEditValuesID = 0;
        editValuesID = "_" + this._currentEditValuesID++;
    }
    
    this._editSessions[editValuesID] = editSession; 

    // editRowMap used for retrieving edit sessions / row
    if (this._editRowMap == null) this._editRowMap = {};
    this._editRowMap[rowNum] = editValuesID;
    return editSession;
},    


//>	@method listGrid.setEditValue()     ([A])
//
//  Modifies a field value being tracked as an unsaved user edit.<P>
//  Use for suggested or reformatted values for edits that remain unsaved.
//
//  @group  editing
//  @visibility external
//
//  @param  rowNum      (number)    Row number (or edit values ID)
//  @param  colNum      (number | string)    Column number of cell, or name of field
//                                           having editValue updated
//  @param  value       (any)       New value for the appropriate field.
//<

setEditValue : function (rowNum, colNum, newValue, suppressDisplay, suppressChange) {

    if (isc.isA.String(rowNum)) {
        isc.logWarn("Warning: first parameter rowNum is a string, expecting a number");
        // bail if we have a bad rowNum
        return;
    }
    
    var fieldName = isc.isA.String(colNum) ? colNum : this.getEditorName(rowNum, colNum, true);
    if (isc.isA.String(colNum)) colNum = this.getFieldNum(colNum);

	// store the new edit value 
    
    var changed = this._storeEditValue(rowNum, colNum, fieldName, newValue, suppressChange);
    
	// only proceed if there was a change 
    if (!changed) return;
    
    // If this is an edit value for a field with a specified displayField, also update the
    // edit value of the displayField
    
    
    var field = this.getField(fieldName);
    if (field && this._useDisplayFieldValue(field)) {
        var displayValue;
        // If we're showing an edit form for this row, ask the edit form for the display
        // field's value
        var editForm = this.getEditForm();
        if (editForm && this.getEditRow() == rowNum && 
            this.fieldValuesAreEqual(field, editForm.getValue(fieldName), newValue)) 
        {
            displayValue = editForm.getValue(field.displayField);
        }
        if (displayValue == null) {
            
            var data = this.data;
            if (isc.ResultSet && isc.isA.ResultSet(data)) data = data.localData;
            if (data) {
                var record = data.find(fieldName, newValue);
                displayValue = (record ? record[field.displayField] : newValue);
            } else {
                displayValue = newValue;
            }
        }
        this.setEditValue(rowNum, field.displayField, displayValue, suppressDisplay, true);
    }
    // If we're not supposed to update the display we're done.
    
    if (suppressDisplay) {
        // If suppress display is passed we do still want to redraw the summary row if there
        // is one since the calling code never handles that
        if (this.summaryRow && this.showGridSummary) this.summaryRow._recalculateSummaries();
        return;
    }
    // Remember which field was most recently modified - this will be passed to callbacks as
    // the 'colNum' param
    this.setRowEditFieldName(rowNum, fieldName);
    
    if (!isc.isA.Number(rowNum)) {
        colNum = this.getEditSessionColNum(rowNum);
        rowNum = this.getEditSessionRowNum(rowNum);

    } else if (!isc.isA.Number(colNum)) {
        
        colNum = this.getFieldNum(colNum);
    }
    var vals = {};
    vals[fieldName] = newValue;
    this._displayNewEditValues(rowNum, colNum, vals);
},

// store an edit value, firing editorChange() notification if there was a change.
// internal helper: call setEditValue() instead

_storeEditValue : function (rowNum, colNum, fieldName, newValue, suppressChange) {

    
    var changed = true, saveEqual = false,
        editSession, editValues, oldValue,
        undef;
        
    editSession = this.getEditSession(rowNum, colNum)
    if (editSession != null) {
        editValues = editSession._editValues;
        // using getFieldValue() will allow us to use dataPath 
        oldValue = isc.Canvas._getFieldValue(fieldName, editValues, this);
    } else {
    	// create a new set of edit values as necessary   
        this.logInfo("creating new edit values for row: " + rowNum, "gridEdit");
        this.initializeEditValues(rowNum, colNum, true);
        editSession = this.getEditSession(rowNum, colNum);
        editValues = editSession._editValues;
    }
    
    // if there's no previous editValue, the old value is the original value from the
	// dataset
    if (oldValue === undef) {
        var record = this.getCellRecord(rowNum, colNum);
        oldValue = record ? isc.Canvas._getFieldValue(fieldName, record, this) : null;
    
    } else saveEqual = true;
    
    
    var field = this.getField(fieldName);        
    if (this.fieldValuesAreEqual(field, oldValue, newValue)) changed = false; // indicate no change

    // store the changed value
    // Note: If newValue was not passed in, clear the field value instead
    
    if (newValue === undef) {
        isc.Canvas._clearFieldValue(fieldName, editValues, this);
    } else if (saveEqual || changed) {
        isc.Canvas._saveFieldValue(fieldName, newValue, editValues, this);
    }

//     this.logWarn("edit value changed: oldValue: " + this.echo(oldValue) +
//                   ", new value: " + this.echo(newValue) + 
//                   ", save equal?:" + saveEqual +
//                   " updated edit vals:" + this.echo(editValues), "gridEdit");
	// fire the change notification.
    if (changed && !suppressChange){
        this._editorChange(rowNum, colNum, newValue, oldValue);
    }

    return changed; // indicate change
},

// setRowEditFieldName() - used to track which field was last being edited for some set of edit values
setRowEditFieldName : function (rowNum, fieldName) {

    var editSession = this.getEditSession(rowNum);
    // Set up empty edit values if necessary
    if (!editSession) {
        var colNum = this.getColNum(fieldName);
        this.setEditValues([rowNum, colNum], null, true);
        editSession = this.getEditSession(rowNum);
    }
    
    if (isc.isA.Number(fieldName)) fieldName = this.getFieldName(fieldName);
    editSession._lastField = fieldName;
},

// getEditField() - given an edited row / valuesID, return the name of the last field being
// edited for that row
getRowEditFieldName : function (editValuesID) {
    var editSession = this.getEditSession(editValuesID);
    return (editSession ? editSession._lastField : null);
},

// getRowEditColNum - returns the colNum for the last edited field within some edit row
getRowEditColNum : function (editValuesID) {
    var fieldName = this.getRowEditFieldName(editValuesID);
    return fieldName ? this.getColNum(fieldName) : null;
},


//>	@method listGrid.getEditValue() 
//
// Returns the current temporary locally stored edit value for some field within a record 
// being edited.
//
// @param  rowNum  (number)    index of the row for which the editValue should be returned
// @param  colNum (number | string) index of the field, or fieldName, for which value should be 
//                              returned
// @return (any)   edit value for the field in question
// @group  editing
// @visibility external
//<

getEditValue : function (rowNum, colNum) {

    var colID = colNum
    if (isc.isA.String(colNum)) colNum = this.getColNum(colNum);
    if (this._editorShowing && 
        (this.getEditRow() == rowNum) && (this.getEditCol() == colNum)) 
    {
        this.storeUpdatedEditorValue();
    }    
    // Note pass the original column ID (colNum or fieldName) on to the _getEditValue() method
    // since there may be no column associated with the fieldName passed in
    return this._getEditValue(rowNum, colID);
},

// Retrieve the stored edit value for some row / field
// (unlike the public method, this will NOT first check for the value present in the editor 
// if showing)
_getEditValue : function (rowNum, colNum) {
	// Note: Edit values are a case where the distinction between a value being unset and
	// being set to null is important... One case implies the user has not edited a cell,
	// the other implies the user has cleared out an edit value from a cell...
    var vals = this._getEditValues(rowNum, colNum);
	// Return undefined rather than explicit null if the edit row doesn't have any stored 
	// values
    if (vals == null) return;
    // Convert the colNum to a fieldName to get the value
    // The 3rd parameter ensures we pick up a field dataPath if specified
    if (!isc.isA.String(colNum)) colNum = this.getEditorName(rowNum, colNum, true);
    
    return isc.Canvas._getFieldValue(colNum, vals, this);
},



//>	@method listGrid.clearEditValue()   ([A])
//
//  Clear a field value being tracked as an unsaved user edit.<P>
//  The saved record value will be displayed in the the appropriate cell instead.
//  Will also discard any validation errors for the specified field / row.
//
//  @group  editing
//  @visibility external
//
//  @param  editValuesID (number | object)    Row number, primary keys object for the record,
//                                              or editValues object
//  @param  colNum (number | string)    Column number, or Name of field for which 
//                                      the value is to be cleared
//<
