/**
 * Copyright (C) 2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.exoplatform.platform.gadgets.services;

import java.util.ArrayList;
import java.util.List;

import org.exoplatform.application.gadget.Gadget;
import org.exoplatform.application.gadget.GadgetRegistryService;
import org.exoplatform.application.registry.ApplicationCategory;
import org.exoplatform.application.registry.ApplicationRegistryService;
import org.exoplatform.container.PortalContainer;
import org.exoplatform.container.component.RequestLifeCycle;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.portal.config.UserACL;
import org.exoplatform.portal.config.model.ApplicationType;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.picocontainer.Startable;

/**
 * @author <a href="mailto:anouar.chattouna@exoplatform.com">Anouar
 *         Chattouna</a>
 * @version $Revision$
 */
public class PopulateGadgetRegisryService implements Startable {

  private static final Log LOG = ExoLogger.getExoLogger(PopulateGadgetRegisryService.class);

  private static final String DEFAULT_GADGETS_CATEGORY_NAME = "Gadgets";

  private String categoryName;
  private String gadgetsCategoryAccessPermission;
  private GadgetRegistryService gadgetRegistryService;
  private ApplicationRegistryService applicationRegistryService;
  private List<Gadget> gadgets;

  public PopulateGadgetRegisryService(GadgetRegistryService gadgetRegistryService,
      ApplicationRegistryService applicationRegistryService, InitParams initParams) {
    categoryName = initParams.getValueParam("gadgetsCategoryName").getValue();
    if (categoryName == null) {
      categoryName = DEFAULT_GADGETS_CATEGORY_NAME;
      LOG.warn("Failed to retrieve " + initParams.getValueParam("gadgetsCategoryName").getName()
          + " init param. Default category name will be used: " + DEFAULT_GADGETS_CATEGORY_NAME);
    }
    gadgetsCategoryAccessPermission = initParams.getValueParam("gadgetsCategoryAccessPermission").getValue();
    if (gadgetsCategoryAccessPermission == null) {
      gadgetsCategoryAccessPermission = UserACL.EVERYONE;
      LOG.warn("Failed to retrieve " + initParams.getValueParam("gadgetsCategoryAccessPermission").getName()
          + " init param. Default access permission will be used: " + UserACL.EVERYONE);
    }
    gadgets = initParams.getObjectParamValues(Gadget.class);
    this.gadgetRegistryService = gadgetRegistryService;
    this.applicationRegistryService = applicationRegistryService;
  }

  /**
   * Saves the gadget read from configuration file in the appropriate
   * application category.
   */
  public void start() {
    for (Gadget gadget : gadgets) {
      RequestLifeCycle.begin(PortalContainer.getInstance());
      try {
        // save the gadget via the GadgetRegistryService
        // check if the gadget was saved elsewhere
        if (gadgetRegistryService.getGadget(gadget.getName()) == null) {
          gadgetRegistryService.saveGadget(gadget);
        }
        ArrayList<String> permissions = new ArrayList<String>();
        String[] permissionEntry = gadgetsCategoryAccessPermission.split(",");
        for (String entry : permissionEntry) {
          permissions.add(entry);
        }
        // creates the registry application
        org.exoplatform.application.registry.Application registryApplication = new org.exoplatform.application.registry.Application();
        registryApplication.setApplicationName(gadget.getName());
        registryApplication.setType(ApplicationType.GADGET);
        registryApplication.setDisplayName(gadget.getTitle());
        registryApplication.setContentId(gadget.getName());
        String description = (gadget.getDescription() == null || gadget.getDescription().length() < 1) ? gadget.getName()
            : gadget.getDescription();
        registryApplication.setDescription(description);
        registryApplication.setAccessPermissions(permissions);
        registryApplication.setCategoryName(categoryName);
        if (applicationRegistryService.getApplicationCategory(categoryName) == null) {
          // creates the application category
          ApplicationCategory category = new ApplicationCategory();
          category.setName(categoryName);
          category.setDisplayName(categoryName);
          category.setDescription(categoryName);
          category.setAccessPermissions(permissions);
          applicationRegistryService.save(category, registryApplication);
        } else {
          applicationRegistryService.save(applicationRegistryService.getApplicationCategory(categoryName), registryApplication);
        }
      } catch (Exception e) {
        LOG.error("Error while saving gadget: " + gadget.getName() + " with " + categoryName + " application category. ", e);
        throw new RuntimeException(e);
      } finally {
        RequestLifeCycle.end();
      }
    }
  }

  public void stop() {}

}
