/**
 * Copyright (C) 2003-2008 eXo Platform SAS.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see<http://www.gnu.org/licenses/>.
 */

package org.exoplatform.ws.security.oauth.http;

import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.oauth.OAuth;
import net.oauth.OAuthAccessor;
import net.oauth.OAuthMessage;
import net.oauth.OAuthProblemException;
import net.oauth.server.OAuthServlet;

import org.apache.commons.logging.Log;
import org.exoplatform.container.ExoContainer;
import org.exoplatform.container.ExoContainerContext;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.ws.security.oauth.OAuthProviderService;

/**
 * @author <a href="mailto:andrew00x@gmail.com">Andrey Parfonov</a>
 * @version $Id: $
 */
public class OAuthAccessTokenServlet extends HttpServlet {

  /**
   * Generated by Eclipse.
   */
  private static final long serialVersionUID = 2950381189568117176L;

  /**
   * Logger.
   */
  private static final Log  LOG              = ExoLogger.getLogger(OAuthAccessTokenServlet.class.getName());

  /**
   * Create access token. {@inheritDoc}
   */
  @Override
  public void service(HttpServletRequest httpRequest, HttpServletResponse httpResponse) throws ServletException,
                                                                                       IOException {
    OAuthMessage oauthMessage = OAuthServlet.getMessage(httpRequest, null);
    ExoContainer container = ExoContainerContext.getCurrentContainer();
    OAuthProviderService providerService = (OAuthProviderService) container.getComponentInstanceOfType(OAuthProviderService.class);
    OAuthAccessor accessor;

    try {
      accessor = providerService.getAccessor(oauthMessage);

      if (LOG.isDebugEnabled())
        LOG.debug("Accessor: " + accessor);

    } catch (OAuthProblemException e) {
      e.printStackTrace();
      httpResponse.sendError(HttpServletResponse.SC_UNAUTHORIZED, "Request token is invalid!");
      return;
    }

    try {
      oauthMessage.validateSignature(accessor);
    } catch (Exception e) {
      e.printStackTrace();
      httpResponse.sendError(HttpServletResponse.SC_UNAUTHORIZED, "Signature validation failed!");
      return;
    }
    
    if (!Boolean.TRUE.equals(accessor.getProperty("authorized"))) {
      httpResponse.sendError(HttpServletResponse.SC_UNAUTHORIZED, "Request token is not authorized!");
      return;
    }

    try {
      providerService.generateAccessToken(accessor);
    } catch (OAuthProblemException e) {
      e.printStackTrace();
      httpResponse.sendError(HttpServletResponse.SC_UNAUTHORIZED, "Can't generate access token!");
      return;
    }

    if (LOG.isDebugEnabled())
      LOG.debug(accessor.accessToken + "/" + accessor.tokenSecret);

    httpResponse.setContentType("text/plain");
    OutputStream out = httpResponse.getOutputStream();
    OAuth.formEncode(OAuth.newList(createParametersList(accessor)), out);
    out.flush();
    out.close();
  }

  /**
   * Create parameters list, as 'name', 'value', 'name', 'value', ...
   * 
   * @param accessor OAuthAccessor
   * @return parameter list
   */
  @SuppressWarnings("unchecked")
  private String[] createParametersList(OAuthAccessor accessor) {
    List<String> l = new ArrayList<String>();
    l.add(OAuth.OAUTH_TOKEN);
    l.add(accessor.accessToken);
    l.add(OAuth.OAUTH_TOKEN_SECRET);
    l.add(accessor.tokenSecret);
    l.add("oauth_user_id");
    l.add((String) accessor.getProperty("oauth_user_id"));
    Collection<String> s = (Collection<String>) accessor.getProperty("oauth_user_roles");
    for (String r : s) {
      l.add("oauth_user_role");
      l.add(r);
    }
    return l.toArray(new String[l.size()]);
  }

}
