/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 Meeds Association
 * contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.exoplatform.services.rest.ext.transport;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;

/**
 * SerialInputData is useful for transfer data represented by stream via RMI.
 * 
 * @author <a href="mailto:andrew00x@gmail.com">Andrey Parfonov</a>
 * @version $Id: $
 */
public class SerialInputData implements Serializable
{

   /**
    * Generated by Eclipse.
    */
   private static final long serialVersionUID = -8333597717110874434L;

   /**
    * Buffer size.
    */
   private static final int MAX_BUFFER_SIZE = 204800;

   /**
    * Data stream.
    */
   private InputStream stream;

   /**
    * Constructs new instance of SerialInputData.
    * 
    * @param stream data stream
    */
   public SerialInputData(InputStream stream)
   {
      if (stream == null)
         throw new IllegalArgumentException("Stream can't be null!");

      this.stream = stream;
   }

   /**
    * Constructs new instance of SerialInputData.
    * 
    * @param bytes data
    */
   public SerialInputData(byte[] bytes)
   {
      this(new ByteArrayInputStream(bytes));
   }

   /**
    * @return data stream
    */
   public InputStream getStream()
   {
      return stream;
   }

   /**
    * Write data stream in supplied {@link ObjectOutputStream}.
    * 
    * @param out See {@link ObjectOutputStream}
    * @throws IOException if any i/o errors occurs
    */
   private void writeObject(ObjectOutputStream out) throws IOException
   {
      byte[] buffer = new byte[8192];
      int bytes = 0;
      while ((bytes = stream.read(buffer)) >= 0)
      {
         if (bytes > 0)
         {
            out.writeInt(bytes);
            out.write(buffer, 0, bytes);
         }
      }
      out.writeInt(0);
      stream.close();
   }

   /**
    * Restore data stream from supplied {@link ObjectInputStream}.
    * 
    * @param in See {@link ObjectInputStream}
    * @throws IOException if any i/o errors occurs
    */
   private void readObject(ObjectInputStream in) throws IOException
   {

      boolean overflow = false;
      byte[] buffer = new byte[8192];

      ByteArrayOutputStream bout = new ByteArrayOutputStream(MAX_BUFFER_SIZE);

      for (int bytes = in.readInt(); bytes > 0; bytes = in.readInt())
      {
         in.readFully(buffer, 0, bytes);
         bout.write(buffer, 0, bytes);
         if (bout.size() > MAX_BUFFER_SIZE)
         {
            overflow = true;
            break;
         }
      }

      if (!overflow)
      {
         // small data , use bytes
         stream = new ByteArrayInputStream(bout.toByteArray());
         return;
      }

      // large data, use file
      final File file = File.createTempFile("restejb-", null);
      OutputStream out = new FileOutputStream(file);

      // copy data from byte array in file
      bout.writeTo(out);

      for (int bytes = in.readInt(); bytes > 0; bytes = in.readInt())
      {
         in.readFully(buffer, 0, bytes);
         out.write(buffer, 0, bytes);
      }

      out.close();

      stream = new FileInputStream(file)
      {

         private boolean removed = false;

         /**
          * {@inheritDoc}
          */
         @Override
         public void close() throws IOException
         {
            try
            {
               super.close();
            }
            finally
            {
               // file must be removed after using
               removed = file.delete();
            }
         }

         /**
          * {@inheritDoc}
          */
         @Override
         protected void finalize() throws IOException
         {
            try
            {
               // if file was not removed
               if (!removed)
                  file.delete();

            }
            finally
            {
               super.finalize();
            }
         }
      };
   }

}
