package org.infinispan.configuration.cache;

import static org.infinispan.configuration.cache.ClusteringConfiguration.BIAS_ACQUISITION;
import static org.infinispan.configuration.cache.ClusteringConfiguration.BIAS_LIFESPAN;
import static org.infinispan.configuration.cache.ClusteringConfiguration.CACHE_SYNC;
import static org.infinispan.configuration.cache.ClusteringConfiguration.CACHE_TYPE;
import static org.infinispan.configuration.cache.ClusteringConfiguration.INVALIDATION_BATCH_SIZE;
import static org.infinispan.configuration.cache.ClusteringConfiguration.REMOTE_TIMEOUT;
import static org.infinispan.util.logging.Log.CONFIG;

import java.util.Arrays;
import java.util.concurrent.TimeUnit;

import org.infinispan.commons.configuration.Builder;
import org.infinispan.commons.configuration.Combine;
import org.infinispan.commons.configuration.attributes.AttributeSet;
import org.infinispan.configuration.global.GlobalConfiguration;
import org.infinispan.partitionhandling.PartitionHandling;

/**
 * Defines clustered characteristics of the cache.
 *
 * @author pmuir
 *
 */
public class ClusteringConfigurationBuilder extends AbstractConfigurationChildBuilder implements
      ClusteringConfigurationChildBuilder, Builder<ClusteringConfiguration> {
   private final HashConfigurationBuilder hashConfigurationBuilder;
   private final L1ConfigurationBuilder l1ConfigurationBuilder;
   private final StateTransferConfigurationBuilder stateTransferConfigurationBuilder;
   private final PartitionHandlingConfigurationBuilder partitionHandlingConfigurationBuilder;
   final AttributeSet attributes;

   ClusteringConfigurationBuilder(ConfigurationBuilder builder) {
      super(builder);
      this.attributes = ClusteringConfiguration.attributeDefinitionSet();
      this.hashConfigurationBuilder = new HashConfigurationBuilder(this);
      this.l1ConfigurationBuilder = new L1ConfigurationBuilder(this);
      this.stateTransferConfigurationBuilder = new StateTransferConfigurationBuilder(this);
      this.partitionHandlingConfigurationBuilder = new PartitionHandlingConfigurationBuilder(this);
   }

   @Override
   public AttributeSet attributes() {
      return attributes;
   }

   /**
    * Cache mode. See {@link CacheMode} for information on the various cache modes available.
    */
   public ClusteringConfigurationBuilder cacheMode(CacheMode cacheMode) {
      cacheType(cacheMode.cacheType());
      cacheSync(cacheMode.isSynchronous());
      return this;
   }

   public CacheMode cacheMode() {
      return CacheMode.of(attributes.attribute(CACHE_TYPE).get(), attributes.attribute(CACHE_SYNC).get());
   }

   public ClusteringConfigurationBuilder cacheType(CacheType type) {
      attributes.attribute(CACHE_TYPE).set(type);
      return this;
   }

   public ClusteringConfigurationBuilder cacheSync(boolean sync) {
      attributes.attribute(CACHE_SYNC).set(sync);
      return this;
   }

   /**
    * This is the timeout used to wait for an acknowledgment when making a remote call, after which
    * the call is aborted and an exception is thrown.
    */
   public ClusteringConfigurationBuilder remoteTimeout(long l) {
      attributes.attribute(REMOTE_TIMEOUT).set(l);
      return this;
   }

   /**
    * This is the timeout used to wait for an acknowledgment when making a remote call, after which
    * the call is aborted and an exception is thrown.
    */
   public ClusteringConfigurationBuilder remoteTimeout(long l, TimeUnit unit) {
      return remoteTimeout(unit.toMillis(l));
   }

   /**
    * For scattered cache, the threshold after which batched invalidations are sent
    */
   public ClusteringConfigurationBuilder invalidationBatchSize(int size) {
      attributes.attribute(INVALIDATION_BATCH_SIZE).set(size);
      return this;
   }

   /**
    * Used in scattered cache. Acquired bias allows reading data on non-owner, but slows
    * down further writes from other nodes.
    */
   public ClusteringConfigurationBuilder biasAcquisition(BiasAcquisition biasAcquisition) {
      attributes.attribute(BIAS_ACQUISITION).set(biasAcquisition);
      return this;
   }

   /**
    * Used in scattered cache. Specifies the duration (in Milliseconds) that acquired bias can be held; while the
    * reads will never be stale, tracking that information consumes memory on the primary owner.
    */
   public ClusteringConfigurationBuilder biasLifespan(long l, TimeUnit unit) {
      attributes.attribute(BIAS_LIFESPAN).set(unit.toMillis(l));
      return this;
   }

   /**
    * Configure hash sub element
    */
   @Override
   public HashConfigurationBuilder hash() {
      return hashConfigurationBuilder;
   }

   /**
    * This method allows configuration of the L1 cache for distributed
    * caches. L1 should be explicitly enabled by calling {@link L1ConfigurationBuilder#enable()}
    */
   @Override
   public L1ConfigurationBuilder l1() {
      return l1ConfigurationBuilder;
   }

   /**
    * Configure the {@code stateTransfer} sub element for distributed and replicated caches.
    * It doesn't have any effect on LOCAL or INVALIDATION-mode caches.
    */
   @Override
   public StateTransferConfigurationBuilder stateTransfer() {
      return stateTransferConfigurationBuilder;
   }

   @Override
   public PartitionHandlingConfigurationBuilder partitionHandling() {
      return partitionHandlingConfigurationBuilder;
   }

   @Override
   public void validate() {
      for (Builder<?> validatable : Arrays.asList(hashConfigurationBuilder, l1ConfigurationBuilder,
            stateTransferConfigurationBuilder, partitionHandlingConfigurationBuilder)) {
         validatable.validate();
      }
      if (cacheMode().isScattered()) {
         if (hash().numOwners() != 1 && hash().isNumOwnersSet()) {
            throw CONFIG.scatteredCacheNeedsSingleOwner();
         }
         hash().numOwners(1);
         org.infinispan.transaction.TransactionMode transactionMode = transaction().transactionMode();
         if (transactionMode != null && transactionMode.isTransactional()) {
            throw CONFIG.scatteredCacheIsNonTransactional();
         }

         if (attributes.attribute(BIAS_ACQUISITION).get() == BiasAcquisition.ON_READ)
            throw new UnsupportedOperationException("Not implemented yet");
      } else {
         if (attributes.attribute(INVALIDATION_BATCH_SIZE).isModified())
            throw CONFIG.invalidationBatchSizeAppliesOnNonScattered();

         if (attributes.attribute(BIAS_ACQUISITION).isModified() || attributes.attribute(BIAS_LIFESPAN).isModified())
            throw CONFIG.biasedReadsAppliesOnlyToScattered();

         if (hash().numOwners() == 1 && partitionHandling().whenSplit() != PartitionHandling.ALLOW_READ_WRITES)
            throw CONFIG.singleOwnerNotSetToAllowReadWrites();
      }
   }

   @Override
   public void validate(GlobalConfiguration globalConfig) {
      if (cacheMode().isClustered() && globalConfig.transport().transport() == null && !builder.template()) {
         throw CONFIG.missingTransportConfiguration();
      }

      for (ConfigurationChildBuilder validatable : Arrays
            .asList(hashConfigurationBuilder, l1ConfigurationBuilder, stateTransferConfigurationBuilder, partitionHandlingConfigurationBuilder)) {
         validatable.validate(globalConfig);
      }
   }

   @Override
   public
   ClusteringConfiguration create() {
      return new ClusteringConfiguration(attributes.protect(), hashConfigurationBuilder.create(),
            l1ConfigurationBuilder.create(), stateTransferConfigurationBuilder.create(), partitionHandlingConfigurationBuilder.create());
   }

   @Override
   public ClusteringConfigurationBuilder read(ClusteringConfiguration template, Combine combine) {
      attributes.read(template.attributes(), combine);
      hashConfigurationBuilder.read(template.hash(), combine);
      l1ConfigurationBuilder.read(template.l1(), combine);
      stateTransferConfigurationBuilder.read(template.stateTransfer(), combine);
      partitionHandlingConfigurationBuilder.read(template.partitionHandling(), combine);

      return this;
   }

   @Override
   public String toString() {
      return "ClusteringConfigurationBuilder [hashConfigurationBuilder=" + hashConfigurationBuilder +
            ", l1ConfigurationBuilder=" + l1ConfigurationBuilder +
            ", stateTransferConfigurationBuilder=" + stateTransferConfigurationBuilder +
            ", partitionHandlingConfigurationBuilder=" + partitionHandlingConfigurationBuilder +
            ", attributes=" + attributes + "]";
   }
}
