/**
 * Licensed to Jasig under one or more contributor license
 * agreements. See the NOTICE file distributed with this work
 * for additional information regarding copyright ownership.
 * Jasig licenses this file to you under the Apache License,
 * Version 2.0 (the "License"); you may not use this file
 * except in compliance with the License. You may obtain a
 * copy of the License at:
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.jasig.i18n.translate;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;

/**
 * Goal which updates a target language file with automatic translations of
 * any missing keys.
 *
 * @goal translate
 * 
 * @phase process-sources
 */
public class AutoTranslateMojo extends AbstractMojo {

    private BingAutoTranslationService translationService;

    private MessageFileService         messageFileService;

    /**
     * @parameter
     */
    private String                     clientId;

    /**
     * @parameter
     */
    private String                     clientSecret;

    /**
     * Messages directory location
     *
     * @parameter default-value="${basedir}/src/main/resources/properties/i18n/"
     */
    private String                     messagesDirectory;

    /**
     * @parameter default-value="Messages.properties"
     */
    private String                     mainMessagesFile;

    /**
     * @parameter
     */
    private final List<String>         languageKeys = Arrays.asList("de");

    /**
     * Message file pattern
     *
     * @parameter default-value="Messages_%s.properties"
     */
    private String                     messageFilePattern;

    @Override
    public void execute() throws MojoExecutionException {
        try {

            this.translationService = new BingAutoTranslationService();

            this.translationService.setClientId(this.clientId);
            this.translationService.setClientSecret(this.clientSecret);

            this.messageFileService = new MessageFileService();

            Set<String> mainKeys = null;

            getLog().info("Locating main messages file at " + getMainMessagesFile());
            final Resource resource = new FileSystemResource(getMainMessagesFile());
            mainKeys = this.messageFileService.getMessageKeysFromFile(resource);

            for (final String key : this.languageKeys) {

                final String fileName = String.format(this.messageFilePattern, key);
                final LanguageFile file = new LanguageFile(this.messagesDirectory, fileName, key);

                if (file.exists() && file.canRead()) {
                    final Resource resource2 = new FileSystemResource(file);

                    final Set<String> targetKeys = this.messageFileService.getMessageKeysFromFile(resource2);
                    final Set<String> missingKeys = new HashSet<String>();
                    missingKeys.addAll(mainKeys);
                    missingKeys.removeAll(targetKeys);

                    if (missingKeys.size() > 0) {
                        getLog().info("Translating message file at " + file.getCanonicalPath());

                        final Resource languageFile = new FileSystemResource(file);
                        final Map<String, String> targetMap = this.messageFileService.getMessageMapFromFile(languageFile);

                        final Map<String, String> mainMap = this.messageFileService.getMessageMapFromFile(resource);

                        final Map<String, String> updatedMap = this.translationService
                                .getAutoUpdatedTranslationMap(mainMap, targetMap, key);

                        this.messageFileService.updateMessageFile(languageFile, updatedMap);
                    }
                } else {

                    getLog().error("Message file for language '" + key + "' (" + key + ") cannot be located");

                }

            }
        } catch (final IOException ex) {
            getLog().error("Main messages file could not be located.");
            throw new MojoExecutionException(ex.getMessage());
        }

        catch (final Exception ex) {
            getLog().error(ex);
            throw new MojoExecutionException(ex.getMessage());
        }

    }

    protected File getMainMessagesFile() {
        return new File(this.messagesDirectory, this.mainMessagesFile);
    }

}
