/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb3.timerservice.naming;

import javax.ejb.TimerService;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.naming.Reference;

import org.jboss.util.naming.Util;

/**
 * Responsible for binding {@link TimerService} into JNDI at a internal JNDI name.
 * 
 * <p>
 *  The internal JNDI name is meant to be used by EJB3 internal implementation and isn't
 *  meant to be used by user deployments. User deployments are expected to use the 
 *  java:comp/TimerService JNDI name, as dictated by EJB3 spec. For further details of 
 *  accessing TimerService in user deployments, refer the EJB3.0/EJB3.1 spec 
 * </p>
 * <p>
 *  The default JNDI name to which the {@link TimerService} is bound is java:internal/TimerService.
 *  This can be overriden by using the {@link #TimerServiceBinder(String, Context)}
 * </p>
 * @author Jaikiran Pai
 * @version $Revision: $
 */
public class TimerServiceBinder
{
   /**
    * Naming context
    */
   private Context context;

   /**
    * The default internal JNDI name to which the {@link TimerService} will be bound
    */
   private String jndiName = "java:internal/TimerService";

   /**
    * Constructs {@link TimerServiceBinder} 
    * <p>
    *   Instantiates a new {@link InitialContext} to bind the {@link TimerService}  
    * </p>
    */
   public TimerServiceBinder()
   {
      try
      {
         this.context = new InitialContext();
      }
      catch (NamingException ne)
      {
         throw new RuntimeException(ne);
      }
   }

   /**
    * Constructs a {@link TimerServiceBinder}
    * 
    * @param jndiName The jndi name to which the {@link TimerService} should be bound
    * @param ctx The naming context to which the {@link TimerService} should be bound
    * 
    * @throws IllegalArgumentException If either of <code>jndiName</code> or <code>ctx</code> is null
    */
   public TimerServiceBinder(String jndiName, Context ctx)
   {
      if (jndiName == null)
      {
         throw new IllegalArgumentException("JNDI name for timerservice cannot be null");
      }
      if (ctx == null)
      {
         throw new IllegalArgumentException("Context for timerservice binding cannot be null");
      }

      this.jndiName = jndiName;
      this.context = ctx;
   }

   /**
    * Constructs a {@link TimerServiceBinder}
    * 
    * @param jndiName The jndi name to which the {@link TimerService} is to be bound
    */
   public TimerServiceBinder(String jndiName)
   {
      this();
      if (jndiName == null)
      {
         throw new IllegalArgumentException("JNDI name for timerservice cannot be null");
      }
      this.jndiName = jndiName;
   }

   /**
    * Constructs a {@link TimerServiceBinder}
    * 
    * @param ctx The naming context to which the {@link TimerService} has to be bound
    */
   public TimerServiceBinder(Context ctx)
   {
      if (ctx == null)
      {
         throw new IllegalArgumentException("Context for timerservice binding cannot be null");
      }
      this.context = ctx;
   }

   /**
    * Returns the {@link Context} to which the {@link TimerService} is bound
    * @return
    */
   public Context getContext()
   {
      return this.context;
   }

   /**
    * Returns the jndi name to which the {@link TimerService} will be bound
    * by this {@link TimerServiceBinder}
    * @return
    */
   public String getTimerServiceJNDIName()
   {
      return this.jndiName;
   }

   /**
    * Binds the {@link TimerService} to {@link #jndiName} under {@link #context}
    * 
    * @throws NamingException If there is any exception during the bind operation
    */
   public void start() throws NamingException
   {
      Reference ref = new Reference(TimerService.class.getName(), TimerServiceObjectFactory.class.getName(), null);
      Util.bind(this.context, jndiName, ref);
   }

   /**
    * Unbinds {@link TimerService} from {@link #jndiName} under {@link #context}
    * @throws NamingException If there is any exception during the bind operation 
    */
   public void stop() throws NamingException
   {
      Util.unbind(this.context, jndiName);
   }
}
