/*
 * Decompiled with CFR 0.152.
 */
package jenkins.util.io;

import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import hudson.Functions;
import hudson.Util;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.attribute.DosFileAttributeView;
import java.nio.file.attribute.PosixFileAttributes;
import java.nio.file.attribute.PosixFilePermission;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import jenkins.util.io.CompositeIOException;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.NoExternalUse;

@Restricted(value={NoExternalUse.class})
public class PathRemover {
    private final RetryStrategy retryStrategy;
    private final PathChecker pathChecker;

    public static PathRemover newSimpleRemover() {
        return new PathRemover(ignored -> false, PathChecker.ALLOW_ALL);
    }

    public static PathRemover newRemoverWithStrategy(@NonNull RetryStrategy retryStrategy) {
        return new PathRemover(retryStrategy, PathChecker.ALLOW_ALL);
    }

    public static PathRemover newFilteredRobustRemover(@NonNull PathChecker pathChecker, int maxRetries, boolean gcAfterFailedRemove, long waitBetweenRetries) {
        return new PathRemover(new PausingGCRetryStrategy(Math.max(maxRetries, 0), gcAfterFailedRemove, waitBetweenRetries), pathChecker);
    }

    private PathRemover(@NonNull RetryStrategy retryStrategy, @NonNull PathChecker pathChecker) {
        this.retryStrategy = retryStrategy;
        this.pathChecker = pathChecker;
    }

    public void forceRemoveFile(@NonNull Path path) throws IOException {
        int retryAttempts = 0;
        Optional<IOException> maybeError;
        while (!(maybeError = this.tryRemoveFile(path)).isEmpty()) {
            if (!this.retryStrategy.shouldRetry(retryAttempts)) {
                IOException error = maybeError.get();
                throw new IOException(this.retryStrategy.failureMessage(path, retryAttempts), error);
            }
            ++retryAttempts;
        }
        return;
    }

    public void forceRemoveDirectoryContents(@NonNull Path path) throws IOException {
        int retryAttempt = 0;
        List<IOException> errors;
        while (!(errors = this.tryRemoveDirectoryContents(path)).isEmpty()) {
            if (!this.retryStrategy.shouldRetry(retryAttempt)) {
                throw new CompositeIOException(this.retryStrategy.failureMessage(path, retryAttempt), errors);
            }
            ++retryAttempt;
        }
        return;
    }

    public void forceRemoveRecursive(@NonNull Path path) throws IOException {
        int retryAttempt = 0;
        List<IOException> errors;
        while (!(errors = this.tryRemoveRecursive(path)).isEmpty()) {
            if (!this.retryStrategy.shouldRetry(retryAttempt)) {
                throw new CompositeIOException(this.retryStrategy.failureMessage(path, retryAttempt), errors);
            }
            ++retryAttempt;
        }
        return;
    }

    private Optional<IOException> tryRemoveFile(@NonNull Path path) {
        try {
            this.removeOrMakeRemovableThenRemove(path.normalize());
            return Optional.empty();
        }
        catch (IOException e) {
            return Optional.of(e);
        }
    }

    private List<IOException> tryRemoveRecursive(@NonNull Path path) {
        Path normalized = path.normalize();
        ArrayList<IOException> accumulatedErrors = Util.isSymlink(normalized) ? new ArrayList() : this.tryRemoveDirectoryContents(normalized);
        this.tryRemoveFile(normalized).ifPresent(accumulatedErrors::add);
        return accumulatedErrors;
    }

    @SuppressFBWarnings(value={"RCN_REDUNDANT_NULLCHECK_WOULD_HAVE_BEEN_A_NPE"}, justification="https://github.com/spotbugs/spotbugs/issues/756")
    private List<IOException> tryRemoveDirectoryContents(@NonNull Path path) {
        Path normalized = path.normalize();
        ArrayList<IOException> accumulatedErrors = new ArrayList<IOException>();
        if (!Files.isDirectory(normalized, new LinkOption[0])) {
            return accumulatedErrors;
        }
        try (DirectoryStream<Path> children = Files.newDirectoryStream(normalized);){
            for (Path child : children) {
                accumulatedErrors.addAll(this.tryRemoveRecursive(child));
            }
        }
        catch (IOException e) {
            accumulatedErrors.add(e);
        }
        return accumulatedErrors;
    }

    @SuppressFBWarnings(value={"RCN_REDUNDANT_NULLCHECK_WOULD_HAVE_BEEN_A_NPE"}, justification="https://github.com/spotbugs/spotbugs/issues/756")
    private void removeOrMakeRemovableThenRemove(@NonNull Path path) throws IOException {
        this.pathChecker.check(path);
        try {
            Files.deleteIfExists(path);
        }
        catch (IOException e) {
            PathRemover.makeRemovable(path);
            try {
                Files.deleteIfExists(path);
            }
            catch (IOException e2) {
                if (Files.isDirectory(path, new LinkOption[0])) {
                    List entries;
                    try (Stream<Path> children = Files.list(path);){
                        entries = children.map(Path::toString).collect(Collectors.toList());
                    }
                    throw new CompositeIOException("Unable to remove directory " + path + " with directory contents: " + entries, e, e2);
                }
                throw new CompositeIOException("Unable to remove file " + path, e, e2);
            }
        }
    }

    private static void makeRemovable(@NonNull Path path) throws IOException {
        Optional<Path> maybeParent;
        if (!Files.isWritable(path)) {
            PathRemover.makeWritable(path);
        }
        if ((maybeParent = Optional.ofNullable(path.getParent()).map(Path::normalize).filter(p -> !Files.isWritable(p))).isPresent()) {
            PathRemover.makeWritable(maybeParent.get());
        }
    }

    private static void makeWritable(@NonNull Path path) throws IOException {
        if (!Functions.isWindows()) {
            try {
                PosixFileAttributes attrs = Files.readAttributes(path, PosixFileAttributes.class, new LinkOption[0]);
                Set<PosixFilePermission> newPermissions = attrs.permissions();
                newPermissions.add(PosixFilePermission.OWNER_WRITE);
                Files.setPosixFilePermissions(path, newPermissions);
            }
            catch (NoSuchFileException ignored) {
                return;
            }
            catch (UnsupportedOperationException ignored) {}
        } else {
            DosFileAttributeView dos = Files.getFileAttributeView(path, DosFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
            if (dos != null) {
                dos.setReadOnly(false);
            }
        }
        path.toFile().setWritable(true);
    }

    private static class PausingGCRetryStrategy
    implements RetryStrategy {
        private final int maxRetries;
        private final boolean gcAfterFailedRemove;
        private final long waitBetweenRetries;
        private final ThreadLocal<Boolean> interrupted = ThreadLocal.withInitial(() -> false);

        private PausingGCRetryStrategy(int maxRetries, boolean gcAfterFailedRemove, long waitBetweenRetries) {
            this.maxRetries = maxRetries;
            this.gcAfterFailedRemove = gcAfterFailedRemove;
            this.waitBetweenRetries = waitBetweenRetries;
        }

        @SuppressFBWarnings(value={"DM_GC"}, justification="Garbage collection happens only when GC_AFTER_FAILED_DELETE is true. It's an experimental feature in Jenkins.")
        private void gcIfEnabled() {
            if (this.gcAfterFailedRemove) {
                System.gc();
            }
        }

        @Override
        public boolean shouldRetry(int retriesAttempted) {
            long delayMillis;
            if (retriesAttempted >= this.maxRetries) {
                return false;
            }
            this.gcIfEnabled();
            long l = delayMillis = this.waitBetweenRetries >= 0L ? this.waitBetweenRetries : (long)(-(retriesAttempted + 1)) * this.waitBetweenRetries;
            if (delayMillis <= 0L) {
                return !Thread.interrupted();
            }
            try {
                Thread.sleep(delayMillis);
                return true;
            }
            catch (InterruptedException e) {
                this.interrupted.set(true);
                return false;
            }
        }

        @Override
        public String failureMessage(@NonNull Path fileToRemove, int retryCount) {
            StringBuilder sb = new StringBuilder();
            sb.append("Unable to delete '");
            sb.append(fileToRemove);
            sb.append("'. Tried ");
            sb.append(retryCount + 1);
            sb.append(" time");
            if (retryCount != 1) {
                sb.append('s');
            }
            if (this.maxRetries > 0) {
                sb.append(" (of a maximum of ");
                sb.append(this.maxRetries + 1);
                sb.append(')');
                if (this.gcAfterFailedRemove) {
                    sb.append(" garbage-collecting");
                }
                if (this.waitBetweenRetries != 0L && this.gcAfterFailedRemove) {
                    sb.append(" and");
                }
                if (this.waitBetweenRetries != 0L) {
                    sb.append(" waiting ");
                    sb.append(Util.getTimeSpanString(Math.abs(this.waitBetweenRetries)));
                    if (this.waitBetweenRetries < 0L) {
                        sb.append("-");
                        sb.append(Util.getTimeSpanString(Math.abs(this.waitBetweenRetries) * (long)(this.maxRetries + 1)));
                    }
                }
                if (this.waitBetweenRetries != 0L || this.gcAfterFailedRemove) {
                    sb.append(" between attempts");
                }
            }
            if (this.interrupted.get().booleanValue()) {
                sb.append(". The delete operation was interrupted before it completed successfully");
            }
            sb.append('.');
            this.interrupted.set(false);
            return sb.toString();
        }
    }

    @Restricted(value={NoExternalUse.class})
    @FunctionalInterface
    public static interface RetryStrategy {
        public boolean shouldRetry(int var1);

        default public String failureMessage(@NonNull Path fileToRemove, int retryCount) {
            StringBuilder sb = new StringBuilder().append("Unable to delete '").append(fileToRemove).append("'. Tried ").append(retryCount).append(" time");
            if (retryCount != 1) {
                sb.append('s');
            }
            sb.append('.');
            return sb.toString();
        }
    }

    @Restricted(value={NoExternalUse.class})
    @FunctionalInterface
    public static interface PathChecker {
        public static final PathChecker ALLOW_ALL = path -> {};

        public void check(@NonNull Path var1) throws SecurityException;
    }
}

