/*
 * Decompiled with CFR 0.152.
 */
package jenkins.model;

import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import hudson.BulkChange;
import hudson.Extension;
import hudson.ExtensionList;
import hudson.Util;
import hudson.XmlFile;
import hudson.model.Computer;
import hudson.model.Messages;
import hudson.model.Node;
import hudson.model.PersistenceRoot;
import hudson.model.Queue;
import hudson.model.Saveable;
import hudson.model.listeners.SaveableListener;
import hudson.slaves.EphemeralNode;
import hudson.slaves.OfflineCause;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import jenkins.model.Jenkins;
import jenkins.model.NodeListener;
import jenkins.util.SystemProperties;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.NoExternalUse;

@Restricted(value={NoExternalUse.class})
public class Nodes
implements PersistenceRoot {
    private static final Logger LOGGER = Logger.getLogger(Nodes.class.getName());
    @Restricted(value={NoExternalUse.class})
    private static final boolean ENFORCE_NAME_RESTRICTIONS = SystemProperties.getBoolean(Nodes.class.getName() + ".enforceNameRestrictions", true);
    @NonNull
    private final Jenkins jenkins;
    private final ConcurrentMap<String, Node> nodes = new ConcurrentSkipListMap<String, Node>();

    Nodes(@NonNull Jenkins jenkins) {
        this.jenkins = jenkins;
    }

    @NonNull
    public List<Node> getNodes() {
        return new ArrayList<Node>(this.nodes.values());
    }

    public void setNodes(final @NonNull Collection<? extends Node> nodes) throws IOException {
        final HashSet toRemove = new HashSet();
        Queue.withLock(new Runnable(){

            @Override
            public void run() {
                toRemove.addAll(Nodes.this.nodes.keySet());
                for (Node n : nodes) {
                    String name = n.getNodeName();
                    toRemove.remove(name);
                    Nodes.this.nodes.put(name, n);
                    n.onLoad(Nodes.this, name);
                }
                Nodes.this.nodes.keySet().removeAll(toRemove);
                Nodes.this.jenkins.updateComputerList();
                Nodes.this.jenkins.trimLabels();
            }
        });
        this.save();
        for (String name : toRemove) {
            LOGGER.fine(() -> "deleting " + new File(this.getRootDir(), name));
            Util.deleteRecursive(new File(this.getRootDir(), name));
        }
    }

    public void addNode(final @NonNull Node node) throws IOException {
        Node old;
        if (ENFORCE_NAME_RESTRICTIONS) {
            Jenkins.checkGoodName(node.getNodeName());
        }
        if (node != (old = this.nodes.put(node.getNodeName(), node))) {
            node.onLoad(this, node.getNodeName());
            this.jenkins.updateNewComputer(node);
            this.jenkins.trimLabels(node, old);
            try {
                node.save();
            }
            catch (IOException | RuntimeException e) {
                Queue.withLock(new Runnable(){

                    @Override
                    public void run() {
                        Nodes.this.nodes.compute(node.getNodeName(), (ignoredNodeName, ignoredNode) -> old);
                        Nodes.this.jenkins.updateComputerList();
                        Nodes.this.jenkins.trimLabels(node, old);
                    }
                });
                throw e;
            }
            if (old != null) {
                NodeListener.fireOnUpdated(old, node);
            } else {
                NodeListener.fireOnCreated(node);
            }
        }
    }

    public XmlFile getConfigFile(Node node) {
        return this.getConfigFile(node.getRootDir());
    }

    public XmlFile getConfigFile(File dir) {
        return new XmlFile(Jenkins.XSTREAM, new File(dir, "config.xml"));
    }

    public XmlFile getConfigFile(String nodeName) {
        return new XmlFile(Jenkins.XSTREAM, new File(this.getRootDir(), nodeName + File.separator + "config.xml"));
    }

    public boolean updateNode(final @NonNull Node node) throws IOException {
        boolean exists;
        try {
            exists = Queue.withLock(new Callable<Boolean>(){

                @Override
                public Boolean call() throws Exception {
                    if (node == Nodes.this.nodes.get(node.getNodeName())) {
                        Nodes.this.jenkins.trimLabels(node);
                        return true;
                    }
                    return false;
                }
            });
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            exists = false;
        }
        if (exists) {
            node.save();
            return true;
        }
        return false;
    }

    public boolean replaceNode(final Node oldOne, final @NonNull Node newOne) throws IOException {
        if (ENFORCE_NAME_RESTRICTIONS) {
            Jenkins.checkGoodName(newOne.getNodeName());
        }
        if (oldOne == this.nodes.get(oldOne.getNodeName())) {
            Queue.withLock(new Runnable(){

                @Override
                public void run() {
                    Nodes.this.nodes.remove(oldOne.getNodeName());
                    Nodes.this.nodes.put(newOne.getNodeName(), newOne);
                    newOne.onLoad(Nodes.this, newOne.getNodeName());
                }
            });
            this.updateNode(newOne);
            if (!newOne.getNodeName().equals(oldOne.getNodeName())) {
                LOGGER.fine(() -> "deleting " + new File(this.getRootDir(), oldOne.getNodeName()));
                Util.deleteRecursive(new File(this.getRootDir(), oldOne.getNodeName()));
            }
            Queue.withLock(() -> {
                this.jenkins.updateComputerList();
                this.jenkins.trimLabels(oldOne, newOne);
            });
            NodeListener.fireOnUpdated(oldOne, newOne);
            return true;
        }
        return false;
    }

    public void removeNode(final @NonNull Node node) throws IOException {
        if (node == this.nodes.get(node.getNodeName())) {
            final AtomicBoolean match = new AtomicBoolean();
            Queue.withLock(new Runnable(){

                @Override
                public void run() {
                    Computer c = node.toComputer();
                    if (c != null) {
                        c.recordTermination();
                        c.disconnect(OfflineCause.create(Messages._Hudson_NodeBeingRemoved()));
                    }
                    match.set(node == Nodes.this.nodes.remove(node.getNodeName()));
                }
            });
            LOGGER.fine(() -> "deleting " + new File(this.getRootDir(), node.getNodeName()));
            Util.deleteRecursive(new File(this.getRootDir(), node.getNodeName()));
            if (match.get()) {
                this.jenkins.updateComputerList();
                this.jenkins.trimLabels(node);
            }
            NodeListener.fireOnDeleted(node);
            SaveableListener.fireOnDeleted(node, this.getConfigFile(node));
        }
    }

    @Override
    public void save() throws IOException {
        if (BulkChange.contains(this)) {
            return;
        }
        for (Node n : this.nodes.values()) {
            if (n instanceof EphemeralNode) continue;
            XmlFile xmlFile = this.getConfigFile(n);
            LOGGER.fine(() -> "saving " + xmlFile);
            xmlFile.write(n);
            SaveableListener.fireOnChange(this, xmlFile);
        }
    }

    @CheckForNull
    public Node getNode(String name) {
        return name == null ? null : (Node)this.nodes.get(name);
    }

    public void load() throws IOException {
        File nodesDir = this.getRootDir();
        File[] subdirs = nodesDir.listFiles(File::isDirectory);
        final TreeMap<String, Node> newNodes = new TreeMap<String, Node>();
        if (subdirs != null) {
            for (File subdir : subdirs) {
                try {
                    this.load(subdir, newNodes);
                }
                catch (IOException e) {
                    LOGGER.log(Level.WARNING, "could not load " + subdir, e);
                }
            }
        }
        Queue.withLock(new Runnable(){

            @Override
            public void run() {
                newNodes.entrySet().removeIf(stringNodeEntry -> ExtensionList.lookup(NodeListener.class).stream().anyMatch(nodeListener -> {
                    if (!nodeListener.allowLoad((Node)stringNodeEntry.getValue())) {
                        LOGGER.log(Level.FINE, () -> "Loading of node " + (String)stringNodeEntry.getKey() + " vetoed by " + nodeListener);
                        return true;
                    }
                    return false;
                }));
                Nodes.this.nodes.entrySet().removeIf(stringNodeEntry -> !(stringNodeEntry.getValue() instanceof EphemeralNode));
                Nodes.this.nodes.putAll(newNodes);
                Nodes.this.jenkins.updateComputerList();
                Nodes.this.jenkins.trimLabels();
            }
        });
    }

    @CheckForNull
    public Node getOrLoad(String name) {
        Node node = this.getNode(name);
        LOGGER.fine(() -> "already loaded? " + node);
        return node == null ? this.load(name) : node;
    }

    @CheckForNull
    public Node load(String name) {
        try {
            XmlFile xmlFile = this.getConfigFile(name);
            if (xmlFile.exists()) {
                Node n = (Node)xmlFile.read();
                this.nodes.put(n.getNodeName(), n);
                n.onLoad(this, n.getNodeName());
                this.jenkins.updateNewComputer(n);
                this.jenkins.trimLabels(n);
                LOGGER.finer(() -> "loading " + xmlFile);
                return n;
            }
            LOGGER.fine(() -> "no such file " + xmlFile);
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "could not load " + name, e);
        }
        return null;
    }

    private Node load(File dir, Map<String, Node> nodesCollector) throws IOException {
        Node n = (Node)this.getConfigFile(dir).read();
        if (n != null) {
            nodesCollector.put(n.getNodeName(), n);
            n.onLoad(this, n.getNodeName());
        }
        return n;
    }

    public void load(File dir) throws IOException {
        Node n = this.load(dir, this.nodes);
        this.jenkins.updateComputerList();
        this.jenkins.trimLabels(n);
    }

    public void unload(Node node) {
        if (node == this.nodes.get(node.getNodeName())) {
            AtomicBoolean match = new AtomicBoolean();
            Queue.withLock(() -> match.set(node == this.nodes.remove(node.getNodeName())));
            if (match.get()) {
                this.jenkins.updateComputerList();
                this.jenkins.trimLabels(node);
            }
        }
    }

    public boolean isLegacy() {
        return !this.getRootDir().isDirectory();
    }

    @Override
    public File getRootDir() {
        return new File(this.jenkins.getRootDir(), "nodes");
    }

    public File getRootDirFor(Node node) {
        return this.getRootDirFor(node.getNodeName());
    }

    private File getRootDirFor(String name) {
        return new File(this.getRootDir(), name);
    }

    @Extension
    public static class ScheduleMaintenanceAfterSavingNode
    extends SaveableListener {
        @Override
        public void onChange(Saveable o, XmlFile file) {
            if (o instanceof Node) {
                Jenkins.get().getQueue().scheduleMaintenance();
            }
        }
    }
}

