package javaposse.jobdsl.dsl

abstract class Item extends AbstractContext {
    final String name

    private final List<Closure> configureBlocks = []

    protected Item(JobManagement jobManagement, String name) {
        super(jobManagement)
        this.name = name
    }

    /**
     * Sets a description for the item.
     */
    void description(String descriptionString) {
        configure { Node project ->
            project /  methodMissing('description', descriptionString)
        }
    }

    /**
     * Sets the name to display instead of the actual name.
     */
    void displayName(String displayName) {
        configure {
            it / methodMissing('displayName', displayName)
        }
    }

    /**
     * Allows direct manipulation of the generated XML.
     *
     * @see <a href="https://github.com/jenkinsci/job-dsl-plugin/wiki/The-Configure-Block">The Configure Block</a>
     */
    void configure(Closure configureBlock) {
        configureBlocks << configureBlock
    }

    /**
     * Postpone all xml processing until someone actually asks for the xml. That lets us execute everything in order,
     * even if the user didn't specify them in order.
     */
    String getXml() {
        Writer xmlOutput = new StringWriter()
        XmlNodePrinter xmlNodePrinter = new XmlNodePrinter(new PrintWriter(xmlOutput), '    ')
        xmlNodePrinter.with {
            preserveWhitespace = true
            expandEmptyElements = true
            quote = "'" // Use single quote for attributes
        }
        xmlNodePrinter.print(node)

        xmlOutput.toString()
    }

    Map getProperties() {
        // see JENKINS-22708
        throw new UnsupportedOperationException()
    }

    Node getNode() {
        Node node = nodeTemplate
        ContextHelper.executeConfigureBlocks(node, configureBlocks)
        node
    }

    protected Node getNodeTemplate() {
        new XmlParser().parse(this.class.getResourceAsStream("${this.class.simpleName}-template.xml"))
    }
}
