/*
 * Copyright 2010-2018 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package org.jetbrains.kotlin.backend.common.lower

import com.intellij.util.containers.addIfNotNull
import org.jetbrains.kotlin.backend.common.BodyLoweringPass
import org.jetbrains.kotlin.backend.common.CommonBackendContext
import org.jetbrains.kotlin.backend.common.DeclarationTransformer
import org.jetbrains.kotlin.backend.common.LoweringContext
import org.jetbrains.kotlin.ir.IrStatement
import org.jetbrains.kotlin.ir.declarations.*
import org.jetbrains.kotlin.ir.expressions.IrBody
import org.jetbrains.kotlin.ir.expressions.impl.IrCompositeImpl
import org.jetbrains.kotlin.ir.util.isEffectivelyExternal
import org.jetbrains.kotlin.ir.visitors.*

/**
 * Moves fields and accessors out from its property.
 */
class PropertiesLowering(@Suppress("unused") context: LoweringContext) : DeclarationTransformer {
    override val withLocalDeclarations: Boolean get() = true

    override fun transformFlat(declaration: IrDeclaration): List<IrDeclaration>? {
        when (declaration) {
            is IrSimpleFunction -> {
                declaration.correspondingPropertySymbol?.owner?.let { property ->
                    if (!property.isEffectivelyExternal()) {
                        return listOf(declaration)
                    }
                }
            }
            is IrField -> {
                declaration.correspondingPropertySymbol?.owner?.let { property ->
                    if (!property.isEffectivelyExternal()) {
                        return listOf(declaration)
                    }
                }
            }
            is IrProperty -> {
                if (!declaration.isEffectivelyExternal()) {
                    return listOf()
                }
            }
        }

        return null
    }
}

class LocalDelegatedPropertiesLowering(private val context: CommonBackendContext) : IrElementTransformerVoid(), BodyLoweringPass {

    override fun lower(irBody: IrBody, container: IrDeclaration) {
        irBody.accept(this, null)
    }

    override fun visitLocalDelegatedProperty(declaration: IrLocalDelegatedProperty): IrStatement {
        declaration.transformChildrenVoid(this)

        return IrCompositeImpl(
            startOffset = declaration.startOffset,
            endOffset = declaration.endOffset,
            type = context.irBuiltIns.unitType,
        ).apply {
            statements.add(declaration.getter)
            statements.addIfNotNull(declaration.setter)
            statements.addIfNotNull(declaration.delegate)
        }
    }
}
