/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package org.jetbrains.kotlin.fir.analysis.jvm.checkers.declaration

import org.jetbrains.kotlin.descriptors.ClassKind
import org.jetbrains.kotlin.descriptors.Visibilities
import org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import org.jetbrains.kotlin.diagnostics.reportOn
import org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import org.jetbrains.kotlin.fir.analysis.checkers.classKind
import org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import org.jetbrains.kotlin.fir.analysis.checkers.declaration.FirFunctionChecker
import org.jetbrains.kotlin.fir.resolve.getContainingClassSymbol
import org.jetbrains.kotlin.fir.analysis.diagnostics.jvm.FirJvmErrors
import org.jetbrains.kotlin.fir.declarations.*
import org.jetbrains.kotlin.fir.declarations.utils.*
import org.jetbrains.kotlin.name.JvmStandardClassIds.JVM_OVERLOADS_CLASS_ID

object FirOverloadsChecker : FirFunctionChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirFunction) {
        val session = context.session
        val annotation = declaration.getAnnotationByClassId(JVM_OVERLOADS_CLASS_ID, session) ?: return

        val ownerOfParametersWithDefaultValues = declaration.symbol.takeIf { !it.isActual }
            ?: declaration.symbol.getSingleMatchedExpectForActualOrNull()
            ?: return

        val containingDeclaration = declaration.getContainingClassSymbol()
        when {
            containingDeclaration?.classKind == ClassKind.INTERFACE ->
                reporter.reportOn(annotation.source, FirJvmErrors.OVERLOADS_INTERFACE)
            declaration.isAbstract ->
                reporter.reportOn(annotation.source, FirJvmErrors.OVERLOADS_ABSTRACT)
            (declaration is FirNamedFunction && declaration.status.visibility == Visibilities.Local) ||
                    context.containingDeclarations.any { it.isLocalClassLike } ->
                reporter.reportOn(annotation.source, FirJvmErrors.OVERLOADS_LOCAL)
            declaration is FirConstructor && containingDeclaration?.classKind == ClassKind.ANNOTATION_CLASS ->
                reporter.reportOn(annotation.source, FirJvmErrors.OVERLOADS_ANNOTATION_CLASS_CONSTRUCTOR_ERROR)
            !declaration.visibility.isPublicAPI && declaration.visibility != Visibilities.Internal ->
                reporter.reportOn(annotation.source, FirJvmErrors.OVERLOADS_PRIVATE)
            ownerOfParametersWithDefaultValues.valueParameterSymbols.none { it.hasDefaultValue } ->
                reporter.reportOn(annotation.source, FirJvmErrors.OVERLOADS_WITHOUT_DEFAULT_ARGUMENTS)
        }
    }
}
