package org.jfrog.jade.plugins.common.ant;

/*
 * Copyright 2005-2006 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.maven.artifact.Artifact;
import org.apache.maven.project.MavenProject;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.types.Path;
import org.jfrog.jade.plugins.common.injectable.MvnInjectable;

import java.io.File;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * @author Fred Simon
 */
public class Maven2AntManagerDefaultImpl implements Maven2AntManager {

    /**
     * The actual map between Maven and Ant projects.
     */
    private Map<MavenProject, Project> mvn2antMap = new HashMap<MavenProject, Project>();

    public Project getAntProject(MvnInjectable injectable) {
        if (injectable == null) {
            throw new IllegalArgumentException("Injectable cannot be null.");
        }
        MavenProject mvnProject = injectable.getProject();
        if (mvnProject == null) {
            throw new IllegalArgumentException("Maven project cannot be null.");
        }

        Project result = mvn2antMap.get(injectable.getProject());
        if (result == null) {
            result = newAntProject(injectable);
            mvn2antMap.put(mvnProject, result);
        }
        return result;
    }

    /**
     * Create a new Ant project from the mvnInjectable.
     * This method can be overrriden if needed by sub class for fine tuning of maven 2 ant.
     *
     * @param injectable
     * @return a new Ant project
     */
    protected Project newAntProject(MvnInjectable injectable) {
        Project antProject = new Project();
        antProject.addBuildListener(new AntBuildListener(injectable.getLog()));
        MavenProject mvnProject = injectable.getProject();
        if (mvnProject == null) {
            throw new IllegalArgumentException("Maven project cannot be null.");
        }
        antProject.setBaseDir(mvnProject.getBasedir());
        antProject.init();
        return antProject;
    }

    public void fillPathFromPaths(Path path, Collection<String> elements) {
        if (elements == null || elements.isEmpty())
            return;

        for (String location : elements) {
            path.createPathElement().setPath(location);
        }
    }

    public void fillPathFromLocations(Path path, Collection<File> elements) {
        if (elements == null || elements.isEmpty())
            return;

        for (File location : elements) {
            path.createPathElement().setLocation(location);
        }
    }

    public void fillPathFromArtifacts(Path path, Collection<Artifact> elements) {
        if (elements == null || elements.isEmpty())
            return;

        for (Artifact artifact : elements) {
            File file = artifact.getFile();
            if (file != null)
                path.createPathElement().setLocation(file);
        }
    }
}
