/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.truststore;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Stream;
import org.jboss.logging.Logger;
import org.keycloak.common.util.KeystoreUtil;

public class TruststoreBuilder {
    public static final String SYSTEM_TRUSTSTORE_KEY = "javax.net.ssl.trustStore";
    public static final String SYSTEM_TRUSTSTORE_PASSWORD_KEY = "javax.net.ssl.trustStorePassword";
    public static final String SYSTEM_TRUSTSTORE_TYPE_KEY = "javax.net.ssl.trustStoreType";
    private static final String CERT_PROTECTION_ALGORITHM_KEY = "keystore.pkcs12.certProtectionAlgorithm";
    public static final String DUMMY_PASSWORD = "keycloakchangeit";
    static final String PKCS12 = "PKCS12";
    private static final Logger LOGGER = Logger.getLogger(TruststoreBuilder.class);

    public static void setSystemTruststore(String[] truststores, boolean trustStoreIncludeDefault, String dataDir) {
        KeyStore truststore = TruststoreBuilder.createMergedTruststore(truststores, trustStoreIncludeDefault);
        File file = TruststoreBuilder.saveTruststore(truststore, dataDir, DUMMY_PASSWORD.toCharArray());
        System.setProperty(SYSTEM_TRUSTSTORE_KEY, file.getAbsolutePath());
        System.setProperty(SYSTEM_TRUSTSTORE_TYPE_KEY, PKCS12);
        System.setProperty(SYSTEM_TRUSTSTORE_PASSWORD_KEY, DUMMY_PASSWORD);
    }

    static File saveTruststore(KeyStore truststore, String dataDir, char[] password) {
        File file = new File(dataDir, "keycloak-truststore.p12");
        file.getParentFile().mkdirs();
        try (FileOutputStream fos = new FileOutputStream(file);){
            String oldValue = System.setProperty(CERT_PROTECTION_ALGORITHM_KEY, "NONE");
            truststore.store(fos, password);
            if (oldValue != null) {
                System.setProperty(CERT_PROTECTION_ALGORITHM_KEY, oldValue);
            } else {
                System.getProperties().remove(CERT_PROTECTION_ALGORITHM_KEY);
            }
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to save truststore: " + file.getAbsolutePath(), e);
        }
        return file;
    }

    static KeyStore createMergedTruststore(String[] truststores, boolean trustStoreIncludeDefault) {
        KeyStore truststore = TruststoreBuilder.createPkcs12KeyStore();
        if (trustStoreIncludeDefault) {
            TruststoreBuilder.includeDefaultTruststore(truststore);
        }
        ArrayList<String> discoveredFiles = new ArrayList<String>();
        TruststoreBuilder.mergeFiles(truststores, truststore, true, discoveredFiles);
        if (!discoveredFiles.isEmpty()) {
            LOGGER.infof("Found the following truststore files under directories specified in the truststore paths %s", discoveredFiles);
        }
        return truststore;
    }

    private static void mergeFiles(String[] truststores, KeyStore truststore, boolean topLevel, List<String> discoveredFiles) {
        for (String file : truststores) {
            File f = new File(file);
            if (f.isDirectory()) {
                TruststoreBuilder.mergeFiles((String[])Stream.of(f.listFiles()).map(File::getAbsolutePath).toArray(String[]::new), truststore, false, discoveredFiles);
                continue;
            }
            if (file.endsWith(".p12") || file.endsWith(".pfx")) {
                TruststoreBuilder.mergeTrustStore(truststore, file, TruststoreBuilder.loadStore(file, PKCS12, null));
                if (topLevel) continue;
                discoveredFiles.add(f.getAbsolutePath());
                continue;
            }
            if (!TruststoreBuilder.mergePemFile(truststore, file, topLevel) || topLevel) continue;
            discoveredFiles.add(f.getAbsolutePath());
        }
    }

    static KeyStore createPkcs12KeyStore() {
        try {
            KeyStore truststore = KeyStore.getInstance(PKCS12);
            truststore.load(null, null);
            return truststore;
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to initialize truststore: cannot create a PKCS12 keystore", e);
        }
    }

    static void includeDefaultTruststore(KeyStore truststore) {
        String originalTruststoreKey = "javax.net.ssl.trustStore.orig";
        String originalTruststoreTypeKey = "javax.net.ssl.trustStoreType.orig";
        String originalTruststorePasswordKey = "javax.net.ssl.trustStorePassword.orig";
        String trustStorePath = System.getProperty(originalTruststoreKey);
        String type = PKCS12;
        String password = null;
        File defaultTrustStore = null;
        if (trustStorePath == null) {
            trustStorePath = System.getProperty(SYSTEM_TRUSTSTORE_KEY);
            if (trustStorePath == null) {
                defaultTrustStore = TruststoreBuilder.getJRETruststore();
            } else {
                type = System.getProperty(SYSTEM_TRUSTSTORE_TYPE_KEY, KeyStore.getDefaultType());
                password = System.getProperty(SYSTEM_TRUSTSTORE_PASSWORD_KEY);
                System.setProperty(originalTruststoreKey, trustStorePath);
                System.setProperty(originalTruststoreTypeKey, type);
                if (password == null) {
                    System.getProperties().remove(originalTruststorePasswordKey);
                } else {
                    System.setProperty(originalTruststorePasswordKey, password);
                }
                defaultTrustStore = new File(trustStorePath);
            }
        } else {
            type = System.getProperty(originalTruststoreTypeKey);
            password = System.getProperty(originalTruststorePasswordKey);
            defaultTrustStore = new File(trustStorePath);
        }
        if (defaultTrustStore.exists()) {
            String path = defaultTrustStore.getAbsolutePath();
            TruststoreBuilder.mergeTrustStore(truststore, path, TruststoreBuilder.loadStore(path, type, password));
        } else {
            LOGGER.warnf("Default truststore was to be included, but could not be found at: %s", (Object)defaultTrustStore);
        }
    }

    static File getJRETruststore() {
        String securityDirectory = System.getProperty("java.home") + File.separator + "lib" + File.separator + "security";
        File jssecacertsFile = new File(securityDirectory, "jssecacerts");
        if (jssecacertsFile.exists() && jssecacertsFile.isFile()) {
            return jssecacertsFile;
        }
        return new File(securityDirectory, "cacerts");
    }

    static KeyStore loadStore(String path, String type, String password) {
        try {
            return KeystoreUtil.loadKeyStore((String)path, (String)password, (String)type);
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to initialize truststore: " + new File(path).getAbsolutePath() + ", type: " + type, e);
        }
    }

    private static boolean mergePemFile(KeyStore truststore, String file, boolean isPem) {
        boolean bl;
        FileInputStream pemInputStream = new FileInputStream(file);
        try {
            CertificateFactory certFactory = CertificateFactory.getInstance("X509");
            boolean loadedAny = false;
            while (pemInputStream.available() > 0) {
                X509Certificate cert;
                try {
                    cert = (X509Certificate)certFactory.generateCertificate(pemInputStream);
                    loadedAny = true;
                }
                catch (CertificateException e) {
                    if (pemInputStream.available() > 0 || !loadedAny) {
                        if (isPem || loadedAny) {
                            throw e;
                        }
                        LOGGER.debugf((Throwable)e, "The file %s may not be in PEM format, it will not be used to create the merged truststore", (Object)new File(file).getAbsolutePath());
                        continue;
                    }
                    LOGGER.debugf((Throwable)e, "The trailing entry for %s generated a certificate exception, assuming instead that the file ends with comments", (Object)new File(file).getAbsolutePath());
                    continue;
                }
                TruststoreBuilder.setCertificateEntry(truststore, cert);
            }
            bl = loadedAny;
        }
        catch (Throwable throwable) {
            try {
                try {
                    pemInputStream.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (Exception e) {
                throw new RuntimeException("Failed to initialize truststore, could not merge: " + new File(file).getAbsolutePath(), e);
            }
        }
        pemInputStream.close();
        return bl;
    }

    private static void setCertificateEntry(KeyStore truststore, Certificate cert) throws KeyStoreException {
        Object alias = null;
        if (cert instanceof X509Certificate) {
            X509Certificate x509Cert = (X509Certificate)cert;
            alias = x509Cert.getSubjectX500Principal().getName() + "_" + x509Cert.getSerialNumber().toString(16);
        } else {
            alias = String.valueOf(Collections.list(truststore.aliases()).size());
        }
        truststore.setCertificateEntry((String)alias, cert);
    }

    private static void mergeTrustStore(KeyStore truststore, String file, KeyStore additionalStore) {
        try {
            for (String alias : Collections.list(additionalStore.aliases())) {
                if (!additionalStore.isCertificateEntry(alias)) continue;
                TruststoreBuilder.setCertificateEntry(truststore, additionalStore.getCertificate(alias));
            }
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to initialize truststore, could not merge: " + new File(file).getAbsolutePath(), e);
        }
    }
}

