/*
 * Decompiled with CFR 0.152.
 */
package org.locationtech.jts.algorithm.hull;

import java.util.ArrayList;
import java.util.List;
import java.util.PriorityQueue;
import org.locationtech.jts.algorithm.hull.HullTri;
import org.locationtech.jts.algorithm.hull.HullTriangulation;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;

public class ConcaveHull {
    private static int PARAM_EDGE_LENGTH = 1;
    private static int PARAM_ALPHA = 2;
    private Geometry inputGeometry;
    private double maxEdgeLengthRatio = -1.0;
    private double alpha = -1.0;
    private boolean isHolesAllowed = false;
    private int criteriaType = PARAM_EDGE_LENGTH;
    private double maxSizeInHull = 0.0;
    private GeometryFactory geomFactory;

    public static double uniformGridEdgeLength(Geometry geom) {
        double areaCH = geom.convexHull().getArea();
        int numPts = geom.getNumPoints();
        return Math.sqrt(areaCH / (double)numPts);
    }

    public static Geometry concaveHullByLength(Geometry geom, double maxLength) {
        return ConcaveHull.concaveHullByLength(geom, maxLength, false);
    }

    public static Geometry concaveHullByLength(Geometry geom, double maxLength, boolean isHolesAllowed) {
        ConcaveHull hull = new ConcaveHull(geom);
        hull.setMaximumEdgeLength(maxLength);
        hull.setHolesAllowed(isHolesAllowed);
        return hull.getHull();
    }

    public static Geometry concaveHullByLengthRatio(Geometry geom, double lengthRatio) {
        return ConcaveHull.concaveHullByLengthRatio(geom, lengthRatio, false);
    }

    public static Geometry concaveHullByLengthRatio(Geometry geom, double lengthRatio, boolean isHolesAllowed) {
        ConcaveHull hull = new ConcaveHull(geom);
        hull.setMaximumEdgeLengthRatio(lengthRatio);
        hull.setHolesAllowed(isHolesAllowed);
        return hull.getHull();
    }

    public static Geometry alphaShape(Geometry geom, double alpha, boolean isHolesAllowed) {
        ConcaveHull hull = new ConcaveHull(geom);
        hull.setAlpha(alpha);
        hull.setHolesAllowed(isHolesAllowed);
        return hull.getHull();
    }

    public ConcaveHull(Geometry geom) {
        this.inputGeometry = geom;
        this.geomFactory = geom.getFactory();
    }

    public void setMaximumEdgeLength(double edgeLength) {
        if (edgeLength < 0.0) {
            throw new IllegalArgumentException("Edge length must be non-negative");
        }
        this.maxSizeInHull = edgeLength;
        this.maxEdgeLengthRatio = -1.0;
        this.criteriaType = PARAM_EDGE_LENGTH;
    }

    public void setMaximumEdgeLengthRatio(double edgeLengthRatio) {
        if (edgeLengthRatio < 0.0 || edgeLengthRatio > 1.0) {
            throw new IllegalArgumentException("Edge length ratio must be in range [0,1]");
        }
        this.maxEdgeLengthRatio = edgeLengthRatio;
        this.criteriaType = PARAM_EDGE_LENGTH;
    }

    public void setAlpha(double alpha) {
        this.alpha = alpha;
        this.maxSizeInHull = alpha;
        this.criteriaType = PARAM_ALPHA;
    }

    public void setHolesAllowed(boolean isHolesAllowed) {
        this.isHolesAllowed = isHolesAllowed;
    }

    public Geometry getHull() {
        if (this.inputGeometry.isEmpty()) {
            return this.geomFactory.createPolygon();
        }
        List<HullTri> triList = HullTriangulation.createDelaunayTriangulation(this.inputGeometry);
        this.setSize(triList);
        if (this.maxEdgeLengthRatio >= 0.0) {
            this.maxSizeInHull = ConcaveHull.computeTargetEdgeLength(triList, this.maxEdgeLengthRatio);
        }
        if (triList.isEmpty()) {
            return this.inputGeometry.convexHull();
        }
        this.computeHull(triList);
        Geometry hull = this.toGeometry(triList, this.geomFactory);
        return hull;
    }

    private void setSize(List<HullTri> triList) {
        for (HullTri tri : triList) {
            if (this.criteriaType == PARAM_EDGE_LENGTH) {
                tri.setSizeToLongestEdge();
                continue;
            }
            tri.setSizeToCircumradius();
        }
    }

    private static double computeTargetEdgeLength(List<HullTri> triList, double edgeLengthRatio) {
        if (edgeLengthRatio == 0.0) {
            return 0.0;
        }
        double maxEdgeLen = -1.0;
        double minEdgeLen = -1.0;
        for (HullTri tri : triList) {
            for (int i = 0; i < 3; ++i) {
                double len = tri.getCoordinate(i).distance(tri.getCoordinate(HullTri.next(i)));
                if (len > maxEdgeLen) {
                    maxEdgeLen = len;
                }
                if (!(minEdgeLen < 0.0) && !(len < minEdgeLen)) continue;
                minEdgeLen = len;
            }
        }
        if (edgeLengthRatio == 1.0) {
            return 2.0 * maxEdgeLen;
        }
        return edgeLengthRatio * (maxEdgeLen - minEdgeLen) + minEdgeLen;
    }

    private void computeHull(List<HullTri> triList) {
        this.computeHullBorder(triList);
        if (this.isHolesAllowed) {
            this.computeHullHoles(triList);
        }
    }

    private void computeHullBorder(List<HullTri> triList) {
        HullTri tri;
        PriorityQueue<HullTri> queue = this.createBorderQueue(triList);
        while (!queue.isEmpty() && !this.isInHull(tri = queue.poll())) {
            if (!this.isRemovableBorder(tri)) continue;
            HullTri adj0 = (HullTri)tri.getAdjacent(0);
            HullTri adj1 = (HullTri)tri.getAdjacent(1);
            HullTri adj2 = (HullTri)tri.getAdjacent(2);
            tri.remove(triList);
            this.addBorderTri(adj0, queue);
            this.addBorderTri(adj1, queue);
            this.addBorderTri(adj2, queue);
        }
    }

    private PriorityQueue<HullTri> createBorderQueue(List<HullTri> triList) {
        PriorityQueue<HullTri> queue = new PriorityQueue<HullTri>();
        for (HullTri tri : triList) {
            this.addBorderTri(tri, queue);
        }
        return queue;
    }

    private void addBorderTri(HullTri tri, PriorityQueue<HullTri> queue) {
        if (tri == null) {
            return;
        }
        if (tri.numAdjacent() != 2) {
            return;
        }
        this.setSize(tri);
        queue.add(tri);
    }

    private void setSize(HullTri tri) {
        if (this.criteriaType == PARAM_EDGE_LENGTH) {
            tri.setSizeToBoundary();
        } else {
            tri.setSizeToCircumradius();
        }
    }

    private boolean isInHull(HullTri tri) {
        return tri.getSize() < this.maxSizeInHull;
    }

    private void computeHullHoles(List<HullTri> triList) {
        List<HullTri> candidateHoles = ConcaveHull.findCandidateHoles(triList, this.maxSizeInHull);
        for (HullTri tri : candidateHoles) {
            if (tri.isRemoved() || tri.isBorder() || tri.hasBoundaryTouch()) continue;
            this.removeHole(triList, tri);
        }
    }

    private static List<HullTri> findCandidateHoles(List<HullTri> triList, double maxSizeInHull) {
        ArrayList<HullTri> candidates = new ArrayList<HullTri>();
        for (HullTri tri : triList) {
            boolean isTouchingBoundary;
            if (tri.getSize() < maxSizeInHull || (isTouchingBoundary = tri.isBorder() || tri.hasBoundaryTouch())) continue;
            candidates.add(tri);
        }
        candidates.sort(null);
        return candidates;
    }

    private void removeHole(List<HullTri> triList, HullTri triHole) {
        HullTri tri;
        PriorityQueue<HullTri> queue = new PriorityQueue<HullTri>();
        queue.add(triHole);
        while (!(queue.isEmpty() || (tri = (HullTri)queue.poll()) != triHole && this.isInHull(tri))) {
            if (tri != triHole && !this.isRemovableHole(tri)) continue;
            HullTri adj0 = (HullTri)tri.getAdjacent(0);
            HullTri adj1 = (HullTri)tri.getAdjacent(1);
            HullTri adj2 = (HullTri)tri.getAdjacent(2);
            tri.remove(triList);
            this.addBorderTri(adj0, queue);
            this.addBorderTri(adj1, queue);
            this.addBorderTri(adj2, queue);
        }
    }

    private boolean isRemovableBorder(HullTri tri) {
        if (tri.numAdjacent() != 2) {
            return false;
        }
        return !tri.isConnecting();
    }

    private boolean isRemovableHole(HullTri tri) {
        if (tri.numAdjacent() != 2) {
            return false;
        }
        return !tri.hasBoundaryTouch();
    }

    private Geometry toGeometry(List<HullTri> triList, GeometryFactory geomFactory) {
        if (!this.isHolesAllowed) {
            return HullTriangulation.traceBoundaryPolygon(triList, geomFactory);
        }
        return HullTriangulation.union(triList, geomFactory);
    }
}

