package org.mapstruct.ap.internal.gem;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.lang.model.element.AnnotationMirror;
import javax.lang.model.element.AnnotationValue;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.element.VariableElement;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.AbstractAnnotationValueVisitor8;
import javax.lang.model.util.ElementFilter;
import org.mapstruct.tools.gem.Gem;
import org.mapstruct.tools.gem.GemValue;


public class JavadocGem implements Gem {

    private final GemValue<String> value;
    private final GemValue<List<String>> authors;
    private final GemValue<String> deprecated;
    private final GemValue<String> since;
    private final boolean isValid;
    private final AnnotationMirror mirror;

    private JavadocGem( BuilderImpl builder ) {
        this.value = builder.value;
        this.authors = builder.authors;
        this.deprecated = builder.deprecated;
        this.since = builder.since;
        isValid = ( this.value != null ? this.value.isValid() : false )
               && ( this.authors != null ? this.authors.isValid() : false )
               && ( this.deprecated != null ? this.deprecated.isValid() : false )
               && ( this.since != null ? this.since.isValid() : false );
        mirror = builder.mirror;
    }

    /**
    * accessor
    *
    * @return the {@link GemValue} for {@link JavadocGem#value}
    */
    public GemValue<String> value( ) {
        return value;
    }

    /**
    * accessor
    *
    * @return the {@link GemValue} for {@link JavadocGem#authors}
    */
    public GemValue<List<String>> authors( ) {
        return authors;
    }

    /**
    * accessor
    *
    * @return the {@link GemValue} for {@link JavadocGem#deprecated}
    */
    public GemValue<String> deprecated( ) {
        return deprecated;
    }

    /**
    * accessor
    *
    * @return the {@link GemValue} for {@link JavadocGem#since}
    */
    public GemValue<String> since( ) {
        return since;
    }

    @Override
    public AnnotationMirror mirror( ) {
        return mirror;
    }

    @Override
    public boolean isValid( ) {
        return isValid;
    }

    public static JavadocGem  instanceOn(Element element) {
        return build( element, new BuilderImpl() );
    }

    public static JavadocGem instanceOn(AnnotationMirror mirror ) {
        return build( mirror, new BuilderImpl() );
    }

    public static  <T> T  build(Element element, Builder<T> builder) {
        AnnotationMirror mirror = element.getAnnotationMirrors().stream()
            .filter( a ->  "org.mapstruct.Javadoc".contentEquals( ( ( TypeElement )a.getAnnotationType().asElement() ).getQualifiedName() ) )
            .findAny()
            .orElse( null );
        return build( mirror, builder );
    }

    public static <T> T build(AnnotationMirror mirror, Builder<T> builder ) {

        // return fast
        if ( mirror == null || builder == null ) {
            return null;
        }

        // fetch defaults from all defined values in the annotation type
        List<ExecutableElement> enclosed = ElementFilter.methodsIn( mirror.getAnnotationType().asElement().getEnclosedElements() );
        Map<String, AnnotationValue> defaultValues = new HashMap<>( enclosed.size() );
        enclosed.forEach( e -> defaultValues.put( e.getSimpleName().toString(), e.getDefaultValue() ) );

        // fetch all explicitely set annotation values in the annotation instance
        Map<String, AnnotationValue> values = new HashMap<>( enclosed.size() );
        mirror.getElementValues().entrySet().forEach( e -> values.put( e.getKey().getSimpleName().toString(), e.getValue() ) );

        // iterate and populate builder
        for ( String methodName : defaultValues.keySet() ) {

            if ( "value".equals( methodName ) ) {
                builder.setValue( GemValue.create( values.get( methodName ), defaultValues.get( methodName ), String.class ) );
            }
            else if ( "authors".equals( methodName ) ) {
                builder.setAuthors( GemValue.createArray( values.get( methodName ), defaultValues.get( methodName ), String.class ) );
            }
            else if ( "deprecated".equals( methodName ) ) {
                builder.setDeprecated( GemValue.create( values.get( methodName ), defaultValues.get( methodName ), String.class ) );
            }
            else if ( "since".equals( methodName ) ) {
                builder.setSince( GemValue.create( values.get( methodName ), defaultValues.get( methodName ), String.class ) );
            }
        }
        builder.setMirror( mirror );
        return builder.build();
    }

    /**
     * A builder that can be implemented by the user to define custom logic e.g. in the
     * build method, prior to creating the annotation gem.
     */
    public interface Builder<T> {

       /**
        * Sets the {@link GemValue} for {@link JavadocGem#value}
        *
        * @return the {@link Builder} for this gem, representing {@link JavadocGem}
        */
        Builder setValue(GemValue<String> methodName );

       /**
        * Sets the {@link GemValue} for {@link JavadocGem#authors}
        *
        * @return the {@link Builder} for this gem, representing {@link JavadocGem}
        */
        Builder setAuthors(GemValue<List<String>> methodName );

       /**
        * Sets the {@link GemValue} for {@link JavadocGem#deprecated}
        *
        * @return the {@link Builder} for this gem, representing {@link JavadocGem}
        */
        Builder setDeprecated(GemValue<String> methodName );

       /**
        * Sets the {@link GemValue} for {@link JavadocGem#since}
        *
        * @return the {@link Builder} for this gem, representing {@link JavadocGem}
        */
        Builder setSince(GemValue<String> methodName );

        /**
         * Sets the annotation mirror
         *
         * @param mirror the mirror which this gem represents
         *
         * @return the {@link Builder} for this gem, representing {@link JavadocGem}
         */
          Builder setMirror( AnnotationMirror mirror );

        /**
         * The build method can be overriden in a custom custom implementation, which allows
         * the user to define his own custom validation on the annotation.
         *
         * @return the representation of the annotation
         */
        T build();
    }

    private static class BuilderImpl implements Builder<JavadocGem> {

        private GemValue<String> value;
        private GemValue<List<String>> authors;
        private GemValue<String> deprecated;
        private GemValue<String> since;
        private AnnotationMirror mirror;

        public Builder setValue(GemValue<String> value ) {
            this.value = value;
            return this;
        }

        public Builder setAuthors(GemValue<List<String>> authors ) {
            this.authors = authors;
            return this;
        }

        public Builder setDeprecated(GemValue<String> deprecated ) {
            this.deprecated = deprecated;
            return this;
        }

        public Builder setSince(GemValue<String> since ) {
            this.since = since;
            return this;
        }

        public Builder  setMirror( AnnotationMirror mirror ) {
            this.mirror = mirror;
            return this;
        }

        public JavadocGem build() {
            return new JavadocGem( this );
        }
    }

}
