/*
 * Copyright (c) 2005, 2013, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package org.openjdk.jmh.infra;

import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.runner.WorkloadParams;
import org.openjdk.jmh.runner.options.TimeValue;
import org.openjdk.jmh.util.Utils;
import org.openjdk.jmh.util.Version;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.concurrent.TimeUnit;

/**
 * Benchmark parameters.
 *
 * <p>{@link BenchmarkParams} handles the parameters used in the current run.</p>
 * <p>This class is dual-purpose:</p>
 * <ol>
 *     <li>It acts as the interface between host JVM and forked JVM, so that the latter
 *     would not have to figure out the benchmark configuration again</li>
 *     <li>It can be injected into benchmark methods to access the runtime configuration
 *     info about the benchmark</li>
 * </ol>
 */
public class BenchmarkParams extends BenchmarkParamsL4 {
    private static final long serialVersionUID = -1068219503090299117L;

    /**
     * Do the class hierarchy trick to evade false sharing, and check if it's working in runtime.
     * @see org.openjdk.jmh.infra.Blackhole description for the rationale
     */
    static {
        Utils.check(BenchmarkParams.class, "benchmark", "generatedTarget", "synchIterations");
        Utils.check(BenchmarkParams.class, "threads", "threadGroups", "forks", "warmupForks");
        Utils.check(BenchmarkParams.class, "warmup", "measurement");
        Utils.check(BenchmarkParams.class, "mode", "params");
        Utils.check(BenchmarkParams.class, "timeUnit", "opsPerInvocation");
        Utils.check(BenchmarkParams.class, "jvm", "jvmArgs");
    }

    public BenchmarkParams(String benchmark, String generatedTarget, boolean synchIterations,
                           int threads, int[] threadGroups, Collection<String> threadGroupLabels,
                           int forks, int warmupForks,
                           IterationParams warmup, IterationParams measurement,
                           Mode mode, WorkloadParams params,
                           TimeUnit timeUnit, int opsPerInvocation,
                           String jvm, Collection<String> jvmArgs,
                           String jdkVersion, String vmName, String vmVersion, String jmhVersion,
                           TimeValue timeout) {
        super(benchmark, generatedTarget, synchIterations,
                threads, threadGroups, threadGroupLabels,
                forks, warmupForks,
                warmup, measurement,
                mode, params,
                timeUnit, opsPerInvocation,
                jvm, jvmArgs,
                jdkVersion, vmName, vmVersion, jmhVersion,
                timeout);
    }
}

abstract class BenchmarkParamsL4 extends BenchmarkParamsL3 {
    private static final long serialVersionUID = -2409216922027695385L;

    private int markerEnd;
    public BenchmarkParamsL4(String benchmark, String generatedTarget, boolean synchIterations,
                             int threads, int[] threadGroups, Collection<String> threadGroupLabels,
                             int forks, int warmupForks,
                             IterationParams warmup, IterationParams measurement,
                             Mode mode, WorkloadParams params,
                             TimeUnit timeUnit, int opsPerInvocation,
                             String jvm, Collection<String> jvmArgs,
                             String jdkVersion, String vmName, String vmVersion, String jmhVersion,
                             TimeValue timeout) {
        super(benchmark, generatedTarget, synchIterations,
                threads, threadGroups, threadGroupLabels,
                forks, warmupForks,
                warmup, measurement,
                mode, params,
                timeUnit, opsPerInvocation,
                jvm, jvmArgs,
                jdkVersion, vmName, vmVersion, jmhVersion,
                timeout);
    }
}

abstract class BenchmarkParamsL3 extends BenchmarkParamsL2 {
    private static final long serialVersionUID = -53511295235994554L;

    private boolean q001, q002, q003, q004, q005, q006, q007, q008;
    private boolean q011, q012, q013, q014, q015, q016, q017, q018;
    private boolean q021, q022, q023, q024, q025, q026, q027, q028;
    private boolean q031, q032, q033, q034, q035, q036, q037, q038;
    private boolean q041, q042, q043, q044, q045, q046, q047, q048;
    private boolean q051, q052, q053, q054, q055, q056, q057, q058;
    private boolean q061, q062, q063, q064, q065, q066, q067, q068;
    private boolean q071, q072, q073, q074, q075, q076, q077, q078;
    private boolean q101, q102, q103, q104, q105, q106, q107, q108;
    private boolean q111, q112, q113, q114, q115, q116, q117, q118;
    private boolean q121, q122, q123, q124, q125, q126, q127, q128;
    private boolean q131, q132, q133, q134, q135, q136, q137, q138;
    private boolean q141, q142, q143, q144, q145, q146, q147, q148;
    private boolean q151, q152, q153, q154, q155, q156, q157, q158;
    private boolean q161, q162, q163, q164, q165, q166, q167, q168;
    private boolean q171, q172, q173, q174, q175, q176, q177, q178;

    public BenchmarkParamsL3(String benchmark, String generatedTarget, boolean synchIterations,
                             int threads, int[] threadGroups, Collection<String> threadGroupLabels,
                             int forks, int warmupForks,
                             IterationParams warmup, IterationParams measurement,
                             Mode mode, WorkloadParams params,
                             TimeUnit timeUnit, int opsPerInvocation,
                             String jvm, Collection<String> jvmArgs,
                             String jdkVersion, String vmName, String vmVersion, String jmhVersion,
                             TimeValue timeout) {
        super(benchmark, generatedTarget, synchIterations,
                threads, threadGroups, threadGroupLabels,
                forks, warmupForks,
                warmup, measurement,
                mode, params,
                timeUnit, opsPerInvocation,
                jvm, jvmArgs,
                jdkVersion, vmName, vmVersion, jmhVersion,
                timeout);
    }
}

abstract class BenchmarkParamsL1 extends BenchmarkParamsL0 {
    private boolean p001, p002, p003, p004, p005, p006, p007, p008;
    private boolean p011, p012, p013, p014, p015, p016, p017, p018;
    private boolean p021, p022, p023, p024, p025, p026, p027, p028;
    private boolean p031, p032, p033, p034, p035, p036, p037, p038;
    private boolean p041, p042, p043, p044, p045, p046, p047, p048;
    private boolean p051, p052, p053, p054, p055, p056, p057, p058;
    private boolean p061, p062, p063, p064, p065, p066, p067, p068;
    private boolean p071, p072, p073, p074, p075, p076, p077, p078;
    private boolean p101, p102, p103, p104, p105, p106, p107, p108;
    private boolean p111, p112, p113, p114, p115, p116, p117, p118;
    private boolean p121, p122, p123, p124, p125, p126, p127, p128;
    private boolean p131, p132, p133, p134, p135, p136, p137, p138;
    private boolean p141, p142, p143, p144, p145, p146, p147, p148;
    private boolean p151, p152, p153, p154, p155, p156, p157, p158;
    private boolean p161, p162, p163, p164, p165, p166, p167, p168;
    private boolean p171, p172, p173, p174, p175, p176, p177, p178;
}

abstract class BenchmarkParamsL0 {
    private int markerBegin;
}

abstract class BenchmarkParamsL2 extends BenchmarkParamsL1 implements Serializable, Comparable<BenchmarkParams> {
    private static final long serialVersionUID = -1068219503090299117L;

    protected final String benchmark;
    protected final String generatedTarget;
    protected final boolean synchIterations;
    protected final int threads;
    protected final int[] threadGroups;
    protected final Collection<String> threadGroupLabels;
    protected final int forks;
    protected final int warmupForks;
    protected final IterationParams warmup;
    protected final IterationParams measurement;
    protected final Mode mode;
    protected final WorkloadParams params;
    protected final TimeUnit timeUnit;
    protected final int opsPerInvocation;
    protected final String jvm;
    protected final Collection<String> jvmArgs;
    protected final String jdkVersion;
    protected final String jmhVersion;
    protected final String vmName;
    protected final String vmVersion;
    protected final TimeValue timeout;

    public BenchmarkParamsL2(String benchmark, String generatedTarget, boolean synchIterations,
                             int threads, int[] threadGroups, Collection<String> threadGroupLabels,
                             int forks, int warmupForks,
                             IterationParams warmup, IterationParams measurement,
                             Mode mode, WorkloadParams params,
                             TimeUnit timeUnit, int opsPerInvocation,
                             String jvm, Collection<String> jvmArgs,
                             String jdkVersion, String vmName, String vmVersion, String jmhVersion,
                             TimeValue timeout) {
        this.benchmark = benchmark;
        this.generatedTarget = generatedTarget;
        this.synchIterations = synchIterations;
        this.threads = threads;
        this.threadGroups = threadGroups;
        this.threadGroupLabels = threadGroupLabels;
        this.forks = forks;
        this.warmupForks = warmupForks;
        this.warmup = warmup;
        this.measurement = measurement;
        this.mode = mode;
        this.params = params;
        this.timeUnit = timeUnit;
        this.opsPerInvocation = opsPerInvocation;
        this.jvm = jvm;
        this.jvmArgs = jvmArgs;
        this.jdkVersion = jdkVersion;
        this.vmName = vmName;
        this.vmVersion = vmVersion;
        this.jmhVersion = jmhVersion;
        this.timeout = timeout;
    }

    /**
     * @return how long to wait for iteration to complete
     */
    public TimeValue getTimeout() {
        return timeout;
    }

    /**
     * @return do we synchronize iterations?
     */
    public boolean shouldSynchIterations() {
        return synchIterations;
    }

    /**
     * @return iteration parameters for warmup phase
     */
    public IterationParams getWarmup() {
        return warmup;
    }

    /**
     * @return iteration parameters for measurement phase
     */
    public IterationParams getMeasurement() {
        return measurement;
    }

    /**
     * @return total measurement thread count
     */
    public int getThreads() {
        return threads;
    }

    /**
     * @return thread distribution within the group
     * @see org.openjdk.jmh.runner.options.ChainedOptionsBuilder#threadGroups(int...)
     */
    public int[] getThreadGroups() {
        return Arrays.copyOf(threadGroups, threadGroups.length);
    }

    /**
     * @return subgroup thread labels
     * @see #getThreadGroups()
     */
    public Collection<String> getThreadGroupLabels() {
        return Collections.unmodifiableCollection(threadGroupLabels);
    }

    /**
     * @return number of forked VM runs, which we measure
     */
    public int getForks() {
        return forks;
    }

    /**
     * @return number of forked VM runs, which we discard from the result
     */
    public int getWarmupForks() {
        return warmupForks;
    }

    /**
     * @return benchmark mode
     */
    public Mode getMode() {
        return mode;
    }

    /**
     * @return benchmark name
     */
    public String getBenchmark() {
        return benchmark;
    }

    /**
     * @return timeUnit used in results
     */
    public TimeUnit getTimeUnit() {
        return timeUnit;
    }

    /**
     * @return operations per invocation used
     */
    public int getOpsPerInvocation() {
        return opsPerInvocation;
    }

    /**
     * @return all workload parameters
     */
    public Collection<String> getParamsKeys() {
        return params.keys();
    }

    /**
     * @param key parameter key; usually the field name
     * @return parameter value for given key
     */
    public String getParam(String key) {
        if (params != null) {
            return params.get(key);
        } else {
            return null;
        }
    }

    /**
     * @return generated benchmark name
     */
    public String generatedBenchmark() {
        return generatedTarget;
    }

    /**
     * @return JVM executable path
     */
    public String getJvm() {
        return jvm;
    }

    /**
     * @return JMH version identical to {@link Version#getPlainVersion()}, but output format should
     *          get there input via bean for testing purposes.
     */
    public String getJmhVersion() {
        return jmhVersion;
    }

    /**
     * @return JVM options
     */
    public Collection<String> getJvmArgs() {
        return Collections.unmodifiableCollection(jvmArgs);
    }

    /**
     * @return version information as returned by the effective target JVM,
     *         via system property {@code java.version} and {@code java.vm.version}
     */
    public String getJdkVersion() {
        return jdkVersion;
    }

    /**
     * @return version information as returned by the effective target JVM,
     *         via system property {@code java.vm.version}
     */
    public String getVmVersion() {
        return vmVersion;
    }

    /**
     * @return name information as returned by the effective target JVM,
     *         via system property {@code java.vm.name}
     */
    public String getVmName() {
        return vmName;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        BenchmarkParams that = (BenchmarkParams) o;

        if (!benchmark.equals(that.benchmark)) return false;
        if (mode != that.mode) return false;
        if (!params.equals(that.params)) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = benchmark.hashCode();
        result = 31 * result + mode.hashCode();
        result = 31 * result + params.hashCode();
        return result;
    }

    @Override
    public int compareTo(BenchmarkParams o) {
        int v = mode.compareTo(o.mode);
        if (v != 0) {
            return v;
        }

        int v1 = benchmark.compareTo(o.benchmark);
        if (v1 != 0) {
            return v1;
        }

        if (params == null || o.params == null) {
            return 0;
        }

        return params.compareTo(o.params);
    }

    public String id() {
        StringBuilder sb = new StringBuilder();
        sb.append(benchmark).append("-");
        sb.append(mode);
        for (String key : params.keys()) {
            sb.append("-");
            sb.append(key).append("-").append(params.get(key));
        }
        return sb.toString();
    }

}
