import ts from "typescript";
import { countChildNodesRecursive, findNodeRecursive, isReturnValueUsed, } from "../utils/utils.js";
import BaseFix from "./BaseFix.js";
import { isSideEffectFree } from "../utils/sideEffects.js";
export default class CallExpressionBaseFix extends BaseFix {
    params;
    nodeTypes = [ts.SyntaxKind.CallExpression];
    containedCallExpressionCount = 0;
    constructor(params, ui5TypeInfo) {
        super(params, ui5TypeInfo);
        this.params = params;
    }
    visitLinterNode(node, sourcePosition, helpers) {
        if (!ts.isCallExpression(node)) {
            return false;
        }
        // If requested, check whether the return value of the call expression is assigned to a variable,
        // passed to another function or used elsewhere.
        if (this.params.mustNotUseReturnValue && isReturnValueUsed(node)) {
            return false;
        }
        const containedCallExpression = findNodeRecursive(node.expression, this.nodeTypes);
        if (containedCallExpression) {
            // Call expression fixes must not affect other call expressions, unless the contained call expression
            // is side-effect free, e.g. sap.ui.getCore().method()
            if (isSideEffectFree(containedCallExpression, helpers.checker)) {
                this.containedCallExpressionCount = 1;
            }
            else {
                return false;
            }
        }
        this.sourcePosition = sourcePosition;
        return true;
    }
    visitAutofixNode(node, position, sourceFile) {
        if (!ts.isCallExpression(node)) {
            return false;
        }
        const count = countChildNodesRecursive(node.expression, this.nodeTypes);
        if (count !== this.containedCallExpressionCount) {
            // The number of call expressions does not match the expected count
            // Reject this node and wait for it's child
            return false;
        }
        this.startPos = node.getStart(sourceFile);
        this.endPos = node.getEnd();
        return true;
    }
}
//# sourceMappingURL=CallExpressionBaseFix.js.map