import path, { dirname } from "node:path";
import { fileURLToPath, pathToFileURL } from "node:url";
const __dirname = dirname(fileURLToPath(import.meta.url));
;
const CONFIG_FILENAMES = [
    "ui5lint.config.js",
    "ui5lint.config.mjs",
    "ui5lint.config.cjs",
];
export default class ConfigManager {
    #projectRootDir;
    #configFile;
    constructor(projectRootDir, configFile) {
        if (!projectRootDir) {
            throw Error("Project's root dir is required");
        }
        this.#projectRootDir = projectRootDir;
        this.#configFile = configFile ?? null;
    }
    #resolveModulePaths(fileName) {
        const resolvedPath = path.join(this.#projectRootDir, fileName);
        return pathToFileURL(resolvedPath).toString();
    }
    async getConfiguration() {
        let config = {};
        if (this.#configFile) {
            // If it's an relative path, transform to POSIX format
            const configFilePath = this.#resolveModulePaths(this.#configFile);
            ({ default: config } = await import(configFilePath));
        }
        else {
            // Find configuration file
            // @ts-expect-error If it's not a missing config file we need to propagate the exception
            ({ default: config } = await Promise.any(CONFIG_FILENAMES.map((filename) => {
                // Relative paths are needed to make it work on Windows
                const configFilePath = this.#resolveModulePaths(filename);
                return import(configFilePath);
            }))
                // Promise.any would throw if nothing is found i.e. there's no config file
                .catch((errs) => {
                const { errors } = errs;
                if (errors?.every((e) => e?.code === "ERR_MODULE_NOT_FOUND")) {
                    return { default: {} };
                }
                const errToThrow = errors?.find((e) => e?.code !== "ERR_MODULE_NOT_FOUND");
                if (errToThrow) {
                    throw errToThrow;
                }
            }));
        }
        return config;
    }
}
//# sourceMappingURL=ConfigManager.js.map