/*
 * Copyright 2024 the original author or authors.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openrewrite.csharp.dependencies;

import com.fasterxml.jackson.databind.MappingIterator;
import com.fasterxml.jackson.dataformat.csv.CsvMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.EqualsAndHashCode;
import lombok.Value;
import org.jspecify.annotations.Nullable;
import org.openrewrite.ExecutionContext;
import org.openrewrite.Option;
import org.openrewrite.ScanningRecipe;
import org.openrewrite.TreeVisitor;
import org.openrewrite.csharp.dependencies.table.VulnerabilityReport;
import org.openrewrite.csharp.dependencies.trait.PackageReference;
import org.openrewrite.internal.StringUtils;
import org.openrewrite.java.dependencies.Vulnerability;
import org.openrewrite.java.dependencies.internal.StaticVersionComparator;
import org.openrewrite.java.dependencies.internal.Version;
import org.openrewrite.java.dependencies.internal.VersionParser;
import org.openrewrite.marker.SearchResult;
import org.openrewrite.semver.LatestPatch;
import org.openrewrite.xml.tree.Xml;

import java.io.IOException;
import java.io.InputStream;
import java.util.*;

import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.partitioningBy;

@Value
@EqualsAndHashCode(callSuper = false)
public class DependencyVulnerabilityCheck extends ScanningRecipe<DependencyVulnerabilityCheck.Accumulator> {
    transient VersionParser versionParser = new VersionParser();
    transient VulnerabilityReport report = new VulnerabilityReport(this);

    @Option(displayName = "Add search markers",
            description = "Report each vulnerability as search result markers. " +
                          "When enabled you can see which dependencies are bringing in vulnerable transitives in the diff view. " +
                          "By default these markers are omitted, making it easier to see version upgrades within the diff.",
            required = false)
    @Nullable
    Boolean addMarkers;

    @Override
    public String getDisplayName() {
        return "Find and fix vulnerable Nuget dependencies";
    }

    @Override
    public String getDescription() {
        //language=markdown
        return "This software composition analysis (SCA) tool detects and upgrades dependencies with publicly disclosed vulnerabilities. " +
               "This recipe both generates a report of vulnerable dependencies and upgrades to newer versions with fixes. " +
               "This recipe **only** upgrades to the latest **patch** version.  If a minor or major upgrade is required to reach the fixed version, this recipe will not make any changes. " +
               "Vulnerability information comes from the [GitHub Security Advisory Database](https://docs.github.com/en/code-security/security-advisories/global-security-advisories/about-the-github-advisory-database), " +
               "which aggregates vulnerability data from several public databases, including the [National Vulnerability Database](https://nvd.nist.gov/) maintained by the United States government. " +
               "Dependencies following [Semantic Versioning](https://semver.org/) will see their _patch_ version updated where applicable.";
    }

    @Value
    public static class Accumulator {
        Map<String, List<Vulnerability>> db;
        Map<NameVersion, Set<Vulnerability>> vulnerabilities;

        @Value
        static class NameVersion {
            /**
             * The name of the package as specified in the package.json.
             */
            String name;

            /**
             * The resolved version actually in use, which may be different from the version specified in the package.json.
             */
            String version;
        }
    }

    @Override
    public Accumulator getInitialValue(ExecutionContext ctx) {
        CsvMapper csvMapper = new CsvMapper();
        csvMapper.registerModule(new JavaTimeModule());
        Map<String, List<Vulnerability>> db = new HashMap<>();

        try (InputStream resourceAsStream = DependencyVulnerabilityCheck.class.getResourceAsStream("/advisories-nuget.csv");
             MappingIterator<Vulnerability> vs = csvMapper.readerWithSchemaFor(Vulnerability.class).readValues(resourceAsStream)) {
            while (vs.hasNextValue()) {
                Vulnerability v = vs.nextValue();
                db.computeIfAbsent(v.getGroupArtifact(), g -> new ArrayList<>()).add(v);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return new Accumulator(db, new HashMap<>());
    }

    @Override
    public TreeVisitor<?, ExecutionContext> getScanner(Accumulator acc) {
        return new PackageReference.Matcher().asVisitor((ref, ctx) -> {
            String dependencyName = ref.getInclude();
            for (Vulnerability v : acc.db.getOrDefault(dependencyName, Collections.emptyList())) {
                String dependencyVersion = ref.getVersion();
                if (isVulnerable(dependencyVersion, v)) {
                    // Add all vulnerable dependencies to the accumulator
                    acc.vulnerabilities
                            .computeIfAbsent(new Accumulator.NameVersion(dependencyName, dependencyVersion), nv -> new LinkedHashSet<>())
                            .add(v);

                    // Insert a row into the report for each vulnerability
                    report.insertRow(ctx, new VulnerabilityReport.Row(
                            v.getCve(),
                            dependencyName,
                            dependencyVersion,
                            v.getFixedVersion(),
                            isFixWithPatchVersionUpdateOnly(dependencyVersion, v),
                            v.getSummary(),
                            v.getSeverity().toString(),
                            0,
                            v.getCwes()
                    ));
                }
            }
            return ref.getTree();
        });
    }


    private static final Comparator<Version> vc = new StaticVersionComparator();

    private boolean isVulnerable(String dependencyVersion, Vulnerability v) {
        Version actual = versionParser.transform(dependencyVersion);
        if (vc.compare(actual, versionParser.transform(v.getIntroducedVersion())) < 0) {
            return false;
        }
        String fixedVersion = v.getFixedVersion();
        if (StringUtils.isBlank(fixedVersion)) {
            return true;
        }
        return vc.compare(actual, versionParser.transform(fixedVersion)) < 0;
    }

    private static final LatestPatch latestPatch = new LatestPatch(null);

    private static boolean isFixWithPatchVersionUpdateOnly(String dependencyVersion, Vulnerability v) {
        return !StringUtils.isBlank(v.getFixedVersion()) &&
               latestPatch.isValid(dependencyVersion, v.getFixedVersion()) &&
               latestPatch.compare(dependencyVersion, dependencyVersion, v.getFixedVersion()) < 0;
    }

    @Override
    public TreeVisitor<?, ExecutionContext> getVisitor(Accumulator acc) {
        return new PackageReference.Matcher().asVisitor((ref, ctx) -> {
            Xml.Tag tag = ref.getTree();

            // Partition vulnerabilities into those that can be fixed with a patch version update and those that can't
            String dependencyVersion = ref.getVersion();
            Map<Boolean, List<Vulnerability>> vulnerabilities = acc.vulnerabilities
                    .getOrDefault(new Accumulator.NameVersion(ref.getInclude(), ref.getVersion()), emptySet())
                    .stream().collect(partitioningBy(v -> isFixWithPatchVersionUpdateOnly(dependencyVersion, v)));

            // Bump to highest fixed patch version
            String highestFixedPatchVersion = vulnerabilities.get(true).stream()
                    .max(Comparator.comparing(v -> versionParser.transform(v.getFixedVersion()), vc))
                    .map(Vulnerability::getFixedVersion)
                    .orElse(null);
            if (highestFixedPatchVersion != null) {
                tag = ref.withVersion(highestFixedPatchVersion);
            }

            // Add marker of vulnerabilities not patched
            List<Vulnerability> remainingVulnerabilities = vulnerabilities.get(false);
            if (Boolean.TRUE.equals(addMarkers) && !remainingVulnerabilities.isEmpty()) {
                tag = SearchResult.found(tag,
                        "This dependency has the following vulnerabilities:\n" +
                        remainingVulnerabilities.stream()
                                .map(v -> String.format("%s (%s severity%s) - %s",
                                        v.getCve(),
                                        v.getSeverity(),
                                        StringUtils.isBlank(v.getFixedVersion()) ? "" : ", fixed in " + v.getFixedVersion(),
                                        v.getSummary()))
                                .collect(joining("\n")));
            }

            return tag;
        });
    }
}
