/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.reflext.api.introspection;

import org.reflext.api.MethodInfo;
import org.reflext.api.TypeInfo;
import org.reflext.api.ClassTypeInfo;
import org.reflext.api.ParameterizedTypeInfo;
import org.reflext.api.AnnotationIntrospector;
import org.reflext.api.HierarchyVisitor;

import java.util.Set;
import java.util.Collection;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.HashSet;
import java.lang.annotation.Annotation;
import java.beans.Introspector;

/**
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
public class MethodIntrospector {

  /** . */
  private final HierarchyScope hierarchyScope;

  /** . */
  private final boolean removeOverrides;

  public MethodIntrospector(HierarchyScope hierarchyScope, boolean removeOverrides) {
    this.hierarchyScope = hierarchyScope;
    this.removeOverrides = removeOverrides;
  }

  public MethodIntrospector(HierarchyScope hierarchyScope) {
    this(hierarchyScope, false);
  }

  public <A extends Annotation> Collection<MethodInfo> resolveMethods(ClassTypeInfo cti, Class<A> annotationClass) {
    ArrayList<MethodInfo> methods = new ArrayList<MethodInfo>();
    AnnotationIntrospector<A> introspector = new AnnotationIntrospector<A>(annotationClass);
    for (MethodInfo method : getMethods(cti)) {
      if (introspector.resolve(method) != null) {
        methods.add(method);
      }
    }
    return methods;
  }

  public Map<String, MethodInfo> getGetterMap(ClassTypeInfo cti) {
    Map<String, MethodInfo> getterMap = new HashMap<String, MethodInfo>();
    for (MethodInfo getter : getGetters(cti)) {
      String getterName = getter.getName();
      String name;
      if (getterName.startsWith("get")) {
        name = Introspector.decapitalize(getterName.substring(3));
      } else {
        name = Introspector.decapitalize(getterName.substring(2));
      }
      getterMap.put(name, getter);
    }
    return getterMap;
  }

  public Iterable<MethodInfo> getGetters(ClassTypeInfo cti) {
    final MethodContainer getters = new MethodContainer();
    HierarchyVisitor visitor = new HierarchyVisitor() {
      public boolean enter(ClassTypeInfo type) {
        return true;
      }
      public void leave(ClassTypeInfo type) {
        if (!type.getName().equals(Object.class.getName())) {
          for (MethodInfo method : type.getDeclaredMethods()) {
            String methodName = method.getName();
            if (
              methodName.startsWith("get") &&
              methodName.length() > 3 &&
              Character.isUpperCase(methodName.charAt(3)) &&
              method.getParameterTypes().size() == 0) {
              getters.add(method);
            }
          }
        }
      }
    };

    //
    cti.accept(visitor);

    //
    return getters;
  }

  public Map<String, Set<MethodInfo>> getSetterMap(ClassTypeInfo cti) {
    Map<String, Set<MethodInfo>> setterMap = new HashMap<String, Set<MethodInfo>>();
    for (MethodInfo setter : getSetters(cti)) {
      String name = Introspector.decapitalize(setter.getName().substring(3));
      Set<MethodInfo> setters = setterMap.get(name);
      if (setters == null) {
        setters = new HashSet<MethodInfo>();
        setterMap.put(name, setters);
      }
      setters.add(setter);
    }
    return setterMap;
  }

  public Iterable<MethodInfo> getSetters(ClassTypeInfo cti) {
    final MethodContainer setters = new MethodContainer();
    HierarchyVisitor visitor = new HierarchyVisitor() {
      public boolean enter(ClassTypeInfo type) {
        return true;
      }
      public void leave(ClassTypeInfo type) {
        if (!type.getName().equals(Object.class.getName())) {
          for (MethodInfo method : type.getDeclaredMethods()) {
            String methodName = method.getName();
            if (
              methodName.startsWith("set") &&
              methodName.length() > 3 &&
              Character.isUpperCase(methodName.charAt(3)) &&
              method.getParameterTypes().size() == 1) {
              setters.add(method);
            }
          }
        }
      }
    };

    //
    cti.accept(visitor);

    //
    return setters;
  }

  public Set<MethodInfo> getMethods(TypeInfo cti) {
    MethodContainer container;
    if (removeOverrides) {
      container = new MethodContainer((ClassTypeInfo)cti);
    } else {
      container = new MethodContainer();
    }
    findMethods(cti, container);
    return container.toCollection();
  }

  private void findMethods(TypeInfo ti, MethodContainer container) {
    if (ti instanceof ClassTypeInfo) {
      findMethods((ClassTypeInfo)ti, container);
    } else if (ti instanceof ParameterizedTypeInfo) {
      findMethods(((ParameterizedTypeInfo)ti).getRawType(), container);
    } else {
      throw new UnsupportedOperationException("Cannot get methods from type " + ti);
    }
  }

  private void findMethods(ClassTypeInfo clazz, MethodContainer container) {
    if (hierarchyScope == HierarchyScope.CLASS) {
      //
    } else {
      TypeInfo superType = clazz.getSuperType();
      if (superType == null || (superType instanceof ClassTypeInfo && ((ClassTypeInfo)superType).getName().equals(Object.class.getName()))) {
        //
      } else {
        findMethods(superType, container);
      }
    }

    //
    if (hierarchyScope == HierarchyScope.ALL) {
      for (TypeInfo interfaceType : clazz.getInterfaces()) {
        findMethods(interfaceType, container);
      }
    }

    //
    for (MethodInfo declaredMethod : clazz.getDeclaredMethods()) {
      container.add(declaredMethod);
    }
  }
}
