/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.reflext.core;

import org.reflext.api.TypeInfo;
import org.reflext.api.ArrayTypeInfo;
import org.reflext.api.ClassTypeInfo;
import org.reflext.api.ParameterizedTypeInfo;
import org.reflext.api.VoidTypeInfo;
import org.reflext.api.TypeVariableInfo;
import org.reflext.api.WildcardTypeInfo;
import org.reflext.api.Visitor;

import java.util.ArrayList;
import java.util.List;

/**
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
public class Utils {

  public static boolean isCovariantReturnType(TypeInfo superTI, TypeInfo extendsTI) {
    if (superTI instanceof ArrayTypeInfo) {
      return isCovariantReturnType((ArrayTypeInfo)superTI, extendsTI);
    } else if (superTI instanceof ClassTypeInfo) {
      return isCovariantReturnType((ClassTypeInfo)superTI, extendsTI);
    } else if (superTI instanceof ParameterizedTypeInfo) {
      return isCovariantReturnType((ParameterizedTypeInfo)superTI, extendsTI);
    } else if (superTI instanceof VoidTypeInfo) {
      return isCovariantReturnType((VoidTypeInfo)superTI, extendsTI);
    } else if (superTI instanceof TypeVariableInfo) {
      return isCovariantReturnType((TypeVariableInfo)superTI, extendsTI);
    } else {
      throw new AssertionError();
    }
  }

  private static boolean isCovariantReturnType(ClassTypeInfo superCTI, TypeInfo extendsTI) {
    if (extendsTI instanceof ClassTypeInfo) {
      ClassTypeInfo cti = (ClassTypeInfo)extendsTI;
      return superCTI.isAssignableFrom(cti);
    } else if (extendsTI instanceof TypeVariableInfo) {
      TypeVariableInfo tvi = (TypeVariableInfo)extendsTI;
      return isCovariantReturnType(superCTI, tvi.getBounds().get(0));
    } else if (extendsTI instanceof ParameterizedTypeInfo) {
      ParameterizedTypeInfo pti = (ParameterizedTypeInfo)extendsTI;
      return isCovariantReturnType(superCTI, pti.getRawType());
    } else if (extendsTI instanceof WildcardTypeInfo) {
      throw new AssertionError();
    }
    return false;
  }

  private static boolean isCovariantReturnType(VoidTypeInfo superVTI, TypeInfo extendsTI) {
    return extendsTI instanceof VoidTypeInfo;
  }

  private static boolean checkVariableExtension(TypeVariableInfo superTVI, TypeVariableInfo extendsTVI, TypeInfo ti) {
    if (ti instanceof ParameterizedTypeInfo) {
      ParameterizedTypeInfo pti = (ParameterizedTypeInfo)ti;
      for (int i = 0;i < pti.getTypeArguments().size();i++) {
        TypeInfo ptiTypeArg = pti.getTypeArguments().get(i);
        if (ptiTypeArg.equals(extendsTVI)) {
          TypeInfo ptiRawType = pti.getRawType();
          if (ptiRawType instanceof ClassTypeInfo) {
            ClassTypeInfo ptiRawTypeCTI = (ClassTypeInfo)ptiRawType;
            if (isCovariantReturnType(superTVI, ptiRawTypeCTI.getTypeParameters().get(i))) {
              return true;
            }
          }
        }
      }
    }
    return false;
  }

  private static boolean isCovariantReturnType(TypeVariableInfo superTVI, TypeInfo extendsTI) {
    if (extendsTI instanceof TypeVariableInfo) {
      TypeVariableInfo extendsTVI = (TypeVariableInfo)extendsTI;
      if (superTVI.equals(extendsTI)) {
        return true;
      } else {
        ClassTypeInfo decl = (ClassTypeInfo)extendsTVI.getGenericDeclaration();
        TypeInfo declSuperTI = decl.getSuperType();
        if (checkVariableExtension(superTVI, extendsTVI, declSuperTI)) {
          return true;
        }
        for (TypeInfo declInterfaceTI : decl.getInterfaces()) {
          if (checkVariableExtension(superTVI, extendsTVI, declInterfaceTI)) {
            return true;
          }
        }
      }
      return false;
    }  else if (extendsTI instanceof ClassTypeInfo) {
      TypeInfo bound = superTVI.getBounds().get(0);
      return isCovariantReturnType(bound, extendsTI);
    } else {
      throw new UnsupportedOperationException();
    }
  }

  private static boolean isCovariantReturnType(ParameterizedTypeInfo superPTI, TypeInfo extendsTI) {
    return isCovariantReturnType(superPTI.getRawType(), extendsTI);
  }

  private static boolean isCovariantReturnType(ArrayTypeInfo superATI, TypeInfo extendsTI) {
    if (extendsTI instanceof ArrayTypeInfo) {
      ArrayTypeInfo extendsATI = (ArrayTypeInfo)extendsTI;
      return isCovariantReturnType(superATI.getComponentType(), extendsATI.getComponentType());
    } else {
      return false;
    }
  }

  public static TypeInfo resolve(ClassTypeInfo context, TypeInfo type) {
    if (type instanceof ClassTypeInfo) {
      return resolve(context, (ClassTypeInfo)type);
    } else if (type instanceof ParameterizedTypeInfo) {
      return resolve(context, (ParameterizedTypeInfo)type);
    } else if (type instanceof TypeVariableInfo) {
      return resolve(context, (TypeVariableInfo)type);
    } else if (type instanceof ArrayTypeInfo) {
      return resolve(context, (ArrayTypeInfo)type);
    } else if (type instanceof WildcardTypeInfo) {
      return resolve(context, (WildcardTypeInfo)type);
    } else {
      throw new UnsupportedOperationException("Cannot resolve type " + type + " with class " + type.getClass().getName());
    }
  }

  static TypeInfo resolve(ClassTypeInfo context, final WildcardTypeInfo wildcardType) {
    final ArrayList<TypeInfo> resolvedUpperBounds = new ArrayList<TypeInfo>();
    for (TypeInfo upperBound : wildcardType.getUpperBounds()) {
      resolvedUpperBounds.add(resolve(context, upperBound));
    }
    final ArrayList<TypeInfo> resolvedLowerBounds = new ArrayList<TypeInfo>();
    for (TypeInfo lowerBound : wildcardType.getLowerBounds()) {
      resolvedLowerBounds.add(resolve(context, lowerBound));
    }
    return new WildcardTypeInfo() {
      public List<TypeInfo> getUpperBounds() {
        return resolvedUpperBounds;
      }
      public List<TypeInfo> getLowerBounds() {
        return resolvedLowerBounds;
      }
      public int hashCode() {
        throw new UnsupportedOperationException();
      }
      public boolean equals(Object obj) {
        throw new UnsupportedOperationException();
      }
      public Object getType() {
        throw new UnsupportedOperationException();
      }
      public void accept(Visitor visitor) {
        throw new UnsupportedOperationException();
      }
      public boolean isCovariantWith(TypeInfo ti) {
        throw new UnsupportedOperationException();
      }
    };
  }

  static TypeInfo resolve(ClassTypeInfo context, ArrayTypeInfo genericArrayType) {
    TypeInfo componentType = genericArrayType.getComponentType();
    final TypeInfo resolvedComponentType = resolve(context, componentType);
    if (resolvedComponentType.equals(componentType)) {
      return genericArrayType;
    } else {
      return new ArrayTypeInfo() {
        public TypeInfo getComponentType() {
          return resolvedComponentType;
        }
        public Object getType() {
          throw new UnsupportedOperationException();
        }
        public void accept(Visitor visitor) {
          throw new UnsupportedOperationException();
        }
        public boolean isCovariantWith(TypeInfo ti) {
          throw new UnsupportedOperationException();
        }
      };
    }
  }

  static <T, M> TypeInfo resolve(ClassTypeInfo context, ClassTypeInfo classType) {
    return classType;
  }

  static <T, M> TypeInfo resolve(ClassTypeInfo context, final ParameterizedTypeInfo parameterizedType) {
    TypeInfo rawType = parameterizedType.getRawType();
    if (rawType instanceof ClassTypeInfo) {
      List<? extends TypeInfo> typeArguments = parameterizedType.getTypeArguments();
      final ArrayList<TypeInfo> resolvedTypeArguments = new ArrayList<TypeInfo>();
      for (TypeInfo typeArgument : typeArguments) {
        TypeInfo resolvedTypeArgument = resolve(context, typeArgument);
        resolvedTypeArguments.add(resolvedTypeArgument);
      }
      return new ParameterizedTypeInfo() {
        public TypeInfo getRawType() {
          return parameterizedType.getRawType();
        }
        public List<TypeInfo> getTypeArguments() {
          return resolvedTypeArguments;
        }
        public TypeInfo getOwnerType() {
          return parameterizedType.getOwnerType();
        }
        public String toString() {
          return "ParameterizedType[rawType" + getRawType() + ",typeArguments=" + getTypeArguments() + "]";
        }
        public int hashCode() {
          throw new UnsupportedOperationException();
        }
        public boolean equals(Object obj) {
          throw new UnsupportedOperationException();
        }
        public Object getType() {
          throw new UnsupportedOperationException();
        }
        public void accept(Visitor visitor) {
          throw new UnsupportedOperationException();
        }
        public boolean isCovariantWith(TypeInfo ti) {
          throw new UnsupportedOperationException();
        }
      };
    } else {
      throw new UnsupportedOperationException();
    }
  }

  static <T, M> TypeInfo resolve(ClassTypeInfo context, TypeVariableInfo type) {
    return resolve((TypeInfo)context, type);
  }

  static <T, M> TypeInfo resolve(TypeInfo context, TypeVariableInfo typeVariable) {
    if (context instanceof ParameterizedTypeInfo) {
      ParameterizedTypeInfo parameterizedContext = (ParameterizedTypeInfo)context;
      TypeInfo rawType = parameterizedContext.getRawType();
      if (!rawType.equals(typeVariable.getGenericDeclaration())) {
        TypeInfo resolvedTypeVariable = resolve(rawType, typeVariable);
        if (resolvedTypeVariable instanceof TypeVariableInfo) {
          if (resolvedTypeVariable.equals(typeVariable)) {
            return resolvedTypeVariable;
          } else {
            return resolve(context, (TypeVariableInfo)resolvedTypeVariable);
          }
        } else {
          return resolvedTypeVariable;
        }
      } else {
        int index = 0;
        for (TypeVariableInfo typeVariableParameter : typeVariable.getGenericDeclaration().getTypeParameters()) {
          if (typeVariable.equals(typeVariableParameter)) {
            return parameterizedContext.getTypeArguments().get(index);
          }
        }
        throw new AssertionError();
      }
    } else if (context instanceof ClassTypeInfo) {
      ClassTypeInfo classContext = (ClassTypeInfo)context;
      for (TypeInfo implementedInterface : classContext.getInterfaces()) {
        TypeInfo resolvedType = resolve(implementedInterface, typeVariable);
        if (!resolvedType.equals(typeVariable)) {
          return resolvedType;
        }
      }
      TypeInfo superClass = classContext.getSuperType();
      if (superClass != null) {
        return resolve(superClass, typeVariable);
      } else {
        return typeVariable;
      }
    } else {
      throw new UnsupportedOperationException();
    }
  }
}
