/**
 * Copyright 2005-2010 Noelios Technologies.
 * 
 * The contents of this file are subject to the terms of one of the following
 * open source licenses: LGPL 3.0 or LGPL 2.1 or CDDL 1.0 or EPL 1.0 (the
 * "Licenses"). You can select the license that you prefer but you may not use
 * this file except in compliance with one of these Licenses.
 * 
 * You can obtain a copy of the LGPL 3.0 license at
 * http://www.opensource.org/licenses/lgpl-3.0.html
 * 
 * You can obtain a copy of the LGPL 2.1 license at
 * http://www.opensource.org/licenses/lgpl-2.1.php
 * 
 * You can obtain a copy of the CDDL 1.0 license at
 * http://www.opensource.org/licenses/cddl1.php
 * 
 * You can obtain a copy of the EPL 1.0 license at
 * http://www.opensource.org/licenses/eclipse-1.0.php
 * 
 * See the Licenses for the specific language governing permissions and
 * limitations under the Licenses.
 * 
 * Alternatively, you can obtain a royalty free commercial license with less
 * limitations, transferable or non-transferable, directly at
 * http://www.noelios.com/products/restlet-engine
 * 
 * Restlet is a registered trademark of Noelios Technologies.
 */

package org.restlet.ext.sip.internal;

import org.restlet.Server;
import org.restlet.data.Parameter;
import org.restlet.engine.connector.Connection;
import org.restlet.engine.connector.ServerOutboundWay;
import org.restlet.engine.header.DateWriter;
import org.restlet.engine.header.HeaderConstants;
import org.restlet.ext.sip.SipRecipientInfo;
import org.restlet.ext.sip.SipRequest;
import org.restlet.ext.sip.SipResponse;
import org.restlet.util.Series;

/**
 * SIP server outbound way.
 * 
 * @author Jerome Louvel
 */
public class SipServerOutboundWay extends ServerOutboundWay {

    /**
     * Constructor.
     * 
     * @param connection
     *            The parent connection.
     */
    public SipServerOutboundWay(Connection<Server> connection) {
        super(connection);
    }

    @Override
    protected void addResponseHeaders(Series<Parameter> headers) {
        SipRequest sipRequest = (SipRequest) getMessage().getRequest();
        SipResponse sipResponse = (SipResponse) getMessage();

        if (!sipResponse.getSipRecipientsInfo().isEmpty()) {
            for (SipRecipientInfo recipient : sipResponse
                    .getSipRecipientsInfo()) {
                // Generate one Via header per recipient
                headers.add(HeaderConstants.HEADER_VIA,
                        SipRecipientInfoWriter.write(recipient));
            }
        }

        if (sipRequest.getCallId() != null) {
            headers.add(SipConstants.HEADER_CALL_ID, sipRequest.getCallId());
        }

        if (sipRequest.getCallSequence() != null) {
            headers.add(SipConstants.HEADER_CALL_SEQ,
                    sipRequest.getCallSequence());
        }

        if (sipRequest.getFrom() != null) {
            headers.add(HeaderConstants.HEADER_FROM,
                    AddressWriter.write(sipRequest.getFrom()));
        }

        if (sipRequest.getTo() != null) {
            headers.add(SipConstants.HEADER_TO,
                    AddressWriter.write(sipRequest.getTo()));
        }

        if (sipResponse.getAlertInfo() != null) {
            headers.add(SipConstants.HEADER_ALERT_INFO,
                    AddressWriter.write(sipResponse.getAlertInfo()));
        }

        if (!sipResponse.getAllowedEventTypes().isEmpty()) {
            headers.add(SipConstants.HEADER_ALLOW_EVENTS,
                    EventTypeWriter.write(sipResponse.getAllowedEventTypes()));
        }

        if (!sipResponse.getCalleeInfo().isEmpty()) {
            headers.add(SipConstants.HEADER_CALL_INFO,
                    AddressWriter.write(sipResponse.getCalleeInfo()));
        }

        if (!sipResponse.getContact().isEmpty()) {
            headers.add(SipConstants.HEADER_CONTACT,
                    ContactInfoWriter.write(sipResponse.getContact()));
        }

        if (sipResponse.getErrorInfo() != null) {
            headers.add(SipConstants.HEADER_ERROR_INFO,
                    AddressWriter.write(sipResponse.getErrorInfo()));
        }

        if (sipResponse.getEvent() != null) {
            headers.add(SipConstants.HEADER_EVENT,
                    EventWriter.write(sipResponse.getEvent()));
        }

        if (sipResponse.getMimeVersion() != null) {
            headers.add(SipConstants.HEADER_MIME_VERSION,
                    sipResponse.getMimeVersion());
        }

        if (sipResponse.getMinExpires() != null) {
            headers.add(SipConstants.HEADER_MIN_EXPIRES,
                    sipResponse.getMinExpires());
        }

        if (sipResponse.getOrganization() != null) {
            headers.add(SipConstants.HEADER_ORGANIZATION,
                    sipResponse.getOrganization());
        }

        if (!sipResponse.getRecordedRoutes().isEmpty()) {
            headers.add(SipConstants.HEADER_RECORD_ROUTE,
                    AddressWriter.write(sipResponse.getRecordedRoutes()));
        }

        if (sipResponse.getReplyTo() != null) {
            headers.add(SipConstants.HEADER_REPLY_TO,
                    AddressWriter.write(sipResponse.getReplyTo()));
        }

        if (sipResponse.getRetryAfter() != null) {
            headers.add(SipConstants.HEADER_RETRY_AFTER,
                    DateWriter.write(sipResponse.getRetryAfter()));
        }

        if (sipResponse.getSipTag() != null) {
            headers.add(SipConstants.HEADER_SIP_ETAG, sipResponse.getSipTag()
                    .format());
        }

        if (!sipResponse.getSupported().isEmpty()) {
            headers.add(SipConstants.HEADER_SUPPORTED,
                    OptionTagWriter.write(sipResponse.getSupported()));
        }

        if (!sipResponse.getUnsupported().isEmpty()) {
            headers.add(SipConstants.HEADER_UNSUPPORTED,
                    OptionTagWriter.write(sipResponse.getUnsupported()));
        }

        super.addResponseHeaders(headers);
    }

}
