/**
 * Copyright 2005-2010 Noelios Technologies.
 * 
 * The contents of this file are subject to the terms of one of the following
 * open source licenses: LGPL 3.0 or LGPL 2.1 or CDDL 1.0 or EPL 1.0 (the
 * "Licenses"). You can select the license that you prefer but you may not use
 * this file except in compliance with one of these Licenses.
 * 
 * You can obtain a copy of the LGPL 3.0 license at
 * http://www.opensource.org/licenses/lgpl-3.0.html
 * 
 * You can obtain a copy of the LGPL 2.1 license at
 * http://www.opensource.org/licenses/lgpl-2.1.php
 * 
 * You can obtain a copy of the CDDL 1.0 license at
 * http://www.opensource.org/licenses/cddl1.php
 * 
 * You can obtain a copy of the EPL 1.0 license at
 * http://www.opensource.org/licenses/eclipse-1.0.php
 * 
 * See the Licenses for the specific language governing permissions and
 * limitations under the Licenses.
 * 
 * Alternatively, you can obtain a royalty free commercial license with less
 * limitations, transferable or non-transferable, directly at
 * http://www.noelios.com/products/restlet-engine
 * 
 * Restlet is a registered trademark of Noelios Technologies.
 */

package org.restlet.util;

import java.util.Map;

import org.restlet.Request;
import org.restlet.Response;
import org.restlet.engine.util.CallResolver;
import org.restlet.engine.util.MapResolver;

/**
 * Resolves a name into a value.
 * 
 * @author Jerome Louvel
 */
public abstract class Resolver<T> {

    /**
     * Creates a resolver that is based on a given map.
     * 
     * @param map
     *            Map between names and values.
     * @return The map resolver.
     */
    public static Resolver<?> createResolver(Map<String, ?> map) {
        return new MapResolver(map);
    }

    /**
     * Creates a resolver that is based on a call (request, response couple). It
     * first looks up the response attributes, then the request attributes and
     * finally the variables below:
     * 
     * <table>
     * <tr>
     * <th>Model property</th>
     * <th>Variable name</th>
     * <th>Content type</th>
     * </tr>
     * <tr>
     * <td>request.confidential</td>
     * <td>c</td>
     * <td>boolean (true|false)</td>
     * </tr>
     * <tr>
     * <td>request.clientInfo.address</td>
     * <td>cia</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>request.clientInfo.upstreamAddress</td>
     * <td>ciua</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>request.clientInfo.agent</td>
     * <td>cig</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>request.challengeResponse.identifier</td>
     * <td>cri</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>request.challengeResponse.scheme</td>
     * <td>crs</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>request.date</td>
     * <td>d</td>
     * <td>Date (HTTP format)</td>
     * </tr>
     * <tr>
     * <td>request.entity.characterSet</td>
     * <td>ecs</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>response.entity.characterSet</td>
     * <td>ECS</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>request.entity.encoding</td>
     * <td>ee</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>response.entity.encoding</td>
     * <td>EE</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>request.entity.expirationDate</td>
     * <td>eed</td>
     * <td>Date (HTTP format)</td>
     * </tr>
     * <tr>
     * <td>response.entity.expirationDate</td>
     * <td>EED</td>
     * <td>Date (HTTP format)</td>
     * </tr>
     * <tr>
     * <td>request.entity.language</td>
     * <td>el</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>response.entity.language</td>
     * <td>EL</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>request.entity.modificationDate</td>
     * <td>emd</td>
     * <td>Date (HTTP format)</td>
     * </tr>
     * <tr>
     * <td>response.entity.modificationDate</td>
     * <td>EMD</td>
     * <td>Date (HTTP format)</td>
     * </tr>
     * <tr>
     * <td>request.entity.mediaType</td>
     * <td>emt</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>response.entity.mediaType</td>
     * <td>EMT</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>request.entity.size</td>
     * <td>es</td>
     * <td>Integer</td>
     * </tr>
     * <tr>
     * <td>response.entity.size</td>
     * <td>ES</td>
     * <td>Integer</td>
     * </tr>
     * <tr>
     * <td>request.entity.tag</td>
     * <td>et</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>response.entity.tag</td>
     * <td>ET</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>request.referrerRef</td>
     * <td>f*</td>
     * <td>Reference (see table below variable name sub-parts)</td>
     * </tr>
     * <tr>
     * <td>request.hostRef</td>
     * <td>h*</td>
     * <td>Reference (see table below variable name sub-parts)</td>
     * </tr>
     * <tr>
     * <td>request.method</td>
     * <td>m</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>request.rootRef</td>
     * <td>o*</td>
     * <td>Reference (see table below variable name sub-parts)</td>
     * </tr>
     * <tr>
     * <td>request.protocol</td>
     * <td>p</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>request.resourceRef</td>
     * <td>r*</td>
     * <td>Reference (see table below variable name sub-parts)</td>
     * </tr>
     * <tr>
     * <td>response.redirectRef</td>
     * <td>R*</td>
     * <td>Reference (see table below variable name sub-parts)</td>
     * </tr>
     * <tr>
     * <td>response.status</td>
     * <td>S</td>
     * <td>Integer</td>
     * </tr>
     * <tr>
     * <td>response.serverInfo.address</td>
     * <td>SIA</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>response.serverInfo.agent</td>
     * <td>SIG</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>response.serverInfo.port</td>
     * <td>SIP</td>
     * <td>Integer</td>
     * </tr>
     * </table>
     * <br>
     * 
     * Below is the list of name sub-parts, for Reference variables, that can
     * replace the asterix in the variable names above:<br>
     * <br>
     * 
     * <table>
     * <tr>
     * <th>Reference property</th>
     * <th>Sub-part name</th>
     * <th>Content type</th>
     * </tr>
     * <tr>
     * <td>authority</td>
     * <td>a</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>baseRef</td>
     * <td>b*</td>
     * <td>Reference</td>
     * </tr>
     * <tr>
     * <td>relativePart</td>
     * <td>e</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>fragment</td>
     * <td>f</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>hostIdentifier</td>
     * <td>h</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>identifier</td>
     * <td>i</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>path</td>
     * <td>p</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>query</td>
     * <td>q</td>
     * <td>String</td>
     * </tr>
     * <tr>
     * <td>remainingPart</td>
     * <td>r</td>
     * <td>String</td>
     * </tr>
     * </table>
     * 
     * @param request
     *            The request.
     * @param response
     *            The response.
     * @return The call resolver.
     */
    public static Resolver<?> createResolver(Request request, Response response) {
        return new CallResolver(request, response);
    }

    /**
     * Resolves a name into a value.
     * 
     * @param name
     *            The name to resolve.
     * @return The resolved value.
     */
    public abstract T resolve(String name);

}
