/*
 * Copyright 2014-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.data.rest.webmvc.support;

import jakarta.servlet.http.HttpServletRequest;

import org.jspecify.annotations.Nullable;

import org.springframework.core.MethodParameter;
import org.springframework.http.HttpMethod;
import org.springframework.web.bind.support.WebDataBinderFactory;
import org.springframework.web.context.request.NativeWebRequest;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.method.support.ModelAndViewContainer;

/**
 * {@link HandlerMethodArgumentResolver} to provide {@link HttpMethod} instances for innjection into MVC controller
 * methods.
 *
 * @author Oliver Gierke
 */
public enum HttpMethodHandlerMethodArgumentResolver implements HandlerMethodArgumentResolver {

	INSTANCE;

	@Override
	public boolean supportsParameter(MethodParameter parameter) {
		return HttpMethod.class.equals(parameter.getParameterType());
	}

	@Override
	public HttpMethod resolveArgument(MethodParameter parameter, @Nullable ModelAndViewContainer mavContainer,
			NativeWebRequest webRequest, @Nullable WebDataBinderFactory binderFactory) throws Exception {

		HttpServletRequest httpServletRequest = webRequest.getNativeRequest(HttpServletRequest.class);

		if (httpServletRequest == null) {
			throw new IllegalStateException("No HttpServletRequest available to resolve HttpMethod argument");
		}

		return HttpMethod.valueOf(httpServletRequest.getMethod().trim().toUpperCase());
	}
}
