/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.xwiki.model.validation.internal;

import java.util.Collections;
import java.util.Set;

import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;

import org.slf4j.Logger;
import org.xwiki.component.annotation.Component;
import org.xwiki.component.manager.ComponentLookupException;
import org.xwiki.component.manager.ComponentManager;
import org.xwiki.model.validation.EntityNameValidation;
import org.xwiki.model.validation.EntityNameValidationConfiguration;
import org.xwiki.model.validation.EntityNameValidationManager;

/**
 * Default manager for name strategies.
 *
 * @version $Id: 8c645f881419c62d6562d1c21fec052a8268678c $
 * @since 12.0RC1
 */
@Component
@Singleton
public class DefaultEntityNameValidationManager implements EntityNameValidationManager
{
    @Inject
    private ComponentManager componentManager;

    @Inject
    private EntityNameValidationConfiguration entityNameValidationConfiguration;

    @Inject
    private ReplaceCharacterEntityNameValidationConfiguration replaceCharacterEntityNameValidationConfiguration;

    @Inject
    private Logger logger;

    @Inject
    @Named(ReplaceCharacterEntityNameValidation.COMPONENT_NAME)
    private EntityNameValidation replaceCharacterValidator;

    @Override
    public EntityNameValidation getEntityReferenceNameStrategy()
    {
        return getEntityReferenceNameStrategy(this.entityNameValidationConfiguration.getEntityNameValidation());
    }

    @Override
    public EntityNameValidation getEntityReferenceNameStrategy(String hint)
    {
        try {
            return this.componentManager.getInstance(EntityNameValidation.class, hint);
        } catch (ComponentLookupException e) {
            this.logger.error("Error while getting the EntityReferenceNameStrategy with hint [{}]", hint, e);
            return null;
        }
    }

    @Override
    public Set<String> getAvailableEntityNameValidations()
    {
        try {
            return this.componentManager.getInstanceMap(EntityNameValidation.class).keySet();
        } catch (ComponentLookupException e) {
            this.logger.error("Error while getting the instance map of the EntityNameValidation", e);
            return Collections.emptySet();
        }
    }

    @Override
    public void resetStrategies()
    {
        ((ReplaceCharacterEntityNameValidation) this.replaceCharacterValidator)
            .setReplacementCharacters(this.replaceCharacterEntityNameValidationConfiguration
                .getCharacterReplacementMap());
    }
}
