/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information collected from managed instances based on your inventory policy document
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InventoryItem implements SdkPojo, Serializable, ToCopyableBuilder<InventoryItem.Builder, InventoryItem> {
    private static final SdkField<String> TYPE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TypeName").getter(getter(InventoryItem::typeName)).setter(setter(Builder::typeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TypeName").build()).build();

    private static final SdkField<String> SCHEMA_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SchemaVersion").getter(getter(InventoryItem::schemaVersion)).setter(setter(Builder::schemaVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchemaVersion").build()).build();

    private static final SdkField<String> CAPTURE_TIME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CaptureTime").getter(getter(InventoryItem::captureTime)).setter(setter(Builder::captureTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CaptureTime").build()).build();

    private static final SdkField<String> CONTENT_HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContentHash").getter(getter(InventoryItem::contentHash)).setter(setter(Builder::contentHash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentHash").build()).build();

    private static final SdkField<List<Map<String, String>>> CONTENT_FIELD = SdkField
            .<List<Map<String, String>>> builder(MarshallingType.LIST)
            .memberName("Content")
            .getter(getter(InventoryItem::content))
            .setter(setter(Builder::content))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Content").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Map<String, String>> builder(MarshallingType.MAP)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    MapTrait.builder()
                                                            .keyLocationName("key")
                                                            .valueLocationName("value")
                                                            .valueFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("value").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<Map<String, String>> CONTEXT_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Context")
            .getter(getter(InventoryItem::context))
            .setter(setter(Builder::context))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Context").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_NAME_FIELD,
            SCHEMA_VERSION_FIELD, CAPTURE_TIME_FIELD, CONTENT_HASH_FIELD, CONTENT_FIELD, CONTEXT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String typeName;

    private final String schemaVersion;

    private final String captureTime;

    private final String contentHash;

    private final List<Map<String, String>> content;

    private final Map<String, String> context;

    private InventoryItem(BuilderImpl builder) {
        this.typeName = builder.typeName;
        this.schemaVersion = builder.schemaVersion;
        this.captureTime = builder.captureTime;
        this.contentHash = builder.contentHash;
        this.content = builder.content;
        this.context = builder.context;
    }

    /**
     * <p>
     * The name of the inventory type. Default inventory item type names start with AWS. Custom inventory type names
     * will start with Custom. Default inventory item types include the following: AWS:AWSComponent, AWS:Application,
     * AWS:InstanceInformation, AWS:Network, and AWS:WindowsUpdate.
     * </p>
     * 
     * @return The name of the inventory type. Default inventory item type names start with AWS. Custom inventory type
     *         names will start with Custom. Default inventory item types include the following: AWS:AWSComponent,
     *         AWS:Application, AWS:InstanceInformation, AWS:Network, and AWS:WindowsUpdate.
     */
    public String typeName() {
        return typeName;
    }

    /**
     * <p>
     * The schema version for the inventory item.
     * </p>
     * 
     * @return The schema version for the inventory item.
     */
    public String schemaVersion() {
        return schemaVersion;
    }

    /**
     * <p>
     * The time the inventory information was collected.
     * </p>
     * 
     * @return The time the inventory information was collected.
     */
    public String captureTime() {
        return captureTime;
    }

    /**
     * <p>
     * MD5 hash of the inventory item type contents. The content hash is used to determine whether to update inventory
     * information. The PutInventory API does not update the inventory item type contents if the MD5 hash has not
     * changed since last update.
     * </p>
     * 
     * @return MD5 hash of the inventory item type contents. The content hash is used to determine whether to update
     *         inventory information. The PutInventory API does not update the inventory item type contents if the MD5
     *         hash has not changed since last update.
     */
    public String contentHash() {
        return contentHash;
    }

    /**
     * Returns true if the Content property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasContent() {
        return content != null && !(content instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The inventory data of the inventory type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasContent()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The inventory data of the inventory type.
     */
    public List<Map<String, String>> content() {
        return content;
    }

    /**
     * Returns true if the Context property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasContext() {
        return context != null && !(context instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of associated properties for a specified inventory type. For example, with this attribute, you can specify
     * the <code>ExecutionId</code>, <code>ExecutionType</code>, <code>ComplianceType</code> properties of the
     * <code>AWS:ComplianceItem</code> type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasContext()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A map of associated properties for a specified inventory type. For example, with this attribute, you can
     *         specify the <code>ExecutionId</code>, <code>ExecutionType</code>, <code>ComplianceType</code> properties
     *         of the <code>AWS:ComplianceItem</code> type.
     */
    public Map<String, String> context() {
        return context;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeName());
        hashCode = 31 * hashCode + Objects.hashCode(schemaVersion());
        hashCode = 31 * hashCode + Objects.hashCode(captureTime());
        hashCode = 31 * hashCode + Objects.hashCode(contentHash());
        hashCode = 31 * hashCode + Objects.hashCode(content());
        hashCode = 31 * hashCode + Objects.hashCode(context());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InventoryItem)) {
            return false;
        }
        InventoryItem other = (InventoryItem) obj;
        return Objects.equals(typeName(), other.typeName()) && Objects.equals(schemaVersion(), other.schemaVersion())
                && Objects.equals(captureTime(), other.captureTime()) && Objects.equals(contentHash(), other.contentHash())
                && Objects.equals(content(), other.content()) && Objects.equals(context(), other.context());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("InventoryItem").add("TypeName", typeName()).add("SchemaVersion", schemaVersion())
                .add("CaptureTime", captureTime()).add("ContentHash", contentHash()).add("Content", content())
                .add("Context", context()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TypeName":
            return Optional.ofNullable(clazz.cast(typeName()));
        case "SchemaVersion":
            return Optional.ofNullable(clazz.cast(schemaVersion()));
        case "CaptureTime":
            return Optional.ofNullable(clazz.cast(captureTime()));
        case "ContentHash":
            return Optional.ofNullable(clazz.cast(contentHash()));
        case "Content":
            return Optional.ofNullable(clazz.cast(content()));
        case "Context":
            return Optional.ofNullable(clazz.cast(context()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InventoryItem, T> g) {
        return obj -> g.apply((InventoryItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InventoryItem> {
        /**
         * <p>
         * The name of the inventory type. Default inventory item type names start with AWS. Custom inventory type names
         * will start with Custom. Default inventory item types include the following: AWS:AWSComponent,
         * AWS:Application, AWS:InstanceInformation, AWS:Network, and AWS:WindowsUpdate.
         * </p>
         * 
         * @param typeName
         *        The name of the inventory type. Default inventory item type names start with AWS. Custom inventory
         *        type names will start with Custom. Default inventory item types include the following:
         *        AWS:AWSComponent, AWS:Application, AWS:InstanceInformation, AWS:Network, and AWS:WindowsUpdate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder typeName(String typeName);

        /**
         * <p>
         * The schema version for the inventory item.
         * </p>
         * 
         * @param schemaVersion
         *        The schema version for the inventory item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaVersion(String schemaVersion);

        /**
         * <p>
         * The time the inventory information was collected.
         * </p>
         * 
         * @param captureTime
         *        The time the inventory information was collected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder captureTime(String captureTime);

        /**
         * <p>
         * MD5 hash of the inventory item type contents. The content hash is used to determine whether to update
         * inventory information. The PutInventory API does not update the inventory item type contents if the MD5 hash
         * has not changed since last update.
         * </p>
         * 
         * @param contentHash
         *        MD5 hash of the inventory item type contents. The content hash is used to determine whether to update
         *        inventory information. The PutInventory API does not update the inventory item type contents if the
         *        MD5 hash has not changed since last update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentHash(String contentHash);

        /**
         * <p>
         * The inventory data of the inventory type.
         * </p>
         * 
         * @param content
         *        The inventory data of the inventory type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder content(Collection<? extends Map<String, String>> content);

        /**
         * <p>
         * The inventory data of the inventory type.
         * </p>
         * 
         * @param content
         *        The inventory data of the inventory type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder content(Map<String, String>... content);

        /**
         * <p>
         * A map of associated properties for a specified inventory type. For example, with this attribute, you can
         * specify the <code>ExecutionId</code>, <code>ExecutionType</code>, <code>ComplianceType</code> properties of
         * the <code>AWS:ComplianceItem</code> type.
         * </p>
         * 
         * @param context
         *        A map of associated properties for a specified inventory type. For example, with this attribute, you
         *        can specify the <code>ExecutionId</code>, <code>ExecutionType</code>, <code>ComplianceType</code>
         *        properties of the <code>AWS:ComplianceItem</code> type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder context(Map<String, String> context);
    }

    static final class BuilderImpl implements Builder {
        private String typeName;

        private String schemaVersion;

        private String captureTime;

        private String contentHash;

        private List<Map<String, String>> content = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> context = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InventoryItem model) {
            typeName(model.typeName);
            schemaVersion(model.schemaVersion);
            captureTime(model.captureTime);
            contentHash(model.contentHash);
            content(model.content);
            context(model.context);
        }

        public final String getTypeName() {
            return typeName;
        }

        @Override
        public final Builder typeName(String typeName) {
            this.typeName = typeName;
            return this;
        }

        public final void setTypeName(String typeName) {
            this.typeName = typeName;
        }

        public final String getSchemaVersion() {
            return schemaVersion;
        }

        @Override
        public final Builder schemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
            return this;
        }

        public final void setSchemaVersion(String schemaVersion) {
            this.schemaVersion = schemaVersion;
        }

        public final String getCaptureTime() {
            return captureTime;
        }

        @Override
        public final Builder captureTime(String captureTime) {
            this.captureTime = captureTime;
            return this;
        }

        public final void setCaptureTime(String captureTime) {
            this.captureTime = captureTime;
        }

        public final String getContentHash() {
            return contentHash;
        }

        @Override
        public final Builder contentHash(String contentHash) {
            this.contentHash = contentHash;
            return this;
        }

        public final void setContentHash(String contentHash) {
            this.contentHash = contentHash;
        }

        public final Collection<? extends Map<String, String>> getContent() {
            return content;
        }

        @Override
        public final Builder content(Collection<? extends Map<String, String>> content) {
            this.content = InventoryItemEntryListCopier.copy(content);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder content(Map<String, String>... content) {
            content(Arrays.asList(content));
            return this;
        }

        public final void setContent(Collection<? extends Map<String, String>> content) {
            this.content = InventoryItemEntryListCopier.copy(content);
        }

        public final Map<String, String> getContext() {
            return context;
        }

        @Override
        public final Builder context(Map<String, String> context) {
            this.context = InventoryItemContentContextCopier.copy(context);
            return this;
        }

        public final void setContext(Map<String, String> context) {
            this.context = InventoryItemContentContextCopier.copy(context);
        }

        @Override
        public InventoryItem build() {
            return new InventoryItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
