/* Jackson JSON-processor.
 *
 * Copyright (c) 2007- Tatu Saloranta, tatu.saloranta@iki.fi
 */

package tools.jackson.core;

import tools.jackson.core.io.ContentReference;

/**
 * Object that encapsulates Location information used for reporting
 * parsing (or potentially generation) errors, as well as current location
 * within input streams.
 *<p>
 * NOTE: users should be careful if using {@link #equals} implementation as
 * it may or may not compare underlying "content reference" for equality.
 * Instead if would make sense to explicitly implementing equality checks
 * using specific criteria caller desires
 * <br />
 * NOTE: in Jackson 2.x this type was named {@code JsonLocation}
 */
public class TokenStreamLocation
    implements java.io.Serializable
{
    private static final long serialVersionUID = 2L;

    /**
     * Shared immutable "N/A location" that can be returned to indicate
     * that no location information is available.
     */
    public final static TokenStreamLocation NA = new TokenStreamLocation(ContentReference.unknown(),
            -1L, -1L, -1, -1);

    private final static String NO_LOCATION_DESC = "[No location information]";

    protected final long _totalBytes;
    protected final long _totalChars;

    protected final int _lineNr;
    protected final int _columnNr;

    /**
     * Reference to input source; never null (but may be that of
     * {@link ContentReference#unknown()}).
     */
    protected final ContentReference _contentReference;

    /**
     * Lazily constructed description for source; constructed if and
     * when {@link #sourceDescription()} is called, retained.
     *
     * @since 2.13
     */
    protected transient String _sourceDescription;

    /*
    /**********************************************************************
    /* Life cycle
    /**********************************************************************
     */

    public TokenStreamLocation(ContentReference contentRef, long totalChars,
            int lineNr, int colNr)
    {
        this(contentRef, -1L, totalChars, lineNr, colNr);
    }

    public TokenStreamLocation(ContentReference contentRef, long totalBytes, long totalChars,
            int lineNr, int columnNr)
    {
        // 14-Mar-2021, tatu: Defensive programming, but also for convenience...
        if (contentRef == null) {
            contentRef = ContentReference.unknown();
        }
        _contentReference = contentRef;
        _totalBytes = totalBytes;
        _totalChars = totalChars;
        _lineNr = lineNr;
        _columnNr = columnNr;
    }

    /*
    /**********************************************************************
    /* Simple accessors
    /**********************************************************************
     */

    /**
     * Accessor for information about the original input source content is being
     * read from. Returned reference is never {@code null} but may not contain
     * useful information.
     *<p>
     * NOTE: not getter, on purpose, to avoid inclusion if serialized using
     * default Jackson serializer.
     *
     * @return Object with information about input source.
     */
    public ContentReference contentReference() {
        return _contentReference;
    }

    /**
     * Access for getting line number of this location, if available.
     * Note that line number is typically not available for binary formats.
     *
     * @return Line number of the location (1-based), if available; {@code -1} if not.
     */
    public int getLineNr() { return _lineNr; }

    /**
     * Access for getting column offset of this location, if available.
     * Note that column position is typically not available for binary formats.
     * Note: this returns an offset that is in units of input, so for {@code byte}-based
     * input sources (like {@link java.io.InputStream}) this does not take into
     * account multi-byte characters: one logical character can be 1, 2 or 3 bytes long.
     * To calculate column position in characters either {@code char}-based input
     * source (like {@link java.io.Reader}) needs to be used, or content needs to be
     * explicitly decoded.
     *
     * @return Column offset of the location (1-based), if available; {@code -1} if not.
     */
    public int getColumnNr() { return _columnNr; }

    /**
     * @return Character offset within underlying stream, reader or writer,
     *   if available; {@code -1} if not.
     */
    public long getCharOffset() { return _totalChars; }

    /**
     * @return Byte offset within underlying stream, reader or writer,
     *   if available; {@code -1} if not.
     */
    public long getByteOffset() { return _totalBytes; }

    /**
     * Accessor for getting a textual description of source reference
     * (Object returned by {@link #contentReference()}), as included in
     * description returned by {@link #toString()}.
     *<p>
     * Note: implementation will simply call
     * {@link ContentReference#buildSourceDescription()})
     *<p>
     * NOTE: not added as a "getter" to prevent it from getting serialized.
     *
     * @return Description of the source reference (see {@link #contentReference()}
     */
    public String sourceDescription() {
        // 04-Apr-2021, tatu: Construct lazily but retain
        if (_sourceDescription == null) {
            _sourceDescription = _contentReference.buildSourceDescription();
        }
        return _sourceDescription;
    }

    /**
     * Accessor for a brief summary of Location offsets (line number, column position,
     * or byte offset, if available).
     *
     * @return Description of available relevant location offsets; combination of
     *    line number and column position or byte offset
     */
    public String offsetDescription() {
        return appendOffsetDescription(new StringBuilder(40)).toString();
    }

    // @since 2.13
    public StringBuilder appendOffsetDescription(StringBuilder sb)
    {
        // 04-Apr-2021, tatu: [core#694] For binary content, we have no line
        //    number or column position indicators; try using what we do have
        //    (if anything)

        if (_contentReference.hasTextualContent()) {
            sb.append("line: ");
            // should be 1-based, but consider -1 to be canonical "got none"
            if (_lineNr >= 0) {
                sb.append(_lineNr);
            } else {
                sb.append("UNKNOWN");
            }
            sb.append(", column: ");
            if (_columnNr >= 0) { // same here
                sb.append(_columnNr);
            } else {
                sb.append("UNKNOWN");
            }
        } else {
            // 04-Apr-2021, tatu: Jackson 2.x had compatibility checks here; 3.x
            //    assumes binary content always implies byte offsets
            sb.append("byte offset: #");
            // For binary formats, total bytes should be the canonical offset
            // for token/current location
            if (_totalBytes >= 0) {
                sb.append(_totalBytes);
            } else {
                sb.append("UNKNOWN");
            }
        }
        return sb;
    }

    /*
    /**********************************************************************
    /* Standard method overrides
    /**********************************************************************
     */

    @Override
    public int hashCode()
    {
        int hash = (_contentReference == null) ? 1 : 2;
        hash ^= _lineNr;
        hash += _columnNr;
        hash ^= (int) _totalChars;
        hash += (int) _totalBytes;
        return hash;
    }

    @Override
    public boolean equals(Object other)
    {
        if (other == this) return true;
        if (other == null) return false;
        if (!(other instanceof TokenStreamLocation)) return false;
        TokenStreamLocation otherLoc = (TokenStreamLocation) other;

        if (_contentReference == null) {
            if (otherLoc._contentReference != null) return false;
        } else if (!_contentReference.equals(otherLoc._contentReference)) {
            return false;
        }

        return (_lineNr == otherLoc._lineNr)
            && (_columnNr == otherLoc._columnNr)
            && (_totalChars == otherLoc._totalChars)
            && (_totalBytes == otherLoc._totalBytes)
            ;
    }

    @Override
    public String toString()
    {
        if (this == NA) {
            return NO_LOCATION_DESC;
        }
        final String srcDesc = sourceDescription();
        StringBuilder sb = new StringBuilder(40 + srcDesc.length())
                .append("[Source: ")
                .append(srcDesc)
                .append("; ");
        return appendOffsetDescription(sb)
                .append(']')
                .toString();
    }

    public StringBuilder toString(StringBuilder sb)
    {
        if (this == NA) {
            return sb.append(NO_LOCATION_DESC);
        }
        sb.append("[Source: ")
                .append(sourceDescription())
                .append("; ");
        return appendOffsetDescription(sb)
                .append(']');
    }
}
