/*
 * Copyright (C) 2024 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.builder.dexing

import com.android.tools.r8.ClassFileResourceProvider
import com.android.tools.r8.ProgramResource
import com.android.tools.r8.origin.Origin
import com.android.tools.r8.tracereferences.TraceReferences
import com.android.tools.r8.tracereferences.TraceReferencesCheckConsumer
import com.android.tools.r8.tracereferences.TraceReferencesCommand
import com.android.tools.r8.tracereferences.TraceReferencesConsumer
import java.nio.file.Path

fun runTraceReferences(
    sourceFiles: List<Path>,
    libraryFiles: List<Path>,
    libraryInMemoryClasses: Map<String, ByteArray>,
) {
    TraceReferences.run(TraceReferencesCommand.builder().apply {
        addSourceFiles(sourceFiles)
        addLibraryFiles(libraryFiles)
        addLibraryResourceProvider(InMemoryClassFileResourceProvider(libraryInMemoryClasses))
        setConsumer(TraceReferencesCheckConsumer(TraceReferencesConsumer.emptyConsumer()))
    }.build())
}


private class InMemoryClassFileResourceProvider(private val classes: Map<String, ByteArray>) : ClassFileResourceProvider {

    private val origin: Origin = object : Origin(root()) {
        override fun part(): String = "Runtime R class that will be generated by consuming app"
    }

    override fun getClassDescriptors(): Set<String> = classes.keys

    override fun getProgramResource(descriptor: String): ProgramResource? {
        return classes[descriptor]?.let {
            ProgramResource.fromBytes(origin, ProgramResource.Kind.CF, it, setOf())
        }
    }
}
