/*
 * Copyright (C) 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.internal.cxx.settings

import com.android.build.gradle.internal.core.Abi
import com.android.build.gradle.internal.cxx.cmake.cmakeBoolean
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.ANDROID_ABI
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.ANDROID_NDK
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.ANDROID_PLATFORM
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.CMAKE_ANDROID_ARCH_ABI
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.CMAKE_ANDROID_NDK
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.CMAKE_CXX_FLAGS
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.CMAKE_C_FLAGS
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.CMAKE_EXPORT_COMPILE_COMMANDS
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.CMAKE_LIBRARY_OUTPUT_DIRECTORY
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.CMAKE_RUNTIME_OUTPUT_DIRECTORY
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.CMAKE_MAKE_PROGRAM
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.CMAKE_FIND_ROOT_PATH
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.CMAKE_SYSTEM_NAME
import com.android.build.gradle.internal.cxx.configure.CmakeProperty.CMAKE_SYSTEM_VERSION
import com.android.build.gradle.internal.cxx.configure.NdkMetaPlatforms
import com.android.build.gradle.internal.cxx.model.CxxAbiModel
import com.android.build.gradle.internal.cxx.model.shouldGeneratePrefabPackages
import com.android.build.gradle.internal.cxx.settings.Environment.NDK
import com.android.build.gradle.internal.cxx.settings.Macro.*
import com.android.build.gradle.internal.cxx.settings.PropertyValue.LookupPropertyValue
import com.android.build.gradle.internal.cxx.settings.PropertyValue.StringPropertyValue
import com.android.utils.FileUtils.join

const val TRADITIONAL_CONFIGURATION_NAME = "traditional-android-studio-cmake-environment"

/**
 * This is a CMakeSettings.json file that is equivalent to the environment CMakeServerJsonGenerator
 * traditionally has run.
 */
fun CxxAbiModel.getCmakeServerDefaultEnvironment(): CMakeSettings {
    val variables = mutableListOf(
        CMakeSettingsVariable(ANDROID_ABI.name, NDK_ABI.ref),
        CMakeSettingsVariable(ANDROID_NDK.name, NDK_DIR.ref),
        CMakeSettingsVariable(ANDROID_PLATFORM.name, NDK_PLATFORM.ref),
        CMakeSettingsVariable(CMAKE_ANDROID_ARCH_ABI.name, NDK_ABI.ref),
        CMakeSettingsVariable(CMAKE_ANDROID_NDK.name, NDK_DIR.ref),
        CMakeSettingsVariable(CMAKE_C_FLAGS.name, NDK_C_FLAGS.ref),
        CMakeSettingsVariable(CMAKE_CXX_FLAGS.name, NDK_CPP_FLAGS.ref),
        CMakeSettingsVariable(CMAKE_EXPORT_COMPILE_COMMANDS.name, "ON"),
        CMakeSettingsVariable(
            CMAKE_LIBRARY_OUTPUT_DIRECTORY.name,
            NDK_DEFAULT_LIBRARY_OUTPUT_DIRECTORY.ref
        ),
        CMakeSettingsVariable(
            CMAKE_RUNTIME_OUTPUT_DIRECTORY.name,
            NDK_DEFAULT_RUNTIME_OUTPUT_DIRECTORY.ref
        ),
        CMakeSettingsVariable(CMAKE_MAKE_PROGRAM.name, NDK_NINJA_EXECUTABLE.ref),
        CMakeSettingsVariable(CMAKE_SYSTEM_NAME.name, "Android"),
        CMakeSettingsVariable(CMAKE_SYSTEM_VERSION.name, NDK_SYSTEM_VERSION.ref)
    )

    if (shouldGeneratePrefabPackages()) {
        variables.add(CMakeSettingsVariable(CMAKE_FIND_ROOT_PATH.name, NDK_PREFAB_PATH.ref))
    }

    return CMakeSettings(
        configurations = listOf(
            CMakeSettingsConfiguration(
                name = TRADITIONAL_CONFIGURATION_NAME,
                description = "Configuration generated by Android Gradle Plugin",
                inheritEnvironments = listOf("ndk"),
                generator = "Ninja",
                buildRoot = NDK_BUILD_ROOT.ref,
                cmakeExecutable = NDK_CMAKE_EXECUTABLE.ref,
                cmakeToolchain = NDK_CMAKE_TOOLCHAIN.ref,
                configurationType = NDK_DEFAULT_BUILD_TYPE.ref,
                variables = variables
            )
        )
    )
}

/**
 * Information that would naturally come from the NDK.
 */
fun CxxAbiModel.getNdkMetaCmakeSettingsJson() : CMakeSettings {
    val environments =
        mutableMapOf<String, Map<Macro, PropertyValue>>()
    val nameTable = mutableMapOf<Macro, PropertyValue>()
    environments[NDK.environment] = nameTable

    nameTable[NDK_MIN_PLATFORM] = LookupPropertyValue { resolveMacroValue(NDK_MIN_PLATFORM) }
    nameTable[NDK_MAX_PLATFORM] = LookupPropertyValue { resolveMacroValue(NDK_MAX_PLATFORM) }
    nameTable[NDK_CMAKE_TOOLCHAIN] = LookupPropertyValue { resolveMacroValue(NDK_CMAKE_TOOLCHAIN) }

    // Per-ABI environments
    for(abiValue in Abi.values()) {
        val abiInfo by lazy {
            variant.module.ndkMetaAbiList.singleOrNull {
                it.abi == abiValue
            }
        }
        val abiNameTable = mutableMapOf<Macro, PropertyValue>()
        environments[Environment.NDK_ABI.environment.replace(NDK_ABI.ref, abiValue.tag)] = abiNameTable
        abiNameTable[NDK_ABI_BITNESS] = LookupPropertyValue {
            abiInfo?.bitness?.toString() ?: "$abiValue" }
        abiNameTable[NDK_ABI_IS_64_BITS] = LookupPropertyValue {
            if (abiInfo != null) cmakeBoolean(abiInfo?.bitness == 64) else ""
        }
        abiNameTable[NDK_ABI_IS_DEPRECATED] = LookupPropertyValue {
            if (abiInfo != null) cmakeBoolean(abiInfo!!.isDeprecated) else ""
        }
        abiNameTable[NDK_ABI_IS_DEFAULT] = LookupPropertyValue {
            if (abiInfo != null) cmakeBoolean(abiInfo!!.isDefault) else ""
        }
    }

    // Per-platform environments. In order to be lazy, promise future platform versions and return
    // blank for PLATFORM_CODE when they are evaluated and don't exist.
    for(potentialPlatform in NdkMetaPlatforms.potentialPlatforms) {
        val platformNameTable = mutableMapOf<Macro, PropertyValue>()
        val environmentName =
            Environment.NDK_PLATFORM.environment.replace(NDK_SYSTEM_VERSION.ref, potentialPlatform.toString())
        environments[environmentName] = platformNameTable
        platformNameTable[NDK_SYSTEM_VERSION] = StringPropertyValue("$potentialPlatform")
        platformNameTable[NDK_PLATFORM] = StringPropertyValue("android-$potentialPlatform")
        platformNameTable[NDK_PLATFORM_CODE] = LookupPropertyValue {
            variant.module.ndkMetaPlatforms!!.aliases.toList().lastOrNull {
                (_, platform) -> platform == potentialPlatform
            } ?.first ?: ""
        }
    }

    val settingsEnvironments =
        environments.map { (name, properties) ->
        val environment = properties.map { it.key.environment }.toSet().single()
        CMakeSettingsEnvironment(
            namespace = environment.namespace,
            environment = name,
            inheritEnvironments = environment.inheritEnvironments.map { it.environment },
            properties = properties
                .map { (macro, property) -> Pair(macro.tag, property) }
                .toMap()
        )
    }
    return CMakeSettings(
        environments = settingsEnvironments,
        configurations = listOf()
    )
}

/**
 * Builds the default android hosting environment.
 */
fun CxxAbiModel.getAndroidGradleCmakeSettings() : CMakeSettings {
    val nameTable = mutableMapOf<Macro, PropertyValue>()
    nameTable[NDK_ABI] = LookupPropertyValue { abi.tag }
    nameTable[NDK_SDK_DIR] = LookupPropertyValue { resolveMacroValue(NDK_SDK_DIR) }
    nameTable[NDK_DIR] = LookupPropertyValue { this.resolveMacroValue(NDK_DIR) }
    nameTable[NDK_CMAKE_EXECUTABLE] = LookupPropertyValue { this.resolveMacroValue(NDK_CMAKE_EXECUTABLE) }
    nameTable[NDK_NINJA_EXECUTABLE] = LookupPropertyValue { this.resolveMacroValue(NDK_NINJA_EXECUTABLE) }
    nameTable[NDK_VERSION] = LookupPropertyValue { this.resolveMacroValue(NDK_VERSION) }
    nameTable[NDK_VERSION_MAJOR] = LookupPropertyValue {
        variant.module.ndkVersion.major.toString()
    }
    nameTable[NDK_VERSION_MINOR] = LookupPropertyValue {
        variant.module.ndkVersion.minor.toString()
    }
    nameTable[NDK_PROJECT_DIR] = LookupPropertyValue { this.resolveMacroValue(NDK_PROJECT_DIR) }
    nameTable[NDK_MODULE_DIR] = LookupPropertyValue { this.resolveMacroValue(NDK_MODULE_DIR) }
    nameTable[NDK_VARIANT_NAME] = LookupPropertyValue { variant.variantName }
    nameTable[NDK_MODULE_NAME] = LookupPropertyValue {
        variant.module.gradleModulePathName
    }
    nameTable[NDK_BUILD_ROOT] = LookupPropertyValue { this.resolveMacroValue(NDK_BUILD_ROOT) }
    nameTable[NDK_DEFAULT_LIBRARY_OUTPUT_DIRECTORY] = LookupPropertyValue {
        this.resolveMacroValue(NDK_DEFAULT_LIBRARY_OUTPUT_DIRECTORY)
    }
    nameTable[NDK_DEFAULT_RUNTIME_OUTPUT_DIRECTORY] = LookupPropertyValue {
        this.resolveMacroValue(NDK_DEFAULT_RUNTIME_OUTPUT_DIRECTORY)
    }
    nameTable[NDK_DEFAULT_BUILD_TYPE] = LookupPropertyValue { resolveMacroValue(NDK_DEFAULT_BUILD_TYPE) }
    nameTable[ENV_THIS_FILE_DIR] = LookupPropertyValue { this.resolveMacroValue(
        ENV_THIS_FILE_DIR
    ) }
    nameTable[ENV_THIS_FILE] = LookupPropertyValue { this.resolveMacroValue(ENV_THIS_FILE) }
    nameTable[NDK_ANDROID_GRADLE_IS_HOSTING] = StringPropertyValue("1")
    nameTable[ENV_PROJECT_DIR] = LookupPropertyValue { this.resolveMacroValue(
        ENV_PROJECT_DIR
    ) }
    nameTable[ENV_WORKSPACE_ROOT] = LookupPropertyValue { resolveMacroValue(ENV_WORKSPACE_ROOT) }
    nameTable[NDK_PREFAB_PATH] = LookupPropertyValue { this.resolveMacroValue(NDK_PREFAB_PATH) }
    val environments = nameTable
        .toList()
        .groupBy { (macro,_) -> macro.environment }
        .map { (environment, properties) ->
            CMakeSettingsEnvironment(
                namespace = environment.namespace,
                environment = environment.environment,
                inheritEnvironments = environment.inheritEnvironments.map { it.environment },
                properties = properties
                    .map { (macro, property) -> Pair(macro.tag, property) }
                    .toMap()
            )
        }

    return CMakeSettings(
        environments = environments,
        configurations = listOf()
    )
}

/**
 * Gather CMake settings from different locations.
 */
fun CxxAbiModel.gatherCMakeSettingsFromAllLocations() : CMakeSettings {
    val settings = mutableListOf<CMakeSettings>()

    // Load the user's CMakeSettings.json if there is one.
    val userSettings = join(variant.module.makeFile.parentFile, "CMakeSettings.json")
    if (userSettings.isFile) {
        settings += createCmakeSettingsJsonFromFile(userSettings)
    }

    // TODO this needs to include environment variables as well.

    // Add the synthetic traditional environment.
    settings += getCmakeServerDefaultEnvironment()

    // Construct settings for gradle hosting environment.
    settings += getAndroidGradleCmakeSettings()

    // Construct synthetic settings for the NDK
    settings += getNdkMetaCmakeSettingsJson()

    return mergeCMakeSettings(*settings.toTypedArray())
}
