/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.android.build.gradle.tasks

import com.android.build.gradle.internal.cxx.configure.convertCMakeToCompileCommandsBin
import com.android.build.gradle.internal.cxx.json.NativeBuildConfigValue
import com.android.build.gradle.internal.cxx.json.PlainFileGsonTypeAdaptor
import com.android.build.gradle.internal.cxx.logging.errorln
import com.android.build.gradle.internal.cxx.logging.warnln
import com.android.build.gradle.internal.cxx.model.CxxAbiModel
import com.android.build.gradle.internal.cxx.model.compileCommandsJsonBinFile
import com.android.build.gradle.internal.cxx.model.compileCommandsJsonFile
import com.android.build.gradle.internal.cxx.model.getBuildCommandArguments
import com.android.build.gradle.internal.cxx.model.ifLogNativeConfigureToLifecycle
import com.android.build.gradle.internal.cxx.model.jsonFile
import com.android.build.gradle.internal.cxx.model.metadataGenerationCommandFile
import com.android.build.gradle.internal.cxx.model.metadataGenerationStderrFile
import com.android.build.gradle.internal.cxx.model.metadataGenerationStdoutFile
import com.android.build.gradle.internal.cxx.process.createProcessOutputJunction
import com.android.ide.common.process.ProcessInfoBuilder
import com.android.utils.cxx.CxxDiagnosticCode.CMAKE_FEATURE_NOT_SUPPORTED_FOR_VERSION
import com.android.utils.cxx.CxxDiagnosticCode.CMAKE_VERSION_IS_UNSUPPORTED
import com.android.utils.tokenizeCommandLineToEscaped
import com.google.gson.GsonBuilder
import com.google.wireless.android.sdk.stats.GradleBuildVariant
import com.google.wireless.android.sdk.stats.GradleNativeAndroidModule.NativeBuildSystemType.CMAKE
import org.gradle.process.ExecOperations
import java.io.File
import java.lang.RuntimeException
import java.nio.charset.StandardCharsets

/**
 * This strategy uses the older custom CMake (version 3.6) that directly generates the JSON file as
 * part of project configuration.
 */
internal class CmakeAndroidNinjaExternalNativeJsonGenerator(
    abi: CxxAbiModel,
    variantBuilder: GradleBuildVariant.Builder?
) : ExternalNativeJsonGenerator(abi, variantBuilder) {
    init {
        variantBuilder?.nativeBuildSystemType = CMAKE
        cmakeMakefileChecks(abi.variant)
    }

    override fun executeProcess(ops: ExecOperations, abi: CxxAbiModel) {
        if(abi.getBuildCommandArguments().isNotEmpty()){
            warnln(
                CMAKE_FEATURE_NOT_SUPPORTED_FOR_VERSION,
                "buildCommandArgs from CMakeSettings.json is not supported for CMake version 3.6 and below."
            )
        }
        val logPrefix = "${abi.variant.variantName}|${abi.abi.tag} :"
        createProcessOutputJunction(
            abi.metadataGenerationCommandFile,
            abi.metadataGenerationStdoutFile,
            abi.metadataGenerationStderrFile,
            getProcessBuilder(abi),
            logPrefix)
            .logStderr()
            .logStdout()
            .logFullStdout(abi.variant.ifLogNativeConfigureToLifecycle { true } ?: false)
            .execute(ops::exec)

        postProcessForkCmakeOutput(abi)
    }

    private fun postProcessForkCmakeOutput(abi: CxxAbiModel) {
        // Process the android_gradle_build.json generated by our fork of CMake and swap the
        // buildTargetsCommand and cleanCommands with the parsed version.
        val gson = GsonBuilder()
            .registerTypeAdapter(File::class.java, PlainFileGsonTypeAdaptor())
            .disableHtmlEscaping()
            .setPrettyPrinting()
            .create()
        abi.jsonFile.takeIf { it.isFile }?.reader(StandardCharsets.UTF_8)?.use {
            val nativeBuildConfigValue = gson.fromJson(it, NativeBuildConfigValue::class.java)
            nativeBuildConfigValue.cleanCommandsComponents =
                nativeBuildConfigValue.cleanCommands?.map { cleanCommand ->
                    cleanCommand.tokenizeCommandLineToEscaped()
                }
            nativeBuildConfigValue.buildTargetsCommandComponents =
                nativeBuildConfigValue.buildTargetsCommand?.tokenizeCommandLineToEscaped()
            nativeBuildConfigValue.cleanCommands = null
            nativeBuildConfigValue.buildTargetsCommand = null
            nativeBuildConfigValue.libraries =
                nativeBuildConfigValue.libraries?.mapValues { (_, library) ->
                    library.buildCommandComponents =
                        library.buildCommand?.tokenizeCommandLineToEscaped()
                    library.buildCommand = null
                    library
                }
            nativeBuildConfigValue
        }?.also { nativeBuildConfigValue ->
            abi.jsonFile.delete()
            abi.jsonFile.writer(StandardCharsets.UTF_8).use { writer ->
                gson.toJson(nativeBuildConfigValue, writer)
            }

            try {
                convertCMakeToCompileCommandsBin(
                    abi.compileCommandsJsonFile,
                    abi.compileCommandsJsonBinFile)
            } catch (e : Exception) {
                // Give the user a way to open a constructive bug
                throw RuntimeException("Please open a bug with zip of ${abi.jsonFile.parentFile.absoluteFile}", e)
            }
        }
    }

    override fun getProcessBuilder(abi: CxxAbiModel): ProcessInfoBuilder {
        val builder = ProcessInfoBuilder()
        builder.setExecutable(abi.variant.module.cmake?.cmakeExe!!)
        builder.addArgs(abi.configurationArguments)
        return builder
    }

    override fun checkPrefabConfig() {
        errorln(
            CMAKE_VERSION_IS_UNSUPPORTED,
            "Prefab cannot be used with CMake 3.6. Use CMake 3.7 or newer."
        )
    }
}
