/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.jira.startup;

import com.atlassian.jira.config.properties.JiraProperties;
import com.atlassian.jira.config.util.JiraHome;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TemporaryDirectoryCleaner {
    public static final String DISABLE_TMP_DIRECTORY_CLEANER_PROPERTY_KEY = "tmpdir.clean.on.startup.disabled";
    private static final Logger log = LoggerFactory.getLogger(TemporaryDirectoryCleaner.class);
    private static final String TMP_DIRECTORY_PROPERTY_KEY = "java.io.tmpdir";
    private static final int LOG_FILES_NOT_REMOVED_MAX_COUNT = 20;
    private final JiraHome jiraHome;
    private final JiraProperties jiraProperties;

    public TemporaryDirectoryCleaner(JiraHome jiraHome, JiraProperties jiraProperties) {
        this.jiraHome = jiraHome;
        this.jiraProperties = jiraProperties;
    }

    public void start() {
        if (Boolean.TRUE.equals(this.jiraProperties.getBoolean(DISABLE_TMP_DIRECTORY_CLEANER_PROPERTY_KEY))) {
            log.info("The temporary directory cleaner has been disabled by the system property '{}'.", (Object)DISABLE_TMP_DIRECTORY_CLEANER_PROPERTY_KEY);
            return;
        }
        File temp = this.getTemporaryDirectory().orElseThrow(() -> new IllegalStateException("The temporary directory property 'java.io.tmpdir' should be defined."));
        if (!temp.exists()) {
            log.warn("The temporary directory [{}] does not exist. Skipping cleaning.", (Object)temp.getAbsolutePath());
            return;
        }
        try {
            if (this.isNotInsideCatalinaDirectories(temp) && this.isNotInsideJiraHome(temp)) {
                log.warn("The temporary directory [{}] should be located in the Jira Home directory [{}] or in one of directories defined by '{}' or '{}' properties. Skipping cleaning.", new Object[]{temp.getAbsolutePath(), this.jiraHome.getLocalHomePath(), "catalina.home", "catalina.base"});
                return;
            }
            this.cleanDirectory(temp);
        }
        catch (IOException e) {
            log.error("Failed to clean the temporary directory located at {}", (Object)temp.getAbsolutePath(), (Object)e);
        }
    }

    private void cleanDirectory(File tmp) throws IOException {
        log.info("Cleaning the temporary directory located at {}", (Object)tmp.getAbsolutePath());
        try (Stream<Path> files = Files.list(tmp.toPath());){
            List<File> filesNotRemoved = files.map(Path::toFile).filter(path -> !FileUtils.deleteQuietly((File)path)).collect(Collectors.toList());
            this.logResult(filesNotRemoved);
        }
    }

    private void logResult(Collection<File> filesNotRemoved) {
        if (!filesNotRemoved.isEmpty()) {
            StringBuilder sb = new StringBuilder();
            filesNotRemoved.stream().limit(20L).forEach(file -> sb.append(file.getAbsolutePath()).append("\n"));
            if (filesNotRemoved.size() > 20) {
                sb.append(String.format("... and %d more.", filesNotRemoved.size() - 20));
            }
            log.warn("Finished cleaning the temporary directory, but couldn't remove the following files:\n{}", (Object)sb);
        } else {
            log.info("Finished cleaning the temporary directory.");
        }
    }

    private boolean isNotInsideJiraHome(File tmp) throws IOException {
        return !FileUtils.directoryContains((File)this.jiraHome.getLocalHome(), (File)tmp);
    }

    private boolean isNotInsideCatalinaDirectories(File tmp) throws IOException {
        return this.isNotInsideDirectoryDefinedByPropertyKey("catalina.base", tmp) && this.isNotInsideDirectoryDefinedByPropertyKey("catalina.home", tmp);
    }

    private boolean isNotInsideDirectoryDefinedByPropertyKey(String propertyKey, File tmp) throws IOException {
        Optional<File> directory = this.getFileFromProperty(propertyKey);
        if (!directory.isPresent()) {
            return true;
        }
        return !FileUtils.directoryContains((File)directory.get(), (File)tmp);
    }

    private Optional<File> getTemporaryDirectory() {
        return this.getFileFromProperty(TMP_DIRECTORY_PROPERTY_KEY);
    }

    private Optional<File> getFileFromProperty(String prop) {
        return Optional.ofNullable(this.jiraProperties.getProperty(prop)).map(File::new);
    }
}

