/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.jira.cluster.service;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.jira.cluster.ClusterManager;
import com.atlassian.jira.cluster.ClusterSettings;
import com.atlassian.jira.cluster.Node;
import com.atlassian.jira.cluster.service.NodeTimeHelper;
import com.atlassian.jira.cluster.service.analytics.ClusterStateChangedEvent;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NonAliveNodesScannerService {
    private static final Logger log = LoggerFactory.getLogger(NonAliveNodesScannerService.class);
    private static Clock clock = Clock.systemUTC();
    static final Duration WARNING_MESSAGE_PERIOD = Duration.ofMinutes(5L);
    private final ClusterManager clusterManager;
    private final NodeTimeHelper nodeTimeHelper;
    private final EventPublisher eventPublisher;
    private volatile Instant nextWarningMessageRun;
    private final Duration retentionPeriod;

    public NonAliveNodesScannerService(ClusterManager clusterManager, NodeTimeHelper nodeTimeHelper, EventPublisher eventPublisher) {
        this.clusterManager = clusterManager;
        this.nodeTimeHelper = nodeTimeHelper;
        this.eventPublisher = eventPublisher;
        this.nextWarningMessageRun = Instant.now(clock);
        this.retentionPeriod = ClusterSettings.getNotAliveNodeRetentionPeriod();
        log.info("{} Service has been registered with retention period {}", (Object)"[CLUSTER-STATE]", (Object)this.retentionPeriod);
    }

    void moveNonAliveNodesToOfflineState() {
        this.warnAboutWrongNodeStates(this.retentionPeriod);
        log.info("{} Service is starting to check the cluster state with retention period {}", (Object)"[CLUSTER-STATE]", (Object)this.retentionPeriod);
        List<String> nodesMovedToOffline = this.clusterManager.moveNodesToOfflineIfOlderThan(this.retentionPeriod);
        if (!nodesMovedToOffline.isEmpty()) {
            log.warn("{} {} nodes moved to OFFLINE state: {}.", new Object[]{"[CLUSTER-STATE]", nodesMovedToOffline.size(), nodesMovedToOffline});
        } else {
            String currentClusterState = this.getCurrentClusterStateData();
            log.info("{} Service did not find any stale ACTIVE without heartbeat nodes. {}", (Object)"[CLUSTER-STATE]", (Object)currentClusterState);
        }
        this.sendAnalytics(nodesMovedToOffline.size());
    }

    private void warnAboutWrongNodeStates(Duration retentionPeriod) {
        Instant now = Instant.now(clock);
        if (now.isAfter(this.nextWarningMessageRun)) {
            this.clusterManager.getAllNodes().stream().filter(this::isNodeActiveNonAlive).forEach(node -> this.logAboutWrongNodeState((Node)node, retentionPeriod));
            this.nextWarningMessageRun = now.plusMillis(WARNING_MESSAGE_PERIOD.toMillis());
        }
    }

    private String getCurrentClusterStateData() {
        int numberOfNodes = this.clusterManager.getAllNodes().size();
        int numberOfActiveNodes = this.clusterManager.findLiveNodes().size();
        int numberOfActiveAndNotAliveNodes = this.clusterManager.findActiveAndNotAliveNodes().size();
        int numberOfOfflineNodes = this.clusterManager.findOfflineNodes().size();
        return String.format("Current cluster state: {numberOfNodes=%d, numberOfActiveNodes=%d, numberOfActiveNotAliveNodes=%d, numberOfOfflineNodes=%d}", numberOfNodes, numberOfActiveNodes, numberOfActiveAndNotAliveNodes, numberOfOfflineNodes);
    }

    private boolean isNodeActiveNonAlive(Node node) {
        return this.clusterManager.isNodeActive(node.getNodeId()) && !this.clusterManager.isNodeAlive(node.getNodeId());
    }

    private void logAboutWrongNodeState(Node node, Duration retentionPeriod) {
        Duration nextRemoveAction = this.nodeTimeHelper.getEstimatedRetentionTime(node, retentionPeriod, clock);
        Duration nodeInState = this.nodeTimeHelper.getTimeOfNodeBeingInCurrentState(node, clock);
        log.warn("{} Node {} is in the ACTIVE state with no heartbeat for {} time. It will be move into the OFFLINE state approximately after {}. If this node should not be part of the cluster please remove it manually. See [https://confluence.atlassian.com/jirakb/remove-abandoned-or-offline-nodes-in-jira-data-center-946616137.html] for details.", new Object[]{"[CLUSTER-STATE]", node.getNodeId(), nodeInState, nextRemoveAction});
    }

    private void sendAnalytics(int numberOfAffectedNodes) {
        if (numberOfAffectedNodes > 0) {
            String actionName = ClusterStateChangedEvent.ActionNames.NODES_MOVED_TO_OFFLINE.getActionName();
            ClusterStateChangedEvent event = new ClusterStateChangedEvent(actionName, numberOfAffectedNodes);
            this.eventPublisher.publish((Object)event);
        }
    }

    @VisibleForTesting
    static void setClock(Clock clock) {
        NonAliveNodesScannerService.clock = clock;
    }
}

