/**
 * Main RapidBoard functions
 */
define("jira-agile/rapid/rapid-board", [
    "jira-agile/rapid/ajax",
    "jira-agile/rapid/ui/notification",
    "jira-agile/rapid/layout",
    "jira/util/formatter",
    "jira/util/events",
    "jira/jquery/deferred",
    "jquery",
    'underscore',
    "jira-agile/rapid/global-events",
    "jira-agile/rapid/analytics-tracker"
], function (Ajax,
             Notification,
             Layout,
             formatter,
             Events,
             Deferred,
             $,
             _,
             GlobalEvents,
             AnalyticsTracker) {
    "use strict";

    const GH = window.GH || {};
    const RapidBoard = GH.RapidBoard || {};

    RapidBoard.initialiseListeners = Deferred();
    RapidBoard.initialiseListenersDone = Deferred();

    /**
     * This API event causes the board to be reloaded.  This is so 3rd party people can
     * cause us to refresh the board WITHOUT refreshing the page
     */
    RapidBoard.API_EVENT_CAUSE_BOARD_RELOAD = "RapidBoard.causeBoardReload";

    RapidBoard.Analytics = {};

    RapidBoard.projectKey = undefined;

    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    RapidBoard.Analytics.toolsMenu = new AnalyticsTracker("RapidBoard.toolsmenu");

    /**
     * Opens the rapid board with the given rapid view id
     */
    RapidBoard.gotoRapidBoardPage = function (rapidViewId, view) {
        window.location.href = RapidBoard.getRapidBoardPageUrl(rapidViewId, view);
    };

    /**
     * Get the URL for a Board
     */
    RapidBoard.getRapidBoardPageUrl = function (rapidViewId, view) {
        // if we have a view, then go to that view specifically, otherwise, use stored settings
        var suffix = (_.isUndefined(view) ? "&useStoredSettings=true" : ("&view=" + view));
        return GH.Navigation.getCanonicRapidBoardPageUrl(rapidViewId) + suffix;
    };

    /**
     * Opens the rapid board configuration
     */
    RapidBoard.gotoRapidViewConfiguration = function (rapidViewId, tabId) {
        window.location.href = RapidBoard.getRapidViewConfigurationUrl(rapidViewId, tabId);
    };

    /**
     * Get the configuration page url, optionally for a given tabId
     */
    RapidBoard.getRapidViewConfigurationUrl = function (rapidViewId, tabId) {
        var baseUrl = Ajax.CONTEXT_PATH + '/secure/RapidView.jspa?rapidView=' + rapidViewId;
        if (tabId) {
            baseUrl = baseUrl + '&tab=' + tabId;
        }
        return baseUrl;
    };

    /**
     * Copy the specified rapid view and then open the new view's configuration
     * @param sourceRapidViewId the id of the rapid view to copy
     * @param sourceRapidViewName the name of the rapid view to copy
     */
    RapidBoard.copyRapidViewAndGotoConfiguration = function (sourceRapidViewId, sourceRapidViewName) {
        // analytics
        RapidBoard.Analytics.toolsMenu.trigger('copy'); // SAFE
        Ajax.put({
            url: '/rapidview/' + sourceRapidViewId + '/copy'
        })
            .done(function (result) {
                var message = formatter.I18n.getText(
                    'gh.rapid.view.copy.success',
                    AJS.escapeHTML(String(sourceRapidViewName)),
                    AJS.escapeHTML(String(result.success.name))
                );
                Notification.addPageLoadMessage(message);
                RapidBoard.gotoRapidViewConfiguration(result.success.id);
            });
    };

    RapidBoard.printCardsInRapidBoard = function () {
        GlobalEvents.trigger('RapidBoard.printCardsInRapidBoard');
    };

    /**
     * Get manage views page url
     */

    RapidBoard.getManageViewsPageUrl = function () {
        return Ajax.CONTEXT_PATH + '/secure/ManageRapidViews.jspa';
    };

    /**
     * Opens the manage views page
     */
    RapidBoard.gotoManageViewsPage = function () {
        window.location.href = RapidBoard.getManageViewsPageUrl();
    };

    /** Get the current page url as used in the returnUrl JIRA request parameter.
     * Note: the contextPath is NOT part of the url in this case
     */
    RapidBoard.getCurrentPageReturnUrl = function () {
        var params = RapidBoard.UrlState.getViewStateRequestParameterString();
        var url = '/secure/RapidBoard.jspa?' + params;
        return encodeURIComponent(url);
    };

    /**
     * Reloads the model from the current view state
     */
    RapidBoard.reload = function () {
        GlobalEvents.trigger('RapidBoard.reload');
    };

    /**
     * Initialise event listeners of sub-components
     */
    RapidBoard.initListeners = function (rapidBoardConfig) {

        RapidBoard.projectKey = $('meta[name="ghx-project-key"]').attr('content');

        // layout info, like viewport sizes etc.
        Layout.init();

        // create view dialog
        GH.Dialog.CreateView.init();

        RapidBoard.initialiseListeners.resolve(rapidBoardConfig);

        // if some one (including third parties) fire this event then we will reload the board
        Events.bind(RapidBoard.API_EVENT_CAUSE_BOARD_RELOAD, RapidBoard.reload);

        // Send analytic for generic board show
        AJS.trigger('analytics', {name: 'RapidBoard.show'});
    };

    RapidBoard.init = function () {
        if ($('body.ghx-print-card-body').length) {
            // we are on the card-printing page
            return;
        }
        GlobalEvents.trigger('pre-initialization');

        // render the base dom tree
        $('#gh').html(GH.tpl.board.x.renderRapidBoard());

        // for the pages that don't use page scrolling
        $('body').addClass('ghx-scroll-columns');

        // touch detection
        if ($.support.touch) {
            $('#gh').removeClass('ghx-no-touch').addClass('ghx-touch');
        }

        // handle delete dialog success
        $(GH).bind('ghx.rapidview.deleted', function (event, data) {
            Notification.addPageLoadMessage(data.message);
            RapidBoard.gotoManageViewsPage();
        });

        var viewId = parseInt(RapidBoard.UrlState.getCurrentUrlState().rapidView);

        // get the rapid view id from local storage if it's not exist in URL params
        if (_.isUndefined(viewId) || _.isNaN(viewId)) {
            //todo this should not be here as it's defined in other scope of js files. RapidBoard should have no knowledge of this object.
            viewId = RapidBoard.State.getRapidViewId();
        }

        GH.RapidViewConfig.fetchConfiguration(viewId).done(function (rapidBoardConfig) {
            // start event handling environment
            RapidBoard.initListeners(rapidBoardConfig);
            RapidBoard.initialiseListenersDone.resolve();
        });
    };

    RapidBoard.viewModelComparator = function (a, b) {
        //ECMA standards, return -1, 0, or 1 (other browsers still accept true/false but not chrome)
        if (typeof a.name !== "string") {
            return -1;
        } else if (typeof b.name !== "string") {
            return 1;
        } else {
            return (a.name.toLowerCase() < b.name.toLowerCase()) ? -1 : (a.name.toLowerCase() > b.name.toLowerCase()) ? 1 : 0;
        }
    };


    return RapidBoard;
});
(function () {
    const rapidBoardDefined = window.GH && window.GH.RapidBoard;
    const rapidBoard = require('jira-agile/rapid/rapid-board');
    if( !rapidBoardDefined ) {
        AJS.namespace('GH.RapidBoard', null, rapidBoard);
    }
})();
