/* globals GH, _, WRM */
(function RapidViewConfig(GH) {
    var configDataClaimed = false;

    GH.RapidViewConfig = {};
    GH.RapidViewConfig.currentData = {};

    /**
     * fetch the configuration data for a given viewIdInfo
     * @param viewIdInfo can be a viewId, null (indicating use from local storage) or a viewIdInfo object
     * @return a jQuery promise that the data will be fetched
     */
    GH.RapidViewConfig.fetchConfiguration = function (viewIdInfo) {
        // adaptive parameter,  if we get an object then we want to treat it slightly
        // different that just a plain id number
        if (_.isNull(viewIdInfo)) {
            viewIdInfo = { viewId: null, fromLocalStorage: true };
        } else if (!_.isObject(viewIdInfo)) {
            viewIdInfo = { viewId: viewIdInfo, fromLocalStorage: false };
        }

        var viewId = viewIdInfo.viewId;
        var returnDefaultBoard = viewIdInfo.fromLocalStorage;

        if (!configDataClaimed) {
            var configData = WRM.data.claim("com.atlassian.greenhopper:rapid-view-config");
            if (configData) {
                GH.RapidViewConfig.handleData(configData);
                GH.RapidViewConfig.currentData.promise = AJS.$.Deferred().resolve(configData.currentViewConfig).promise();
            }
            configDataClaimed = true;
        }

        // If the id differs, request a new set of configurations
        if (GH.RapidViewConfig.currentData.id !== viewId) {
            // fetch the config - if we don't have a viewId, let the server work out what to give us
            var data = {
                returnDefaultBoard: returnDefaultBoard
            };

            if (viewId) {
                data.rapidViewId = viewId;
            }
            var promise = GH.Ajax.get({
                url: '/xboard/config.json',
                data: data,
                deferErrorHandling: true
            }, 'rapidViewConfig');
            promise.done(GH.RapidViewConfig.handleData);
            // pipe the promise, listeners should only care about the currentViewConfig
            GH.RapidViewConfig.currentData = {
                id: viewId,
                promise: promise.pipe(function (data) {
                    return data.currentViewConfig;
                }, GH.RapidViewConfig.handleRapidLoadError),
                data: null
            };
        }

        return GH.RapidViewConfig.currentData.promise;
    };

    GH.RapidViewConfig.handleRapidLoadError = function (jqXHR, textStatus) {
        if (textStatus.status !== 404) {
            return;
        }
        var error;
        if (GH.Ajax.getUserFromAJS() === 'anonymous') {
            error = AJS.I18n.getText('gh.error.noview.anonymous', GH.Navigation.getLoginUrl());
        } else {
            error = AJS.I18n.getText('gh.rapid.view.error.noview');
        }
        GH.Notification.addPageLoadMessage(error, 'warning');
        GH.RapidBoard.gotoManageViewsPage();
    };

    GH.RapidViewConfig.handleData = function (data) {
        // set the global data
        GH.RapidViewConfig.storeGlobalConfig(data.globalConfig);

        // set the current view data
        GH.RapidViewConfig.currentData.id = data.currentViewConfig.id;
        GH.RapidViewConfig.currentData.data = data.currentViewConfig;
        GH.RapidViewConfig.canManageSprints = data.currentViewConfig.canManageSprints;

        GH.RapidViewConfig.canEdit = data.currentViewConfig.canEdit;
        GH.RapidViewConfig.kanPlanEnabled = data.currentViewConfig.kanPlanEnabled;
        GH.RapidViewConfig.isBoardAdmin = data.currentViewConfig.isBoardAdmin;
        GH.RapidViewConfig.canManageSprints = data.currentViewConfig.canManageSprints;

        // modify a "global" flag in the DOM to indicate sprint support
        GH.RapidViewConfig.modifySprintSupport();
    };

    /**
     * Distributes the global config to the individual components
     * @param globalConfig : object
     */
    GH.RapidViewConfig.storeGlobalConfig = function (globalConfig) {
        // Set NumberFormat and TimeFormat information - at the moment this is global
        GH.NumberFormat.groupingSeparator = globalConfig.numberGroupingSeparator;
        GH.NumberFormat.decimalSeparator = globalConfig.numberDecimalSeparator;

        // Time formatting
        GH.TimeFormat.timeTrackingConfigOpts = globalConfig.timeTrackingConfig;
        GH.TimeFormat.dateTimePickerFormat = globalConfig.dateTimePickerFormat;
        GH.TimeFormat.userTimeZoneLabel = globalConfig.userTimeZoneLabel;

        // Date format
        // TODO: Put into a more generic place
        GH.ChartTimeFrames.dateFormat = globalConfig.dateFormat;
        GH.ChartTimeFrames.timeFormat = globalConfig.dateTimePickerFormat;
        GH.VersionQuickCreate.setDatePickerFormat(globalConfig.dateFormat);
        GH.VersionView.setDatePickerFormat(globalConfig.dateFormat);

        // whether we allow > 1 sprint to be started
        GH.RapidBoard.parallelSprints = globalConfig.parallelSprints;

        GH.RapidBoard.stickySprintHeaders = globalConfig.stickySprintHeaders;

        // whether we allow backlog for kanban board
        GH.RapidBoard.kanPlanLabEnabled = globalConfig.kanPlanLabEnabled;

        // storing a globally accessible epics config because on a Kanban board, the Epics data might not be present
        GH.EpicConfig.setEpicConfig(globalConfig.epicConfig);

        GH.SprintConfig.setSprintConfig(globalConfig.sprintConfig);

        // User Data
        GH.UserData.setGlobalConfig(globalConfig);

        GH.Config.isSampleDataAvailable = globalConfig.isSampleDataAvailable;
    };

    // TODO: find a better place for this
    GH.RapidViewConfig.modifySprintSupport = function () {
        var body = AJS.$('body');
        // remove any previous flags
        body.removeClass('ghx-days-in-column-disabled ghx-days-in-column-enabled ghx-sprint-support ghx-no-sprint-support');

        // set a flag on the body for styling
        if (GH.RapidBoard.State.isScrumBoard()) {
            body.addClass('ghx-sprint-support');
        } else {
            body.addClass('ghx-no-sprint-support');
        }
        if (GH.RapidViewConfig.currentData.data.showDaysInColumn) {
            body.addClass('ghx-days-in-column-enabled');
        } else {
            body.addClass('ghx-days-in-column-disabled');
        }
    };
})(GH);