define('jira-agile/rapid/configuration/card-layout-config-table', ["jira/util/formatter", "underscore", "jquery"], function (formatter, _, $) {
    'use strict';

    var Messages = AJS.messages;
    var RestfulTable = AJS.RestfulTable;
    var Ajax = GH.Ajax;

    function CardLayoutConfigTable(mode, model, el) {
        this.mode = mode;
        this.model = model;
        this.el = $(el);
    }

    CardLayoutConfigTable.prototype = {
        render: function render() {
            var _this = this;

            var model = this.model;
            var canEdit = model.canEdit;

            var fields = _.filter(model.currentFields, function (field) {
                return field.mode === this.mode;
            }, this);

            var availableFields = this._getAvailableCardLayoutFields(false);

            if (fields.length === 0 && availableFields.length === 0) {
                return;
            }

            // create new or replace existing table for RestfulTable
            this.el.html('<table></table>');

            // setup the RestfulTable
            this.restfulTable = new RestfulTable({
                el: this.el.find('table'), // table to add entries to. Entries are appended to the table's <tbody> element
                resources: {
                    all: function all(next) {
                        return next(fields);
                    },
                    self: Ajax.buildRestUrl('/cardlayout/' + model.rapidViewId + '/' + this.mode + '/field/')
                },
                columns: this._getTableColumns(),
                allowEdit: false,
                allowDelete: canEdit,
                allowReorder: canEdit,
                allowCreate: canEdit
            });

            if (this.model.canEdit) {
                // Need to keep the create row up to date when values are removed (add max allowed, then remove some).
                $(this.restfulTable).bind(RestfulTable.Events.ROW_REMOVED, function () {
                    _this.restfulTable.getCreateRow().render({ values: {}, errors: {} });
                });

                // This only triggers with a response code of 400.
                this.restfulTable.getCreateRow().bind(RestfulTable.Events.VALIDATION_ERROR, function (errors) {
                    Messages.warning(_this.el, {
                        body: errors.cardlayoutConfig,
                        insert: 'prepend'
                    });
                });
            }
        },

        _getTableColumns: function _getTableColumns() {
            return [{
                id: 'fieldId',
                header: formatter.I18n.getText('gh.rapid.config.cardlayout.table.heading.field'),
                fieldName: 'fieldId',
                readView: RestfulTable.CustomReadView.extend({
                    render: function render() {
                        return GH.tpl.cardlayoutconfig.renderFieldCell(this.model.toJSON());
                    }
                }),
                createView: this._getCustomCreateView()
            }];
        },

        _getCustomCreateView: function _getCustomCreateView() {
            var configTable = this;
            return RestfulTable.CustomCreateView.extend({
                render: function render(renderData) {
                    var customCreateView = this;
                    var $select = $('<select class="select"></select>').attr('name', renderData.name);

                    // this part won't be used, the actually displayed values are set in the promise callback bellow
                    var values = configTable._getAvailableCardLayoutFields(false);

                    // render the initial options or hide the select if empty
                    if (values.length) {
                        $select.html(this.renderOptions(values));
                    } else {
                        $select.hide();
                    }

                    // load the new values
                    configTable._getAvailableCardLayoutFields(true).done(function (result) {
                        values = result.fields;
                        values.sort(function (field1, field2) {
                            return field1.name.localeCompare(field2.name);
                        });
                        if (values.length) {
                            configTable.restfulTable.getCreateRow().enable();
                            $select.html(customCreateView.renderOptions(values)).show();
                        } else {
                            $select.hide();
                            configTable.restfulTable.getCreateRow().disable();
                        }
                    });
                    return $select;
                },
                renderOptions: function renderOptions(values) {
                    return _.reduce(values, function ($options, optionData) {
                        var $option = $('<option></option>').val(optionData.fieldId).text(optionData.name);
                        return $options.add($option);
                    }, $());
                }
            });
        },

        /**
         * Return the available fields that can be added (i.e. the list which will populate the dropdown)
         * @param {boolean} defer use true to return a deferred result; false to return the list as it is in the model (will not be up to date)
         * @return {*}
         */
        _getAvailableCardLayoutFields: function _getAvailableCardLayoutFields(defer) {
            var model = this.model;
            if (defer) {
                return Ajax.get({
                    url: '/cardlayout/' + model.rapidViewId + '/' + this.mode + '/available'
                }, 'cardLayoutConfig').done(function (result) {
                    model.availableFields = result.fields;
                });
            } else {
                return model.availableFields;
            }
        }
    };

    return CardLayoutConfigTable;
});

AJS.namespace('GH.CardLayoutConfigTable', null, require('jira-agile/rapid/configuration/card-layout-config-table'));