
/* globals GH, _ */
(function () {
    GH.RapidBoard.Config = {};
    GH.RapidBoard.Config.BoardAdmins = {
        // Inline edit data. Used to finish editing properly, because
        // the board admins picker does not integrate with the lifecycle of
        // GH.RapidBoard.Util.InlineEditable
        editData: null
    };

    GH.RapidBoard.Config.UNMAPPED_STATUSES_WITH_ISSUES = "unmappedStatusesWithIssues";
    GH.RapidBoard.Config.NO_UNMAPPED_STATUSES_WITH_ISSUES = "noUnmappedStatusesWithIssues";

    (function () {
        var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
        GH.RapidBoard.Config.analytics = {
            fieldChange: new AnalyticsTracker("gh.config.field"),
            tabChange: new AnalyticsTracker("gh.config.tab")
        };
        GH.RapidBoard.Config.BoardAdmins.analytics = {
            update: new AnalyticsTracker("gh.config.boardadmins", "update")
        };
    })();

    // default is the left-most tab
    GH.RapidBoard.Config.defaultTab = "filter";

    // storage for tab selection
    GH.RapidBoard.Config.storedTabKey = 'selectedConfigTab';

    // Events
    GH.RapidBoard.Config.BACKLOG_COLUMN_STATUS_UPDATE = 'backlog.column.status.update';

    /**
     * Initializes the RapidViewConfig page
     */
    GH.RapidBoard.Config.init = function () {
        // create view dialog
        GH.Dialog.CreateView.init();

        // load the model
        GH.RapidBoard.Config.loadEditModel();

        // handle delete dialog success
        AJS.$(GH).bind('ghx.rapidview.deleted', function (event, data) {
            GH.Notification.addPageLoadMessage(data.message);
            GH.RapidBoard.gotoManageViewsPage();
        });

        // rapid view name edit
        GH.RapidBoard.Config.initRapidViewNameEdit();
        GH.RapidBoard.Config.initRapidViewBoardAdminsEdit();

        // TODO this will probably have to move when the big merge happens
        GH.PersistentMessages.CreateViewMessage.renderMessage();
    };

    /**
     * Calls the rapid board data rest service.
     */
    GH.RapidBoard.Config.loadEditModel = function () {
        // find the right rapidViewId
        var state = GH.State.toViewState(window.location.href);
        var rapidViewId = state.rapidView;

        if (!rapidViewId) {
            window.location.href = GH.Ajax.CONTEXT_PATH + "/secure/ManageRapidViews.jspa";
            return;
        }

        var configurationTabsResponse = GH.Ajax.get({
            url: '/xboard/configuration-tabs',
            data: {
                rapidViewId: rapidViewId
            },
            deferErrorHandling: true
        });

        var rapidViewConfigurationResponse = GH.Ajax.get({
            url: '/rapidviewconfig/editmodel.json',
            data: {
                rapidViewId: rapidViewId
            },
            deferErrorHandling: true
        }, 'rapidBoardConfig');

        return AJS.$.when(rapidViewConfigurationResponse, configurationTabsResponse).done(function (modelResponse, configurationTabsResponse) {
            var model = modelResponse[0];
            var configurationTabs = configurationTabsResponse[0].values;
            GH.RapidBoard.Config.renderEditScreen(model, configurationTabs);
        }).fail(GH.RapidViewConfig.handleRapidLoadError);
    };

    /** Holds the id of the currently edited rapidView */
    GH.RapidBoard.Config.editedRapidViewId = undefined;

    /**
     * Renders the complete edit screen
     */
    GH.RapidBoard.Config.renderEditScreen = function (model, configurationTabs) {

        var EstimationConfig = require('jira-agile/rapid/configuration/estimation-config');

        GH.RapidBoard.State.setRapidViewId(model.id);

        // initialize global configuration
        GH.RapidViewConfig.storeGlobalConfig(model.globalConfig);

        var isSprintSupportEnabled = model.isSprintSupportEnabled;

        // TODO this is duplicated from RapidViewConfig.modifySprintSupport
        var body = AJS.$('body');
        // remove anyAJS.$('body') previous flags
        body.removeClass('ghx-sprint-support ghx-no-sprint-support ghx-days-in-column-enabled ghx-days-in-column-disabled');
        // set a flag on the body for styling the days in column
        if (model.showDaysInColumn) {
            body.addClass('ghx-days-in-column-enabled');
        } else {
            body.addClass('ghx-days-in-column-disabled');
        }

        if (isSprintSupportEnabled) {
            body.addClass('ghx-sprint-support');
        } else {
            body.addClass('ghx-no-sprint-support');
        }

        // identify the page
        body.addClass('ghx-board-configure');

        // render the content shell
        AJS.$('#ghx-content-main').html(GH.tpl.rapid.view.renderConfigurationTabs({
            rapidViewId: model.id,
            configurationTabs: configurationTabs
        }));

        // touch detection
        if (AJS.$.support.touch) {
            AJS.$('#gh').removeClass('ghx-no-touch').addClass('ghx-touch');
        }

        GH.JQLAutoComplete.setModel(model);

        // render the header
        GH.RapidBoard.Config.renderHeader(model);

        // render the name edit
        GH.RapidBoard.Config.renderViewNameForm(model);

        // save the model for convenience
        _.extend(GH.RapidBoard.Config.BoardAdmins, {
            model: model.boardAdmins,
            canEdit: model.canEdit,
            needsWarning: model.warnBeforeEditingOwner,
            canUsePicker: model.canUseBoardAdminsPicker
        });

        // render the board admins edit
        GH.RapidBoard.Config.renderViewBoardAdminsForm();

        // render the filter part
        GH.RapidBoard.Config.Filter.init(model);

        // column configuration
        GH.RapidBoard.Config.Columns.init(model);

        // swimlane configuration
        GH.RapidBoard.Config.Swimlanes.init(model);

        // quick filter configuration
        GH.RapidBoard.Config.QuickFilter.init(model);

        if (isSprintSupportEnabled) {
            // estimation configuration
            EstimationConfig.init(model);
        }

        // Working days configuration
        GH.WorkingDaysConfig.init(model);

        // card colors
        GH.CardColorConfig.init(model);

        // card layout
        new GH.CardLayoutConfig(model, '#ghx-config-cardLayout').render();

        // detail view fields
        GH.DetailViewConfig.init(model);

        // retain the view id
        GH.RapidBoard.Config.editedRapidViewId = model.id;

        // NEW_CONTENT_ADDED is called whenever a arbitrary new content is added by JIRA code, such as when the RestfulTable
        // adds an edit row. Check whether the context really contains a row, and if so initialize the jql autocomplete
        JIRA.bind(JIRA.Events.NEW_CONTENT_ADDED, function (e, context) {
            var textfield = AJS.$(context).find('.js-jql-autocomplete-field');
            var error = AJS.$(context).find('.js-jql-autocomplete-error');
            if (textfield.length > 0 && error.length > 0) {
                var textFieldId = textfield.attr('id');
                var errorId = error.attr('id');
                // Restful table creates the create row before actually adding it to the dom.
                // As JQLAutoComplete fetches the elements from the dom it ends up not finding what it is looking for and blowing up.
                // also ensure the field and error is actually available in the dom
                textfield = AJS.$('#' + textFieldId);
                error = AJS.$('#' + errorId);
                if (textfield.length > 0 && error.length > 0) {
                    GH.JQLAutoComplete.initialize(textFieldId, errorId);
                }
            }
        });

        // initialise which tab we're looking at
        GH.RapidBoard.Config.initState();

        // initialise the tab and state-change listeners
        GH.RapidBoard.Config.initListeners();

        _.delay(function () {
            AJS.$('#ghx-config-header').attr("data-rendered", new Date().getTime());
        });
    };

    /**
     * Renders the general information
     */
    GH.RapidBoard.Config.renderHeader = function (model) {

        // render the header html
        GH.RapidBoard.Config.renderTitle(model);

        // render the config header html
        var canCreateView = GH.UserData.hasUser();

        // render actions
        GH.ViewActions.renderOnConfiguration(model);

        GH.RapidBoard.Config.addAnalyticEventsForToolsMenu();

        // TODO: Remove ghx-config-header and move error messaging to a more sensible place
        var header = AJS.$('#ghx-config-header');
        // render any error messaging
        header.html(GH.tpl.rapid.view.renderConfigPermissionsWarning({ 'model': model }));

        // if user cannot edit we change some styles, below adds/removes the hook
        if (model.canEdit) {
            AJS.$('#gh').addClass('ghx-editor').removeClass('ghx-reader');
        } else {
            AJS.$('#gh').addClass('ghx-reader').removeClass('ghx-editor');
        }
    };

    GH.RapidBoard.Config.renderTitle = function (model) {
        AJS.$('#ghx-header').html(GH.tpl.rapid.view.renderConfigurationHeader({ name: model.name, rapidViewId: model.id }));
        AJS.$('#back-to-board').click(function () {
            AJS.trigger('analyticsEvent', { name: 'gh.config.back.to.board' });
        });
    };

    GH.RapidBoard.Config.addAnalyticEventsForToolsMenu = function () {
        AJS.$('#board-tools-section-content .js-view-action-copy').bind('click', function () {
            AJS.trigger('analyticsEvent', { name: 'jira-software.config.boardmenu.click.copy-board' });
        });

        AJS.$('#board-tools-section-content .js-view-action-delete').bind('click', function () {
            AJS.trigger('analyticsEvent', { name: 'jira-software.config.boardmenu.click.delete-board' });
        });

        AJS.$('#board-tools-section-content .js-view-action-create').bind('click', function () {
            AJS.trigger('analyticsEvent', { name: 'jira-software.config.boardmenu.click.create-board' });
        });
    };

    GH.RapidBoard.Config.updateTitle = function (viewName) {
        AJS.$('#js-nav-view-name').text(viewName);
    };

    //
    // Rapid view name edit
    //


    /**
     * Registers the live handler for the rapid view name edit
     */
    GH.RapidBoard.Config.initRapidViewNameEdit = function () {
        // edit view name
        GH.RapidBoard.Util.InlineEditable.register('.js-edit-rapidViewName', {
            validate: GH.RapidBoard.Config.validateViewName,
            save: GH.RapidBoard.Config.saveViewName,
            renderView: GH.RapidBoard.Config.renderViewNameField,
            postEdit: GH.RapidBoard.Config.handleTitleUpdate
        });
    };

    /**
     * Initializes the view name form
     */
    GH.RapidBoard.Config.renderViewNameForm = function (model) {
        // render the markup
        var viewNameEdit = AJS.$('#ghx-view-name-edit');
        viewNameEdit.html(GH.tpl.rapid.view.renderViewNameEditForm({ rapidViewName: model.name, canEdit: model.canEdit }));
    };

    /**
     * Validates the rapid view name
     */
    GH.RapidBoard.Config.validateViewName = function (editData) {
        var newValue = editData.editElement.val();

        // perform validation
        if (!GH.Validation.notBlank(editData.editElement, AJS.I18n.getText('gh.rapid.view.error.name.required'))) {
            return false;
        }

        editData.newValue = newValue;
        return true;
    };

    /**
     * Saves the rapid view name
     */
    GH.RapidBoard.Config.saveViewName = function (editData) {
        GH.Ajax.put({
            url: '/rapidviewconfig/name',
            data: {
                'id': GH.RapidBoard.Config.editedRapidViewId,
                'name': editData.newValue
            }
        }).done(function () {
            GH.RapidBoard.Config.analytics.fieldChange.trigger('boardname.update');
        });

        // give fast feedback
        GH.RapidBoard.Util.InlineEditable.updateView(editData);
    };

    /**
     * Updates the header according to the new name
     */
    GH.RapidBoard.Config.handleTitleUpdate = function (editData) {
        if (editData.canceled) {
            return;
        }

        GH.RapidBoard.Config.updateTitle(editData.newValue);
    };

    /**
     * Renders the view name field
     */
    GH.RapidBoard.Config.renderViewNameField = function (editData) {
        var displayView = AJS.$(GH.tpl.rapid.view.renderViewNameField({
            rapidViewName: editData.newValue,
            canEdit: true // you were allowed to edit in the first place...
        }));
        return displayView;
    };

    /**
     * Initialise the inline edit for board admins
     */
    GH.RapidBoard.Config.initRapidViewBoardAdminsEdit = function () {
        GH.RapidBoard.Util.InlineEditable.register('.js-edit-rapidViewBoardAdmin', {
            renderEdit: GH.RapidBoard.Config.renderViewBoardAdminsEditField,
            activate: null
        });
    };

    /**
     * Render the standard board admins view
     */
    GH.RapidBoard.Config.renderViewBoardAdminsForm = function () {
        var viewBoardAdminsEdit = AJS.$("#ghx-view-board-admins-edit");

        viewBoardAdminsEdit.html(GH.tpl.rapid.view.renderViewBoardAdminsEditForm({
            model: GH.RapidBoard.BoardAdminsPicker.getOrderedUnion(GH.RapidBoard.Config.BoardAdmins.model),
            canEdit: GH.RapidBoard.Config.BoardAdmins.canEdit,
            canUsePicker: GH.RapidBoard.Config.BoardAdmins.canUsePicker
        }));
    };

    /**
     * Render the board admins edit form depending on the user's permission
     * Uses inline text fields if the user cannot use the User Picker (.canUsePicker property)
     * and the multiselect otherwise
     */
    GH.RapidBoard.Config.renderViewBoardAdminsEditField = function (editData) {
        GH.RapidBoard.Config.BoardAdmins.editData = editData;
        if (GH.RapidBoard.Config.BoardAdmins.canUsePicker) {
            GH.RapidBoard.Config.renderViewBoardAdminsMultiselect();
        } else {
            GH.RapidBoard.Config.renderViewBoardAdminsNoPicker();
        }
    };

    /**
     * Render the board admins edit field without a picker
     */
    GH.RapidBoard.Config.renderViewBoardAdminsNoPicker = function () {
        var boardAdminsEditField = AJS.$("#ghx-field-view-board-admins");
        boardAdminsEditField.html(GH.tpl.rapid.view.renderBoardAdminsEditNoPicker({
            model: GH.RapidBoard.Config.BoardAdmins.model
        }));

        // focus the users field
        AJS.$("#ghx-board-admin-edit-groups").focus();

        AJS.$("#ghx-board-admin-edit-groups, #ghx-board-admin-edit-users").bind('keydown', function (e) {
            if (e.keyCode === 13) {
                GH.RapidBoard.Config.BoardAdmins.handleReturn(e);
            } else if (e.keyCode === 27) {
                GH.RapidBoard.Config.BoardAdmins.handleEsc(e);
            }
        });

        GH.RapidBoard.Config.bindNonPickerEvents();
    };

    /**
     * Set up the blur events for the no picker fields
     */
    GH.RapidBoard.Config.bindNonPickerEvents = function () {
        AJS.$("#ghx-board-admin-edit-groups, #ghx-board-admin-edit-users").blur(function () {
            GH.RapidBoard.Config.areBothNonPickerFieldsBlurred().done(function () {
                // end the form, hide the field, reload the page
                GH.RapidBoard.Config.endBoardAdminsForm();
            });
        });
    };

    /**
     * Disable the blur event for the fields, used when Enter/Esc pressed to stop
     * both firing
     */
    GH.RapidBoard.Config.unbindNonPickerEvents = function () {
        AJS.$("#ghx-board-admin-edit-groups, #ghx-board-admin-edit-users").unbind("blur");
    };

    /**
     * Event Handlers for the multiselect / dual input form
     */
    GH.RapidBoard.Config.BoardAdmins.handleReturn = function () {
        GH.RapidBoard.Config.unbindNonPickerEvents();
        // immediately save the form
        if (!GH.RapidBoard.Config.BoardAdmins.canUsePicker) {
            GH.RapidBoard.Config.endBoardAdminsForm(true);
        }
        // enter should save only if the user hasn't typed anything
        else if (GH.RapidBoard.Config.BoardAdmins.form.multiselect._getUserInputValue().length === 0) {
                GH.RapidBoard.Config.endBoardAdminsForm(true);
            }
    };

    GH.RapidBoard.Config.BoardAdmins.handleEsc = function () {
        // BoardAdminsPicker doesn't integrate into the InlineEditable lifecycle, hence the need to call this
        GH.RapidBoard.Util.InlineEditable.removeFromActiveEdits(GH.RapidBoard.Config.BoardAdmins.editData);

        GH.RapidBoard.Config.unbindNonPickerEvents();
        GH.RapidBoard.Config.renderViewBoardAdminsForm();
    };

    /**
     * Test whether both non picker fields are blurred
     */
    GH.RapidBoard.Config.areBothNonPickerFieldsBlurred = function () {
        var deferred = AJS.$.Deferred();

        setTimeout(function () {
            var focusedInputs = AJS.$("#ghx-board-admin-edit-groups:focus, #ghx-board-admin-edit-users:focus");
            if (focusedInputs.length > 0) {
                deferred.reject();
            } else {
                deferred.resolve();
            }
        }, 90);

        return deferred.promise();
    };

    /**
     * Render the board admins multiselect
     */
    GH.RapidBoard.Config.renderViewBoardAdminsMultiselect = function () {
        GH.RapidBoard.Config.BoardAdmins.form = new GH.RapidBoard.BoardAdminsPicker({
            selector: '#ghx-board-admins',
            initialData: GH.RapidBoard.Config.formatBoardAdminsInitData(GH.RapidBoard.Config.BoardAdmins.model),
            blur: function blur() {
                clearTimeout(GH.RapidBoard.Config.BoardAdmins.tId);
                var tId = setTimeout(GH.RapidBoard.Config.endBoardAdminsForm, 150);
                GH.RapidBoard.Config.BoardAdmins.tId = tId;
            }
        });

        var form = GH.RapidBoard.Config.BoardAdmins.form;
        form.show();
        form.multiselect.INVALID_KEYS.Esc = false;
        form.multiselect.keys.Return = function () {
            if (!form.multiselect.suggestionsVisible) {
                GH.RapidBoard.Config.BoardAdmins.handleReturn();
            }
        };
        form.multiselect.keys.Esc = function () {
            if (!form.multiselect.suggestionsVisible) {
                GH.RapidBoard.Config.BoardAdmins.handleEsc();
            }
        };
    };

    /**
     * Format the data for the multiselect and make it parseable later
     */
    GH.RapidBoard.Config.formatBoardAdminsInitData = function (model) {
        function formatElement(prefix, admin) {
            return {
                value: prefix + ':' + admin.key,
                label: admin.displayName,
                html: AJS.escapeHTML(String(admin.displayName))
            };
        }
        var formattedUsers = _.map(model.userKeys, function (user) {
            return formatElement('user', user);
        });
        var formattedGroups = _.map(model.groupKeys, function (group) {
            return formatElement('group', group);
        });
        return _.union(formattedUsers, formattedGroups);
    };

    /**
     * Validate, save and close the board admins form
     */
    GH.RapidBoard.Config.endBoardAdminsForm = function (enterPressed) {
        clearTimeout(GH.RapidBoard.Config.BoardAdmins.tId);

        // BoardAdminsPicker doesn't integrate into the InlineEditable lifecycle, hence the need to call this
        GH.RapidBoard.Util.InlineEditable.removeFromActiveEdits(GH.RapidBoard.Config.BoardAdmins.editData);

        if (GH.RapidBoard.Config.BoardAdmins.canUsePicker && !GH.RapidBoard.Config.shouldHideBoardAdminsForm()) {
            if (!enterPressed) {
                return;
            }
        }

        var updatedModel;

        if (GH.RapidBoard.Config.BoardAdmins.canUsePicker) {
            updatedModel = GH.RapidBoard.Config.BoardAdmins.form.getAjaxValue();
        } else {
            updatedModel = GH.RapidBoard.Config.resolveNewBoardAdminsModelFromTextFields();
        }

        GH.RapidBoard.Config.validateBoardAdmins(updatedModel).done(GH.RapidBoard.Config.saveBoardAdmins).fail(function () {
            GH.RapidBoard.Config.renderViewBoardAdminsForm();
        });
    };

    GH.RapidBoard.Config.resolveNewBoardAdminsModelFromTextFields = function () {
        var checkNotEmpty = function checkNotEmpty(e) {
            return e === "";
        };
        var splitRegex = /\s*,\s*/;

        var convertToArray = function convertToArray(str) {
            return _.uniq(_.reject(str.split(splitRegex), checkNotEmpty));
        };

        return {
            userKeys: convertToArray(AJS.$("#ghx-board-admin-edit-users").val()),
            groupKeys: convertToArray(AJS.$("#ghx-board-admin-edit-groups").val())
        };
    };

    GH.RapidBoard.Config.saveBoardAdmins = function () {

        var ajaxValue;
        if (GH.RapidBoard.Config.BoardAdmins.canUsePicker) {
            ajaxValue = GH.RapidBoard.Config.BoardAdmins.form.getAjaxValue();
        } else {
            ajaxValue = GH.RapidBoard.Config.resolveNewBoardAdminsModelFromTextFields();
        }

        GH.RapidBoard.Config.BoardAdmins.analytics.update.trigger(ajaxValue.groupKeys.length + "," + ajaxValue.userKeys.length);

        return GH.Ajax.put({
            url: '/rapidviewconfig/boardadmins',
            data: {
                id: GH.RapidBoard.Config.editedRapidViewId,
                boardAdmins: ajaxValue
            }
        }).done(function (xhr, status, response) {
            // fast response
            GH.RapidBoard.Config.BoardAdmins.model = JSON.parse(response.responseText);
            GH.RapidBoard.Config.renderViewBoardAdminsForm();

            // now reload the edit model to render any permission warnings / disable editable fields
            GH.RapidBoard.Config.loadEditModel();
            GH.RapidBoard.Config.analytics.fieldChange.trigger('boardadmins.update');
        }).fail(function () {
            GH.RapidBoard.Config.renderViewBoardAdminsForm();
        });
    };

    /**
     * Check whether the form has lost focus, which means it should hide
     */
    GH.RapidBoard.Config.shouldHideBoardAdminsForm = function () {
        return AJS.$("textarea:focus").length === 0;
    };

    /**
     * Validate the new board admins
     */
    GH.RapidBoard.Config.validateBoardAdmins = function (newModel) {
        var deferred = AJS.$.Deferred();

        if (newModel.userKeys.length === 0 && newModel.groupKeys.length === 0) {
            return deferred.reject();
        }

        if (!GH.RapidBoard.Config.hasBoardAdminsChanged(newModel)) {
            return deferred.reject();
        }

        GH.RapidBoard.Config.needsBoardAdminsConfirm(newModel).done(function () {
            var message = AJS.I18n.getText('gh.configuration.board.admin.lost');
            var dialog = GH.ConfirmDialog.create("ghx-boardadmins-confirm-dialog", {
                content: message,
                onConfirmFn: deferred.resolve,
                onCancelFn: deferred.reject,
                height: 230
            });

            dialog.show();
        }).fail(function () {
            deferred.resolve();
        });

        return deferred.promise();
    };

    /**
     * Check to see if the new board admins differ from the original ones
     * that were present when inline edit began.
     */
    GH.RapidBoard.Config.hasBoardAdminsChanged = function (newBoardAdmins) {
        return !_.isEqual(newBoardAdmins, GH.RapidBoard.Config.BoardAdmins.model);
    };

    /**
     * Check whether the new list of board admins will lock the user
     * out of making future changes, prompt with confirm if so
     */
    GH.RapidBoard.Config.needsBoardAdminsConfirm = function (newModel) {
        var deferred = AJS.$.Deferred();

        // if they are a JIRA admin, this will be false
        // so we don't need to warn them
        if (!GH.RapidBoard.Config.BoardAdmins.needsWarning) {
            return deferred.reject();
        }
        // otherwise they aren't a JIRA admin
        AJS.$.ajax({
            url: AJS.contextPath() + '/rest/api/2/user?expand=groups',
            type: 'get',
            dataType: 'json',
            data: {
                username: GH.UserData.getUserName()
            }
        }).done(function (user) {

            var isCurrentUserKeyInList = function isCurrentUserKeyInList(userKey, list) {
                return _.contains(list, userKey);
            };

            var isCurrentUserInGroups = function isCurrentUserInGroups(userGroups, adminGroups) {
                return _.find(userGroups, function (group) {
                    return _.contains(adminGroups, group.name);
                });
            };

            user.key = user.key ? user.key : user.name;

            if (isCurrentUserKeyInList(user.key, newModel.userKeys) || isCurrentUserInGroups(user.groups.items, newModel.groupKeys)) {
                return deferred.reject();
            } else {
                return deferred.resolve();
            }
        }).fail(function () {
            return deferred.reject();
        });

        return deferred.promise();
    };

    //
    // tabs / url state
    //

    /**
     * Sets the active config tab to the one represented by tabKey
     */
    GH.RapidBoard.Config.setTab = function (tabKey) {
        function getEscapedSelectorForId(id) {
            // these need to be escaped ! " # $ % & ' ( ) * + , . / : ; < = > ? @ [ \ ] ^ ` { | } ~
            // because section id may comes from module descriptor key and it may contains not allowed characters for css class or id.
            return id.replace(/(!|"|#|\$|%|&|'|\(|\)|\*|\+|,|\.|\/|:|;|<|=|>|\?|@|\[|\\|\]|\^|`|\{|\||\}|~)/g, "\\$1");
        }

        //get the tab id based on the key we're given
        var revertToDefault = !tabKey;

        // double check that this tab still exists
        var $selectedTab;
        if (!revertToDefault) {
            $selectedTab = AJS.$("#" + getEscapedSelectorForId(tabKey));
            revertToDefault = $selectedTab.length === 0;
        }

        // If we're given a dud tab key, we want to revert to the default
        if (revertToDefault) {
            tabKey = GH.RapidBoard.Config.defaultTab;
            $selectedTab = AJS.$("#" + getEscapedSelectorForId(tabKey));
            //If we're setting the default tab we may as well replace it in the history
            GH.State.replace({
                "rapidView": GH.RapidBoard.Config.editedRapidViewId,
                "tab": tabKey
            });
        }

        // Store the current tab selection
        GH.BoardState.setPerViewValue(GH.RapidBoard.Config.storedTabKey, tabKey);
        GH.RapidBoard.Config.analytics.tabChange.trigger(tabKey + ".view");

        // Hide all the panel content items
        AJS.$('.ghx-page-panel-content-item').css({ 'display': 'none' });
        // Show the panel content item for the current 'tab'
        $selectedTab.css({ 'display': 'block' });

        var $menuItems = AJS.$('#ghx-config-nav').find('li');
        // Remove any selected state on the menu items and add to current item
        $menuItems.removeClass('aui-nav-selected').filter('[data-tabitem="' + tabKey + '"]').addClass('aui-nav-selected');

        if (tabKey === "columns") {
            GH.RapidBoard.Config.Columns.tabLoaded();
        }
    };

    /**
     * Sets the initial state
     */
    GH.RapidBoard.Config.initState = function () {
        //Get the state out of the uri
        var state = GH.State.getStateFromUri(document.location.href);
        var tab = state.tab;

        if (!tab) {
            tab = GH.BoardState.getPerViewValue(GH.RapidBoard.Config.storedTabKey, false);
        }

        //set the selected tab to the one specified in the url
        GH.RapidBoard.Config.setTab(tab);
    };

    GH.RapidBoard.Config.initListeners = function () {
        var statusDndState = require('jira-agile-column-configuration-status-dnd-state');
        var flags = require('jira-agile/rapid/configuration/kanban-backlog-flags');

        // Event listener for vertical nav
        AJS.$('#ghx-config-nav').find('li').bind('simpleClick', GH.Util.wrapPreventDefault(function () {
            var selectedTab = AJS.$(this).data("tabitem");

            // the tab has been selected, update the url if it doesn't reflect the new state
            if (selectedTab) {
                var state = GH.State.toViewState(window.location.href);
                if (selectedTab !== state.tab) {
                    GH.State.push({
                        "rapidView": GH.RapidBoard.Config.editedRapidViewId,
                        "tab": selectedTab
                    });
                }
            }
        }));

        // upon history event, select the correct tab
        GH.State.registerStateChangeListener(function () {
            var state = GH.State.toViewState(window.location.href);
            GH.RapidBoard.Config.setTab(state.tab);
        });

        // Handle dragging status in/out Kanban backlog column to update the edit model and layout according to the status of KanPlan feature for board
        GH.RapidBoard.Config.Columns.on(GH.RapidBoard.Config.Columns.BACKLOG_COLUMN_STATUS_UPDATE, function ($srcColumn, $toColumn) {
            var columns = GH.RapidBoard.Config.Columns.model.mappedColumns;
            var isKanbanBacklogColumnMapped = !!_.find(columns, function (column) {
                return column.isKanPlanColumn === true && !_.isEmpty(column.mappedStatuses);
            });
            if (GH.RapidBoard.Config.Columns.kanPlanConfig.isKanPlanEnabled !== isKanbanBacklogColumnMapped) {
                GH.RapidBoard.Config.loadEditModel().done(function sendAnalytics(modelResponse) {
                    var model = modelResponse[0];
                    var data = statusDndState.basicEventData(model.rapidListConfig);
                    data.kanplanEnabled = model.isKanPlanEnabled;
                    AJS.trigger('analytics', { name: 'jira-software.rapidboard.data.loaded', data: data });
                });
            }

            var params = statusDndState.getBoardStateAfterStatusDnd(GH.RapidBoard.Config.Columns.model, $srcColumn, $toColumn);

            if (params.kanplanDisabledAfterOperation) {
                flags.kanplanDisabledFlag(GH.RapidBoard.Config.Columns.model, $srcColumn, $toColumn);
            } else if (params.kanplanEnabledAfterOperation) {
                flags.kanplanEnabledFlag(GH.RapidBoard.Config.Columns.model, $srcColumn, $toColumn);
            }
        }, this);
    };
})();