AJS.test.require(["com.pyxis.greenhopper.jira:gh-test-common", "com.pyxis.greenhopper.jira:gh-rapid"], function () {
    module("RapidBoardUrlState tests", {
        assertUrlToNormalized: function assertUrlToNormalized(urlParams, normalizedInput, expectedNormalized) {
            GH.RapidBoardUrlState.getNormalizedFromUrl(urlParams, normalizedInput);
            deepEqual(normalizedInput, expectedNormalized, "Normalized state was incorrect");
        },

        assertNormalizedToUrl: function assertNormalizedToUrl(normalized, urlParamsInput, expectedUrlParams) {
            GH.RapidBoardUrlState.toUrl(normalized, urlParamsInput);
            deepEqual(urlParamsInput, expectedUrlParams, "Url params were incorrect");
        }
    });

    test("Project key is removed from state when URL doesn't contain it", sinon.test(function () {
        var urlParams = { "useStoredSettings": "false", "rapidView": "1203" };
        var normalized = { "projectKey": "SHOULDREMOVE" };

        this.assertUrlToNormalized(urlParams, normalized, { "detailOpened": false, "mode": "work", "rapidView": 1203 });
    }));

    test("Project key is removed from state when URL doesn't contain it even if using stored settings", sinon.test(function () {
        var urlParams = { "useStoredSettings": "true", "rapidView": "1203" };
        var normalized = { "projectKey": "SHOULDREMOVE" };

        this.assertUrlToNormalized(urlParams, normalized, { "rapidView": 1203 });
    }));

    test("Project key is in state when URL contains it", sinon.test(function () {
        var urlParams = { "useStoredSettings": "false", "rapidView": "1203", "projectKey": "NEWKEY" };
        var normalized = { "projectKey": "SHOULDREMOVE" };

        this.assertUrlToNormalized(urlParams, normalized, { "detailOpened": false, "mode": "work", "rapidView": 1203, "projectKey": "NEWKEY" });
    }));

    test("Normalized state is set correctly based on URL", sinon.test(function () {
        this.stub(GH.RapidBoard.State, "isPlanModeAvailable", function () {
            return true;
        });

        var normalized = { "projectKey": "SHOULDREMOVE" };

        var urlParams = { "useStoredSettings": "false", "rapidView": "1203", "projectKey": "NEWKEY", "view": "reporting" };
        this.assertUrlToNormalized(urlParams, normalized, { "mode": "report", "rapidView": 1203, "projectKey": "NEWKEY" });

        urlParams = { "useStoredSettings": "false", "rapidView": "1203", "projectKey": "NEWKEY", "view": "planning" };
        this.assertUrlToNormalized(urlParams, normalized, { "detailOpened": true, "mode": "plan", "rapidView": 1203, "projectKey": "NEWKEY" });

        urlParams = { "useStoredSettings": "false", "rapidView": "1203", "projectKey": "NEWKEY", "view": "planning.nodetail" };
        this.assertUrlToNormalized(urlParams, normalized, { "detailOpened": false, "mode": "plan", "rapidView": 1203, "projectKey": "NEWKEY" });

        urlParams = { "useStoredSettings": "false", "rapidView": "1203", "projectKey": "NEWKEY", "view": "detail" };
        this.assertUrlToNormalized(urlParams, normalized, { "detailOpened": true, "mode": "work", "rapidView": 1203, "projectKey": "NEWKEY" });

        urlParams = { "useStoredSettings": "false", "rapidView": "1203", "projectKey": "NEWKEY", "view": "" };
        this.assertUrlToNormalized(urlParams, normalized, { "detailOpened": false, "mode": "work", "rapidView": 1203, "projectKey": "NEWKEY" });

        urlParams = { "useStoredSettings": "false", "rapidView": "1203", "projectKey": "NEWKEY" };
        this.assertUrlToNormalized(urlParams, normalized, { "detailOpened": false, "mode": "work", "rapidView": 1203, "projectKey": "NEWKEY" });
    }));

    test("Normalized state should contain the default mode when the url has no view parameter and does not contain useStoredSettings", sinon.test(function () {
        var urlParams = { "rapidView": "1203" };
        this.assertUrlToNormalized(urlParams, {}, { "mode": "work", "detailOpened": false, "rapidView": 1203 });
    }));

    test("Normalized state should contain the mode as specified by the url when the url has a view parameter and does not contain useStoredSettings", sinon.test(function () {
        var urlParams = { "rapidView": "1203", "view": "reporting" };
        this.assertUrlToNormalized(urlParams, {}, { "mode": "report", "rapidView": 1203 });
    }));

    test("Normalized state should not contain any mode when the url has no view parameter and contains useStoredSettings", sinon.test(function () {
        var urlParams = { "rapidView": "1203", "useStoredSettings": true };
        this.assertUrlToNormalized(urlParams, {}, { "rapidView": 1203 });
    }));

    test("Normalized state should contain the mode as specified by the url when the url has a view parameter and contains useStoredSettings", sinon.test(function () {
        var urlParams = { "rapidView": "1203", "useStoredSettings": true, "view": "reporting" };
        this.assertUrlToNormalized(urlParams, {}, { "mode": "report", "rapidView": 1203 });
    }));

    test("ProjectKey is not added to URL when not in normalized", sinon.test(function () {
        var normalized = { "rapidView": 123 };
        var urlParams = { "projectKey": "SHOULDREMOVE" };

        this.assertNormalizedToUrl(normalized, urlParams, { "rapidView": 123 });
    }));

    test("ProjectKey is added to URL when in normalized", sinon.test(function () {
        var normalized = { "rapidView": 123, "projectKey": "HSP" };
        var urlParams = { "projectKey": "SHOULDREMOVE" };

        this.assertNormalizedToUrl(normalized, urlParams, { "rapidView": 123, "projectKey": "HSP" });
    }));

    test("URL params added correctly based on normalized state", sinon.test(function () {
        var urlParams = {};

        var normalized = { "rapidView": 123 };
        this.assertNormalizedToUrl(normalized, urlParams, { "rapidView": 123 });

        normalized = { "rapidView": 123, mode: "work" };
        this.assertNormalizedToUrl(normalized, urlParams, { "rapidView": 123 });

        normalized = { "rapidView": 123, mode: "work", detailOpened: true };
        this.assertNormalizedToUrl(normalized, urlParams, { "rapidView": 123, "view": "detail" });

        normalized = { "rapidView": 123, mode: "report" };
        this.assertNormalizedToUrl(normalized, urlParams, { "rapidView": 123, "view": "reporting" });

        normalized = { "rapidView": 123, mode: "plan" };
        this.assertNormalizedToUrl(normalized, urlParams, { "rapidView": 123, "view": "planning.nodetail" });

        normalized = { "rapidView": 123, mode: "plan", detailOpened: true };
        this.assertNormalizedToUrl(normalized, urlParams, { "rapidView": 123, "view": "planning" });
    }));

    test("Normalized does not include the project key when not set internally", sinon.test(function () {
        var normalized = { "projectKey": "SHOULDREMOVE" };

        this.stub(GH.RapidBoard.State, "getRapidViewId").returns(1234);

        GH.RapidBoardUrlState.getNormalizedFromInternal(normalized);

        var expectedNormalized = { "detailOpened": false, "mode": "work", "rapidView": 1234 };
        deepEqual(normalized, expectedNormalized, "Normalized params were incorrect");
    }));

    test("Normalized does include the project key when set internally", sinon.test(function () {
        var normalized = { "projectKey": "SHOULDREMOVE" };

        this.stub(GH.RapidBoard.State, "getRapidViewId").returns(1234);
        this.stub(GH.RapidBoard.State, "getProjectKey").returns("HSP");

        GH.RapidBoardUrlState.getNormalizedFromInternal(normalized);

        var expectedNormalized = { "detailOpened": false, "mode": "work", "rapidView": 1234, "projectKey": "HSP" };
        deepEqual(normalized, expectedNormalized, "Normalized params were incorrect");
    }));

    test("Normalized sets detail openened correctly for work mode", sinon.test(function () {
        var normalized = { "projectKey": "SHOULDREMOVE" };

        this.stub(GH.RapidBoard.State, "getRapidViewId").returns(1234);
        this.stub(GH.RapidBoard.State, "isDetailsViewOpened").returns(true);

        GH.RapidBoardUrlState.getNormalizedFromInternal(normalized);

        var expectedNormalized = { "detailOpened": true, "mode": "work", "rapidView": 1234 };
        deepEqual(normalized, expectedNormalized, "Normalized params were incorrect");
    }));

    test("Normalized sets detail openened correctly for work mode", sinon.test(function () {
        var normalized = {};

        this.stub(GH.RapidBoard.State, "getRapidViewId").returns(1234);
        this.stub(GH.RapidBoard.State, "getMode").returns("plan");
        this.stub(GH.RapidBoard.State, "isDetailsViewPlanOpened").returns(true);

        GH.RapidBoardUrlState.getNormalizedFromInternal(normalized);

        var expectedNormalized = { "detailOpened": true, "mode": "plan", "rapidView": 1234 };
        deepEqual(normalized, expectedNormalized, "Normalized params were incorrect");
    }));

    test("Internal state does removes the project key when normalized doesn't include it", sinon.test(function () {
        var normalized = { "rapidView": 123 };

        var setRapidViewIdSpy = this.spy(GH.RapidBoard.State, "setRapidViewId");
        var setProjectKeySpy = this.spy(GH.RapidBoard.State, "setProjectKey");
        GH.RapidBoardUrlState.updateInternalFromNormalized(normalized);

        ok(setRapidViewIdSpy.called);
        ok(setProjectKeySpy.calledWith(undefined), "Project key did not get cleared from data. Calling setProjectKey with undefined has the effect of deleting it in data.");
    }));

    test("Internal state set project key when normalized includes it", sinon.test(function () {
        var normalized = { "rapidView": 123, "projectKey": "HEY" };

        var setRapidViewIdSpy = this.spy(GH.RapidBoard.State, "setRapidViewId");
        var setProjectKeySpy = this.spy(GH.RapidBoard.State, "setProjectKey");
        GH.RapidBoardUrlState.updateInternalFromNormalized(normalized);

        ok(setRapidViewIdSpy.called);
        ok(setProjectKeySpy.calledWith("HEY"));
    }));

    test("the view mode should replace into detail while loading KanPlan while it's disabled", sinon.test(function () {
        var urlParams = { "view": "planning", "rapidView": "1203" };
        var normalized = { "projectKey": "SHOULDREMOVE" };

        this.stub(GH.RapidBoard.State, "isPlanModeAvailable", function () {
            return false;
        });

        this.assertUrlToNormalized(urlParams, normalized, { "detailOpened": true, "mode": "work", "rapidView": 1203 });
    }));

    test("the view mode should replace into detail while loading KanPlan while it's disabled - with 'nodetail' in the mode", sinon.test(function () {
        var urlParams = { "view": "planning.nodetail", "rapidView": "1203" };
        var normalized = { "projectKey": "SHOULDREMOVE" };

        this.stub(GH.RapidBoard.State, "isPlanModeAvailable", function () {
            return false;
        });

        this.assertUrlToNormalized(urlParams, normalized, { "detailOpened": true, "mode": "work", "rapidView": 1203 });
    }));
});