define("jira-agile/rapid/state/url-state", ["underscore", "jquery"], function(_, $) {
    /**
     * Manages the state of the url.
     */
    const UrlState = {};

    UrlState.dateFormat = "%Y-%m-%d";

    /**
     *
     * Url parameter normalization utils
     *
     */

    /**
     * Normalizes a string param into a number.
     * @return the number or null if not a valid number, NaN and Infinite.
     */
    UrlState.normalizeNumberParam = function (value, defaultValue) {
        if (_.isUndefined(defaultValue)) {
            defaultValue = null;
        }
        if (value === undefined) {
            return defaultValue;
        }
        if (value === null) {
            return defaultValue;
        }
        var numberVal = parseInt(value, 10); // 10 radix
        if (!isFinite(numberVal)) {
            return defaultValue;
        } // _.isNumber returns true for Infinite, isFinite returns true for null. both return false for undefined.
        return numberVal;
    };

    /**
     * Normalizes a boolean param
     */
    UrlState.normalizeBooleanParam = function (value, defaultValue) {
        if (_.isUndefined(defaultValue)) {
            defaultValue = null;
        }
        if (value === undefined) {
            return defaultValue;
        }
        if (value === null) {
            return defaultValue;
        }
        if (value === "false") {
            return false;
        }
        return Boolean(value);
    };

    /**
     * Normalizes a date param of the form yyyy-mm-dd
     * @return the date object or null if not set. Note that an invalid date will translate into "now"
     */
    UrlState.normalizeDateParam = function (value, defaultValue) {
        if (_.isUndefined(defaultValue)) {
            defaultValue = null;
        }
        if (value === undefined) {
            return defaultValue;
        }
        if (value === null) {
            return defaultValue;
        }
        var dateVal = Date.parseDate(value, UrlState.dateFormat);
        return dateVal;
    };

    /**
     * Translates a parameter value (which might be undefined, a string or an array of strings) into a number array.
     */
    UrlState.normalizeNumberArrayParam = function (value) {
        var res = [];
        if (_.isString(value)) {
            // convert to number
            var numberVal = UrlState.normalizeNumberParam(value);
            if (numberVal !== null) {
                res.push(numberVal);
            }
        } else if (_.isArray(value)) {
            // convert each value
            _.each(value, function (v) {
                var numberVal = UrlState.normalizeNumberParam(v);
                if (numberVal !== null) {
                    res.push(numberVal);
                }
            });
        }
        return res;
    };

    /**
     * Normalizes a string param.
     * @param defaultValue. What value to use in case in case of null, undefined or an empty string. defaultValue defaults to null
     */
    UrlState.normalizeStringParam = function (value, defaultValue) {
        // ensure we have a default value
        if (_.isUndefined(defaultValue)) {
            defaultValue = null;
        }

        // for arrays we use the first array element
        if (_.isArray(value) && value.length > 0) {
            var val = value[0]; // use the first value in case we got multiple ones
            if (val !== '') {
                return val;
            }
            else {
                return defaultValue;
            }
        }
        // for string simply apply the normalization
        if (_.isString(value)) {
            if (value !== '') {
                return value;
            }
            else {
                return defaultValue;
            }
        }
        // fall back to defaulValue
        return defaultValue;
    };

    /**
     * Removes empty arrays, strings, undefined, null parameters from a normalized parameter object
     */
    UrlState.removeEmptyParams = function (normalized) {
        var params = {};
        _.each(normalized, function (value, key) {
            if (_.isArray(value)) {
                if (value.length > 0) {
                    params[key] = value;
                }
            }
            else if (_.isString(value)) {
                // remove empty strings
                if (!_.isEmpty(value)) {
                    params[key] = value;
                }
            }
            else if (value === null || value === undefined) {
            } // ignore
            else {
                // if in doubt add
                params[key] = value;
            }
        });
        return params;
    };

    /**
     * Get the view state as request parameters.
     * The parameters are concatenated using &, no ? is prepended
     */
    UrlState.toUrlParameterString = function (params) {
        var query = '';
        _.each(params, function (value, key) {
            if (_.isArray(value)) {
                _.each(value, function (e) {
                    if (query.length > 0) {
                        query += '&';
                    }
                    query += key + "=" + encodeURIComponent(value);
                });
            }
            else {
                if (query.length > 0) {
                    query += '&';
                }
                query += key + "=" + encodeURIComponent(value);
            }
        });
        return query;
    };


    /**
     *
     * Handlers that provide the actual mapping functionality
     *
     */

    UrlState.setHandlers = function (handlers) {
        UrlState.handlers = handlers;
    };

    UrlState.handlers = [];

    /**
     *
     * Normalized <===> url
     *
     */

    /**
     * Returns the url params in a normalized way.
     */
    UrlState.getNormalizedUrlState = function () {
        var params = UrlState.getCurrentUrlState();
        var normalized = {};

        // let each handler do its thing
        _.each(UrlState.handlers, function (handler) {
            if (handler.isApplicable(normalized)) {
                handler.getNormalizedFromUrl(params, normalized);
            }
        });

        return normalized;
    };

    UrlState.toUrlState = function (normalized) {

        var urlParams = {};

        // let each handler do its thing
        _.each(UrlState.handlers, function (handler) {
            if (handler.isApplicable(normalized)) {
                handler.toUrl(normalized, urlParams);
            }
        });

        return urlParams;
    };

    /**
     * Called to execute a deferred push state. The push is ignored if pushDate is older than the last push/replace operation
     * @param pushDate
     */
    UrlState.deferredPushState = function (pushDate) {
        if (pushDate < UrlState.lastPushReplace) {
            return;
        }
        UrlState.pushState();
    };

    /**
     * Keeps track of the last time we changed the url
     */
    UrlState.lastPushReplace = 0;

    /**
     * Updates the url state with the current values
     */
    UrlState.pushState = function () {
        UrlState.lastPushReplace = new Date().getTime();

        // fetch then normalized rapid board state
        var normalized = UrlState.getNormalizedRapidBoardState();

        // remove empty values and reorder
        var params = UrlState.toUrlState(normalized);

        var urlNormalized = UrlState.getNormalizedUrlState();
        GH.log("url normalized : " + JSON.stringify(urlNormalized), GH.Logger.Contexts.state);
        GH.log("new normalized : " + JSON.stringify(normalized), GH.Logger.Contexts.state);
        if (_.isEqual(normalized, urlNormalized)) {
            GH.log("Not pushing, url state already up to date");
            return;
        }

        // push the state
        GH.log("pushing state  : " + JSON.stringify(params), GH.Logger.Contexts.state);
        GH.State.push(params);
    };

    /**
     * Replaces the url state with the current values
     */
    UrlState.replaceState = function (forceReplace) {
        UrlState.lastPushReplace = new Date().getTime();

        // fetch then normalized rapid board state
        var normalized = UrlState.getNormalizedRapidBoardState();

        // remove empty values and reorder
        var params = UrlState.toUrlState(normalized);

        var urlNormalized = UrlState.getNormalizedUrlState();
        GH.log("url normalized : " + JSON.stringify(urlNormalized), GH.Logger.Contexts.state);
        GH.log("new normalized : " + JSON.stringify(normalized), GH.Logger.Contexts.state);
        if (_.isEqual(normalized, urlNormalized) && !forceReplace) {
            GH.log("Not replacing, url state already up to date", GH.Logger.Contexts.state);
            return;
        }

        // replace the state
        GH.log("replacing state: " + JSON.stringify(params), GH.Logger.Contexts.state);
        GH.State.replace(params);
    };

    /**
     *
     * Board <===> normalized state
     *
     */

    /**
     * Returns the rapid board state in a normalized way
     */
    UrlState.getNormalizedRapidBoardState = function () {
        var normalized = {};

        // let each handler do its thing
        _.each(UrlState.handlers, function (handler) {
            if (handler.isApplicable(normalized)) {
                handler.getNormalizedFromInternal(normalized);
            }
        });

        return normalized;
    };

    /**
     * Updates the rapid board state with the url state
     */
    UrlState.loadState = function () {
        // fetch the normalized url params
        var normalized = UrlState.getNormalizedUrlState();

        // let each handler do its thing
        _.each(UrlState.handlers, function (handler) {
            if (handler.isApplicable(normalized)) {
                handler.updateInternalFromNormalized(normalized);
            }
        });
    };


    /**
     *
     * Current state fetching, comparison board <===> url
     *
     */

    /**
     * Get the current url state, not normalized
     */
    UrlState.getCurrentUrlState = function () {
        return GH.State.toViewState(window.location.href);
    };

    /**
     * Is there any state currently encoded in the url. This merely checks whether there are parameters present
     */
    UrlState.hasUrlState = function () {
        return !_.isEmpty(UrlState.getCurrentUrlState());
    };

    /**
     * Compares the state of the url to the current internal state
     */
    UrlState.isUrlStateUpToDate = function () {
        // fetch internal and url state
        var boardNormalized = UrlState.getNormalizedRapidBoardState();
        var urlNormalized = UrlState.getNormalizedUrlState();

        // compare
        return _.isEqual(boardNormalized, urlNormalized);
    };

    /**
     * Get the view state as request parameters.
     * The parameters are concatenated using &, no ? is prepended
     */
    UrlState.getViewStateRequestParameterString = function () {
        // fetch then normalized rapid board state
        var normalized = UrlState.getCurrentUrlState();

        // remove empty values
        var params = UrlState.removeEmptyParams(normalized);

        // transform to url parameter string
        return UrlState.toUrlParameterString(params);
    };

    UrlState.HISTORY_CHANGED = 'historyDidChange';

    /**
     * Called when the history changed.
     */
    UrlState.historyChanged = function () {
        if (UrlState.isUrlStateUpToDate()) {
            GH.log("UrlState.historyChanged state up to date", GH.Logger.Contexts.state);
            $(GH).trigger(UrlState.HISTORY_CHANGED);
            return;
        }

        GH.log("UrlState.historyChanged outdated state", GH.Logger.Contexts.state);
        UrlState.logDifferenceInState();

        // apply the new state
        UrlState.loadState();

        // for now we simply reload the complete board
        GH.RapidBoard.reload();
        $(GH).trigger(UrlState.HISTORY_CHANGED);
    };

    UrlState.logDifferenceInState = function () {
        var boardState = UrlState.getNormalizedRapidBoardState();
        var urlState = UrlState.getNormalizedUrlState();
        GH.log("boardState: " + JSON.stringify(boardState), GH.Logger.Contexts.state);
        GH.log("  urlState: " + JSON.stringify(urlState), GH.Logger.Contexts.state);
    };

    /**
     * Initializes the view state object
     */
    UrlState.init = function () {
        // register a listener for history change events
        // Delay registration, pre html 5 browsers fake the events, which is fired after the board is initialized
        // (we specifically want to ignore the initial pushState event)
        setTimeout(function () {
            GH.State.registerStateChangeListener(UrlState.historyChanged);
        }, 200);
    };

    return UrlState;
});

AJS.namespace("GH.RapidBoard.UrlState", null, require("jira-agile/rapid/state/url-state"));