/* global GH */
/**
 * Burndown Chart View
 * @module jira-agile/rapid/ui/chart/burndown-chart-view
 * @requires module:jquery
 * @requires module:underscore
 * @requires module:jira-agile/rapid/global-events
 * @requires module:jira-agile/rapid/ui/chart/burndown-chart-controller
 * @requires module:jira-agile/rapid/ui/chart/burndown-chart-model
 * @requires module:jira-agile/rapid/ui/chart/chart
 * @requires module:jira-agile/rapid/ui/chart/chart-colors
 * @requires module:jira-agile/rapid/ui/chart/chart-controller
 * @requires module:jira-agile/rapid/ui/chart/chart-view
 * @requires module:jira-agile/rapid/ui/chart/non-working-days-util
 */
define('jira-agile/rapid/ui/chart/burndown-chart-view', ['require'], function (require) {
    'use strict';

    // REQUIRES

    var $ = require('jquery');
    var _ = require('underscore');
    var BurndownChartController;
    var BurndownChartModel;
    var ReportController;
    var Chart = require('jira-agile/rapid/ui/chart/chart');
    var ChartColors = require('jira-agile/rapid/ui/chart/chart-colors');
    var ChartController = require('jira-agile/rapid/ui/chart/chart-controller');
    var ChartView = require('jira-agile/rapid/ui/chart/chart-view');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var NonWorkingDaysUtils = require('jira-agile/rapid/ui/chart/non-working-days-util');

    // GLOBALS... FIX ME
    var ChartTimeFrames = GH.ChartTimeFrames;
    var ChartTooltip = GH.ChartTooltip;
    var FlotChartUtils = GH.FlotChartUtils;
    var RapidBoard = GH.RapidBoard;
    var tpl = GH.tpl;

    // Resolve circular dependency
    GlobalEvents.on('pre-initialization', function () {
        BurndownChartController = require('jira-agile/rapid/ui/chart/burndown-chart-controller');
        BurndownChartModel = require('jira-agile/rapid/ui/chart/burndown-chart-model');
        ReportController = require('jira-agile/rapid/ui/chart/report-controller');
    });

    /**
     * Burndown Chart logic
     */
    var BurndownChartView = {};

    BurndownChartView.gadgetMode = false;
    BurndownChartView.wallboardMode = false;

    BurndownChartView.nonWorkingDaysShown = undefined;
    BurndownChartView.renderStats = true;
    BurndownChartView.renderNonWorkingDayControls = true;

    BurndownChartView.renderDataPoints = true;

    BurndownChartView.setNonWorkingDaysShown = function (workingDaysShown) {
        BurndownChartView.nonWorkingDaysShown = workingDaysShown;
    };

    BurndownChartView.setGadgetMode = function (gadgetMode) {
        BurndownChartView.gadgetMode = gadgetMode;
        BurndownChartView.renderStats = !gadgetMode;
        BurndownChartView.renderNonWorkingDayControls = !gadgetMode;

        BurndownChartView.renderDataPoints = !gadgetMode;
    };

    BurndownChartView.setWallboardMode = function (wallboardMode) {
        BurndownChartView.wallboardMode = wallboardMode;
    };

    /*
     * Show chart loading
     */

    BurndownChartView.renderChartLoading = function () {
        // controls structure
        ChartView.hideChartControls();

        // render the view
        ChartView.getChartContentElem(true).html(tpl.burndownchart.renderChartBody());

        // show the spinner
        ChartView.showSpinner();
    };

    /**
     * Shows warning that there is no issues to display.
     */
    BurndownChartView.showWarning = function (warningMessage) {
        var warning = AJS.$(tpl.burndownchart.renderWarningMessage({
            message: warningMessage
        }));
        warning.prependTo('#ghx-chart-content');
        ChartView.hideSpinner();
    };

    /**
     * Shows the chart for the provided data
     */
    BurndownChartView.showChart = function () {
        ChartView.hideSpinner();

        if (BurndownChartModel.getWarning()) {
            BurndownChartView.showWarning(BurndownChartModel.getWarning());
        }

        // do not display empty chart
        if (BurndownChartModel.isEmpty()) {
            return;
        }

        // then render the chart
        BurndownChartView.renderUI();

        if (!BurndownChartView.gadgetMode) {
            // handle re-open button
            BurndownChartView.handleReopenButton();
        }
    };

    BurndownChartView.handleReopenButton = function () {
        ReportController.renderReopenButton(ChartView.getReopenButton(true), RapidBoard.State.getRapidViewId(), BurndownChartController.sprintPicker.getItems(), BurndownChartController.sprintPicker.getSelectedItem());
    };

    /**
     * Render the entire UI: chart plus table
     */
    BurndownChartView.renderUI = function () {

        // ensure we show the entire chart area
        ChartView.showChartGroup();

        // render the chart area
        BurndownChartView.renderChart();

        // render the table statistic
        if (BurndownChartView.renderStats) {
            BurndownChartView.renderChartStats();
        }
    };

    /**
     * Renders the data table for the chart
     */
    BurndownChartView.renderChartStats = function () {
        // fetch the rows for the table
        var dataTable = BurndownChartModel.getDataTable();

        // render the info
        var dataDiv = $(tpl.burndownchart.renderBurndownChartTable({
            statistics: dataTable.statistics,
            rows: dataTable.rows
        }));
        $('#ghx-chart-data').empty().append(dataDiv);
    };

    /*
     * Decide whether to show a chart or error message if no working days
     * If we want to display a chart, then decide whether or not to show working days
     */

    BurndownChartView.renderChart = function () {
        if (!BurndownChartModel.hasAnyWorkingDays()) {
            $('#ghx-chart-wrap').hide();
            $('#ghx-chart-empty').html(tpl.burndownchart.renderNoWorkingDaysMessage({})).show();
            return;
        } else {
            $('#ghx-chart-empty').hide();
            $('#ghx-chart-wrap').show();
        }

        if (BurndownChartView.nonWorkingDaysShown) {
            BurndownChartView.renderChartWithNonWorkingDays();
        } else {
            BurndownChartView.renderChartWithoutNonWorkingDays();
        }
    };

    /*
     * Set chart options for chart with working days only then call render function
     */
    BurndownChartView.renderChartWithoutNonWorkingDays = function () {
        // fetch the series and stats
        var chartStartTime = BurndownChartModel.getChartStartTime();
        var series = NonWorkingDaysUtils.transformSeriesExcludingNonWorkingDays(BurndownChartModel.getSeries(), chartStartTime);
        BurndownChartView.showNotificationIfEmpty(series);

        var formatXAxisDate = function formatXAxisDate(originalVal, xAxis) {
            // calculate the date taking non working days into account
            var date = NonWorkingDaysUtils.transformDateIfInNonWorkingDays(originalVal, BurndownChartModel.getChartStartTime());

            var format = '%b %d';

            if (xAxis.tickSize[1] === 'hour') {
                format += ' %h:%M';
            }
            return $.plot.formatDate(date, format);
        };

        // setup the chart
        var options = BurndownChartView.getDefaultPlotOptions();
        options.xaxis.tickFormatter = formatXAxisDate;

        BurndownChartView.renderBurndownChart(series, options);
    };

    /*
     * Set chart options for non-working day chart then call render function
     */
    BurndownChartView.renderChartWithNonWorkingDays = function () {
        // fetch the series and stats
        var series = BurndownChartModel.getSeries();

        BurndownChartView.showNotificationIfEmpty(series);

        // setup the chart
        var options = BurndownChartView.getDefaultPlotOptions();
        if (!options.grid.markings) {
            options.grid.markings = ChartController.getNonWorkingBlocks;
        }
        BurndownChartView.renderBurndownChart(series, options);

        if (BurndownChartView.renderNonWorkingDayControls) {
            BurndownChartView.addNonWorkingDaysToLegend();
        }
    };

    /*
     * create the chart itself
     */
    BurndownChartView.renderBurndownChart = function (series, options) {

        // set the chart labels
        FlotChartUtils.setAndAlignAxisLabels(AJS.I18n.getText('gh.rapid.chart.cfd.xaxis.label'), BurndownChartView.getYAxisLabel());

        // element that holds the actual chart - do not use cached element
        var plotHolder = ChartView.getChartView(true);

        // plot the series
        var plot = Chart.draw(BurndownChartController.id, plotHolder, series, options);

        // register hooks to handle dot painting
        if (BurndownChartView.renderDataPoints) {
            plot.hooks.drawSeries.push(BurndownChartView.gatherPoints);
            plot.hooks.draw.push(FlotChartUtils.deferredRenderPoints);

            BurndownChartView.bindEvents(plotHolder);
        }

        if (BurndownChartView.renderNonWorkingDayControls) {
            BurndownChartView.addNonWorkingDaysToLegend();
        }
    };

    BurndownChartView.addNonWorkingDaysToLegend = function () {
        var legendTable = ChartView.getChartView().find('.legend').find('tbody');

        var checkbox = legendTable.find('#ghx-chart-show-non-working-days');
        if (checkbox.length === 0) {
            // only draw the checkbox if it's not already there
            var input = $(tpl.burndownchart.renderShowNonWorkingDaysOption({ checked: BurndownChartView.nonWorkingDaysShown }));
            legendTable.append(input);

            legendTable.find('#ghx-chart-show-non-working-days').click(BurndownChartView.toggleNonWorkingDays);
        }
    };

    BurndownChartView.toggleNonWorkingDays = function () {
        // toggle setting
        BurndownChartView.nonWorkingDaysShown = $('#ghx-chart-show-non-working-days').is(':checked');
        RapidBoard.State.setNonWorkingDaysShown('burndown', BurndownChartView.nonWorkingDaysShown);
        BurndownChartView.renderChart();
    };

    BurndownChartView.showNotificationIfEmpty = function (series) {
        if (_.isEmpty(series)) {
            var notification = $(tpl.burndownchart.renderBurndownNoSprint()).css({ 'margin-top': '0px' });
            // append to the body (which will give us the correct with)
            notification.prependTo('#ghx-chart-content');
        }
    };

    BurndownChartView.renderAndShowSprintPicker = function () {
        BurndownChartController.sprintPicker.render(ChartView.getChartSelector(true));
        ChartView.showChartGroup();
    };

    /**
     * Create an object containing the options common to chart view both with and without working days shown
     * @return {Object}
     */
    BurndownChartView.getDefaultPlotOptions = function () {

        var options = {
            xaxis: {
                mode: 'time'
            },
            yaxis: BurndownChartModel.getYAxis(),
            series: {
                points: {
                    show: false,
                    radius: 3,
                    symbol: 'circle'
                },
                lines: {
                    show: true
                }
            },
            grid: {
                hoverable: true,
                clickable: false,
                autoHighlight: false
            },
            mouseEnterExitEvents: true,
            legend: {
                container: null,
                backgroundOpacity: 0.5,
                position: 'ne'
            }
        };

        if (BurndownChartView.gadgetMode || BurndownChartView.wallboardMode) {
            options.series = {
                lines: {
                    show: true,
                    lineWidth: BurndownChartView.wallboardMode ? 4 : 2
                },
                points: { show: false },
                shadowSize: 0
            };

            if (BurndownChartView.wallboardMode) {
                // wallboard
                options.grid = {
                    backgroundColor: ChartColors.wallboard.backgroundColor,
                    borderWidth: 1, // in pixels
                    borderColor: ChartColors.wallboard.borderColor, // set if different from the grid color
                    color: ChartColors.wallboard.textColor, // primary color used for outline and labels
                    noGridBottomOffsetDelta: 0,
                    tickColor: ChartColors.wallboard.tickColor, // color used for the ticks
                    markings: BurndownChartView.getMarkingsForGadget
                };
                //            options.xaxis = {
                //                show: false // pending design review
                //            };
            } else {
                options.grid = {
                    color: '#333', // primary color used for outline and labels
                    borderWidth: 1, // in pixels
                    borderColor: '#999', // set if different from the grid color
                    tickColor: '#eee' // color used for the ticks
                };
            }
        }

        return options;
    };

    BurndownChartView.getMarkingsForGadget = function () {
        var wallboardStripes = ChartColors.nonWorkingDaysWallboard;
        var markings = ChartController.getNonWorkingBlocks();
        return _.map(markings, function (marking) {
            marking.color = wallboardStripes;
            return marking;
        });
    };
    /* Events */

    BurndownChartView.bindEvents = function (plotHolder) {
        // mouse enter/leave events
        plotHolder.unbind('flotMouseEnter flotMouseLeave').bind('flotMouseEnter flotMouseLeave', FlotChartUtils.handleMouseOverStateChange);

        // register plothover and plotclick event handlers
        // we first unbind here, otherwise we add on each invocation a new click handler
        plotHolder.unbind('plothover plotclick plotselected dblclick');
        plotHolder.bind('plothover', BurndownChartView.handlePlotHover);
        plotHolder.bind('plotclick', FlotChartUtils.handlePlotClick);
        plotHolder.bind('plotselected', BurndownChartView.handleRangeSelection);
        plotHolder.bind('dblclick', BurndownChartView.resetRangeSelection);
    };

    //
    // Tooltip support
    //

    /**
     * Handler for  plot hover events.
     */
    BurndownChartView.handlePlotHover = function (event, pos, item) {
        var $tooltip = $('#ghx-tooltip');
        // fetch the issue for that index
        if (item) {
            // fetch the data for the given point
            var seriesId = item.series.id;
            var dataIndex = item.dataIndex;
            var dataPoint = BurndownChartModel.getDataBySeriesAtIndex(seriesId, dataIndex);

            if (!_.isEmpty(dataPoint)) {
                if (BurndownChartView.previousPlotHoverPoint !== item.dataIndex) {
                    BurndownChartView.previousPlotHoverPoint = item.dataIndex;

                    // remove the previous tooltip and add a new one
                    $tooltip.remove();
                    BurndownChartView.showChartTooltip(item, seriesId, dataPoint, item.datapoint[0]);

                    // if the tooltip has no warnings then the issue is clickable
                    if (!$tooltip.hasClass('ghx-has-warnings')) {
                        ChartView.getChartView().addClass('ghx-clickable');
                    }
                }
                return;
            }
        }
        $tooltip.remove();
        BurndownChartView.previousPlotHoverPoint = null;
        ChartView.getChartView().removeClass('ghx-clickable');
    };

    /** Holds the last hovered point. */
    BurndownChartView.previousPlotHoverPoint = undefined;

    BurndownChartView.roundDuration = function (duration) {
        return duration !== 0 ? Math.round(duration * 10) / 10 : 0;
    };

    /**
     * Displays a data point tooltip
     */
    BurndownChartView.showChartTooltip = function (item, seriesId, dataPoint) {
        if (!dataPoint) {
            return;
        }

        // render the tooltip
        var tooltipData = BurndownChartModel.getTooltipData(seriesId, dataPoint);
        var tooltip = new ChartTooltip(item, tooltipData);
        tooltip.render();
    };

    /**
     * Gather the data points we want to draw
     *
     * @param plot : the chart object
     * @param ctx : the canvas 2d context
     * @param series : series currently being drawn
     */
    BurndownChartView.gatherPoints = function (plot, ctx, series) {
        if (!plot.mouseEntered) {
            return;
        } // Don't add data points when mouse is outside of the plot

        // fetch the data for the series
        var seriesId = series.id;
        var seriesData = BurndownChartModel.getDataBySeries(seriesId);
        if (!seriesData) {
            return;
        }

        var realPoints = FlotChartUtils.gatherPoints(series, seriesData);

        plot.realPointsPerSeries[seriesId] = realPoints;
    };

    /**
     * Get the Y axis label
     */
    BurndownChartView.getYAxisLabel = function () {
        return BurndownChartModel.getStatistic().name;
    };

    /**
     * Called when a range selection has been made
     */
    BurndownChartView.handleRangeSelection = function (event, ranges) {
        // set to custom and use from and to, round from to the day down, to to the day up
        //var from = ChartTimeFrames.roundMillisToBeginOfDay(ranges.xaxis.from);
        //var to = ChartTimeFrames.roundMillisToNextDay(ranges.xaxis.to);
        var from = ranges.xaxis.from;
        var to = ranges.xaxis.to;
        ChartTimeFrames.setNewChartLimits(-1, from, to);
    };

    /**
     * Called on a double click to reset the selection
     */
    BurndownChartView.resetRangeSelection = function (event) {
        // set to All Time
        ChartTimeFrames.setNewChartLimits(0, 0, 0);
    };

    return BurndownChartView;
});