/**
 * Chart filters
 */
GH.ChartFilters = {};

GH.ChartFilters.rapidViewConfig = {};

GH.ChartFilters.columns = [];
GH.ChartFilters.quickFilters = [];
GH.ChartFilters.swimlanes = [];
GH.ChartFilters.chartType = null;

/**
 * The default columns for the currently displayed chart. They're supplied on GH.ChartFilters.update, since they differ between CC and CFD.
 * These aren't actually column objects or column ids, as one might suspect, but they're booleans that match the respective active state
 * of the columns in GH.ChartFilters.columns. Which makes this essentially a bitset.
 */
GH.ChartFilters.defaultColumns = [];

/**
 * Initialize
 */
GH.ChartFilters.init = function() {
    GH.ChartUtils.registerInlineDialogHacks("chartFiltersInlineDialog");
};

/**
 * Refreshes the filters according to the given chart id
 */
GH.ChartFilters.update = function(chartType, rapidViewConfig, chartFilterConfig) {
    GH.ChartFilters.chartType = chartType;
    GH.ChartFilters.rapidViewConfig = rapidViewConfig;

    // register the defaults provided by the chartFilterConfig
    GH.ChartFilters.defaultColumns = chartFilterConfig.defaultColumns;

    // initialize the columns, quickFilters and swimlanes. This will setup the internal data structure for these filters
    GH.ChartFilters.initializeColumns(chartType);
    GH.ChartFilters.initializeQuickFilters(chartType);
    GH.ChartFilters.initializeSwimlanes(chartType);

    // load the active state configuration from the settings, or set default configurations if not available
    var defaultColumnIds = GH.ChartFilters.getDefaultColumnIds();
    var activeColumnIds = GH.RapidBoard.State.getNumberArrayBoardSetting(null, 'chart.filters.columns.'+chartType, defaultColumnIds);
    // upon first visit, the default will fall back to an empty array. Can't know in advance what this will be as we don't have a view
    // yet to figure out the ids to add. Simply use defaultColumnIds if no columns are selected
    if (activeColumnIds.length < 1) {
        activeColumnIds = defaultColumnIds;
    }
    GH.ChartFilters.setActiveColumnIds(activeColumnIds);

    var defaultQuickFilterIds = GH.ChartFilters.getDefaultQuickFilterIds();
    var activeQuickFilterIds = GH.RapidBoard.State.getNumberArrayBoardSetting(null, 'chart.filters.quickFilters', defaultQuickFilterIds);
    GH.ChartFilters.setActiveQuickFilterIds(activeQuickFilterIds);

    var defaultSwimlaneIds = GH.ChartFilters.getDefaultSwimlaneIds();
    var activeSwimlaneIds = GH.RapidBoard.State.getNumberArrayBoardSetting(null, 'chart.filters.swimlanes', defaultSwimlaneIds);
    // same as for column. Use all if none selected
    if (activeSwimlaneIds.length < 1) {
        activeSwimlaneIds = defaultSwimlaneIds;
    }
    GH.ChartFilters.setActiveSwimlaneIds(activeSwimlaneIds);

    // update the filter info
    GH.ChartFilters.updateFilterInfo();
};

/**
 * Updates the filter information according to the current selection
 */
GH.ChartFilters.updateFilterInfo = function() {
    // check whether we are in the default configuration
    var isDefault = GH.ChartFilters.isDefaultColumns() &&
                    GH.ChartFilters.isDefaultQuickFilters() &&
                    GH.ChartFilters.isDefaultSwimlanes();

    // add or remove the info
    var refineButton = AJS.$('#js-chart-filter');
    var refineIconSpan = AJS.$(refineButton).find('.aui-icon');
    var refineIconText = AJS.$(refineIconSpan).text();
    if (isDefault) {
        refineButton.addClass('ghx-icon-hide');
        refineIconSpan.removeAttr('title');
    } else {
        refineButton.removeClass('ghx-icon-hide');
        refineIconSpan.attr('title',refineIconText);
    }
};

/**
 * Adds request parameters required for the current filters
 */
GH.ChartFilters.addRequestParams = function(data) {
    // add all selected swimlanes
    var swimlaneIds = [];
    _.each(GH.ChartFilters.swimlanes, function(swimlane, index) {
        if (swimlane.active) {
            swimlaneIds.push(swimlane.id);
        }
    });
    if (swimlaneIds.length > 0) {
        data.swimlaneId = swimlaneIds;
    }

    // add all selected quick filters
    var quickFilterIds = [];
    _.each(GH.ChartFilters.quickFilters, function(quickFilter, index) {
        if (quickFilter.active) {
            quickFilterIds.push(quickFilter.id);
        }
    });
    if (quickFilterIds.length > 0) {
        data.quickFilterId = quickFilterIds;
    }

    // add all selected columns
    var columnIds = [];
    _.each(GH.ChartFilters.columns, function(column) {
        if (column.active) {
            columnIds.push(column.id);
        }
    });
    if (columnIds.length > 0) {
        data.columnId = columnIds;
    }
};

/*
 *
 * Columns
 *
 */

GH.ChartFilters.initializeColumns = function(chartType) {
    // create the internal columns structure
    var inputColumns = GH.ChartFilters.rapidViewConfig.columns;
    var columns = [];
    for (var i=0; i < inputColumns.length; i++) {
        var c = inputColumns[i];
        columns.push({
            id: c.id,
            name: c.name,
            active: false
        });
    }
    GH.ChartFilters.columns = columns;
};

/**
 * Get all columns
 */
GH.ChartFilters.getColumns = function() {
    return GH.ChartFilters.columns;
};

/**
 * Returns whether the current configuration is the default one
 */
GH.ChartFilters.isDefaultColumns = function() {
    var defaultActive = GH.ChartFilters.getDefaultColumns();
    for (var i=0; i < GH.ChartFilters.columns.length; i++) {
        if (GH.ChartFilters.columns[i].active != defaultActive[i]) {
            return false;
        }
    }
    return true;
};

/**
 * Returns the ids of the default active columns
 */
GH.ChartFilters.getDefaultColumnIds = function() {
    var defaultColumns = GH.ChartFilters.getDefaultColumns();
    var defaultIds = [];
    for (var i=0; i < GH.ChartFilters.columns.length; i++) {
        if (defaultColumns[i]) {
            defaultIds.push(GH.ChartFilters.columns[i].id);
        }
    }
    return defaultIds;
};

/**
 * @returns the columns that are currently selected in the filter
 */
GH.ChartFilters.getActiveColumns = function() {
    return _.filter(GH.ChartFilters.columns, function(column) {
        return column.active;
    });
};

/**
 * Returns the ids of the active columns
 */
GH.ChartFilters.getActiveColumnIds = function() {
    var activeIds = [];
    for (var i=0; i < GH.ChartFilters.columns.length; i++) {
        if (GH.ChartFilters.columns[i].active) {
            activeIds.push(GH.ChartFilters.columns[i].id);
        }
    }
    return activeIds;
};

/**
 * Activates the provided columns given their ids
 */
GH.ChartFilters.setActiveColumnIds = function(columnIds) {
    _.each(GH.ChartFilters.columns, function(column) {
        column.active=false;
        _.each(columnIds, function(columnId) {
            if (columnId == column.id) {
                column.active = true;
            }
        });
    });
    // update the state
    GH.RapidBoard.State.setNumberArrayBoardSetting('chart.filters.columns.'+GH.ChartFilters.chartType, columnIds);
};

/**
 * Returns a boolean array that says whether the column at position i is active
 */
GH.ChartFilters.getDefaultColumns = function() {
    return GH.ChartFilters.defaultColumns;
};

/**
 * Returns whether a column is active, given the column id
 */
GH.ChartFilters.isColumnActive = function(id) {
    var col = GH.ChartFilters.getColumn(id);
    if (col) {
        return col.active;
    }
    return false;
};

/**
 * Sets a column as active or inactive
 */
/*
GH.ChartFilters.setColumnActive = function(id, active) {
    var col = GH.ChartFilters.getColumn(id);
    if (!col) return false;
    if (col.active == active) return false;
    col.active = active;
    return true;
};
*/

/**
 * Returns the column data for a column given its id
 */
GH.ChartFilters.getColumn = function(id) {
    for (var i=0; i < GH.ChartFilters.columns.length; i++) {
        var c = GH.ChartFilters.columns[i];
        if (c.id == id) {
            return c;
        }
    }
    return false;
};



/*
 *
 * Quick filters
 *
 */

GH.ChartFilters.initializeQuickFilters = function(chartType) {
    // create the internal columns structure
    var inputFilters = GH.ChartFilters.rapidViewConfig.quickFilters;
    var quickFilters = [];
    for (var i=0; i < inputFilters.length; i++) {
        var f = inputFilters[i];
        quickFilters.push({
            id: f.id,
            name: f.name,
            active: false
        });
    }
    GH.ChartFilters.quickFilters = quickFilters;
};

/**
 * Checks whether the default configuration is selected
 */
GH.ChartFilters.isDefaultQuickFilters = function() {
    return GH.ChartFilters.getActiveQuickFilterIds().length === 0;
};

GH.ChartFilters.getDefaultQuickFilterIds = function() {
    return [];
};

GH.ChartFilters.getActiveQuickFilterIds = function() {
    var activeIds = [];
    _.each(GH.ChartFilters.quickFilters, function(quickFilter) {
        if (quickFilter.active) {
            activeIds.push(quickFilter.id);
        }
    });
    return activeIds;
};

GH.ChartFilters.setActiveQuickFilterIds = function(quickFilterIds) {
    _.each(GH.ChartFilters.quickFilters, function(quickFilter) {
        quickFilter.active = false;
        _.each(quickFilterIds, function(quickFilterId) {
            if (quickFilterId == quickFilter.id) {
                quickFilter.active = true;
            }
        });
    });
    // update the state
    GH.RapidBoard.State.setNumberArrayBoardSetting('chart.filters.quickFilters', quickFilterIds);
};

/**
 * Get all columns
 */
GH.ChartFilters.getQuickFilters = function() {
    return GH.ChartFilters.quickFilters;
};

/**
 * Returns whether a column is active, given the column id
 */
GH.ChartFilters.isQuickFilterActive = function(id) {
    var filter = GH.ChartFilters.getQuickFilter(id);
    if (filter) {
        return filter.active;
    }
    return false;
};

/**
 * Sets a column as active or inactive
 */
/*
GH.ChartFilters.setQuickFilterActive = function(id, active) {
    var filter = GH.ChartFilters.getQuickFilter(id);
    if (!filter) return false;
    if (filter.active == active) return false;
    filter.active = active;
    return true;
};
*/

/**
 * Returns the column data given its id
 */
GH.ChartFilters.getQuickFilter = function(id) {
    for (var i=0; i < GH.ChartFilters.quickFilters.length; i++) {
        var f = GH.ChartFilters.quickFilters[i];
        if (f.id == id) {
            return f;
        }
    }
    return false;
};


/*
 *
 * Swimlanes
 *
 */

GH.ChartFilters.initializeSwimlanes = function(chartType) {
    // create the internal columns structure
    var inputFilters = GH.ChartFilters.rapidViewConfig.swimlanes;
    var swimlanes = [];
    for (var i=0; i < inputFilters.length; i++) {
        var f = inputFilters[i];
        swimlanes.push({
            id: f.id,
            name: f.name,
            active: false
        });
    }
    GH.ChartFilters.swimlanes = swimlanes;
};

GH.ChartFilters.isDefaultSwimlanes = function() {
    // only the case if all swimlanes are selected
    return GH.ChartFilters.getDefaultSwimlaneIds().length == GH.ChartFilters.getActiveSwimlaneIds().length;
};

/**
 * Get the default swimlane ids that should be selected
 */
GH.ChartFilters.getDefaultSwimlaneIds = function() {
    var swimlaneIds = [];
    _.each(GH.ChartFilters.swimlanes, function(swimlane) {
        swimlaneIds.push(swimlane.id);
    });
    return swimlaneIds;
};

/**
 * Returns the ids of the active swimlanes
 */
GH.ChartFilters.getActiveSwimlaneIds = function() {
    var swimlaneIds = [];
    _.each(GH.ChartFilters.swimlanes, function(swimlane) {
        if (swimlane.active) {
            swimlaneIds.push(swimlane.id);
        }
    });
    return swimlaneIds;
};

/**
 * Sets the ids of the active swimlanes
 */
GH.ChartFilters.setActiveSwimlaneIds = function(swimlaneIds) {
    // GHS-4585: detect when swimlane IDs are specified which are not valid
    var validSwimlaneIds = _.filter(swimlaneIds, function(swimlaneId) {
        return _.find(GH.ChartFilters.swimlanes, function(swimlane) {
            return (swimlaneId == swimlane.id);
        });
    });

    if (validSwimlaneIds.length === 0) {
        // need to use defaults -- all swimlanes in config
        validSwimlaneIds = _.pluck(GH.ChartFilters.swimlanes, "id");
    }

    _.each(GH.ChartFilters.swimlanes, function(swimlane) {
        swimlane.active = false;
        _.each(validSwimlaneIds, function(swimlaneId) {
            if (swimlaneId == swimlane.id) {
                swimlane.active = true;
            }
        });
    });

    // update the state
    GH.RapidBoard.State.setNumberArrayBoardSetting('chart.filters.swimlanes', validSwimlaneIds);
};

/**
 * Get all swimlanes
 */
GH.ChartFilters.getSwimlanes = function() {
    return GH.ChartFilters.swimlanes;
};

/**
 * Returns whether a swimlane is active, given the swimlane id
 */
GH.ChartFilters.isSwimlaneActive = function(id) {
    var swimlane = GH.ChartFilters.getSwimlane(id);
    if (swimlane) {
        return swimlane.active;
    }
    return false;
};

/**
 * Sets a swimlane as active or inactive
 */
/*
GH.ChartFilters.setSwimlaneActive = function(id, active) {
    var swimlane = GH.ChartFilters.getSwimlane(id);
    if (!swimlane) return false;
    if (swimlane.active == active) return false;
    swimlane.active = active;
    return true;
};
*/

/**
 * Returns the swimlane data given its id
 */
GH.ChartFilters.getSwimlane = function(id) {
    for (var i=0; i < GH.ChartFilters.swimlanes.length; i++) {
        var s = GH.ChartFilters.swimlanes[i];
        if (s.id == id) {
            return s;
        }
    }
    return false;
};

/**
 * Renders the dropdown menu for the Chart Menu items & attach events
 */
GH.ChartFilters.initializeDialog = function() {
    // kill any previous inline dialogs
    AJS.$('#inline-dialog-chartFiltersInlineDialog').remove();
    if (AJS.InlineDialog.current) {
        AJS.InlineDialog.current.hide();
    }

    // initialize inline dialogs
    var options = {
        width: 500,
        cacheContent : false, // don't cache the dialog content
        hideDelay : 60000 // set longer timeout (default is 10 seconds)
    };
    AJS.InlineDialog("#js-chart-filter", "chartFiltersInlineDialog", GH.ChartFilters.renderChartFiltersDialog, options);
};


/**
 * Renders a chart filters dialog
 */
GH.ChartFilters.renderChartFiltersDialog = function(contents, trigger, showPopup) {
    var columns = GH.ChartFilters.getColumns();
    var swimlanes = GH.ChartFilters.getSwimlanes();
    var quickFilters = GH.ChartFilters.getQuickFilters();

    contents.html(GH.tpl.chartfilters.renderChartFilters({
        'columns': columns,
        'swimlanes': swimlanes,
        'quickFilters': quickFilters
        }));

    // mark this for GH styling
    contents.closest('.aui-inline-dialog').addClass('ghx-inline-dialog');

    // handle filter clicks - we can use the same for columns, swimlanes and quickFilters
    contents.find('.js-toggle').click(function(e) {
        e.preventDefault(); // stop the button submitting early
        var button = AJS.$(this);
        var isActive = button.hasClass('active');
        var newState = !isActive;
        if (newState) {
            button.addClass('active');
        } else {
            button.removeClass('active');
        }
        // check whether the configuration is valid for quick feedback
        GH.ChartFilters.checkFilterValidity(contents);
    });

    // handle click on update and cancel
    contents.find('.js-save-button').click(function(e) {
        e.preventDefault();
        GH.ChartFilters.handleDialogSave(contents);
    });
    contents.find('.js-cancel-button').click(function(e) {
        AJS.InlineDialog.current.hide();
    });

    showPopup();
};

/**
 * Checks whether all filters are valid, or shows error if not.
 * @return true if valid
 */
GH.ChartFilters.checkFilterValidity = function(contents) {
    // clear errors
    contents.find('.ghx-error').remove();

    var valid = true;

    // check columns
    if (contents.find('.js-column-options .js-toggle.active').length < 1) {
        valid = false;
        contents.find('.js-column-options').after('<div class="ghx-error">'+AJS.I18n.getText('gh.rapid.charts.filters.error.nocolumns')+'</div>');
    }

    // check swimlanes
    if (contents.find('.js-swimlane-options .js-toggle.active').length < 1) {
        valid = false;
        contents.find('.js-swimlane-options').after('<div class="ghx-error">'+AJS.I18n.getText('gh.rapid.charts.filters.error.noswimlanes')+'</div>');
    }

    return valid;
};

/**
 * Called to save the dialog data
 */
GH.ChartFilters.handleDialogSave = function handleDialogSave(contents) {
    // ensure the configuration is valid
    if (!GH.ChartFilters.checkFilterValidity(contents)) {
        return;
    }

    // update the columns
    var columnIds = [];
    contents.find('.js-column-options .js-toggle').each(function(index, elem) {
        var button = AJS.$(elem);
        var id = button.attr('data-column-id');
        var isActive = button.hasClass('active');
        if (isActive) {
            columnIds.push(parseInt(id, 10));
        }
    });

    // update the swimlanes
    var swimlaneIds = [];
    contents.find('.js-swimlane-options .js-toggle').each(function(index, elem) {
        var button = AJS.$(elem);
        var id = button.attr('data-swimlane-id');
        var isActive = button.hasClass('active');
        if (isActive) {
            swimlaneIds.push(parseInt(id, 10));
        }
    });

    // update the quickFilters
    var quickFilterIds = [];
    contents.find('.js-quickfilter-options .js-toggle').each(function(index, elem) {
        var button = AJS.$(elem);
        var id = button.attr('data-quickfilter-id');
        var isActive = button.hasClass('active');
        if (isActive) {
            quickFilterIds.push(parseInt(id, 10));
        }
    });

    // compare to previous state before updating
    var swimlanesChanged = !_.isEqual(swimlaneIds, GH.ChartFilters.getActiveSwimlaneIds());
    var quickFiltersChanged = !_.isEqual(quickFilterIds, GH.ChartFilters.getActiveQuickFilterIds());
    var columnsChanged = !_.isEqual(columnIds, GH.ChartFilters.getActiveColumnIds());

    // set the new values
    GH.ChartFilters.setActiveColumnIds(columnIds);
    GH.ChartFilters.setActiveSwimlaneIds(swimlaneIds);
    GH.ChartFilters.setActiveQuickFilterIds(quickFilterIds);

    // push the new values
    GH.RapidBoard.State.pushState();

    // fire an event
    AJS.$(GH).trigger('chartFiltersChanged', {
        swimlanesChanged: swimlanesChanged,
        quickFiltersChanged: quickFiltersChanged,
        columnsChanged: columnsChanged
    });

    // update the filter button
    GH.ChartFilters.updateFilterInfo();

    // hide dialog
    AJS.InlineDialog.current.hide();
};
