(function (_, d3, timeFrames) {
    // Note: there is an implicit dependency on Calendar.js due to the use of
    // Date.parseDate and Date.prototype.print.

    var NORMALIZED_DATE_FORMAT = '%Y-%m-%d';
    var DEFAULT_TIME_FRAME = 90;
    var CUSTOM_TIME_FRAME = 'custom';
    var day = d3.time.day;

    function checkIsDate(date, paramName) {
        if (!_.isDate(date)) {
            throw paramName + " must be a Date object";
        }
        return date;
    }

    /**
     * Time 00:00 of the next day.
     *
     * @param {Date} date
     * @returns {Date}
     */
    function nextDay(date) {
        return day.floor(day.offset(date, 1));
    }

    /**
     * Time 23:59 of the given day.
     *
     * @param {Date} date
     * @returns {Date}
     */
    function endOfDay(date) {
        return new Date(nextDay(date) - 1);
    }

    /**
     * Represents a chart's time frame settings.
     *
     * @constructor
     */
    function ChartTimeFrameModel() {}

    ChartTimeFrameModel.prototype = {
        /**
         * @type {string|number}
         */
        timeFrame: null,

        /**
         * @type {Date}
         */
        fromDate: null,

        /**
         * @type {Date}
         */
        toDate: null,

        /**
         * Set time frame based on from and to dates.
         * These dates are treated as indicating the day only - hours, minutes, seconds and milliseconds are ignored.
         * Both from and to dates are inclusive.
         *
         * @param {Date} fromDate
         * @param {Date} toDate
         * @returns {ChartTimeFrameModel}
         */
        setSimpleDates: function setSimpleDates(fromDate, toDate) {
            this.timeFrame = CUSTOM_TIME_FRAME;
            this.fromDate = day.floor(checkIsDate(fromDate, 'fromDate'));
            this.toDate = endOfDay(checkIsDate(toDate, 'toDate'));
            return this;
        },

        /**
         * Set time frame based on from and to dates.
         * These dates are treated as exact, to the millisecond.
         * Both from and to dates are inclusive.
         *
         * @param {Date} fromDate
         * @param {Date} toDate
         * @returns {ChartTimeFrameModel}
         */
        setExactDates: function setExactDates(fromDate, toDate) {
            this.timeFrame = CUSTOM_TIME_FRAME;
            this.fromDate = checkIsDate(fromDate);
            this.toDate = checkIsDate(toDate);
            return this;
        },

        /**
         * Set time frame based on number of days before the current date.
         *
         * @param {number} days
         * @returns {ChartTimeFrameModel}
         */
        setDays: function setDays(days) {
            this.timeFrame = days;
            return this;
        },

        /**
         * Set time frame to be from the earliest available date to the current date.
         *
         * @returns {ChartTimeFrameModel}
         */
        setAllTime: function setAllTime() {
            this.timeFrame = 0;
            return this;
        },

        /**
         * Set time frame to be the default time frame.
         *
         * @returns {ChartTimeFrameModel}
         */
        resetToDefault: function resetToDefault() {
            this.timeFrame = DEFAULT_TIME_FRAME;
            return this;
        },

        /**
         * Returns boolean indicating if ChartTimeFrameModel is in default state
         *
         * @returns {boolean}
         */
        isDefault: function isDefault() {
            return this.timeFrame === DEFAULT_TIME_FRAME;
        },

        /**
         * Set the earliest date and the current date so that actual dates can be calculated
         * for time frames set to All Time or number of days before the current date.
         *
         * @param {Date} earliestDate
         * @param {Date} currentDate
         * @returns {ChartTimeFrameModel}
         */
        setBounds: function setBounds(earliestDate, currentDate) {
            this.earliestDate = checkIsDate(earliestDate, 'earliestDate');
            this.currentDate = checkIsDate(currentDate, 'currentDate');
            return this;
        },

        /**
         * Returns the time frame represented as days before the current date, 'custom', or 0 for all time.
         *
         * If the number of days is not found in timeFrames, it will return custom.
         *
         * @returns {string|number}
         */
        getTimeFrame: function getTimeFrame() {
            if (!_.findWhere(timeFrames, { days: this.timeFrame })) {
                return CUSTOM_TIME_FRAME;
            }
            return this.timeFrame;
        },

        /**
         * Returns an object containing the from and to Date objects that represent the time frame.
         * Both from and to dates are inclusive.
         *
         * @returns {{fromDate: Date, toDate: Date}}
         */
        getTimeFrameDates: function getTimeFrameDates() {
            if (!this.earliestDate || !this.currentDate) {
                throw "Cannot calculate time frame dates without earliestDate and currentDate";
            }

            var fromDate;
            var toDate;
            if (this.timeFrame > 0) {
                // timeFrame is a set amount of days
                fromDate = day.offset(nextDay(this.currentDate), -1 * this.timeFrame);
                toDate = endOfDay(this.currentDate);
            } else if (this.timeFrame <= 0) {
                // timeFrame is all time
                fromDate = day.floor(this.earliestDate);
                toDate = endOfDay(this.currentDate);
            } else if (this.timeFrame === CUSTOM_TIME_FRAME) {
                // time window is custom
                fromDate = this.fromDate;
                toDate = this.toDate;
            }
            return {
                fromDate: fromDate,
                toDate: toDate
            };
        },

        /**
         * Returns the duration in milliseconds of the time frame.
         *
         * @returns {number}
         */
        getTimeFrameDuration: function getTimeFrameDuration() {
            var dates = this.getTimeFrameDates();
            return dates.toDate - dates.fromDate + 1;
        },

        /**
         * Updates the model using the normalized representation of the model.
         *
         * @param {Object} normalized
         * @returns {ChartTimeFrameModel}
         */
        fromNormalized: function fromNormalized(normalized) {
            if (normalized.days == null) {
                this.resetToDefault();
            } else if (normalized.days > 0) {
                this.setDays(Number(normalized.days));
            } else if (normalized.days === CUSTOM_TIME_FRAME) {
                try {
                    this.setSimpleDates(Date.parseDate(normalized.from, NORMALIZED_DATE_FORMAT), Date.parseDate(normalized.to, NORMALIZED_DATE_FORMAT));
                } catch (e) {
                    // if the normalized dates are not valid, switch to the default case
                    this.resetToDefault();
                }
            } else {
                this.setAllTime();
            }
            return this;
        },

        /**
         * Returns an object containing the internal state of the model in a normalized format.
         *
         * @returns {{days: number|string, from: string, to: string}}
         */
        toNormalized: function toNormalized() {
            var normalized = {};
            if (_.isNumber(this.timeFrame) && this.timeFrame !== DEFAULT_TIME_FRAME) {
                normalized.days = this.timeFrame;
            } else if (this.timeFrame === CUSTOM_TIME_FRAME) {
                normalized.days = this.timeFrame;
                normalized.from = this.fromDate.print(NORMALIZED_DATE_FORMAT);
                normalized.to = this.toDate.print(NORMALIZED_DATE_FORMAT);
            }
            return normalized;
        },

        /**
         * Creates a clone of this model. Bounds are not copied.
         *
         * @returns {ChartTimeFrameModel}
         */
        clone: function clone() {
            return new ChartTimeFrameModel().fromNormalized(this.toNormalized());
        }
    };

    GH.Reports.ChartTimeFrameModel = ChartTimeFrameModel;
})(_, d3, GH.ChartTimeFrames.timeFrames);