/* Globals c3 */
define('jira-agile/rapid/ui/chart/v2/burnup/burnup-chart', ['require'], function (require) {
    'use strict';

    // REQUIRES

    var _ = require('underscore');
    var d3 = require('jira-agile/d3');
    var GlobalEvents = require('jira-agile/rapid/global-events');

    // GLOBALS.. FIX ME
    var _c3;
    var FlotChartUtils;

    GlobalEvents.on("pre-initialization", function () {
        _c3 = c3;
        FlotChartUtils = GH.FlotChartUtils;
    });

    function SprintBurnupChart() {}

    SprintBurnupChart.prototype = {
        hide: function hide(container) {
            GH.Reports.BurnupChartDialog.hide();
            container.html('');
        },

        show: function show(model, container) {

            // clear the chart
            container.html('');

            var scrubber = GH.Reports.BurnupChartScrubber();

            container = container.append("svg").attr('height', parseInt(container.style('height'))).attr('width', parseInt(container.style('width')));

            var xBounds = [model.start, model.end];
            var yBounds = [d3.min(model.scope.concat(model.work), function (d) {
                return d.value;
            }), d3.max([{ value: 8 }].concat(model.scope.concat(model.work)), function (d) {
                return d.value;
            })];

            var chart = _c3.component().extend(function () {
                scrubber.enableScrubber();
                GH.Reports.BurnupChartDialog.hide();
            }).extend(_c3.borderLayout()).extend(_c3.utils.defaultViewBox).extend({
                xDomain: function xDomain() {
                    var domain = xBounds;
                    _.each(domain, function (n) {
                        _c3.checkIsNumber(n);
                    });
                    return domain;
                },
                yDomain: function yDomain() {
                    var domain = yBounds;
                    _.each(domain, function (n) {
                        _c3.checkIsNumber(n);
                    });
                    return domain;
                },
                clusters: _c3.prop()
            }).xScaleConstructor(d3.time.scale).yScaleConstructor(d3.scale.linear);

            var xAxis = _c3.labelledAxis().axisConstructor(function () {
                return d3.svg.axis().tickFormat(function (timestamp) {
                    return model.formatShortDate(timestamp);
                });
            }).extend(function () {
                // Remove all ticks positioned before start of axis
                // Temporary workaround while we sort out bad Date object effects on d3
                this.selection().selectAll('g.tick').filter(function () {
                    return this.getAttribute('transform').indexOf('translate(-') !== -1;
                }).remove();
            }).height(50).text(AJS.I18n.getText('gh.rapid.chart.cfd.xaxis.label')).orient('bottom');

            var yTicks = this.calculateYTicks(yBounds[1], model.statisticField);

            var yAxis = _c3.labelledAxis().axisConstructor(function () {
                return d3.svg.axis().tickValues(yTicks.tickValues).tickFormat(model.formatStatistic).ticks(8);
            }).width(60).text(model.statisticField.name).orient('left');

            var fillingRect = _c3.fillWithRect().elementClass('ghx-dragging');

            var guideline = this.buildGuideline(model);

            var plots = _c3.component().extend(_c3.interactive().domContext(AJS.$("#ghx-chart-group")).region(fillingRect)).extend(_c3.layerable());

            if (!model.sprintCompleted) {
                // sprint in progress - add marker showing current time.
                plots.addLayer('today-line', _c3.linePlot().extend({
                    data: function data() {
                        return [[model.now, yBounds[0]], [model.now, yBounds[1]]];
                    }
                }).elementClass('today-line')).addLayer('today-line-label', _c3.label().extend({
                    width: function width() {
                        // likes to put the label in the middle of the thing.  2x gets it
                        //   to center of over the line
                        return chart.xScale()(model.now) * 2;
                    },
                    height: function height() {
                        return chart.yScale()(yBounds[1]);
                    },
                    orient: function orient() {
                        return 'bottom';
                    },
                    text: function text() {
                        return AJS.I18n.getText('gh.chart.today');
                    }
                }));
            }

            plots.extend(scrubber)
            //.clipped(true)
            .addLayer('grid', _c3.gridLines().orient('left')).addLayer('non-work-days', _c3.areaPlot().extend({
                data: function data() {
                    // build a set of points for each non work period
                    return chart.data().nonWorkPeriods.map(function (p) {
                        return [[p.start, yBounds[0]], [p.start, yBounds[1]], [p.end, yBounds[1]], [p.end, yBounds[0]]];
                    }).reduce(function (prev, curr) {
                        // flatten them into a single array.
                        return prev.concat(curr);
                    }, []);
                }
            }).elementClass('non-working-days')).addLayer('guideline', _c3.linePlot().extend({
                data: function data() {
                    return guideline;
                }
            }).elementClass('guide-line')).addLayer('scope-projection', _c3.linePlot().extend({
                data: function data() {
                    var currentScope = chart.data().scope[model.scope.length - 1];
                    return [[currentScope.timestamp, currentScope.value], [model.end, currentScope.value]];
                }
            }).elementClass('scope-projection-line')).addLayer('scope', this.steppedLinePlot().extend({
                data: function data() {
                    return chart.data().scope.map(function (o) {
                        return [o.timestamp, o.value];
                    });
                }
            }).elementClass('scope-line')).addLayer('work', this.steppedLinePlot().extend({
                data: function data() {
                    return chart.data().work.map(function (o) {
                        return [o.timestamp, o.value];
                    });
                }
            }).elementClass('work-line')).addLayer('filler', fillingRect).addLayer('scope-marker', _c3.circlePlot().xAccessor(function (d) {
                return d.timestamp;
            }).yAccessor(function (d) {
                return d.value;
            }).extend({
                data: function data() {
                    // don't put markers on the first (start sprint) and last (end sprint) points
                    return chart.data().scope.slice(1, -1);
                }
            }).enter(function (event) {
                event.selection.on('click.dialog', function (d) {
                    GH.Reports.BurnupChartDialog.show(this, model, d);

                    // don't show the scrubber if the issue dialog is open
                    scrubber.hideScrubber();
                });
            }).elementClass('scope-line-marker')).addLayer('work-marker', _c3.circlePlot().xAccessor(function (d) {
                return d.timestamp;
            }).yAccessor(function (d) {
                return d.value;
            }).extend({
                data: function data() {
                    // don't put markers on the first (start sprint) and last (end sprint) points
                    return chart.data().work.slice(1, -1);
                }
            }).enter(function (event) {
                event.selection.on('click.dialog', function (d) {
                    GH.Reports.BurnupChartDialog.show(this, model, d);

                    // don't show the scrubber if the issue dialog is open
                    scrubber.hideScrubber();
                });
            }).elementClass('work-line-marker'));

            chart = chart.center(plots).north(_c3.withDimensions().height(20)).south(xAxis).east(_c3.withDimensions().width(40)).west(yAxis);

            chart.data(model);

            chart(container);
        },

        buildGuideline: function buildGuideline(model) {

            // guideline starts at starttime, 0 scope.
            var guideline = [[model.start, 0]];

            // the target scope to aim for
            var finalScope = model.scope[model.scope.length - 1].value;

            // calculate the total milliseconds available for work
            var availableMs = model.end - model.start;
            model.nonWorkPeriods.forEach(function (p) {
                availableMs -= p.end - p.start;
            });

            var nonWorkMsCompleted = 0;

            // build the line up by drawing flat lines through each non-work period and joining the start / ends between them
            model.nonWorkPeriods.forEach(function (p) {

                // how many ms of work is completed at this point in time.
                var completeMs = p.start - model.start;

                // exclude all time from nonwork periods in the past.
                completeMs -= nonWorkMsCompleted;

                // ensure faction doesn't go above 1 (rounding errors).
                var fractionComplete = Math.min(completeMs / availableMs, 1);

                guideline.push([p.start, fractionComplete * finalScope]);
                guideline.push([p.end, fractionComplete * finalScope]);

                // update the nonWorkPeriods in the past to include this one.
                nonWorkMsCompleted += p.end - p.start;
            });

            // guideline always finishes at the end time at final scope.
            guideline.push([model.end, finalScope]);

            return guideline;
        },

        steppedLinePlot: function steppedLinePlot() {
            return _c3.component('steppedlinePlot').extend(_c3.linePlot()).update(function (event) {
                var line = this.lineConstructor()().x(this.x()).y(this.y());

                line.interpolate('step-after');

                event.selection.attr('d', line(this.data())).attr('stroke', 'black').attr('fill', 'none');
            });
        },

        calculateYTicks: function calculateYTicks(maxYValue, statisticField) {
            var flotYAxis = FlotChartUtils.calculateYAxis(maxYValue, statisticField);
            var tickSize = flotYAxis.tickSize || flotYAxis.minTickSize;
            var tickValues;
            if (tickSize) {
                tickValues = [];
                for (var tick = 0; tick <= maxYValue; tick += tickSize) {
                    tickValues.push(tick);
                }
            }

            return {
                tickValues: tickValues // only defined for issueCount and timeEstimates, otherwise, d3 will use default values
            };
        }
    };

    return SprintBurnupChart;
});