define('jira-agile/rapid/ui/chart/v2/burnup/burnup-report-controller', ['require'], function (require) {
    // REQUIRES
    var _ = require('underscore');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var SprintBurnupReportView = require('jira-agile/rapid/ui/chart/v2/burnup/burnup-report-view');
    var SprintBurnupChartDataService = require('jira-agile/rapid/ui/chart/v2/burnup/burnup-chart-data-service');

    var sprintBurnupChartDataService = new SprintBurnupChartDataService();

    // GLOBALS... FIX ME
    var PromiseKeeper;
    var RapidBoardState;
    var VersionsDataService;
    var RapidBoard = GH.RapidBoard;
    var sprintsDataService;

    GlobalEvents.on("pre-initialization", function () {
        PromiseKeeper = GH.PromiseKeeper;
        RapidBoardState = GH.RapidBoard.State;
        VersionsDataService = new GH.VersionsDataService();
        sprintsDataService = new GH.SprintsDataService();
    });

    function SprintBurnupReportController() {}

    SprintBurnupReportController.prototype = {
        SETTINGS_KEY: 'chart.config.sprintBurnup',
        displayed: false,

        _sprints: null,

        _model: null,
        _rapidViewData: null,
        _tracker: null,

        setRapidView: function setRapidView(rapidViewData) {
            this._rapidViewData = rapidViewData || {};
        },

        show: function show() {
            this.displayed = true;
            this._tracker = new AnalyticsTracker('gh.report.burnup');

            RapidBoard.State.pushState();

            // Initialise view
            var view = this._view = new SprintBurnupReportView();
            view.on('sprintSelected', _.bind(function (sprint) {
                this._model.sprint = sprint;
                this._tracker.trigger('sprintChanged');
                this.update();
            }, this)).on('estimateSelected', _.bind(function (estimateId) {
                this._model.estimate = estimateId;
                this._tracker.trigger('estimateChanged');
                this.update();
            }, this)).on('optionChanged', _.bind(function (changes) {
                this._model.zeroBaseline = changes.zeroBaseline;
                this.update();
            }, this));
            view.init(this._model);
            view.showSpinner();

            this._tracker.trigger('show');

            this._dataPromiseKeeper = new PromiseKeeper();
            var sprintsPromise = sprintsDataService.getSimpleSprints(this._rapidViewData.id);
            this._dataPromiseKeeper.add(sprintsPromise).done(_.bind(function (sprints) {
                if (!sprints.length) {
                    this._model.sprint = null;
                    view.showBlankState(true, true);
                    this._tracker.trigger('blank');
                    view.hideSpinner();
                } else {
                    sprints.reverse();
                    if (this._model.sprint == null) {
                        this._model.sprint = sprints[0].id;
                    }
                    this._sprints = sprints;
                    view.setSprints(sprints);
                    this.update(true);
                }
            }, this));
        },

        hide: function hide() {
            this.displayed = false;
            if (this._dataPromiseKeeper) {
                this._dataPromiseKeeper.reset();
            }
            if (this._view) {
                this._view.destroy();
                this._view = null;
            }
            sprintBurnupChartDataService.clearCache();
        },

        update: function update(isAfterLoadingVersions) {

            var controller = this;

            // if the selected sprint isn't in the list of selectable sprints (deleted, stale local data, etc)
            if (!this.isSprintSelectable(controller._model.sprint)) {
                //  select the most recent sprint instead.
                controller._model.sprint = controller._sprints[0].id;
            }

            if (isAfterLoadingVersions) {
                RapidBoard.State.replaceState();
            } else {
                // clears everything but the legend and howto
                RapidBoard.State.pushState();
            }

            if (this._model.sprint == null) {
                return;
            }

            this.saveState();

            // Get data and update chart
            controller._view.showSpinner();

            var dataPromise = sprintBurnupChartDataService.get(controller._rapidViewData.id, controller._model.sprint, controller._model.estimate);

            return controller._dataPromiseKeeper.add(dataPromise).done(function (data) {
                // Toggle blankstate
                if (data.chartData.estimatedIssueEvents === 0) {
                    // no events on the chart had estimates
                    controller._view.showBlankState(true);
                    controller._tracker.trigger('blank');
                } else {
                    controller._view.hideBlankState();
                }

                data.chartData.sprintId = controller._model.sprint;
                controller._view.setSprints(controller._sprints);
                controller._view.update(data.chartData);
                controller._view.updateTable(data);
            }).always(function (data) {
                if (data !== 'abort') {
                    controller._view.hideSpinner();
                }
            });
        },

        saveState: function saveState() {
            RapidBoard.State.setBoardSetting(this.SETTINGS_KEY, this._model);
        },

        loadState: function loadState() {
            this._model = RapidBoard.State.getBoardSetting(RapidBoard.State.getRapidViewId(), this.SETTINGS_KEY);
        },

        /**
         * Is the chart available for this type of board
         *
         * @param {Object} rapidViewConfig
         * @returns {boolean}
         */
        isApplicable: function isApplicable(rapidViewConfig) {
            return rapidViewConfig.sprintSupportEnabled;
        },

        /**
         * Return a normalized representation of the chart state.
         *
         * @returns {Object}
         */
        getNormalizedFromInternal: function getNormalizedFromInternal() {
            if (this._model === null) {
                this.loadState();
            }
            return _.defaults(_.clone(this._model), {});
        },

        /**
         * Update the chart state from a normalized representation.
         * Saves the state to localStorage as part of the update.
         *
         * @param {Object} normalized
         */
        updateInternalFromNormalized: function updateInternalFromNormalized(normalized) {
            if (!this._model) {
                this.loadState();
            }
            this._model = _.defaults(_.clone(normalized), this._model);
        },

        /**
         * Convert a normalized representation of the chart state into url query params.
         *
         * @param {Object} normalized
         * @returns {Object}
         */
        toUrl: function toUrl(normalized) {
            return {
                sprint: normalized.sprint !== null ? normalized.sprint : undefined,
                estimate: normalized.estimate !== null ? normalized.estimate : undefined
            };
        },

        /**
         * check if the provided sprint is in the list of selectable sprints.
         * @param currentSprint
         * @returns {boolean}
         */
        isSprintSelectable: function isSprintSelectable(currentSprint) {
            return this._sprints.filter(function (s) {
                return s.id === currentSprint;
            }).length > 0;
        },

        /**
         * Convert url query params into a normalized representation of the chart state.
         * Types must be cast correctly from string otherwise the URL manager will get confused when comparing equality.
         *
         * @param {Object} params
         * @returns {Object}
         */
        getNormalizedFromUrl: function getNormalizedFromUrl(params) {
            return {
                sprint: isNaN(params.sprint) ? undefined : +params.sprint,
                estimate: params.estimate
            };
        }
    };

    return SprintBurnupReportController;
});